/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-08-18: ChangeLog started
*/

#include "oagMapperSimple.h"
#include "oagFunc.h"
#include "oagFuncModGraph.h"
#include "oagFuncManager.h"

#include "oagMapperDebug.h"

namespace oagMapper {

// *****************************************************************************
// Simple()
//
/// \brief Constructor.
///
/// \param andGate the library cell to be used to map and operations
/// \param notGate the library cell to be used to map complementation
/// \param sequentialGate the library cell to be used to map all sequential logic
// *****************************************************************************
Simple::Simple(oa::oaDesign *andGate, oa::oaDesign *notGate,
               oa::oaDesign *sequentialGate) {
    assert(andGate);
    assert(notGate);
    assert(sequentialGate);

    this->andGate = andGate;
    identifyAndTerminals();
    this->notGate = notGate;
    identifyNotTerminals();    
    this->sequentialGate = sequentialGate;
    identifySeqTerminals();
    
    resetNet = clockNet = NULL;
    
    andCount = notCount = seqCount = 0;
}


// *****************************************************************************
// identifyAndTerminals()
//
/// \brief Identifies the function of each terminal of an AND gate.
///
/// There should be exactly three signal terminals, including two symmetric
/// inputs and one output.  The results are stored in the appropriate 
/// class variables.
//
// *****************************************************************************
void
Simple::identifyAndTerminals() {
    assert(this->andGate);
    assert(this->andGate->getTopModule());
    
    andInput0 = andInput1 = andOutput = NULL;
    
    oa::oaModTerm *term;
    oa::oaIter<oa::oaModTerm> termIter(andGate->getTopModule()->getTerms(oacTermIterSingleBit));
    int inputNumber = 0, outputNumber = 0;
    while((term = termIter.getNext())) {
        oa::oaModNet *net = term->getNet();
        oa::oaTermType type = term->getTermType();
        
        if (net->getSigType() == oa::oacSignalSigType) {
            if (type == oa::oacInputTermType) {
                if (inputNumber == 0) {
                    andInput0 = term;
                } else if (inputNumber == 1) {
                    andInput1 = term;
                } else {
                    cerr << "ERROR: Library AND-type gate has more than two signal inputs" << endl;
                    QUIT_ON_ERROR;                
                }
                inputNumber++;
            }
        
            if (type == oa::oacOutputTermType) {
                if (outputNumber == 0) {
                    andOutput = term;
                } else {
                    cerr << "ERROR: Library AND-type gate has more than one signal output" << endl;
                    QUIT_ON_ERROR;                
                }
                outputNumber++;
            }  
        }
    }
    
    if (!andInput0 || !andInput1 || !andOutput) {
        cerr << "ERROR: Could not find the necessary terminals on the library AND-type gate" << endl;
        QUIT_ON_ERROR;
    }
}


// *****************************************************************************
// identifyNotTerminals()
//
/// \brief Identifies the function of each terminal of a NOT gate.
///
/// There should be exactly two signal terminals, including one
/// input and one output.  The results are stored in the appropriate 
/// class variables.
//
// *****************************************************************************
void
Simple::identifyNotTerminals() {
    assert(notGate);
    assert(notGate->getTopModule()); 
    
    notInput = notOutput = NULL;
    
    oa::oaModTerm *term;
    oa::oaIter<oa::oaModTerm> termIter(notGate->getTopModule()->getTerms(oacTermIterSingleBit));
    int inputNumber = 0, outputNumber = 0;
    while((term = termIter.getNext())) {
        oa::oaModNet *net = term->getNet();
        oa::oaTermType type = term->getTermType();        
        
        if (net->getSigType() == oa::oacSignalSigType) {
            if (type == oa::oacInputTermType) {
                if (inputNumber == 0) {
                    notInput = term;
                } else {
                    cerr << "ERROR: Library NOT-type gate has more than one signal input" << endl;
                    QUIT_ON_ERROR;                
                }
                inputNumber++;
            }
        
            if (type == oa::oacOutputTermType) {
                if (outputNumber == 0) {
                    notOutput = term;
                } else {
                    cerr << "ERROR: Library NOT-type gate has more than one signal output" << endl;
                    QUIT_ON_ERROR;                
                }
                outputNumber++;
            }  
        }
    }
    
    if (!notInput || !notOutput) {
        cerr << "ERROR: Could not find the necessary terminals on the library NOT-type gate" << endl;
        QUIT_ON_ERROR;
    }    
}


// *****************************************************************************
// identifySeqTerminals()
//
/// \brief Identifies the function of each terminal of a SEQ gate.
///
/// The gate terminals are identified by name.  Every terminal described here
/// should be a scalar terminal.  
///
/// The following MUST be present:
/// \li "D" : data / next state input
/// \li "Q" : output
/// \li "CLK", "CK", "C" : clock
///
/// One of the following may also be present:
/// \li "RN" : asynchronous reset (negative edge triggered)
/// \li "R" : asynchronous reset (positive edge triggered)
///
/// One of the following may also be present:
/// \li "SN" : asynchronous preset (negative edge triggered)
/// \li "S" : asynchronous preset (positive edge triggered)
//
// *****************************************************************************
void
Simple::identifySeqTerminals() {
    assert(sequentialGate);
    oa::oaModule *sequentialModule = sequentialGate->getTopModule();
    assert(sequentialModule);
    
    const oa::oaVerilogNS verilogNS;
    seqInput = seqOutput = seqClock = seqReset = seqPreset = NULL;    
    
    // input pin
    oa::oaModScalarTerm *term = oa::oaModScalarTerm::find(sequentialModule,
        oa::oaScalarName(verilogNS, "D"));
    if (!term) {
        cerr << "ERROR: Could not find \"D\" terminal on sequential gate" << endl;
        exit(0);
    }
    seqInput = term;
    
    // output pin
    term = oa::oaModScalarTerm::find(sequentialModule,
        oa::oaScalarName(verilogNS, "Q"));
    if (!term) {
        cerr << "ERROR: Could not find \"Q\" terminal on sequential gate" << endl;
        QUIT_ON_ERROR;
    }
    seqOutput = term;
    
    // clock pin
    term = oa::oaModScalarTerm::find(sequentialModule,
        oa::oaScalarName(verilogNS, "CK"));
    if (!term) {
        term = oa::oaModScalarTerm::find(sequentialModule,
            oa::oaScalarName(verilogNS, "CLK"));   
    }
    if (!term) {
        term = oa::oaModScalarTerm::find(sequentialModule,
            oa::oaScalarName(verilogNS, "C"));   
    }
    if (!term) {
        cerr << "ERROR: Could not find clock terminal on sequential gate" << endl;
        QUIT_ON_ERROR;
    }
    seqClock = term;
    seqClockTrigger = POSEDGE; // assume posedge trigger
    
    // reset pin
    term = oa::oaModScalarTerm::find(sequentialModule,
        oa::oaScalarName(verilogNS, "RN"));
    if (term) {
        cout << "NOTE: Found a negedge triggered reset pin on sequential gate" << endl;
        seqReset = term;
        seqResetTrigger = NEGEDGE;        
    } else {
        term = oa::oaModScalarTerm::find(sequentialModule,
            oa::oaScalarName(verilogNS, "R"));
        if (term) {
            cout << "NOTE: Found a posedge triggered reset pin on sequential gate" << endl;
            seqReset = term;
            seqResetTrigger = POSEDGE;
        } else {
            cout << "WARNING: Did not find a reset pin on sequential gate" << endl;
        }
    }

    // preset pin
    term = oa::oaModScalarTerm::find(sequentialModule,
        oa::oaScalarName(verilogNS, "SN"));
    if (term) {
        cout << "NOTE: Found a negedge triggered preset pin on sequential gate" << endl;
        seqPreset = term;
        seqPresetTrigger = NEGEDGE;        
    } else {
        term = oa::oaModScalarTerm::find(sequentialModule,
            oa::oaScalarName(verilogNS, "S"));
        if (term) {
            cout << "NOTE: Found a posedge triggered preset pin on sequential gate" << endl;
            seqPreset = term;
            seqPresetTrigger = POSEDGE;
        } else {
            cout << "WARNING: Did not find a preset pin on sequential gate" << endl;
        }
    }
}


// *****************************************************************************
// removeSeqControlsFromLogic()
//
/// \brief Removes sequential control signals from logic.
///
/// All sequential control signals are synthesized in a very general manner.
/// They are added as generic triggers on a sequential node, and any 
/// additional functional behavior is implemented in logic surrounding the sequential 
/// node.
///
/// Two common examples of this are asynchronous resets and presets.
/// These are triggers, but also affect the function being latched (i.e.
/// the '0' function will be latched on an asynchronous reset and
/// the '1' function will be latched on an asynchronous preset).
///
/// In library gates that have asynchronous resets and presets, these inputs
/// will both trigger the gate and set its state to the appropriate value.
/// There is no need to keep the logic on the input of the sequential element
/// that had previously driven the next state input to the appropriate reset
/// values.  In fact, this logic is entirely useless and may confuse some
/// equivalence checkers.
///
/// This function removes the asynchronous resets and presets from the
/// input logic of sequential elements, replacing them with a constant that
/// reflects the unasserted value of the signal.
//
// *****************************************************************************
void
Simple::removeSeqControlsFromLogic() {

    // identify all reset nets
    if (resetNet) {
        oagFunc::ModRef resetRef = oagFunc::ModGraph::getNetToAiConnection(resetNet);
        assert(!oagFunc::ModGraph::isNull(resetRef));
 
        if (resetNetTrigger == POSEDGE) {
            // replace posedge resets with a zero
            oagFunc::ModGraph::resubstitute(resetRef, oagFunc::ModGraph::constantZero(resetRef.module));
        } else {
            // replace negedge resets with a one
            oagFunc::ModGraph::resubstitute(resetRef, oagFunc::ModGraph::constantOne(resetRef.module));
        }
       
    } else {
        cout << "WARNING: Reset nets may remain in the logic fan-in of registers (removal yet unimplemented)" << endl;
    }
}

    
// *****************************************************************************
// techmap()
//
/// \brief Maps a design to the specified gates.
///
/// All of the functional behavior of the provided design is implemented using
/// structural library components.
///
/// The result is an implementation of the AI graph with concrete AND and NOT
/// gates.  The structure of the graph will be preserved.
///
/// \param target
//
// *****************************************************************************
void
Simple::techmap(oa::oaModule *target) {
    assert(target);

    oagFunc::Manager *currentManager = oagFunc::Manager::get(target->getDesign());
    if (currentManager->isStructural()) {
        // the design is already entirely structural
        return;
    }

    mappedRefs.clear();
    andCount = notCount = seqCount = 0;

    oa::oaString moduleString;
    const oa::oaVerilogNS verilogNS;
    target->getName(verilogNS, moduleString);
    cout << "Mapping module: " << moduleString << endl;

    removeSeqControlsFromLogic();
    
    // map all nets
    oa::oaModNet *net;
    oa::oaIter<oa::oaModNet> netIter(target->getNets(oacNetIterSingleBit));
    while((net = netIter.getNext())) {
        oa::oaSigTypeEnum type(net->getSigType());
        switch(type) {
          case oa::oacPowerSigType:
          case oa::oacGroundSigType:
          case oa::oacTieoffSigType:
          case oa::oacTieHiSigType:
          case oa::oacTieLoSigType:
            continue;
          case oa::oacAnalogSigType:
            cout << "Warning: Ignoring analog net " << endl;
            continue;
          case oa::oacScanSigType:
            cout << "Warning: Ignoring scan net " << endl;
            continue;
          case oa::oacSignalSigType:
          case oa::oacClockSigType:
          case oa::oacResetSigType:
            // fall through
            break;
        }
        oa::oaModBitNet *bit = oagFunc::toBitNet(net);
        oagFunc::ModRef ref = oagFunc::ModGraph::getNetToAiConnection(bit);
        if (!oagFunc::ModGraph::isNull(ref)) {
            assert(oagFunc::ModGraph::isTerminal(ref));
            oagFunc::ModRef driver = oagFunc::ModGraph::getTerminalDriver(ref);
            if (!oagFunc::ModGraph::isNull(driver)) {
                oa::oaModBitNet *mapped_bit = techmap(driver);
                oagFunc::ModGraph::decrementExternalReferences(ref);
                oagFunc::ModGraph::removeNetToAiConnection(bit);
                bit->makeEquivalent(mapped_bit);
            }
        }
    }
    
    // detach all mapped nodes
    for(map<oagFunc::ModRef, oa::oaModBitNet*>::iterator refIter = mappedRefs.begin();
            refIter != mappedRefs.end(); ++refIter) {
        // nodes only need to be detached once.  ignore inverted references
        if (!oagFunc::ModGraph::isInverted(refIter->first)) {
            oagFunc::ModGraph::detach(refIter->first);
        }
    }
    mappedRefs.clear();
    
    // garbage collect
    oagFunc::Manager::get(target->getDesign())->garbageCollect();
    
    cout << "\tused " << andCount << " ANDs, " 
         << notCount << " NOTs, " 
         << seqCount << " SEQs" << endl;
}


// *****************************************************************************
// techmap()
//
/// \brief Recursively maps an Ai reference to structural equivalents.
//
// *****************************************************************************
oa::oaModBitNet*
Simple::techmap(oagFunc::ModRef ref) {
    assert(!oagFunc::ModGraph::isNull(ref));

    oa::oaModule *currentModule = ref.module;

    // has this reference already been mapped?
    if (mappedRefs.find(ref) != mappedRefs.end()) {
        return mappedRefs[ref];    
    }
    
    if (oagFunc::ModGraph::isInverted(ref)) {
        DEBUG_PRINTLN("\t mapping to NOT: " << ref.ref)
        ++notCount;
        
        // create a net for mapped output
        oa::oaModBitNet *mapped_out = oa::oaModScalarNet::create(currentModule);
        
        // recursively map inputs
        mappedRefs[ref] = mapped_out;
        oa::oaModBitNet *mapped_in0 = techmap(oagFunc::ModGraph::notOf(ref));

        // map to a NOT gate
        oa::oaModScalarInst *inst = oa::oaModScalarInst::create(currentModule, notGate);
        oa::oaModInstTerm::create(mapped_in0, inst, notInput);
        oa::oaModInstTerm::create(mapped_out, inst, notOutput);
        
        return mapped_out;

    } else if (oagFunc::ModGraph::getNodeType(ref) == oagAi::Node::AND) {
        DEBUG_PRINTLN("\t mapping to AND: " << ref.ref)
        ++andCount;

        // map to an AND gate
        oa::oaModScalarInst *inst = oa::oaModScalarInst::create(currentModule, andGate);
        
        // create a net for mapped output
        oa::oaModBitNet *mapped_out = oa::oaModScalarNet::create(currentModule);
        oa::oaModInstTerm::create(mapped_out, inst, andOutput); 
        mappedRefs[ref] = mapped_out;
        
        // recursively map inputs
	oagFunc::ModRef left = oagFunc::ModGraph::getAndLeft(ref);
	oagFunc::ModRef right = oagFunc::ModGraph::getAndRight(ref);
	
	if (!oagFunc::ModGraph::isNull(left)) {
	    oa::oaModBitNet *mapped_in0 = techmap(left);
	    oa::oaModInstTerm::create(mapped_in0, inst, andInput0);
	} else {
	    cout << "WARNING: Floating input on ";
	    ref.print();
	}
	if (!oagFunc::ModGraph::isNull(right)) {
	    oa::oaModBitNet *mapped_in1 = techmap(right);
	    oa::oaModInstTerm::create(mapped_in1, inst, andInput1);
	} else {
	    cout << "WARNING: Floating input on ";
	    ref.print();
	}

        return mapped_out;
        
    } else if (oagFunc::ModGraph::getNodeType(ref) == oagAi::Node::SEQUENTIAL) {
        DEBUG_PRINTLN("\t mapping to SEQ: " << ref.ref)
        ++seqCount;
        
        // map to a SEQ gate
        oa::oaModScalarInst *inst = oa::oaModScalarInst::create(currentModule, sequentialGate);

        // create a net for mapped output
        oa::oaModBitNet *mapped_out = oa::oaModScalarNet::create(currentModule);
        oa::oaModInstTerm::create(mapped_out, inst, seqOutput);
        mappedRefs[ref] = mapped_out;
        
        // recursively map inputs
	oagFunc::ModRef nextState = oagFunc::ModGraph::getNextState(ref);

	if (!oagFunc::ModGraph::isNull(nextState)) {
	    oa::oaModBitNet *mapped_in0 = techmap(nextState);
	    oa::oaModInstTerm::create(mapped_in0, inst, seqInput);
	} else {
	    cout << "WARNING: Floating input on ";
	    ref.print();
	}
  
        connectSeqControls(ref, inst);
        
        return mapped_out;
        
    } else if (oagFunc::ModGraph::getNodeType(ref) == oagAi::Node::CONSTANT0) {
        oa::oaModNet *net = oa::oaModNet::find(currentModule, oa::oaScalarName(oa::oaNativeNS(), "tie0"));
        assert(net);
        
        return oagFunc::toBitNet(net);
        
    } else if (oagFunc::ModGraph::getNodeType(ref) == oagAi::Node::TERMINAL) {
        oa::oaModBitNet *net = oagFunc::ModGraph::getNetToAiConnection(ref);
        if (!net) {
	    oagFunc::ModRef driver = oagFunc::ModGraph::getTerminalDriver(ref);
	    if (!oagFunc::ModGraph::isNull(driver)) {
		net = techmap(driver);
	    } else {
		cout << "WARNING: Floating input on ";
		ref.print();	
		net = oa::oaModScalarNet::create(currentModule);
	    }
        } else if (net->isImplicit()) {
            oa::oaModBitNet *implicitNet = net;
            net = oa::oaModScalarNet::create(currentModule);
            net->makeEquivalent(implicitNet);
        }
        mappedRefs[ref] = net;
        
        return net;
    }

    assert(false);
    return NULL;
}


// *****************************************************************************
// connectSeqControls()
//
/// \brief Connects the sequential controls, such as clocks, resets, etc.
///
/// \param ref
/// \param inst
//
// *****************************************************************************
void
Simple::connectSeqControls(oagFunc::ModRef ref, oa::oaModInst *inst) {

    oa::oaModule *currentModule = ref.module;

    oagAi::Node::SequentialData *data = oagFunc::ModGraph::getSequentialData(ref);   
     
    // the node must be a BLACK_BOX abstraction level
    assert(data->abstractionLevel == oagAi::Node::SequentialData::BLACK_BOX);

    // if global clock net, connect it...
    if (clockNet) {
        // are the trigger type of the global net and the library cell the same?
        if (seqClockTrigger != clockNetTrigger) {
            // if not, insert an inverted on the input            
            oa::oaModScalarInst *notInst = oa::oaModScalarInst::create(currentModule, notGate);
            oa::oaModInstTerm::create(clockNet, notInst, notInput);
            oa::oaModScalarNet *localInvertedClockNet = oa::oaModScalarNet::create(currentModule);
            oa::oaModInstTerm::create(localInvertedClockNet, notInst, notOutput);
            ++notCount;

            oa::oaModInstTerm::create(localInvertedClockNet, inst, seqClock);
        } else {
            oa::oaModInstTerm::create(clockNet, inst, seqClock);
        }
    } else {
        // otherwise, connect the first posedge trigger
        if (data->signals.find("trigger_0_posedge") != data->signals.end()) {
            oagFunc::ModRef nextRef(data->signals["trigger_0_posedge"], currentModule);
            oa::oaModBitNet *mapped_clock = techmap(nextRef);           
            oa::oaModInstTerm::create(mapped_clock, inst, seqClock);            
        } else {
                cout << "WARNING: Clock unconnected.  No posedge trigger for register at node: " << ref.ref << endl;
            }
        }
        
    // if sequential element has a reset...
    if (seqReset) {
        // if global reset net, connect it...
        if (resetNet) {
            // are the trigger type of the global net and the library cell the same?
            if (seqResetTrigger != resetNetTrigger) {
                // if not, insert an inverted on the input
                oa::oaModScalarInst *notInst = oa::oaModScalarInst::create(currentModule, notGate);
                oa::oaModInstTerm::create(resetNet, notInst, notInput);
                oa::oaModScalarNet *localInvertedResetNet = oa::oaModScalarNet::create(currentModule);
                oa::oaModInstTerm::create(localInvertedResetNet, notInst, notOutput);
                ++notCount;

                oa::oaModInstTerm::create(localInvertedResetNet, inst, seqReset);
            } else {
                oa::oaModInstTerm::create(resetNet, inst, seqReset);
            }
        } else {
            // otherwise, choose the second posedge or first negedge signal
            if (data->signals.find("trigger_1_posedge") != data->signals.end()) {
                oagFunc::ModRef nextRef(data->signals["trigger_1_posedge"], currentModule);
                // does this trigger type need to be inverted to match the library gate?
                if (seqResetTrigger == NEGEDGE) {
                    nextRef = oagFunc::ModGraph::notOf(nextRef);
                }
                oa::oaModBitNet *mapped_reset = techmap(nextRef);           
                oa::oaModInstTerm::create(mapped_reset, inst, seqReset);
            } else if (data->signals.find("trigger_1_negedge") != data->signals.end()) {
                oagFunc::ModRef nextRef(data->signals["trigger_1_negedge"], currentModule);
                // does this trigger type need to be inverted to match the library gate?
                if (seqResetTrigger == POSEDGE) {
                    nextRef = oagFunc::ModGraph::notOf(nextRef);
                }
                oa::oaModBitNet *mapped_reset = techmap(nextRef);
                oa::oaModInstTerm::create(mapped_reset, inst, seqReset); 
            } else {
                cout << "WARNING: Could not identify reset for register at node: " << ref.ref << endl;
                oa::oaModNet *tieOff;
                const oa::oaVerilogNS verilogNS;
                if (seqResetTrigger == POSEDGE) {
                    tieOff = oa::oaModNet::find(currentModule, oa::oaName(verilogNS, "tie0"));
                } else {
                    tieOff = oa::oaModNet::find(currentModule, oa::oaName(verilogNS, "tie1"));
                }
                assert(tieOff);                    
                oa::oaModInstTerm::create(tieOff, inst, seqReset);
            }
        }
    }

    // if sequential element has a preset...
    if (seqPreset) {
        // tie off
        oa::oaModNet *tieOff;
        const oa::oaVerilogNS verilogNS;
        if (seqPresetTrigger == POSEDGE) {
            tieOff = oa::oaModNet::find(currentModule, oa::oaName(verilogNS, "tie0"));
        } else {
            tieOff = oa::oaModNet::find(currentModule, oa::oaName(verilogNS, "tie1"));
        }
        assert(tieOff);
        oa::oaModInstTerm::create(tieOff, inst, seqPreset);
    }
}


// *****************************************************************************
// mergeEquivalentNets()
//
/// \brief Replaces all groups of equivalent nets with a single net.
///
/// \param module
//
// *****************************************************************************
void
Simple::mergeEquivalentNets(oa::oaModule *module) {
    assert(module);

    // merge all nets
    bool changed = true;
    while(changed) {
        changed = false;

        // make all bus net bits explicit
        oa::oaModNet *net;
        oa::oaIter<oa::oaModNet> netIter2(module->getNets(oacNetIterNotImplicit));
        while((net = netIter2.getNext())) {
            if(!net->isImplicit() && net->getNumBits() > 1) {
                net->scalarize();
            }
        }

        oa::oaIter<oa::oaModNet> netIter(module->getNets(oacNetIterSingleBit));
        while(!changed && (net = netIter.getNext())) {
            oa::oaModBitNet *preferred = oagFunc::toBitNet(net)->getPreferredEquivalent();

            oa::oaModBitNet *equivNet;
            oa::oaIter<oa::oaModBitNet> equivIter(preferred->getEquivalentNets());      
            while((equivNet = equivIter.getNext())) {

                // move instTerms
                oa::oaModInstTerm *instTerm;
                oa::oaIter<oa::oaModInstTerm> instTermIter(equivNet->getInstTerms());      
                while((instTerm = instTermIter.getNext())) {
                    instTerm->addToNet(preferred);
                }
                
                // move terms
                oa::oaModTerm *term;
                oa::oaIter<oa::oaModTerm> termIter(equivNet->getTerms());      
                while((term = termIter.getNext())) {
                    term->moveToNet(preferred);
                }
               
                // resubstitute graph
                oagFunc::ModRef target = oagFunc::ModGraph::getNetToAiConnection(equivNet);
                if (!oagFunc::ModGraph::isNull(target)) {
                    oagFunc::ModRef replacement = oagFunc::ModGraph::prepareNetToAiConnection(preferred);
                    oagFunc::ModGraph::resubstitute(target, replacement);
                    oagFunc::ModGraph::removeNetToAiConnection(equivNet);
                }
                
                // destroy net or at least break equivalence
                equivNet->destroy();
                changed = true;
            }
        }
    }   
}

}
