/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron Hurst <ahurst@eecs.berkeley.edu>

ChangeLog:
2005-07-20: ChangeLog started
*/

#include <stdlib.h>
#include <iostream>
#include <sstream>
#include "oaDesignDB.h"
#include "oagFunc.h"
#include "oagFuncSynthesis.h"
#include "oagFuncOccGraph.h"
#include "oagFuncModGraph.h"
#include "oagFuncSimOcc.h"
#include "oagFuncSimMod.h"
#include "oagUtilOption.h"
#include "oagUtilOptionParser.h"


// *****************************************************************************
// Global constant definitions.
// *****************************************************************************

const static int NAME_COL_WIDTH = 40;
const static int CYCLE_COL_WIDTH = 3;
const static int VECTOR_COL_WIDTH = 3;


// *****************************************************************************
// findOpenLib()
//
/// \brief Find an already open, opens from disk, or creates a library.
///
// *****************************************************************************
oa::oaLib *
findOpenLib(const oa::oaScalarName libName, const char *libString) {
    oa::oaLib *lib = NULL;

    if ((lib = oa::oaLib::find(libName)))
        ;
    else if (!oa::oaLib::exists(libString)) {
        lib = oa::oaLib::create(libName, libString);
        oa::oaLibDefList *defList = oa::oaLibDefList::get("lib.defs", 'a');
        oa::oaLibDef::create(defList, libName, libString);
        defList->save();
    } else {
        lib = oa::oaLib::open(libName, libString); 
    }

    return lib;
}


// *****************************************************************************
// printVector()
//
/// \brief Prints a vector in a particular format.
///
/// \param val
///
// *****************************************************************************
void
printVector(const unsigned int v, const char outputFormat) {

  double int_max = INT_MAX;
  const int DEC_DIGITS = static_cast<int>(log(int_max)/log(10.0)+1);
  const int OCT_DIGITS = sizeof(int)<<5;

  switch(outputFormat) {
  case 'd':
    cout << ios::dec;
    cout.width(DEC_DIGITS);
    cout << v;
    break;
  case 'o':
    cout << ios::oct;
    cout.width(OCT_DIGITS);
    cout << v;
  case 'b':
    for(int i=sizeof(int)*8-1; i >= 0; i-=1) {
      std::cout << static_cast<char>('0' + ((v >> i) & 0x1));
      if (i%8==0) {
        std::cout << " ";
      }
    }
    break;
  case 'h':
  default:
    for(int i=sizeof(int)*8-4; i >= 0; i-=4) {
      char c;
      std::cout << static_cast<char>((c = (v >> i) & 0xf) <= 0x9 ? '0'+c : 'A'+c-0xA );
    }
    break;
  }
}


// *****************************************************************************
// doModSim()
//
/// \brief Performs a random simulation on a design in the module domain.
///
/// Significantly faster than an occurrence-domain simulation, but unable to
/// handle hierarchy.
///
/// \param design the design to simulate
/// \param cycles the number of add'l cycles to simulate
///
// *****************************************************************************
void 
doModSim(oa::oaDesign *design, int cycles, char outputFormat) {

    const oa::oaVerilogNS verilogNS;

    // Create the simulator and generate an initial set of input/state vectors
    oagFunc::SimMod sim(design);
    sim.generateRandomInputVectors();
    sim.generateRandomStateVectors();

    // oagFunc::Manager::get(design)->print();  // DEBUG: print graph info

    // Print input vectors
    std::cout << "[primary input vectors]" << std::endl;
    list<oagFunc::ModRef> inputs;
    oagFunc::ModGraph::getInputs(design->getTopModule(), inputs);
    for(list<oagFunc::ModRef>::iterator it=inputs.begin();
        it!=inputs.end(); ++it) {
        oagFunc::SimulationVector vec;
        sim.getVector(*it, vec);
        oa::oaModBitNet *net = oagFunc::ModGraph::getNetToAiConnection(*it);
        assert(net);
        oa::oaString netNameString;
        net->getName(verilogNS, netNameString);
        std::cout.width(NAME_COL_WIDTH);
        std::cout << netNameString << ": ";
        printVector(vec, outputFormat);
        std::cout << std::endl;
    }
    
    // Print state vectors
    std::cout << std::endl << "[initial state vectors]" << std::endl;
    list<oagFunc::ModRef> states;
    oagFunc::ModGraph::getLocalStates(design->getTopModule(), states);
    for(list<oagFunc::ModRef>::iterator it=states.begin();
        it!=states.end(); ++it) {

        std::stringstream stateNameString;
        stateNameString << "state_" << it->ref;
        std::cout.width(NAME_COL_WIDTH);
        std::cout << stateNameString.str() << ": ";

        oagFunc::SimulationVector vec;
        sim.getVector(*it, vec);
        printVector(vec, outputFormat);
        std::cout << std::endl;
    }
 
    // Simulate multiple cycles
    for(int iter = 0; iter < cycles+1; ++iter) {
        std::cout << "-------- cycle ";
        std::cout.width(CYCLE_COL_WIDTH);
        std::cout << iter;
        std::cout << "----------------------------------------" << std::endl;

        // Run the simulation
        sim.runFull();

        // Print output vectors
        std::cout << "[primary output vectors]" << std::endl;
        oa::oaModTerm *term;
        oa::oaIter<oa::oaModTerm> termIter(design->getTopModule()->getTerms(oacTermIterSingleBit));
        while((term = termIter.getNext())) {
          if (term->getTermType() == oa::oacOutputTermType) {
            oa::oaModBitNet *net = oagFunc::toBitNet(term->getNet());
            assert(net);
            
            oagFunc::ModRef ref = oagFunc::ModGraph::getNetToAiConnection(net);
            assert(!oagFunc::ModGraph::isNull(ref));
            oagFunc::SimulationVector vec;
            sim.getVector(ref, vec);
            
            oa::oaString termNameString;
            term->getName(verilogNS, termNameString);
            std::cout.width(NAME_COL_WIDTH);
            std::cout << termNameString << ": ";
            printVector(vec, outputFormat);
            std::cout << std::endl;
          }
        }
        std::cout << std::endl;
        
        // Advance another time step
        sim.nextCycle();
    }
}


// *****************************************************************************
// doOccSim()
//
/// \brief Performs a random simulation on a design in the occurrence domain.
///
/// In general, occurrence domain simulation is much slower than module
/// domain simulation, especially if the design has a large number of atomic
/// functional units (i.e. a mapped design).  The simulator must traverse 
/// and unfold the design hierarchy as necessary to propagate the vectors.  
/// In most cases, it's preferrable to flatten the design first and run a 
/// module domain simulation.
///
/// \param design the design to simulate
/// \param cycles the number of add'l cycles to simulate
///
// *****************************************************************************
void 
doOccSim(oa::oaDesign *design, int cycles, char outputFormat) {

    const oa::oaVerilogNS verilogNS;

    // Create the simulator and generate an initial set of input/state vectors
    oagFunc::SimOcc sim(design);
    sim.generateRandomInputVectors();
    sim.generateRandomStateVectors();

    // oagFunc::Manager::get(design)->print();  // DEBUG: print graph info

    // Print input vectors
    std::cout << "[primary input vectors]" << std::endl;
    list<oagFunc::OccRef> inputs;
    oagFunc::OccGraph::getInputs(design->getTopOccurrence(), inputs);
    for(list<oagFunc::OccRef>::iterator it=inputs.begin();
        it!=inputs.end(); ++it) {
        oagFunc::SimulationVector vec;
        if (sim.getVector(*it, vec)) {
            oa::oaOccBitNet *net = oagFunc::OccGraph::getNetToAiConnection(*it);
            assert(net);
            oa::oaString netNameString;
            net->getName(verilogNS, netNameString);
            std::cout.width(NAME_COL_WIDTH);
            std::cout << netNameString << ": ";
            printVector(vec, outputFormat);
            std::cout << std::endl;
        } else {
            std::cerr << "ERROR: No vector for an input" << std::endl;
        }
    }
    
    // Print state vectors
    std::cout << std::endl << "[initial state vectors]" << std::endl;
    list<oagFunc::OccRef> states;
    oagFunc::OccGraph::getStates(design->getTopOccurrence(), states);
    for(list<oagFunc::OccRef>::iterator it=states.begin();
        it!=states.end(); ++it) {
        oagFunc::SimulationVector vec;
        if (sim.getVector(*it, vec)) {
            oa::oaString occString;
            it->occurrence->getOccInst()->getPathName(oa::oaVerilogNS(), occString);
            std::stringstream stateNameString;
            stateNameString << "state_" << occString << "_" << it->ref;
            std::cout.width(NAME_COL_WIDTH);
            std::cout << stateNameString.str() << ": ";

            printVector(vec, outputFormat);
            std::cout << std::endl;
        } else {
            std::cerr << "ERROR: No vector for an input" << std::endl;
        }
    }
 
    // Simulate multiple cycles
    for(int iter = 0; iter < cycles+1; ++iter) {
        std::cout << "-------- cycle ";
        std::cout.width(CYCLE_COL_WIDTH);
        std::cout << iter;
        std::cout << "----------------------------------------" << std::endl;

        // Run the simulation
        sim.runFull();

        // Print output vectors
        std::cout << "[primary output vectors]" << std::endl;
        oa::oaOccTerm *term;
        oa::oaIter<oa::oaOccTerm> termIter(design->getTopOccurrence()->getTerms(oacTermIterSingleBit));
        while((term = termIter.getNext())) {
          if (term->getTermType() == oa::oacOutputTermType) {
            oa::oaOccBitNet *net = oagFunc::toBitNet(term->getNet());
            assert(net);
            
            oagFunc::OccRef ref = oagFunc::OccGraph::getNetToAiConnection(net);
            assert(!oagFunc::OccGraph::isNull(ref));
            oagFunc::SimulationVector vec;
            if (!sim.getVector(ref, vec)) {
                std::cerr << "ERROR: No vector for an output" << std::endl;
                exit(0);
            }
            
            oa::oaString termNameString;
            term->getName(verilogNS, termNameString);
            std::cout.width(NAME_COL_WIDTH);
            std::cout << termNameString << ": ";
            printVector(vec, outputFormat);
            std::cout << std::endl;
          }
        }
        std::cout << std::endl;
        
        // Advance another time step
        sim.nextCycle();
    }
}


// *****************************************************************************
// main()
//
// *****************************************************************************
int main(int argc, const char **argv) {

    std::cout << "*****************************************************************************" << std::endl;
    std::cout << "Tool:   simulate        Package:    oagFunc     version:  01/2007" << std::endl;
    std::cout << "*****************************************************************************" << std::endl;
    std::cout << std::endl;

    oagUtil::OptionParser options("Simple test program to demonstrate the use "
            "of the OAGear Func package, to read a design from OpenAccess or "
            "Verilog, and simulate the behavior using a set of random input"
            "vectors");

    oagUtil::Option *libOpt = options.add("lib", "Input and output library "
            "name", true, "library");
    oagUtil::Option *viewOpt = options.add("view", "Input and output view name "
            "(default: netlist)", false, "view");
    oagUtil::Option *cellOpt = options.add("cell", "Input cell "
            "name", true, "cell");
    oagUtil::Option *libertyOpt = options.add("liberty", ".lib library file", 
            false, "file");
    oagUtil::Option *verilogOpt = options.add("verilog", "Verilog file", false, 
            "file");
    oagUtil::Option *leafLibsOpt = options.add("leafLibs", "Library files for leaf cells",
             false, "\"libList\"");
    oagUtil::Option *leafViewsOpt = options.add("leafViews", "Library views for leaf cells", 
             false, "\"viewList\"");
    oagUtil::Option *numOpt = options.add("num", "Number of random vectors to "
            "simulate (default: 1)", false, "int");
    oagUtil::Option *cycleOpt = options.add("cycles", "Number of addt'l cycles "
            "per vector to simulate (default: 0)", false, "int");
    oagUtil::Option *hierOpt = options.add("hierarchy", "Treatment of design "
            "hierarchy (default: flatten)", false,"ignore|flatten|keep");
    oagUtil::Option *seedOpt = options.add("seed", "Use a specific random seed"
            " (default: use system time)", false, "int");
    oagUtil::Option *formatOpt = options.add("format", "Display all simulation"
             " vectors in format (default: hex)", false,"bin|oct|dec|hex");

    if (!options.parse(argc, argv)) {
        std::cerr << options.getMessage();
        exit(0);
    }

    const char *libString = libOpt->getValue();
    const char *viewString 
        = viewOpt->isGiven() ? viewOpt->getValue(): "netlist";

    // parse output format
    char outputFormat = 0;
    if (formatOpt->isGiven()) {
      outputFormat = formatOpt->getValue()[0];
    }

    try {

        // Initialize OpenAccess

        oa::oaDesignInit();
        oa::oaLibDefList::openLibs();
        oagFunc::initialize();

        const oa::oaNativeNS nativeNS;
        const oa::oaVerilogNS verilogNS;
        const oa::oaScalarName libName(nativeNS, libString);
        const oa::oaScalarName viewName(nativeNS, viewString);

        // Find, create, or load the design library and view

        oa::oaLib *library = findOpenLib(libName, libString);
        assert(library);

        // Find, create, or load additional libraries (if given)
        
        if (leafLibsOpt->isGiven()) {
            assert(strlen(leafLibsOpt->getValue())<256);
            char opt[256];
            strcpy(opt, leafLibsOpt->getValue());
            const char *tok = strtok(opt, "\" ,");
            while(tok) {
              oa::oaScalarName leafLibName(nativeNS, tok);
              oa::oaLib *leafLib = findOpenLib(leafLibName, tok);
              assert(leafLib);
              oagFunc::Synthesis::addLeafLibrary(leafLib);
              tok = strtok(NULL, "\" ,");
            }
            if (leafViewsOpt->isGiven()) {
              assert(strlen(leafViewsOpt->getValue())<256);
              char opt[256];
              strcpy(opt, leafViewsOpt->getValue());
              char *tok = strtok(opt, "\" ,");
              while(tok) {
                oa::oaScalarName leafViewName(nativeNS, tok);
                oagFunc::Synthesis::addLeafView(leafViewName);
                tok = strtok(NULL, "\" ,");
              }
            } else {
                std::cout << "WARNING: No leaf views provided. Assuming \"netlist\" view" << endl;
                oa::oaScalarName leafViewName(nativeNS, "netlist");
                oagFunc::Synthesis::addLeafView(leafViewName);
            }
        }

        // Read designs from a Liberty file

        if (libertyOpt->isGiven()) {
            // read in a cell library
            oagFunc::readLiberty(library, viewName, libertyOpt->getValue());
        }

        // Read designs from a Verilog file

        if (verilogOpt->isGiven()) {
            // read in functional Verilog description
            oagFunc::readVerilog(library, viewName, verilogOpt->getValue());
        }

        // Open design

        const char *cellString = cellOpt->getValue();
        const oa::oaScalarName cellName(nativeNS, cellString);
        oa::oaDesign *design;
        if ((design = oa::oaDesign::find(libName, cellName, viewName))) {
            ;
        } else {
            // open for read only
            design = oa::oaDesign::open(libName, cellName, viewName, 'r');
        }

        if (!design) {
            std::cerr << "ERROR: Could not open design" << std::endl;
            exit(1);
        }
        
        // Seed random generator w/ user-specified num or system time
        
        if (seedOpt->isGiven()) {
            srand(atoi(seedOpt->getValue()));
        } else {
            srand(time(NULL));
        }
        
        // Run simulation

        int cycles = 0;
        if (cycleOpt->isGiven()) {
            cycles = atoi(cycleOpt->getValue());
        }

        int numVectors = 1;
        if (numOpt->isGiven()) {
            numVectors = atoi(numOpt->getValue());
        }

        char hierOptChar = 'f';
        if (hierOpt->isGiven()) {
          hierOptChar = hierOpt->getValue()[0];
        }
        switch(hierOptChar) {
        case 'k':
          for (int n=0; n<numVectors; n++) {
            std::cout << "======== vector ";
            std::cout.width(VECTOR_COL_WIDTH);
            std::cout << n;
            std::cout << "========================================" << std::endl;
            doOccSim(design, cycles, outputFormat);
          }
          break;
        case 'f':
          std::cout << "Flattening design..." << std::endl << std::endl;
          oagFunc::collapseAllInstancesAllLevels(design->getTopModule());
        case 'i':
          for (int n=0; n<numVectors; n++) {
            std::cout << "======== vector ";
            std::cout.width(VECTOR_COL_WIDTH);
            std::cout << n;
            std::cout << "========================================" << std::endl;
            doModSim(design, cycles, outputFormat);
          }
          break;
        default:
          std::cerr << "ERROR: Can't understand option for design hierarchy treatment " 
                    << hierOpt->getValue() << endl;
          exit(0);
          break;
        }
    
        // Close design

        oa::oaDesign *openDesign;
        oa::oaIter<oa::oaDesign> designIter(oa::oaDesign::getOpenDesigns());
        while((openDesign = designIter.getNext())) {
            oa::oaString designString;
            openDesign->getTopModule()->getName(nativeNS, designString);
            openDesign->purge();
        }

        // Handle errors

    } catch(oa::oaException &e) {
        std::cerr << "ERROR: " << e.getMsg() << std::endl;
        return 1;
    }

    return 0;
}
