/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-06-01: ChangeLog started
*/

#if !defined(oagFuncReasoningEngine_P)
#define oagFuncReasoningEngine_P

#include <stdint.h>

namespace oagFunc {

// *****************************************************************************
// GenericFunction
//
// The actual representation that a user will choose for a logic function is
// not known a priori.  We either must have them 1. cast a void* to their
// desired type 2. obscure the type through another layer of polymorphism.
// Because this callback will be used quite heavily, the former solution is
// selected for its performance advantages.
//
// *****************************************************************************

typedef void* GenericFunction;


// *****************************************************************************
// GenericReasoningEngine
//
/// \brief The abstract callback interface for an engine to reason about functionality in an unknown representation.
//
// *****************************************************************************

class GenericReasoningEngine {

public:

    // --- callback functions

    // return the function of a false expression
    virtual GenericFunction         genericGetZero( ) = 0;

    // logic operations on preexisting functions
    virtual GenericFunction         genericAndOf( GenericFunction a, GenericFunction b ) = 0;
    virtual GenericFunction         genericNotOf( GenericFunction a ) = 0;
    
};


// *****************************************************************************
// ReasoningEngine
//
/// \brief The abstract callback interface for an engine to reason about functionality in a known representation.
//
// This class is implemented by the user to perform functional queries using
// a functional representation of their choice (BDDs, CNF, etc.)
//
// *****************************************************************************

template <class SpecificFunction>
class ReasoningEngine : public GenericReasoningEngine {

public:

    // return the function of a false expression
    virtual SpecificFunction        getZero( ) = 0;

    // logic operations on preexisting functions
    virtual SpecificFunction        andOf( SpecificFunction a, SpecificFunction b ) = 0;
    virtual SpecificFunction        notOf( SpecificFunction a ) = 0;

    // bashing of type system
    static GenericFunction          castG(SpecificFunction a) {
	// query objects store GenericFunction, so this is required
	assert(sizeof(GenericFunction) >= sizeof(SpecificFunction));
	assert(sizeof(uintptr_t)       >= sizeof(SpecificFunction));
	uintptr_t tmpa = (uintptr_t) a;
	return (GenericFunction) tmpa;
    }
    static SpecificFunction         castS(GenericFunction  a) {
	uintptr_t tmpa = (uintptr_t) a;
	return (SpecificFunction) tmpa;
    }

    GenericFunction                 genericGetZero() 
	{ return castG(getZero()); }
    GenericFunction                 genericAndOf( GenericFunction a, GenericFunction b ) 
	{ return castG(andOf(castS(a), castS(b))); }
    GenericFunction                 genericNotOf( GenericFunction a ) 
	{ return castG(notOf(castS(a))); }

};


}

#endif
