/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-06-01: ChangeLog started
*/

#include "oaDesignDB.h"
#include "oagFunc.h"
#include "oagFuncObserver.h"
#include "oagFuncManager.h"

#include "oagFuncDebug.h"

namespace oagFunc {

// *****************************************************************************
// observer()
//
// Observer class catches open/modify/save actions.
//
// *****************************************************************************
Observer::Observer() : oa::oaObserver<oa::oaDesign>(10) { 
    DEBUG_PRINTLN("initialized observer")
}


// *****************************************************************************
// onFirstOpen()
//
/// This callback is invoked when an oaDesign is first opened.
//
// *****************************************************************************
void    
Observer::onFirstOpen(oa::oaDesign *design) {
    if(design->getTopModule()) {
        oa::oaString moduleNameString;
        design->getTopModule()->getName(oa::oaNativeNS(), moduleNameString);
        DEBUG_PRINTLN("open " << moduleNameString)
    } else {
        DEBUG_PRINTLN("open ")
    }

    if( !design->getAppDefs().includes(streamAppDef) ) {
        DEBUG_PRINTLN("\tno functional description");
        return;
    }
    oa::oaByte *buf = new oa::oaByte[streamAppDef->getSize(design)];
    streamAppDef->get(design, buf);
    int size = Manager::unserialize(static_cast<void*>(buf), design);
    static_cast<void>(size);
    DEBUG_PRINTLN("\t" << "serialized design occupied " << size << " bytes")
    
    streamAppDef->destroy(design);
    DEBUG_PRINTLN("\t" << "serialized info destroyed")
    delete [] buf;
}


// *****************************************************************************
// onPurge()
//
/// This callback is invoked before an oaDesign is removed from memory.
//
// *****************************************************************************
void    
Observer::onPurge(oa::oaDesign *design) { 
    if(design->getTopModule()) {
        oa::oaString moduleNameString;
        design->getTopModule()->getName(oa::oaNativeNS(), moduleNameString);
        DEBUG_PRINTLN("purge " << moduleNameString)
    } else {
        DEBUG_PRINTLN("purge ")
    }
    
    Manager::destroy(design);
}


// *****************************************************************************
// onPreSave()
//
/// This callback is invoked before a design is saved.
//
// *****************************************************************************
void    
Observer::onPreSave(oa::oaDesign *design) { 
    if(design->getTopModule()) {
        oa::oaString moduleNameString;
        design->getTopModule()->getName(oa::oaNativeNS(), moduleNameString);
        DEBUG_PRINTLN("presave " << moduleNameString)
    } else {
        DEBUG_PRINTLN("presave ")
    }
    
    if(!Manager::hasManager(design) || Manager::get(design)->isStructural()) {
        return;
    }
        
    int bytes = Manager::getSerializedSize(design);
    oa::oaByte *buf = new oa::oaByte[bytes];
    Manager::serialize(static_cast<void*>(buf), design);
    DEBUG_PRINTLN("\t" << "serialized design occupies " << bytes << " bytes")
    streamAppDef->set(design, bytes, buf);
    delete [] buf;
}


// *****************************************************************************
// onPostSave()
//
/// This callback is invoked after a design is saved.
//
// *****************************************************************************
void    
Observer::onPostSave(oa::oaDesign *design) { 
    if(design->getTopModule()) {
        oa::oaString moduleNameString;
        design->getTopModule()->getName(oa::oaNativeNS(), moduleNameString);
        DEBUG_PRINTLN("postsave " << moduleNameString)
    } else {
        DEBUG_PRINTLN("postsave ")
    }
    
    if(!Manager::hasManager(design)) {
        return;
    }

    streamAppDef->destroy(design);

    DEBUG_PRINTLN("\t" << "serialized info destroyed")
}

}
