/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-08-25: ChangeLog started
*/

#include "oagFunc.h"
#include "oagFuncModGraph.h"
#include <set>

// #define DEBUG

#include "oagFuncDebug.h"

namespace oagFunc {


// *****************************************************************************
// failIfInDifferentModules()
//
/// \brief Generates an error if the two modules are not identical.
///
/// \param x
/// \param y
//
// *****************************************************************************
inline void
ModGraph::failIfInDifferentModules(oa::oaModule *x, oa::oaModule *y) {
    if (x != y) {
        cerr << "ERROR: Expected ModRefs that refer to nodes in the same module" << endl;    
        QUIT_ON_ERROR;
    }
}     


// *****************************************************************************
// convertRefListToModRefList()
//
/// \brief Converts a list of Refs in a given module into a list of ModRefs.
///
/// \param source the list of original Refs
/// \param module the module
/// \param result the list of resulting ModRefs
//
// *****************************************************************************
void
ModGraph::convertRefListToModRefList(const list<oagAi::Ref> & source,
                                     oa::oaModule *module,
                                     list<ModRef> & result) {
    for(list<oagAi::Ref>::const_iterator it = source.begin(); it != source.end(); ++it) {
        result.push_back(ModRef(*it, module));
    }
}


// *****************************************************************************
// convertModRefListToRefList()
//
/// \brief Converts a list of ModRefs in a given module into a list of Refs.
///
/// \param source the list of original Refs
/// \param result the list of resulting ModRefs
//
// *****************************************************************************
oa::oaModule*
ModGraph::convertModRefListToRefList(const list<ModRef> & source,
                                     list<oagAi::Ref> & result) {
    if (source.empty()) return NULL;
    oa::oaModule *module = source.front().module;
    for(list<ModRef>::const_iterator it = source.begin(); it != source.end(); ++it) {
      assert(it->module == module);
      result.push_back(it->ref);
    }
    return module;
}


// *****************************************************************************
// convertRefVectorToModRefVector()
//
/// \brief Converts a vector of Refs in a given module into a vector of ModRefs.
///
/// \param source the vector of oagAI Refs
/// \param module the module
/// \param result the vector of resulting ModRefs
//
// *****************************************************************************
void
ModGraph::convertRefVectorToModRefVector(const vector<oagAi::Ref> & source,
                                         oa::oaModule *module,
                                         vector<ModRef> & result) {
  result.reserve(source.size());
  for(vector<oagAi::Ref>::const_iterator it = source.begin(); it != source.end(); ++it) {
    result.push_back(ModRef(*it, module));
  }
}


// *****************************************************************************
// newAnd()
//
/// \brief Returns a new AND node.
///
/// A new node will be created, even if it could be structually hashed with
/// an isomorphic equivalent.
///
/// \param x
/// \param y
/// \return a reference to a new AND node
//
// *****************************************************************************
ModRef
ModGraph::newAnd(ModRef x, ModRef y) {
    failIfInDifferentModules(x,y);
    
    return ModRef(getGraph(x)->newAnd(x.ref, y.ref), x.module);
}


// *****************************************************************************
// andOf()
///
/// \brief Returns the logic and of two functions.
///
/// A new node may or may not be created, depending on whether the resulting
/// function can be trivially reduced (i.e. one of the operands is constant)
/// or if the node is structurally isomorphic to an existing one and hashing
/// is enabled.
///
/// \param x
/// \param y
/// \return a reference to an AND node
//
// *****************************************************************************
ModRef
ModGraph::andOf(ModRef x, ModRef y) {
    failIfInDifferentModules(x,y);
    
    return ModRef(getGraph(x)->andOf(x.ref, y.ref), x.module);
}
    

// *****************************************************************************
// setTerminalDriver()
///
/// \brief Sets the driver of a given terminal.
///
/// The reference must be to a TERMINAL node.
/// The terminal and driver must be in the same module.
///
/// \param terminal
/// \param driver
//
// *****************************************************************************
void
ModGraph::setTerminalDriver(ModRef terminal, ModRef driver) {
    failIfInDifferentModules(terminal, driver);

    getGraph(terminal)->setTerminalDriver(terminal.ref, driver.ref);
}


// *****************************************************************************
// setAndLeft()
//
/// \brief Sets the left operand of a given AND node.
///
/// The reference must be to an AND node.
/// The AND reference and its operand must be in the same module.
///
/// \param x
/// \param left
//
// *****************************************************************************
void
ModGraph::setAndLeft(ModRef x, ModRef left) {
    failIfInDifferentModules(x,left);

    getGraph(x)->setAndLeft(x.ref, left.ref);
}


// *****************************************************************************
// setAndRight()
//
/// \brief Sets the right operand of a given AND node.
///
/// The reference must be to an AND node.
/// The AND reference and its operand must be in the same module.
///
/// \param x
/// \param right
//
// *****************************************************************************
void
ModGraph::setAndRight(ModRef x, ModRef right) {
    failIfInDifferentModules(x,right);

    getGraph(x)->setAndRight(x.ref, right.ref);
}


// *****************************************************************************
// setNextState()
//
/// \brief Sets the next state input of a given SEQUENTIAL node.
///
/// The reference must be to a SEQUENTIAL node.
/// The sequential reference and its new next state must be in the same module.
///
/// \param sequential
/// \param nextState
//
// *****************************************************************************
void
ModGraph::setNextState(ModRef sequential, ModRef nextState) {
    failIfInDifferentModules(sequential, nextState);

    getGraph(sequential)->setNextState(sequential.ref, nextState.ref);
}


// *****************************************************************************
// resubsitute()
//
/// \brief Resubstitutes one node for another.
///
/// This function takes all of the references to a node and 
/// replaces them with another.
///
/// References in sequential node control signals will not be replaced.
///
/// \param original the node to which the references will be replaced
/// \param replacement the new node to which the updated references will refer
//
// *****************************************************************************
void
ModGraph::resubstitute(ModRef original, ModRef replacement) {
    failIfInDifferentModules(original, replacement);

    getGraph(original)->resubstitute(original.ref, replacement.ref);
}


// *****************************************************************************
// resubsitute()
//
/// \brief Resubstitutes one node for another.
///
/// This function takes one reference to a node and replaces it with a 
/// reference to another.
///
/// References in sequential node control signals will not be replaced.
///
/// \param target the node whose reference will be changed
/// \param original the node to which the reference will be replaced
/// \param replacement the new node to which the updated reference will refer
//
// *****************************************************************************
void
ModGraph::resubstitute(ModRef original, ModRef replacement, ModRef target) {
    failIfInDifferentModules(original, replacement);
    failIfInDifferentModules(original, target);
    
    getGraph(original)->resubstitute(original.ref, replacement.ref, target.ref);
}


// *****************************************************************************
// detach()
//
/// \brief Detaches a node from its fan-in and fan-out.
///
/// The inputs of this node are reset to null, its fan-outs are cleared, and all
/// references to it are also reset to null.
///
/// \param x
//
// *****************************************************************************
void
ModGraph::detach(ModRef x) {

    getGraph(x)->detach(x.ref);
}


// *****************************************************************************
//  hasCombinationalCycle()
///
/// \brief Tests if there are any combinational cycles in a module's design.
///
/// All modules in the same design will be checked.
///
/// \param module
/// \return true, if one or more combinational cycles exist
//
// *****************************************************************************
bool
ModGraph::hasCombinationalCycle(oa::oaModule *module) {
    return getGraph(module)->hasCombinationalCycle();
}


// *****************************************************************************
// enumerateKfeasibleCuts()
///
/// \brief Returns the list of cuts in the transitive fan-in of a given node up to a given size.
///
/// Information about cut sets is cached between subsequent calls.  This cache
/// can be reset by clearKfeasibleCuts().
///
/// For further information, see oagAi::enumerateKfeasibleCuts()
///
/// \param x a graph node
/// \param maxCutSize the bound on the width of the cut
/// \param maxCutCount the maximum number of cuts to return
/// \param maxCutDepth the maximum levels backward to search
/// \param includeConstantNode should the constant node ever be included
/// \return the set of K-feasible for the graph node
//
// *****************************************************************************
ModGraph::CutSet
ModGraph::enumerateKfeasibleCuts(ModRef x, unsigned int maxCutSize, 
                                 int maxCutCount, int maxCutDepth,
                                 bool includeConstantNode) {
  oagAi::Graph::CutSet *source = getGraph(x)->enumerateKfeasibleCuts(x.ref, 
                                                                     maxCutSize, maxCutCount, maxCutDepth,
                                                                     includeConstantNode);

    // convert cuts in Refs to cuts in ModRefs
    CutSet result;
    for(oagAi::Graph::CutSet::iterator setIter = source->begin(); setIter != source->end(); ++setIter) {
        Cut    resultCut;
        for(oagAi::Graph::Cut::iterator cutIter = setIter->begin(); cutIter != setIter->end(); ++cutIter) {
            resultCut.insert(ModRef(*cutIter, x.module));
        }
        result.push_back(resultCut);
    }
    return result;
}


// *****************************************************************************
// clearKfeasibleCuts()
//
/// \brief Clears the cached sets of cuts in a given module.
///
/// \param module
//
// *****************************************************************************
void
ModGraph::clearKfeasibleCuts(oa::oaModule *module) {
    getGraph(module)->clearKfeasibleCuts();
}


// *****************************************************************************
// getFanout()
//
/// \brief Returns the fan-outs of a node that lie within the same module.
///
/// An error is generated if fanout is not being maintained.
///
/// \param x
/// \return the fanout list
//
// *****************************************************************************
list<ModRef>
ModGraph::getFanout(ModRef x) {
    list<ModRef> result;
    convertRefListToModRefList(getGraph(x)->getFanout(x.ref), x.module, result);
    return result;
}
    
    
// *****************************************************************************
// getFanout()
//
/// \brief Returns the fan-outs of a node that lie within the same module.
///
/// An error is generated if fanout is not being maintained.
///
/// \param x
/// \param result the resulting list of nodes in the fan-out
//
// *****************************************************************************
void
ModGraph::getFanout(ModRef x, list<ModRef> &result) {
    convertRefListToModRefList(getGraph(x)->getFanout(x.ref), x.module, result);
}


// *****************************************************************************
// getTransitiveFanin()
//
/// \brief Returns a list of all nodes in the transitive fan-in of one node.
///
/// The nodes in the transitive fan-in are appended to the provided list
/// in forward topological order.  The initial node is not included.  
/// All references will be non-inverted.
///
/// See oagAi::getTransitiveFanin() for more details.
///
/// \param x
/// \param transitiveFanin the resulting list of nodes in the transitive fan-in
/// \param includeRoots true if root nodes should be included
/// \param crossSequential true if the traversal should cross sequential nodes
//
// *****************************************************************************
void
ModGraph::getTransitiveFanin(ModRef x, list<ModRef> &transitiveFanin, 
                             bool includeRoots, bool crossSequential) {
    list<oagAi::Ref> flatResult;
    getGraph(x)->getTransitiveFanin(x.ref, flatResult, 
                                    includeRoots, crossSequential);
    convertRefListToModRefList(flatResult, x.module, transitiveFanin);
}


// *****************************************************************************
// getTransitiveFanout()
//
/// \brief Returns a list of all nodes in the transitive fan-out of one node.
///
/// The nodes in the transitive fan-out are appended to the provided list
/// in forward topological order.  The initial node is not included.  
/// All references will be non-inverted.
///
/// See oagAi::getTransitiveFanout() for more details.
///
/// \param x
/// \param transitiveFanout the resulting list of nodes in the transitive fan-out
/// \param includeRoots true if root nodes should be included
/// \param crossSequential true if the traversal should cross sequential nodes
//
// *****************************************************************************
void
ModGraph::getTransitiveFanout(ModRef x, list<ModRef> &transitiveFanout,
                              bool includeRoots, bool crossSequential) {
    list<oagAi::Ref> flatResult;
    getGraph(x)->getTransitiveFanout(x.ref, flatResult, 
                                     includeRoots, crossSequential);
    convertRefListToModRefList(flatResult, x.module, transitiveFanout);
}


// *****************************************************************************
// getTransitiveFanin()
//
/// \brief Returns a vector of all nodes in the transitive fan-in of one node.
///
/// The nodes in the transitive fan-in are appended to the provided vector
/// in forward topological order.  The initial node is not included.  
/// All references will be non-inverted.
///
/// See oagAi::getTransitiveFanin() for more details.
///
/// \param x
/// \param transitiveFanin the resulting vector of nodes in the transitive fan-in
/// \param includeRoots true if root nodes should be included
/// \param crossSequential true if the traversal should cross sequential nodes
//
// *****************************************************************************
void
ModGraph::getTransitiveFanin(ModRef x, vector<ModRef> &transitiveFanin,
                             bool includeRoots, bool crossSequential) {
    vector<oagAi::Ref> flatResult;
    getGraph(x)->getTransitiveFanin(x.ref, flatResult, 
                                    includeRoots, crossSequential);
    convertRefVectorToModRefVector(flatResult, x.module, transitiveFanin);
}


// *****************************************************************************
// getTransitiveFanout()
//
/// \brief Returns a vector of all nodes in the transitive fan-out of one node.
///
/// The nodes in the transitive fan-out are appended to the provided vector
/// in forward topological order.  The initial node is not included.  
/// All references will be non-inverted.
///
/// See oagAi::getTransitiveFanout() for more details.
///
/// \param x
/// \param transitiveFanout the resulting vector of nodes in the transitive fan-out
/// \param includeRoots true if root nodes should be included
/// \param crossSequential true if the traversal should cross sequential nodes
//
// *****************************************************************************
void
ModGraph::getTransitiveFanout(ModRef x, vector<ModRef> &transitiveFanout,
                              bool includeRoots, bool crossSequential) {
    vector<oagAi::Ref> flatResult;
    getGraph(x)->getTransitiveFanout(x.ref, flatResult, 
                                     includeRoots, crossSequential);
    convertRefVectorToModRefVector(flatResult, x.module, transitiveFanout);
}


// *****************************************************************************
// getTransitiveFanin()
//
/// \brief Returns a list of all nodes in the transitive fan-in of several nodes.
///
/// The nodes in the transitive fan-in are appended to the provided list
/// in forward topological order.  The initial node is not included.  
/// All references will be non-inverted.
///
/// See oagAi::getTransitiveFanin() for more details.
///
/// \param x
/// \param transitiveFanin the resulting list of nodes in the transitive fan-in
/// \param includeRoots true if root nodes should be included
/// \param crossSequential true if the traversal should cross sequential nodes
//
// *****************************************************************************
void
ModGraph::getTransitiveFanin(list<ModRef> x, list<ModRef> &transitiveFanin, 
                             bool includeRoots, bool crossSequential) {
    if (x.empty()) return;
    list<oagAi::Ref> flatInput;
    oa::oaModule *module = convertModRefListToRefList(x, flatInput);
    list<oagAi::Ref> flatResult;
    getGraph(module)->getTransitiveFanin(flatInput, flatResult, 
                                         includeRoots, crossSequential);
    convertRefListToModRefList(flatResult, module, transitiveFanin);
}


// *****************************************************************************
// getTransitiveFanout()
//
/// \brief Returns a list of all nodes in the transitive fan-out of several nodes.
///
/// The nodes in the transitive fan-out are appended to the provided list
/// in forward topological order.  The initial node is not included.  
/// All references will be non-inverted.
///
/// See oagAi::getTransitiveFanout() for more details.
///
/// \param x
/// \param transitiveFanout the resulting list of nodes in the transitive fan-out
/// \param includeRoots true if root nodes should be included
/// \param crossSequential true if the traversal should cross sequential nodes
//
// *****************************************************************************
void
ModGraph::getTransitiveFanout(list<ModRef> x, list<ModRef> &transitiveFanout,
                              bool includeRoots, bool crossSequential) {
    if (x.empty()) return;
    list<oagAi::Ref> flatInput;
    oa::oaModule *module = convertModRefListToRefList(x, flatInput);
    list<oagAi::Ref> flatResult;
    getGraph(module)->getTransitiveFanout(flatInput, flatResult, 
                                          includeRoots, crossSequential);
    convertRefListToModRefList(flatResult, module, transitiveFanout);
}


// *****************************************************************************
// getTransitiveFanin()
//
/// \brief Returns a vector of all nodes in the transitive fan-in of several nodes.
///
/// The nodes in the transitive fan-in are appended to the provided vector
/// in forward topological order.  The initial node is not included.  
/// All references will be non-inverted.
///
/// See oagAi::getTransitiveFanin() for more details.
///
/// \param x
/// \param transitiveFanin the resulting vector of nodes in the transitive fan-in
/// \param includeRoots true if root nodes should be included
/// \param crossSequential true if the traversal should cross sequential nodes
//
// *****************************************************************************
void
ModGraph::getTransitiveFanin(list<ModRef> x, vector<ModRef> &transitiveFanin,
                             bool includeRoots, bool crossSequential) {
    if (x.empty()) return;
    list<oagAi::Ref> flatInput;
    oa::oaModule *module = convertModRefListToRefList(x, flatInput);
    vector<oagAi::Ref> flatResult;
    getGraph(module)->getTransitiveFanin(flatInput, flatResult, 
                                         includeRoots, crossSequential);
    convertRefVectorToModRefVector(flatResult, module, transitiveFanin);
}


// *****************************************************************************
// getTransitiveFanout()
//
/// \brief Returns a vector of all nodes in the transitive fan-out of several nodes.
///
/// The nodes in the transitive fan-out are appended to the provided vector
/// in forward topological order.  The initial node is not included.  
/// All references will be non-inverted.
///
/// See oagAi::getTransitiveFanout() for more details.
///
/// \param x
/// \param transitiveFanout the resulting vector of nodes in the transitive fan-out
/// \param includeRoots true if root nodes should be included
/// \param crossSequential true if the traversal should cross sequential nodes
//
// *****************************************************************************
void
ModGraph::getTransitiveFanout(list<ModRef> x, vector<ModRef> &transitiveFanout,
                              bool includeRoots, bool crossSequential) {
    if (x.empty()) return;
    list<oagAi::Ref> flatInput;
    oa::oaModule *module = convertModRefListToRefList(x, flatInput);
    vector<oagAi::Ref> flatResult;
    getGraph(module)->getTransitiveFanout(flatInput, flatResult, 
                                          includeRoots, crossSequential);
    convertRefVectorToModRefVector(flatResult, module, transitiveFanout);
}


// *****************************************************************************
// getFaninRoots()
//
/// \brief Returns the roots of the transitive fan-in of a node.
///
/// \param x
/// \param faninRoots the resulting list of roots of the transitive fan-in
//
// *****************************************************************************
void
ModGraph::getFaninRoots(ModRef x, list<ModRef> &faninRoots) {
    list<oagAi::Ref> flatResult;
    getGraph(x)->getFaninRoots(x.ref, flatResult);
    
    convertRefListToModRefList(flatResult, x.module, faninRoots);
}


// *****************************************************************************
// getFanoutRoots()
//
/// \brief Returns the roots of the transitive fan-out of a node.
///
/// \param x
/// \param fanoutRoots the resulting list of roots of the transitive fan-out
//
// *****************************************************************************
void
ModGraph::getFanoutRoots(ModRef x, list<ModRef> &fanoutRoots) {
    list<oagAi::Ref> flatResult;
    getGraph(x)->getFanoutRoots(x.ref, flatResult);
    
    convertRefListToModRefList(flatResult, x.module, fanoutRoots);
}


// *****************************************************************************
// testEquivalence()
//
/// \brief Tests if two references are explicitly marked as equivalent.
///
/// \param x
/// \param y
//
// *****************************************************************************
bool
ModGraph::testEquivalence(ModRef x, ModRef y) {
    failIfInDifferentModules(x,y);

    return getGraph(x)->testEquivalence(x.ref, y.ref);
}


// *****************************************************************************
// setEquivalent()
//
/// \brief Marks two references as being functionally equivalent.
///
/// To prevent obfuscation, TERMINAL nodes can not be marked as equivalent.
//
// *****************************************************************************
void
ModGraph::setEquivalent(ModRef x, ModRef y) {
    failIfInDifferentModules(x,y);

    getGraph(x)->setEquivalent(x.ref, y.ref);
}


// *****************************************************************************
// getEquivalents()
//
/// \brief Returns all references that have been marked equivalent.
///
/// The set of equivalent references will be appended to the provided list.
/// The list will include the reference provided.
///
/// \param x
/// \param result
//
// *****************************************************************************
void
ModGraph::getEquivalents(ModRef x, list<ModRef> & result) {
    list<oagAi::Ref> flatResult;
    getGraph(x)->getEquivalents(x.ref, flatResult);
    
    convertRefListToModRefList(flatResult, x.module, result);
}


// *****************************************************************************
// getFanoutOfEquivalentNodes()
//
/// \brief Returns the fan-out of a node and all of its equivalents.
///
/// \param x
/// \param result the fan-out of a node and all of its equivalents
//
// *****************************************************************************
void
ModGraph::getFanoutOfEquivalentNodes(ModRef x, list<ModRef> &result) {
    list<oagAi::Ref> flatResult;
    getGraph(x)->getFanoutOfEquivalentNodes(x.ref, flatResult);
    
    convertRefListToModRefList(flatResult, x.module, result);
}


// *****************************************************************************
// prepareNetToAiConnection()
//
/// \brief Creates an terminal node to represent the function of a net.
///
/// If an terminal node is already associated with the net, then the existing
/// one is returned.
///
/// If a new terminal node is created, it will implement a null function.
///
/// \param net
/// \return the terminal node attached to this net
//
// *****************************************************************************
ModRef   
ModGraph::prepareNetToAiConnection(oa::oaModBitNet *net) {
    Manager *manager = Manager::get(net->getDesign());
    return ModRef(manager->prepareNetToAiConnection(net), net->getModule());
}


// *****************************************************************************
// getNetToAiConnection()
//
/// \brief Returns the OpenAccess net connected to a given graph TERMINAL.
///
/// \param x
/// \return the module net, or NULL if there is none
//
// *****************************************************************************
oa::oaModBitNet *   
ModGraph::getNetToAiConnection(ModRef x) {
    Manager *manager = Manager::get(x.module->getDesign());
    return manager->getNetToAiConnection(x.ref);
}


// *****************************************************************************
// getNetToAiConnection()
//
/// \brief Returns the graph node connected to a given OpenAccess net.
///
/// \param net
/// \return a reference to the connected graph TERMINAL node, or a null ModRef if none exists
//
// *****************************************************************************
ModRef
ModGraph::getNetToAiConnection(oa::oaModBitNet *net) {
    Manager *manager = Manager::get(net->getDesign());
    return ModRef(manager->getNetToAiConnection(net), net->getModule());
}


// *****************************************************************************
// setNetToAiConnection()
//
/// \brief Creates a logical connection between an OpenAccess net and a graph TERMINAL node.
///
/// \param net
/// \param x
//
// *****************************************************************************
void
ModGraph::setNetToAiConnection(oa::oaModBitNet *net, ModRef x) {
    failIfInDifferentModules(net->getModule(), x.module);

    Manager *manager = Manager::get(x.module->getDesign());
    manager->setNetToAiConnection(net, x.ref);
}


// *****************************************************************************
// removeNetToAiConnection()
//
/// \brief Destroys a logical connection between an OpenAccess net and a graph TERMINAL node.
///
/// \param net
//
// *****************************************************************************
void
ModGraph::removeNetToAiConnection(oa::oaModBitNet *net) {
    Manager *manager = Manager::get(net->getDesign());
    manager->removeNetToAiConnection(net);
}


// *****************************************************************************
// getOutputs()
//
/// \brief Gets all output terminals inside a module.
///
/// An output terminal is a TERMINAL node that is connected to an output oaModTerm.
///
/// If the bit net attached to the oaModTerm does not have a connection to the
/// graph, one is created.
///
/// \param module
/// \param result the list of all output terminals inside the module
//
// *****************************************************************************
void         
ModGraph::getOutputs(oa::oaModule *module, list<ModRef> &result) {
    assert(module);
    oa::oaModTerm *term;
    oa::oaIter<oa::oaModTerm> termIter(module->getTerms(oacTermIterSingleBit));
    while((term = termIter.getNext())) {
        if(term->getTermType() == oa::oacOutputTermType) {
            result.push_back(prepareNetToAiConnection(toBitNet(term->getNet())));
        }
    }
}


// *****************************************************************************
// getInputs()
//
/// \brief Gets all input terminals inside a module.
///
/// An input terminal is a TERMINAL node that is connected to an input oaModTerm.
///
/// If the bit net attached to the oaModTerm does not have a connection to the
/// graph, one is created.
///
/// \param module
/// \param result the list of all input terminals inside the module
//
// *****************************************************************************
void         
ModGraph::getInputs(oa::oaModule *module, list<ModRef> &result) {
    assert(module);
    oa::oaModTerm *term;
    oa::oaIter<oa::oaModTerm> termIter(module->getTerms(oacTermIterSingleBit));
    while((term = termIter.getNext())) {
        if(term->getTermType() == oa::oacInputTermType) {
            result.push_back(prepareNetToAiConnection(toBitNet(term->getNet())));
        }
    }
}


// *****************************************************************************
// getAllNodes()
//
/// \brief Gets all nodes inside of a module.
///
/// \param module
/// \param result the list of all nodes inside the module
//
// *****************************************************************************
void         
ModGraph::getAllNodes(oa::oaModule *module, list<ModRef> &result) {
    assert(module);

    // add the state bits inside this object
    list<oagAi::Ref> flatResult;
    getGraph(module)->getAll(flatResult);
    convertRefListToModRefList(flatResult, module, result);

}


// *****************************************************************************
// getLocalStates()
//
/// \brief Gets all state bits inside a module.
///
/// \param module
/// \param result the list of all state bits inside the module
//
// *****************************************************************************
void         
ModGraph::getLocalStates(oa::oaModule *module, list<ModRef> &result) {
    assert(module);

    // add the state bits inside this object
    list<oagAi::Ref> flatResult;
    getGraph(module)->getAllSequential(flatResult);
    convertRefListToModRefList(flatResult, module, result);
}


// *****************************************************************************
// getAllConnections()
//
/// \brief Returns the set of nets and refs that are logically connected.
///
/// \param net
/// \param connectedNets the resulting list of connected nets
/// \param connectedRefs the resulting list of connected refs
/// \param searchForwardThroughGraph follow TERMINAL fan-out to functionally identical refs
/// \param searchBackwardThroughGraph follow TERMINAL drivers to functionally identical refs
/// \param searchThroughEquivNets search all nets that are marked equivalent in the database
/// \param searchThroughEquivRefs follow all Refs that are marked equivalent in the graph
/// \param includeSelf include the starting node
//
// *****************************************************************************
void
ModGraph::getAllConnections(oa::oaModBitNet *net, 
                            set<oa::oaModBitNet*> &connectedNets,
                            set<ModRef> &connectedRefs,
                            bool searchForwardThroughGraph,
                            bool searchBackwardThroughGraph,
                            bool searchThroughEquivNets,
                            bool searchThroughEquivRefs,
                            bool includeSelf) {
    assert(net);
    
    // has this net been visited?
    if (connectedNets.find(net) == connectedNets.end()) {
        connectedNets.insert(net);
    } else {
        return;
    }

    // --- directly connected refs
    getAllConnections(getNetToAiConnection(net), connectedNets, connectedRefs,
                searchForwardThroughGraph, searchBackwardThroughGraph,
                searchThroughEquivNets, searchThroughEquivRefs, true);
    
    // --- equivalent nets (if searchThroughEquivNets)
    if (searchThroughEquivNets) {
        oa::oaModBitNet *equivNet;
        oa::oaIter<oa::oaModBitNet> equivIter(net->getEquivalentNets());
        while((equivNet = equivIter.getNext())) {
            getAllConnections(equivNet, connectedNets, connectedRefs,
                searchForwardThroughGraph, searchBackwardThroughGraph,
                searchThroughEquivNets, searchThroughEquivRefs, true);
        }
    }
    
    // -- exclude self
    if (!includeSelf) {
        connectedNets.erase(net);
    }
}


// *****************************************************************************
// getAllConnections()
//
/// \brief Returns the set of nets and refs that are logically connected.
///
/// \param x

/// \param net
/// \param connectedNets the resulting list of connected nets
/// \param connectedRefs the resulting list of connected refs
/// \param searchForwardThroughGraph follow TERMINAL fan-out to functionally identical refs
/// \param searchBackwardThroughGraph follow TERMINAL drivers to functionally identical refs
/// \param searchThroughEquivNets search all nets that are marked equivalent in the database
/// \param searchThroughEquivRefs follow all Refs that are marked equivalent in the graph
/// \param includeSelf include the starting node
//
// *****************************************************************************    
void 
ModGraph::getAllConnections(ModRef x, 
                            set<oa::oaModBitNet*> &connectedNets,
                            set<ModRef> &connectedRefs,
                            bool searchForwardThroughGraph,
                            bool searchBackwardThroughGraph,
                            bool searchThroughEquivNets,
                            bool searchThroughEquivRefs,
                            bool includeSelf) {
    if (isNull(x)) {
        return;
    }

    // has this ref been visited?
    if (connectedRefs.find(x) == connectedRefs.end()) {
        connectedRefs.insert(x);
    } else {
        return;
    }
    
    // --- directly connected nets
    if (isTerminal(x) && !isInverted(x)) {
        oa::oaModBitNet *net;
        if ((net = getNetToAiConnection(x))) {
            getAllConnections(net, connectedNets, connectedRefs,
                searchForwardThroughGraph, searchBackwardThroughGraph,
                searchThroughEquivNets, searchThroughEquivRefs, true);
        }
    }

    // --- connected backward through graph (if searchBackwardThroughGraph)
    if (searchBackwardThroughGraph) {
        // terminal drivers
        if (isTerminal(x)) {
            ModRef driver = getTerminalDriver(x);
            driver = isInverted(x) ? notOf(driver) : driver;
            getAllConnections(driver, connectedNets, connectedRefs,
                searchForwardThroughGraph, searchBackwardThroughGraph,
                searchThroughEquivNets, searchThroughEquivRefs, true);
        }
    }

    // --- connected forward through graph (if searchForwardThroughGraph)
    if (searchForwardThroughGraph) {
        // terminals in fanout
        list<ModRef> fanoutList;
        getFanout(x, fanoutList);
        while(!fanoutList.empty()) {
            ModRef fanout(fanoutList.front());
            fanoutList.pop_front();
            if(isTerminal(fanout)) {
                fanout = isInverted(x) ? notOf(fanout) : fanout;
                fanout = isInverted(getTerminalDriver(fanout)) ? notOf(fanout) : fanout;
                getAllConnections(fanout, connectedNets, connectedRefs,
                    searchForwardThroughGraph, searchBackwardThroughGraph,
                    searchThroughEquivNets, searchThroughEquivRefs, true);
            }
        }
    }

    // --- equivalent nodes (if searchThroughEquivRefs)
    if (searchThroughEquivRefs) {
        list<ModRef> equivList;
        getEquivalents(x, equivList);
        while(!equivList.empty()) {
            ModRef equiv(equivList.front());
            equivList.pop_front();
            getAllConnections(equiv, connectedNets, connectedRefs,
                searchForwardThroughGraph, searchBackwardThroughGraph,
                searchThroughEquivNets, searchThroughEquivRefs, true);
        }
    }
    
    // -- exclude self
    if (!includeSelf) {
        connectedRefs.erase(x);
    }
}


// *****************************************************************************
// findDriverOfEquivalentNets()
//
/// \brief Returns the equivalent net that is driving a net.
///
/// The driving net is the first equivalent net that satisfies one of these conditions:
/// 1. connected to an input Term
/// 2. connected to a TERMINAL node that is not driven by NULL
///
/// In the case where the given net is itself the driver for the group, NULL 
/// be returned.
///
/// \param net
/// \return driver driving net
//
// *****************************************************************************
oa::oaModBitNet *
ModGraph::findDriverOfEquivalentNets(oa::oaModBitNet *net) {
    assert(net);

    // ignore if there are no equivalent nets
    if (net->getEquivalentNets().isEmpty()) {
      return NULL;
    }

    // ignore if there is already a driver in the graph
    ModRef termRef = getNetToAiConnection(net);
    if (!isNull(termRef) && !isNull(getTerminalDriver(termRef))) {    
      return NULL;
    }

    // iterate over equivalent nets to find driver
    oa::oaModBitNet *equivNet;
    oa::oaIter<oa::oaModBitNet> equivNetIter(net->getEquivalentNets());

    DEBUG_PRINT("Equivalent set : ");
    while ((equivNet = equivNetIter.getNext())) {
      assert(equivNet);
      
      oa::oaString netString;
      equivNet->getName(oa::oaVerilogNS(), netString);
      DEBUG_PRINTMORE(netString << " ");

      // is this driven in the graph?
      ModRef termRef = getNetToAiConnection(equivNet);
      if (!isNull(termRef) && !isNull(getTerminalDriver(termRef))) {
        // avoid combinational cycles
        ModRef driverRef = termRef;
        while(true) {
          oa::oaModBitNet *net2 = getNetToAiConnection(driverRef);
          if (net2 && net2 == net) {
            // net can't be in a cycle with itself
            return NULL;
          }
          driverRef = getTerminalDriver(driverRef);
          if (isNull(driverRef) || !isTerminal(driverRef)) {
            break;
          }
        }
        return equivNet;
      }
      
      // is an input terminal attached?
      oa::oaModTerm *term;
      oa::oaIter<oa::oaModTerm> termIter(equivNet->getTerms(oacTermIterSingleBit));
      while((term = termIter.getNext())) {
        assert(term);
        if(term->getTermType() == oa::oacInputTermType) {
          return equivNet;
        }
      }
    }
    DEBUG_PRINTMORE("\n");
    
    // is there a preferred equivalent?
    /*
    // NOTE: This appears to be broken.  A preferred net is sometimes
    // returned when none has been set.  Reason unknown.

    if (net->getPreferredEquivalent() != net) {
      return net->getPreferredEquivalent();
    }
    */

    return NULL;
}


// *****************************************************************************
// connectEquivalentNetsInGraph()
//
/// \brief Connects the TERMINAL nodes of equivalent nets.
///
/// Functionally identical nets can also be marked
/// equivalent in the OpenAccess database itself.  The semantics of this
/// are unfortunately
/// very unclear as to whether this means either (1) the two nets already
/// both implement the same functionality or (2) the nets don't already
/// implement the same functionality but they should (for example by
/// physical connection).
///
/// For example, the verilog2oa imports continuous assignments by marking the
/// nets as equivalent.  In Verilog, this construct implies (2).  In such
/// cases, it may
/// be convenient and/or necessary to reflect this functional dependency
/// in the graph, this routine provides the means to do so.
///
/// For a group of nets that are marked equivalent, the one that is being
/// driven (as an input or in the graph) is first discovered.  The TERMINAL
/// node of this net is then set as the driver as the TERMINAL nodes of the
/// other nets.
///
/// If no driver is found, the  nets are left as is.
///
/// If there are multiple equivalent nets that are being driven, the existing
/// drivers will not be modified.  This should only be the case if the
/// equivalent nets feature in the OpenAccess database is being
/// used in the manner of (1).  Because only the undriven nets are modified,
/// this should be compatible with such as use case as well.
///
/// \param module
// *****************************************************************************
void
ModGraph::connectEquivalentNetsInGraph(oa::oaModule *module) {
  assert(module);

  oa::oaModNet *net;
  oa::oaIter<oa::oaModNet> netIter(module->getNets(oacNetIterSingleBit));
  while((net = netIter.getNext())) {
    oa::oaModBitNet *bitNet = toBitNet(net);
    assert(bitNet);
    ModRef termNode = getNetToAiConnection(bitNet);
    assert(!isNull(termNode));
    if (!isNull(getTerminalDriver(termNode))) {
      continue;
    }

    oa::oaModBitNet *driverNet = findDriverOfEquivalentNets(bitNet);

    if (driverNet) {      
      ModRef driver = getNetToAiConnection(driverNet);
#if defined(DEBUG)
      cout << "Moving equivalent nets to graph : node ";
      termNode.printName(false);
      cout << " was ";
      getTerminalDriver(termNode).printName();
      cout << " is now driven by ";
      driver.printName(true);
      cout << endl;
#endif
      setTerminalDriver(termNode, driver);
    }
  }
}


// *****************************************************************************
// print()
//
/// \brief Prints entire graph.
//
// *****************************************************************************
void                
ModGraph::print(oa::oaModule *module) {
  assert(module);

  list<ModRef> all;
  getAllNodes(module, all);
  for(list<ModRef>::iterator it = all.begin(); it!=all.end(); it++) {
    it->print();
  }
}


// *****************************************************************************
// print()
//
/// \brief Prints full information about a ModRef.
///
/// \param lineFeed
//
// *****************************************************************************
void                
ModRef::print(bool lineFeed) {
    const int TYPE_COL_WIDTH = 4;
    const int MODULE_COL_WIDTH = 20;
    const int REF_COL_WIDTH = 5;

    oa::oaString modString;
    if (!module) {
        modString = "(null)";
    } else {
        module->getName(oa::oaVerilogNS(), modString);
    }

    if (ModGraph::getNodeType(*this) == oagAi::Node::AND) {
        cout.width(TYPE_COL_WIDTH);
        cout << "AND ";
        cout.width(MODULE_COL_WIDTH);
        cout << modString << ".";
        cout.width(REF_COL_WIDTH);
        cout << ref << " :\t";
        cout << ModGraph::getAndLeft(*this).ref << ", " << ModGraph::getAndRight(*this).ref;
    } else if (ModGraph::getNodeType(*this) == oagAi::Node::SEQUENTIAL) {
        cout.width(TYPE_COL_WIDTH);
        cout << "SEQ ";
        cout.width(MODULE_COL_WIDTH);
        cout << modString << ".";
        cout.width(REF_COL_WIDTH);
        cout << ref << " :\t";
        cout << ModGraph::getNextState(*this).ref;
    } else if (ModGraph::getNodeType(*this) == oagAi::Node::CONSTANT0) {
        cout.width(TYPE_COL_WIDTH);
        cout << "ZRO ";
        cout.width(MODULE_COL_WIDTH);
        cout << modString << ".";
        cout.width(REF_COL_WIDTH);
        cout << ref << " :\t";
    } else if (ModGraph::getNodeType(*this) == oagAi::Node::TERMINAL) {
        cout.width(TYPE_COL_WIDTH);
        cout << "TRM ";
        cout.width(MODULE_COL_WIDTH);
        cout << modString << ".";
        cout.width(REF_COL_WIDTH);
        cout << ref << " :\t";
        cout << ModGraph::getTerminalDriver(*this).ref;
        cout << " net= ";
        oa::oaModBitNet *net = ModGraph::getNetToAiConnection(*this);
        if (net) {
          oa::oaString netString;
          net->getName(oa::oaVerilogNS(), netString);
          cout << netString;
        } else {
          cout << "*none*";
        }
    } else if (ModGraph::getNodeType(*this) == oagAi::Node::NONE) {
        cout.width(TYPE_COL_WIDTH);
        cout << "NUL ";
        cout.width(MODULE_COL_WIDTH);
        cout << modString << ".";
        cout.width(REF_COL_WIDTH);
        cout << ref << " :\t";
    }

    if (ModGraph::isInverted(*this)) {
        cout << " (inverted)";
    }
    
    if (lineFeed) {
        cout << endl;
    }
}


// *****************************************************************************
// printName()
//
/// \brief Prints concise information about a ModRef.
///
/// \param lineFeed
//
// *****************************************************************************
void                
ModRef::printName(bool lineFeed) {

    oa::oaString modString;
    if (!module) {
        modString = "(null)";
    } else {
        module->getName(oa::oaVerilogNS(), modString);
    }

    cout << modString << "." << ref;
    
    if (lineFeed) {
        cout << endl;
    }
}


// *****************************************************************************
// getFaninCone()
//
/// \brief Returns a list of the nodes between a specified cut and a node.
///
/// The nodes in the transitive fan-in are appended to the provided list
/// in forward topological order.  The initial node is not included.  
/// All references will be non-inverted.
///
/// See oagAi::getFaninCone() for more details.
///
/// \param x
/// \param coneRoots the backward boundary of the cone
/// \param transitiveFanin the resulting list of nodes in the transitive fan-in
/// \param includeRoots true, if cut and/or root SEQUENTIAL, TERMINAL, and CONSTANT nodes should be included
//
// *****************************************************************************
void
ModGraph::getFaninCone(ModRef x, const list<ModRef> &coneRoots, 
                       list<ModRef> &transitiveFanin, bool includeRoots) {

  list<oagAi::Ref> flatCut;
  oa::oaModule *module = convertModRefListToRefList(coneRoots, flatCut);
  assert(coneRoots.empty() || module == x.module);
  list<oagAi::Ref> flatResult;
  getGraph(x.module)->getFaninCone(x.ref, flatCut, flatResult, includeRoots);
  convertRefListToModRefList(flatResult, x.module, transitiveFanin);
}


// *****************************************************************************
// getFanoutCone()
//
/// \brief Returns a list of the nodes between a node and a specified cut.
///
/// The nodes in the transitive fan-out are appended to the provided list
/// in forward topological order.  The initial node is not included.  
/// All references will be non-inverted.
///
/// See oagAi::getFanoutCone() for more details.
///
/// \param x
/// \param coneRoots the forward boundary of the cone
/// \param transitiveFanout the resulting list of nodes in the transitive fan-out
/// \param includeRoots true, if cut and/or root SEQUENTIAL, TERMINAL, and CONSTANT nodes should be included
//
// *****************************************************************************
void
ModGraph::getFanoutCone(ModRef x, const list<ModRef> &coneRoots, 
                        list<ModRef> &transitiveFanout, bool includeRoots) {
  
  list<oagAi::Ref> flatCut;
  oa::oaModule *module = convertModRefListToRefList(coneRoots, flatCut);
  assert(module == x.module);
  list<oagAi::Ref> flatResult;
  getGraph(x.module)->getFanoutCone(x.ref, flatCut, flatResult, includeRoots);
  convertRefListToModRefList(flatResult, x.module, transitiveFanout);
}

}
