/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-05-31: ChangeLog started
*/

#include "oagFunc.h"
#include "oagFuncManager.h"
#include "oagAiGraph.h"
#include <list>
#include <map>
#include <assert.h>
#include <iostream>

#include "oagFuncDebug.h"

namespace oagFunc {
 

// *****************************************************************************
// Class static variable definitions.
// *****************************************************************************

#if defined(CACHE_LAST_MANAGER_LOOKUP)
  oa::oaDesign *Manager::lastManagerDesign;
  Manager      *Manager::lastManagerObject;
#endif


// *****************************************************************************
// Manager() constructor
///
/// Constructs a Manager for the given design.  If it already has a
/// Manager, an error is generated.  
///
/// All information about the function behavior of a design is stored and 
/// managed by this class.  A Manager is required to describe any functional
/// relationships between OpenAccess objects other than simple equivalence.
/// Conversely, objects that are purely structural in nature (that is, do
/// not have any complex behavior, other than simple connections) do not
/// require a Manager.
///
/// \param design
//
// *****************************************************************************
Manager::Manager(oa::oaDesign *design) {

    // does this design already have a Manager associated with it?
    if(hasManager(design)) {
        cerr << "ERROR: Manager for design already exists" << endl;
        QUIT_ON_ERROR;
    }
    
    // initialize variables and caches
    this->design = design;
    preflattenedCache.clear();

    // associate this manager with the design by setting an AppDef
    managerAppDef->set(design, this);
    
    // design must have a top module
    oa::oaModule *module;
    if (!(module = design->getTopModule())) {
        module = oa::oaModule::create(design);
        design->setTopModule(module);
    }
     
}
 

// *****************************************************************************
// create()
//
/// \brief Returns the preexisting Manager for the given design or creates a new one.
//
// *****************************************************************************
Manager *
Manager::create(oa::oaDesign *design) {

    if (hasManager(design)) {
       return static_cast<Manager*>(managerAppDef->get(design));
    }
    
    // there is no Manager yet associated with this design
    Manager *manager = new Manager(design);
    
    // create an terminal AI node for each net in each module
    oa::oaModule *module;
    oa::oaIter<oa::oaModule> modIter(design->getModules());
    while((module = modIter.getNext())) { 
      oa::oaModBitNet *net;
      oa::oaIter<oa::oaModBitNet> netIter(module->getNets(oacNetIterAll));
      while((net = netIter.getNext())) {
        if(net->getNumBits() > 1) {
          continue;
        }
        
        // clear an existing AppDef, if it exists
        AiRefAppDef->destroy(net);
        
        // create a terminal to connect the net to the graph
        manager->prepareNetToAiConnection(net);
      }

      // tie the 0 and 1 nets appropriately
      oa::oaNativeNS      nativeNS;
      oa::oaModScalarNet *logicConstantNet;
      logicConstantNet = oa::oaModScalarNet::find(module, oa::oaScalarName(nativeNS, "tie0"));
      if (logicConstantNet) {
        manager->ai.setTerminalDriver(AiRefAppDef->get(logicConstantNet), manager->ai.constantZero());
      }
      logicConstantNet = oa::oaModScalarNet::find(module, oa::oaScalarName(nativeNS, "tie1"));
      if (logicConstantNet) {
        manager->ai.setTerminalDriver(AiRefAppDef->get(logicConstantNet), manager->ai.constantOne());
      }
    }
    
    return manager;
}


// *****************************************************************************
// destory()
//
/// \brief Destroys a manager object and the functional description of a design.
//
// *****************************************************************************
void
Manager::destroy(oa::oaDesign *design) {
  // is there a Manager?
  if (!hasManager(design)) {
    return;
  }
    
  // remove links from nets to AI graph
  oa::oaModule *module;
  oa::oaIter<oa::oaModule> modIter(design->getModules());
  while((module = modIter.getNext())) {   
    oa::oaModBitNet *net;
    oa::oaIter<oa::oaModBitNet> netIter(module->getNets(oacNetIterAll));
    while((net = netIter.getNext())) {
      if(net->getNumBits() > 1) {
        continue;
      }
      
      // clear an existing AppDef, if it exists
      AiRefAppDef->destroy(net);
    }
  }
  
  // delete manager object
  delete static_cast<Manager*>(managerAppDef->get(design));
  managerAppDef->destroy(design);
}


// *****************************************************************************
// ~Manager() destructor
///
/// \brief Destructor for Manager object.
///
/// The Manager should only be deleted from a destroy() call.
//
// *****************************************************************************
Manager::~Manager() {
}


// *****************************************************************************
// prepareNetToAiConnection()
//
/// \brief Creates an terminal node to represent the function of a net.
///
/// If an terminal node is already associated with the net, then the existing
/// one is returned.
///
/// If a new terminal node is created, it will implement a null function.
///
/// \param net
/// \return the terminal node attached to this net
//
// *****************************************************************************
oagAi::Ref
Manager::prepareNetToAiConnection(oa::oaModBitNet *net) {
    assert(net);
    
    // is there already an terminal node associated with this net?
    if (net->getAppDefs().includes(AiRefAppDef)) {
        oagAi::Ref terminal = AiRefAppDef->get(net);  
        assert(ai.isTerminal(terminal));
        return terminal;
    }

    // create a new terminal node
    oagAi::Ref terminal = ai.newTerminal(ai.getNull());
    ai.incrementExternalReferences(terminal);
    setNetToAiConnection(net, terminal);
    return terminal;
}


// *****************************************************************************
// getSerializedSize()
//
/// \brief Computes the number of bytes necessary for the serialized representation.
///
/// \return the number of bytes
//
// *****************************************************************************
int         
Manager::getSerializedSize(oa::oaDesign *design) {
    assert(design);

    if (!Manager::hasManager(design)) {
        return 0;
    }

    Manager *manager = Manager::get(design);

    int bytes = 0;
    
    long numNodes = manager->ai.getNumNodes() - manager->ai.NODES_TO_IGNORE;
    bytes += sizeof(numNodes);

    while(--numNodes >= 0) {
        oagAi::Node *node = manager->ai.getNode((numNodes+manager->ai.NODES_TO_IGNORE)*2);
        bytes += node->getSerializedSize();
    }
    
    return bytes;
}


// *****************************************************************************
// serialize()
//
/// \brief Serializes the functional description of a design.
///
/// \param buf the pre-allocated buffer into which the serialized data is placed
/// \param design the design to serialize.
//
// *****************************************************************************
void         
Manager::serialize(void* buf, oa::oaDesign *design) {
    assert(buf);
    assert(design);
    assert(design->getTopModule());
    
    oa::oaString moduleNameString;
    design->getTopModule()->getName(oa::oaVerilogNS(), moduleNameString);
    DEBUG_PRINTLN("serializing " << moduleNameString)
    
    // is there any functional information to serialize?
    if (!Manager::hasManager(design)) {
        return;
    }
    
    Manager *manager = Manager::get(design);
    
    void *ptr = buf;
    
    long numNodes = manager->ai.getNumNodes() - manager->ai.NODES_TO_IGNORE;
    memcpy(ptr, &numNodes, sizeof(numNodes));
    char *char_ptr = static_cast<char*>(ptr);
    char_ptr += sizeof(numNodes);
    ptr = char_ptr;
    DEBUG_PRINTLN("\twriting " << numNodes << " AI nodes")
    
    oagAi::Ref currentRef = manager->ai.NODES_TO_IGNORE*2;
    while(--numNodes >= 0) {
        oagAi::Node *node = manager->ai.getNode(currentRef);
        node->serialize(ptr);
        char *char_ptr = static_cast<char*>(ptr);
        char_ptr += node->getSerializedSize();
        ptr = char_ptr;
        currentRef += 2;
    }

#ifdef DEBUG
    long bytes = reinterpret_cast<long>(ptr)-reinterpret_cast<long>(buf);
    DEBUG_PRINTLN("\twrote " << bytes << " bytes");
#endif
}


// *****************************************************************************
// unserialize()
//
/// \brief Unserializes the functional description of a design.
///
/// \param buf a pointer to a buffer containing a serialized functional description.
/// \param design the design to annotate with this functional description.
/// \return the number of bytes read from the buffer
//
// *****************************************************************************
int
Manager::unserialize(void *buf, oa::oaDesign *design) {
    assert(design);
    assert(design->getTopModule());
    
    oa::oaString moduleNameString;
    design->getTopModule()->getName(oa::oaVerilogNS(), moduleNameString);
    DEBUG_PRINTLN("unserializing " << moduleNameString)

    if (Manager::hasManager(design)) {
        QUIT_ON_INTERNAL_ERROR;
    }
    Manager *manager = new Manager(design);
    void *ptr = buf;
    
    long numNodes;
    memcpy(&numNodes, ptr, sizeof(numNodes));
    char *char_ptr = static_cast<char*>(ptr);
    char_ptr += sizeof(numNodes);    
    ptr = char_ptr;
    DEBUG_PRINTLN("\treading " << numNodes << " AI nodes")

    // recreate all nodes
    assert(manager->ai.getNumNodes() == manager->ai.NODES_TO_IGNORE);
    while(--numNodes >= 0) {
        oagAi::Node *node = manager->ai.newNode();
        char *char_ptr = static_cast<char*>(ptr);
        char_ptr += node->unserialize(ptr);
        ptr = char_ptr;
    }
    // rebuild all non-persistent data
    oagAi::Ref currentRef = manager->ai.NODES_TO_IGNORE*2;
    oagAi::Ref lastRef = (manager->ai.getNumNodes()-1)*2;
    while(currentRef <= lastRef) {
        switch(manager->ai.getNodeType(currentRef)) {
          case oagAi::Node::TERMINAL: {
            // update fan-outs and reference counts
            manager->ai.addToFanout(manager->ai.getTerminalDriver(currentRef), currentRef);
            break;
            }
          case oagAi::Node::SEQUENTIAL: {
            // update fan-outs and reference counts
            manager->ai.addToFanout(manager->ai.getNextState(currentRef), currentRef);
            manager->ai.seqNodes++;
            break;
            }
          case oagAi::Node::AND: {
            // update fan-outs and reference counts          
            manager->ai.addToFanout(manager->ai.getAndLeft(currentRef), currentRef);
            manager->ai.addToFanout(manager->ai.getAndRight(currentRef), currentRef);            
    
            // add node to structural hash table
            int hash_key = manager->ai.getStructuralHashKey(manager->ai.getAndLeft(currentRef),manager->ai.getAndLeft(currentRef));
            manager->ai.getNode(currentRef)->next = manager->ai.structuralHash[hash_key];
            manager->ai.structuralHash[hash_key] = currentRef;

            manager->ai.andNodes++;
            break; 
            }
          default:
            break;
        }
        currentRef += 2;
    }

    DEBUG_PRINTLN("\t\tfinal num nodes =  " << manager->ai.getNumNodes())
    DEBUG_PRINTLN("\t\t\tlastPage = " << manager->ai.lastPage << " lastIndex = " << manager->ai.lastIndex)

    // 1. increment refcounts of nodes that are attached to nets
    // 2. set external_terminal_connection pointers to nets

    oa::oaModule *module;
    oa::oaIter<oa::oaModule> modIter(design->getModules());
    while((module = modIter.getNext())) {      
        oa::oaModNet *net;
        oa::oaIter<oa::oaModNet> netIter(module->getNets(oacNetIterSingleBit));
        while((net = netIter.getNext())) {
            oa::oaModBitNet *bit = toBitNet(net);
            if (!manager->ai.isNull(currentRef = manager->getNetToAiConnection(bit))) {
                manager->ai.incrementExternalReferences(currentRef);
                manager->ai.setExternalTerminalConnection(currentRef, reinterpret_cast<void *>(bit));
            }
        }
    }
    
    long bytes = reinterpret_cast<long>(ptr)-reinterpret_cast<long>(buf);
    DEBUG_PRINTLN("\tread " << bytes << " bytes");
    return bytes;
}


// *****************************************************************************
// removeNetToAiConnection()
//
/// \brief Disconnections a net from the Ai graph.
///
/// \param net a net
//
// *****************************************************************************
void
Manager::removeNetToAiConnection(oa::oaModBitNet *net) {
    assert(net);

    // remove the pointer from the node to the net    
    if (net->getAppDefs().includes(AiRefAppDef)) {
        oagAi::Ref ref = AiRefAppDef->get(net);
        assert(!ai.isNull(ref));
        ai.setExternalTerminalConnection(ref, NULL);    
    }
    
    // remove the pointer from the net to the node
    AiRefAppDef->destroy(net);
}


// *****************************************************************************
// setNetToAiConnection()
//
/// \brief Connects a net to an Ai graph reference.
///
/// \param ref a reference to an terminal node
/// \param net a net
//
// *****************************************************************************
void
Manager::setNetToAiConnection(oa::oaModBitNet *net, oagAi::Ref ref) {
    if(ai.getNodeType(ref) != oagAi::Node::TERMINAL) {
        cerr << "ERROR: Nets can only be connected to TERMINAL nodes:";
        ai.print(ref);
        QUIT_ON_ERROR;
    } else if (ai.isInverted(ref)) {
        cerr << "ERROR: Nets can only be connected to non-inverted references: ";
        ai.print(ref);
        QUIT_ON_ERROR;
    }
    
    oa::oaString str;
    net->getName(oa::oaVerilogNS(), str);
    AiRefAppDef->set(net, ref);
    ai.setExternalTerminalConnection(ref, reinterpret_cast<void*>(net));
}


// *****************************************************************************
// getNetToAiConnection()
//
/// \brief Returns the net connected to a graph reference.
///
/// NULL is returned if this graph reference is not connected to any net.
///
/// \param ref a reference to an terminal node
/// \return the associated net, or NULL if there is none
//
// *****************************************************************************
oa::oaModBitNet *
Manager::getNetToAiConnection(oagAi::Ref ref) {
    assert(ai.getNodeType(ref) == oagAi::Node::TERMINAL);

    // is the pointer to the connected net stored in the terminal node?
    void *external_terminal_connection = ai.getExternalTerminalConnection(ref);
    oa::oaModBitNet *bit;
    if (external_terminal_connection) {
        bit = reinterpret_cast<oa::oaModBitNet*>(external_terminal_connection);
        assert(getNetToAiConnection(bit) == oagAi::Graph::getNonInverted(ref));
        return bit;
    }
    
    // NOTE: the following code is deprecated

#if 0
    // search all nets for a connection to this reference

    // in all modules in design...
    oa::oaModule *module;
    oa::oaIter<oa::oaModule> modIter(design->getModules());
    while((module = modIter.getNext())) {
    
        // in all nets in module...
        oa::oaModNet *net;
        oa::oaIter<oa::oaModNet> netIter(module->getNets(oacNetIterAll));
        while((net = netIter.getNext())) {
            if (net->getNumBits() > 1) {
                continue;
            }
        
            bit = toBitNet(net);

            // does this net attach to the graph?
            if (bit->getAppDefs().includes(AiRefAppDef)) {
                // does this attach to the reference in question?
                if (ref == static_cast<unsigned int>(AiRefAppDef->get(bit))) {
                    return bit;
                }
            }
        }
    }
#endif

    return NULL;
}


// *****************************************************************************
// isStructural()
//
/// \brief Returns true if a design is entirely structural.
///
/// A structural module involves only interconnections between components.
/// There are no non-equivalence functional relationships between any nets in
/// the module.
///
/// \return true, if the module is structural
//
// *****************************************************************************
bool
Manager::isStructural() {
    assert(design);
    assert(design->getTopModule());

    // if there are any AND or SEQ nodes in the graph, the design
    // is not structural
    if (ai.getNumAndNodes() > 0 || ai.getNumSeqNodes() > 0) {
        return false;
    }

    // if any modules have nets connected to terminals that are 
    // driven by inverted references, the design
    // is not structural
    oa::oaModule *module;
    oa::oaIter<oa::oaModule> modIter(design->getModules());
    while ((module = modIter.getNext())) {
        oa::oaModNet *net;
        oa::oaIter<oa::oaModNet> netIter(module->getNets(oacNetIterSingleBit));
        while ((net = netIter.getNext())) {
            oagAi::Ref ref = getNetToAiConnection(toBitNet(net));;
            if (!oagAi::Graph::isNull(ref)) {
                ref = ai.getTerminalDriver(ref);
                if (!oagAi::Graph::isNull(ref) && oagAi::Graph::isInverted(ref)) {
                    return false;
                }
            }
        }            
    }
    return true;
}


// *****************************************************************************
// getNetToAiConnection()
//
/// \brief Returns the Ai graph reference connected to this net.
///
/// A null Ai graph reference is returned if this net is not connected to the
/// graph.
///
/// \param net
/// \return the associated Ai graph reference, or a null reference is there is none
//
// *****************************************************************************
oagAi::Ref
Manager::getNetToAiConnection(oa::oaModBitNet *net) {

    oagAi::Ref ref = ai.getNull();
    
    // examine each one of the functionally equivalent nets
    if (net->getAppDefs().includes(AiRefAppDef)) {
        ref = AiRefAppDef->get(net);
        assert(!ai.isNull(ref));
        assert(ai.isTerminal(ref));
        return ref;
    }
    
    return ai.getNull();
}


// *****************************************************************************
// getLocalAIsize()
//
/// Returns the number of nodes in the local graph.
///
/// \return the number of nodes in the local graph.
//
// *****************************************************************************
int
Manager::getLocalAIsize() {
    return ai.getNumNodes();
}


// *****************************************************************************
// getHierarchicalAIsize()
//
/// \brief Returns the number of nodes in the unfolded graph.
///
/// This is computed by traversing the hierarchy of DesignInsts.  The number 
/// may include nodes in modules in a subdesign that aren't instantiated.
///
/// \return the number of nodes in the unfolded graph.
//
// *****************************************************************************
int
Manager::getHierarchicalAIsize() {
    assert(design);
    
    return getHierarchicalAIsize(design);
}


// *****************************************************************************
// getHierarchicalAIsize()
//
/// \brief Returns the number of nodes in the unfolded graph of a design.
///
/// This is computed by traversing the hierarchy of DesignInsts.  The number 
/// may include nodes in modules in a subdesign that aren't instantiated.
///
/// \param design
/// \return the number of nodes in the unfolded graph of a design.
//
// *****************************************************************************
int
Manager::getHierarchicalAIsize(oa::oaDesign *design) {
    assert(design);
    assert(design->getTopOccurrence());

    int num = 0;
    
    // if this design is not entirely structural, add the number of nodes here
    if (Manager::hasManager(design)) {
        num += Manager::get(design)->ai.getNumNodes();
    }
    
    // traverse the ModuleInst hierarchy, and recurse on any DesignInsts
    list<oa::oaModule*> toBeVisited;
    toBeVisited.push_back(design->getTopModule());
 
    while(!toBeVisited.empty()) {
        oa::oaModule *module = toBeVisited.front();
        toBeVisited.pop_front();

        oa::oaModInst *inst;
        oa::oaIter<oa::oaModInst> instIter(module->getInsts());
        while((inst = instIter.getNext())) {
            assert(inst);

            oa::oaModule *subMod = inst->getMasterModule();
            if (!subMod) {
              assert(inst->isModDesignInst());
              oa::oaModDesignInst *designInst = static_cast<oa::oaModDesignInst *>(inst);
              oa::oaString libNameString, cellNameString, viewNameString;
              designInst->getLibName(oa::oaNativeNS(), libNameString);
              designInst->getCellName(oa::oaNativeNS(), cellNameString);
              designInst->getViewName(oa::oaNativeNS(), viewNameString);
              cerr << "ERROR: Can not bind to " << libNameString << "/" << cellNameString << "/" << viewNameString << endl;
              QUIT_ON_ERROR;
            }

            if (inst->isModDesignInst()) {
                num += getHierarchicalAIsize(subMod->getDesign());
            } else if (inst->isModModuleInst()) {
                toBeVisited.push_front(subMod);
            } else {
                QUIT_ON_INTERNAL_ERROR;
            }
        }
    }
      
    return num;
}


// *****************************************************************************
// getLocalAndAIsize()
//
/// Returns the number of AND nodes in the local graph.
///
/// \return the number of AND nodes in the local graph.
//
// *****************************************************************************
int
Manager::getLocalAndAIsize() {
    return ai.getNumAndNodes();
}


// *****************************************************************************
// getHierarchicalAndAIsize()
//
/// \brief Returns the number of nodes in the unfolded graph.
///
/// This is computed by traversing the hierarchy of DesignInsts.  The number 
/// may include nodes in modules in a subdesign that aren't instantiated.
///
/// \return the number of nodes in the unfolded graph.
//
// *****************************************************************************
int
Manager::getHierarchicalAndAIsize() {
    assert(design);
    
    return getHierarchicalAndAIsize(design);
}


// *****************************************************************************
// getHierarchicalAndAIsize()
//
/// \brief Returns the number of AND nodes in the unfolded graph of a design.
///
/// This is computed by traversing the hierarchy of DesignInsts.  The number 
/// may include nodes in modules in a subdesign that aren't instantiated.
///
/// \param design
/// \return the number of AND nodes in the unfolded graph of a design.
//
// *****************************************************************************
int
Manager::getHierarchicalAndAIsize(oa::oaDesign *design) {
    assert(design);
    assert(design->getTopOccurrence());

    int num = 0;
    
    // if this design is not entirely structural, add the number of AND nodes here
    if (Manager::hasManager(design)) {
        num += Manager::get(design)->ai.getNumAndNodes();
    }
    
    // traverse the ModuleInst hierarchy, and recurse on any DesignInsts
    list<oa::oaModule*> toBeVisited;
    toBeVisited.push_back(design->getTopModule());
 
    while(!toBeVisited.empty()) {
        oa::oaModule *module = toBeVisited.front();
        toBeVisited.pop_front();

        oa::oaModInst *inst;
        oa::oaIter<oa::oaModInst> instIter(module->getInsts());
        while((inst = instIter.getNext())) {
            assert(inst);

            oa::oaModule *subMod = inst->getMasterModule();
            if (!subMod) {
              assert(inst->isModDesignInst());
              oa::oaModDesignInst *designInst = static_cast<oa::oaModDesignInst *>(inst);
              oa::oaString libNameString, cellNameString, viewNameString;
              designInst->getLibName(oa::oaNativeNS(), libNameString);
              designInst->getCellName(oa::oaNativeNS(), cellNameString);
              designInst->getViewName(oa::oaNativeNS(), viewNameString);
              cerr << "ERROR: Can not bind to " << libNameString << "/" << cellNameString << "/" << viewNameString << endl;
              QUIT_ON_ERROR;
            }

            if (inst->isModDesignInst()) {
                num += getHierarchicalAndAIsize(subMod->getDesign());
            } else if (inst->isModModuleInst()) {
                toBeVisited.push_front(subMod);
            } else {
                QUIT_ON_INTERNAL_ERROR;
            }
        }
    }
      
    return num;
}

// *****************************************************************************
// getLocalSeqNodeCount()
//
/// \brief Returns the number of SEQUENTIAL nodes in the local graph of a design.
///
/// The graphs inside instances are not included in the count.
///
/// \return the number of AND nodes in the local graph.
//
// *****************************************************************************
int
Manager::getLocalSeqNodeCount() {
    return ai.getNumSeqNodes();
}


// *****************************************************************************
// getHierarchicalSeqNodeCount()
//
/// \brief Returns the number of SEQUENTIAL nodes in the unfolded graph of a design.
///
/// This is computed by traversing the hierarchy of DesignInsts.  The number 
/// may include nodes in modules in a subdesign that aren't instantiated.
///
/// \return the number of SEQUENTIAL nodes in the unfolded graph.
//
// *****************************************************************************
int
Manager::getHierarchicalSeqNodeCount() {
    assert(design);
    
    return getHierarchicalSeqNodeCount(design);
}


// *****************************************************************************
// getHierarchicalSeqNodeCount
//
/// \brief Returns the number of SEQUENTIAL nodes in the unfolded graph of a design.
///
/// This is computed by traversing the hierarchy of DesignInsts.  The number 
/// may include nodes in modules in a subdesign that aren't instantiated.
///
/// \param design
/// \return the number of AND nodes in the unfolded graph of a design.
//
// *****************************************************************************
int
Manager::getHierarchicalSeqNodeCount(oa::oaDesign *design) {
    assert(design);
    assert(design->getTopOccurrence());

    int num = 0;
    
    // if this design is not entirely structural, add the number of AND nodes here
    if (Manager::hasManager(design)) {
        num += Manager::get(design)->ai.getNumSeqNodes();
    }
    
    // traverse the ModuleInst hierarchy, and recurse on any DesignInsts
    list<oa::oaModule*> toBeVisited;
    toBeVisited.push_back(design->getTopModule());
 
    while(!toBeVisited.empty()) {
        oa::oaModule *module = toBeVisited.front();
        toBeVisited.pop_front();

        oa::oaModInst *inst;
        oa::oaIter<oa::oaModInst> instIter(module->getInsts());
        while((inst = instIter.getNext())) {
            assert(inst);

            oa::oaModule *subMod = inst->getMasterModule();
            if (!subMod) {
              assert(inst->isModDesignInst());
              oa::oaModDesignInst *designInst = static_cast<oa::oaModDesignInst *>(inst);
              oa::oaString libNameString, cellNameString, viewNameString;
              designInst->getLibName(oa::oaNativeNS(), libNameString);
              designInst->getCellName(oa::oaNativeNS(), cellNameString);
              designInst->getViewName(oa::oaNativeNS(), viewNameString);
              cerr << "ERROR: Can not bind to " << libNameString << "/" << cellNameString << "/" << viewNameString << endl;
              QUIT_ON_ERROR;
            }

            if (inst->isModDesignInst()) {
                num += getHierarchicalSeqNodeCount(subMod->getDesign());
            } else if (inst->isModModuleInst()) {
                toBeVisited.push_front(subMod);
            } else {
                QUIT_ON_INTERNAL_ERROR;
            }
        }
    }
      
    return num;
}


// *****************************************************************************
// getNumTotalInstances()
//
/// \brief Returns the number of module instances in this design.
///
/// \return the number of module instances
//
// *****************************************************************************
int
Manager::getNumTotalInstances() {
    assert(design);
    
    return getNumTotalInstances(design) + 1;
}


// *****************************************************************************
// getNumTotalInstances()
//
/// \brief Returns the number of module instances in this design.
///
/// \return the number of module instances
//
// *****************************************************************************
int
Manager::getNumTotalInstances(oa::oaDesign *design) {
    assert(design);
    assert(design->getTopOccurrence());
    
    int num = 0;
    
    // add the number of instances inside this instance
    num += design->getTopOccurrence()->getInsts().getCount();
    
    // add the number of instances inside each instance
    oa::oaOccInst *inst;
    oa::oaIter<oa::oaOccInst> instIter(design->getTopOccurrence()->getInsts());
    while((inst = instIter.getNext())) {
        assert(inst->getMasterOccurrence());
        assert(inst->getMasterOccurrence()->getModule());
        num += getNumTotalInstances(inst->getMasterOccurrence()->getModule()->getDesign());
    }

    return num;
}


// *****************************************************************************
// getNumLeafInstances()
//
/// \brief Returns the number of module leaf instances in this design.
///
/// \return the number of module leaf instances
//
// *****************************************************************************
int
Manager::getNumLeafInstances() {
    assert(design);
    
    return getNumLeafInstances(design);
}


// *****************************************************************************
// getNumLeafInstances()
//
/// \brief Returns the number of module leaf instances in this design.
///
/// \return the number of module leaf instances
//
// *****************************************************************************
int
Manager::getNumLeafInstances(oa::oaDesign *design) {
    assert(design);
    assert(design->getTopOccurrence());
    
    int num = 0;
    
    // add the number of leaves in each instance
    oa::oaOccInst *inst;
    oa::oaIter<oa::oaOccInst> instIter(design->getTopOccurrence()->getInsts());
    while((inst = instIter.getNext())) {
        assert(inst->getMasterOccurrence());
        assert(inst->getMasterOccurrence()->getModule());    
        num += getNumLeafInstances(inst->getMasterOccurrence()->getModule()->getDesign());
    }
    
    if (num != 0) {
        // non-leaf
        return num;
    } else {
        // leaf
        return 1;    
    }
}


// *****************************************************************************
// print()
//
/// \brief Dumps information about the managed graph.
///
/// Prints the graph node that every ModBitNet in the design is connected to
/// (or if it is not connected), and prints information about every node in
/// the graph.
//
// *****************************************************************************
void
Manager::print() {

    const oa::oaVerilogNS verilogNS;
   
    oa::oaModule *module; 
    oa::oaIter<oa::oaModule> modIter(design->getModules());
    while((module = modIter.getNext())) {
        
        oa::oaModNet *net;
        oa::oaIter<oa::oaModNet> netIter(module->getNets(oacNetIterSingleBit));
        while((net = netIter.getNext())) {
            oa::oaString netNameString;
            net->getName(verilogNS, netNameString);

            oagAi::Ref ref = getNetToAiConnection(toBitNet(net));
            if (!ai.isNull(ref)) {
                cout << "net \"" << netNameString << "\" -> " << ref << endl;
            } else {
                cout << "net \"" << netNameString << "\" is not connected to a graph node" << endl;
            }
        }
    }

    ai.print();
}

}


