/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron Hurst <ahurst@eecs.berkeley.edu>

ChangeLog:
2005-07-20: ChangeLog started
*/

#include <iostream>
#include "oaDesignDB.h"
#include "oagFuncManager.h"
#include "oagFuncPrint.h"
#include "oagUtilOption.h"
#include "oagUtilOptionParser.h"

#include "oagFuncDebug.h"


// *****************************************************************************
// findOpenLib()
//
/// \brief Find an already open, opens from disk, or creates a library.
///
// *****************************************************************************
oa::oaLib *
findOpenLib(const oa::oaScalarName libName, const char *libString) {
    oa::oaLib *lib = NULL;

    if ((lib = oa::oaLib::find(libName)))
        ;
    else if (!oa::oaLib::exists(libString)) {
        lib = oa::oaLib::create(libName, libString);
        oa::oaLibDefList *defList = oa::oaLibDefList::get("lib.defs", 'a');
        oa::oaLibDef::create(defList, libName, libString);
        defList->save();
    } else {
        lib = oa::oaLib::open(libName, libString); 
    }

    return lib;
}


// *****************************************************************************
// main()
//
// *****************************************************************************
int main(int argc, const char **argv) {

    std::cout << "*****************************************************************************" << std::endl;
    std::cout << "Tool:   oaFunc2verilog   Package:    oagFunc     version:  01/2007" << std::endl;
    std::cout << "*****************************************************************************" << std::endl;
    std::cout << std::endl;

    oagUtil::OptionParser options("Convert OpenAccess (including "
        "functional information) to a Verilog description");

    oagUtil::Option *libOpt = options.add("lib", "Input library "
            "name", true, "library");
    oagUtil::Option *viewOpt = options.add("view", "Input view name "
            "(default: netlist)", false, "view");
    oagUtil::Option *cellOpt = options.add("cell", "Input cell (if not specified, "
            "entire library)", false, "cell");
    oagUtil::Option *verilogOpt = options.add("verilog", "Verilog file", true, 
                                              "file");

    if (!options.parse(argc, argv)) {
        std::cerr << options.getMessage();
        exit(0);
    }

    const char *libString = libOpt->getValue();
    const char *viewString 
        = viewOpt->isGiven() ? viewOpt->getValue(): "netlist";

    try {

        // Initialize OpenAccess

        oa::oaDesignInit();
        oa::oaLibDefList::openLibs();
        oagFunc::initialize();

        const oa::oaNativeNS nativeNS;
        const oa::oaScalarName libName(nativeNS, libString);
        const oa::oaScalarName viewName(nativeNS, viewString);
        const oa::oaViewType *viewType = oa::oaViewType::get(oa::oacNetlist);

        // Find, create, or load the first library

        oa::oaLib *library = findOpenLib(libName, libString);
        assert(library);
        library->getAccess(oa::oacReadLibAccess);
        oa::oaView *view = oa::oaView::find(library, viewName, viewType);
        if (!view) {
            cerr << "ERROR: No such view exists in library" << endl;
            exit(0);
        }

        // Find, create, or load additional libraries (if given)
        /*        
        if (leafLibsOpt->isGiven()) {
            assert(strlen(leafLibsOpt->getValue())<256);
            char opt[256];
            strcpy(opt, leafLibsOpt->getValue());
            const char *tok = strtok(opt, "\" ,");
            while(tok) {
              oa::oaScalarName leafLibName(nativeNS, tok);
              oa::oaLib *leafLib = findOpenLib(leafLibName, tok);
              assert(leafLib);
              tok = strtok(NULL, "\" ,");
            }
        }
        */

        // Print statistics about a module (if given)

        if (cellOpt->isGiven()) {

            const oa::oaScalarName cellName(nativeNS, cellOpt->getValue());
            oa::oaDesign *design;
            if ((design = oa::oaDesign::find(libName, cellName, viewName))) {
                ;
            } else {
                // open for read and write
                design = oa::oaDesign::open(libName, cellName, viewName, 'r');
            }
        }

        // Write designs to a Verilog file

        if (cellOpt->isGiven()) {
          // write only one design
          std::cout << "Appending design \'" << cellOpt->getValue() 
                    << "\' to Verilog file " << verilogOpt->getValue() 
                    << "..." << std::endl;
          oa::oaDesign *design;
          const oa::oaScalarName cellName(nativeNS, cellOpt->getValue());
          if ((design = oa::oaDesign::find(libName, cellName, viewName)) ||
              (design = oa::oaDesign::open(libName, cellName, viewName, 'r'))) {
            oagFunc::writeVerilog(design, verilogOpt->getValue());;
          } else {
            cerr << "ERROR: Could not open design " << cellOpt->getValue() << endl;
            QUIT_ON_ERROR;
          }
        } else {
          // write all designs in library
          std::cout << "Appending library to Verilog file " 
                    << verilogOpt->getValue() << "..." << std::endl;
          oa::oaView *view = oa::oaView::find(library, viewName, viewType);
          oagFunc::writeVerilog(library, view, verilogOpt->getValue());
        }

        // Close

        oa::oaDesign *openDesign;
        oa::oaIter<oa::oaDesign> designIter(oa::oaDesign::getOpenDesigns());
        while((openDesign = designIter.getNext())) {
            openDesign->purge();
        }
        std::cout << std::endl;

        // Handle errors

    } catch(oa::oaException &e) {
        std::cerr << "ERROR: " << e.getMsg() << std::endl;
        return 1;
    }

    return 0;
}
