/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron Hurst <ahurst@eecs.berkeley.edu>

ChangeLog:
2005-07-20: ChangeLog started
*/

#include <iostream>
#include "oaDesignDB.h"
#include "oagFuncOccGraph.h"
#include "oagFuncManager.h"
#include "oagFuncSynthesis.h"
#include "oagFuncPrint.h"
#include "oagUtilOption.h"
#include "oagUtilOptionParser.h"
#include "oagFuncQueryOcc.h"
#include "oagFuncDdReasoningEngine.h"

using namespace std;

const oa::oaNativeNS nativeNS;
const oa::oaVerilogNS verilogNS;

// *****************************************************************************
// findOpenLib()
//
/// \brief Find an already open, opens from disk, or creates a library.
///
// *****************************************************************************
oa::oaLib *
findOpenLib(const oa::oaScalarName libName, const char *libString) {
    oa::oaLib *lib = NULL;

    if ((lib = oa::oaLib::find(libName)))
        ;
    else if (!oa::oaLib::exists(libString)) {
        lib = oa::oaLib::create(libName, libString);
        oa::oaLibDefList *defList = oa::oaLibDefList::get("lib.defs", 'a');
        oa::oaLibDef::create(defList, libName, libString);
        defList->save();
    } else {
        lib = oa::oaLib::open(libName, libString); 
    }

    return lib;
}

void printAllFunctionality(oa::oaDesign *design, bool printVariables, bool outputCubes, bool nextStateCubes) {

    cout << "---------------------------------------" << endl;

    // Print module name

    oa::oaModule *module = design->getTopModule();
    oa::oaString    str;
    module->getName(nativeNS, str);
    cout << "MODULE: " << str << endl;
    
    // Get (or create) Manager
    
    oagFunc::Manager *manager = oagFunc::Manager::get(design);

    // Print statistics

    if (manager->isStructural()) {
        cout << "\tlocal module is entirely structural" << endl;
    }
    
    cout << "\tlocal AI graph size: " << manager->getLocalAIsize() << endl;
    cout << "\thier. AI graph size: " << manager->getHierarchicalAIsize() << endl;
    cout << "\thier. num AND nodes: " << manager->getHierarchicalAndAIsize() << endl;
    cout << "\ttotal num  insts: " << manager->getNumTotalInstances() << endl;
    cout << "\ttotal leaf insts: " << manager->getNumLeafInstances() << endl;

    // Create a decision diagram reasoning engine and a query object
    oagFunc::DdReasoningEngine engine;
    oagFunc::QueryOcc          query(design, &engine);
    
    // Print terminals
    
    oa::oaOccurrence *occurrence = design->getTopOccurrence();
    list<oagFunc::OccRef> inputs, outputs, states;
    oagFunc::OccGraph::getInputs(occurrence, inputs);
    oagFunc::OccGraph::getOutputs(occurrence, outputs);
    oagFunc::OccGraph::getStates(occurrence, states);
    cout << "\tnum output bits: " << outputs.size() << endl;    
    cout << "\tnum input bits: " << inputs.size() << endl;
    cout << "\tnum state bits: " << states.size() << endl;
    cout << "\tterminals..." << endl;
    for(list<oagFunc::OccRef>::iterator it = outputs.begin(); it!=outputs.end(); it++) {
        oa::oaOccBitNet *net = oagFunc::OccGraph::getNetToAiConnection(*it);
        assert(net);
        net->getName(verilogNS, str); 
        cout << "\t\tterminal: " << str << " (output)" << endl;
    }    
    for(list<oagFunc::OccRef>::iterator it = inputs.begin(); it!=inputs.end(); it++) {
        oa::oaOccBitNet *net = oagFunc::OccGraph::getNetToAiConnection(*it);
        assert(net);    
        net->getName(verilogNS, str);
        cout << "\t\tterminal: " << str << " (input)" << endl;
    }
    
    // Set variables and print states

    cout << "\tstates..." << endl;
    for(list<oagFunc::OccRef>::iterator it = states.begin(); it!=states.end(); it++) {
        cout << "\t\tstate bit: " << getRefName(*it) << endl;
    }
    
    // Print input functions

    cout << "\tfunctions..." << endl;
    
    for(list<oagFunc::OccRef>::iterator it = inputs.begin(); it!=inputs.end(); it++) {
        DdNode *f = engine.newVariable();
        query.set(*it, f);

        oa::oaOccBitNet *net = oagFunc::OccGraph::getNetToAiConnection(*it);
        assert(net);
        net->getName(verilogNS, str);
        cout << "\t\tinput " << str << " = " << f << endl;

        if (printVariables) {        
            engine.print(f);        
        }
    }

    // Print state functions

    for(list<oagFunc::OccRef>::iterator it = states.begin(); it!=states.end(); it++) {
        DdNode *f = engine.newVariable();
        query.set(*it, f);

        cout << "\t\tstate " << getRefName(*it) << " = " << f << endl;

        if (printVariables) {        
            engine.print(f);        
        }
    }
  
    // Print output functions
    
    for(list<oagFunc::OccRef>::iterator it = outputs.begin(); it!=outputs.end(); it++) {
        oa::oaOccBitNet *net = oagFunc::OccGraph::getNetToAiConnection(*it);
        net->getName(nativeNS, str);
        void *f = query.get(*it);
        cout << "\t\toutput " << str << " = " << f << endl;
        if (outputCubes) {
            engine.print(static_cast<DdNode*>(f));
        }
    }

    // Print next state functions
    
    for(list<oagFunc::OccRef>::iterator it = states.begin(); it!=states.end(); it++) {
        void *f = query.getNextState(*it);
        
        cout << "\t\tnext state " << getRefName(*it) << " = " << f << endl;
        
        if (nextStateCubes) {        
            engine.print(static_cast<DdNode*>(f));       
        }
            
    }
}


void printNetFunctionality(oa::oaDesign *design, bool printVariables, oa::oaOccBitNet *net) {
    assert(net);

    cout << "---------------------------------------" << endl;

    // Print module name

    oa::oaModule *module = design->getTopModule();
    oa::oaString    str;
    module->getName(nativeNS, str);
    cout << "MODULE: " << str << endl;
    
    // Get (or create) Manager
    
    oagFunc::Manager *manager = oagFunc::Manager::get(design);

    // Print statistics

    if (manager->isStructural()) {
        cout << "\tlocal module is entirely structural" << endl;
    }
    
    cout << "\tlocal AI graph size: " << manager->getLocalAIsize() << endl;
    cout << "\thier. AI graph size: " << manager->getHierarchicalAIsize() << endl;
    cout << "\thier. num AND nodes: " << manager->getHierarchicalAndAIsize() << endl;
    cout << "\ttotal num  insts: " << manager->getNumTotalInstances() << endl;
    cout << "\ttotal leaf insts: " << manager->getNumLeafInstances() << endl;

    // Create a decision diagram reasoning engine and a query object
    oagFunc::DdReasoningEngine engine;
    oagFunc::QueryOcc          query(design, &engine);
    
    // Print terminals
    
    oa::oaOccurrence *occurrence = design->getTopOccurrence();
    list<oagFunc::OccRef> inputs, outputs, states;
    oagFunc::OccGraph::getInputs(occurrence, inputs);
    oagFunc::OccGraph::getOutputs(occurrence, outputs);
    oagFunc::OccGraph::getStates(occurrence, states);
    cout << "\tnum output bits: " << outputs.size() << endl;    
    cout << "\tnum input bits: " << inputs.size() << endl;
    cout << "\tnum state bits: " << states.size() << endl;
    cout << "\tterminals..." << endl;
    for(list<oagFunc::OccRef>::iterator it = outputs.begin(); it!=outputs.end(); it++) {
        oa::oaOccBitNet *net = oagFunc::OccGraph::getNetToAiConnection(*it);
        assert(net);
        net->getName(verilogNS, str); 
        cout << "\t\tterminal: " << str << " (output)" << endl;
    }    
    for(list<oagFunc::OccRef>::iterator it = inputs.begin(); it!=inputs.end(); it++) {
        oa::oaOccBitNet *net = oagFunc::OccGraph::getNetToAiConnection(*it);
        assert(net);    
        net->getName(verilogNS, str);
        cout << "\t\tterminal: " << str << " (input)" << endl;
    }
    
    // Set variables and print states

    cout << "\tstates..." << endl;
    for(list<oagFunc::OccRef>::iterator it = states.begin(); it!=states.end(); it++) {
        cout << "\t\tstate bit: " << getRefName(*it) << endl;
    }
    
    // Print input functions

    cout << "\tfunctions..." << endl;
    
    for(list<oagFunc::OccRef>::iterator it = inputs.begin(); it!=inputs.end(); it++) {
        DdNode *f = engine.newVariable();
        query.set(*it, f);

        oa::oaOccBitNet *net = oagFunc::OccGraph::getNetToAiConnection(*it);
        assert(net);
        net->getName(verilogNS, str);
        cout << "\t\tinput " << str << " = " << f << endl;

        if (printVariables) {        
            engine.print(f);        
        }
    }

    // Print state functions

    for(list<oagFunc::OccRef>::iterator it = states.begin(); it!=states.end(); it++) {
        DdNode *f = engine.newVariable();
        query.set(*it, f);

        cout << "\t\tstate " << getRefName(*it) << " = " << f << endl;

        if (printVariables) {        
            engine.print(f);        
        }
    }
  
    // Print net function
    
    cout << "---------------------------------------" << endl;
    oagFunc::OccRef netRef = oagFunc::OccGraph::getNetToAiConnection(net);
    assert(!oagFunc::OccGraph::isNull(netRef));
    void *f = query.get(netRef);
    oa::oaString netNameString;
    net->getPathName(verilogNS, netNameString);
    cout << "\t\tnet " << netNameString << " = " << f << endl;
    engine.print(static_cast<DdNode*>(f));       
    cout << "---------------------------------------" << endl;

}


int main(int argc, const char **argv) {

    std::cout << "*****************************************************************************" << std::endl;
    std::cout << "Tool:   cuddTest        Package:    oagFunc     version:  01/2007" << std::endl;
    std::cout << "*****************************************************************************" << std::endl;
    std::cout << std::endl;

    // Parse input

    oagUtil::OptionParser options("Simple test program to demonstrate the use of the OAGear Func package.");

    oagUtil::Option *libOpt = options.add("lib", "Input/output library name", true, "lib");
    oagUtil::Option *viewOpt = options.add("view", "Input/output view name (default: netlist)", false, "view");
    oagUtil::Option *cellOpt = options.add("cell", "Print info about this cell or module", false, "cell");
    oagUtil::Option *libertyOpt = options.add("liberty", ".lib library file", false, "file");
    oagUtil::Option *verilogOpt = options.add("verilog", "Verilog file", false, "file");
    oagUtil::Option *leafLibsOpt = options.add("leafLibs", "Library files for leaf cells",
             false, "\"libList\"");
    oagUtil::Option *leafViewsOpt = options.add("leafViews", "Library views for leaf cells", 
             false, "\"viewList\"");
    oagUtil::Option *reuseOpt = options.add("reuse", "Re-use existing structures "
            "and append functional descriptions", false);
    oagUtil::Option *nextStateCubesOpt = options.add("next_state_cubes", "Print a cube cover of every next state function", false);
    oagUtil::Option *outputCubesOpt = options.add("output_cubes", "Print a cube cover of every output", false);
    oagUtil::Option *variablesOpt = options.add("variables", "Print the variable of every input/state", false);
    oagUtil::Option *flattenOpt = options.add("flatten", "Flatten the cell (cell needs to be specified)", false);
    oagUtil::Option *uniquifyOpt = options.add("uniquify", "Uniquify the cell (only possible if -flatten is not used)", false, "cell");
    oagUtil::Option *saveOpt = options.add("save", "Save into OpenAccess", false);
    oagUtil::Option *netOpt = options.add("net", "Print only the functionality for a particular net", false, "net");

    if (!options.parse(argc, argv)) {
        cerr << options.getMessage();
        exit(0);
    }

    const char *libString = libOpt->getValue();
    const char *viewString = viewOpt->isGiven() ? viewOpt->getValue(): "netlist";

    try {

    // Initialize OpenAccess
    
    oa::oaDesignInit();
    oa::oaLibDefList::openLibs();
    oagFunc::initialize();

    const oa::oaScalarName libName(nativeNS, libString);
    const oa::oaScalarName viewName(nativeNS, viewString);

    // Find, create, or load the design library and view
    
    oa::oaLib *library = findOpenLib(libName, libString);
    assert(library);
    
    // Find, create, or load additional libraries (if given)
    
    if (leafLibsOpt->isGiven()) {
        assert(strlen(leafLibsOpt->getValue())<256);
        char opt[256];
        strcpy(opt, leafLibsOpt->getValue());
        const char *tok = strtok(opt, "\" ,");
        while(tok) {
            oa::oaScalarName leafLibName(nativeNS, tok);
            oa::oaLib *leafLib = findOpenLib(leafLibName, tok);
            assert(leafLib);
            oagFunc::Synthesis::addLeafLibrary(leafLib);
            tok = strtok(NULL, "\" ,");
        }
    }
    if (leafViewsOpt->isGiven()) {
        assert(strlen(leafViewsOpt->getValue())<256);
        char opt[256];
        strcpy(opt, leafViewsOpt->getValue());
        char *tok = strtok(opt, "\" ,");
        while(tok) {
            oa::oaScalarName leafViewName(nativeNS, tok);
            oagFunc::Synthesis::addLeafView(leafViewName);
            tok = strtok(NULL, "\" ,");
        }
    }
    
    // Should existing structures be re-used?

    if (reuseOpt->isGiven()) {
        oagFunc::Synthesis::setOverwriteStructure(false);
    }
   
    // Read designs from a Liberty file
    
    if (libertyOpt->isGiven()) {
        // read in a cell library
        oagFunc::readLiberty(library, viewName, libertyOpt->getValue());
    }
    
    // Read designs from a Verilog file
    
    if (verilogOpt->isGiven()) {
        // read in functional Verilog description
        oagFunc::readVerilog(library, viewName, verilogOpt->getValue());
    }

    // Print statistics about a module (if given)

    if (cellOpt->isGiven()) {

        const char *cellString = cellOpt->getValue();
        const oa::oaScalarName cellName(nativeNS, cellString);
        oa::oaDesign *design;
        if ((design = oa::oaDesign::find(libName, cellName, viewName)))
            ;
        else if (saveOpt->isGiven()) {
            // open for read and write
            design = oa::oaDesign::open(libName, cellName, viewName, 'a');
        } else {
            // open for read only
            design = oa::oaDesign::open(libName, cellName, viewName, 'r');
        }

        // Print function for a net (if given)
        
        if (netOpt->isGiven()) {

            oa::oaName netName(verilogNS, netOpt->getValue());
            oa::oaOccNet *net = oa::oaOccNet::find(design->getTopOccurrence(), netName);

            if (!net) {
                cerr << "ERROR: Could not find net: " << netOpt->getValue()<< endl;
                exit(0);
            }
            if(net->getNumBits() != 1) {
                cerr << "ERROR: Not a single bit net: " << netOpt->getValue()<< endl;
                exit(0);
            }

            printNetFunctionality(design, variablesOpt->isGiven(), oagFunc::toBitNet(net));

        } else {

            printAllFunctionality(design, variablesOpt->isGiven(), 
                              outputCubesOpt->isGiven(),
                              nextStateCubesOpt->isGiven());
        
            if (flattenOpt->isGiven()) {
                cout << "Flattening \'" << cellString << "\' ..." << endl;
                oagFunc::collapseAllInstancesAllLevels(design->getTopModule());
                
                printAllFunctionality(design, variablesOpt->isGiven(), 
                                  outputCubesOpt->isGiven(),
                                  nextStateCubesOpt->isGiven());
                
            } else if (uniquifyOpt->isGiven()) {
                
                oa::oaSimpleName occName(verilogNS, uniquifyOpt->getValue());
                oa::oaOccInst *occInst = oa::oaOccInst::find(design->getTopOccurrence(), occName);
                
                if (!occInst) {
                    cerr << "ERROR: Could not find instance: " << uniquifyOpt->getValue() << endl;
                    exit(0);
                }

                cout << "Uniquifying \'" << uniquifyOpt->getValue() << "\' ..." << endl;

                oa::oaOccurrence *occurrence = occInst->getMasterOccurrence();
                assert(occurrence);
                oagFunc::uniquify(occurrence);
                
                printAllFunctionality(design, variablesOpt->isGiven(), 
                                      outputCubesOpt->isGiven(),
                                      nextStateCubesOpt->isGiven());
            }
        }
    }

    // Save and close

    oa::oaDesign *openDesign;
    oa::oaIter<oa::oaDesign> designIter(oa::oaDesign::getOpenDesigns());
    while((openDesign = designIter.getNext())) {
        if (saveOpt->isGiven() &&
            openDesign->getLib() == library && openDesign->getMode() != 'r') {
            oa::oaString designString;
            openDesign->getTopModule()->getName(nativeNS, designString);
            cout << "Saving design \'" << designString << "\' ..." << endl;
            openDesign->save();
        }
        openDesign->purge();
    }
    
    // Handle errors

    } catch(oa::oaException &e) {
        std::cerr << "ERROR :" << e.getMsg() << std::endl;
        return 1;
    }

    return 0;
}
