/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Zhong Xiu <zxiu@andrew.cmu.edu>

ChangeLog:
2004-09-15: ChangeLog started
2005-08-11: Move to separate package
*/

#include <vector>
#include "oagTimerTimer.h"
#include "oagBufferSimpleInserter.h"

using namespace oa;
using namespace oagTimer;
using namespace std;

namespace oagBuffer {

///////////////////////////////////////////////////////////////////////////////
/// Constructor.
///
/// \param block Block on which to operate.
/// \param timer Timer to use.

SimpleInserter::SimpleInserter(oa::oaBlock      *block,
                               oagTimer::Timer  *timer) :
    _block(block), _timer(timer)
{
    assert(block);
    assert(timer);
}

///////////////////////////////////////////////////////////////////////////////
/// Destructor.

SimpleInserter::~SimpleInserter()
{
    // do nothing
}

///////////////////////////////////////////////////////////////////////////////
/// Inserts a buffer into a net.
/// Initially the buffer should not have any net attached to its output.
/// All terms and instTerms on the given net which are being driven
/// by that net are moved to a new net which is created and attached
/// to the output of the buffer.
///
/// \param inst Buffer to be inserted.
/// \param net Net to perform insertion on.
/// \return The new net driven by the buffer.

oa::oaScalarNet *
SimpleInserter::insertBuf(oa::oaScalarInst* inst,
                          oa::oaScalarNet*  net) {
    oaScalarNet *net1;
    net1 = oaScalarNet::create(_block);

    // move driven terms to new net
    oaIter<oaTerm> termIter(net->getTerms());
    while (oaTerm *i = termIter.getNext()) {
        oaTermType type(i->getTermType());
        if (type == oacOutputTermType) {
            i->moveToNet(net1);
        }
    }

    // move driven instTerms to new net
    oaIter<oaInstTerm> instTermIter(net->getInstTerms());
    while (oaInstTerm *i = instTermIter.getNext()) {
        oaTerm *masterTerm = i->getTerm();
        assert(masterTerm);
        oaTermType type(masterTerm->getTermType());
        if (type == oacInputTermType) {
            i->addToNet(net1);
        }
    }

    // attach buffer instTerms to appropriate net
    oaIter<oaInstTerm> iTermIter(inst->getInstTerms());
    while (oaInstTerm *i = iTermIter.getNext()) {
        oaTerm *masterTerm = i->getTerm();
        assert(masterTerm);
        oaTermType type(masterTerm->getTermType());
        if (type == oacInputTermType) {
            i->addToNet(net);
        } else if (type == oacOutputTermType) {
            i->addToNet(net1);
        }
    }

    return net1;
}

///////////////////////////////////////////////////////////////////////////////
/// Unattaches the given buffer from the design.
/// The terms and instTerms attached to the output net of the buffer are
/// moved to the input net of the buffer and the output net is
/// destroyed.  The buffer is not destroyed.
///
/// \param inst Buffer to remove.

void
SimpleInserter::removeBuf(oa::oaScalarInst *inst)
{
    oaNet *inNet = 0;
    oaNet *outNet = 0;

    // find input and output nets
    oaIter<oaInstTerm> iTermIter(inst->getInstTerms());
    while (oaInstTerm *i = iTermIter.getNext()) {
        oaTerm *masterTerm = i->getTerm();
        assert(masterTerm);
        oaTermType type(masterTerm->getTermType());
        if (type == oacInputTermType) {
            assert(!inNet);
            inNet = i->getNet();
        } else if (type == oacOutputTermType) {
            assert(!outNet);
            outNet = i->getNet();
        }
    }
    assert(inNet);
    assert(outNet);

    // move terms on outNet
    oaIter<oaTerm> termIter(outNet->getTerms());
    while (oaTerm *i = termIter.getNext()) {
        oaTermType type(i->getTermType());
        if (type == oacOutputTermType) {
            i->moveToNet(inNet);
        } else {
            // error: the output net has multiple drivers
            assert(0);
        }
    }

    // move instTerms on outNet
    oaIter<oaInstTerm> instTermIter(outNet->getInstTerms());
    while (oaInstTerm *i = instTermIter.getNext()) {
        oaTerm *masterTerm = i->getTerm();
        assert(masterTerm);
        oaTermType type(masterTerm->getTermType());
        if (type == oacInputTermType) {
            i->addToNet(inNet);
        } else {
            // error: the output net has multiple drivers
            assert(0);
        }
    }

    outNet->destroy();
}

///////////////////////////////////////////////////////////////////////////////
/// Function object used to sort terms/instTerms into ascending slack order.

class SimpleInserter::SortBySlack {
  public:
    /// Constructor.
    /// \param timer Timer to use to compare slacks.
    SortBySlack(Timer *timer) : _timer(timer) {}
    /// Comparison function.
    /// \param a An oaTerm/oaInstTerm.
    /// \param b An oaTerm/oaInstTerm.
    /// \return True if the slack of a is less than that of b.
    bool operator() (oaOccObject *a,
                     oaOccObject *b)
    { 
        if (_timer->getSlack(a) < _timer->getSlack(b)) {
            return true;
        }
        return false;
    }
  private:
    Timer *_timer;
};


///////////////////////////////////////////////////////////////////////////////
/// Inserts a buffer into a net using timing analysis to determine
/// the best insertion point.
/// Initially the buffer should not have a net attached to its output
/// terminal.  A new net will be created and attached to the buffer output.
///
/// \param inst Buffer to be inserted.
/// \param net Net to perform insertion on.

void
SimpleInserter::optimalInsert(oa::oaScalarInst* inst,
                              oa::oaScalarNet*  net)
{
    vector<oaOccObject*> sinkPins;

    // find all sinks for the net
    oaIter<oaTerm> termIter(net->getTerms());
    while (oaTerm *i = termIter.getNext()) {
        oaTermType type(i->getTermType());
        if (type == oacOutputTermType) {
            sinkPins.push_back(i->getOccTerm());
        }
    } 
    oaIter<oaInstTerm> instTermIter(net->getInstTerms());
    while (oaInstTerm *i = instTermIter.getNext()) {
        oaTerm *masterTerm = i->getTerm();
        assert(masterTerm);
        oaTermType type(masterTerm->getTermType());
        if (type == oacInputTermType) {
            sinkPins.push_back(i->getOccInstTerm());
        }
    }

    // sort all the sink pins by increasing slack
    std::stable_sort(sinkPins.begin(), sinkPins.end(), SortBySlack(_timer));

    // add the buffer and move all pins to output net
    oaScalarNet *outNet = insertBuf(inst, net);
    
    double bestSlack = _timer->getWorstSlack();
    // bestPos will contain the index of the first sink pin which should be
    // moved to the new output net to obtain the best timing
    unsigned bestPos = 0;

    // move sink pins back to the original net one by one
    // to find the best timing
    // use size - 1 to ensure at least one sink lies on the output net
    for (unsigned i = 0; i < sinkPins.size() - 1; i++) {
        if (sinkPins[i]->isOccTerm()) {
            (static_cast<oaOccTerm*>(sinkPins[i]))
                    ->getTerm()->moveToNet(net);
        } else {
            (static_cast<oaOccInstTerm*>(sinkPins[i]))
                    ->getInstTerm()->addToNet(net);
        }
        double t = _timer->getWorstSlack();
        if (t > bestSlack) {
            bestSlack = t;
            bestPos = i + 1;
        }
    }

    // now realize the best timing
    for (unsigned i = bestPos; i < sinkPins.size(); i++) {
        if (sinkPins[i]->isOccTerm()) {
            (static_cast<oaOccTerm*>(sinkPins[i]))
                    ->getTerm()->moveToNet(outNet);
        } else {
            (static_cast<oaOccInstTerm*>(sinkPins[i]))
                    ->getInstTerm()->addToNet(outNet);
        }
    }
}

//-----------------------------------------------------------------------------

/*! 
  This function optimally inserts a buf into the critical path.
  \param lib the library name 
  \param cell the cell name 
  \param view the view name 
  \param path the path vector
*/
void
SimpleInserter::optInsToPath(const oa::oaScalarName     lib,
                             const oa::oaScalarName     cell, 
                             const oa::oaScalarName     view,
                             oagTimer::nodesSlopeDir    &path)
{
    oaDesign *c = oaDesign::open(lib, cell, view, 'r');
    double old;

    old = _timer->getWorstSlack();
    if (path.empty()) {
        return;
    }

    // create the inst
    oaScalarInst *inst = oaScalarInst::create(_block, c,
                                              oaTransform(0, 0, oacR0));

    double s = 0.0;
    int n1 = 0, n2 = 0;
    oaNet *net1, *net2 = NULL;

    // perform the insertion in a net
    for (unsigned i=0; i<path.size(); i++) {
        TPoint *p = TPoint::get(path[i].first);
        if (!p->isPO() && path[i].first->isOccTerm()) {
            net1 = (static_cast<oaOccTerm*>(path[i].first))
                    ->getTerm()->getNet();
        } else if (p->type == TIMING_POINT_SIGNAL_OUT) {
            net1 = (static_cast<oaOccInstTerm*>(path[i].first))
                    ->getInstTerm()->getNet();
        } else {
            continue;
        }
        if (net1) {
            //fullTimingAnalysis();
            double gain = optInsToNet(inst, net1, old, n1, false);
            if (s<gain) {
                s = gain;
                n2 = n1;
                net2 = net1;
            }
        }
    }

    // insert or not, if not, destroy all the timing points and the inst
    if (s == ModelType::ZERO_DELAY()) {
        inst->destroy();
    } else {
        // insert
        insToNet(inst, net2, n2);
    }
    c->close();
}

//-----------------------------------------------------------------------------

/*! 
  This function optimally inserts a buf into a net. 
  \param inst the instance name 
  \param net the net name 
  \param oldWorstSlack the old worst slack
  \param position the position of the insertion which gives the best result
  \param insert insert or not 
*/
double
SimpleInserter::optInsToNet(oa::oaScalarInst    *inst,
                            oa::oaNet           *net,
                            double              oldWorstSlack, 
                            int                 &position,
                            bool                insert)
{

    double p = oldWorstSlack;
    std::vector<oaOccObject*> sinkPins;

    oaIter<oaTerm> termIter(net->getTerms());
    while (oaTerm *i = termIter.getNext()) {
        oaTermType type(i->getTermType());
        if (type == oacOutputTermType) {
            sinkPins.push_back(i->getOccTerm());
        }
    }

    oaIter<oaInstTerm> instTermIter(net->getInstTerms());
    while (oaInstTerm *i = instTermIter.getNext()) {
        oaTerm *masterTerm = i->getTerm();
        assert(masterTerm);
        oaTermType type(masterTerm->getTermType());
        if (type == oacInputTermType) {
            sinkPins.push_back(i->getOccInstTerm());
        }
    }

    // sort all the oaBlocks according to their slacks
    sortBySlacks(sinkPins);

    for (unsigned i=0; i<sinkPins.size(); i++) {
        // add a new net
        oaScalarNet *net1 = oaScalarNet::create(_block,
                                                oacSignalSigType,
                                                false);
        for (unsigned j=i; j<sinkPins.size(); j++) {
            if (sinkPins[j]->isOccTerm()) {
                (static_cast<oaOccTerm*>(sinkPins[j]))
                        ->getTerm()->moveToNet(net1);
            } else {
                (static_cast<oaOccInstTerm*>(sinkPins[j]))
                        ->getInstTerm()->addToNet(net1);
            }
        }

        oaIter<oaInstTerm> iTermIter(inst->getInstTerms());
        while (oaInstTerm *i = iTermIter.getNext()) {
            oaTerm *masterTerm = i->getTerm();
            assert(masterTerm);
            oaTermType type(masterTerm->getTermType());
            if (type == oacInputTermType) {
                i->addToNet(net);
            } else if (type == oacOutputTermType) {
                i->addToNet(net1);
            }
        }
     
        // another timing analysis here
        // fullTimingAnalysis();
        double c = _timer->getWorstSlack();
        if (c > p) {
            p = c;
            position = i;
        } 
        //std::cout << c << std::endl;

        // the original nets structure
        for (unsigned j=i; j<sinkPins.size(); j++) {
            if (sinkPins[j]->isOccTerm()) {
                (static_cast<oaOccTerm*>(sinkPins[j]))
                        ->getTerm()->moveToNet(net);
            } else {
                (static_cast<oaOccInstTerm*>(sinkPins[j]))
                        ->getInstTerm()->addToNet(net);
            }
        }
        iTermIter=inst->getInstTerms();
        while (oaInstTerm *i = iTermIter.getNext()) {
            oaTerm *masterTerm = i->getTerm();
            assert(masterTerm);
            oaTermType type(masterTerm->getTermType());
            if (type == oacInputTermType) {
                i->removeFromNet();
            }
        }
        net1->destroy();
    }

    if (insert) {
        oaScalarNet *net2 = oaScalarNet::create(_block,
                                                oacSignalSigType,
                                                false);
        for (unsigned i=position; i<sinkPins.size(); i++) {
            if (sinkPins[i]->isOccTerm()) {
                (static_cast<oaOccTerm*>(sinkPins[i]))
                        ->getTerm()->moveToNet(net2);
            } else {
                (static_cast<oaOccInstTerm*>(sinkPins[i]))
                        ->getInstTerm()->addToNet(net2);
            }
        }
        oaIter<oaInstTerm> iIter(inst->getInstTerms());
        while (oaInstTerm *i = iIter.getNext()) {
            oaTerm *masterTerm = i->getTerm();
            assert(masterTerm);
            oaTermType type(masterTerm->getTermType());
            if (type == oacInputTermType) {
                i->addToNet(net);
            } else if (type == oacOutputTermType) {
                i->addToNet(net2);
            }
        }
    }

    sinkPins.clear();
    return (p-oldWorstSlack);
}

//-----------------------------------------------------------------------------

/*! 
  This function sort the vector of oaTerms and oaInstTerms 
  according to the slack values. 
  This function should be used in incremental timing analysis, 
  the function first checks if the current data is valid,
  if yes, returns the value, if not, it updates the timing information
  and then returns the value.
  \param blocks the vector of oaBlockObjects need to be sorted
*/
void
SimpleInserter::sortBySlacks(std::vector<oa::oaOccObject*> & blocks)
{
    for (unsigned i=0; i<blocks.size(); i++) {
        _timer->getSlack(blocks[i]);
    }

    std::stable_sort(blocks.begin(), blocks.end(), SortBySlack(_timer));
}

//-----------------------------------------------------------------------------

/*! 
  This function performs the insertion of a buf into a net. 
  \param inst the instance name 
  \param net the net name 
  \param n the position of insertion
*/
void
SimpleInserter::insToNet(oa::oaScalarInst   *inst,
                         oa::oaNet          *net,
                         int                n)
{
    std::vector<oaOccObject*> sinkPins;

    oaIter<oaTerm> termIter(net->getTerms());
    while (oaTerm *i = termIter.getNext()) {
        oaTermType type(i->getTermType());
        if (type == oacOutputTermType) {
            sinkPins.push_back(i->getOccTerm());
        }
    }

    oaIter<oaInstTerm> instTermIter(net->getInstTerms());
    while (oaInstTerm *i = instTermIter.getNext()) {
        oaTerm *masterTerm = i->getTerm();
        assert(masterTerm);
        oaTermType type(masterTerm->getTermType());
        if (type == oacInputTermType) {
            sinkPins.push_back(i->getOccInstTerm());
        }
    }

    // sort all the oaBlocks according to their slacks
    sortBySlacks(sinkPins);
    oaScalarNet *net1 = oaScalarNet::create(_block,
                                            oacSignalSigType,
                                            false);
    for (unsigned i=n; i<sinkPins.size(); i++) {
        if (sinkPins[i]->isOccTerm()) {
            (static_cast<oaOccTerm*>(sinkPins[i]))
                    ->getTerm()->moveToNet(net1);
        } else {
            (static_cast<oaOccInstTerm*>(sinkPins[i]))
                    ->getInstTerm()->addToNet(net1);
        }
    }
    oaIter<oaInstTerm> iIter(inst->getInstTerms());
    while (oaInstTerm *i = iIter.getNext()) {
        oaTerm *masterTerm = i->getTerm();
        assert(masterTerm);
        oaTermType type(masterTerm->getTermType());
        if (type == oacInputTermType) {
            i->addToNet(net);
        } else if (type == oacOutputTermType) {
            i->addToNet(net1);
        }
    }

    sinkPins.clear();
}

//-----------------------------------------------------------------------------

}
