/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: David Papa <iamyou@umich.edu>

ChangeLog:
2004-09-15: ChangeLog started
 */

#include <limits>
#include <set>
#include <assert.h>
#include <iostream>

#include "SchemInstRect.h"
#include "../QoagCommon/QoagCommon.h"
#include "OAGearCommon.h"
#include "DebugLog.h"

using std::set;
using std::pair;
using std::numeric_limits;
using std::cerr;
using std::endl;

using oa::oaSimpleName;
using oa::oaString;
using oa::oaModule;
using oa::oaNativeNS;
using oa::oaIter;
using oa::oaModInst;
using oa::oaModInstTerm;
using oa::oaBlock;
using oa::oaBox;

namespace oagBazaar
{

    namespace Qoag
    {
        using oagBazaar::operator<<;

        QoaSchemInstRect::
            QoaSchemInstRect(oa::oaModInst *inst):
                oaBox(numeric_limits<int>::max(), 
                        numeric_limits<int>::max(), 
                        numeric_limits<int>::max(), 
                        numeric_limits<int>::max()),
                _inst(inst),
                _instName("DEBUG:instName"),
                _masterName("DEBUG:masterName"),
                _displayList(glGenLists(1)),
                _selected(0)
            {
                _instName = getNameFromOA(inst);
                oaModule* master = inst->getMasterModule();
                assert(master);
                _masterName = getNameFromOA(master); 
                learnTerms();
            }

        QoaSchemInstRect::~QoaSchemInstRect()
        {
            glDeleteLists(_displayList,1);
            //TODO, unallocate terms.
        }

        void QoaSchemInstRect::learnTerms(void)
        {
            DebugLog::ostream()<< "SCHEMINSTRECTDEBUG:  call to QoaSchemInstRect::learnTerms("<<_instName<<")."<<endl;


            _allTerms.clear();
            _allOAInstTerms.clear();
            _leftTerms.clear();
            _rightTerms.clear();
            _oaInstTerm2QoaSchemTerm.clear();
            _QoaSchemTerm2oaInstTerm.clear();
            _oaInstTerm2QoaSchemTerm.clear();

            //oaBlock* masterBlock = _getMasterBlock();

            //first build a set of terms to add to ensure uniqueness
            std::set<oaModInstTerm*> inst_terms_to_add;
            oaIter<oaModInstTerm> instTermIter(_inst->getInstTerms());
            oaModInstTerm* instTerm = NULL;
            while( (instTerm = instTermIter.getNext()) )
            {
                assert(instTerm);
                inst_terms_to_add.insert(instTerm);
                //DebugLog::ostream()<<"TERM ADD DEBUG: "<<instTerm<<endl;
            }
            //DebugLog::ostream()<<"SCHEMINST-TERMDEBUG: found "<<inst_terms_to_add.size()<<" instTerms to add"<<endl;

            //this will loop over all terms, and decide which list they belong in
            for(std::set<oaModInstTerm*>::iterator it  = inst_terms_to_add.begin(); 
                    it != inst_terms_to_add.end(); ++it)
            {
                QoaSchemTerm* schemTerm = NULL;
                std::string name = getNameFromOA((*it)->getTerm());

                if( (*it)->getTerm()->getTermType() == oa::oacInputTermType ||
                        (*it)->getTerm()->getTermType() == oa::oacInputOutputTermType )
                {
                    //LEFT SIDE CELLS -- inputs or bidirectional
                    schemTerm = new QoaSchemTerm(*it);
                    assert(schemTerm);
                    _leftTerms.push_back(schemTerm);
                    _allTerms.push_back(schemTerm);
                    _allOAInstTerms.push_back(*it);
                    _oaInstTerm2QoaSchemTerm.insert( pair< oa::oaModInstTerm*, QoaSchemTerm* >(*it, schemTerm) );
                    _QoaSchemTerm2oaInstTerm.insert( pair< QoaSchemTerm*, oa::oaModInstTerm* >(schemTerm, *it) );
                    //DebugLog::ostream()<<"SCHEMINSTRECTDEBUG: adding a term to the left side left.size="<<
                    //          _leftTerms.size()<<endl;
                }
                else if( (*it)->getTerm()->getTermType() == oa::oacOutputTermType )
                {
                    //RIGHT SIDE CELLS -- outputs
                    schemTerm = new QoaSchemTerm(*it);
                    assert(schemTerm);
                    _rightTerms.push_back(schemTerm);
                    _allTerms.push_back(schemTerm);
                    _allOAInstTerms.push_back(*it);
                    _oaInstTerm2QoaSchemTerm.insert( pair< oa::oaModInstTerm*, QoaSchemTerm* >(*it, schemTerm) );
                    _QoaSchemTerm2oaInstTerm.insert( pair< QoaSchemTerm*, oa::oaModInstTerm* >(schemTerm, *it) );
                    //DebugLog::ostream()<<"SCHEMINSTRECTDEBUG: adding a term to the right side right.size="<<
                    //          _rightTerms.size()<<endl;
                }
                else
                {
                    assert(0);
                }  
            }//for all terms_to_add


            unsigned nterms = _inst->getInstTerms().getCount();
            //unsigned nterms = _getNumTermsFromMaster();
            assert( _allTerms.size() == nterms );
            DebugLog::ostream()<<"SCHEMINSTRECTDEBUG: number of terms of \""<<_instName<<"\": "<<_allTerms.size()<<endl;
            debug_double_check_terms_for_consistancy();
        }

        //set vertical spacing for uniform whitespace
        //Whitespace is expressed in a percentage normalized to the range 0-1, ex: 80% == 0.8
        void QoaSchemInstRect::setTermLocations(const std::vector<QoaSchemTerm*>& term_list,
                double x_column_position,
                double top_edge,
                double available_height,
                double whitespace)
        {
            DebugLog::ostream()<<"SCHEMINSTRECTDEBUG: call to setTermLocations size ("<<_instName<<"): "<<term_list.size()<<
                                                                                                          " x: "<<x_column_position<<" top edge: "<<top_edge<<
                                                                                                              "avail: "<<available_height<<" ws: "<<whitespace<<endl;


            //equally spaced among portion of the height with spacing accounting for specified whitespace
            //+1 is to leave 1 inst's height of space equally spaced for a margin at top and bottom.
            //leave 1/2 inst height margin at top and bottom
            double term_total_space = (available_height) / (term_list.size());
            double accum = term_total_space * 0.5;
            double width = right() - left();
            DebugLog::ostream()<<"SCHEMINSTRECTDEBUG: y_incr: "<<term_total_space<<" accum: "<<accum<< " width: "<<width<<
                " term_list size: "<<term_list.size()<<endl;

            for(unsigned it = 0; it < term_list.size(); ++it )
            {
                term_list[it]->setTermGeom( internalPointT(static_cast<int>(x_column_position),static_cast<int>(top_edge-accum)), width );
                DebugLog::ostream()<<"SCHEMINSTRECTDEBUG: Setting Term location (x,y): ("<<
                    x_column_position <<"," << 
                        top_edge - accum <<")."<<endl;
                accum += term_total_space;
            }
        }

        oaModule* QoaSchemInstRect::_getMasterModule(void)
        {
            oaModule *master = _inst->getMasterModule();
            assert(master);
            return master;
        }

        unsigned QoaSchemInstRect::_getNumTermsFromMaster(void)
        {
            oaModule* master = _getMasterModule();
            return master->getTerms().getCount();
        }

        void QoaSchemInstRect::debug_double_check_terms_for_consistancy(void)
        {
            assert( _allTerms.size() != 0 );
            assert( _allTerms.size() == _leftTerms.size() + _rightTerms.size() );
            assert( _inst );

            //unsigned nterms =  _inst->getInstTerms().getCount();
            oa::oaCollection<oaModInstTerm, oaModInst> tempCol(_inst->getInstTerms());
            unsigned nterms =  tempCol.getCount();
            assert(_allTerms.size() == nterms);

            std::set< oa::oaModInstTerm const* > unique_terms;
            for(unsigned i = 0; i<_allTerms.size(); ++i)
            {
                unique_terms.insert(_allTerms[i]->getInstTerm());
            }
            assert(unique_terms.size() == _allTerms.size());
        }

        //this function returns a QString containing
        //A new-line separated string of Term Names from the left side.
        QString QoaSchemInstRect::getLeftTermsFullString(void)
        {
            QString s("");
            for(unsigned i=0;i<_leftTerms.size();i++)
            {
                s.append(QString((_leftTerms[i]->getName() + std::string("\n")).c_str()));
            }
            return s;
        }

        //this function returns a QString containing
        //A new-line separated string of Term Names from the right side.
        QString QoaSchemInstRect::getRightTermsFullString(void)
        {
            QString s("");
            for(unsigned i=0;i<_rightTerms.size();i++)
            {
                s.append(QString((_rightTerms[i]->getName() + std::string("\n")).c_str()));
            }
            return s;
        }

        void QoaSchemInstRect::setRect(const oa::oaBox& bboxNew)
        {
            DebugLog::ostream()<<"SCHEMINSTRECTDEBUG: Call to setRect! (l,b,r,t): "<<bboxNew<<endl;

            double newLeft=bboxNew.left(); double newBottom=bboxNew.bottom();
            double newRight=bboxNew.right(); double newTop=bboxNew.top();
            set(static_cast<int>(newLeft),
                    static_cast<int>(newBottom),
                    static_cast<int>(newRight),
                    static_cast<int>(newTop) );

            double top_edge = top();
            double available_height = (top() - bottom())*0.8;//TODO: remove hardcode
            double width = right() - left();
            double left_x_column_position = width * _L_term_margin + left();
            double right_x_column_position = right() - width * _R_term_margin;

            debug_double_check_terms_for_consistancy();

            setTermLocations(_leftTerms, left_x_column_position, top_edge, available_height);
            setTermLocations(_rightTerms, right_x_column_position, top_edge, available_height);
        }

        void QoaSchemInstRect::setSelected(bool selected)
        {
            _selected=selected;
        }

        QoaSchemTerm* QoaSchemInstRect::getTerm(int term_num)//TODO: make typesafe
        {
            assert( unsigned(term_num) <= _allTerms.size() );
            return _allTerms[term_num];
        }
        oa::oaModInstTerm* QoaSchemInstRect::getOAInstTerm(int inst_term_num)//TODO: make typesafe
        {
            assert( unsigned(inst_term_num) <= _allOAInstTerms.size() );
            return _allOAInstTerms[inst_term_num];
        }

        //Caution!  This function currently searches all terms, use it sparingly.
        //TODO:  Make the following function call an O(1) lookup, 
        //         and change to overload of getTerm
        QoaSchemTerm* QoaSchemInstRect::getInstTerm(oa::oaModInstTerm* instTerm)
        {
            if( _oaInstTerm2QoaSchemTerm.find(instTerm) != _oaInstTerm2QoaSchemTerm.end() )
                return (_oaInstTerm2QoaSchemTerm.find(instTerm)->second);
            else
                return 0;
        }

        oa::oaPoint QoaSchemInstRect::getTermLoc(int term_num)//TODO: make typesafe
        {
            assert(unsigned(term_num) < _allTerms.size());
            return oa::oaPoint(_allTerms[term_num]->getTermCenterPoint());
        }

        QString QoaSchemInstRect::getFullName(void)
        {
            return QString(( _masterName+std::string("::")+_instName).c_str() );
        }

        QoaSchemInstRect::QoaSchemInstRect(const QoaSchemInstRect& orig):
            oaBox(orig),
            _inst(orig._inst),
            _instName(orig._instName),
            _masterName(orig._masterName),
            _displayList(orig._displayList),
            _selected(orig._selected)
        {
            for(unsigned i = 0; i<orig._allTerms.size(); ++i)
            {
                QoaSchemTerm* new_term=new QoaSchemTerm(*(orig._allTerms[i]) );
                _allTerms.push_back( new_term );
                if( find(orig._leftTerms.begin(), orig._leftTerms.end(), new_term ) != orig._leftTerms.end() )
                {
                    _leftTerms.push_back( new_term );
                }
                else
                { 
                    _rightTerms.push_back( new_term );
                }

            }
        }




        QoaSchemInstRect& QoaSchemInstRect::operator=(const QoaSchemInstRect& orig)
        {
            int origLeft = orig.left();
            int origBottom = orig.bottom();
            int origRight = orig.right();
            int origTop = orig.top();
            set(origLeft, origBottom, origRight, origTop);
            _inst = orig._inst;
            _instName=orig._instName;
            _masterName=orig._masterName;
            _displayList=orig._displayList;
            _selected = orig._selected;

            for(unsigned i = 0; i<orig._allTerms.size(); ++i)
            {
                QoaSchemTerm* new_term=new QoaSchemTerm(*(orig._allTerms[i]) );
                _allTerms.push_back( new_term );
                if( find(orig._leftTerms.begin(), orig._leftTerms.end(), new_term ) != orig._leftTerms.end() )
                {
                    _leftTerms.push_back( new_term );
                }
                else
                { 
                    _rightTerms.push_back( new_term );
                }

            }

            return *this;
        }

        void QoaSchemInstRect::draw(void)
        {
            _checkBoundsInit(static_cast<oaBox>(*this), "QoaSchemInstRect");
            DebugLog::ostream() << (*this) << endl; 


            debug_double_check_terms_for_consistancy();  

            //make sure that all of the terms' display lists are up to date
            DebugLog::ostream()<<"SCHEMINSTRECTDEBUG:  numterms to draw for inst:\" "<<  
                _instName << "\"== " << _allTerms.size() << endl;

            for(unsigned i=0;i<_allTerms.size();++i)
            {
                _allTerms[i]->draw();
            }

            std::vector<double> selected_inst_foreground_color;
            setVectorColor(selected_inst_foreground_color, 1.0, 0.0, 0.0); //<-- full red
            std::vector<double> selected_inst_border_color;
            setVectorColor(selected_inst_border_color, 0.0, 0.25, 0.7);//<--- mostly blue

            std::vector<double> unselected_inst_foreground_color;
            setVectorColor(unselected_inst_foreground_color, 0.5, 0.0, 0.0); //<--- darker red
            std::vector<double> unselected_inst_border_color;
            setVectorColor(unselected_inst_border_color, 0.0, 0.25, 0.7); //<---- same blue

            glDeleteLists(_displayList,1);
            _displayList = glGenLists(1);
            glNewList(_displayList,GL_COMPILE);

            if(_selected)
                QoaGlFrameRectd(_displayList, static_cast<oaBox>(*this),
                        Qoag::_border_size, Qoag::_border_size,
                        selected_inst_border_color,selected_inst_foreground_color);
            else
                QoaGlFrameRectd(_displayList, static_cast<oaBox>(*this),
                        Qoag::_border_size, Qoag::_border_size,
                        unselected_inst_border_color,unselected_inst_foreground_color);

            glPushMatrix();
            glTranslated(0.0, 0.0, -0.25);
            for(unsigned i=0;i<_allTerms.size();++i)
            {
                glCallList( _allTerms[i]->getGlDisplayList() );
            }
            glPopMatrix();
            glEndList();

            //draw_test_pattern();  

        }

        void QoaSchemInstRect::draw_test_pattern(void)
        {
            GLuint dispList=glGenLists(1);
            glNewList(dispList,GL_COMPILE);
            glColor3d(0.5,0.5,0.0);
            glRectd(5.0,5.0,95.0,95.0);
            glEndList();
            glCallList(dispList);
            glDeleteLists(dispList,1);
        }


        void QoaSchemInstRect::setName(const std::string& name)
        {
            _instName=name;
        }

        void QoaSchemInstRect::setMasterName(const std::string& masterName)
        {
            _masterName=masterName;
        }

        std::ostream& operator<<(std::ostream& os, QoaSchemInstRect r)
        {
            os << "SCHEMINSTRECTDEBUG: QoaSchemRect Info: "<<endl;
            os <<     "\tName: " << r.getName() << endl;
            os <<     "\tBounding Box(l,b,r,t): ("<<r.left()<<","<<r.bottom()<<","<<r.right()<<","<<r.top()<<")" << endl;
            os <<     "\tAll Terms: "<<endl;
            for(std::vector<QoaSchemTerm*>::iterator it = r._allTerms.begin();
                    it != r._allTerms.end(); ++it)
            {
                os <<         "\t\t" << (*(*it)) << endl;
            }
            os << "SCHEMINSTRECTDEBUG: End QoaSchemRect Info." << endl;

            return os;
        }

    };//end namespace Qoag

}


