/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */
#include <iostream>
#include <iomanip>

#include <qtable.h>
#include <qlineedit.h>

#include "timerPlugIn.h"
#include "QoagCommon/QoagCommonTempl.h"

#include <assert.h>
#include "oaDesignDB.h"
#include "oagTimerTimer.h"
#include "oagTimerReport.h"
#include "oagUtilOption.h"
#include "oagUtilOptionParser.h"

using std::vector;
using std::string;

int RunTimerPlugInCmd(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char *argv[])
{
    static_cast<void>(clientData);
    static_cast<void>(interp);

    oagUtil::OptionParser options("Analyze the timing of an OpenAccess "
            "design");

    oagUtil::Option *libOpt = options.add("lib", "Input library name", true, 
            "library");
    oagUtil::Option *cellOpt = options.add("cell", "Input cell name", true, 
            "cell");
    oagUtil::Option *viewOpt = options.add("view", "Input view name", true, 
            "view");
    oagUtil::Option *libertyOpt = options.add("liberty", ".lib library file", 
            true, "file");
    oagUtil::Option *libertyLibOpt = options.add("liberty_lib",
            "Lib name for timing data (default: from .lib file)",
            false, "library");
    oagUtil::Option *libertyViewOpt = options.add("liberty_view",
            "View name for timing data (default: abstract)",
            false, "view");
    oagUtil::Option *sdcOpt = options.add("sdc", "SDC constraint file", false, 
            "file");
    oagUtil::Option *reportFromOpt = options.add("report_from", "Report worst "
            "timing path from this pin or port", false, "pin/port");
    oagUtil::Option *reportToOpt = options.add("report_to", "Report worst "
            "timing path to this pin or port", false, "pin/port");
    oagUtil::Option *reportThroughOpt = options.add("report_through", "Report "
            "worst timing path through this pin or port", false, "pin/port");
    oagUtil::Option *reportOpt = options.add("report", "Report the timing of "
            "all points, of all endpoints path with the worst slack", false, 
            "timing | endpoint_slack | critical");

    if (!options.parse(argc, argv)) {
        std::cerr << options.getMessage();
    }

    try {
        oa::oaDesignInit();
        oa::oaLibDefList::openLibs();

        const oa::oaNativeNS nativeNS;
        const oa::oaScalarName cellName(nativeNS, cellOpt->getValue());
        const oa::oaScalarName libName(nativeNS, libOpt->getValue());
        const oa::oaScalarName viewName(nativeNS, viewOpt->getValue());

        // take top level cellView
        oa::oaDesign *cv = oa::oaDesign::open(libName, cellName, viewName, 'a');
        assert(cv);
        oa::oaOccurrence *occ = cv->getTopOccurrence();
        assert(occ);

        // initiate the timing project
        oagTimer::Timer *myTiming = new oagTimer::Timer();
        myTiming->readLibrary(libertyOpt->getValue(),
                              libertyLibOpt->getValue(),
                              libertyViewOpt->getValue());
        myTiming->add(cv);

        oagTimer::Report myReport(cv, myTiming);

        oa::oaOccObject *t;

        if (sdcOpt->isGiven()) {
            myTiming->readConstraints(sdcOpt->getValue(), cv);
        }

        if (reportFromOpt->isGiven()) {
            t = oagTimer::Timer::findOccObj(occ, reportFromOpt->getValue());
            if (!t) {
                std::cout << " Not found: " << reportFromOpt->getValue() 
                    << std::endl;
            } else {
                oagTimer::nodesSlopeDir path;
                myTiming->fromPath(path, t);
                myReport.pathReport(path);
            }
        }

        if (reportToOpt->isGiven()) {
            t = oagTimer::Timer::findOccObj(occ, reportToOpt->getValue());
            if (!t) {
                std::cout << " Not found: " << reportToOpt->getValue() 
                    << std::endl;
            } else {
                oagTimer::nodesSlopeDir path;
                myTiming->toPath(path, t);
                myReport.pathReport(path);
            }
        }

        if (reportThroughOpt->isGiven()) {
            t = oagTimer::Timer::findOccObj(occ, reportThroughOpt->getValue());
            if (!t) {
                std::cout << " Not found: " << reportThroughOpt->getValue() 
                    << std::endl;
            } else {
                oagTimer::nodesSlopeDir path;
                myTiming->throughPath(path, t);
                myReport.pathReport(path);
            }
        }

        if (reportOpt->isGiven()) {
            if (strcmp(reportOpt->getValue(), "timing")==0) {
                myReport.reportAll();
            } else if (strcmp(reportOpt->getValue(), "endpoint_slack") == 0) {
                myReport.reportEndpointsSlacks();
            } else if (strcmp(reportOpt->getValue(), "critical") == 0) {
                oagTimer::nodesSlopeDir path;
                myTiming->findWorstPath(path);
                myReport.pathReport(path);
            } else {
                std::cerr << "Check the parameter after -report"
                    << " [timing | endpoint_slack | critical]: " 
                    << reportOpt->getValue() << std::endl;
            }
        }

        delete myTiming;
        cv->close();
    }

    catch (oa::oaException &e) {
        std::cerr << "ERROR: " << e.getMsg() << std::endl;
        return TCL_ERROR;
    }

        return TCL_OK;
}


oagBazaar::oagBazaarPlugInInterface * newPlugIn(void)
{
    TimerPlugIn* plugIn = new TimerPlugIn; //memory leak!
    return plugIn;
}


void TimerPlugIn::load(std::auto_ptr<oagBazaar::oagBazaarPlugInAPI> bazaarAPI)
{
   //DO THIS FIRST BEFORE ANYTHING ELSE 
   _bazaarAPI = bazaarAPI; //take ownership of the API object pointer

   _bazaarAPI->registerTCLCommand(&RunTimerPlugInCmd, "run_timer", this);

   QPopupMenu* timerMenu = _bazaarAPI->addMenu("Timer");
   QToolBar* timerToolBar = _bazaarAPI->addToolBar("Timer Tool Bar");

   QAction* runTimerAction = _bazaarAPI->createAction("run_timer", "Run OAGear Timer", "Run OAGear's Timer.", "stopwatch_icon.png");

   _bazaarAPI->addMenuItem(timerMenu, runTimerAction);
   _bazaarAPI->addToolBarButton(timerToolBar, runTimerAction);

}



