/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: David Papa <iamyou@umich.edu>

ChangeLog:
2004-09-15: ChangeLog started
 */

#include <string>
#include <iostream>
#include <vector>
#include <algorithm>
#include <string>
#include <stdlib.h>

#include <qscrollview.h>
#include <qapplication.h>
#include <qmenubar.h>
#include <qpopupmenu.h>
#include <qpushbutton.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qmessagebox.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qmultilineedit.h>
#include <qwidgetfactory.h>
#include <qsizegrip.h>

#include "oaBase.h"
#include "oaDesignDB.h"

#include "../../QoagCommon/QoagCommon.h"
#include "similarityLayoutViewWnd.h"
#include "oagConnect.h"
#include "similarityLayoutInstView.h"
#include "OAGearCommon.h"
#include "DebugLog.h"

using std::string;
using std::cout;
using std::cerr;
using std::endl;
using std::vector;
using std::min;
using std::max;

using oa::oaInst;
using oa::oaInstTerm;
using oa::oaIter;
using oa::oaBox;
using oa::oaNet;
using oa::oaCollection;
using oa::oaTerm;
using oa::oaTermType;
using oa::oacInputTermType;
using oa::oacOutputTermType;
using oa::oacInputOutputTermType;

namespace oagBazaar
{

    void similarityLayoutViewWnd::apply_zoom(void)
    {
        _layout->setZoom(_zoom_factor);

        QSize minSize(200, 200);
        //prevent max from being less than min
        QSize layoutPlusFrame(sizeLayoutPlusFrame());
        QSize maxSize(max(minSize.width(), layoutPlusFrame.width()), max(minSize.height(), layoutPlusFrame.height()));
        setMinimumSize(minSize);
        setMaximumSize(maxSize); 
    }

    void similarityLayoutViewWnd::zoom_in(void)
    {
        DebugLog::ostream()<<"LAYOUTDEBUG: Calling zoom_in()"<<endl;
        _zoom_factor*=2.0;
        apply_zoom();
    }
    void similarityLayoutViewWnd::zoom_out(void)
    {

        DebugLog::ostream() << "LAYOUTDEBUG: Calling zoom_out()" << endl;
        _zoom_factor /= 2.0;
        apply_zoom();
    }

    void similarityLayoutViewWnd::showCells(void)
    {
        _drawCells = true;
    }

    void similarityLayoutViewWnd::hideCells(void)
    {
        _drawCells = false;
    }

    void similarityLayoutViewWnd::showPins(void)
    {
        _drawPins = true;
    }

    void similarityLayoutViewWnd::hidePins(void)
    {
        _drawPins = false;
    }

    void similarityLayoutViewWnd::showNets(void)
    {
        _drawNets = true;
    }

    void similarityLayoutViewWnd::hideNets(void)
    {
        _drawNets = false;
    }

    void similarityLayoutViewWnd::setModified(void)
    {
        if(!_isModified)
        {
        _isModified = true;
        setcaption();
        }
    }

    void similarityLayoutViewWnd::setUnmodified(void)
    {
        if(_isModified)
        {
        _isModified = false;
        setcaption();
        }
    }


    void similarityLayoutViewWnd::showPlacing(void)
    {
        _drawPlacing=true;
        _placingLabel = new QLabel(QString("Placing..."), viewport(), "placing");
        _placingLabel->setAlignment(Qt::AlignCenter);
        _placingLabel->setFrameStyle( QFrame::Panel | QFrame::Raised );
        _placingLabel->setLineWidth(3);
        QSize viewport_size = viewport()->size();
        int lx = viewport()->x()+viewport_size.width()/3;
        int ly = viewport()->y()+viewport_size.height()/3;
        int w = viewport_size.width()/3;
        int h = viewport_size.height()/3;
        DebugLog::ostream()<<" LAYOUTVIEWWNDDEBUG: Placing geom: "<<lx<<" "<<ly<<" "<<w<<" "<<h<<endl;
        int cx=0, cy=0, cw=0, ch=0;
        viewportToContents(lx, ly, cx, cy);
        viewportToContents(lx+w, ly+w, cw, ch);
        cw = cw-cx;
        ch = ch-cy;
        DebugLog::ostream()<<" LAYOUTVIEWWNDDEBUG: Placing geom: "<<cx<<" "<<cy<<" "<<cw<<" "<<ch<<endl;
        _placingLabel->setGeometry(cx, cy, cw, ch); 
        _placingLabel->show();
        addChild(_placingLabel, cx, cy);
        repaintContents(true);
    }

    void similarityLayoutViewWnd::hidePlacing(void)
    {
        _drawPlacing=false;
        removeChild(_placingLabel);
        delete _placingLabel;
        _placingLabel = 0;
    }

    void similarityLayoutViewWnd::initDesign(void)
    {
        removeChild(_layout);
        if(_layout)
            _layout->close(TRUE);
    }

    void similarityLayoutViewWnd::readOA(void)
    {
        assert(_layout);
        _layout->clear();    

        DebugLog::ostream()<<"LAYOUTDEBUG: Calling readOA()"<<endl;

        //use size_factor to scale all designs to size 300 to start 
        double size_factor = _CoreRegionBBox.getWidth()/300.0;

        int cw=static_cast<int>(_CoreRegionBBox.getWidth()/size_factor),
            ch=static_cast<int>(_CoreRegionBBox.getHeight()/size_factor);

        oaBox coreRegion(0, 0, cw, ch);
        _layout->resizeCR(coreRegion);

        int i = 0;
        oaIter<oaInst> instIter(_block->getInsts());
        while (oaInst   *inst = instIter.getNext())
        {
            oaBox           bbox;
            bbox = Qoag::getPlacementBBox(inst);
            int posx= static_cast<int>(bbox.left()/size_factor);
            int posy= static_cast<int>(bbox.bottom()/size_factor);
            int cellWidth = static_cast<int>(bbox.getWidth()/size_factor);
            int cellHeight = static_cast<int>(bbox.getHeight()/size_factor);
            oaBox  currInstBBox(posx,posy,posx+cellWidth,posy+cellHeight);
            if(i++ % 100 == 0)
            {
                DebugLog::ostream() << "Read "<<i<<" insts, last having BBox "<<currInstBBox<<endl;
            }
            //cout<<"DPDEBUG: testing readOA, calling checkOutputNetDiff"<<endl;
            bool output_different = checkOutputNetDiff(inst);
            bool input_different = checkInputNetDiff(inst);
           // bool color_different = !input_different && output_different;
            bool color_different = output_different;
            _layout->addInst(currInstBBox, color_different?1:0 );
        }   
        DebugLog::ostream()<<"LAYOUTDEBUG: numInsts: "<<i<<endl;
    }

    void similarityLayoutViewWnd::refresh(void)
    {
        DebugLog::ostream()<<"LAYOUTDEBUG: Calling refresh()"<<endl;
        readOA();
        apply_zoom();
        repaintContents(true);
        _layout->paintGL();
    }

    void similarityLayoutViewWnd::createLayout(void)
    {
        _layout = new glSimilarityLayoutWidget( viewport(), oaBox(0,0,0,0));
    }

    void similarityLayoutViewWnd::setcaption(void)
    {
        string caption;
        if(_isModified)
          caption+="<Modified> ";
        caption+="Similarity Viewer: ";

        oa::oaNativeNS ns;
        oa::oaString name;
        _des->getCellName(ns,name);
        caption+=static_cast<const char*>(name);

        setCaption(caption.c_str());
    }

    similarityLayoutViewWnd::similarityLayoutViewWnd(QWidget* parent, string lib, string cell, string view,
            const std::vector<oaNet*>* diffNets) :
        QScrollView(parent, "LayoutView", WStaticContents), _drawCells(true),
        _drawPins(false), _drawNets(false), _drawPlacing(false), _isModified(false),
        _zoom_factor(1.0), _layout(0), _lib(lib),
        _cell(cell), _view(view), _diffNets(diffNets)
    {

        //stole this code from oagConnect, perhaps should be factored out of bazaar
        // Get the oa names 
        oa::oaString        libstr(_lib.c_str());
        oa::oaString        cellstr(_cell.c_str());
        oa::oaString        viewstr(_view.c_str());

        // Treat the names  as belonging 
        //to the Native namespace and convert them to oaNames.
        oa::oaNativeNS      ns;
        oa::oaScalarName    libName(ns, libstr);
        oa::oaScalarName    cellName(ns, cellstr);
        oa::oaScalarName    viewName(ns, viewstr);

        try 
        {
            try
            {
                _des  = oa::oaDesign::open(libName, cellName, viewName, 'r') ;
            }
            catch (oa::oaException &excp) 
            {
                cerr<<"Can't read CellView "<<static_cast<const char *>(libstr)
                    <<static_cast<const char *>(cellstr)
                    <<static_cast<const char *>(viewstr) 
                    <<" (check libDefFile)" << endl; 
                cerr<<"OpenAccess exception, Error: \t"<<
                    static_cast<const char *>(excp.getMsg()) <<endl;
                exit(1);
            }

            _block = _des->getTopBlock() ;
            _block->getBBox(_CoreRegionBBox);
        }
        catch (oa::oaException &excp) 
        {
            cerr<<"OpenAcess exception opening design, Error: "<<
                static_cast<const char *>(excp.getMsg())<<endl;
            exit(1);
        }
  


        setcaption();

        setVScrollBarMode(QScrollView::AlwaysOn);
        setHScrollBarMode(QScrollView::AlwaysOn);

        setMinimumSize(QSize(200,200));
        setMaximumSize(QSize(400,400)); //will change when zoom is applied

        createLayout();
        readOA();
        assert(_layout);
        addChild(_layout);   
        apply_zoom();
        repaintContents(true);
        _layout->paintGL();

        DebugLog::ostream()<<"LAYOUTDEBUG: Contents is: "<<contentsWidth() << " wide and "<<contentsHeight()<<" high"<<endl;
        DebugLog::ostream()<<"LAYOUTDEBUG: Viewport is: "<<viewport()->width()<< " wide and "<<viewport()->height()<<" high"<<endl;
        DebugLog::ostream()<<"LAYOUTDEBUG: Layout is: "<<_layout->width() << " wide and "<<_layout->height()<<" high"<<endl;
    }

    QSize similarityLayoutViewWnd::sizeLayoutPlusFrame(void)
    {
        int sx=0, sy=0;
        if(horizontalScrollBar())
            sy = horizontalScrollBar()->height()+frameWidth()+1;
        else
            sy = frameWidth()+1;

        if(verticalScrollBar())
            sx = verticalScrollBar()->width()+frameWidth()+1;
        else
            sx = frameWidth()+1;

        return QSize( _layout->width()+sx, _layout->height()+sy);
    }

    bool similarityLayoutViewWnd::checkInputNetDiff(oaInst * inst)
    {
        if(!_diffNets) return false;

        //cout<<"DPDEBUG diffnets:"<<endl;
        //for(vector<oaNet*>::const_iterator it = _diffNets->begin(); it != _diffNets->end(); ++it)
        //{
        //    cout<<"Net is: "<<(*it)<<endl;
        //}
       
        //go over the term and collect all nets on output terms
        vector<oaNet*> inputNets;
        oaIter<oaInstTerm> oaITiter(inst->getInstTerms());
        for(oaInstTerm* it = oaITiter.getNext();it!=0; it=oaITiter.getNext())
        {
            oaTerm * t = it->getTerm();
            if(!t)
            {
               cerr<<"Warning! found an inst term with no term, ignoring this inst term!"<<endl;
               continue;
            }
            oaTermType tt = t->getTermType();
            if(tt == oacInputTermType  )
            {
               inputNets.push_back(it->getNet());
            }
        }

        //cout<<"DPDEBUG: size of output nets: "<<outputNets.size()<<endl;

        for(vector<oaNet*>::iterator it = inputNets.begin(); it!= inputNets.end(); ++it)
        {
            //cout<<"\t Net name is : "<<oagBazaar::Qoag::getNameFromOA(*it)<<endl;
            //cout<<"Net2 is: "<<(*it)<<endl;
            //if(std::find(_diffNets->begin(), _diffNets->end(), *it) != _diffNets->end())
            for(vector<oaNet*>::const_iterator it2 = _diffNets->begin(); it2 != _diffNets->end(); ++it2)
       
            {
                 
               //cout<<"\t Net names are : "<<oagBazaar::Qoag::getNameFromOA(*it)<<", "<< oagBazaar::Qoag::getNameFromOA(*it2) <<endl;
               if( oagBazaar::Qoag::getNameFromOA(*it) == oagBazaar::Qoag::getNameFromOA(*it2) )
               {
                   //cout<<"Found an inst driving a diff net!"<<endl;
                   return true; 
               }
            }
        } 

        return false;  
    }
   
    bool similarityLayoutViewWnd::checkOutputNetDiff(oaInst * inst)
    {
        if(!_diffNets) return false;

        //cout<<"DPDEBUG diffnets:"<<endl;
        //for(vector<oaNet*>::const_iterator it = _diffNets->begin(); it != _diffNets->end(); ++it)
        //{
        //    cout<<"Net is: "<<(*it)<<endl;
        //}
       
        //go over the term and collect all nets on output terms
        vector<oaNet*> outputNets;
        oaIter<oaInstTerm> oaITiter(inst->getInstTerms());
        for(oaInstTerm* it = oaITiter.getNext();it!=0; it=oaITiter.getNext())
        {
            oaTerm * t = it->getTerm();
            if(!t)
            {
               cerr<<"Warning! found an inst term with no term, ignoring this inst term!"<<endl;
               continue;
            }
            oaTermType tt = t->getTermType();
            if(tt == oacOutputTermType  || tt == oacInputOutputTermType )
            {
               outputNets.push_back(it->getNet());
            }
        }

        //cout<<"DPDEBUG: size of output nets: "<<outputNets.size()<<endl;

        for(vector<oaNet*>::iterator it = outputNets.begin(); it!= outputNets.end(); ++it)
        {
            //cout<<"\t Net name is : "<<oagBazaar::Qoag::getNameFromOA(*it)<<endl;
            //cout<<"Net2 is: "<<(*it)<<endl;
            //if(std::find(_diffNets->begin(), _diffNets->end(), *it) != _diffNets->end())
            for(vector<oaNet*>::const_iterator it2 = _diffNets->begin(); it2 != _diffNets->end(); ++it2)
       
            {
                 
               //cout<<"\t Net names are : "<<oagBazaar::Qoag::getNameFromOA(*it)<<", "<< oagBazaar::Qoag::getNameFromOA(*it2) <<endl;
               if( oagBazaar::Qoag::getNameFromOA(*it) == oagBazaar::Qoag::getNameFromOA(*it2) )
               {
                   //cout<<"Found an inst driving a diff net!"<<endl;
                   return true; 
               }
            }
        } 

        return false;  
    }

} //namespace oagBazaar


