/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

   This file is part of the OA Gear distribution.  See the COPYING file in
   the top level OA Gear directory for copyright and licensing information. */

/*
Author: David Papa <iamyou@umich.edu>

ChangeLog:
2004-09-15: ChangeLog started
 */

#include <string>
#include <iostream>
#include <vector>
#include <algorithm>
#include <string>
#include <stdlib.h>

#include <qscrollview.h>
#include <qapplication.h>
#include <qmenubar.h>
#include <qpopupmenu.h>
#include <qpushbutton.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qmessagebox.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qmultilineedit.h>
#include <qwidgetfactory.h>
#include <qsizegrip.h>

#include "oaBase.h"
#include "oaDesignDB.h"

#include "../QoagCommon/QoagCommon.h"
#include "LayoutViewWnd.h"
#include "oagConnect.h"
#include "LayoutInstView.h"
#include "OAGearCommon.h"
#include "DebugLog.h"

using std::string;
using std::cerr;
using std::cout;
using std::endl;
using std::vector;
using std::min;
using std::max;

using oa::oaInst;
using oa::oaInstTerm;
using oa::oaIter;
using oa::oaBox;

namespace oagBazaar
{

    void LayoutViewWnd::apply_zoom(void)
    {
        _layout->setZoom(_zoom_factor);

        QSize minSize(200, 200);
        //prevent max from being less than min
        //QSize layoutPlusFrame(sizeLayoutPlusFrame());
        //QSize maxSize(max(minSize.width(), layoutPlusFrame.width()), max(minSize.height(), layoutPlusFrame.height()));
        setMinimumSize(minSize);
        //setMaximumSize(maxSize); 
    }

    void LayoutViewWnd::zoom_in(void)
    {
        DebugLog::ostream()<<"LAYOUTDEBUG: Calling zoom_in()"<<endl;
        //_zoom_factor*=2.0;
        _zoom_factor += 1.0;
        apply_zoom();
    }
    void LayoutViewWnd::zoom_out(void)
    {

        DebugLog::ostream() << "LAYOUTDEBUG: Calling zoom_out()" << endl;
        //_zoom_factor /= 2.0;
        _zoom_factor -= 1.0;
        apply_zoom();
    }

    void LayoutViewWnd::showCells(void)
    {
        _drawCells = true;
    }

    void LayoutViewWnd::hideCells(void)
    {
        _drawCells = false;
    }

    void LayoutViewWnd::showPins(void)
    {
        _drawPins = true;
    }

    void LayoutViewWnd::hidePins(void)
    {
        _drawPins = false;
    }

    void LayoutViewWnd::showNets(void)
    {
        _drawNets = true;
    }

    void LayoutViewWnd::hideNets(void)
    {
        _drawNets = false;
    }

    void LayoutViewWnd::setModified(void)
    {
        if(!_isModified)
        {
            _isModified = true;
            setcaption();
        }
    }

    void LayoutViewWnd::setUnmodified(void)
    {
        if(_isModified)
        {
            _isModified = false;
            setcaption();
        }
    }


    void LayoutViewWnd::showLabel(const std::string& label_str)
    {
        _drawPlacing=true;
        _placingLabel = new QLabel(QString(label_str.c_str()), this, "placing");
        _placingLabel->setAlignment(Qt::AlignCenter);
        _placingLabel->setFrameStyle( QFrame::Panel | QFrame::Raised );
        _placingLabel->setLineWidth(3);
        QSize viewport_size = contentsRect().size();
        int lx = contentsRect().x()+viewport_size.width()/3;
        int ly = contentsRect().y()+viewport_size.height()/3;
        int w = viewport_size.width()/3;
        int h = viewport_size.height()/3;
        DebugLog::ostream()<<" LAYOUTVIEWWNDDEBUG: Placing geom: "<<lx<<" "<<ly<<" "<<w<<" "<<h<<endl;
        _placingLabel->setGeometry(lx, ly, w, h); 
        _placingLabel->show();
    }

    void LayoutViewWnd::hideLabel(void)
    {
        _drawPlacing=false;
        removeChild(_placingLabel);
        delete _placingLabel;
        _placingLabel = 0;
    }

    void LayoutViewWnd::initDesign(void)
    {
        removeChild(_layout);
        if(_layout)
            _layout->close(TRUE);
    }

    void LayoutViewWnd::readOA(void)
    {
        assert(_layout);
        _layout->clear();    

        DebugLog::ostream()<<"LAYOUTDEBUG: Calling readOA()"<<endl;

        //use size_factor to scale all designs to size 300 to start 
        double size_factor = _dbConnection->getCoreRegionBBox().getWidth()/300.0;

        int cw=static_cast<int>(_dbConnection->getCoreRegionBBox().getWidth()/size_factor),
            ch=static_cast<int>(_dbConnection->getCoreRegionBBox().getHeight()/size_factor);

        oaBox coreRegion(0, 0, cw, ch);
        _layout->resizeCR(coreRegion);

        int i = 0;
        oagDesignBlock oagDBlk = _dbConnection->getOAGDesignBlock();
        oaIter<oaInst> instIter(oagDBlk.block->getInsts());
        while (oaInst   *inst = instIter.getNext())
        {
            oaBox           bbox;
            bbox = Qoag::getPlacementBBox(inst);
            int posx= static_cast<int>(bbox.left()/size_factor);
            int posy= static_cast<int>(bbox.bottom()/size_factor);
            int cellWidth = static_cast<int>(bbox.getWidth()/size_factor);
            int cellHeight = static_cast<int>(bbox.getHeight()/size_factor);
            oaBox  currInstBBox(posx,posy,posx+cellWidth,posy+cellHeight);
            if(i++ % 100 == 0)
            {
                DebugLog::ostream() << "Read "<<i<<" insts, last having BBox "<<currInstBBox<<endl;
            }
            _layout->addInst(currInstBBox);
        }   
        DebugLog::ostream()<<"LAYOUTDEBUG: numInsts: "<<i<<endl;
    }

    void LayoutViewWnd::refresh(void)
    {
        DebugLog::ostream()<<"LAYOUTDEBUG: Calling refresh()"<<endl;
        readOA();
        apply_zoom();
        //repaintContents(true);
        _layout->paintGL();
    }

    void LayoutViewWnd::createLayout(void)
    {
        _layout = new glLayoutWidget( this, oaBox(0,0,0,0));
    }

    void LayoutViewWnd::setcaption(void)
    {
        string caption;
        if(_isModified)
            caption+="<Modified> ";
        caption+="Layout: ";

        oa::oaNativeNS ns;
        oa::oaString name;
        _dbConnection->getOAGDesignBlock().design->getCellName(ns,name);
        caption+=static_cast<const char*>(name);

        setCaption(caption.c_str());
    }

    LayoutViewWnd::LayoutViewWnd(QWidget* parent, oagConnection* dbConnection) :
        QFrame(parent, "LayoutView", WStaticContents), _drawCells(true),
        _drawPins(false), _drawNets(false), _drawPlacing(false), _isModified(false),
        _zoom_factor(0.0), _layout(0), _horiz(0), _vert(0), _lib(dbConnection->getLibName()),
        _cell(dbConnection->getCellName()), _view(dbConnection->getViewName()),
        _dbConnection(dbConnection)
    {
        setcaption();

        _horiz = new QScrollBar( QScrollBar::Horizontal, this , "qt_hbar" );
        _vert = new QScrollBar( QScrollBar::Vertical, this , "qt_vbar" );
        updateScrollBars();

        setMinimumSize(QSize(200,200));
        //setMaximumSize(QSize(400,400)); //will change when zoom is applied

        createLayout();
        connect( horizontalScrollBar(), SIGNAL(valueChanged(int)),
                 _layout, SLOT(scrollHorizValue(int)));
        connect( verticalScrollBar(), SIGNAL(valueChanged(int)),
                 _layout, SLOT(scrollVertValue(int)));
        readOA();
        assert(_layout);
        apply_zoom();
        _layout->resize(viewportSize());
        _layout->resizeGL(_zoom_factor);
        //cout<<"DPDEBUG: viewportSize() : "<<viewportSize().width()<<" , "<<viewportSize().height()<<endl;
        _layout->updateGL();



        //DebugLog::ostream()<<"LAYOUTDEBUG: Contents is: "<<contentsWidth() << " wide and "<<contentsHeight()<<" high"<<endl;
        //DebugLog::ostream()<<"LAYOUTDEBUG: Viewport is: "<<viewport()->width()<< " wide and "<<viewport()->height()<<" high"<<endl;
        //DebugLog::ostream()<<"LAYOUTDEBUG: Layout is: "<<_layout->width() << " wide and "<<_layout->height()<<" high"<<endl;
    }

    QSize LayoutViewWnd::viewportSize(void)
    {
        int sx=0, sy=0;
        if(horizontalScrollBar())
            sy = contentsRect().height() - horizontalScrollBar()->height();
        else
            sy = contentsRect().height();

        if(verticalScrollBar())
            sx = contentsRect().width() - verticalScrollBar()->width();
        else
            sx = contentsRect().width();

        return QSize(sx, sy);
    }

    QScrollBar*  LayoutViewWnd::horizontalScrollBar() 
    {
        return _horiz;
    }

    QScrollBar*  LayoutViewWnd::verticalScrollBar() 
    {
        return _vert;
    }

    //code stolen from qt version of this function and tweaked for our purposes
    void LayoutViewWnd::updateScrollBars()
    {
        if(!horizontalScrollBar() && !verticalScrollBar())
            return;

        // I support this should use viewportSize()... but it needs
        // so many of the temporary variables from viewportSize.  hm.
        int fw = frameWidth();
        int lmarg = fw;
        int rmarg = fw;
        int tmarg = fw;
        int bmarg = fw;

        int w = width();
        int h = height();

        int portw, porth;

        bool needh;
        bool needv;
        bool showh;
        bool showv;
        bool showc = TRUE;

        int hsbExt = horizontalScrollBar()->sizeHint().height();
        int vsbExt = verticalScrollBar()->sizeHint().width();

        //QSize oldVisibleSize( contentsRect().width(), contentsRect().height() );

        //      // Do we definitely need the scrollbar?
        //      needh = w-lmarg-rmarg < contentsRect().width();
        //      needv = h-tmarg-bmarg < contentsRect().height();

        //      showh = needh;
        //      showv = needv;

        needh = showh = TRUE;
        needv = showv = TRUE;


        //  calculate viewport size
        if ( showh ) {
            porth=h-hsbExt-tmarg-bmarg;
        } else {
            porth=h-tmarg-bmarg;
        }
        if ( showv ) {
            portw=w-vsbExt-lmarg-rmarg;
        } else {
            portw=w-lmarg-rmarg;
        }

        // Configure scrollbars that we will show
        _vert->setRange( 0, 100);
        _vert->setValue( 50);
        //_vert->setSteps( QScrollView::d->vbar->lineStep(), porth );
        _horiz->setRange( 0, 100 );
        _horiz->setValue( 50);
        //_horiz->setSteps( QScrollView::d->hbar->lineStep(), portw );

        // Position the scrollbars, viewport and corner widget.
        int bottom;
        bool reverse = QApplication::reverseLayout();
        int xoffset = ( reverse && (showv || cornerWidget() )) ? vsbExt : 0;
        int xpos = reverse ? 0 : w - vsbExt;
        bool frameContentsOnly = false;
        //style().styleHint(QStyle::SH_ScrollView_FrameOnlyAroundContents);

        if( ! frameContentsOnly ) {
            if ( reverse )
                xpos += fw;
            else
                xpos -= fw;
        }
        if ( showh ) {
            int right = ( showc || showv || cornerWidget() ) ? w-vsbExt : w;
            _horiz->setGeometry( fw + xoffset, h-hsbExt-fw, right-fw-fw, hsbExt );
            bottom=h-hsbExt;
        }
        if ( showv ) {
            _vert->setGeometry( xpos, fw, vsbExt, h-hsbExt-fw-fw );
        }

        //QWidget *corner = d->corner;
        //if ( !d->corner )
        //corner = d->defaultCorner;
        //if ( ! frameContentsOnly )
        //corner->setGeometry( xpos,
        //		     h-hsbExt-fw,
        //		     vsbExt,
        //		     hsbExt );
        //else
        //corner->setGeometry( xpos,
        //		     h-hsbExt,
        //		     vsbExt,
        //		     hsbExt );


        // Finally, show the scroll bars
        _horiz->show();
        _vert->show();

    }

    void LayoutViewWnd::resizeEvent( QResizeEvent* event )
    {
        QFrame::resizeEvent( event );
        updateScrollBars();
        _layout->resize(viewportSize());
        _layout->resizeGL(_zoom_factor);
        _layout->updateGL();
    }


} //namespace oagBazaar


