/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

#include <iostream>
#include <fstream>
#include <dlfcn.h>

#include "oaDesignDB.h"

#include "tclCommands.h"
#include "controller.h"
#include "dockingCommandWindow.h"
#include "mainwindow.h"
#include "oagException.h"
#include "oagConnect.h"
#include "OAGearCommon.h"

#include "oagLoader.h"


using std::cout;
using std::cerr;
using std::endl;
using std::ifstream;
using std::string;

using oa::oaLibDefList;
using oa::oaLib;
using oa::oaString;

namespace oagBazaar
{

    void registerTclCommands(Tcl_Interp *interp)
    {
        registerTCLCommand(interp, &library_definition_file, "library_definition_file");
        registerTCLCommand(interp, &echo_debug_log, "echo_debug_log");
        registerTCLCommand(interp, &load_layout, "load_layout");
        registerTCLCommand(interp, &load_schematic, "load_schematic");
        registerTCLCommand(interp, &print_job_queue, "print_job_queue");
        registerTCLCommand(interp, &save, "save");
        registerTCLCommand(interp, &save_as, "save_as");
        registerTCLCommand(interp, &load_plugin, "load_plugin");
    }

    void registerTCLCommand(Tcl_Interp* interp, Tcl_CmdProc* cmd, const std::string& cmdName, ClientData clientData, Tcl_CmdDeleteProc *deleteProc)
    {
        Tcl_CreateCommand(interp, cmdName.c_str(), cmd, clientData, deleteProc);
    }




    int library_definition_file(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char *argv[])
    {
        //get rid of compiler unused warnings
        static_cast<void>(clientData);
        static_cast<void>(interp);

        if(argc != 2)
        {
            cout<<"Usage: library_definition_file /path/to/lib.defs"<<endl;
            return TCL_OK;
        }

        ifstream checkFile(argv[1]);
        if(!checkFile.good())
        {
            cerr<<"No such file: \""<<argv[1]<<"\"."<<endl;
            return TCL_ERROR;
        }

        oaLibDefList::openLibs(argv[1]);
        return TCL_OK;
    }


    int echo_debug_log(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char *argv[])
    {
        //get rid of compiler unused warnings
        static_cast<void>(clientData);
        static_cast<void>(interp);
        static_cast<void>(argv);

        if(argc != 1)
        {
            cout<<"Usage: echo_debug_log"<<endl;
            return TCL_ERROR;
        }

        ControllerRegistry::interpToController(interp).getCommandWindow()->echoDebugStream();

        return TCL_OK;
    }

    int load_layout(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char *argv[])
    {
        //get rid of compiler unused warnings
        static_cast<void>(clientData);

        if(argc != 4)
        {
            std::string message("load_layout");
            message+=": Open a layout view window\n";
            message+="Usage: load_layout [library] [cell] [view]\n"
                "\tlibrary - the OpenAccess library name\n"
                "\tcell    - the OpenAccess cell name\n"
                "\tview    - the OpenAccess view name\n";
            cout<<message;
            return TCL_OK;
        }

        std::string lib=argv[1];
        std::string cell=argv[2];
        std::string view=argv[3];

        Controller& controller = ControllerRegistry::interpToController(interp);
        oagConnection* dbConnection = controller.openConnection(lib, cell, view);
        controller.getView().loadLayout(dbConnection);

        return TCL_OK;
    }


    int load_schematic(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char *argv[])
    {
        //get rid of compiler unused warnings
        static_cast<void>(clientData);

        if(argc != 4)
        {
            std::string message("load_schematic");
            message+=": Open a schematic view window\n";
            message+="Usage: load_schematic [library] [cell] [view]\n"
                "\tlibrary - the OpenAccess library name\n"
                "\tcell    - the OpenAccess cell name\n"
                "\tview    - the OpenAccess view name\n";
            cout<<message;
            return TCL_OK;
        }

        std::string lib=argv[1];
        std::string cell=argv[2];
        std::string view=argv[3];

        Controller& controller = ControllerRegistry::interpToController(interp);

        controller.getView().loadSchematic(lib,cell,view);

        return TCL_OK;
    }

    int save(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char *argv[])
    {
        //get rid of compiler unused warnings
        static_cast<void>(clientData);

        if(argc != 4)
        {
            std::string message("save");
            message+=": save an oaDesign\n";
            message+="Usage: save [library] [cell] [view]\n"
                "\tlibrary - the OpenAccess library name\n"
                "\tcell    - the OpenAccess cell name\n"
                "\tview    - the OpenAccess view name\n";
            cout<<message;
            return TCL_OK;
        }

        std::string lib=argv[1];
        std::string cell=argv[2];
        std::string view=argv[3];

        Controller& controller = ControllerRegistry::interpToController(interp);
        oagConnection* dbConnection = NULL;
        try
        {
            dbConnection = controller.getConnection(lib, cell, view);
        }
        catch( oagNoConnection& err )
        {
            cerr<<"Cannot save design "<<lib<<"/"<<cell<<"/"<<view<<" because it no such connection exists"<<endl;
        }

        dbConnection->getWriteAccess();
        oagDesignBlock oaDBlk = dbConnection->getOAGDesignBlock();
        oaDBlk.design->save();

        return TCL_OK;
    }

    int save_as(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char *argv[])
    {
        //get rid of compiler unused warnings
        static_cast<void>(clientData);

        if(argc != 7)
        {
            std::string message("save_as");
            message+=": save an oaDesign using a specified name\n";
            message+="Usage: save_as [library] [cell] [view] [newlibrary] [newcell] [newview]\n"
                "\tlibrary    - the OpenAccess library name of the design\n"
                "\tcell       - the OpenAccess cell name of the design\n"
                "\tview       - the OpenAccess view name of the design\n"
                "\tnewlibrary - the OpenAccess library name for the new design\n"
                "\tnewcell - the OpenAccess cell name for the new design\n"
                "\tnewview - the OpenAccess view name for the new design\n";
            cout<<message;
            return TCL_OK;
        }

        std::string lib=argv[1];
        std::string cell=argv[2];
        std::string view=argv[3];

        Controller& controller = ControllerRegistry::interpToController(interp);
        oagConnection* dbConnection = NULL;
        try
        {
            dbConnection = controller.getConnection(lib, cell, view);
        }
        catch( oagNoConnection& err )
        {
            cerr<<"Cannot save design "<<lib<<"/"<<cell<<"/"<<view<<" because it no such connection exists"<<endl;
        }

        dbConnection->getWriteAccess();
        oagDesignBlock oaDBlk = dbConnection->getOAGDesignBlock();

        std::string newlib=argv[4];
        std::string newcell=argv[5];
        std::string newview=argv[6];

        oa::oaNativeNS ns;
        oa::oaScalarName newLib(ns, newlib.c_str()),
            newCell(ns, newcell.c_str()),
            newView(ns, newview.c_str());


        try
        {

            //This library creation block ensures that if the library doesnt exist, it gets created
            //Try to get a pointer to the library using oaLib::find()
            //If this fails, open or create the library.
            oaLib *lib = oaLib::find(newLib);
            if (!lib)
            {
                cerr<<"Did not find library, attempting to create it"<<endl;

                //get the top oaLibDefList in order to grab the path
                oaLibDefList *list = oaLibDefList::getTopList();
                oaString libraryPath = getLibraryPath(list);
                libraryPath += newlib.c_str();
                cerr<<"Goodie, we got a path, it is: "<<libraryPath<<endl;

                lib = oaLib::create(newLib, libraryPath);

                if (lib)
                {
                    cerr<<"Happy day, we made the lib and everything looks fine, adding it to lib.defs"<<endl;
                    // We need to update the user's lib.def file since we either
                    // found or created the library without a lib.defs reference.
                    updateLibDefsFile(newLib, libraryPath);
                }
                else
                {
                    // Print error mesage
                    cerr << "ERROR : Unable to create " << libraryPath << "/";
                    cerr << newlib << endl;
                    exit(1);
                }
            }
            //end library creation block

            oaDBlk.design->saveAs(newLib, newCell, newView);
        }
        catch (oa::oaException &excp)
        {
            cerr << "OA threw an exception while trying to save_as, Error: " << excp.getMsg() << endl;
            cerr << "Ignoring this error and continuing, please try to save your work to avoid losing data, have a nice day :) "<<endl;
        }

        return TCL_OK;
    }

    int print_job_queue(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char *argv[])
    {
        //get rid of compiler unused warnings
        static_cast<void>(clientData);
        static_cast<void>(argv);

        if(argc != 1)
        {
            std::string message("print_job_queue");
            message+=": Print jobs waiting on the queue\n";
            message+="Usage: print_job_queue \n";
            cout<<message;
            return TCL_OK;
        }


        Controller& controller = ControllerRegistry::interpToController(interp);

        controller.getProcessManager().printQueue();

        return TCL_OK;
    }

    int load_plugin(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char* argv[])
    {
        //get rid of compiler unused warnings
        static_cast<void>(clientData);
        static_cast<void>(argv);
        static_cast<void>(argc);

        if(argc < 2)
        {
            std::string message("load_plugin");
            message+=": load the SimEqui plugin\n";
            message+="Usage: load_plugin sharedObjectLibrary.so [LoadFunction]\n";
            message+="            Default LoadFunction = newPlugIn\n";
            cout<<message;
            return TCL_OK;
        }

        Controller& controller = ControllerRegistry::interpToController(interp);

        void *handle = dlopen(argv[1], RTLD_LAZY | RTLD_GLOBAL);
        if(handle == 0)
        { 
            cerr<<"Could not open shared object file: \""<<argv[1]<<"\"."<<endl;
            return TCL_ERROR;
        }
   
        oagBazaar::oagBazaarPlugInInterface * (*newPlugIn)(void);

        string loadFunction = "newPlugIn";
        if(argc >= 3)
          loadFunction = argv[2];
   
        newPlugIn = (oagBazaar::oagBazaarPlugInInterface* (*)(void)) dlsym(handle, loadFunction.c_str());
        *(void **)(&newPlugIn) =dlsym(handle, loadFunction.c_str());
        if(newPlugIn == 0)
        {
           cerr<<"Could not get symbol \""<<loadFunction.c_str()<<"\" from object file \""<<argv[1]<<"\"."<<endl;
           return TCL_ERROR;
        }

        oagBazaar::oagBazaarPlugInInterface * plugIn = (*newPlugIn)();
        oagLoader loader;
        loader.load(&controller.getView(), plugIn);

        return TCL_OK;
    }

}

