/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
  Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
  ChangeLog:
  2005-05-31: ChangeLog started
*/

#if !defined(oagAiGraph_P)
#define oagAiGraph_P

#include <assert.h>
#include "oagAiRef.h"
#include "oagAiNode.h"
#include <vector>
#include <list>
#include <set>
#include <string>

using namespace std;

namespace oagFunc {
  class Manager;
};

namespace oagAi {

  // *****************************************************************************
  // Graph
  //
  /// \brief An and/inverter graph with extensions.
  ///
  /// For more detailed documentation, see the main page of the package documentation.
  //
  // *****************************************************************************

  class Graph {
    friend class GraphUnitTest;
    friend class oagFunc::Manager;

  public:

    typedef set<Ref>    Cut;
    typedef list<Cut>   CutSet;
  
    /// \name Constructor and destructor
      /// @{

      Graph();
      ~Graph();

      /// @}
      /// \name Global options
      /// @{
    
      void                enableStructuralHashing();
      void                disableStructuralHashing();
      void                enableGarbageCollection();
      void                disableGarbageCollection();    
    
      /// @}
      /// \name Informational queries
      /// @{
      void                dot(string &number);
      int                 getNumNodes() const;
      int                 getNumAndNodes() const;
      int                 getNumSeqNodes() const;

      inline unsigned int getUserData(Ref x, unsigned int index) {
        /// \brief Returns a user data field on a node.
        /// \param x
        /// \param index the data word index
        /// \return the user data
        Node *node = getNode(x);
        assert(node);
        assert(index < NUM_WORDS_AINODE_USER_DATA);
        return node->data[index];
      }
	  inline void         setUserData(Ref x, unsigned int index, unsigned int data) {
	    /// \brief Sets a user data field on a node.
	    /// \param x
	    /// \param index the data word index
	    /// \param data
	    Node *node = getNode(x);
	    assert(node);
	    assert(index < NUM_WORDS_AINODE_USER_DATA);
	    node->data[index] = data;
	  }
    
	  /// @}
	    /// \name Functional operations
	    /// @{
    
	    static inline Ref   getNull() {
	      /// \brief Returns a null reference.
	      /// \return a Ref to a null function
	      return NULL_AIREF;
	    }
	    static inline bool  isNull(Ref x) {
	      /// \brief Tests a reference to see if it is null.
	      /// \param x
	      /// \return true, if the reference is to a null function
	      return (x == NULL_AIREF || x == notOf(NULL_AIREF));
	    }
	    inline Ref          constantOne() const {
	      /// \brief Returns a reference to the function that is always true.
	      return notOf(CONSTANT0_AIREF);
	    }
	    inline Ref          constantZero() const {
	      /// \brief Returns a reference to the function that is always false.
	      return CONSTANT0_AIREF;
	    }
	    static inline Ref   notOf(Ref x) {
	      /// \brief Returns the complement of a function.  
	      /// The structure of the graph will not
	      /// be modified and no new nodes will be created.
	      /// \param x the function to complement
	      /// \return an Ref for the resulting function
	      return (x ^ 0x1);
	    }
	    static inline Ref   notCondOf(Ref x, bool b) {
	      /// \brief Returns the complement of a function (or not)
	      /// The structure of the graph will not
	      /// be modified and no new nodes will be created.
	      /// \param x the function to complement
	      /// \param b 
	      /// \return an Ref for the resulting function
	      return (x ^ b);
	    }
	    static inline bool  isInverted(Ref x) {
	      /// Tests whether a reference that points to a node is inverted.
	      /// \param x
	      /// \return true if the reference is inverted
	      return (x & 0x1);
	    }
	    static inline Ref   getNonInverted(Ref x) {
	      /// \brief Returns a reference that is not inverted.
	      /// \param x
	      /// \return non-inverted reference
	      return (x & ~0x1);
	    }
	    static inline unsigned getNodeID(Ref x) {
	      return x >> 1;
	    }
	    static inline unsigned getRefFromID(unsigned x, bool inv = false) {
	      return (x << 1) | inv;
	    }
    
	    /// @}
	      /// \name Graph modification
	      /// @{
    
	      Ref               newSequential(Ref nextState);
	      Ref               newTerminal(Ref terminal);
	      Ref               newAnd(Ref x, Ref y);    
	      Ref               andOf(Ref x, Ref y);
	      void              setTerminalDriver(Ref terminal, Ref driver);
	      void              setNextState(Ref sequential, Ref nextState);
	      void              setAndLeft(Ref x, Ref left);
	      void              setAndRight(Ref x, Ref right);
    
	      void              resubstitute(Ref original, Ref replacement);
	      void              resubstitute(Ref original, Ref replacement, Ref target);
	      void              detach(Ref x);

	      void              rehash();

	      /// @}
		/// \name Incremental graph traversal
		/// @{
    
		inline Node::Type     getNodeType(Ref x) const {
		  /// \brief Returns the type of the referenced graph node.
		  /// \param x
		  /// \return the type of the referenced node
		  return getNode(x)->type;
		}
    
		//    on terminal nodes...
		inline Ref            getTerminalDriver(Ref terminal) const {
		  /// \brief Returns the function that an terminal node mirrors.
		  /// \param terminal
		  /// \return a reference to the function that the terminal node mirrors
		  Node *node = getNode(terminal);
		  assert(node);
		  assert(node->type == Node::TERMINAL);
		  return node->terminal_driver;
		}
		bool                  isTerminal(Ref x) const;
		void *                getExternalTerminalConnection(Ref terminal) const;
		void                  setExternalTerminalConnection(Ref terminal, void *connection);

		//    on sequential nodes...
		Ref            getNextState(Ref sequential) const;
		bool           isSequential(Ref x) const;
		Node::SequentialData *getSequentialData(Ref sequential);

		//    on and nodes...
		bool                  isAnd(Ref x) const;
		inline Ref            getAndLeft(Ref x) const {
		  /// \brief Returns the left operand of an AND node.
		  /// \param x
		  /// \return reference to left operand
		  Node *node = getNode(x);
		  assert(node->type == Node::AND);
		  return node->left;
		}
		inline Ref            getAndRight(Ref x) const {
		  /// \brief Returns the right operand of an AND node.
		  /// \param x
		  /// \return reference to right operand
		  Node *node = getNode(x);
		  assert(node->type == Node::AND);
		  return node->right;
		}
		//    on all nodes...
		inline const list<Ref>&     getFanout(Ref x) const {
		  /// \brief Returns the fanout of a node.
		  ///
		  /// An error is generated if fanout is not being maintained.
		  /// \param x
		  /// \return the fanout list
		  assert(MAINTAIN_FANOUT);
		  return getNode(x)->fanout;   
		}
		inline bool         hasFanout(Ref x) const {
		  /// \brief Returns the fanout of a node.
		  ///
		  /// An error is generated if fanout is not being maintained.
		  /// \param x
		  /// \return true, if the node has any fanout, false otherwise
		  assert(MAINTAIN_FANOUT);        
		  return getNode(x)->fanoutCount > 0;
		}

		/// @}
		  /// \name Advanced graph traversal
		  /// @{
    
		  CutSet*             enumerateKfeasibleCuts(Ref x, unsigned int maxCutWidth, 
							     int maxCutCount = -1, int maxCutDepth = -1,
							     bool includeConstantNode = true);
		  void                clearKfeasibleCuts();
    
		  bool                hasCombinationalCycle();

		  void                getTransitiveFanin(Ref x, list<Ref> &transitiveFanin, 
							 bool includeRoots = true, bool crossSequential = false);
		  void                getTransitiveFanin(Ref x, vector<Ref> &transitiveFanin, 
							 bool includeRoots = true, bool crossSequential = false);
		  void                getTransitiveFanin(list<Ref> x, list<Ref> &transitiveFanin, 
							 bool includeRoots = true, bool crossSequential = false);
		  void                getTransitiveFanin(list<Ref> x, vector<Ref> &transitiveFanin, 
							 bool includeRoots = true, bool crossSequential = false);

		  void                getTransitiveFanout(Ref x, list<Ref> &transitiveFanout, 
							  bool includeRoots = true, bool crossSequential = false);
		  void                getTransitiveFanout(Ref x, vector<Ref> &transitiveFanout, 
							  bool includeRoots = true, bool crossSequential = false);
		  void                getTransitiveFanout(list<Ref> x, list<Ref> &transitiveFanout, 
							  bool includeRoots = true, bool crossSequential = false);
		  void                getTransitiveFanout(list<Ref> x, vector<Ref> &transitiveFanout, 
							  bool includeRoots = true, bool crossSequential = false);

		  void                getFaninRoots(Ref x, list<Ref> &faninRoots);
		  void                getFanoutRoots(Ref x, list<Ref> &fanoutRoots);
    
		  void                getFaninCone(Ref x, const list<Ref> &coneRoots,
						   list<Ref> &transitiveFanin, bool includeRoots = true);
		  void                getFanoutCone(Ref x, const list<Ref> &coneRoots,
						    list<Ref> &transitiveFanout, bool includeRoots = true);
		  void                getFaninCone(Ref x, const vector<Ref> &coneRoots,
						   vector<Ref> &transitiveFanin, bool includeRoots = true);
		  void                getFanoutCone(Ref x, const vector<Ref> &coneRoots,
						    vector<Ref> &transitiveFanout, bool includeRoots = true);

		  void                getAllSequential(list<Ref> &sequential) const;
		  void                getAllSequential(vector<Ref> &sequential) const;
		  void                getAll(list<Ref> &all) const;
		  void                getAll(vector<Ref> &all) const;

		  /// @}
		    /// \name Equivalence marking
		    /// @{
    
		    bool                testEquivalent(Ref x, Ref y) const;
		    void                setEquivalent(Ref x, Ref y);
		    void                getEquivalent(Ref x, list<Ref> & l) const;
		    void                removeEquivalent(Ref x);
		    void                chooseEquivalent(Ref x);
		    void                getFanoutOfEquivalentNodes(Ref x, list<Ref> & l) const;

		    bool                testEquivalence(Ref x, Ref y) const; // deprecated
		    void                getEquivalents(Ref x, list<Ref> & l) const; // deprecated
		    void                removeEquivalences(Ref x); // deprecated

		    /// @}
		      /// \name Traversal and dirty marking
		      /// @{
    
		      bool                isDirty(unsigned int marker) const;
		      unsigned int        getDirtyMarker() const;

		      unsigned int        newTraversalID();

		      inline void         markVisited(Ref x) { 
			/// \brief Marks a node as having been visited in the current traversal.
			/// \param x
			getNode(x)->traversalID = currentTraversalID; 
		      }
		      inline void         unmarkVisited(Ref x) { 
			/// \brief Unmarks a node as having been visited in the current traversal.
			/// \param x
			getNode(x)->traversalID = 0;
		      }
		      inline bool         isVisited(Ref x) const {
			/// \brief Tests if a node has been visited in the current traversal.
			/// \param x
			/// \return true, if node has been visited
			return getNode(x)->traversalID == currentTraversalID;
		      }

		      /// @}
			/// \name Memory management    
			/// @{
    
			void                garbageCollect();
			void                incrementExternalReferences(Ref x);
			void                decrementExternalReferences(Ref x);
			void                clearExternalReferences(Ref x);

			/// @}
			  /// \name Output / debugging
			  /// @{

			  void                print() const;
			  void                print(Ref x) const;
    
			  /// @}
    
			    protected:

			  static const int    NODES_TO_IGNORE = 2; // i.e. null & const0

			  unsigned int        andNodes, seqNodes; // number of nodes
			  unsigned int        dirtyMarker;
  
			  static const bool   MAINTAIN_FANOUT = true;
  
			  Ref                 CONSTANT0_AIREF;
			  static const Ref    NULL_AIREF = 0;
  
			  // Node creation
			  Node*               newNode();
  
			  // Settings
			  bool                structuralHashingEnabled;
			  bool                garbageCollectionEnabled;
    
			  // Utility functions
			  inline Node*        getNode(Ref x) const {
			    /// \brief Returns a pointer to the referenced Node data structure.
			    assert((x >> 1) <= (lastUsedPage*PAGE_SIZE + lastUsedIndex));
			    Node *page = data[getPage(x)];
			    return &(page[getIndex(x)]);  
			  }
			  inline Ref          getRef(unsigned int page, unsigned int index) const {
			    return (page*PAGE_SIZE+index) << 1;
			  }
			  inline unsigned int getPage(Ref x) const {
			    return (x >> 1) / PAGE_SIZE;
			  }
			  inline unsigned int getIndex(Ref x) const {
			    return (x >> 1) % PAGE_SIZE;
			  }
    
			  // Maintaining fanout
			  void                addToFanout(Ref x, Ref fanout);
			  void                removeFromFanout(Ref x, Ref fanout);

			  // Structural hashing
			  inline unsigned int getStructuralHashKey(Ref x, Ref y) const {
			    /// \brief Computes a structural hash key.
			    ///
			    /// The hash function is generated from a pair of references.  
			    /// If these are the left and right predecessors of a particular 
			    /// node, structurally isomorphic nodes will hash to the same value.
			    ///
			    /// \param x
			    /// \param y
			    /// \return a hash key

			    // reorder such that x < y
			    if (x>y) { Ref t = x; x = y; y = t; }
			    // arbitrary hash function
			    return (x % structuralHashSize * 3) * (y % structuralHashSize + 7) % structuralHashSize;
			  }
			  void                resizeStructuralHash();
			  void                updateStructuralHash(Ref x, int oldhash);

#define STRUCTURAL_HASH_TARGET_UTIL 0.5
#define STRUCTURAL_HASH_RESIZE_UTIL 5.0

			  // Cut hashing
			  map<Node*, CutSet*>       cutHash;
    
			  // Memory management
			  void                      repackMemory();
    
			  // Traversal marking
			  unsigned int              currentTraversalID;

			  void   getTransitiveFanin_recursive(Ref x, vector<Ref> &transitiveFanin, 
							      bool includeRoots, bool crossSequential = false);
			  void   getTransitiveFanout_recursive(Ref x, vector<Ref> &transitiveFanout, 
							       bool includeRoots, bool crossSequential = false);
			  void   getTransitiveFanin_recursive(Ref x, list<Ref> &transitiveFanin, 
							      bool includeRoots, bool crossSequential = false);
			  void   getTransitiveFanout_recursive(Ref x, list<Ref> &transitiveFanout, 
							       bool includeRoots, bool crossSequential = false);
			  bool   hasCombinationalCycle_recursive(Ref x, unsigned int startingTraversalID);

  private:
   
			 
			  void dotEdge(Ref fromRef, Ref toRef, bool nextState, ofstream &outfile);
			  void dotNode(Ref ref, ofstream &outfile);
                          const char * typeStr(const Node *node);
			  
                          // Memory management
			  vector<Node*>             data;
			  static const unsigned int PAGE_SIZE = 1024;   
			  unsigned int              totalNodes;
    
			  unsigned int              nextFreePage, lastUsedPage;
			  unsigned int              nextFreeIndex, lastUsedIndex;

			  // Structural hashing
			  Ref*                structuralHash;
			  unsigned int        structuralHashSize;
   
  };

}

#endif
