/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
  Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
  ChangeLog:
  2005-05-31: ChangeLog started
*/

#include "oagAiGraph.h"
#include <stdlib.h>
#include <list>
#include <vector>
#include <assert.h>
#include <iostream>
#include <fstream>
#include "oagAiDebug.h"

namespace oagAi {


  // *****************************************************************************
  // Class static variable definitions.
  // *****************************************************************************

  // none

  // *****************************************************************************
  // Graph()
  //
  // This is the constructor for the Graph class.  It performs necessary
  // initialization.
  //
  // *****************************************************************************
  Graph::Graph() {
    // initialize
    dirtyMarker = 0;
    nextFreePage = nextFreeIndex = 0;
    lastUsedPage = lastUsedIndex = 0;
    totalNodes = andNodes = seqNodes = 0;
    garbageCollectionEnabled = false;
    structuralHashingEnabled = true;
    currentTraversalID = 1;

    // allocate first page
    Node *page = new Node[PAGE_SIZE];
    assert(page);
    data.push_back(page);

    // create structural hash table
    structuralHash = NULL;
    resizeStructuralHash();
    
    // create null node
    Node *none = newNode();
    assert(NULL_AIREF == none->self);
    none->type = Node::NONE;
    // add a reference so that node is never deleted
    none->refCount++;
    
    // create constant0 node
    Node *constant0 = newNode();
    CONSTANT0_AIREF = constant0->self;
    constant0->type = Node::CONSTANT0;
    // add a reference so that node is never deleted
    constant0->refCount++;
  }


  // *****************************************************************************
  // ~Graph()
  //
  // This is the destructor for the Graph class.  It frees all dynamically
  // allocated memory.
  //
  // *****************************************************************************
  Graph::~Graph() {

    return;

    // free all pages
    while(!data.empty()) {
      delete [] data.back();
      data.pop_back();    
    }
    
    // free structural hash
    if (structuralHash) {
      free(structuralHash);
    }
  }


  // *****************************************************************************
  // getNumNodes()
  //
  /// \brief Returns the total number of nodes in the graph.
  ///
  /// \return the total number of nodes in the graph.
  //
  // *****************************************************************************
  int 
  Graph::getNumNodes() const {
    return totalNodes;
  }


  // *****************************************************************************
  // getNumAndNodes()
  //
  /// \brief Returns the number of AND nodes in the graph.
  ///
  /// \return the number of AND nodes in the graph.
  //
  // *****************************************************************************
  int 
  Graph::getNumAndNodes() const {
    return andNodes;
  }


  // *****************************************************************************
  // getNumSeqNodes()
  //
  /// \brief Returns the number of SEQUENTIAL nodes in the graph.
  ///
  /// \return the number of SEQUENTIAL nodes in the graph.
  //
  // *****************************************************************************
  int 
  Graph::getNumSeqNodes() const {
    return seqNodes;
  }


  // *****************************************************************************
  // dot()
  //
  /// \brief Prints the graph in a comprehensive way
  ///
  ///
  /// Prints the graph to *.dot, which can be viewed with 
  ///
  /// "dot -Tps *.dot -o *.ps" and "gv *.ps"
  /// 
  /// additional documentation for "dot" is available at
  ///
  /// http://graphviz.org/Documentation.php
  ///
  /// In case of redundancy removal two graphs are plotted. One (flatt.ps) before 
  /// redundancy removal is applied and one (after.ps) after the redundancies are 
  /// removed
  //
  // *****************************************************************************

  void
  Graph::dot(string &number)
  {
    Node *node;
    
    ofstream outfile;
    string name = number + ".dot";
    const char * filename;
    filename = name.c_str();
    
    outfile.open(filename);

    unsigned int page = 0;
    unsigned int index = 0;
  
    outfile << "digraph G {" <<  endl;
    outfile << "size = \"7,10\";" <<  endl;

    //cout << "last used index: " << lastUsedIndex << endl ;
    //cout << "last used page:  " << lastUsedPage << endl ;

    while(1){
      if (index == PAGE_SIZE) {
	index = 0;
	page++;
      }
      
      oagAi::Ref ref = getRef(page, index);
     
      if (getNodeType(ref) == Node::AND){
	node = getNode(ref);
        dotEdge(node->left, ref, false, outfile);
	dotEdge(node->right, ref, false, outfile);	
      }
      else if (getNodeType(ref) == Node::SEQUENTIAL) {
	node = getNode(ref);
        if (node->nextState != NULL_AIREF )
	  dotEdge(node->nextState,ref, true, outfile);
      } 
      else if (getNodeType(ref) == Node::TERMINAL) {
	node = getNode(ref);
	if (node->terminal_driver != NULL_AIREF )
	  dotEdge(node->terminal_driver,ref, false, outfile);
      }
      if (index == lastUsedIndex && page == lastUsedPage) break;
      index++;
    }	
    index = 0;
    page = 0;

    while (1) {
      if (index == PAGE_SIZE) {
	index = 0;
	page++;
      }
      oagAi::Ref ref = getRef(page, index);
	
      dotNode(ref, outfile);
      if (index == lastUsedIndex && page == lastUsedPage) break;
      index++;
    }
    outfile << "}" <<  endl;

    outfile.close();
  }


  // *****************************************************************************
  // newTerminal()
  //
  /// \brief Creates a new TERMINAL node, driven by the specified function.
  ///
  /// \param driver the function driving the terminal
  /// \return a reference to the new TERMINAL node
  ///
  // *****************************************************************************
  Ref 
  Graph::newTerminal(Ref driver) {

    // create a new terminal node
    Node *node = newNode();
    node->type = Node::TERMINAL;
    node->terminal_driver = driver;
    addToFanout(driver, node->self);
    DEBUG_PRINTLN("new TERM = " << node->self << "  \tdriver=" << driver)
      return node->self;
  }


  // *****************************************************************************
  // newSequential()
  //
  /// \brief Creates a new SEQUENTIAL node.
  ///
  /// The node's sequentialData will be allocated, and by default it will be a
  /// BLACK_BOX type abstraction.
  ///
  /// \param nextState the data / next state input
  /// \return a reference to the new SEQUENTIAL node
  //
  // *****************************************************************************
  Ref 
  Graph::newSequential(Ref nextState) {

    // create a new SEQUENTIAL node
    seqNodes++;        
    Node *node = newNode();
    node->type = Node::SEQUENTIAL;
    node->sequentialData = new Node::SequentialData(Node::SequentialData::BLACK_BOX);
    node->nextState = nextState;
    addToFanout(nextState, node->self);
    DEBUG_PRINTLN("new SEQ  = " << node->self << "  \tnextState=" << nextState)    
      return node->self;
  }


  // *****************************************************************************
  // andOf()
  //
  /// \brief Returns the logical and of two functions.  
  ///
  /// A new node may or may not be
  /// be created, depending on structual hashing and whether an isomorphic
  /// node already exists.
  ///
  /// \param left the left operand
  /// \param right the right operand
  /// \return a reference to the resulting function
  //
  // *****************************************************************************
  Ref 
  Graph::andOf(Ref left, Ref right) {

    // check for constants
    if (left == CONSTANT0_AIREF || right == CONSTANT0_AIREF) {
      return CONSTANT0_AIREF;
    } else if (left == notOf(CONSTANT0_AIREF)) {
      return right;
    } else if (right == notOf(CONSTANT0_AIREF)) {
      return left;
    } else if (left == right) {
      return left;
    } else if (left == notOf(right)) {
      return CONSTANT0_AIREF;
    }

    // if structural hashing is enabled, search for an isomorphic node
    if(structuralHashingEnabled) {
      Ref current = structuralHash[getStructuralHashKey(left,right)];
      Node *node;
      while(current) {
	node = getNode(current);
	if (node->left == left && node->right == right ||
	    node->left == right && node->right == left) {
	  DEBUG_PRINTLN("iso AND  = " << node->self << "  \tleft=" << left << " right=" << right)
	    return node->self;
	} else {
	  current = node->next;
	}
      }
    }

    // create and return a new node
    return newAnd(left,right);
  }


  // *****************************************************************************
  // newAnd()
  //
  /// \brief Creates a new AND node.
  ///
  /// A new node will always be created, even if an isomorphic
  /// node already exists or the operands can be trivially reduced.
  ///
  /// \param left the left operand
  /// \param right the right operand
  /// \return a refernece to the resulting function
  //
  // *****************************************************************************
  Ref 
  Graph::newAnd(Ref left, Ref right) {

    // create a new "and" node
    andNodes++;
    Node *node = newNode();
    node->type = Node::AND;
    node->left = left;
    node->right = right;
    
    // update fan-outs
    addToFanout(left, node->self);
    addToFanout(right, node->self);
    
    // add node to structural hash table
    node->next = structuralHash[getStructuralHashKey(left,right)];
    structuralHash[getStructuralHashKey(left,right)] = node->self;
    
    DEBUG_PRINTLN("new AND  = " << node->self << " \tleft=" << left << " right=" << right)        
      return node->self;
  }


  // *****************************************************************************
  // enableStructuralHashing()
  //
  /// \brief Turns on structural hashing.  
  ///
  /// No two structurally isomorphic nodes will be
  /// created.  Maintaining the hash requires some memory and inflicts a small
  /// runtime penalty when nodes are created.
  ///
  /// This does not perform perform retroactive hashing.  If two structural
  /// isomorphic nodes were created while hashing was off, they will remain
  /// independent until rehash() is called.
  //
  // *****************************************************************************
  void 
  Graph::enableStructuralHashing() {
    structuralHashingEnabled = true;
  }


  // *****************************************************************************
  // disableStructuralHashing()
  //
  /// \brief Turns off structural hashing.  
  ///
  /// There may be a marginal
  /// speedup for node creation at the expense of creating structural isomorphic
  /// nodes and using more memory.
  //
  // *****************************************************************************
  void 
  Graph::disableStructuralHashing() {
    structuralHashingEnabled = false;
  }


  // *****************************************************************************
  // enableGarbageCollection()
  //
  /// \brief Turns on the automatic deallocation of unused memory.
  ///
  // *****************************************************************************
  void 
  Graph::enableGarbageCollection() {
    garbageCollectionEnabled = true;
  }


  // *****************************************************************************
  // disableGarbageCollection()
  //
  /// \brief Turns off the automatic deallocation of unused memory.
  ///
  // *****************************************************************************
  void 
  Graph::disableGarbageCollection() {
    garbageCollectionEnabled = false;
  }


  // *****************************************************************************
  // incrementExternalReferences()
  //
  /// \brief Increment the reference count of a particular node.  
  ///
  /// If there are any 
  /// external references to a node, it will not be removed during garbage 
  /// collection.
  ///
  /// \param x
  //
  // *****************************************************************************
  void 
  Graph::incrementExternalReferences(Ref x) {
    getNode(x)->refCount++;
  }


  // *****************************************************************************
  // decrementExternalReferences()
  //
  /// \brief Decrement the reference count of a particular node.  
  /// 
  /// If there are any external references to a node, it will not be removed 
  /// during garbage collection.
  ///
  /// \param x
  //
  // *****************************************************************************
  void 
  Graph::decrementExternalReferences(Ref x) {
    getNode(x)->refCount--;
  }


  // *****************************************************************************
  // clearExternalReferences()
  //
  /// \brief Clears the reference count of a particular node.
  ///
  /// In conjunction with clearExternalReferences(), this function is useful
  /// for marking a node as ready for removal.  Doing both actions will guarantee its
  /// garbage collection, though no memory is deallocated here.
  ///
  /// \param x
  //
  // *****************************************************************************
  void 
  Graph::clearExternalReferences(Ref x) {
    getNode(x)->refCount = 0;
  }


  // *****************************************************************************
  // garbageCollect()
  //
  /// \brief Frees any memory being used by unreferenced nodes.  
  /// 
  /// Unreferenced nodes are removed, and the remaining nodes are packed 
  /// to remove the resulting holes.  Any pages that are entirely empty 
  /// will be deallocated.
  ///
  /// To guarantee that nodes that are in-use are not destroyed, use 
  /// incrementExternalReferences() to mark them.  Any nodes that are necessary
  /// to represent the function of those nodes (i.e. the transitive fan-in)
  /// will also be preserved.
  //
  // *****************************************************************************
  void 
  Graph::garbageCollect() {
    DEBUG_PRINTLN("garbage collecting...")
      DEBUG_PRINTLN("\told next free ref " << getRef(nextFreePage,nextFreeIndex) )
      DEBUG_PRINTLN("\told last used ref " << getRef(lastUsedPage,lastUsedIndex) )    
    
      // find existing deletion candidates
      list<Ref> pendingDeletion;
    unsigned int page = 0;
    unsigned int index = 0;
    Node *node;
    while( !(index > lastUsedIndex && page == lastUsedPage || page > lastUsedPage) ) {
      if (index == PAGE_SIZE) {
	index = 0;
	page++;
      }
      node = &(data[page][index]);
        
      if (node->refCount + node->fanoutCount == 0 && node->type != Node::NONE) {
	pendingDeletion.push_front(getRef(page,index));
      }
      index++;        
    }
    
    // delete recursively
    vector<Ref> alreadyDeleted;
    while(!pendingDeletion.empty()) {
      Ref ref = pendingDeletion.front();
      Node *node = getNode(ref);
      DEBUG_PRINTLN("\tfreeing ref " << ref);
      --totalNodes;
        
      // is this the earliest hole?  if so, move nextFreeIndex pointer back
      if (getPage(ref) < nextFreePage ||
	  getPage(ref) == nextFreePage && getIndex(ref) < nextFreeIndex) {
	nextFreePage = getPage(ref);
	nextFreeIndex = getIndex(ref);    
      }
    
      // remove equivalences
      removeEquivalent(ref);
    
      // decrement fan-in refCounts and search for more nodes to delete
      if (node->type == Node::AND) {
	--andNodes;
	Node *left = getNode(node->left);
	removeFromFanout(node->left, ref);
	if(left->fanoutCount + left->refCount == 0) {
	  pendingDeletion.push_back(node->left);
	}
	Node *right = getNode(node->right);
	removeFromFanout(node->right, ref);
	if(right->fanoutCount + right->refCount == 0) {
	  pendingDeletion.push_back(node->right);
	}
      }
      else if (node->type == Node::SEQUENTIAL) {
	--seqNodes;
	Node *nextState = getNode(node->nextState);
	removeFromFanout(node->nextState, ref);
	if(nextState->fanoutCount + nextState->refCount == 0) {
	  pendingDeletion.push_back(node->nextState);        
	}
      }
      else if (node->type == Node::TERMINAL) {
	Node *terminal = getNode(node->terminal_driver);
	removeFromFanout(node->terminal_driver, ref);
	if(terminal->fanoutCount + terminal->refCount == 0) {
	  pendingDeletion.push_back(node->terminal_driver);      
	}
      }

      node->clear();
      alreadyDeleted.push_back(ref);
      pendingDeletion.pop_front();
    }
    
    // repack nodes in memory
    // repackMemory();
 
    // rebuild structural hash
    resizeStructuralHash();
    
    // find the new last index/page
    unsigned int prevLastPage = lastUsedPage;
    while(data[lastUsedPage][lastUsedIndex].type == Node::NONE) {
      if (lastUsedIndex == 0) {
	assert(lastUsedPage != 0);
	--lastUsedPage;
	lastUsedIndex = PAGE_SIZE;
      }
      --lastUsedIndex;
    }
    DEBUG_PRINTLN("\tnew next free ref " << getRef(nextFreePage,nextFreeIndex) )
      DEBUG_PRINTLN("\tnew last used ref " << getRef(lastUsedPage,lastUsedIndex) )

      // free the newly empty pages    
      for(; prevLastPage>lastUsedPage; --prevLastPage) {
	DEBUG_PRINTLN("\t\tdeallocating page " << prevLastPage)
	  delete [] data[prevLastPage];
      }
  }


  // *****************************************************************************
  // repackMemory()
  //
  /// \brief Reorder nodes to be as packed as possible.
  ///
  /// This function reorders nodes that don't have an external reference.
  /// It moves nodes from end of data to holes left by deleted nodes.  Some
  /// memory may be freed.
  //
  // *****************************************************************************
  void
  Graph::repackMemory() {

    DEBUG_PRINTLN("Repacking node data in memory");
    // NEEDS DEBUGGING
    QUIT_ON_INTERNAL_ERROR;

    /*
      sort(alreadyDeleted.begin(), alreadyDeleted.end());
      vector<Ref>::iterator nextHole = alreadyDeleted.begin();
      map<Ref,Ref> movedNodes;
      page = lastUsedPage;
      index = lastUsedIndex;
      while( nextHole != alreadyDeleted.end() ) {
      node = &(data[page][index]);
      if (node->type != Node::NONE && node->refCount == 0) {
      Node *hole = getNode(*nextHole);
      DEBUG_PRINTLN("\tmoving " << getRef(page,index) << " to hole " << *nextHole)
      memcpy(hole, node, sizeof(Node));
      hole->self = node->self;
      movedNodes[getRef(page,index)] = *nextHole;
      nextHole++;
      } else {
      movedNodes[getRef(page,index)] = getRef(page,index);
      }

      if (index == 0) {
      index = PAGE_SIZE;
      if (page == 0) {
      break;
      }
      page--;
      }
      index--;        
      }
    
      // update references and recalculate {next/last}{Page/Index}
      page = index = 0;
      nextFreeIndex = lastUsedIndex;
      nextFreePage = lastUsedPage;
      unsigned int prevLastIndex = lastUsedIndex;
      unsigned int prevLastPage = lastUsedPage;
      while( !(index == prevLastIndex && page == prevLastPage) ) {
      if (index == PAGE_SIZE) {
      index = 0;
      page++;
      }
      node = &(data[page][index]);
      // update references
      switch(node->type) {
      case Node::SEQUENTIAL:
      if (movedNodes.find(getNonInverted(node->nextState)) != movedNodes.end()) {
      node->nextState = movedNodes[getNonInverted(node->nextState)] | (node->nextState & 0x1);
      }
      // UNIMPLEMENTED : update SequentialData references : FIX THIS NOW!!!
      break;
      case Node::TERMINAL:
      if (movedNodes.find(getNonInverted(node->terminal_driver)) != movedNodes.end()) {
      node->terminal_driver = movedNodes[getNonInverted(node->terminal_driver)] | (node->terminal_driver & 0x1);
      }
      break;
      case Node::AND:
      if (movedNodes.find(getNonInverted(node->left)) != movedNodes.end()) {
      node->left = movedNodes[getNonInverted(node->left)] | (node->left & 0x1);
      }
      if (movedNodes.find(getNonInverted(node->right)) != movedNodes.end()) {
      node->right = movedNodes[getNonInverted(node->right)] | (node->right & 0x1);
      }
      break;
      default:
      break;
      }

      if (node->type == Node::NONE) {
      if (nextFreeIndex == lastUsedIndex && nextFreePage == lastUsedPage) {
      nextFreeIndex = index;
      nextFreePage = page;
      }
      } else {
      lastUsedIndex = index+1;
      lastUsedPage = page;
      }

      index++;        
      }

    */

    DEBUG_PRINTLN("\tnext free ref " << getRef(nextFreePage,nextFreeIndex) )
      DEBUG_PRINTLN("\tlast free ref " << getRef(lastUsedPage,lastUsedIndex) )
      }

    
  // *****************************************************************************
  // newNode()
  //
  /// \brief Allocates a new graph node.
  ///
  /// The new node is allocated at the end of the last valid page, and if need 
  /// be, a new page is created.
  ///
  /// \return a pointer to the new node
  //
  // *****************************************************************************
  Node*
  Graph::newNode() {

    totalNodes++;

    // does the structural hash table need to be resized?
    if (structuralHashingEnabled && totalNodes*1.0/structuralHashSize > STRUCTURAL_HASH_RESIZE_UTIL) {
      resizeStructuralHash();
    }

    // find the next hole
    Node *page = NULL;
    do {

      if (nextFreeIndex == PAGE_SIZE) {
	nextFreePage++;
	nextFreeIndex = 0;
	if (data.size() <= nextFreePage) {
	  // allocate new page
	  page = new Node[PAGE_SIZE];
	  assert(page);
	  data.push_back(page);
	} else {
	  page = data[nextFreePage];
	}
      } else {
	page = data[nextFreePage];  
      }

    } while (page[nextFreeIndex++].type != Node::NONE);
    --nextFreeIndex;

    lastUsedIndex = nextFreePage > lastUsedPage || nextFreePage == lastUsedPage && nextFreeIndex > lastUsedIndex ? nextFreeIndex : lastUsedIndex;
    lastUsedPage = nextFreePage > lastUsedPage ? nextFreePage : lastUsedPage;
    
    page[nextFreeIndex].self = getRef(nextFreePage, nextFreeIndex);
    assert(getNode(page[nextFreeIndex].self) == &(page[nextFreeIndex]));
    page[nextFreeIndex].refCount = 0;
    page[nextFreeIndex].next = 0;
    return &(page[nextFreeIndex++]);
  }


  // *****************************************************************************
  // resizeStructuralHash()
  //
  /// \brief Resizes the hash table to a fraction of the number of nodes.
  ///
  /// The number of hash entries in the resized table will be the current
  /// number of nodes times the STRUCTURAL_HASH_TARGET_UTIL plus one.
  //
  // *****************************************************************************
  void
  Graph::resizeStructuralHash() {
    // resize memory
    if (structuralHash) free(structuralHash);
    structuralHashSize = (int)(totalNodes*STRUCTURAL_HASH_TARGET_UTIL+1);
    structuralHash = static_cast<Ref *>(calloc(structuralHashSize, sizeof(Ref)));
    assert(structuralHash);

    // rebuild table
    unsigned int page = 0;
    unsigned int index = 0;
    Node *node;
    while( !(index > lastUsedIndex && page == lastUsedPage || page > lastUsedPage) ) {
      if (index == PAGE_SIZE) {
	index = 0;
	page++;
      }
      node = &(data[page][index]);
      if (node->type == Node::AND) {
	node->next = structuralHash[getStructuralHashKey(node->left, node->right)];
	structuralHash[getStructuralHashKey(node->left, node->right)] = node->self;
      }
      index++;
    }
  }

  void
  Graph::updateStructuralHash(Ref x, int oldkey)
  {
    Node *node = getNode(x); 
    assert(node->type == Node::AND);
    const int newkey = getStructuralHashKey(node->left, node->right);
    // update hash 
    // 1) remove from old
    Ref current = structuralHash[oldkey];
    if (current == node->self) {
      structuralHash[oldkey] = node->next;
    } else {
      while(current) {
	Node *currentNode = getNode(current);
	if (currentNode->next == node->self) {
	  currentNode->next = node->next;
	  break;
	}
	current = currentNode->next;
      }
    }
    // 2) add to new
    node->next = structuralHash[newkey];
    structuralHash[newkey] = node->self;
  }

  // *****************************************************************************
  // print()
  //
  /// \brief Prints the full AI graph to standard output.
  //
  // *****************************************************************************
  void                
  Graph::print() const {
    unsigned int page = 0;
    unsigned int index = 0;
    while (1) {
      if (index == PAGE_SIZE) {
	index = 0;
	page++;
      }
      if (getNodeType(getRef(page, index)) != Node::NONE)
	print(getRef(page, index));
      if (index == lastUsedIndex && page == lastUsedPage) break;
      index++;
    }
  }


  // *****************************************************************************
  // print()
  //
  /// \brief Prints information about a node to standard output.
  //
  // *****************************************************************************
  void                
  Graph::print(Ref x) const {
    const int TYPE_COL_WIDTH = 4;
    const int PAGE_COL_WIDTH = 3;
    const int INDEX_COL_WIDTH = 3;
    const int REF_COL_WIDTH = 5;
    const int FANOUT_COL_WIDTH = 3;
    const int EXTERNAL_COL_WIDTH = 3;

    Node *node = getNode(x);
    if (node->type == Node::AND) {
      cout.width(TYPE_COL_WIDTH);
      cout << "AND ";
      cout.width(REF_COL_WIDTH);
      cout << node->self << "   (@";
      cout.width(PAGE_COL_WIDTH);
      cout << (node->self>>1)/PAGE_SIZE << ",";
      cout.width(INDEX_COL_WIDTH);
      cout << (node->self>>1)%PAGE_SIZE << ") : ";
      cout << " fout=";
      cout.width(FANOUT_COL_WIDTH);
      cout << node->fanout.size();
      cout << " ext=";
      cout.width(EXTERNAL_COL_WIDTH);
      cout << node->refCount << " : \t";
      // AND specific
      cout << node->left << " & " << node->right << endl;
    } else if (node->type == Node::SEQUENTIAL) {
      cout.width(TYPE_COL_WIDTH);
      cout << "SEQ ";
      cout.width(REF_COL_WIDTH);
      cout << node->self << "   (@";
      cout.width(PAGE_COL_WIDTH);
      cout << (node->self>>1)/PAGE_SIZE << ",";
      cout.width(INDEX_COL_WIDTH);
      cout << (node->self>>1)%PAGE_SIZE << ") : ";
      cout << " fout=";
      cout.width(FANOUT_COL_WIDTH);
      cout << node->fanout.size();
      cout << " ext=";
      cout.width(EXTERNAL_COL_WIDTH);
      cout << node->refCount << " : \t";
      // SEQ specific
      cout << node->nextState << endl;
    } else if (node->type == Node::CONSTANT0) {
      cout.width(TYPE_COL_WIDTH);
      cout << "ZRO ";
      cout.width(REF_COL_WIDTH);
      cout << node->self << "   (@";
      cout.width(PAGE_COL_WIDTH);
      cout << (node->self>>1)/PAGE_SIZE << ",";
      cout.width(INDEX_COL_WIDTH);
      cout << (node->self>>1)%PAGE_SIZE << ") : ";
      cout << " fout=";
      cout.width(FANOUT_COL_WIDTH);
      cout << "x";
      cout << " ext=";
      cout.width(EXTERNAL_COL_WIDTH);
      cout << node->refCount << " : \t";
      // CONSTANT0 specific
      cout << endl;
    } else if (node->type == Node::TERMINAL) {
      cout.width(TYPE_COL_WIDTH);
      cout << "TRM ";
      cout.width(REF_COL_WIDTH);
      cout << node->self << "   (@";
      cout.width(PAGE_COL_WIDTH);
      cout << (node->self>>1)/PAGE_SIZE << ",";
      cout.width(INDEX_COL_WIDTH);
      cout << (node->self>>1)%PAGE_SIZE << ") : ";
      cout << " fout=";
      cout.width(FANOUT_COL_WIDTH);
      cout << node->fanout.size();
      cout << " ext=";
      cout.width(EXTERNAL_COL_WIDTH);
      cout << node->refCount << " : \t";
      // TRM specific
      cout << node->terminal_driver << endl;
    } else if (node->type == Node::NONE) {
      cout.width(TYPE_COL_WIDTH);
      cout << "NUL ";
      cout.width(REF_COL_WIDTH);
      cout << node->self << "   (@";
      cout.width(PAGE_COL_WIDTH);
      cout << (node->self>>1)/PAGE_SIZE << ",";
      cout.width(INDEX_COL_WIDTH);
      cout << (node->self>>1)%PAGE_SIZE << ") : ";
      cout << " fout=";
      cout.width(FANOUT_COL_WIDTH);
      cout << "x";
      cout << " ext=";
      cout.width(EXTERNAL_COL_WIDTH);
      cout << node->refCount << " : \t";
      // NUL specific
      cout << endl;
    }

  }


  // *****************************************************************************
  // hasCombinationalCycle()
  //
  /// \brief Tests if there are any combinational cycles in the graph.
  ///
  /// A combinational cycle is any cycle in the graph of AND and TERMINAL nodes.
  ///
  /// This function will modify traversal IDs.
  ///
  /// \return true, if one or more combinational cycles exist
  //
  // *****************************************************************************
  bool
  Graph::hasCombinationalCycle() {
    unsigned int startingTravID = newTraversalID();

    unsigned int page = 0;
    unsigned int index = 0;
    while( !(index > lastUsedIndex && page == lastUsedPage || page > lastUsedPage) ) {
      if (index == PAGE_SIZE) {
	index = 0;
	page++;
      }
      newTraversalID();
      Ref ref = getRef(page,index);
      if (hasCombinationalCycle_recursive(ref, startingTravID)) {
	DEBUG_PRINTLN("Combinational cycle on node " << ref);
	return true;
      }
      index++;
    }  
  
    return false;
  }


  // *****************************************************************************
  // hasCombinationalCycle()
  //
  /// \brief Tests if there are any combinational cycles in the graph from a node.
  ///
  /// \return true, if one or more combinational cycles exist
  //
  // *****************************************************************************
  bool
  Graph::hasCombinationalCycle_recursive(Ref x, unsigned int startingTravID) {
    Node *node = getNode(x);
    // is this node non-combinational?
    if (node->type == Node::SEQUENTIAL ||
	node->type == Node::NONE ||
	node->type == Node::CONSTANT0) {
      return false;
    }
    // does this form a cycle?
    if (node->traversalID == currentTraversalID) {
      return true;
    }
    // has this already been proved to be cycle-free?
    if (node->traversalID >= startingTravID) {
      return false;
    }
    node->traversalID = currentTraversalID;
    // otherwise, continue backward
    if (node->type == Node::AND) {
      bool result = hasCombinationalCycle_recursive(node->left, startingTravID) ||
	hasCombinationalCycle_recursive(node->right, startingTravID);
      node->traversalID = startingTravID;
      return result;
    }
    else if (node->type == Node::TERMINAL) {
      bool result = hasCombinationalCycle_recursive(node->terminal_driver, startingTravID);
      node->traversalID = startingTravID;
      return result;
    }

    QUIT_ON_INTERNAL_ERROR;
    return false;
  }


  // *****************************************************************************
  // addToFanout()
  //
  /// \brief Adds a reference to a fanout list.
  ///
  /// The noninverted version of the references is added to the fanout list.
  ///
  /// If fanout is being ignored (MAINTAIN_FANOUT == false), then this call has
  /// no effect.
  ///
  /// \param x the node whose fanout list onto which a reference is appended
  /// \param fanout the reference to append
  //
  // *****************************************************************************
  void
  Graph::addToFanout(Ref x, Ref fanout) {
    if (isNull(x) || isNull(fanout))
      return;

    Node *node = getNode(x);
    ++node->fanoutCount;

    if (MAINTAIN_FANOUT) {
      node->fanout.push_back(getNonInverted(fanout));
    }
  }


  // *****************************************************************************
  // removeFromFanout()
  //
  /// \brief Removes a reference from a fanout list.
  ///
  /// Either the reference or its complement will be removed from the fanout list,
  /// if present.  If either appears more than once in the list, only one copy is
  /// removed.
  ///
  /// If fanout is being ignored (MAINTAIN_FANOUT == false), then this call has
  /// no effect.
  ///
  /// \param x the node whose fanout list onto which a reference is appended
  /// \param fanout the reference to append
  //
  // *****************************************************************************
  void
  Graph::removeFromFanout(Ref x, Ref fanout) {   
    if (isNull(x) || isNull(fanout))
      return;

    Node *node = getNode(x);
    --node->fanoutCount;

    if (MAINTAIN_FANOUT) {
      list<Ref>::iterator it = node->fanout.begin();
      while(it != node->fanout.end()) {
	Ref current = *it;
	if (getNonInverted(current) == getNonInverted(fanout)) {
	  node->fanout.erase(it);
	  return;
	}
	++it;
      }

      // node not found in fanout list?
      cerr << "ERROR: Node " << fanout << " not found in fanout of " << x << endl;
      QUIT_ON_INTERNAL_ERROR;
    }
  }


  // *****************************************************************************
  // isAnd()
  //
  /// \brief Tests if a reference is to an AND node.
  ///
  /// \param x
  /// \return true, if the reference is to an AND node
  //
  // *****************************************************************************
  bool
  Graph::isAnd(Ref x) const { 
    Node *node = getNode(x);
    assert(node);
    return (node->type == Node::AND);
  }


  // *****************************************************************************
  // isTerminal()
  //
  /// \brief Tests if a reference is to a TERMINAL node.
  ///
  /// \param x
  /// \return true, if the reference is to a TERMINAL node
  //
  // *****************************************************************************
  bool
  Graph::isTerminal(Ref x) const { 
    Node *node = getNode(x);
    assert(node);
    return (node->type == Node::TERMINAL);
  }


  // *****************************************************************************
  // setAndRight()
  //
  /// \brief Sets the right operand of an AND node.
  ///
  /// The graph is marked as dirty.
  ///
  /// \param x the AND node to be modified
  /// \param right the new right operand
  //
  // *****************************************************************************
  void
  Graph::setAndRight(Ref x, Ref right) { 
    Node *node = getNode(x);
    assert(node);
    assert(node->type == Node::AND);

    // mark dirty; existing graph structure may be modified
    dirtyMarker++;

    // decrement old fan-in
    removeFromFanout(node->right, x);
    // update right operand
    const int oldkey = getStructuralHashKey(node->left, node->right);
    node->right = right;
    updateStructuralHash(node->self, oldkey);
    // increment new fan-in
    addToFanout(right, x);
  }


  // *****************************************************************************
  // setAndLeft()
  //
  /// \brief Sets the left operand of an AND node.
  ///
  /// The graph is marked as dirty.
  ///
  /// \param x the AND node to be modified
  /// \param left the new left operand
  //
  // *****************************************************************************
  void
  Graph::setAndLeft(Ref x, Ref left) { 
    Node *node = getNode(x);
    assert(node);
    assert(node->type == Node::AND);

    // mark dirty; existing graph structure may be modified
    dirtyMarker++;

    // decrement old fan-in
    removeFromFanout(node->left, x);
    // update left operand
    const int oldkey = getStructuralHashKey(node->left, node->right);
    node->left = left;
    updateStructuralHash(node->self, oldkey);
    // increment new fan-in
    addToFanout(left, x);
  }


  // *****************************************************************************
  // setTerminalDriver()
  //
  /// \brief Sets the driver of a TERMINAL node.
  ///
  /// The graph is marked as dirty.
  ///
  /// \param terminal an reference to the TERMINAL node
  /// \param driver a reference to new driver of the terminal
  //
  // *****************************************************************************
  void
  Graph::setTerminalDriver(Ref terminal, Ref driver) { 
    Node *node = getNode(terminal);
    assert(node);
    assert(node->type == Node::TERMINAL);

    // mark dirty; existing graph structure may be modified
    dirtyMarker++;

    // decrement old fan-in
    removeFromFanout(node->terminal_driver, terminal);
    // update terminal driver
    node->terminal_driver = driver;
    // increment new fan-in
    addToFanout(driver, terminal);
  }


  // *****************************************************************************
  // isSequential()
  //
  /// \brief Tests if a reference is to a SEQUENTIAL node.
  ///
  /// \param x
  /// \return true, if this is a reference to a sequential node
  //
  // *****************************************************************************
  bool
  Graph::isSequential(Ref x) const { 
    Node *node = getNode(x);
    assert(node);
    return (node->type == Node::SEQUENTIAL);
  }


  // *****************************************************************************
  // getSequentialData()
  //
  /// \brief Returns the annotated sequention functional description for a SEQUENTIAL node.
  ///
  /// \param sequential
  /// \return sequential functional description, or NULL if it doesn't exist
  //
  // *****************************************************************************
  Node::SequentialData*
  Graph::getSequentialData(Ref sequential) { 
    Node *node = getNode(sequential);
    assert(node);
    assert(node->type == Node::SEQUENTIAL);
    if (!node->sequentialData) {
      return (node->sequentialData = new Node::SequentialData(Node::SequentialData::BLACK_BOX));
    }
    return node->sequentialData;    
  }


  // *****************************************************************************
  // getNextState()
  //
  /// \brief Returns the next state input for a SEQUENTIAL node.
  ///
  /// \param sequential
  /// \return the next state function
  //
  // *****************************************************************************
  Ref
  Graph::getNextState(Ref sequential) const { 
    Node *node = getNode(sequential);
    assert(node);
    assert(node->type == Node::SEQUENTIAL);
    return node->nextState;
  }


  // *****************************************************************************
  // getExternalTerminalConnection()
  //
  /// \brief Returns the external terminal connection pointer for a TERMINAL node.
  ///
  /// \param terminal
  /// \return the external terminal connection pointer
  //
  // *****************************************************************************
  void *
  Graph::getExternalTerminalConnection(Ref terminal) const {
    Node *node = getNode(terminal);
    assert(node);
    assert(node->type == Node::TERMINAL);
    return node->external_terminal_connection;
  }


  // *****************************************************************************
  // setExternalTerminalConnection()
  //
  /// \brief Sets the external terminal connection pointer for a TERMINAL node.
  ///
  /// \param terminal
  /// \param connection
  /// \return the external terminal connection pointer
  //
  // *****************************************************************************
  void
  Graph::setExternalTerminalConnection(Ref terminal, void *connection) {
    Node *node = getNode(terminal);
    assert(node);
    assert(node->type == Node::TERMINAL);
    node->external_terminal_connection = connection;
  }

    
  // *****************************************************************************
  // setNextState()
  //
  /// \brief Sets the next state input for a SEQUENTIAL node.
  ///
  /// The graph is marked as dirty.
  ///
  /// \param sequential
  /// \param nextState the new next state function
  //
  // *****************************************************************************
  void
  Graph::setNextState(Ref sequential, Ref nextState) { 
    Node *node = getNode(sequential);
    assert(node);
    assert(node->type == Node::SEQUENTIAL);

    // mark dirty; existing graph structure may be modified
    dirtyMarker++;

    // decrement old fan-in
    removeFromFanout(node->nextState, node->self);
    // update next state reference
    node->nextState = nextState;
    // increment new fan-in
    addToFanout(nextState, node->self);
  }


  // *****************************************************************************
  // enumerateKfeasibleCuts()
  //
  /// \brief Enumerates the k-feasible graph cuts of a node.
  ///
  /// A CutSet is allocated and returned.  It is also stored for future
  /// use, along with all of the CutSets of the other nodes used to
  /// generate it.  The user should not deallocate this memory.  If the CutSets
  /// are no longer needed or CutSets of a different width are needed,
  /// the stored information can be cleared with clearKfeasibleCuts().
  ///
  /// Entirely non-inverted graph edges will be returned in the cut.
  ///
  /// If maxCuts < 0, all cuts will be returned.  Otherwise, the set
  /// of cuts will be restricted to |maxCuts|.  The ordering (both
  /// with and without a maximum number) will be: self cuts, empty
  /// cuts, each cut from the left node in order times each cut
  /// from the right node in order.
  ///
  /// If includeConstantNode is false, then no cuts that include the
  /// constant node will be returned (it will always be subsumed into
  /// the logic past the cut).  If it is true, both cuts with the
  /// constant node subsumed and with it explicit in the cut will
  /// be returned.
  ///
  /// To do: ideally, when the number of cuts is limited, the 
  /// cross-product should give equal weighting to the left and
  /// right predecessor (instead of favoring the left).
  ///
  /// References:
  /// \li A. Mishchenko, S. Chatterjee, R. Brayton, X. Wang, T. Kam.  
  /// Technology Mapping with Boolean Matching, Supergates and Choices. 
  /// Submitted to DAC 05.
  ///
  /// \param x a graph node
  /// \param maxCutWidth the bound on the width of the cut
  /// \param maxCutCount the maximum number of cuts to return
  /// \param maxCutDepth the maximum levels backward to search
  /// \param includeConstantNode should the constant node ever be included
  /// \return the set of K-feasible for the graph node
  //
  // *****************************************************************************
  Graph::CutSet*
  Graph::enumerateKfeasibleCuts(Ref x, unsigned int maxCutWidth, 
				int maxCutCount, int maxCutDepth,
				bool includeConstantNode) {

    Node *node = getNode(x);
    
    // has this cutset been precomputed?
    if (cutHash.find(node) != cutHash.end()) {
      return cutHash[node];
    }
    
    CutSet *cuts = new CutSet;

    if (maxCutDepth != 0) {
      switch(node->type) {
      case Node::AND: {
	// compute k-feasible cuts for right and left subgraphs
	CutSet *leftCuts = enumerateKfeasibleCuts(node->left, maxCutWidth, 
						  maxCutCount, maxCutDepth-1, 
						  includeConstantNode),
	  *rightCuts = enumerateKfeasibleCuts(node->right, maxCutWidth,
					      maxCutCount, maxCutDepth-1, 
					      includeConstantNode);
	// merge cut sets
	Cut mergeCut;
	for(CutSet::iterator leftCutIter = leftCuts->begin(); 
	    leftCutIter != leftCuts->end() && 
	      (maxCutCount < 0 || cuts->size() < static_cast<unsigned int>(maxCutCount)); 
	    ++leftCutIter) {
	  for(CutSet::iterator rightCutIter = rightCuts->begin(); 
	      rightCutIter != rightCuts->end() && 
		(maxCutCount < 0 || cuts->size() < static_cast<unsigned int>(maxCutCount));
	      ++rightCutIter) {
	    mergeCut.clear();
	    set_union(leftCutIter->begin(), leftCutIter->end(),
		      rightCutIter->begin(), rightCutIter->end(),
		      insert_iterator<Cut>(mergeCut, mergeCut.begin()));
	    if (mergeCut.size() <= maxCutWidth) {
	      cuts->push_front(mergeCut);
	    }
	  }
	}
	break;
      }
      case Node::TERMINAL:
	if (!isNull(node->terminal_driver)) {
	  return enumerateKfeasibleCuts(node->terminal_driver, maxCutWidth,
					maxCutCount, maxCutDepth-1, 
					includeConstantNode);
	}
      default:
	break;
      }
    }

    // handle constant node
    if (node->type == Node::CONSTANT0) {
      set<Ref> emptyCut;
      cuts->push_front(emptyCut);
      // if !includeConstantNode, return only an empty cut
      if (!includeConstantNode) {
	cutHash[node] = cuts;
	return cuts;
      }
    }

    set<Ref> selfCut;
    selfCut.insert(node->self);
    cuts->push_front(selfCut);
    
    // hash cut set
    cutHash[node] = cuts;

    return cuts;
  }


  // *****************************************************************************
  // clearKfeasibleCuts()
  //
  /// \brief Clears the hashed sets stored by previous calls to enumerate cutsets.
  //
  // *****************************************************************************
  void
  Graph::clearKfeasibleCuts() {
    for(map<Node*, CutSet*>::iterator hashIter = cutHash.begin(); 
	hashIter != cutHash.end();
	++hashIter) {
      delete hashIter->second;
    }
    cutHash.clear();
  }


  // *****************************************************************************
  // testEquivalent()
  //
  /// \brief Tests if two references are explicitly marked as equivalent.
  ///
  /// This does not test for structural isomorphism or functional equivalence.
  /// It tests if the application has explicitly marked the two nodes as
  /// equivalent using the setEquivalent method.
  //
  // *****************************************************************************
  bool
  Graph::testEquivalent(Ref x, Ref y) const {

    // trivial cases
    if (x == y) {
      return true;    
    }
    if (x == notOf(y)) {
      return false;
    }

    // XXX: maintain a circular list in the first place!
    Node *node = getNode(x);
    if (isNull(node->equivalent)) {
      // x isn't equivalent to anything
      return false;
    }

    // follow the circular list around until we reach the start
    Ref current = node->equivalent;
    while (getNonInverted(current) != getNonInverted(x)) {

      // if at any point we encounter a reference to y,
      // the two nodes have been marked equivalent
      if (current == y) {
	return true;
      }
        
      node = getNode(current);
      current = isInverted(current) ? notOf(node->equivalent) : node->equivalent;
        
      assert(!isNull(current));
    }

    // error if we've been marked equivalent to our inverse
    assert(isInverted(current) == isInverted(x));
    
    return false;
  }


  // *****************************************************************************
  // testEquivalence()
  //
  /// \brief DEPRECATED. Use testEquivalent instead.
  ///
  // *****************************************************************************
  bool
  Graph::testEquivalence(Ref x, Ref y) const {
    return testEquivalent(x, y);
  }


  // *****************************************************************************
  // setEquivalent()
  //
  /// \brief Marks two references as being functionally equivalent.
  ///
  /// To prevent obfuscation, TERMINAL nodes can not be marked as equivalent.
  //
  // *****************************************************************************
  void
  Graph::setEquivalent(Ref x, Ref y) {

    // are the two nodes already marked as equivalent?
    if (testEquivalent(x,y))
      return;

    // two functions can not be both equivalent and inversely equivalent
    assert(!testEquivalent(x, notOf(y)));
       
    Node *xNode = getNode(x),
      *yNode = getNode(y);
    assert(!(xNode->type == Node::TERMINAL || xNode->type == Node::NONE));
    assert(!(yNode->type == Node::TERMINAL || yNode->type == Node::NONE));
    Ref   yNonInverted = getNonInverted(y),
      xNonInverted = getNonInverted(x);

    // create self-loops if necessary
    if (isNull(xNode->equivalent)) {
      xNode->equivalent = x;
    }
    if (isNull(yNode->equivalent)) {
      yNode->equivalent = y;
    }

    // Splice into circular linked list. Note that we in general are merging 
    // two linked lists rather than adding a single element.
    Ref xx = getNode(x)->equivalent;
    Ref yy = getNode(y)->equivalent;
    getNode(x)->equivalent = notCondOf(yy, isInverted(x)^isInverted(y));
    getNode(y)->equivalent = notCondOf(xx, isInverted(x)^isInverted(y));
  }


  // *****************************************************************************
  // removeEquivalent()
  //
  /// \brief Removes all equivalences to/from a reference and its inverse.
  ///
  /// This function should always be invoked when the graph structure is 
  /// modified such that the functionality of this reference is changed, 
  /// relative to other nodes that may be equivalent to it.
  //
  // *****************************************************************************
  void
  Graph::removeEquivalent(Ref x) {
    Node *node = getNode(x);
    if (isNull(node->equivalent)) {
      // x isn't equivalent to anything
      return;
    }
    
    // follow the circular list to find the reference to x
    while (getNonInverted(node->equivalent) != getNonInverted(x)) {
        
      node = getNode(node->equivalent);
        
      assert(!isNull(node->equivalent));
    }

    // move the reference to x to the reference of x, 
    // splicing x out of the equivalence list
    node->equivalent = isInverted(node->equivalent) ? notOf(getNode(x)->equivalent) : getNode(x)->equivalent;

    // mark x as not equivalent to anything
    getNode(x)->equivalent = getNull();
  }


  // *****************************************************************************
  // removeEquivalences()
  //
  /// \brief DEPRECATED.  Use removeEquivalent instead.
  //
  // *****************************************************************************
  void
  Graph::removeEquivalences(Ref x) {
    removeEquivalent(x);
  }


  // *****************************************************************************
  // resubsitute()
  //
  /// \brief Resubstitutes one node for all fan-outs.
  ///
  /// This function takes all of the references to a node and 
  /// replaces them with another.
  ///
  /// The replacement will occur when the original appears in the following
  /// situations: in the nextState input of a SEQUENTIAL node, as the driver
  /// of a TERMINAL node, or as the left or right input of an AND node.
  /// Control signals of a SEQUENTIAL node referring to the original will not 
  /// be replaced.
  ///
  /// Inverted references to the original node will be replaced with inverted
  /// references to the replacement.
  ///
  /// The graph is marked as dirty.
  ///
  /// \param original
  /// \param replacement
  //
  // *****************************************************************************
  void
  Graph::resubstitute(Ref original, Ref replacement) {
    Node *originalNode = getNode(original);
    
    // mark the graph as dirty
    if (!originalNode->fanout.empty()) dirtyMarker++;

    // get fanout of target node, and reconnect to replacement node
    while(!originalNode->fanout.empty()) {
      Node *targetNode = getNode(originalNode->fanout.front());

      switch(targetNode->type) {
      case Node::SEQUENTIAL:
	if (targetNode->nextState == original) {
	  targetNode->self = replacement;
	} else if (targetNode->nextState == notOf(original)) {
	  targetNode->nextState = notOf(replacement);
	}
	break;
      case Node::TERMINAL:
	if (targetNode->terminal_driver == original) {
	  targetNode->terminal_driver = replacement;
	} else if (targetNode->terminal_driver == notOf(original)) {
	  targetNode->terminal_driver = notOf(replacement);
	}
	break;
      case Node::AND: {
	const int oldkey = getStructuralHashKey(targetNode->left, targetNode->right);
	if (targetNode->left == original) {
	  targetNode->left = replacement;
	} else if (targetNode->left == notOf(original)) {
	  targetNode->left = notOf(replacement);
	}
	if (targetNode->right == original) {
	  targetNode->right = replacement;
	} else if (targetNode->right == notOf(original)) {
	  targetNode->right = notOf(replacement);
	}
	updateStructuralHash(targetNode->self, oldkey);
	break;
      }
      default:
	QUIT_ON_INTERNAL_ERROR;
      }

      // move from original to replacement node fanout list
      originalNode->fanout.pop_front();
      --originalNode->fanoutCount;
      addToFanout(replacement, targetNode->self);
    }

  }


  // *****************************************************************************
  // resubsitute()
  //
  /// \brief Resubstitutes one node for another in one node.
  ///
  /// This function takes one reference to a node and replaces it with a 
  /// reference to another.
  ///
  /// The replacement will occur when the original appears in the following
  /// situation: in the nextState input of a SEQUENTIAL node, as the driver
  /// of a TERMINAL node, or as the left or right input of an AND node.
  /// Control signals of a SEQUENTIAL node referring to the original will 
  /// not be replaced.
  ///
  /// Inverted references to the original node will be replaced with inverted
  /// references to the replacement.
  ///
  /// The graph is marked as dirty.
  ///
  /// \param target the node whose reference will be changed
  /// \param original the node to which the reference will be replaced
  /// \param replacement the new node to which the reference will refer
  //
  // *****************************************************************************
  void
  Graph::resubstitute(Ref original, Ref replacement, Ref target) {
    Node *targetNode = getNode(target);

    switch(targetNode->type) {
    case Node::SEQUENTIAL:
      if (targetNode->nextState == original) {
	targetNode->nextState = replacement;
	dirtyMarker++;
      } else if (targetNode->nextState == notOf(original)) {
	targetNode->nextState = notOf(replacement);
	dirtyMarker++;        
      }
      break;
    case Node::TERMINAL:
      if (targetNode->terminal_driver == original) {
	targetNode->terminal_driver = replacement;
	dirtyMarker++;
      } else if (targetNode->terminal_driver == notOf(original)) {
	targetNode->terminal_driver = notOf(replacement);
	dirtyMarker++;
      }
      break;
    case Node::AND: {
      const int oldkey = getStructuralHashKey(targetNode->left, targetNode->right);
      if (targetNode->left == original) {
	targetNode->left = replacement;
	dirtyMarker++;
      } else if (targetNode->left == notOf(original)) {
	targetNode->left = notOf(replacement);
	dirtyMarker++;
      }
      if (targetNode->right == original) {
	targetNode->right = replacement;
	dirtyMarker++;
      } else if (targetNode->right == notOf(original)) {
	targetNode->right = notOf(replacement);
	dirtyMarker++;
      }
      updateStructuralHash(targetNode->self, oldkey);
      break;
    }
    default:
      QUIT_ON_INTERNAL_ERROR;
    }

    // move from original to replacement node fanout list
    removeFromFanout(original, targetNode->self);
    addToFanout(replacement, targetNode->self);
  }


  // *****************************************************************************
  // detach()
  //
  /// \brief Detaches a node from its fan-in and fan-out.
  ///
  /// The inputs of this node are reset to null, its fan-outs are cleared, and all
  /// references to it are also reset to null.
  ///
  /// In conjunction with clearExternalReferences(), this function is useful
  /// for marking a node as ready for removal.  Doing both actions will guarantee its
  /// garbage collection, though no memory is deallocated here.
  ///
  /// \param x
  //
  // *****************************************************************************
  void
  Graph::detach(Ref x) {

    Node *node = getNode(x);
    switch(node->type) {
    case Node::SEQUENTIAL:
      setNextState(x, getNull());
      break;
    case Node::TERMINAL:
      setTerminalDriver(x, getNull());
      break;
    case Node::AND:
      setAndLeft(x, getNull());
      setAndRight(x, getNull());
      break;
    case Node::CONSTANT0:
      break;
    default:
      QUIT_ON_INTERNAL_ERROR;
    }
    
    resubstitute(x, getNull());
  }


  // *****************************************************************************
  // chooseEquivalent()
  //
  /// \brief Resubstitutes a chosen node for all of its equivalent nodes.
  ///
  /// Each of several nodes which are marked as equivalent may have fan-out.
  /// This function takes all of the references to the equivalent nodes and 
  /// reconnects them to only one of the equivalent nodes.
  ///
  /// The graph will not be marked as dirty.
  ///
  /// \param x
  //
  // *****************************************************************************
  void
  Graph::chooseEquivalent(Ref x) {

    Ref chosenRef = getNonInverted(x);
    Node *chosenNode = getNode(chosenRef);
    if (isNull(chosenNode->equivalent)) {
      // x isn't equivalent to anything
      return;
    }

    // keep the old dirty marker
    int oldDirtyMarker = dirtyMarker;

    // follow the circular list around until we reach the start
    Ref equivRef = chosenNode->equivalent;
    while (getNonInverted(equivRef) != getNonInverted(chosenRef)) {
      Node *equivNode = getNode(equivRef);
      assert(equivNode);
      resubstitute(equivRef, chosenRef);
      equivRef = isInverted(equivRef) ? notOf(equivNode->equivalent) : equivNode->equivalent;        
      assert(!isNull(equivRef));
    }

    // keep the old dirty marker
    dirtyMarker = oldDirtyMarker;
  }


  // *****************************************************************************
  // getEquivalent()
  //
  /// \brief Returns all references that have been marked equivalent.
  ///
  /// The set of equivalent references will be appended to the provided list.
  /// The list will include the reference provided.
  ///
  /// \param x
  /// \param l
  //
  // *****************************************************************************
  void
  Graph::getEquivalent(Ref x, list<Ref> & l) const {
    l.push_back(x);
    
    Node *node = getNode(x);
    if (isNull(node->equivalent)) {
      // x isn't equivalent to anything
      return;
    }
    
    // follow the circular list around until we reach the start
    Ref current = node->equivalent;
    while (getNonInverted(current) != getNonInverted(x)) {
      l.push_back(current);
        
      node = getNode(current);
      current = isInverted(current) ? notOf(node->equivalent) : node->equivalent;
        
      assert(!isNull(current));
    }
  }


  // *****************************************************************************
  // getEquivalents()
  //
  /// \brief DEPRECATED. Use getEquivalent instead.
  //
  // *****************************************************************************
  void
  Graph::getEquivalents(Ref x, list<Ref> & l) const {
    getEquivalent(x, l);
  }


  // *****************************************************************************
  // getFanoutOfEquivalentNodes()
  //
  /// \brief Returns the fanout of the references node and all nodes marked equivalent to it.
  ///
  /// The fanouts of this node and all marked equivalents of it are appended 
  /// to the provided list. If two equivalent nodes fanout to the same node, 
  /// that node will appear twice in the fanout list.
  ///
  /// An error is generated if fanout is not being maintained.
  ///
  /// \param x
  /// \param l the list onto which to append the fanout references
  //
  // *****************************************************************************
  void
  Graph::getFanoutOfEquivalentNodes(Ref x, list<Ref> & l) const {
    assert(MAINTAIN_FANOUT);

    list<Ref> equivalentNodes;
    getEquivalent(x, equivalentNodes);
    
    for(list<Ref>::iterator it = equivalentNodes.begin(); it != equivalentNodes.end(); ++it) {
      list<Ref> singleFanout(getFanout(*it));
      // copy the fanout list of getFanout, as the result is modified

      singleFanout.sort();        
      l.merge(singleFanout);
    }
  }


  // *****************************************************************************
  // isDirty()
  //
  /// \brief Compares the state of the current graph against a previous state.
  ///
  /// In conjunction with the getDirtyMarker() function, this can be used to
  /// identify if any changes have been made to the graph that may affect
  /// the functionality of any external references.  If there have been
  /// any structural changes to the graph since the marker provided from
  /// a previous call to the getDirtyMarker() function, it will differ from the
  /// current marker and true will be returned.
  ///
  /// The marker will be incremented if the input of any pre-existing node is
  /// changed by an external call (i.e. a direct set or a resubstitution).  
  /// It will not be changed in the course of equivalence marking or rehashing.
  /// 
  /// \param marker the dirty marker against which to compare
  /// \return true, if the function of any existing reference may have changed
  //
  // *****************************************************************************
  bool
  Graph::isDirty(unsigned int marker) const {
    return marker == dirtyMarker;
  }


  // *****************************************************************************
  // getDirtyMarker()
  //
  /// \brief Returns a marker to check against a future graph against for change.
  //
  // *****************************************************************************
  unsigned int
  Graph::getDirtyMarker() const {
    return dirtyMarker;
  }


  // *****************************************************************************
  // getTransitiveFanin()
  //
  /// \brief Returns a vector of all nodes in the transitive fan-in.
  ///
  /// The nodes in the transitive fan-in are appended to the provided list
  /// in forward topological order.  The initial node is not included.  
  /// All references will be non-inverted.
  ///
  /// The backward traversal will terminate at terminal nodes
  /// that point to a null reference, the constant zero node, and sequential
  /// nodes (if the parameter crossSequential is false).  If the parameter
  /// includeRoots is true, these root nodes will be included in the result.
  ///
  /// If the initial  reference is to a sequential node, the transitive fan-in 
  /// of its nextState input will be returned.
  ///
  /// This function will modify traversal IDs.
  ///
  /// Cycles are permitted.
  ///
  /// \param x the vertex of the fan-in cone
  /// \param transitiveFanin the vector onto which to append the results
  /// \param includeRoots true if root nodes should be included
  /// \param crossSequential true if the traversal should cross sequential nodes
  //
  // *****************************************************************************
  void
  Graph::getTransitiveFanin(Ref x, vector<Ref> &transitiveFanin, 
			    bool includeRoots, bool crossSequential) {
    newTraversalID();
    getTransitiveFanin_recursive(x, transitiveFanin, includeRoots, crossSequential);
    // remove initial node
    transitiveFanin.pop_back(); 
  }


  // *****************************************************************************
  // getTransitiveFanin()
  //
  /// \brief Returns a vector of all nodes in the transitive fan-in of several nodes.
  ///
  /// The nodes in the transitive fan-in are appended to the provided list
  /// in forward topological order.  All references will be non-inverted.
  ///
  /// The initial nodes will not be included unless they themselves lie in the
  /// fan-in of another initial node.
  ///
  /// The backward traversal will terminate at terminal nodes
  /// that point to a null reference, the constant zero node, and sequential
  /// nodes (if the parameter crossSequential is false).  If the parameter
  /// includeRoots is true, these root nodes will be included in the result.
  ///
  /// If the initial  reference is to a sequential node, the transitive fan-in 
  /// of its nextState input will be returned.
  ///
  /// This function will modify traversal IDs.
  ///
  /// Cycles are permitted.
  ///
  /// \param x the vertices of the fan-in cone
  /// \param transitiveFanin the vector onto which to append the results
  /// \param includeRoots true if root nodes should be included
  /// \param crossSequential true if the traversal should cross sequential nodes
  //
  // *****************************************************************************
  void
  Graph::getTransitiveFanin(list<Ref> x, vector<Ref> &transitiveFanin, 
			    bool includeRoots, bool crossSequential) {

    // transitiveFanin = { transitiveFanin, fanin(x_1), fanin(x_2) ... fanin(x_n) }
  
    newTraversalID();
    for(list<Ref>::iterator xIter = x.begin(); xIter != x.end(); xIter++) {
      getTransitiveFanin_recursive(*xIter, transitiveFanin, 
				   includeRoots, crossSequential);
      // remove initial nodes
      transitiveFanin.pop_back();
    }
  }


  // *****************************************************************************
  // getTransitiveFanin_recursive()
  //
  /// \brief Internal function.
  ///
  /// The result is returend in NORMAL forward topological order.
  //
  // *****************************************************************************
  void
  Graph::getTransitiveFanin_recursive(Ref x, vector<Ref> &transitiveFanin,
				      bool includeRoots, bool crossSequential) {
    x = getNonInverted(x);
    markVisited(x);
      
    Ref   fanin1, fanin2 = getNull();
    Node *node = getNode(x);
    switch(node->type) {
    case Node::AND:
      fanin1 = node->left;
      fanin2 = node->right;
      break;
    case Node::TERMINAL:
      fanin1 = node->terminal_driver;
      break;
    case Node::SEQUENTIAL:
      fanin1 = node->nextState;
      break;
    case Node::CONSTANT0:
      return;
    default:
      QUIT_ON_INTERNAL_ERROR;
    }

    if (!isNull(fanin1) && !isVisited(fanin1)) {
      if (isTerminal(fanin1) && isNull(getTerminalDriver(fanin1)) 
	  || getNonInverted(fanin1) == constantZero()
	  || isSequential(fanin1) && !crossSequential) {
	// only include if includeRoots is true
	if (includeRoots) {
	  markVisited(fanin1);
	  transitiveFanin.push_back(fanin1);
	}
      } else {
	getTransitiveFanin_recursive(fanin1, transitiveFanin, 
				     includeRoots, crossSequential);
      }
    }

    if (!isNull(fanin2) && !isVisited(fanin2)) {
      if (isTerminal(fanin2) && isNull(getTerminalDriver(fanin2)) 
	  || getNonInverted(fanin2) == constantZero()
	  || isSequential(fanin2) && !crossSequential) {
	// only include if includeRoots is true
	if (includeRoots) {
	  markVisited(fanin2);
	  transitiveFanin.push_back(fanin2);
	}
      } else {
	getTransitiveFanin_recursive(fanin2, transitiveFanin, 
				     includeRoots, crossSequential);
      }
    }

    transitiveFanin.push_back(x);
  }


  // *****************************************************************************
  // getTransitiveFanout()
  //
  /// \brief Returns a vector of all nodes in the transitive fan-out.
  ///
  /// The nodes in the transitive fan-in are appended to the provided vector
  /// in forward topological order. The initial node is not included.  
  /// All references will be non-inverted.
  ///
  /// The forward traversal will terminate at sequential nodes (if the
  /// crossSequential parameter is false) and nodes that
  /// do not have any fan-out.  If includesRoots is false, the root
  /// nodes will not be included in the result.
  ///
  /// If the initial reference is to a sequential node,
  /// the traversal will continue along its fanout and not terminate immediately.
  ///
  /// If fan-out is not being maintained, the function returns immediately
  /// without modifying the list.
  ///
  /// This function will modify traversal IDs.
  ///
  /// Cycles are permitted.
  ///
  /// \param x the vertex of the fan-out cone
  /// \param transitiveFanout the vector onto which to append the results
  /// \param includeRoots true if root nodes should be included
  /// \param crossSequential true if the traversal should cross sequential nodes
  //
  // *****************************************************************************
  void
  Graph::getTransitiveFanout(Ref x, vector<Ref> &transitiveFanout, 
			     bool includeRoots, bool crossSequential) {
    newTraversalID();
    // generate the reversed forward topological ordering of transitive fanout
    vector<Ref> newReversedTransitiveFanout;
    getTransitiveFanout_recursive(x, newReversedTransitiveFanout, includeRoots, crossSequential);
    // remove initial node
    newReversedTransitiveFanout.pop_back(); 
    // append reversed result to original vector
    transitiveFanout.reserve(transitiveFanout.size()+newReversedTransitiveFanout.size());
    for(vector<Ref>::reverse_iterator it = newReversedTransitiveFanout.rbegin();
	it != newReversedTransitiveFanout.rend(); it++) {
      transitiveFanout.push_back(*it);
    }
  }


  // *****************************************************************************
  // getTransitiveFanout()
  //
  /// \brief Returns a vector of all nodes in the transitive fan-out of several nodes.
  ///
  /// The nodes in the transitive fan-in are appended to the provided vector
  /// in forward topological order.  All references will be non-inverted.
  ///
  /// The initial nodes will not be included unless they themselves lie in the
  /// fan-in of another initial node.
  ///
  /// The forward traversal will terminate at sequential nodes (if the
  /// crossSequential parameter is false) and nodes that
  /// do not have any fan-out.  If includesRoots is false, the root
  /// nodes will not be included in the result.
  ///
  /// If the initial reference is to a sequential node,
  /// the traversal will continue along its fanout and not terminate immediately.
  ///
  /// If fan-out is not being maintained, the function returns immediately
  /// without modifying the list.
  ///
  /// This function will modify traversal IDs.
  ///
  /// Cycles are permitted.
  ///
  /// \param x the vertex of the fan-out cone
  /// \param transitiveFanout the vector onto which to append the results
  /// \param includeRoots true if root nodes should be included
  /// \param crossSequential true if the traversal should cross sequential nodes
  //
  // *****************************************************************************
  void
  Graph::getTransitiveFanout(list<Ref> x, vector<Ref> &transitiveFanout,
			     bool includeRoots, bool crossSequential) {

    // transitiveFanout = { transitiveFanout, rev(fanout(x_n)) ... rev(fanout(2)), rev(fanout(1)) }

    newTraversalID();
    vector<Ref> newReversedTransitiveFanout;
    for(list<Ref>::iterator it = x.begin(); it != x.end(); it++) {
      // generate the reversed forward topological ordering of transitive fanout
      getTransitiveFanout_recursive(*it, newReversedTransitiveFanout,
				    includeRoots, crossSequential);
      // remove initial node
      newReversedTransitiveFanout.pop_back(); 
    }
    // append reversed result to original vector
    transitiveFanout.reserve(transitiveFanout.size()+newReversedTransitiveFanout.size());
    for(vector<Ref>::reverse_iterator it2 = newReversedTransitiveFanout.rbegin();
	it2 != newReversedTransitiveFanout.rend(); it2++) {
      transitiveFanout.push_back(*it2);
    }
  }


  // *****************************************************************************
  // getTransitiveFanout_recursive()
  //
  /// \brief Internal function.
  ///
  /// NOTE: Because the result is returned in a vector and the efficiencies of
  /// adding to the front and back are not symmetrical, the ordering of the
  /// result is in REVERSE forward topological order.
  //
  // *****************************************************************************
  void
  Graph::getTransitiveFanout_recursive(Ref x, vector<Ref> &transitiveFanout, 
				       bool includeRoots, bool crossSequential) {
    x = getNonInverted(x);
    markVisited(x);

    if (getNodeType(x) == Node::TERMINAL && 
	hasFanout(x) && !includeRoots) {
      // only include if includeRoots is true
      return;
    }

    const list<Ref> *immediateFanout = &getFanout(x);
    for(list<Ref>::const_iterator it = immediateFanout->begin(); 
	it != immediateFanout->end(); ++it) {
      Ref next = getNonInverted(*it);

      // there shouldn't be fanout to a null or constant node
      assert(!isNull(next));
      assert(getNodeType(next) != Node::CONSTANT0);

      // have we already explored this node?
      if (isVisited(next)) {
	continue;
      }
            
      if (getNodeType(next) == Node::SEQUENTIAL && !crossSequential) {
	// only include if includeRoots is true
	if (includeRoots) {
	  markVisited(next);
	  transitiveFanout.push_back(next);
	}
      } else {
	// continue traversal
	getTransitiveFanout_recursive(next, transitiveFanout, includeRoots);
      }
    }

    transitiveFanout.push_back(x);
  }


  // *****************************************************************************
  // getTransitiveFanin()
  //
  /// \brief Returns a list of all nodes in the transitive fan-in of a node.
  ///
  /// The nodes in the transitive fan-in are appended to the provided list
  /// in forward topological order.  The initial node is not included.  
  /// All references will be non-inverted.
  ///
  /// The backward traversal will terminate at terminal nodes
  /// that point to a null reference, the constant zero node, and sequential
  /// nodes (if the parameter crossSequential is false).  If the parameter
  /// includeRoots is true, these root nodes will be included in the result.
  ///
  /// If the initial  reference is to a sequential node, the transitive fan-in 
  /// of its nextState input will be returned.
  ///
  /// This function will modify traversal IDs.
  ///
  /// Cycles are permitted.
  ///
  /// \param x the vertex of the fan-in cone
  /// \param transitiveFanin the list onto which to append the results
  /// \param includeRoots true if root nodes should be included
  /// \param crossSequential true if the traversal should cross sequential nodes
  //
  // *****************************************************************************
  void
  Graph::getTransitiveFanin(Ref x, list<Ref> &transitiveFanin,
			    bool includeRoots, bool crossSequential) {
    newTraversalID();
    getTransitiveFanin_recursive(x, transitiveFanin, includeRoots, crossSequential);
    // remove initial node
    transitiveFanin.pop_back(); 
  }


  // *****************************************************************************
  // getTransitiveFanin()
  //
  /// \brief Returns a list of all nodes in the transitive fan-in of several nodes.
  ///
  /// The nodes in the transitive fan-in are appended to the provided list
  /// in forward topological order.  The initial node is not included.  
  /// All references will be non-inverted.
  ///
  /// The backward traversal will terminate at terminal nodes
  /// that point to a null reference, the constant zero node, and sequential
  /// nodes (if the parameter crossSequential is false).  If the parameter
  /// includeRoots is true, these root nodes will be included in the result.
  ///
  /// If the initial  reference is to a sequential node, the transitive fan-in 
  /// of its nextState input will be returned.
  ///
  /// This function will modify traversal IDs.
  ///
  /// Cycles are permitted.
  ///
  /// \param x the vertex of the fan-in cone
  /// \param transitiveFanin the list onto which to append the results
  /// \param includeRoots true if root nodes should be included
  /// \param crossSequential true if the traversal should cross sequential nodes
  //
  // *****************************************************************************
  void
  Graph::getTransitiveFanin(list<Ref> x, list<Ref> &transitiveFanin,
			    bool includeRoots, bool crossSequential) {

    // transitiveFanin = { transitiveFanin, fanin(x_1), fanin(x_2) ... fanin(x_n-1), fanin(x_n) }

    newTraversalID();
    for(list<Ref>::iterator xIter = x.begin(); xIter!=x.end(); xIter++) {
      getTransitiveFanin_recursive(*xIter, transitiveFanin, 
				   includeRoots, crossSequential);
      // remove initial nodes
      transitiveFanin.pop_back();
    }
  }


  // *****************************************************************************
  // getTransitiveFanout()
  //
  /// \brief Returns a list of all nodes in the transitive fan-out of several nodes.
  ///
  /// The nodes in the transitive fan-out are appended to the provided list
  /// in forward topological order.  The initial node is not included.  
  /// All references will be non-inverted.
  ///
  /// The forward traversal will terminate at sequential nodes (if the
  /// crossSequential parameter is false) and nodes that
  /// do not have any fan-out.  If includesRoots is false, the root
  /// nodes will not be included in the result.
  ///
  /// If the initial reference is to a sequential node,
  /// the traversal will continue along its fanout and not terminate immediately.
  ///
  /// If fan-out is not being maintained, the function returns immediately
  /// without modifying the list.
  ///
  /// This function will modify traversal IDs.
  ///
  /// Cycles are permitted.
  ///
  /// \param x the vertex of the fan-in cone
  /// \param transitiveFanout the list onto which to append the results
  /// \param includeRoots true if root nodes should be included
  /// \param crossSequential true if the traversal should cross sequential nodes
  //
  // *****************************************************************************
  void
  Graph::getTransitiveFanout(list<Ref> x, list<Ref> &transitiveFanout,
			     bool includeRoots, bool crossSequential) {

    // transitiveFanout = { transitiveFanout, fanout(x_n) ... fanout(2), fanout(1) }

    newTraversalID();
    list<Ref> allNewFanout;
    for(list<Ref>::iterator it = x.begin(); it != x.end(); it++) {
      list<Ref> oneNewFanout;
      getTransitiveFanout_recursive(*it, oneNewFanout, includeRoots, crossSequential);
      // remove initial node
      oneNewFanout.pop_front();
      allNewFanout.splice(allNewFanout.begin(), oneNewFanout);
    }
    transitiveFanout.splice(transitiveFanout.end(), allNewFanout);
  }


  // *****************************************************************************
  // getTransitiveFanin_recursive()
  //
  /// \brief Internal function.
  ///
  /// The ordering of the result is in NORMAL forward topological order.
  //
  // *****************************************************************************
  void
  Graph::getTransitiveFanin_recursive(Ref x, list<Ref> &transitiveFanin,
				      bool includeRoots, bool crossSequential) {
    x = getNonInverted(x);
    markVisited(x);

    Ref   fanin1, fanin2 = getNull();
    Node *node = getNode(x);
    switch(node->type) {
    case Node::AND:
      fanin1 = node->left;
      fanin2 = node->right;
      break;
    case Node::TERMINAL:
      fanin1 = node->terminal_driver;
      break;
    case Node::SEQUENTIAL:
      fanin1 = node->nextState;
      break;
    case Node::CONSTANT0:
      return;
    default:
      QUIT_ON_INTERNAL_ERROR;
    }

    if (!isNull(fanin1) && !isVisited(fanin1)) {
      if (isTerminal(fanin1) && isNull(getTerminalDriver(fanin1)) 
	  || getNonInverted(fanin1) == constantZero()
	  || isSequential(fanin1) && !crossSequential) {
	// only include if includeRoots is true
	if (includeRoots) {
	  markVisited(fanin1);
	  transitiveFanin.push_back(fanin1);
	}
      } else {
	getTransitiveFanin_recursive(fanin1, transitiveFanin, 
				     includeRoots, crossSequential);
      }
    }

    if (!isNull(fanin2) && !isVisited(fanin2)) {
      if (isTerminal(fanin2) && isNull(getTerminalDriver(fanin2)) 
	  || getNonInverted(fanin2) == constantZero()
	  || isSequential(fanin2) && !crossSequential) {
	// only include if includeRoots is true
	if (includeRoots) {
	  markVisited(fanin2);
	  transitiveFanin.push_back(fanin2);
	}
      } else {
	getTransitiveFanin_recursive(fanin2, transitiveFanin, 
				     includeRoots, crossSequential);
      }
    }
    transitiveFanin.push_back(x);
  }


  // *****************************************************************************
  // getTransitiveFanout()
  //
  /// \brief Returns a list of all nodes in the transitive fan-out.
  ///
  /// The nodes in the transitive fan-in are appended to the provided list
  /// in forward topological order.  The initial node is not included.  
  /// All references will be non-inverted.
  ///
  /// The forward traversal will terminate at sequential nodes (if the
  /// crossSequential parameter is false) and nodes that
  /// do not have any fan-out.  If includesRoots is false, the root
  /// nodes will not be included in the result.
  ///
  /// If the initial reference is to a sequential node,
  /// the traversal will continue along its fanout and not terminate immediately.
  ///
  /// If fan-out is not being maintained, the function returns immediately
  /// without modifying the list.
  ///
  /// This function will modify traversal IDs.
  ///
  /// Cycles are permitted.
  ///
  /// \param x the vertex of the fan-out cone
  /// \param transitiveFanout the list onto which to append the results
  /// \param includeRoots true if root nodes should be included
  /// \param crossSequential true if the traversal should cross sequential nodes
  //
  // *****************************************************************************
  void
  Graph::getTransitiveFanout(Ref x, list<Ref> &transitiveFanout,
			     bool includeRoots, bool crossSequential) {
    newTraversalID();
    list<Ref> newTransitiveFanout;
    getTransitiveFanout_recursive(x, newTransitiveFanout, 
				  includeRoots, crossSequential);
    // remove initial node
    newTransitiveFanout.pop_front(); 
    // append reversed result to original vector
    transitiveFanout.splice(transitiveFanout.end(), newTransitiveFanout);
  }


  // *****************************************************************************
  // getTransitiveFanout_recursive()
  //
  /// \brief Internal function.
  ///
  /// The ordering of the result is in NORMAL forward topological order.
  //
  // *****************************************************************************
  void
  Graph::getTransitiveFanout_recursive(Ref x, list<Ref> &transitiveFanout, 
				       bool includeRoots, bool crossSequential) {
    x = getNonInverted(x);
    markVisited(x);

    if (getNodeType(x) == Node::TERMINAL && 
	hasFanout(x) && !includeRoots) {
      // only include if includeRoots is true
      return;
    }

    const list<Ref> *immediateFanout = &getFanout(x);
    for(list<Ref>::const_iterator it = immediateFanout->begin(); 
	it != immediateFanout->end(); ++it) {
      Ref next = getNonInverted(*it);

      // there shouldn't be fanout to a null or constant node
      assert(!isNull(next));
      assert(getNodeType(next) != Node::CONSTANT0);

      // have we already explored this node?
      if (isVisited(next)) {
	continue;
      }
            
      if (getNodeType(next) == Node::SEQUENTIAL && !crossSequential) {
	// only include if includeRoots is true
	if (includeRoots) {
	  markVisited(next);
	  transitiveFanout.push_front(next);
	}
      } else {
	// continue traversal
	getTransitiveFanout_recursive(next, transitiveFanout, includeRoots);
      }
    }

    transitiveFanout.push_front(x);
  }


  // *****************************************************************************
  // getFaninRoots()
  //
  /// \brief Returns the roots of the transitive fan-in.
  ///
  /// The roots of the transitive fan-in are the backward root in transitive 
  /// fan-in.  A root is either: an terminal node that points to a null 
  /// reference, a sequential node, or constant zero.
  ///
  /// The roots will be appended to the provided list.  If the initial reference 
  /// is to a sequential node, the traversal will continue and not terminate 
  /// immediately.  All references will be non-inverted.
  ///
  /// This function will modify traversal IDs.
  ///
  /// \param x the vertex of the fan-in cone
  /// \param faninRoots the list onto which to append the results
  //
  // *****************************************************************************
  void
  Graph::getFaninRoots(Ref x, list<Ref> &faninRoots) {
    x = getNonInverted(x);

    map<Ref, bool> alreadyVisited;
    list<Ref> toBeVisited;
    toBeVisited.push_back(x);
    alreadyVisited[x] = true;
        
    while(!toBeVisited.empty()) {
      Ref current = toBeVisited.front();
    
      Ref   fanin1, fanin2 = getNull();
      Node *node = getNode(current);
      switch(node->type) {
      case Node::AND:
	fanin1 = node->left;
	fanin2 = node->right;
	break;
      case Node::TERMINAL:
	fanin1 = node->terminal_driver;
	if (isNull(fanin1)) {
	  faninRoots.push_back(current);
	  goto continue_loop;
	}
	break;
      case Node::SEQUENTIAL:
	fanin1 = node->nextState;
	break;
      case Node::CONSTANT0:
	return;
      default:
	QUIT_ON_INTERNAL_ERROR;
      }
    
      fanin1 = getNonInverted(fanin1);    
      // have we already explored this node?
      if (alreadyVisited.find(fanin1) == alreadyVisited.end()) {
	alreadyVisited[fanin1] = true;
	if (isTerminal(fanin1) && isNull(getTerminalDriver(fanin1)) 
	    || getNonInverted(fanin1) == constantZero()
	    || isSequential(fanin1)) {
	  faninRoots.push_back(fanin1);   
	} else {
	  toBeVisited.push_back(fanin1);
	}
      }
        
      if (isNull(fanin2)) {
	// we're done; there was only one fanin
	goto continue_loop;
      }

      fanin2 = getNonInverted(fanin2);    
      // have we already explored this node?
      if (alreadyVisited.find(fanin2) == alreadyVisited.end()) {
	alreadyVisited[fanin2] = true;
	if (isTerminal(fanin2) && isNull(getTerminalDriver(fanin2)) 
	    || getNonInverted(fanin2) == constantZero()
	    || isSequential(fanin2)) {
	  faninRoots.push_back(fanin2);   
	} else {
	  toBeVisited.push_back(fanin2);
	}
      }

    continue_loop:
      toBeVisited.pop_front();
    }
  }


  // *****************************************************************************
  // getFanoutRoots()
  //
  /// \brief Returns the roots of the transitive fan-out.
  ///
  /// The roots of the transitive fan-out are the forward roots in transitive 
  /// fan-out.  A root is either: a sequential node or a node without any
  /// fan-out. 
  ///
  /// The roots will be appended to the provided list.  If the initial reference 
  /// is to a sequential node, the traversal will continue and not terminate 
  /// immediately.  All references will be non-inverted.
  ///
  /// If fan-out is not being maintained, the function returns immediately
  /// without completing the fan-out list.
  ///
  /// Traversal IDs will not be affected.
  ///
  /// \param x the vertex of the fan-out cone
  /// \param fanoutRoots the list onto which to append the results
  //
  // *****************************************************************************
  void
  Graph::getFanoutRoots(Ref x, list<Ref> &fanoutRoots) {
    
    if (!MAINTAIN_FANOUT) {
      return;
    }

    x = getNonInverted(x);
    
    map<Ref, bool> alreadyVisited;
    list<Ref> toBeVisited;
    toBeVisited.push_back(x);
    alreadyVisited[x] = true;
        
    while(!toBeVisited.empty()) {
      Ref current = toBeVisited.front();

      const list<Ref> *immediateFanout = &getFanout(current);

      if (immediateFanout->empty()) {
	fanoutRoots.push_back(current);    
      }

      for(list<Ref>::const_iterator it = immediateFanout->begin(); 
	  it != immediateFanout->end(); 
	  ++it) {
        
	Ref next = getNonInverted(*it);
	// there shouldn't be fanout to a null or constant node        
	assert(!isNull(next));
	assert(getNodeType(next) != Node::CONSTANT0);

	// have we already explored this node?
	if (alreadyVisited.find(next) != alreadyVisited.end()) {
	  continue;
	}
	alreadyVisited[next] = true;
            
	if (getNodeType(next) == Node::SEQUENTIAL) {
	  fanoutRoots.push_back(next);
	} else {
	  toBeVisited.push_back(next);
	}
      }
    
      toBeVisited.pop_front();
    }
  }


  // *****************************************************************************
  // getAllSequential()
  //
  /// \brief Returns all of the sequential references in a graph.
  ///
  /// The sequential bits will be appended to the provided list.  
  /// All references will be non-inverted.
  ///
  /// \param sequential the list onto which to append the results
  //
  // *****************************************************************************
  void
  Graph::getAllSequential(list<Ref> &sequential) const {
    unsigned int page = 0;
    unsigned int index = 0;
    while( !(index > lastUsedIndex && page == lastUsedPage || page > lastUsedPage) ) {
      if (index == PAGE_SIZE) {
	index = 0;
	page++;
      }
      Ref ref = getRef(page,index);
      if (getNodeType(ref) == Node::SEQUENTIAL) {
	sequential.push_back(getNonInverted(ref));
      }
      index++;        
    }
  }


  // *****************************************************************************
  // getAllSequential()
  //
  /// \brief Returns all of the sequential references in a graph.
  ///
  /// The sequential bits will be appended to the provided vector.  
  /// All references will be non-inverted.
  ///
  /// \param sequential the vector onto which to append the results
  //
  // *****************************************************************************
  void
  Graph::getAllSequential(vector<Ref> &sequential) const {
    unsigned int page = 0;
    unsigned int index = 0;
    while( !(index > lastUsedIndex && page == lastUsedPage || page > lastUsedPage) ) {
      if (index == PAGE_SIZE) {
	index = 0;
	page++;
      }
      Ref ref = getRef(page,index);
      if (getNodeType(ref) == Node::SEQUENTIAL) {
	sequential.push_back(getNonInverted(ref));
      }
      index++;        
    }
  }


  // *****************************************************************************
  // getAll()
  //
  /// \brief Returns all of the references in a graph.
  ///
  /// The results will be appended to the provided list.  
  /// All references will be non-inverted.
  ///
  /// \param all the list onto which to append the results
  //
  // *****************************************************************************
  void
  Graph::getAll(list<Ref> &all) const {
    unsigned int page = 0;
    unsigned int index = 0;
    while( !(index > lastUsedIndex && page == lastUsedPage || page > lastUsedPage) ) {
      if (index == PAGE_SIZE) {
	index = 0;
	page++;
      }
      Ref ref = getRef(page,index);
      if (getNodeType(ref) != Node::NONE) {
	all.push_back(getNonInverted(ref));
      }
      index++;        
    }
  }


  // *****************************************************************************
  // getAll()
  //
  /// \brief Returns all of the references in a graph.
  ///
  /// The results will be appended to the provided vector.  
  /// All references will be non-inverted.
  ///
  /// \param all the vector onto which to append the results
  //
  // *****************************************************************************
  void
  Graph::getAll(vector<Ref> &all) const {
    unsigned int page = 0;
    unsigned int index = 0;
    while( !(index > lastUsedIndex && page == lastUsedPage || page > lastUsedPage) ) {
      if (index == PAGE_SIZE) {
	index = 0;
	page++;
      }
      Ref ref = getRef(page,index);
      if (getNodeType(ref) != Node::NONE) {
	all.push_back(getNonInverted(ref));
      }
      index++;        
    }
  }


  // *****************************************************************************
  // rehash()
  //
  /// \brief Rehashes the nodes by structure.
  ///
  /// After rehashing, there will be no two structurally isomorphic AND nodes.
  ///
  /// If two isomorphic nodes are discovered, their fan-outs and equivalences
  /// will be merged and imparted to one representative.
  ///
  /// If fan-out is not maintained, this routine may significantly slower, as
  /// it must be iterated until no more changes are seen, rather than
  /// selective rehashing only the local fan-out as new equivalences are found.
  ///
  /// The graph will not be marked as dirty.
  //
  // *****************************************************************************
  void
  Graph::rehash() {
    assert(structuralHash);
   
    // keep the old dirty marker
    int oldDirtyMarker = dirtyMarker;

    // walk through all nodes
    list<Node *> toBeHashed;
    unsigned int page = 0;
    unsigned int index = 0;
    Node *node;
    while( !(index > lastUsedIndex && page == lastUsedPage || page > lastUsedPage) ) {
      if (index == PAGE_SIZE) {
	index = 0;
	page++;
      }
      node = &(data[page][index]);
      if (node->type == Node::AND) {
	toBeHashed.push_back(node);
      }
      index++;
    }

    // hash until no more changes are seen
    bool modified = false;
    while(!toBeHashed.empty()) {
      node = toBeHashed.front();
      toBeHashed.pop_front();

      int key = getStructuralHashKey(node->left, node->right);
        
      Ref current = structuralHash[key];
      while(current) {
	Node *currentNode = getNode(current);

	// I don't know what the point of "self" is
	assert(currentNode->self == current);
	// no need to do anything to merge with itself
	if (node->self == currentNode->self) {
	  break;
	}

	if (currentNode->left == node->left && currentNode->right == node->right ||
	    currentNode->right == node->left && currentNode->left == node->right) {
	  // isomorphism found
	  modified = true;

	  // 1. must rehash fanout
	  for(list<Ref>::iterator it = node->fanout.begin(); it != node->fanout.end(); it++) {
	    Node *fanoutNode = getNode(*it);
	    if (fanoutNode->type != Node::AND) continue; // only ANDs need to be rehashed

	    toBeHashed.push_back(fanoutNode);
	  }

	  // 2. merge equivalences
	  setEquivalent(node->self, currentNode->self);
	  removeEquivalent(node->self);

	  // 3. merge fanout
	  resubstitute(node->self, currentNode->self);
                
	  // 4. destroy non-representative
	  detach(node->self);

	  break;
	}

	current = currentNode->next;
      }
    }

    // repeat if fan-out is not being maintained
    if (modified && !MAINTAIN_FANOUT) rehash();

    // keep the old dirty marker
    dirtyMarker = oldDirtyMarker;
  }


  // *****************************************************************************
  // newTraversalID()
  //
  /// \brief Generates a unique ID for a new graph traversal.
  ///
  /// Traversal IDs can be used to selectively mark nodes during a graph traversal
  /// operation.  A node is considered "marked" if its traversal ID is the same
  /// as the current glocal traversal ID; if its traversal ID is out of date or
  /// cleared, then it is considered "unmarked".
  ///
  /// The IDs are disposable in that no explicit unmarking is
  /// necessary; once a new ID is generated, any nodes marked with old IDs
  /// are implicitly moved to unmarked status.
  ///
  /// \return the value of the new traversal ID
  ///
  // *****************************************************************************
  unsigned int 
  Graph::newTraversalID() {

    // clear all travsersalIDs
    if (++currentTraversalID == 0) {
      unsigned int page = 0;
      unsigned int index = 0;
      while( !(index > lastUsedIndex && page == lastUsedPage || page > lastUsedPage) ) {
	if (index == PAGE_SIZE) {
	  index = 0;
	  page++;
	}
	data[page][index++].traversalID = 0;
      }    
      currentTraversalID = 1;
    }

    return currentTraversalID;
  }


  // *****************************************************************************
  // getFaninCone()
  //
  /// \brief Returns a list of the nodes between a specified cut and a node.
  ///
  /// The nodes in the transitive fan-in are appended to the provided list
  /// in forward topological order.  The initial node is not included.  
  /// All references will be non-inverted.
  ///
  /// The backward traversal will terminate at sequential nodes, terminal nodes
  /// that point to a null reference, and the constant zero node.  If the parameter
  /// includeRoots is true, these nodes will be included in the result.
  ///
  /// The backward traversal will also terminate at any of the nodes in the
  /// provided cut.  The cut need not be complete, but if it is not, the
  /// result may include nodes that a topological predecessors of the cut.
  ///
  /// If the initial  reference is to a sequential node, the transitive fan-in 
  /// of its nextState input will be returned.
  ///
  /// This function will modify traversal IDs.
  ///
  /// Cycles are permitted.
  ///
  /// \param x the vertex of the fan-in cone
  /// \param coneRoots the roots of the cone that limit traversal
  /// \param transitiveFanin the vector onto which to append the results
  /// \param includeRoots true if root SEQUENTIAL, TERMINAL, and CONSTANT nodes should be included
  //
  // *****************************************************************************
  void
  Graph::getFaninCone(Ref x, const list<Ref> &coneRoots,
		      list<Ref> &transitiveFanin, bool includeRoots) {
    newTraversalID();
    // block traversal from proceeding past cut
    for(list<Ref>::const_iterator it = coneRoots.begin(); 
	it != coneRoots.end(); it++) {
      markVisited(*it);
      // add roots if desired
      if (includeRoots) {
	transitiveFanin.push_back(*it);
      }
    }
    // do backward traversal
    getTransitiveFanin_recursive(x, transitiveFanin, includeRoots);
    // remove initial node
    transitiveFanin.pop_back(); 
  }



  // *****************************************************************************
  // getFaninCone()
  //
  /// \brief Returns a vector of the nodes between a specified cut and a node.
  ///
  /// The nodes in the transitive fan-in are appended to the provided list
  /// in forward topological order.  The initial node is not included.  
  /// All references will be non-inverted.
  ///
  /// The backward traversal will terminate at sequential nodes, terminal nodes
  /// that point to a null reference, and the constant zero node.  If the parameter
  /// includeRoots is true, these nodes will be included in the result.
  ///
  /// The backward traversal will also terminate at any of the nodes in the
  /// provided cut.  The cut need not be complete, but if it is not, the
  /// result may include nodes that a topological predecessors of the cut.
  ///
  /// If the initial  reference is to a sequential node, the transitive fan-in 
  /// of its nextState input will be returned.
  ///
  /// This function will modify traversal IDs.
  ///
  /// Cycles are permitted.
  ///
  /// \param x the vertex of the fan-in cone
  /// \param coneRoots the roots of the cone that limit traversal
  /// \param transitiveFanin the vector onto which to append the results
  /// \param includeRoots true if root SEQUENTIAL, TERMINAL, and CONSTANT nodes should be included
  //
  // *****************************************************************************
  void
  Graph::getFaninCone(Ref x, const vector<Ref> &coneRoots,
		      vector<Ref> &transitiveFanin, bool includeRoots) {
    newTraversalID();
    // block traversal from proceeding past cut
    for(vector<Ref>::const_iterator it = coneRoots.begin(); 
	it != coneRoots.end(); it++) {
      markVisited(*it);
      // add roots if desired
      if (includeRoots) {
	transitiveFanin.push_back(*it);
      }
    }
    // do backward traversal
    getTransitiveFanin_recursive(x, transitiveFanin, includeRoots);
    // remove initial node
    transitiveFanin.pop_back(); 
  }


  // *****************************************************************************
  // getFanoutCone()
  //
  /// \brief Returns a list of the nodes between a node and a specified cut.
  ///
  /// The nodes in the transitive fan-out are appended to the provided list
  /// in forward topological order.  The initial node is not included.  
  /// All references will be non-inverted.
  ///
  /// The forward traversal will terminate at sequential nodes and nodes that
  /// do not have any fan-out.  If includesRoots is false, the sequential
  /// nodes and terminal nodes without any fanout will not be included in the
  /// result.
  ///
  /// The backward traversal will also terminate at any of the nodes in the
  /// provided cut.  The cut need not be complete, but if it is not, the
  /// result may include nodes that lie in the fan-in of the cut 
  ///
  /// If the initial reference is to a sequential node,
  /// the traversal will continue along its fanout and not terminate immediately.
  ///
  /// If fan-out is not being maintained, the function returns immediately
  /// without modifying the list.
  ///
  /// This function will modify traversal IDs.
  ///
  /// Cycles are permitted.
  ///
  /// \param x the vertex of the fan-out cone
  /// \param coneRoots the roots of the cone that limit traversal
  /// \param transitiveFanout the vector onto which to append the results
  /// \param includeRoots true if root SEQUENTIAL and TERMINAL nodes should be included
  //
  // *****************************************************************************
  void
  Graph::getFanoutCone(Ref x, const list<Ref> &coneRoots,
		       list<Ref> &transitiveFanout, bool includeRoots) {
    newTraversalID();
    // block traversal from proceeding past cut
    for(list<Ref>::const_iterator it = coneRoots.begin(); 
	it != coneRoots.end(); it++) {
      markVisited(*it);
    }
    // do forward traversal
    list<Ref> newTransitiveFanout;
    getTransitiveFanout_recursive(x, newTransitiveFanout, includeRoots);
    // remove initial node
    newTransitiveFanout.pop_front();
    // add roots if desired
    if (includeRoots) {
      for(list<Ref>::const_iterator it = coneRoots.begin(); 
	  it != coneRoots.end(); it++) {
	newTransitiveFanout.push_back(*it);
      }
    }
    // append result to original vector
    transitiveFanout.splice(transitiveFanout.end(), newTransitiveFanout);
  }

  // *****************************************************************************
  // dotEdge()
  //
  /// \brief Prints the edges of a graph 
  ///
  /// \param fromRef the starting node of the edge
  /// \param toRef the ending node of the edge
  /// \param nextState the indication for an edge to an sequential node
  /// \param outfile the output file
  //
  // *****************************************************************************
  void
  Graph::dotEdge(Ref fromRef,Ref toRef, bool nextState, ofstream &outfile)
  {
    unsigned int i = getIndex(fromRef);
    unsigned int j = getIndex(toRef);

    outfile << "N_" << i << " -> N_" << j ;
    if (oagAi::Graph::isInverted(fromRef) && nextState) {
      outfile << " [style=dotted, label=\"NS\"];";
    }
    else if (oagAi::Graph::isInverted(fromRef)) {
      outfile << " [style=dotted];";
    }
    else if (nextState) {
      outfile << " [label=\"NS\"];";
    }
    outfile << endl;
  }

  // *****************************************************************************
  // dotNode()
  //
  /// \brief Prints the nodes of a graph 
  /// 
  /// \param ref the reference of the node
  /// \param outfile the output file
  //
  // *****************************************************************************
  void
  Graph::dotNode(Ref ref, ofstream &outfile)
  {
    Node *node = getNode(ref);
    unsigned int i = Graph::getIndex(ref);
    outfile << "N_" << i << " [label=\"N_" << i << "\\n" << typeStr(node);

    

    oagAi::Ref iref = oagAi::Graph::notOf(ref);

    
    //char *s = getInfo(ref);
    //char *si = getInfo(iref);
    //if (s) {
    //  outfile << "\\n" << s;
    //}
    //if (si) {
    //  outfile << "\\ni:" << si;
    //}
    if(node->type == Node::TERMINAL)     
      outfile << "\",shape=box";
    else
      outfile << "\"";
    outfile << "];" <<  endl;
  }


const char *
Graph::typeStr(const Node *node) {
  switch (node->type) {
   case Node::NONE: return "NONE";
   case Node::SEQUENTIAL: return "SEQ";
   case Node::TERMINAL: return "TERM";
   case Node::AND: return "AND";
   case Node::CONSTANT0: return "CONST0";
  }
  assert(0);
  return "UNKNOWN";
}

//-----------------------------------------------------------------------------




}
