/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron Hurst <ahurst@eecs.berkeley.edu>

ChangeLog:
2005-08-12: ChangeLog started
*/

#include <iostream>
#include <algorithm>
#include "oaDesignDB.h"
#include "oagAiGraph.h"
#include "oagAiGraphUnitTest.h"
#include <string>

using namespace std;

namespace oagAi {

bool areListsIdentical(list<Ref> &l1, list<Ref> &l2) {
    l1.sort();
    l2.sort();

    // print out lists
    //copy(l1.begin(), l1.end(), ostream_iterator<int>(cout, " ")); cout << endl;
     // copy(l2.begin(), l2.end(), ostream_iterator<int>(cout, " ")); cout << endl;         

    return equal(l1.begin(), l1.end(), l2.begin())&&equal(l2.begin(), l2.end(), l1.begin());
}

int GraphUnitTest::runAllTests(void) {

    Graph graph;
    
    // Graph()
    
    logOneTest(graph.getNumNodes() == graph.NODES_TO_IGNORE, "Checking count of fixed nodes");

    // create test setup
    // ----------------    
    int numAndNodes = 0, numNodes = 0, numSeqNodes = 0;
    // ----------------
    Ref i1 = graph.newTerminal(graph.getNull());
    numNodes++;    
    Ref i2 = graph.newTerminal(graph.getNull());    
    numNodes++;
    Ref i3 = graph.newTerminal(graph.getNull());
    numNodes++;
    Ref s1 = graph.newSequential(graph.getNull());
    numNodes++; numSeqNodes++;
    Ref s2 = graph.newSequential(i2);
    numNodes++; numSeqNodes++;
    graph.disableStructuralHashing();
    Ref a1 = graph.andOf(i1,i2);
    numNodes++; numAndNodes++;
    Ref a2 = graph.andOf(i1,i2);
    numNodes++; numAndNodes++;
    graph.enableStructuralHashing(); 
    Ref a3 = graph.andOf(i2,i3);
    numNodes++; numAndNodes++;
    Ref a4 = graph.andOf(i2,i3); 
    // this should not create a new node
    
    Ref a5 = graph.andOf(a1,a3);
    numNodes++; numAndNodes++;
    Ref a6 = graph.andOf(a5,s2);
    numNodes++; numAndNodes++; 
    Ref s3 = graph.newSequential(a6);    
    numNodes++;numSeqNodes++;
    Ref i4 = graph.newTerminal(a6); 
    

    // ---------------
    
    //Dot  test;
    string str ("unit");//char * name = "unit";
    graph.dot(str);
    //graph.


    // getNull() / isNull()
    
    logOneTest(graph.isNull(graph.getNull()), "Checking null reference is null");
    logOneTest(!graph.isNull(i1), "Checking non-null reference is non-null");
   
    // structural hashing
    
    logOneTest(a1 != a2, "Checking that structural hashing is disabled");
    logOneTest(a3 == a4, "Checking that identical nodes are not created with structual hashing on");

    // getNumAndNodes() / getNumNodes() / getNumSeqNodes()
    
    logOneTest(graph.getNumSeqNodes() == numSeqNodes, "Checking that the count of sequential nodes is correct");
    logOneTest(graph.getNumAndNodes() == numAndNodes, "Checking that the count of and nodes is correct");

    // notOf() / isInverted() / getNonInverted()
    
    Ref not_i1 = graph.notOf(i1);
    logOneTest(graph.isInverted(not_i1), "Checking that complementation inverts reference");
    logOneTest(graph.getNonInverted(not_i1) == i1, "Checking that noninverted reference is properly returned");
    logOneTest(graph.notOf(not_i1) == i1, "Checking that two complementations returns original reference");
    
    // constantZero() / constantOne()
    
    logOneTest(graph.notOf(graph.constantZero()) == graph.constantOne(), "Checking that constant zero and one are complements");
    
    // getTransitiveFanin()

    list<Ref> actual, expected;
    
    actual.clear(); expected.clear();    
    graph.getTransitiveFanin(a6, actual);
    expected.push_back(a1);
    expected.push_back(a3);    
    expected.push_back(a5);
    expected.push_back(s2);
    expected.push_back(i1);
    expected.push_back(i2);
    expected.push_back(i3);
    logOneTest(areListsIdentical(actual, expected), "Checking transitive fanin computation 1");
    
    actual.clear();
    graph.getTransitiveFanin(s3, actual);
    expected.push_back(a6);
    logOneTest(areListsIdentical(actual, expected), "Checking transitive fanin computation 2");
    
    actual.clear();
    graph.getTransitiveFanin(i4, actual);
    logOneTest(areListsIdentical(actual, expected), "Checking transitive fanin computation 3");    

    actual.clear(); expected.clear();
    graph.getTransitiveFanin(i1, actual);
    logOneTest(areListsIdentical(actual, expected), "Checking transitive fanin computation 4");

    // getFaninRoots()

    actual.clear(); expected.clear();
    graph.getFaninRoots(a6, actual);
    expected.push_back(s2);
    expected.push_back(i1);
    expected.push_back(i2);
    expected.push_back(i3);
    logOneTest(areListsIdentical(actual, expected), "Checking fanin root computation 1");
    
    actual.clear();
    graph.getFaninRoots(s3, actual);
    logOneTest(areListsIdentical(actual, expected), "Checking fanin root computation 2");
    
    actual.clear();
    graph.getFaninRoots(i4, actual);
    logOneTest(areListsIdentical(actual, expected), "Checking fanin root computation 3");    

    actual.clear(); expected.clear();
    graph.getFaninRoots(i1, actual);
    expected.push_back(i1);
    logOneTest(areListsIdentical(actual, expected), "Checking fanin root computation 4");
    
    // fanout
    
    if (graph.MAINTAIN_FANOUT) {
        logOneTest(graph.getFanout(i2).size() == 4, "Fanout test 1");

        // getTransitiveFanout()

        actual.clear(); expected.clear();
        graph.getTransitiveFanout(a3, actual);
        expected.push_back(a5);
        expected.push_back(a6);
        expected.push_back(i4);
        expected.push_back(s3);
        logOneTest(areListsIdentical(actual, expected), "Checking transitive fanout computation 1");

        actual.clear();
        graph.getTransitiveFanout(i2, actual);
        expected.push_back(a1);
        expected.push_back(a2);
        expected.push_back(a3);
        expected.push_back(s2);
        logOneTest(areListsIdentical(actual, expected), "Checking transitive fanout computation 2");
        
        actual.clear(); expected.clear();
        graph.getTransitiveFanout(s2, actual);
        expected.push_back(a6);
        expected.push_back(i4);
        expected.push_back(s3);
        logOneTest(areListsIdentical(actual, expected), "Checking transitive fanout computation 3");

        actual.clear(); expected.clear();
        graph.getTransitiveFanout(i4, actual);
        logOneTest(areListsIdentical(actual, expected), "Checking transitive fanout computation 4");

        actual.clear(); expected.clear();
        graph.getTransitiveFanout(s3, actual);
        logOneTest(areListsIdentical(actual, expected), "Checking transitive fanout computation 5");
        
        // getFanoutRoots()
        
        actual.clear(); expected.clear();
        graph.getFanoutRoots(a3, actual);
        expected.push_back(i4);
        expected.push_back(s3);
        logOneTest(areListsIdentical(actual, expected), "Checking fanout root computation 1");
        
        actual.clear();
        graph.getFanoutRoots(s2, actual);
        logOneTest(areListsIdentical(actual, expected), "Checking fanout root computation 2");
        
        actual.clear();
        graph.getFanoutRoots(i2, actual);
        expected.push_back(s2);
        expected.push_back(a2);    
        logOneTest(areListsIdentical(actual, expected), "Checking fanout root computation 3");
        
        actual.clear(); expected.clear();
        graph.getFanoutRoots(s3, actual);
        expected.push_back(s3);
        logOneTest(areListsIdentical(actual, expected), "Checking fanout root computation 4");

        actual.clear(); expected.clear();
        graph.getFanoutRoots(i4, actual);
        expected.push_back(i4);
        logOneTest(areListsIdentical(actual, expected), "Checking fanout root computation 5");       
    }
    
    // garbage collection
    
    int preCollect = graph.getNumNodes();
    graph.setNextState(s1, a2);
    graph.incrementExternalReferences(i4);
    graph.incrementExternalReferences(s3);
    graph.garbageCollect();
    // two nodes should be destroyed: a2, s1
    int postCollect = graph.getNumNodes();
    logOneTest(preCollect == postCollect+2, "Checking garbage collection 1");
    graph.clearExternalReferences(i4);
    graph.clearExternalReferences(s3);
    graph.garbageCollect();
    // all other nodes should be destroyed
    postCollect = graph.getNumNodes();
    logOneTest(preCollect == postCollect+12, "Checking garbage collection 2");

    i1 = graph.newTerminal(graph.getNull());
    i2 = graph.newTerminal(graph.getNull());
    a1 = graph.andOf(i1,i2);
    logOneTest(graph.getNumAndNodes() == 1, "Checking functionality post-garbage collection");
    
    return getReturnCode();

}

}
