# Requires either lp_solve http://lpsolve.sourceforge.net/5.5/ (FREE) or 
# Mosek http://www.mosek.com (Commercial)
# if using Mosek, 
# set the MOSEK flag to 1 if you'd like to use MOSEK (faster)
# set MOSEK 0 if you'd like to use lp_solve (better support, but slower)

#Implementation of David Nguyen, Abhijit Davare, Michael Orshansky, David Chinnery, Brandon Thompson, and Kurt Keutzer. Minimization of Dynamic and Static Power Through Joint Assignment of Threshold Voltages and Sizing Optimization. Proceedings of the International Symposium on Low Power Electronics and Design 2003 (ISLPED'03). pp. 348-353. 

# Differences between the paper - 
# 1. Delta Slack is used instead of Delta Delay - this improves issues with slews
# 2. Slacks are also used to check each move - this helps to guarantee a timing feasible
#    outcome. However, this also slows the algorithm significantly.

#


# compute the sensitivity of the gate for the
# LP Slack Allocation Objective
proc getLPSASensitivity { cgate } { 
    set gname [get_property $cgate hierarchical_name]
    
    set sensMin 1000000.0
    set optBest [list $gname 0 0]
    
    set opts [evaluateOptions $gname]
    
    foreach opt $opts { 
        set cellnameN [lindex $opt 0]
        set DS [lindex $opt 1]
        set DP [lindex $opt 2]
        
        if { $DP < 0 } {
            if {$DS == 0.0} {
                set sens 0.0
            } else {
                set sens  [expr "- $DP / $DS"]
            }
            
            if {$DS > 0.0} {
                # don't allocate negative slacks in LP
                set DS 0.0 
            }
                       
            if {$sens < $sensMin} {
                set sensMin $sens
                set optBest [list $gname $sens $DS]
            }
        }
    }
    
    return $optBest
}

# Main LP Slack allocation procedure
proc LPSlackAllocation { } {
    puts "Running LP Slack Allocation"
 
    setEcoMode -refinePlace false
    # Allocate slacks according to sensitivities
    # Load the upsize and downsize procedures information
    #source /w/design/puneet/projects/eco/scripts/sizingProcs.tcl
    
    set freeList [list]
    global gatesList
    numberGates
    set riseOffset 0
    set fallOffset [llength $gatesList]
    setEcoMode -refinePlace false

    set cc [get_clocks]
    set clockPeriod [get_property $cc period]

    set loopf 1
    
    # to use mosek instead of lp_solve
    set MOSEK 0
    
    set fidl [open "lpsa.log" "w"]
    puts "OPENING LOG FILE lpsa.log"
    writeIterateInfo "Initial" $fidl
    
    if { [getMinSlack] <= 0.0 } {
        puts $fidl "Negative initial slack - exiting"
        close $fidl
        return
    }
    
    set iters 0
    while { $loopf } {
        incr iters 1
        set OUTPUTFILE "LPSA.${iters}.lp";
        set fid [open $OUTPUTFILE "w"]
        puts "OPENING FILE ${OUTPUTFILE}"

        
        set smin [getMinSlack]
        puts "$iters: slack $smin"
        
        if {$MOSEK} {
          puts $fid "minimize"
        } else {
          puts $fid "min: "
        }
        
        set plusFlag false
        set bounds [list]
        # Write LP
        foreach_in_collection iCell [get_cells] {
            if {[get_property $iCell is_sequential] == "false"} {
                set gname [get_property $iCell hierarchical_name]
                set iPin [getOutputPin $gname]
                set phname [get_property $iPin hierarchical_name]
                set cgate [get_cells $gname]
                set cellname0 [get_property $cgate ref_lib_cell_name]
                set cpins [get_pins $gname/*]
                set CInd [getGateIndex $iCell]
                set sensL [getLPSASensitivity $iCell]
                lappend bounds $sensL
                set sens [lindex $sensL 1]
                
                if { $sens != 0.0 } { 
                  if { $sens < 0.0 } {
                     set plusFlag true
                  } else {
                      if {$plusFlag == "true"} {
                        puts $fid " + " nonewline
                    } else {
                        set plusFlag true
                    }
                  }
                  
                  puts $fid "${sens} G_${gname}_s" nonewline
                }
            }
        }
        if {$MOSEK} {
          puts $fid ""
          puts $fid "subject to"
        } else {
          puts $fid ";"
        }
          
        # Write LP
        foreach_in_collection iCell [get_cells] {
            set gname [get_property $iCell hierarchical_name]
            set cgate [get_cell $gname]
            set cellname0 [get_property $cgate ref_lib_cell_name]
            set cpins [get_pins $gname/*]
            set CInd [getGateIndex $iCell]

            foreach_in_collection iPin $cpins {
                
                # flip-flop outputs 
                if {([get_property $iCell is_sequential] == "true") && 
                    ([get_property $iPin direction]  == "out") } {
                    set pname [get_property $iPin ref_lib_pin_name]
                    set phname [get_property $iPin hierarchical_name]
                    
                    set ac [get_arcs -from $phname]
                    if {[sizeof_collection $ac] != 0} {
                        set amr [get_property $iPin arrival_max_rise]
                        set amf [get_property $iPin arrival_max_fall]
                        set outVar $CInd
                        puts $fid "-G_${phname}_r${outVar} <= -$amr" nonewline
                        if {! $MOSEK } {
                          puts $fid ";"
                        } else {
                          puts $fid "\n" nonewline
                        }
                          
                        puts $fid "-G_${phname}_f${outVar} <= -$amf" nonewline
                        if { ! $MOSEK } {
                          puts $fid ";"
                        } else {
                          puts $fid "\n" nonewline
                        }
                    }
                }

                #input pin loop
                if {([get_property $iPin direction] == "in") && 
                    ([get_property $iPin is_async]  == "false") && 
                    ([get_property $iPin is_clock]  == "false") } {
                    set pname [get_property $iPin ref_lib_pin_name]
                    set phname [get_property $iPin hierarchical_name]
                    if { [get_property $iPin is_rise_edge_triggered_data] == "true" } {
                        # Flip-flop Inputs - handled differently
                        # compute the delay into this gate
                        set ac [get_arcs -to $phname]
                        set FIGInd -1
                        foreach_in_collection ca $ac {
                            if { [get_property $ca arc_type] == "setup_rising"} {
                                set setupRise [get_property $ca delay_max_rise]
                                set setupFall [get_property $ca delay_max_fall]
                            } else {
                                set sourcepin [get_property $ca source_pin]
                                if { [get_property $sourcepin is_port] } {
                                    # add the delay at the port
                                    set amr [get_property $sourcepin arrival_max_rise]
                                    set amf [get_property $sourcepin arrival_max_fall]
                                } else {
                                    set srcGate [getGateFromPin $sourcepin]
                                    set FIGInd [getGateIndex $srcGate]
                                    set FIPinName [get_property $sourcepin hierarchical_name] 
                                }
                                set dmr [get_property $ca delay_max_rise]
                                set dmf [get_property $ca delay_max_fall]
                            }
                        }
                        
                        #adjust the times to account for clock latency
                        set ipamr [getArrivalRisePin $iPin]
                        set ipamf [getArrivalFallPin $iPin]
                        set ipsmr [getSlackRisePin $iPin]
                        set ipsmf [getSlackFallPin $iPin]
                        
                        set dcr [expr $ipamr + $ipsmr + $setupRise]
                        set dcf [expr $ipamf + $ipsmf + $setupFall]
                        
                        # rising path from input
                        if {$FIGInd != "-1"} {
                            set constb [expr -$dmr - $setupRise + $dcr]
                            set fiVar $FIGInd
                            puts $fid "G_${FIPinName}_r${fiVar} <= ${constb}" nonewline
                            if { ! $MOSEK } {
                              puts $fid ";"
                            } else {
                              puts $fid "\n" nonewline
                            }
                        } 
                        
                        # falling path from input

                        if {$FIGInd != "-1"} {
                            set constb [expr -$dmf - $setupFall + $dcf]
                            set fiVar $FIGInd
                            puts $fid "G_${FIPinName}_f${fiVar} <= ${constb}" nonewline
                            if { ! $MOSEK } {
                              puts $fid ";"
                            } else {
                              puts $fid "\n" nonewline
                            }
                        } 
                        
                        
                    } else {
                        # compute the delay into this gate
                        set ac [get_arcs -to $phname]
                        set FIGInd -1
                        foreach_in_collection ca $ac {
                            set sourcepin [get_property $ca source_pin]
                            if { [get_property $sourcepin is_port] } {
                                # add the delay at the port
                                set amr [get_property $sourcepin arrival_max_rise]
                                set amf [get_property $sourcepin arrival_max_fall]
                            } else {
                                set srcGate [getGateFromPin $sourcepin]
                                set FIGInd [getGateIndex $srcGate]
                                set FIPinName [get_property $sourcepin hierarchical_name] 
                            }
                            set dmr [get_property $ca delay_max_rise]
                            set dmf [get_property $ca delay_max_fall]
                        }
                        
                        set ac [get_arcs -from $phname]
                        foreach_in_collection ca $ac {
                            set sinkpin [get_property $ca sink_pin]
                            set tsense [get_property $ca sense]
                            set sinkPinName [get_property $sinkpin hierarchical_name]
                            #puts $fid "$phname to $sinkPinName $tsense" 
                            
                            # rising path from input
                            set outVar $CInd                            
                            set constb [expr -$dmr]
                            if {$FIGInd != "-1"} {
                                set fiVar $FIGInd
                                if { $tsense == "negative_unate" } {
                                    puts $fid "G_${FIPinName}_f${fiVar} " nonewline
                                } 
                                if { $tsense == "positive_unate" } {
                                    puts $fid "G_${FIPinName}_r${fiVar} " nonewline
                                }
                                lappend freeList "G_${sinkPinName}_r${outVar}"
                            } else {
                                set constb [expr $constb - $amr]
                            }
                            

                            puts $fid "-G_${sinkPinName}_r${outVar} + G_${gname}_s" nonewline
                            set constb [expr $constb - [get_property $ca delay_max_rise]]
                            
                            puts $fid " <= ${constb}" nonewline
                            if {! $MOSEK } {
                              puts $fid ";"
                            } else {
                              puts $fid "\n" nonewline
                            }
                            
                            # falling path from input
                            set outVar $CInd
                            set constb [expr -$dmf]
                            if {$FIGInd != "-1"} {
                                set fiVar $FIGInd
                                if { $tsense == "negative_unate" } {
                                    puts $fid "G_${FIPinName}_r${fiVar} " nonewline
                                } 
                                if { $tsense == "positive_unate" } {
                                    puts $fid "G_${FIPinName}_f${fiVar} " nonewline
                                }                           
                                 lappend freeList "G_${sinkPinName}_f${outVar}"
                            } else {
                                set constb [expr $constb - $amf]
                            }

                            puts $fid "-G_${sinkPinName}_f${outVar} + G_${gname}_s" nonewline
                            set constb [expr $constb - [get_property $ca delay_max_fall]]
                            puts $fid " <= ${constb}" nonewline
                            if { ! $MOSEK } {
                              puts $fid ";"
                            } else {
                              puts $fid "\n" nonewline
                            }
                            
                        }
                        
                    }
                    
                }
            }
        }
        
        
        foreach_in_collection iPort [get_ports] {
            set pname [get_property $iPort hierarchical_name]
            set s0 [min [get_property $iPort slack_max_rise]  [get_property $iPort slack_max_fall]]
            set ac [get_arcs -to $pname]
            set FIGInd -1
            foreach_in_collection ca $ac {
                set sourcepin [get_property $ca source_pin]
                set sourcePinName [get_property $sourcepin hierarchical_name]
                set dmr [get_property $ca delay_max_rise]
                set dmf [get_property $ca delay_max_rise]
                if { [get_property $sourcepin is_port] } {
                    # add the delay at the port
                    set amr [get_property $sourcepin arrival_max_rise]
                    set amf [get_property $sourcepin arrival_max_fall]
                } else {
                    set srcGate [getGateFromPin $sourcepin]
                    set FIGInd [getGateIndex $srcGate]
                }

                #adjust the times to account for clock latency
                                        set ipamr [getArrivalRisePin $iPin]
                        set ipamf [getArrivalFallPin $iPin]
                        set ipsmr [getSlackRisePin $iPin]
                        set ipsmf [getSlackFallPin $iPin]
                
                set ipamr [getArrivalRisePin $iPort]
                set ipamf [getArrivalFallPin $iPort]
                set ipsmr [getSlackRisePin $iPort]
                set ipsmf [getSlackFallPin $iPort]
                        
                set dcr [expr $ipamr + $ipsmr]
                set dcf [expr $ipamf + $ipsmf]
                
                # falling path from input
                
                if {$FIGInd != "-1"} {
                    set fiVar $FIGInd
                    set constb [expr -$dmf+$dcf]
                    
                    puts $fid "G_${sourcePinName}_f${fiVar} " nonewline
                    puts $fid " <= ${constb}" nonewline
                    if { ! $MOSEK } {
                      puts $fid ";"
                    } else {
                      puts $fid "\n" nonewline
                    }
                } 
                

                # rising path from input
                if {$FIGInd != "-1"} {
                    set fiVar [expr $FIGInd]
                    set constb [expr -$dmr+$dcr]
                    
                    puts $fid "G_${sourcePinName}_r${fiVar} " nonewline
                    puts $fid " <= ${constb}" nonewline
                    if { ! $MOSEK } {
                      puts $fid ";"
                    } else {
                      puts $fid "\n" nonewline
                    }
                } 
                
            }
        }
        
        if { $MOSEK } {
          puts $fid "bounds"
        } 
        
        foreach s $bounds {
            set vname [lindex $s 0]
            set dmax [expr 0- [lindex $s 2]]
            if {$dmax < 0} {
                puts $fid " 0 <= G_${vname}_s <= 0" nonewline
                if {! $MOSEK} {
                  puts $fid ";"
                } else {
                          puts $fid "\n" nonewline
                        }
            } else {
                set dmax [expr $dmax*1.1]
                puts $fid " 0 <= G_${vname}_s <= ${dmax}" nonewline
                if {! $MOSEK} {
                  puts $fid ";"
                } else {
                          puts $fid "\n" nonewline
                        }
            }
        }
        
        if { ! $MOSEK } {
            puts $fid "free " nonewline
        } 
        
        set flag 0
        
        foreach fc $freeList {
            if {! $MOSEK } {
                if {$flag} {
                    puts $fid "," nonewline
                } else {
                    set flag 1
                }
            }
            puts $fid " $fc" nonewline
            if {$MOSEK} {
                puts $fid " free\n" nonewline
            }
        }
        if {! $MOSEK } {
            puts $fid ";\n\n" nonewline
        } else {
            puts "end\n\n" nonewline
        }
        
        
        close $fid
        
        if {$MOSEK} {
          exec mosek -p /w/design/puneet/projects/gs_survey/tcl/mosek.par LPSA.${iters}.lp > output.${iters}.lp
          set slacks [readLPSAoutput LPSA.sol]
        } else {
            if {[catch {exec lp_solve LPSA.${iters}.lp > output.${iters}.lp}] } {
                puts "LP solver infeas - breaking" 
                break
            }
            if {![checkLPSAoutput output.${iters}.lp]} {
                puts "Infeasible - breaking" 
                break
            }
            set slacks [readLPSAoutput output.${iters}.lp]
        }
        set loopf 0
        foreach s $slacks {
            set gname [lindex $s 0]
            set sa [lindex $s 1]
            
            ## if {$sa > 0 } {
            ##  puts $fidl "$gname / slack $sa" 
            ##  
            ## }
            ## puts "gate name $gname / slack $sa "
            
            if { [applySlackLPSA $gname $sa] } {
                  set loopf 1
            }
        }
        if { $iters > 50} {
          break
        }
        writeIterateInfo "Iterate" $fidl
    }
        
    setEcoMode -refinePlace true
    writeIterateInfo "Final" $fidl    
    close $fidl
}

# applies the slack allocation $sa to gate $gname
proc applySlackLPSA { gname sa } {
    #if {$sa <= 0} {
    #  return 0
    #} 
    
    set cgate [get_cell $gname]
    set cellname0 [get_property $cgate ref_lib_cell_name]
    set leakage0 [getLeakagePowerLibCell $cellname0]

    set iPin [getOutputPin $gname]
    
    set cellnameF $cellname0
    set opts [evaluateOptions $gname]
    
    set opts [lsort -real -index 2 -decreasing $opts] 
   
    foreach opt $opts { 
        set cellnameN [lindex $opt 0]
        set DS [lindex $opt 1]
        set DP [lindex $opt 2]
        set Sc [lindex $opt 3]
        
        if { $DP < 0 } {
            if { ($DS <= $sa) && ($Sc >= 0) } {
              # if the slack change is within the allocated slack
              if { [LPSALocalSlack $iPin] >= 0.0 } {
                set cellnameF $cellnameN
              }
            }
        }
    }
    
    if {$cellname0 != $cellnameF} {
      ecoChangeCell -inst $gname -cell $cellnameF
      set leakageF [getLeakagePowerLibCell $cellnameF]
      puts "$gname / $sa / $cellname0 ($leakage0) -> $cellnameF ($leakageF)"
      return 1
    } else {    
      return 0
    }
}

# Computes the slack a neighborhood around the current gate.
# This is to ensure that no timing violations are present
proc LPSALocalSlack { cpin } {
  
  set smin 1000
  set dcc  [all_fanin -pin_levels 3 -to $cpin ]
  foreach_in_collection dc $dcc {
    set sc [getSlackPin $dc]
    if { $sc < $smin } {
      set smin $sc
    }
  }
  set dcc  [all_fanout -pin_levels 3 -from $cpin ]
  foreach_in_collection dc $dcc {
    set sc [getSlackPin $dc]
    if { $sc < $smin } {
      set smin $sc
    }
  }
  return $smin
}

# reads the output fo the LP solver to check if
# the solver returned infeasible
proc checkLPSAoutput { fname } {
    set fid [open $fname "r"]
    set slacks [list]
    if {[gets $fid line] >= 0} {
        set words [regexp -all -inline {\S+} $line]
        if {[llength $words] == 4 } {
            set word [lindex $words 3]
            if {$word == "infeasible"} {
                return 0
            }
        }
    }
    
    close $fid
    return 1
}

# reads the variables from the solver log
proc readLPSAoutput { fname } {
    set fid [open $fname "r"]
    set slacks [list]
    while {[gets $fid line] >= 0} {
        set words [regexp -all -inline {\S+} $line]
        if {[llength $words] == 2 } {
            set gname [lindex $words 0]
            if { [string range $gname end-1 end] == "_s"} {
                set gname2 [string range $gname 2 end-2]
                set salloc [lindex $words 1]
                lappend slacks [list $gname2 $salloc]
                ## if { $salloc > 0} {
                ##  puts "read $gname2 $salloc / $gname $salloc"
                  
                ##}
                #puts "read $gname / $gname2"  
            }
        } 
        if {[llength $words] == 8 } {
            set gname [lindex $words 1]
            if { [string range $gname end-1 end] == "_s"} {
                set gname [string range $gname 2 end-2]
                set salloc [lindex $words 3]
                lappend slacks [list $gname2 $salloc]
                #puts "$gname $salloc"
            }
        } 
    }
    close $fid
    return $slacks
}

# This is just used for debugging purposes
proc LPDelayCheck { } {
    # result of the LP should be the minimum clock period
    # Load the upsize and downsize procedures information
    #source /w/design/puneet/projects/eco/scripts/sizingProcs.tcl
    
    # to use mosek instead of lp_solve
    set MOSEK 0
    
    set freeList [list]
    global gatesList
    numberGates
    set riseOffset 0
    set fallOffset [llength $gatesList]
    setEcoMode -refinePlace false

    set OUTPUTFILE "LPassignment.lp";
    set fid [open $OUTPUTFILE "w"]
    puts "OPENING FILE ${OUTPUTFILE}"
    
    
    
    set cc [get_clocks]
    set clockPeriod [get_property $cc period]
    
    set minDelay true
    
    if {$MOSEK} {
        puts $fid "minimize\n tmax\n" nonewline
    } else {
        puts $fid "min: tmax" nonewline
    }
    
    if { ! $MOSEK} {
        puts $fid ";\n0 <= tmax;\n" nonewline
    } else {
     puts $fid "subject to\n" nonewline 
    }

    # Write LP
    foreach_in_collection iCell [get_cells] {
        set gname [get_property $iCell hierarchical_name]
        set cgate [get_cells $gname]
        set cellname0 [get_property $cgate ref_lib_cell_name]
        set cpins [get_pins $gname/*]
        set CInd [getGateIndex $iCell]
        #puts  "$gname $CInd"

        foreach_in_collection iPin $cpins {
            

            if {([get_property $iCell is_sequential] == "true") && 
                ([get_property $iPin direction]  == "out") } {
                
                # **** FLIP FLOP OUTPUTS **** 
                set pname [get_property $iPin ref_lib_pin_name]
                set phname [get_property $iPin hierarchical_name]
                
                set ac [get_arcs -from $phname]
                if {[sizeof_collection $ac] != 0} {
                    set amr [get_property $iPin arrival_max_rise]
                    set amf [get_property $iPin arrival_max_fall]
                    set outVar $CInd
                    puts $fid "-G_${phname}_r${outVar} <= -$amr" nonewline
                    if {! $MOSEK} {
                      puts $fid ";" nonewline
                    } 
                    puts $fid "\n" nonewline
                    
                    puts $fid "-G_${phname}_f${outVar} <= -$amf" nonewline
                    if {! $MOSEK} {
                      puts $fid ";" nonewline
                    } 
                    puts $fid "\n" nonewline
                    #lappend freeList "G_${phname}_f${outVar}"
                    #lappend freeList "G_${phname}_r${outVar}"
                }
            }


            if {([get_property $iPin direction] == "in") && 
                ([get_property $iPin is_async]  == "false") && 
                ([get_property $iPin is_clock]  == "false") } {
                # INPUTS 
                set pname [get_property $iPin ref_lib_pin_name]
                set phname [get_property $iPin hierarchical_name]
                if { [get_property $iPin is_rise_edge_triggered_data] == "true" } {
                    # FLIP-FLOP INPUTS
                    # compute the delay into this gate
                    set ac [get_arcs -to $phname]
                    set FIGInd -1
                    foreach_in_collection ca $ac {
                        if { [get_property $ca arc_type] == "setup_rising"} {
                            set setupRise [get_property $ca delay_max_rise]
                            set setupFall [get_property $ca delay_max_fall]
                        } else {
                            set sourcepin [get_property $ca source_pin]
                            if { [get_property $sourcepin is_port] } {
                                # add the delay at the port
                                set amr [get_property $sourcepin arrival_max_rise]
                                set amf [get_property $sourcepin arrival_max_fall]
                            } else {
                                set srcGate [getGateFromPin $sourcepin]
                                set FIGInd [getGateIndex $srcGate]
                                set FIPinName [get_property $sourcepin hierarchical_name] 
                            }
                            set dmr [get_property $ca delay_max_rise]
                            set dmf [get_property $ca delay_max_fall]
                        }
                    }
                    
                    
                    # rising path from input
                    if {$FIGInd != "-1"} {
                        if {$minDelay} {
                            set constb [expr -$dmr - $setupRise]
                            puts $fid "-tmax + " nonewline
                        } else {
                            set constb [expr -$dmr - $setupRise + $clockPeriod]
                        }
                        set fiVar $FIGInd
                        puts $fid "G_${FIPinName}_r${fiVar} <= ${constb}" nonewline
                        if {! $MOSEK} {
                          puts $fid ";" nonewline
                        }
                        puts $fid "\n" nonewline
                    } 
                    
                    
                    # falling path from input

                    if {$FIGInd != "-1"} {
                        if {$minDelay} {
                            set constb [expr -$dmf - $setupFall]
                            puts $fid "-tmax + " nonewline
                        } else {
                            set constb [expr -$dmf - $setupFall + $clockPeriod]
                        }
                        set fiVar $FIGInd
                        puts $fid "G_${FIPinName}_f${fiVar} <= ${constb}" nonewline
                        if {! $MOSEK} {
                          puts $fid ";" nonewline
                        }
                        puts $fid "\n" nonewline
                    } 
                    
                    
                } else {
                    # REGULAR GATE INPUTS
                    # compute the delay into this gate
                    set ac [get_arcs -to $phname]
                    set FIGInd -1
                    foreach_in_collection ca $ac {
                        set sourcepin [get_property $ca source_pin]
                        if { [get_property $sourcepin is_port] } {
                            # add the delay at the port
                            set amr [get_property $sourcepin arrival_max_rise]
                            set amf [get_property $sourcepin arrival_max_fall]
                        } else {
                            set srcGate [getGateFromPin $sourcepin]
                            set FIGInd [getGateIndex $srcGate]
                            set FIPinName [get_property $sourcepin hierarchical_name] 
                        }
                        set dmr [get_property $ca delay_max_rise]
                        set dmf [get_property $ca delay_max_fall]
                    }
                    
                    set ac [get_arcs -from $phname]
                    foreach_in_collection ca $ac {
                        set sinkpin [get_property $ca sink_pin]
                        set tsense [get_property $ca sense]
                        set sinkPinName [get_property $sinkpin hierarchical_name]
                        #puts $fid "$phname to $sinkPinName $tsense" 
                        
                        # rising path from input
                        set outVar $CInd
                        set constb [expr -$dmr]
                        if {$FIGInd != "-1"} {
                            set fiVar $FIGInd
                            if { $tsense == "negative_unate" } {
                                puts $fid "G_${FIPinName}_f${fiVar} " nonewline
                            } 
                            if { $tsense == "positive_unate" } {
                                puts $fid "G_${FIPinName}_r${fiVar} " nonewline
                            }
                            lappend freeList "G_${sinkPinName}_r${outVar}"
                        } else {
                            set constb [expr $constb - $amr]
                            if {$MOSEK} {
                                lappend freeList "G_${sinkPinName}_r${outVar}"
                            }
                        }
                        

                        puts $fid "-G_${sinkPinName}_r${outVar}" nonewline
                        set constb [expr $constb - [get_property $ca delay_max_rise]]
                        
                        puts $fid " <= ${constb}" nonewline
                        if {! $MOSEK} {
                          puts $fid ";" nonewline
                        }
                        puts $fid "\n" nonewline

                        
               
                        # falling path from input
                        set outVar $CInd
                        set constb [expr -$dmf]
                        if {$FIGInd != "-1"} {
                            set fiVar $FIGInd
                            if { $tsense == "negative_unate" } {
                                puts $fid "G_${FIPinName}_r${fiVar} " nonewline
                            } 
                            if { $tsense == "positive_unate" } {
                                puts $fid "G_${FIPinName}_f${fiVar} " nonewline
                            }                       
                            lappend freeList "G_${sinkPinName}_f${outVar}"
                        } else {
                            set constb [expr $constb - $amf]
                            if {$MOSEK} {
                                lappend freeList "G_${sinkPinName}_f${outVar}"
                            }
                        }

                        puts $fid "-G_${sinkPinName}_f${outVar}" nonewline
                        set constb [expr $constb - [get_property $ca delay_max_fall]]
                        puts $fid " <= ${constb}" nonewline

                        if {! $MOSEK} {
                          puts $fid ";" nonewline
                        }
                        puts $fid "\n" nonewline


                        
                    }
                    
                }
                
            }
        }
    }
    
    
    foreach_in_collection iPort [get_ports] {
        set pname [get_property $iPort hierarchical_name]
        set s0 [min [get_property $iPort slack_max_rise]  [get_property $iPort slack_max_fall]]
        set ac [get_arcs -to $pname]
        set FIGInd -1
        foreach_in_collection ca $ac {
            set sourcepin [get_property $ca source_pin]
            set sourcePinName [get_property $sourcepin hierarchical_name]
            set dmr [get_property $ca delay_max_rise]
            set dmf [get_property $ca delay_max_rise]
            if { [get_property $sourcepin is_port] } {
                # add the delay at the port
                set amr [get_property $sourcepin arrival_max_rise]
                set amf [get_property $sourcepin arrival_max_fall]
            } else {
                set srcGate [getGateFromPin $sourcepin]
                set FIGInd [getGateIndex $srcGate]
            }

            # falling path from input
            
            if {$FIGInd != "-1"} {
                set fiVar $FIGInd
                if {$minDelay} {
                    puts $fid "-tmax + " nonewline
                    set constb [expr -$dmf]
                } else {
                    set constb [expr -$dmf+$clockPeriod]
                }
                puts $fid "G_${sourcePinName}_f${fiVar} " nonewline
                puts $fid " <= ${constb}" nonewline
                if {! $MOSEK} {
                  puts $fid ";" nonewline
                }
                puts $fid "\n" nonewline
            } 
            

            # rising path from input
            if {$FIGInd != "-1"} {
                set fiVar [expr $FIGInd]
                if {$minDelay} {
                    puts $fid "-tmax + " nonewline
                    set constb [expr -$dmr]
                } else { 
                    set constb [expr -$dmr+$clockPeriod]
                }
                puts $fid "G_${sourcePinName}_r${fiVar} " nonewline
                puts $fid " <= ${constb}" nonewline
                if {! $MOSEK} {
                  puts $fid ";" nonewline
                }
                puts $fid "\n" nonewline
            } 
            
        }
    }
    
    if { ! $MOSEK } {
        puts $fid "free " nonewline
    } else {
        puts $fid "bounds\n" nonewline
    }
    
    set flag 0
    
    foreach fc $freeList {
        if {! $MOSEK } {
            if {$flag} {
                puts $fid "," nonewline
            } else {
                set flag 1
            }
        }
        puts $fid " $fc" nonewline
        if {$MOSEK} {
            puts $fid " free\n" nonewline
        }
    }
    if {! $MOSEK } {
        puts $fid ";\n\n" nonewline
    } else {
        puts "end\n\n" nonewline
    }
    
    close $fid
}


proc numberGates { } {
    global gatesList
    
    # create an empty list
    set gatesList [list]

    foreach_in_collection iCell [get_cells] {
        lappend gatesList [get_property $iCell hierarchical_name]
    }

}

proc getGateIndexName { cg } {
    global gatesList
    return [expr [lsearch $gatesList $cg] + 1]
}

proc getGateIndex { cg } {
    global gatesList
    return [expr [lsearch $gatesList [get_property $cg hierarchical_name]] + 1]
}

proc getGateFromPin { cpin } {
    set cpname [get_property $cpin hierarchical_name]
    set slind [string last "/" $cpname]
    set gname [string range $cpname 0 [expr $slind - 1]]
    return [get_cell $gname]

}
