/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Philip Chong <pchong@cadence.com>

ChangeLog:
2005-07-22: ChangeLog started
*/

#if !defined(oagUtilOptionParser_P)
#define oagUtilOptionParser_P

#include <vector>
#include <sstream>

#include "oagUtilOption.h"

namespace oagUtil {

/// \brief A command line options parser.
///
/// Command lines are expected to be in the form
/// \verbatim
/// command_name -opt1 [arg1] -opt2 [arg2] ...
/// \endverbatim
/// where opt{n} is the name of the {n}th option and arg{n} is the
/// corresponding argument.
///
/// Command line options can either take arguments or not;  there is
/// currently no support for options with optional arguments.  Options
/// may also be mandatory.  The parse() method flags an error if a
/// mandatory option is not present on the command line being parsed.
/// There is no support for parsing of arguments without a corresponding
/// option flag.  The order of options on the command line is ignored.
///
/// Users should first construct a OptionParser instance, giving a
/// short string which will be displayed when help is given (e.g. when
/// the -h option is encountered).
/// Users should then initialize the instance by using the add() method
/// to add all the options which should be accepted by the parser.
/// The add() method returns a pointer to an oagUtil::Option instance
/// which the user can use to reference the results from parsing.
/// Calling the parse() method parses the given command line,
/// storing the result in the internal state of the OptionParser instance.
/// The user should then use the isGiven() and getValue() methods
/// on the oagUtil::Options instances to query for the results of parsing.
///
/// If the option takes an argument, an "argument name" must be
/// specified for that option.
/// The argument name is a one-word string which will be displayed
/// when the user requests help (e.g. with the -h option).
///
/// For example:
/// \verbatim
/// code fragment:
///     parser.add("infile", "Input file", true, "file");
///     parser.add("outfile", "Output file", true, "file");
///
/// displayed help:
///     -infile file           Input file
///     -outfile file          Output file
/// \endverbatim

class OptionParser {

    friend class OptionParserUnitTest;

    typedef std::vector<Option *> OptionMap;

    public:
        /// Constuct an OptionParser.
        /// 
        /// A string must be provided.  The string must contain
        /// a short help message which will be displayed when
        /// the user invokes help (e.g. with the -h option).
        /// This message should explain the overall use of the command.
        ///
        /// \param helpString Help message.
        OptionParser(const char *helpString);

        ~OptionParser();

        /// Adds an option to the parser.
        ///
        /// The name of the option is the same as the string which
        /// appears after the dash on the command line
        /// (e.g. an option -foo has the name "foo")
        /// A string with a short help message explaining the meaning
        /// of the option must also be given.
        ///
        /// \param name Name of the option.
        /// \param helpString Help message.
        /// \param isMandatory True if the option is mandatory.
        /// \param argName Name for the argument to accept or NULL if
        ///        the option takes no argument.
        Option *add(const char  *name,
                    const char  *helpString,
                    bool        isMandatory,
                    const char  *argName = 0);

        /// Return the option corresponding to the given name.
        ///
        /// \param name Option name.
        /// \return The corresponding option or NULL if no option with the
        ///         given name was found.
        Option *getOption(const char *name);

        /// Parses the given command line.  The command line should
        /// include the command name itself at the front of the string.
        /// The caller should abort execution if this function returns false.
        ///
        /// \todo Not implemented yet.
        ///
        /// \param commandLine The command line to parse.
        /// \return True if the command line was successfully parsed.
        bool parse(const char *commandLine);

        /// Parses the given command line.  The command line is assumed
        /// to be already split into individual tokens in the standard
        /// argc/argv format.  Note that the command name itself should
        /// be in argv[0] and the last token is argv[argc-1].
        /// The caller should abort execution if this function returns false.
        ///
        /// \param argc Number of tokens to parse.
        /// \param argv Array of tokens to parse.
        /// \return True if the command line was successfully parsed.
        bool parse(int          argc,
                   const char   *argv[]);

        /// Gets the error message generated by the last call to parse().
        ///
        /// The caller should use this to get a message to print whenever
        /// the call to parse() returns false.
        ///
        /// \return Error message string.
        std::string getMessage() {
            return _message.str();
        }

    private:
        const char  *_helpString;
        OptionMap   _registry;

        std::stringstream _message;

        void _clearParseData();
        bool _checkMandatory();
        void _printHelp();
};

}

#endif
