/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Zhong Xiu <zxiu@andrew.cmu.edu>

ChangeLog:
2004-09-15: ChangeLog started
*/

#if !defined(oagTimerTimer_P)
#define oagTimerTimer_P

#include <iostream>
#include "oaDesignDB.h"

#include "oagTimerTPoint.h"
//#include "oagTimerCellData.h"
#include "oagTimerTPointMaster.h"
#include <map>
#include <set>
#include <string>

#include "oagTimerModel.h"
#include "oagTimerLibData.h"
#include "oagTimerLibParserInt.h"
#include "oagTimerWireModel.h"
#include "oagTimerElmoreWireModel.h"

namespace oagTimer {

// forward decls
class TermObserver;
class InstTermObserver;
class ElmoreWireModel;

/*!
  The vector of pairs, the first member of the pair is the oaNet pointer,
  the second member of the pair is the slack value of this net.
*/
typedef std::vector<std::pair<oaOccNet*, DelayType> > netsSlacks;
/*! 
  The vector of pairs, 
  the first member of the pair is the oaOccObject pointer,
  the second member of the pair is a boolean value, 
  true for rise and false for fall.
*/
typedef std::vector<std::pair<oaOccObject*, bool> > nodesSlopeDir;

//! The Timer class has all APIs for the user to use this timer package.
/*!
  The Timer class provides the standardized APIs for the user to use this
  timer package, including .lib .sdc files parsers, create and destroy
  the timing points, full and incremental timing analyisis functions.
*/
class Timer {
  friend class CellData;
  friend class DesignTool;
  friend class Opt;
  friend class TPoint;
  friend class Report;
  friend class TimerUnitTest;
  friend class Sensitivity;
  friend class DuetSensitivity;
  friend class Util;
  friend class TimerUtil;
  friend class KickMove;
  friend class SubTimer;

  /* To be removed */
  friend class PowerSenseOpt;

///////////////////////////////////////////////////////////////////////////////

  public:
  
    ///////////////////////////////////////////////////////////////////////////
    /// @name Constructors and Destructors
    //@{

    /// Constructor.
    Timer();

    /// Destructor.
    ~Timer();

    //@}

    ///////////////////////////////////////////////////////////////////////////
    /// @name Initialization Functions
    //@{

    /// Read in the timing library file.
    ///
    /// \param filename The timing library filename.
    /// \param libertyLibname The name of the lib containing the standard
    ///        library cells to annotate with the timing data from the
    ///        specified timing library file.  If NULL, use the name
    ///        specified withinn the library file itself.
    /// \param libertyViewname The name for the view to be annotated
    ///        with the timing data.  If NULL, use the default viewname
    ///        "abstract".
    void readLibrary(const char *filename,
                     const char *libertyLibname = 0,
                     const char *libertyViewname = 0);

    /// Adds a design to the Timer.
    /// Designs must be registered to the Timer using this function before
    /// any timing operations can be done.
    ///
    /// \param design Design to add.
    void add(oaDesign *design);

    void debug_TimingFlag(oaDesign *design);

    /// Removes a design from the Timer.
    /// Designs must be unregistered from the Timer using this function
    /// before they are purged.
    ///
    /// \param design Design to remove.
    void remove(oaDesign *design);

    /// Read an SDC file and annotate the given
    /// design with the timing constraints from that file.
    ///
    /// \param filename The SDC contraint filename.
    /// \param design The design to annotate.
    void readConstraints(const char *filename,
                         oaDesign   *design);

    //void createSwapList(oaDesign *design);

    /// Set the wire model.
    ///
    /// \param model The WireModel to use.
    void setWireModel(WireModel *model) {
        _wireModel = model;
	elmoreAnalysis = false;
    }

    /// Set Elmore Wire Model
    ///
    /// \param model The Elmore wire model to use
    void setWireModel(ElmoreWireModel *model){
	_elmoreWireModel = model;
	const char *tUnit = libParseData.tUnit;
	const char *capUnit = libParseData.capUnit;
	std::cout << "Time Unit: " << tUnit << std::endl;
	std::cout << "Cap Load Unit: " << capUnit << std::endl;
	_elmoreWireModel->setUnit(tUnit,capUnit);
	elmoreAnalysis = true;
    }

    /// Set Slew degradation computation to on
    ///
    void setSlewDegr(){
	slewDegr = true;
    }

    /// Set the worst arrival time of this design
    TimeType setWorstArr(oaDesign *design);

    //@}

    ///////////////////////////////////////////////////////////////////////////
    /// @name Query Functions
    //@{

    /// Returns the slack of an OccTerm/OccInstTerm. 
    /// This function run full timing analysis to update
    /// the value if it is not valid.
    ///
    /// \param oPtr OccTerm/OccInstTerm to query.
    /// \return Slack for the queried object.
    DelayType getFullSlack(oaOccObject *oPtr);
    
    /// Returns the slack of an OccTerm/OccInstTerm. 
    /// This function uses incremental timing analysis to update
    /// the value if it is not valid.
    ///
    /// \param oPtr OccTerm/OccInstTerm to query.
    /// \return Slack for the queried object.
    DelayType getSlack(oaOccObject *oPtr);

    /// Returns the arrival time of an OccTerm/OccInstTerm. 
    /// This function uses incremental timing analysis to update
    /// the value if it is not valid.
    ///
    /// \param oPtr OccTerm/OccInstTerm to query.
    /// \return Arrival time for the queried object.
    TimeType getArr(oaOccObject *oPtr);

    /// Returns the required arrival time of an OccTerm/OccInstTerm. 
    /// This function uses incremental timing analysis to update
    /// the value if it is not valid.
    ///
    /// \param oPtr OccTerm/OccInstTerm to query.
    /// \return Rquired arrival time for the queried object.
    TimeType getReq(oaOccObject *oPtr);

    /// Returns the fall arrival time of an OccTerm/OccInstTerm. 
    /// This function uses incremental timing analysis to update
    /// the value if it is not valid.
    ///
    /// \param oPtr OccTerm/OccInstTerm to query.
    /// \return Fall arrival time for the queried object.
    TimeType getFallArr(oaOccObject *oPtr);

    /// Returns the rise arrival time of an OccTerm/OccInstTerm. 
    /// This function uses incremental timing analysis to update
    /// the value if it is not valid.
    ///
    /// \param oPtr OccTerm/OccInstTerm to query.
    /// \return Rise arrival time for the queried object.
    TimeType getRiseArr(oaOccObject *oPtr);

    /// Returns the fall required arrival time of a OccTerm/OccInstTerm. 
    /// This function updates the value if it is not valid.
    ///
    /// \param oPtr OccTerm/OccInstTerm to query.
    /// \return Fall required arrival time for the queried object.
    TimeType getFallReq(oaOccObject *oPtr);

    /// Returns the rise required arrival time of a OccTerm/OccInstTerm. 
    /// This function updates the value if it is not valid.
    ///
    /// \param oPtr OccTerm/OccInstTerm to query.
    /// \return Rise required arrival time for the queried object.
    TimeType getRiseReq(oaOccObject *oPtr);

    /// Returns the transition time of an OccInstTerm. 
    /// This function does not perform timing analysis to update
    /// the value if it is not valid.
    ///
    /// \param oPtr OccInstTerm to query.
    /// \return tranistion time for the queried object.
    DelayType getSlew(oaOccObject *oPtr);

    /// Returns the fall transition time of an OccTerm/OccInstTerm. 
    /// This function does not perform timing analysis to update
    /// the value if it is not valid.
    ///
    /// \param oPtr OccTerm/OccInstTerm to query.
    /// \return Fall transition time for the queried object.
    DelayType getFallSlew(oaOccObject *oPtr);

    /// Returns the rise transition time of an OccTerm/OccInstTerm. 
    /// This function does not perform timing analysis to update
    /// the value if it is not valid.
    ///
    /// \param oPtr OccTerm/OccInstTerm to query.
    /// \return Rise transition time for the queried object.
    DelayType getRiseSlew(oaOccObject *oPtr);

    //@}


///////////////////////////////////////////////////////////////////////////////

    DelayType getWorstSlack();
    DelayType getWorstArr();
    double getSetupTime(oaOccInstTerm *iTerm);
    
    // Check that timing is not violated at PO
    bool isValidTiming();

  public:
    /** @name Timing analyisis on nets 
     */
    //@{
    DelayType getNetSlack(const oaOccNet *net);
    //@}

    /** @name Path query functions 
     */
    //@{
    void findWorstPath(nodesSlopeDir & path);
    void fromPath(nodesSlopeDir & path, oaOccObject *block);
    void toPath(nodesSlopeDir & path, oaOccObject *block);
    void throughPath(nodesSlopeDir & path, oaOccObject *block);
    //@}

    double getPeriod();
  public:
    static void invalidateFanout(oaOccNet *net);
    static void invalidateFanin(oaOccNet *net);
/*To be removed*/
    void updateAll(){
	//std::cout << "UpdateAllTiming()" << std::endl;
	updateAllArr();
	updateAllReq();
    }
  private:
    static void invalidateAllFanin(oaOccInst *inst);

  private:
    /** @name Incremental timing analyisis 
     *  Incrementally update the arrival time, required arrival time; 
     *  do fan in and fan out cones propagation
     */
    //@{
    void updateArr(oaOccObject *block);
    void updateClockArr(oaOccObject *block);
    oaOccNet* getOccNet(oaOccObject *block);
    void updateNetPathReq(TPoint *p, TPoint *tp);
    void initReq(oaOccObject *block);
    void initPOReq(oaOccObject *block);
    void initPIArr(oaOccObject *block);
    void initClockPIArr(oaOccObject *block);
    void updateNetLoadDelay(oaOccObject *block);
    void updateNetArr(oaOccObject *block);
    void updateClockNetArr(oaOccObject *block);
    void updateCellArr(oaOccObject *block);
    void updateClockCellArr(oaOccObject *block);
    void updateNetReq(oaOccObject *block);
    void updateCellReq(oaOccObject *block);
    void takeWorstSlew(oaOccObject *block);
    int  getClockIndex(oaOccObject *block);
    void setMultiCyclePaths(oaOccObject *block); 
    bool timingDataCompare(std::vector<timingData*> &v1, std::vector<timingData*> &v2);
    void timingDataCopy(std::vector<timingData*> &v1, std::vector<timingData*> &v2);
    void slewDataCopy(TPoint *p); 
    void insertTimingData(std::vector<timingData*> &timingDataVector, timingData *d);
    int  findTimingData (TPoint *p, int clockIndex);
    void updateNetPathArr(TPoint *p, TPoint *tp);
    void updateClockNetPathArr(TPoint *p, TPoint *tp);
    void setMultiCycleConstraint(oaOccObject *block);
    DelayType greatestCommonDivider(DelayType d1, DelayType d2);
    std::string getBlockName(oaOccObject *block);
    void setSlewTimingValidOpt(TPoint *p); 
    void clearVector(std::vector<timingData*> &v); 
    void initVector(std::vector<timingData*> &v);
    //int createNewClock(int x);

    void updateCellPathReq(TPointMaster::pathVector::iterator i, 
                           TPoint *p,
                           oaOccInstTerm *other);

   void updateCellPathArr(TPointMaster *tm,
            TPoint *p, 
            TPointMaster::pathVector::iterator i, 
            TPoint *tp);

   bool updateClockCellPathArr(TPointMaster *tm,
            TPoint *p, 
            TPointMaster::pathVector::iterator i, 
            TPoint *tp);

    void updateReq(oaOccObject *block);
    void updateAllArr();
    void updateAllReq();
    //@}

    void fromPath(nodesSlopeDir & path, oaOccObject *block,
                  bool isStartingClock);
    void toPath(nodesSlopeDir & path, oaOccObject *block,
                bool isEndingClock);
    void compareSlack(oaOccObject *block, bool currentRise,
                      TPoint *tp, bool & isRise,
                      DelayType & worstSlack, 
                      oaOccObject * & blockWithWorstSlack);

  private:
    /*! the vector of all clocks */
    std::set<oaOccObject *> clockPins;  

    /*! an callback pointer for oaTerm */
    TermObserver *_termCB;
    /*! an callback pointer for oaInstTerm */
    InstTermObserver *_instTermCB;
    /*! the standard cell library name */
    oaScalarName stdLib;


///////////////////////////////////////////////////////////////////////////////

  private:
    WireModel *_wireModel;
    ElmoreWireModel *_elmoreWireModel;
    bool elmoreAnalysis;
    std::set<oaDesign *> _watchedDesigns;
    bool slewDegr;
    TimeType worstArr;

    //To be removed
    oaDesign *dsgn;

  public:
    static oaOccObject *findOccObj(oaOccurrence *occ, std::string name);
    static oaOccInstTerm *getOccFromMod(oaOccInst *inst, oaModTerm *term);

    // Use to enable incremental timing speedup by propagating stored slew and delay
    // values
    // This happen only when input slew does not change by certain percentage
    bool optIncrTimer;
    
    void clearDesignTimingData();
    void printCell(oaOccInst *i);
    void printDetailTiming();
    void printAll();
    void printAllPO();
    void checkRatFlags();

///////////////////////////////////////////////////////////////////////////////

};

}

#endif
