/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 
This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Zhong Xiu <zxiu@andrew.cmu.edu>

ChangeLog:
2004-09-15: ChangeLog started
*/

#if !defined(oagTPointMaster_P)
#define oagTPointMaster_P

#include <assert.h>
#include <vector>
#include "oaDesignDB.h"

#include "oagTimerModel.h"
#include "oagTimerTPoint.h"

namespace oagTimer {

/*! 
  This structure contains the necessary information of an internal arc.
*/
struct TimerPath {
    oaModTerm *other; //!< the associated term
    ModelType *delay; //!< the delay model of this path
    ModelType *slew; //!< the slew model of this path
    int transition; //!< the transition type
};

/*!
  The TPointMaster class defines the necessary information
  at a point on the standard cell, which is an instTerm in the design
  and aTerm in the standard cell. When we do timing analysis, we need
  to go and find the corresponding TPointMaster and get all
  the timing information such as load, cap, delay table and slew table. 

  \todo Timing models are explicitly copied into the TPointMaster,
  once for each path which uses the model.
  This should be replaced by a pointer reference instead.
*/
class TPointMaster {
  friend class TPoint;
  friend class Timer;
  friend class Report;
  friend class LibData;
  friend class ElmoreWireModel;
  friend class PowerSenseOpt;
  friend class DuetSensitivity;
  friend class Opt;
  friend class Util;
  friend class TimerUtil;
  friend class SubTimer;

  public:
    static const int TRANSITION_SRC_RISE = 0x01;
    static const int TRANSITION_SRC_FALL = 0x02;
    static const int TRANSITION_DST_RISE = 0x10;
    static const int TRANSITION_DST_FALL = 0x20;

  public:
    /*! The constructor. */
    TPointMaster() { loadLimit = 0.0; }
    /*! The destructor. */
    ~TPointMaster();

  public:
    /*! The capacitance at this timing point master. */
    double cap;
    /*! The load limit at this timing point master. */
    double loadLimit;

  private:
    /*! This defines the vector of the paths */
    typedef std::vector<TimerPath> pathVector;
    /*! A vector of paths which end at this timing point */
    pathVector inPaths;
    /*! A vector of paths which goes from this timing point */
    pathVector outPaths;

  private:
    /*! This defines the vector of the models */
    typedef std::vector<ModelType *> modelVector;
    /*! The vector of models associated with this timing point */
    modelVector _models;

///////////////////////////////////////////////////////////////////////////////

  private:
    // Static vars
    static oaVoidPointerAppDef<oaModTerm> *_termAppDef;

    /// Initialize the appDefs associated with TPointMasters.
    static void initAppDefs();

    /// Create a TPointMaster for the given ModTerm.
    ///
    /// It is a fatal error to try to create a TPointMaster for a ModTerm
    /// which already has a TPointMaster.
    ///
    /// \param term The ModTerm for which to create TPointMaster.
    /// \return The new TPointMaster.
    static TPointMaster *create(oaModTerm *term);

    /// Get the TPointMaster for the given ModTerm, or 0 if 
    /// the ModTerm has no TPointMaster.
    ///
    /// \param term The ModTerm to query.
    /// \return The corresponding TPointMaster.
    static TPointMaster *get(oaModTerm *term);

  private:
    /// Fill in the data fields for a TimerPath.
    ///
    /// \todo This should be moved to a TimerPath constructor.
    ///
    /// \param p Reference to the TimerPath to fill in.
    /// \param term The ModTerm for the other point on the path.
    /// \param delay The delay model of this path.
    /// \param slew The slew model of this path.
    /// \param transition The type of this path's transition.
    void _makePath(TimerPath        &p,
                   oaModTerm        *term,
                   const ModelType  *delay,
                   const ModelType  *slew,
                   int              transition);

  public:
    /// Create a new path and add it to the list of input paths for this
    /// TPointMaster.
    ///
    /// \param term The ModTerm for the other point on the path.
    /// \param delay The delay model of this path.
    /// \param slew The slew model of this path.
    /// \param transition The type of this path's transition.
    void addInPath(oaModTerm        *term,
                   const ModelType  *delay,
                   const ModelType  *slew,
                   int              transition);

    /// Create a new path and add it to the list of output paths for this
    /// TPointMaster.
    ///
    /// \param term The ModTerm for the other point on the path.
    /// \param delay The delay model of this path.
    /// \param slew The slew model of this path.
    /// \param transition The type of this path's transition.
    void addOutPath(oaModTerm       *term,
                    const ModelType *delay,
                    const ModelType *slew,
                    int             transition);

///////////////////////////////////////////////////////////////////////////////

};

}

#endif
