/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Philip Chong <pchong@cadence.com>

ChangeLog:
2005-07-24: ChangeLog started
*/

#include "oagUtilOptionParserUnitTest.h"
#include "oagUtilOptionParser.h"

#include <cstring>

namespace oagUtil {

int OptionParserUnitTest::runAllTests(void) {
    OptionParser m("help string");
    bool r;
    OptionParser::OptionMap::iterator p;
    
    // add() tests
    // add 1.1 : Parser should start off empty
    logOneTest(m._registry.empty(), "add 1.1");
    // add 1.2 : Adding an option should return a new oagUtil::Option
    Option *o1 = m.add("opt1", "opt1 help", false);
    logOneTest(o1, "add 1.2");
    // add 1.3 : Added option should be in registry
    p = std::find(m._registry.begin(), m._registry.end(), o1);
    logOneTest(p != m._registry.end(), "add 1.3");

    // add 2.1 : Adding an option should return a new oagUtil::Option
    Option *o2 = m.add("opt2", "opt2 help", true, "arg2");
    logOneTest(o2, "add 2.1");
    // add 2.2 : Added option should be in registry
    p = std::find(m._registry.begin(), m._registry.end(), o2);
    logOneTest(p != m._registry.end(), "add 2.2");
    // add 2.3 : First option should still exist after second option added
    p = std::find(m._registry.begin(), m._registry.end(), o1);
    logOneTest(p != m._registry.end(), "add 2.3");

    // add 3.1 : Option 1 name should be set properly
    logOneTest(std::strcmp(o1->_name, "opt1") == 0, "add 3.1");
    // add 3.2 : Option 1 helpstring should be set properly
    logOneTest(std::strcmp(o1->_helpString, "opt1 help") == 0, "add 3.2");
    // add 3.3 : Option 1 isMandatory should be false
    logOneTest(!o1->_isMandatory, "add 3.3");
    // add 3.4 : Option 1 argName should not be set
    logOneTest(!o1->_argName, "add 3.4");
    // add 3.5 : Option 2 name should be set properly
    logOneTest(std::strcmp(o2->_name, "opt2") == 0, "add 3.5");
    // add 3.6 : Option 2 helpstring should be set properly
    logOneTest(std::strcmp(o2->_helpString, "opt2 help") == 0, "add 3.6");
    // add 3.7 : Option 2 isMandatory should be true
    logOneTest(o2->_isMandatory, "add 3.7");
    // add 3.8 : Option 2 argName should be set
    logOneTest(std::strcmp(o2->_argName, "arg2") == 0, "add 3.8");

    // parse() tests
    // parse 1.1 : valid command line should be accepted by parser
    const char *argv1[] = {"command_name", "-opt2", "testarg"};
    r = m.parse(3, argv1);
    logOneTest(r, "parse 1.1");
    // parse 1.2 : parser should not touch option 1
    logOneTest(!o1->isGiven(), "parse 1.2");
    // parse 1.3 : parser should touch option 2
    logOneTest(o2->isGiven(), "parse 1.3");
    // parse 1.4 : parser should set argument in option 2
    logOneTest(std::strcmp(o2->getValue(), "testarg") == 0, "parse 1.4");
    // parse 1.5 : invalid command line should be rejected by parser
    r = m.parse(2, argv1);
    logOneTest(!r, "parse 1.5");
    // parse 1.6 : invalid command line should be rejected by parser
    r = m.parse(1, argv1);
    logOneTest(!r, "parse 1.6");

    // parse 2.1 : valid command line should be accepted by parser
    const char *argv2[] = {"command_name", "-opt2", "testarg", "-opt1"};
    r = m.parse(4, argv2);
    logOneTest(r, "parse 2.1");
    // parse 2.2 : parser should touch option 1
    logOneTest(o1->isGiven(), "parse 2.2");
    // parse 2.3 : parser should touch option 2
    logOneTest(o2->isGiven(), "parse 2.3");
    // parse 2.4 : parser should set argument in option 2
    logOneTest(std::strcmp(o2->getValue(), "testarg") == 0, "parse 2.4");

    // parse 3.1 : invalid command line should be rejected by parser
    const char *argv3[] = {"command_name", "-bad", "-opt2", "testarg"};
    r = m.parse(4, argv3);
    logOneTest(!r, "parse 3.1");
    // parse 3.2 : invalid command line should be rejected by parser
    r = m.parse(2, argv3);
    logOneTest(!r, "parse 3.2");
    // parse 3.3 : invalid command line should be rejected by parser
    r = m.parse(1, argv3);
    logOneTest(!r, "parse 3.3");

    // getOption() tests
    // getOption 1.1 : option 1 should be recognized by parser
    logOneTest(m.getOption("opt1") == o1, "getOption 1.1");
    // getOption 1.2 : option 2 should be recognized by parser
    logOneTest(m.getOption("opt2") == o2, "getOption 1.2");
    // getOption 1.3 : option 3 does not exist
    logOneTest(!m.getOption("opt3"), "getOption 1.3");

    return getReturnCode();
}

}
