#include "solverIF.h"

namespace solver {
  MSKenv_t solverIF::env = NULL;
  MSKtask_t solverIF::task = NULL;
  int solverIF::solverType = SIF_MOSEK;
  
  static void MSKAPI printstr(void *handle, char str[]) {
    //printf("%s",str);       
  } /* printstr */
  
  
  void solverIF::initializeSolver(int nVars) {
    //cout << "Initializing solver" << endl;
    
    if (solverIF::solverType == SIF_LPSOLVE) {
      _lp = make_lp(0, nVars);
      set_add_rowmode( _lp, TRUE);
    }
    
    if (solverIF::solverType == SIF_MOSEK) {   
      r = MSK_makeenv(&solverIF::env, NULL, NULL, NULL, NULL);
      
      
      if (r == MSK_RES_OK ) {
        MSK_linkfunctoenvstream(solverIF::env, MSK_STREAM_LOG, NULL, printstr);
        
      }
      else 
        cout << "MOSEK error" << endl;
      
      r = MSK_initenv(solverIF::env);
      
      
      if (r == MSK_RES_OK ) {
        MSK_echoenv(solverIF::env, MSK_STREAM_MSG, "Making the MOSEK optimization task\n");
        r = MSK_maketask(solverIF::env, 0, 0, &solverIF::task); // initializes empty task
        
        //cout << "linking environment streams" << endl;
        MSK_linkfunctotaskstream(solverIF::task, MSK_STREAM_LOG, NULL, printstr);
        MSK_linkfiletotaskstream(solverIF::task, MSK_STREAM_LOG, "moseklog.txt", 0);
        MSK_linkfiletotaskstream(solverIF::task, MSK_STREAM_MSG, "moseklog.txt", 0);
        MSK_linkfiletotaskstream(solverIF::task, MSK_STREAM_ERR, "moseklog.txt", 0);
        MSK_linkfiletotaskstream(solverIF::task, MSK_STREAM_WRN, "moseklog.txt", 0);
      }
      else
        cout << "MOSEK error" << endl;
      
      if (r == MSK_RES_OK ) {
        r = MSK_putmaxnumvar(solverIF::task, nVars);
      }
      
      if (r == MSK_RES_OK ) {
        r = MSK_append(solverIF::task, MSK_ACC_VAR, nVars);
      }
      
      if (r == MSK_RES_OK ) {
        r = MSK_putcfix(solverIF::task, 0.0);
      }
      else
        cout << "MOSEK error" << endl;
    }
    _nConstrs = 0;
    _nVars = nVars;
    
  }
  
  void solverIF::setBound(int n, double lb, double ub) {
    
    if (solverIF::solverType == SIF_LPSOLVE) {
      set_bounds(_lp, n, lb, ub);
    }
    
    if (solverIF::solverType == SIF_MOSEK) {
      if (lb == ub)
        r = MSK_putbound(solverIF::task, MSK_ACC_VAR, (n-1), MSK_BK_FX, lb, ub);
      else
        r = MSK_putbound(solverIF::task, MSK_ACC_VAR, (n-1), MSK_BK_RA, lb, ub);
      if (r != MSK_RES_OK )
        cout << "MOSEK error: setBound " << n <<", " << lb << " / " << ub << endl;
    }
  }
  
  void solverIF::setUpperBound(int n, double bd) {
    
    if (solverIF::solverType == SIF_LPSOLVE) {
      //set_free( _lp, n);
      set_bounds( _lp, n, -get_infinite(_lp), bd);
    } //
    
    if (solverIF::solverType == SIF_MOSEK) {      
      r = MSK_putbound(solverIF::task, MSK_ACC_VAR, (n-1), MSK_BK_UP, -MSK_INFINITY, bd);
      if (r != MSK_RES_OK )
        cout << "MOSEK error: setLowerBound" << endl;
    } //
  }
  
  void solverIF::setLowerBound(int n, double bd) {
    
    if (solverIF::solverType == SIF_LPSOLVE) {
      set_lowbo( _lp, n, bd);
    } //
    
    if (solverIF::solverType == SIF_MOSEK) {      
      r = MSK_putbound(solverIF::task, MSK_ACC_VAR, (n-1), MSK_BK_LO, bd, +MSK_INFINITY);
      if (r != MSK_RES_OK )
        cout << "MOSEK error: setLowerBound" << endl;
    } //
  }
  
  
  void solverIF::setFree(int n) {
    
    if (solverIF::solverType == SIF_LPSOLVE) {
      set_free(_lp, n);
    } //
    
    if (solverIF::solverType == SIF_MOSEK) {      
      r = MSK_putbound(solverIF::task, MSK_ACC_VAR, (n-1), MSK_BK_FR, -MSK_INFINITY, +MSK_INFINITY);
      if (r != MSK_RES_OK )
        cout << "MOSEK error: setFree" << endl;
    } //
  }
  
  
  void solverIF::addConstraint(int N, double * varCoeffs, int * varInds, int bdtype, double bd) {
    
    if (solverIF::solverType == SIF_LPSOLVE) {
      switch (bdtype) {
        case BD_LE:
          add_constraintex(_lp, N, varCoeffs, varInds, LE, bd);
          break;
        case BD_GE:
          add_constraintex(_lp, N, varCoeffs, varInds, GE, bd);
          break;
        case BD_EQ:
          add_constraintex(_lp, N, varCoeffs, varInds, EQ, bd);
          break;
        default:
          cout << "Unknown bound type: " << bdtype << endl;
          break;
      }
      ++_nConstrs;
    } //
    
    
    if (solverIF::solverType == SIF_MOSEK) {
      
      int fvarInds[N];
      double fvarCoeffs[N];
      int fN = 0;
      
      for (int i = 0; i < N; ++i) {
        if (fabs(varCoeffs[i]) > 1e10) {
            cout << "Magnitude error" << endl;
            assert(0);
          
        }
        if (!(fabs(varCoeffs[i]) < 1e-10)) { 
          fvarInds[fN] = (varInds[i]-1);
          fvarCoeffs[fN] = varCoeffs[i];
          ++fN;
        }
        
      }
      if (fN > 0) {
        r = MSK_append(solverIF::task, MSK_ACC_CON, 1); // add space for another constraint
        if (r != MSK_RES_OK )
          cout << "MOSEK error: addConstraint / append" << endl;
        switch (bdtype) {
          case BD_LE:
            r = MSK_putbound(solverIF::task, MSK_ACC_CON, _nConstrs, MSK_BK_UP, -MSK_INFINITY, bd);
            break;
          case BD_GE:
            r = MSK_putbound(solverIF::task, MSK_ACC_CON, _nConstrs, MSK_BK_LO, bd, +MSK_INFINITY);
            break;
          case BD_EQ:
            r = MSK_putbound(solverIF::task, MSK_ACC_CON, _nConstrs, MSK_BK_FX, bd, bd);
            break;
          default:
            cout << "Unknown bound type: " << bdtype << endl;
            break;
        }     
        if (r != MSK_RES_OK )
          cout << "MOSEK error: addConstraint / putbound" << endl;
        
        r = MSK_putavec(solverIF::task, MSK_ACC_CON, _nConstrs, fN, fvarInds, fvarCoeffs);
        if (r != MSK_RES_OK ) {
          cout << "MOSEK error: addConstraint / putavec " << _nConstrs << " / " << fN << "/" << N << "  nvars " << _nVars << endl;
          for (int i = 0; i < fN; ++i) {
            cout << "     "  << fvarInds[i] << ": " << fvarCoeffs[i] << endl;
          }
        }
        ++_nConstrs;
      }
    } //

  }
  
  /* Only input the lower triangular part */
  void solverIF::addQuadObjective(int N, double * val, int * indi, int * indj) {
    int findi[N];
    int findj[N];

    if (solverIF::solverType == SIF_LPSOLVE) {
      cout << "LP_SOLVE CANNOT HANDLE QP" << endl;
    } //
    
    
    if (solverIF::solverType == SIF_MOSEK) {
      for (int i = 0; i < N; ++i) {
        if (indi[i] < indj[i]) {
          findj[i] = indi[i]-1;
          findi[i] = indj[i]-1;
        }
        else {
          findi[i] = indi[i]-1;
          findj[i] = indj[i]-1;
        }       
      }
      r = MSK_putqobj(solverIF::task, N, findi, findj, val);
      if (r != MSK_RES_OK ) {
        cout << "MOSEK error: addQuadObjective " << endl;
      }
    } 
  }
  
  /* Only input the lower triangular part */
  void solverIF::addQuadObjectiveEntry(double val, int indi, int indj) {

    if (solverIF::solverType == SIF_LPSOLVE) {
      cout << "LP_SOLVE CANNOT HANDLE QP" << endl;
    } //
    
    
    if (solverIF::solverType == SIF_MOSEK) {
      if (indi < indj) {
        r = MSK_putqobjij(solverIF::task, (indj-1), (indi-1), val);
      }
      else {
        r = MSK_putqobjij(solverIF::task, (indi-1), (indj-1), val);
      }
      if (r != MSK_RES_OK ) {
        cout << "MOSEK error: addQuadObjective " << endl;
      }
    } 
  }
  
  void solverIF::setObjective(int N, double * varCoeffs, int * varInds) {
        
    if (solverIF::solverType == SIF_LPSOLVE) {
      set_add_rowmode( _lp, FALSE);
      set_obj_fnex(_lp, N, varCoeffs, varInds);
      set_minim(_lp);
    } //
    
    
    if (solverIF::solverType == SIF_MOSEK) {      
      for (int i = 0; i < N; ++i) {  
        r = MSK_putcj(solverIF::task, (varInds[i]-1), varCoeffs[i]); 
	  if (r != MSK_RES_OK ){
              std::cout << "MOSEK error: ind " << varInds[i] << "/ coeff " << varCoeffs[i] << std::endl;	
	    //break;
	  }
      }
      if (r != MSK_RES_OK ) {
        cout << "MOSEK error: setObjective" << endl;
        for (int i = 0; i < N; ++i) {  
            r = MSK_putcj(solverIF::task, (varInds[i]-1), varCoeffs[i]); 
        }
      }
    } //
  }
  
  int solverIF::solveLP(double * xx) {
    time_t etime;
    char * tstr;
    string sstr;
    if (solverIF::solverType == SIF_LPSOLVE) {
      /* Constraints END */
	std::cout << "contraint end" << std::endl;
      set_add_rowmode( _lp, FALSE);
      cout << get_Nrows(_lp) << " constraints" << endl;

      write_lp(_lp, "lpsolve.lp");
      
      set_verbose(_lp, IMPORTANT);
      

      time(&etime);
      tstr = ctime(&etime);
      sstr = string(tstr);
      //cout << "TIMEx Calling solver at " << sstr << endl;
      
      int ret = solve(_lp);
      double rt = time_elapsed(_lp);
      cout << "DUMPrt Solver Runtime " << rt << endl;
      
      time(&etime);
      tstr = ctime(&etime);
      sstr = string(tstr);
      //cout << "TIMEx Solver finished at " << sstr << endl;
      
      if (ret == OPTIMAL) {
        get_variables(_lp, xx);
        //for (uint i = 0; i < _nVars; ++i) {
       //   cout << i+1 << ": " << xx[i] << endl;
        //}
        //if (slackOpt)
        if (PRINT_LP) {
          print_solution(_lp, _nVars);
        }
        double obj = get_objective(_lp);
        return 1;  
      }
      //delete_lp(_lp);
      
      switch (ret) {
        case OPTIMAL:
          return 1;
          break;
        case NOMEMORY:
          cout << "Solver ran out of memory" << endl;
          return 0;
          break;
        case SUBOPTIMAL:
          cout << "Solution is suboptimal" << endl;
          return 0;
          break;
        case INFEASIBLE:
          cout << "Problem is infeasible" << endl;
          return 0;
          break;
        case UNBOUNDED:
          cout << "Problem is unbounded" << endl;
          return 0;
          break;
        case DEGENERATE:
          cout << "Problem is degenerate" << endl;
          return 0;
          break;
        case NUMFAILURE:
          cout << "Solver encountered a numerical failure" << endl;
          return 0;
          break;
        default:
          cout << "Solver encountered an unknown error: "<< ret << endl;
          return 0;
          break;
      }
    } //
    
    
    if (solverIF::solverType == SIF_MOSEK) {
      if (r == MSK_RES_OK) 
        r = MSK_putobjsense(solverIF::task, MSK_OBJECTIVE_SENSE_MINIMIZE);    
      MSK_writedata(task,"taskdump.lp");
      
      time(&etime);
      tstr = ctime(&etime);
      sstr = string(tstr);
      //cout << "TIMEx Calling solver at " << sstr << endl;
      
      if (r == MSK_RES_OK) 
        r = MSK_optimize(solverIF::task);
      else
        cout << "Mosek not OK , not optimizing" << endl;
      
      time(&etime);
      tstr = ctime(&etime);
      sstr = string(tstr);
      //cout << "TIMEx Solver finished at " << sstr << endl;
      
      MSKprostae prosta; 
      MSKsolstae solsta;
      MSK_getsolutionstatus( solverIF::task, MSK_SOL_BAS, &prosta, &solsta ); 
      
      MSK_solutionsummary(solverIF::task, MSK_STREAM_LOG);
      
      switch (solsta) {
        case MSK_SOL_STA_OPTIMAL:
          /* Successful */
	  //std::cout << "OPTIMAL" << std::endl;
          MSK_getsolutionslice(solverIF::task, MSK_SOL_BAS, MSK_SOL_ITEM_XX, 0, _nVars, xx); 
          //MSK_deletetask(&solverIF::task);
          //MSK_deleteenv(&solverIF::env);
          return 1;
          break;
        case MSK_SOL_STA_PRIM_INFEAS_CER:
        case MSK_SOL_STA_NEAR_PRIM_INFEAS_CER:
          //MSK_deletetask(&solverIF::task);
          //MSK_deleteenv(&solverIF::env);
          return 0;
          break;
        default:
          cout << "Other solution status (UNKNOWN): " << solsta << endl;
          return 0;
          break;
      }
    } //
    
    return 1;
  }
   int solverIF::solveQP(double * xx) {
    time_t etime;
    char * tstr;
    string sstr;
    
    
    if (solverIF::solverType == SIF_MOSEK) {
      if (r == MSK_RES_OK) 
        r = MSK_putobjsense(solverIF::task, MSK_OBJECTIVE_SENSE_MINIMIZE);    
      //MSK_writedata(task,"taskdump.lp");
      
      time(&etime);
      tstr = ctime(&etime);
      sstr = string(tstr);
      //cout << "TIMEx Calling solver at " << sstr << endl;
      
      if (r == MSK_RES_OK) 
        r = MSK_optimize(solverIF::task);
      else
        cout << "Mosek not OK , not optimizing" << endl;
      
      time(&etime);
      tstr = ctime(&etime);
      sstr = string(tstr);
      //cout << "TIMEx Solver finished at " << sstr << endl;
      
      MSKprostae prosta; 
      MSKsolstae solsta;
      MSK_getsolutionstatus( solverIF::task, MSK_SOL_ITR, NULL, &solsta ); 
      
      MSK_solutionsummary(solverIF::task, MSK_STREAM_LOG);
      
      switch (solsta) {
        case MSK_SOL_STA_OPTIMAL:
          /* Successful */
          MSK_getsolutionslice(solverIF::task, MSK_SOL_ITR, MSK_SOL_ITEM_XX, 0, _nVars, xx); 
          //MSK_deletetask(&solverIF::task);
          //MSK_deleteenv(&solverIF::env);
          return 1;
          break;
	case MSK_SOL_STA_NEAR_OPTIMAL:
          /* Good Enough */
          MSK_getsolutionslice(solverIF::task, MSK_SOL_ITR, MSK_SOL_ITEM_XX, 0, _nVars, xx); 
          //MSK_deletetask(&solverIF::task);
          //MSK_deleteenv(&solverIF::env);
          return 1;
          break;
        case MSK_SOL_STA_PRIM_INFEAS_CER:
        case MSK_SOL_STA_NEAR_PRIM_INFEAS_CER:
          //MSK_deletetask(&solverIF::task);
          //MSK_deleteenv(&solverIF::env);
          return 0;
          break;
        default:
          cout << "Other solution status (UNKNOWN): " << solsta << endl;
          return 0;
          break;
      }
    } //
    
    return 1;
  }
  void solverIF::deleteSolver() {
    if (solverIF::solverType == SIF_LPSOLVE) {
      delete_lp(_lp);
    }
      
    if (solverIF::solverType == SIF_MOSEK) {
      MSK_deletetask(&solverIF::task);
      MSK_deleteenv(&solverIF::env);
    }
  }

}
