/************************************************************************
* oagTimerUtil.cpp
* Added functionalities to compute cell area, cell leakage power
* and total leakage power of design
* by: Santiago Mok <smok@ee.ucla.edu>
* advised by: Prof. Puneet Gupta
*
**************************************************************************/

#include <iostream>
#include <cmath>
#include <algorithm>
#include "string.h"
#include "assert.h"
#include "oaDesignDB.h"
#include "oagTimerDesignTool.h"
#include "oagTimerUtil.h"
#include "oagTimerCellMaster.h"
#include "oagTimerTimer.h"
#include "oagTimerLibData.h"
#include "oagTimerLibParserInt.h"

#define MODE_READ 'r'
namespace oagTimer{

/*---------------------------------------------------------*/
using namespace oa;
using namespace std;
/*---------------------------------------------------------*/
//Constructor
Util::Util(){
}

//Destructor
Util::~Util(){}
/*---------------------------------------------------------*/
void
Util::getCellFootprint(oaString cellName, oaString &footprint){
    oaNativeNS NS;
    oaScalarName libName(NS, libParseData.libString);
    oaScalarName libCell(NS, cellName);
    oaScalarName libView(NS, libParseData.viewString);
    oaDesign *des = oaDesign::open(libName, libCell, libView, MODE_READ);
    oaModule *m = des->getTopModule();
    assert(m);
    CellMaster *cm = CellMaster::get(m);
    assert(cm);
    footprint=cm->footprint;
}
/*---------------------------------------------------------*/
double
Util::getCellArea(oaInst *inst){
    assert(inst);
    oaDesign *master = inst->getMaster();
    assert(master);
    oaModule *mod = master->getTopModule();
    assert(mod);
    CellMaster *cm = CellMaster::get(mod);
    assert(cm);
    return cm->area;
}
/*---------------------------------------------------------*/
double 
Util::getCellLeakagePower(oaModule *m){
    assert(m);
    CellMaster *cm = CellMaster::get(m);
    if(cm)
	return cm->leakage_power;
    return 0;
}
/*---------------------------------------------------------*/
double 
Util::getCellLeakagePower(oaModInst* modInst){
    //oaModule* mod = modInst->getTopModule();
    oaModule* mod = modInst->getMasterModule();
    return Util::getCellLeakagePower(mod);
}
/*---------------------------------------------------------*/
double 
Util::getCellLeakagePower(oaOccInst* occInst){
    oaInst* inst = occInst->getInst();
    return Util::getCellLeakagePower(inst);
}
/*---------------------------------------------------------*/
double 
Util::getCellLeakagePower(oaInst *inst){
    oaModule *mod = inst->getMaster()->getTopModule();
    assert(mod);
    CellMaster *cm = CellMaster::get(mod);
    if(cm)
	return cm->leakage_power;
    return 0;
}
/*---------------------------------------------------------*/
double 
Util::getCellLeakagePower(const char *s){
    oaNativeNS NS;
    oaString cellName = s;
    oaScalarName libName(NS, libParseData.libString);
    oaScalarName libCell(NS, cellName);
    oaScalarName libView(NS, libParseData.viewString);
    oaDesign *des = oaDesign::open(libName, libCell, libView, MODE_READ);
    oaModule *m = des->getTopModule();
    return Util::getCellLeakagePower(m);
}
/*---------------------------------------------------------*/
double 
Util::getCellLeakagePower(oaString cellName){
    oaNativeNS NS;
    oaScalarName libName(NS, libParseData.libString);
    oaScalarName libCell(NS, cellName);
    oaScalarName libView(NS, libParseData.viewString);
    oaDesign *des = oaDesign::open(libName, libCell, libView, MODE_READ);
    oaModule *m = des->getTopModule();
    assert(m);
    return Util::getCellLeakagePower(m);
}
/*---------------------------------------------------------*/
double 
Util::getTotalLeakPower(const oaDesign *design){
    assert(design);
    oaBlock *block = design->getTopBlock();
    assert(block);
    float leakPower = 0.0;
    oaIter<oaInst> iIter (block->getInsts());
    while(oaInst *inst = iIter.getNext()){
	leakPower += Util::getCellLeakagePower(inst);
    }
    return leakPower;
}
/*---------------------------------------------------------*/
double
Util::getPowerSensitivity(const char *curr, const char *other) {
    return (Util::getCellLeakagePower(other) - Util::getCellLeakagePower(curr));
}
/*---------------------------------------------------------*/
double 
Util::getCellInputCap(oaOccInstTerm *j, oaString s, Timer *timer){
    TPoint *pin = TPoint::get(j);
    assert(pin);
    if(pin->type != TIMING_POINT_SIGNAL_IN) return 0.0;

    oaName name;
    j->getTerm()->getName(name);
    oaModule* topCell = DesignTool::getCellTopMod(s);
    assert(topCell);
    oaModTerm *inTerm = oaModTerm::find(topCell,name);
    TPointMaster *tm = TPointMaster::get(inTerm);
    return tm->cap;
}
/*---------------------------------------------------------*/
oaOccInstTerm*
Util::findOccInTerms(oaDesign *design, oaModScalarInst *m){
    oaOccurrence *occ = design->getTopOccurrence();
    oaOccInstTerm *outputTerm;
    oaIter<oaOccInst> occInstIter (m->getOccInsts(occ));
    while(oaOccInst *occInst = occInstIter.getNext() ){
	oaIter<oaOccInstTerm> itIter (occInst->getInstTerms());
	while(oaOccInstTerm *occInstTerm = itIter.getNext()){		
	    oaTermType type(occInstTerm->getTerm()->getTermType());
	    if(type == oacInputTermType){
		return occInstTerm;
	    }
	}
    }
    //return outputTerm;
}
/*---------------------------------------------------------*/
oaOccInstTerm*
Util::findOccOutTerm(oaDesign *design, oaModScalarInst *m){
    oaOccurrence *occ = design->getTopOccurrence();
    oaOccInstTerm *outputTerm;
    oaIter<oaOccInst> occInstIter (m->getOccInsts(occ));
    while(oaOccInst *occInst = occInstIter.getNext() ){
	oaIter<oaOccInstTerm> itIter (occInst->getInstTerms());
	while(oaOccInstTerm *occInstTerm = itIter.getNext()){		
	    oaTermType type(occInstTerm->getTerm()->getTermType());
	    if(type == oacOutputTermType){
		return occInstTerm;
	    }
	}
    }
}
/*---------------------------------------------------------*/
oaModInstTerm*
Util::getOutInstTerm(oaModNet *net){
    oaIter<oaModInstTerm> iter (net->getInstTerms());
    while(oaModInstTerm *i = iter.getNext()){
	oaTermType type(i->getTerm()->getTermType());
	if(type == oacOutputTermType){
	    return i;
	}
    }
}
/*---------------------------------------------------------*/
DelayType 
Util::getIOSlack(oaModInst* i, oaDesign* design, Timer* timing, io_Type io){
#if defined(DEBUG)
    std::cout << "getIOSlack()" << std::endl;
#endif
    if(io == INPUT){
	//oaOccInstTerm* inputTerm = findOccIOTerm(design,static_cast<oaModScalarInst*>(i),io_Type(INPUT));
	//return timing->getSlack(inputTerm);
    }else if(io == OUTPUT){
	oaOccInstTerm* outputTerm = findOccOutTerm(design,static_cast<oaModScalarInst*>(i));
	return timing->getSlack(outputTerm);
    }
}
/*---------------------------------------------------------*/
TimeType 
Util::getIOArr(oaModInst* i, oaDesign* design, Timer* timing, io_Type io){
#if defined(DEBUG)
    std::cout << "getIOArr()" << std::endl;
#endif
    
    if(io == INPUT){
	//oaOccInstTerm* inputTerm = findOccIOTerm(design,static_cast<oaModScalarInst*>(i),io_Type(INPUT));
	//return timing->getArr(inputTerm);
    }else if(io == OUTPUT){
	oaOccInstTerm* outputTerm = findOccOutTerm(design,static_cast<oaModScalarInst*>(i));
	return timing->getArr(outputTerm);
    }
}
/*---------------------------------------------------------*/
TimeType 
Util::getIOArr(oaOccInst* i, oaDesign* design, Timer* timing, io_Type io){
#if defined(DEBUG)
    std::cout << "getIOArr()" << std::endl;
#endif
    
    if(io == INPUT){
	//oaOccInstTerm* inputTerm = findOccIOTerm(design,static_cast<oaModScalarInst*>(i->getModInst()),io_Type(INPUT));
	//return timing->getArr(inputTerm);
    }else if(io == OUTPUT){
	oaOccInstTerm* outputTerm = findOccOutTerm(design,static_cast<oaModScalarInst*>(i->getModInst()));
	return timing->getArr(outputTerm);
    }
}
/*---------------------------------------------------------*/
void 
Util::getMatchCells(vector<oaString> &list,int sizes){

    vector<oaString>::iterator iter;
    size_t name_index, size_index;
    oaString s = list.back();
    list.pop_back();
    string curr = static_cast<string>(s);
    name_index = curr.find_first_of("_");
    size_index = curr.find_last_of("_");

    string currCellName = curr.substr(0,name_index);
    string currCellSize = curr.substr(size_index+2);
    int currSize = atoi(currCellSize.c_str());
    for(iter = libParseData.libCellNames.begin(); 
	    iter != libParseData.libCellNames.end(); ++iter){
	string otherLibCell = static_cast<string>(oaString(*iter));
	string otherLibCellName = otherLibCell.substr(0,name_index);
	if(otherLibCellName == currCellName){
	    string otherLibCellSize = otherLibCell.substr(size_index+2);
	    int otherSize = atoi(otherLibCellSize.c_str());
	    if( (sizes == 1) && (otherSize > currSize)){
		list.push_back(*iter);	//Upsize only
	    }else if( (sizes == 2) && (otherSize < currSize)){
		list.push_back(*iter); //Downsize only
	    }else if( (sizes == 0) && (otherSize != currSize)){
		list.push_back(*iter); //All sizes
	    }
	}
    }
}
/*---------------------------------------------------------*/
bool
Util::isGreaterThan(oaString current, oaString other){
    size_t name_index, size_index;
    string curr = static_cast<string>(current);
    string othr = static_cast<string>(other);
    name_index = curr.find_first_of("_");
    size_index = curr.find_last_of("_");

    string currCellSize = curr.substr(size_index+2);
    string othrCellSize = othr.substr(size_index+2);

    int currSize = atoi(currCellSize.c_str());
    int othrSize = atoi(othrCellSize.c_str());

    if(othrSize > currSize)
	return true;
    return false;
}
/*---------------------------------------------------------*/
bool
Util::isLessThan(oaString current, oaString other){
    size_t name_index, size_index;
    string curr = static_cast<string>(current);
    string othr = static_cast<string>(other);
    name_index = curr.find_first_of("_");
    size_index = curr.find_last_of("_");

    string currCellSize = curr.substr(size_index+2);
    string othrCellSize = othr.substr(size_index+2);

    int currSize = atoi(currCellSize.c_str());
    int othrSize = atoi(othrCellSize.c_str());

    if(othrSize < currSize)
	return true;
    return false;
}
/*---------------------------------------------------------*/
vector<oaModInst*> 
Util::getOtherFanin(oaModInst* currFanin, oaModInst* currCell){

    vector<oaModInst*> fanins;
    oaIter<oaModInstTerm> modInstIter (currCell->getInstTerms());
    while(oaModInstTerm *modInstTerm = modInstIter.getNext() ){
	oaTermType type_in(modInstTerm->getTerm()->getTermType());
	if(type_in == oacInputTermType ){
	    oaModNet* net = modInstTerm->getNet();
	    oaIter<oaModInstTerm> outInstTmIter (net->getInstTerms());
	    while(oaModInstTerm* modOutInstTerm = outInstTmIter.getNext() ){
		oaTermType type_out(modOutInstTerm->getTerm()->getTermType());
		oaModInst* m = modOutInstTerm->getInst();
		if(type_out == oacOutputTermType 
			&& m != currFanin){
		    fanins.push_back(m);
		}
	    }
	}
    }
    return fanins;

}
/*---------------------------------------------------------*/
void
Util::updateFanIOTiming(oaDesign* design,Timer* t,oaModInst *m){
    oaModScalarInst *MScaInst = static_cast<oaModScalarInst*>(m);
    assert(MScaInst);
    oaOccurrence *occ = design->getTopOccurrence();
    oaIter<oaOccInst> occInstIter (m->getOccInsts(occ));
    while(oaOccInst *occInst = occInstIter.getNext() ){
	oaIter<oaOccInstTerm> itIter (occInst->getInstTerms());
	while(oaOccInstTerm *occInstTerm = itIter.getNext()){		
	    oaTermType type(occInstTerm->getTerm()->getTermType());
	    if(type == oacInputTermType){
		oaOccNet *net = occInstTerm->getNet();
		t->invalidateFanout(net);
		t->updateArr(occInstTerm);
	    }
	    if(type == oacOutputTermType){
		oaOccNet *net = occInstTerm->getNet();
		t->invalidateFanin(net);
		t->updateReq(occInstTerm);
	    }
	}
    }
}
/*---------------------------------------------------------*/
void
Util::invalidateFanIOTiming(oaDesign* design,Timer* t,oaModInst *m){
    oaModScalarInst *MScaInst = static_cast<oaModScalarInst*>(m);
    assert(MScaInst);
    oaOccurrence *occ = design->getTopOccurrence();
    oaIter<oaOccInst> occInstIter (m->getOccInsts(occ));
    while(oaOccInst *occInst = occInstIter.getNext() ){
	oaIter<oaOccInstTerm> itIter (occInst->getInstTerms());
	while(oaOccInstTerm *occInstTerm = itIter.getNext()){		
	    oaTermType type(occInstTerm->getTerm()->getTermType());
	    if(type == oacInputTermType){
		oaOccNet *net = occInstTerm->getNet();
		t->invalidateFanout(net);
	    }
	    if(type == oacOutputTermType){
		oaOccNet *net = occInstTerm->getNet();
		t->invalidateFanin(net);
	    }
	}
    }
}
/*---------------------------------------------------------*/
int
Util::swapCell(oaString cellName,oaModInst* currModInst){
    oaNativeNS NS;
    const oaScalarName libertyName(NS, libParseData.libString);
    const oaScalarName libertyView(NS, libParseData.viewString);
    const oaScalarName libertyCell(NS, cellName);

    oaDesign *cell_des = oaDesign::open(
	    libertyName, libertyCell, libertyView, MODE_READ);
    oaModScalarInst* scalarInst = static_cast<oaModScalarInst*>
	    (currModInst);
    assert(scalarInst);
    if(scalarInst)
	scalarInst->setMaster(cell_des);
    else{
	cell_des->close();
	return 1;
    }
    cell_des->close();
    return 0;
}
/*---------------------------------------------------------*/
double
Util::getCellDelayEstimate(oaOccInstTerm *inpTerm, 
			    oaString otherSize,
			    double load,
			    Timer* timer){
#if defined(DEBUG)
    std::cout << " getCellDelayEstimate(" << getBlockName(inpTerm) << "," 
	<< otherSize << "," << load << ")" << std::endl;
#endif

    double delay = -DBL_MAX;
    oaModule* topCell = DesignTool::getCellTopMod(otherSize);
    assert(topCell);
    oaModTerm *outTerm;
    oaIter<oaModTerm> termIter = topCell->getTerms();
    while(oaModTerm *term = termIter.getNext()){
	oaTermType type(term->getTermType());
	if(type == oacOutputTermType){
	    outTerm = term;
	}
    }
    TPointMaster *otherTM = TPointMaster::get(outTerm);
    assert(otherTM);
    oaModule *master = inpTerm->getInst()->getModInst()->getMasterModule();
    assert(master);

    /*oaOccInstTerm *outInstTerm = DesignTool::getOccOutInstTerm(inpTerm->getInst());
    TPoint *p = TPoint::get(outInstTerm);
    assert(p);
    std::cout << " load1:" << p->load << " load2:" << load << std::endl;
    */
    TPointMaster::pathVector::iterator i;
    for (i = otherTM->inPaths.begin(); i != otherTM->inPaths.end(); ++i) {
        assert(i->other);
	oaName name;
	i->other->getName(name);
	oaModTerm *inTerm = oaModTerm::find(master,name);
	assert(inTerm);
        oaOccInstTerm *other = timer->getOccFromMod(inpTerm->getInst(), inTerm);
        if (!other) continue;
	if(other == inpTerm){
	    TPoint *otherTP = TPoint::get(other); 
	    for (unsigned int j = 0; j < otherTP->multiClockData.size(); ++j) { 
		timingData *td = otherTP->multiClockData[j];
		DelayType d = 0.0;
		DelayType sl1 = td->riseSlew;
		DelayType sl2 = td->fallSlew;
		if (i->transition & TPointMaster::TRANSITION_SRC_RISE) {
		    d = i->delay->lookup(load, sl1, otherTM->loadLimit);
		} 
		else if (i->transition & TPointMaster::TRANSITION_SRC_FALL)
		{
		    d = i->delay->lookup(load, sl2, otherTM->loadLimit);
		}
		if(d > delay) delay = d;
	    }
	}
    }
    return delay;
}
/*---------------------------------------------------------*/
double
Util::getDeltaDelay(oaOccObject *block, 
	TPointMaster *otherTM,
	Timer* timer){
    DelayType currAT = timer->getArr(block);
    DelayType newAT = 0.0;
    TPoint *p = TPoint::get(block);
    assert(p);
    oaOccInst *inst = (static_cast<oaOccInstTerm*>(block))->getInst();
    assert(inst);
    oaModule *master = inst->getModInst()->getMasterModule();
    assert(master);
    TPointMaster::pathVector::iterator i;
    for (i = otherTM->inPaths.begin(); i != otherTM->inPaths.end(); ++i) {
        assert(i->other);
	oaName name;
	i->other->getName(name);
	oaModTerm *inTerm = oaModTerm::find(master,name);
	assert(inTerm);
        oaOccInstTerm *other = timer->getOccFromMod(inst, inTerm);
        if (!other) continue;
        TPoint *otherTP = TPoint::get(other); 
	for (unsigned int j = 0; j < otherTP->multiClockData.size(); ++j) { 
	    timingData *td = otherTP->multiClockData[j];
	    DelayType d = 0.0;
	    DelayType sl1 = td->riseSlew;
	    DelayType sl2 = td->fallSlew;
	    if (i->transition & TPointMaster::TRANSITION_SRC_RISE) {
		d = i->delay->lookup(p->load, sl1, otherTM->loadLimit);
		d  += td->riseArr;
	    } 
	    else if (i->transition & TPointMaster::TRANSITION_SRC_FALL)
	    {
		d = i->delay->lookup(p->load, sl2, otherTM->loadLimit);
		d  += td->fallArr;
	    }
	    if(d > newAT) newAT = d;
	}
    }
    //std::cout << " " << currAT << ":" << newAT << std::endl;
    return (newAT-currAT);
}
/*---------------------------------------------------------*/
DelayType
Util::getCellRiseDelay(oaOccInstTerm *inPin){
    TPoint *tp = TPoint::get(inPin);
    assert(tp);
    return tp->riseDelay;
}
/*---------------------------------------------------------*/
DelayType
Util::getCellFallDelay(oaOccInstTerm *inPin){
    TPoint *tp = TPoint::get(inPin);
    assert(tp);
    return tp->fallDelay;
}
/*---------------------------------------------------------*/
DelayType
Util::getCellDelay(oaOccInstTerm *inPin){
    DelayType t1 = Util::getCellRiseDelay(inPin);
    DelayType t2 = Util::getCellFallDelay(inPin);
    DelayType s = (t1 > t2) ? t1 : t2;
    return s;
}
/*---------------------------------------------------------*/
/*DelayType
Util::getCellDelay(Timer *timer, oaOccInstTerm *inPin, oaOccInstTerm *outPin){
    double delay;
    TPoint *tp = TPoint::get(inPin);
    assert(tp);
    timingData *td = tp->multiClockData[0];
    //delay = (timer->getRiseArr(outPin) > timer->getFallArr(outPin)) ? p->delay1 : p->delay2;
    delay = ((td->riseArr+tp->delay2) > (td->fallArr+tp->delay1)) ? tp->delay2 : tp->delay1;
    std::cout << " SrcRise:" << td->riseArr << " delay1:" << tp->delay1 
	<< " SrcFall:" << td->fallArr << " delay2:" << tp->delay2 << endl;
    return delay;
}*/
/*---------------------------------------------------------*/
/*DelayType
Util::getCellDelay(Timer *timer, oaOccInstTerm *inPin, oaOccInstTerm *outPin){
    TPoint *p = TPoint::get(outPin);
    assert(p);
    oaModTerm *masterTerm = outPin->getModInstTerm()->getTerm();
    assert(masterTerm);
    TPointMaster *tm = TPointMaster::get(masterTerm);
    assert(tm);
    oaOccInst *inst = outPin->getInst();
    assert(inst);
    DelayType delay = 0.0;
    TPointMaster::pathVector::iterator i;
    for (i = tm->inPaths.begin(); i != tm->inPaths.end(); ++i) {
        assert(i->other);
        oaOccInstTerm *other = timer->getOccFromMod(inst, i->other);
        if (!other) continue;
	if (other != inPin) continue;
        TPoint *otherTP = TPoint::get(other); 
	for (unsigned int j = 0; j < otherTP->multiClockData.size(); ++j) { 
	    timingData *td = otherTP->multiClockData[j];
	    DelayType d = 0.0;
	    DelayType sl1 = td->riseSlew;
	    DelayType sl2 = td->fallSlew;
	    if (i->transition & TPointMaster::TRANSITION_SRC_RISE) {
		d = i->delay->lookup(p->load, sl1, tm->loadLimit);
		//std::cout << " **SrcRise sl1:" << sl1 << " load:" << p->load
		 //   << " loadLimit" << tm->loadLimit << std::endl;
	    } 
	    else if (i->transition & TPointMaster::TRANSITION_SRC_FALL)
	    {
		d = i->delay->lookup(p->load, sl2, tm->loadLimit);
		//std::cout << " **SrcFall sl2:" << sl2 << " load:" << p->load
		  //  << " loadLimit" << tm->loadLimit << std::endl;
	    }
	    if(d > delay) delay = d;
	}
    }
    //std::cout << "Delay from:" << timer->getBlockName(inPin) 
	//<< " to" << timer->getBlockName(outPin) << ":" << delay << std::endl;
    return delay; 
}*/
/*---------------------------------------------------------*/
DelayType
Util::getWorstInputSlack(Timer* t, oaOccInst* inst){
    //cout << "Get Worst Input Slack" << endl;
    DelayType ws = DBL_MAX;
    oaIter<oaOccInstTerm> itIter (inst->getInstTerms());
    while(oaOccInstTerm *i = itIter.getNext()){		
	oaTermType type(i->getTerm()->getTermType());
	if(type == oacInputTermType){
	    DelayType slack = t->getSlack(i);
	    if(slack < ws) ws = slack;
	    //std::cout << t->getBlockName(i) << " WS:" << ws << endl;
	}else{
	    //std::cout << t->getBlockName(i) << " Arr:" << t->getArr(i) << endl;
	}
    }
    return ws;
}
/*---------------------------------------------------------*/
double
Util::getUpSizeDeltaDelay(oaDesign* design, Timer *timing, CellData *c){
    vector<oaString> size_vec = c->getGreaterThanCurrentSize();
    if(size_vec.empty()) return DBL_MAX;

    oaOccInst *occInst = DesignTool::getOccInst(design,c->inst);
    oaOccInstTerm *outInstTerm = DesignTool::getOccOutInstTerm(occInst);
    //vector<oaString>::iterator it=size_vec.begin();
    vector<oaString>::iterator it= --size_vec.end();
    oaModule* topCell = DesignTool::getCellTopMod(*it);
    assert(topCell);
    oaModTerm *outTerm;
    oaIter<oaModTerm> termIter = topCell->getTerms();
    while(oaModTerm *term = termIter.getNext()){
	oaTermType type(term->getTermType());
	if(type == oacOutputTermType){
	    outTerm = term;
	}
    }
    TPointMaster *new_tm = TPointMaster::get(outTerm);
    assert(new_tm);
    DelayType dDelay = Util::getDeltaDelay(outInstTerm,new_tm,timing);
    /*std::cout << " Upsizing Inst:" << c->instName << " from:" << c->master->name << " to:" << *it << std::endl;
    std::cout << "  Delta Delay:" << dDelay  << std::endl;*/
    return dDelay;
}
/*---------------------------------------------------------*/
double
Util::getDownSizeDeltaDelay(oaDesign* design, Timer* timing, CellData *c){
    vector<oaString> size_vec = c->getLessThanCurrentSize();
    if(size_vec.empty()) return DBL_MAX;

    oaOccInst *occInst = DesignTool::getOccInst(design,c->inst);
    oaOccInstTerm *outInstTerm = DesignTool::getOccOutInstTerm(occInst);
    vector<oaString>::iterator it=size_vec.begin();
    oaModule* topCell = DesignTool::getCellTopMod(*it);
    assert(topCell);
    oaModTerm *outTerm;
    oaIter<oaModTerm> termIter = topCell->getTerms();
    while(oaModTerm *term = termIter.getNext()){
	oaTermType type(term->getTermType());
	if(type == oacOutputTermType){
	    outTerm = term;
	}
    }
    TPointMaster *new_tm = TPointMaster::get(outTerm);
    assert(new_tm);
    DelayType dDelay = Util::getDeltaDelay(outInstTerm,new_tm,timing);
    /*std::cout << " Downsizing Inst:" << c->instName << " from:" << c->master->name << " to:" << *it << std::endl;
    std::cout << "  Delta Delay:" << dDelay  << std::endl;*/
    return dDelay;
}
/*---------------------------------------------------------*/
DelayType
Util::getDeltaDelay(oaDesign* design, Timer* timing, CellData *c, oaString newSize){
    oaOccInst *occInst = DesignTool::getOccInst(design,c->inst);
    oaOccInstTerm *outInstTerm = DesignTool::getOccOutInstTerm(occInst);
    oaModule* topCell = DesignTool::getCellTopMod(newSize);
    assert(topCell);
    oaModTerm *outTerm;
    oaIter<oaModTerm> termIter = topCell->getTerms();
    while(oaModTerm *term = termIter.getNext()){
	oaTermType type(term->getTermType());
	if(type == oacOutputTermType){
	    outTerm = term;
	}
    }
    TPointMaster *new_tm = TPointMaster::get(outTerm);
    assert(new_tm);
    DelayType dDelay = Util::getDeltaDelay(outInstTerm,new_tm,timing);
    /*std::cout << " Downsizing Inst:" << c->instName << " from:" << c->master->name << " to:" << *it << std::endl;
    std::cout << "  Delta Delay:" << dDelay  << std::endl;*/
    return dDelay;
}
/*---------------------------------------------------------*/
bool 
Util::hasValidArrTime(Timer *timer, CellData *cell){
    int checkLevel = 2;
    oaDesign *top = *(timer->_watchedDesigns.begin()); 	

    std::cout << cell->instName << " Fanouts:" << std::endl;
    //Verify 2 level forward fanout(s);
    instVector fanouts; 
    if(!fanouts.empty()) fanouts.clear();
    DesignTool::getFanouts(cell->inst,fanouts,checkLevel);
    if(fanouts.empty()){
	DesignTool::getFanouts(cell->inst,fanouts,1);
    }
    if(fanouts.empty()){
	oaOccInst *occInst = DesignTool::getOccInst(top,cell->inst);
	oaOccInstTerm *outTerm = DesignTool::getOccOutInstTerm(occInst);
	if(timer->getSlack(outTerm) < 0) return false;
    }
    for(instVector::iterator it=fanouts.begin();
	   it!=fanouts.end(); ++it){
	std::cout << " " << Util::getInstName(*it) << std::endl;
	oaOccInst *occInst = DesignTool::getOccInst(top,*it);
	oaOccInstTerm *outTerm = DesignTool::getOccOutInstTerm(occInst);
	if(timer->getSlack(outTerm) < 0) return false;
    }
    
    //Verify 2 level back fanin(s);
    std::cout << cell->instName << " Fanins:" << std::endl;
    instVector fanins; 
    int fInLvl = checkLevel;
    do{
	if(!fanins.empty()) fanins.clear();
	DesignTool::getFanins(cell->inst,fanins,fInLvl);
	for(instVector::iterator it=fanins.begin();
	       it!=fanins.end(); ++it){
	    std::cout << " " << Util::getInstName(*it) << std::endl;
	    oaOccInst *occInst = DesignTool::getOccInst(top,*it);
	    vector<oaOccInstTerm*> inTerms = DesignTool::getOccInInstTerm(occInst);
	    for(vector<oaOccInstTerm*>::iterator it=inTerms.begin();
		    it!=inTerms.end(); ++it){
		if(timer->getSlack(*it) < 0) return false;
	    }
	}
    }while(--fInLvl);

    oaOccInst *occInst = DesignTool::getOccInst(top,cell->inst);
    vector<oaOccInstTerm*> inTerms = DesignTool::getOccInInstTerm(occInst);
    for(vector<oaOccInstTerm*>::iterator it=inTerms.begin();
	    it!=inTerms.end(); ++it){
	if(timer->getSlack(*it) < 0) return false;
    }
    return true;
}
/*---------------------------------------------------------*/
double
Util::getMaxNetDelay(oaOccInstTerm* t){
    TPoint *tp = TPoint::get(t);
    assert(tp);
    if(tp->type != TIMING_POINT_SIGNAL_OUT) return 0.0;	

    double maxDelay = -DBL_MAX;
    oaIter<oaOccTerm> tIter = t->getNet()->getTerms();
    while(oaOccTerm *term = tIter.getNext()){
	TPoint *p = TPoint::get(term);
	assert(p);
	if(p->type == TIMING_POINT_PO){ 
	    if(p->netDelay > maxDelay) maxDelay = p->netDelay;
	}
    }
    oaIter<oaOccInstTerm> itIter = t->getNet()->getInstTerms();
    while(oaOccInstTerm *iTerm = itIter.getNext()){
	TPoint *p = TPoint::get(iTerm);
	assert(p);
	if(p->type == TIMING_POINT_SIGNAL_IN){ 
	    if(p->netDelay > maxDelay) maxDelay = p->netDelay;
	}
    }
    return maxDelay;
}
/*---------------------------------------------------------*/
oaString
Util::getCellName(oaModInst* i){
    oaNativeNS NS;
    oaString name;
    oaModule *master = i->getMasterModule();
    assert(master);
    master->getName(NS,name);
    return name;
}
/*---------------------------------------------------------*/
oaString 
Util::getCellName(oaOccInst *i){
    oaModInst *modInst = i->getModInst();
    assert(modInst);
    return getCellName(modInst);
}
/*---------------------------------------------------------*/
void
Util::linear_fit(double &a,double &b, double *x, double *y, int N){
    int iter = 0;
    double fval, det;
    double z[2];
    double dz[2];
    double d2z[4];
    double d2zinv[4];

    if(N==1){
	a = y[0]/x[0];
	b = 0.0; 
	return;
    }
    dz[0] = 1;
    dz[1] = 0;
    z[0] = 0;
    z[1] = 0;

    while(sqrt(dz[0]*dz[0] + dz[1]*dz[1]) > 1e-10){
	fval = 0;
	dz[0] = dz[1] = 0;
	d2z[0] = d2z[1] = d2z[2] = d2z[3] = 0;

	for(int i=0; i<N; ++i){
	    fval += (z[0]*x[i]+z[1]-y[i])*(z[0]*x[i]+z[1]-y[i]);
	    dz[0] += 2*z[0]*x[i]*x[i]+2*z[1]*x[i]-2*x[i]*y[i];
	    dz[1] += 2*z[1]+2*z[0]*x[i]-2*y[i];
	    d2z[0] += 2*x[i]*x[i];
	    d2z[1] += 2*x[i];
	    d2z[2] += 2*x[i];
	    d2z[3] += 2;
	}

	det = d2z[0]*d2z[3]-d2z[1]*d2z[2];
	if(det == 0){
	    std::cout << "affine fit: 0 determinant, breaking" << std::endl;
	    assert(0);
	    break;
	}
	d2zinv[0] = d2z[3]/det;
	d2zinv[1] = -d2z[1]/det;
	d2zinv[2] = -d2z[2]/det;
	d2zinv[3] = d2z[0]/det;
	z[0] = z[0]-(d2zinv[0]*dz[0]+d2zinv[1]*dz[1]);
	z[1] = z[1]-(d2zinv[2]*dz[0]+d2zinv[3]*dz[1]);
	if(iter > 10){
	    std::cout << "f:" << fval 
		<< "\t|df|:" << sqrt(dz[0]*dz[0]+dz[1]*dz[1]) << std::endl;
	    std::cout << "Affine fit: max iterations!" << std::endl;
	    break;
	}else iter += 1;

	a = z[0];
	b = z[1];
    }
}
/*---------------------------------------------------------*/
void
Util::downSizeAll(oaDesign *design, Timer *timer){
std::cout << "DownSizeAll()" << std::endl;
assert(design);
oaModule *mod = design->getTopModule();
assert(mod);
oaIter<oaModInst> modIter ( mod->getInsts());
while(oaModInst *modInst = modIter.getNext()){
    CellData *cell = CellData::get(modInst);
    assert(cell);
    cout << " Inst:" << cell->instName << ":" << cell->master->name 
	<< " cnt:"  << cell->getDownSizeCount() << endl;
    if(cell->getDownSizeCount()){
	vector<oaString> downSizes = cell->getLessThanCurrentSize();
	if(downSizes.empty()) cout << "Error: no downsize" << std::endl;
	int minPos = downSizes.size()-1;
	cell->swap(downSizes[minPos]);
	cell->commitSwap();
    }
    //timer->clearDesignTimingData();
    //timer->updateAll();
}
}
/*---------------------------------------------------------*/
void 
Util::reportSlacks(oaDesign* design, Timer *timer){
    map<double,int> slackTable;
    map<double,int>::iterator iter;
    int max = 10;
    vector<int> slackVector(max,0);
    double maxSlack = 0.0;
    double minSlack = DBL_MAX;
    Util ut;

    //Query total no. of gates
    oaModule *mod = design->getTopModule();
    assert(mod);
    double totGates = 0;
    oaIter<oaModInst> modIter ( mod->getInsts());
    while(oaModInst *modInst = modIter.getNext()){
      ++totGates;
    }
    oaOccurrence *occ = design->getTopOccurrence();
    assert(occ);
    oaIter<oaOccInstTerm> itIter ( occ->getInstTerms());
    while(oaOccInstTerm *iTerm = itIter.getNext()){
	oaTermType type(iTerm->getTerm()->getTermType());
	if(type == oacInputTermType){
	    double slack = timer->getSlack(iTerm);
	    if (slack > maxSlack) maxSlack = slack;
	    if (slack < minSlack) minSlack = slack;
	    iter = slackTable.find(slack);
	    if(iter!=slackTable.end()){
		int n = iter->second;
		slackTable[slack] = ++n;
	    }else slackTable[slack] = 1;
	}
    }
    double denom = maxSlack - minSlack;
    cout << "Slack Distribution Report" << endl;
    cout << " min/max:" << minSlack << "/" << maxSlack << endl;
    for(iter=slackTable.begin(); iter!=slackTable.end(); ++iter){
	double num = iter->first - minSlack;
	double pos = floor((num/denom)*10);
	int indx = int(pos);
	if (indx > 9) indx = 9;
	int count = slackVector[pos]; 
	slackVector[indx] = count + iter->second; 
	/*cout << " * " << iter->first << " pos:" << pos
	    << " index:" << indx << " bin:" << ((pos+1)/10)*maxSlack << " count:" << iter->second << endl;*/
    }
    for(int i = 0; i < max; ++i){
	cout << (double(i+1)/10)*maxSlack << ":" << double(slackVector[i])/totGates << endl;
    }
}
/*---------------------------------------------------------*/
void 
Util::reportGates(oaDesign* design){
  map<oaString,int> stat_table;
  map<oaString,int>::iterator iter;
  //vector<int> foCounter (5,0);
  map<int,int> foCounter;
  map<int,int>::iterator it;
  int maxFanout = 0;
  int maxLibSize = 0;
  foCounter.clear();
  Util ut;
  oaModule *mod = design->getTopModule();
  assert(mod);
  int count = 0;
  oaIter<oaModInst> modIter ( mod->getInsts());
  while(oaModInst *modInst = modIter.getNext()){
      CellData *cell = CellData::get(modInst);
      assert(cell);
      //std::cout << cell->instName << "(" << cell->master->name << "):" << std::endl;
      vector<oaString> sVec = cell->getAllSizes();
      if(int(sVec.size()) > maxLibSize) maxLibSize = sVec.size();

      oaString masterName = ut.getCellName(modInst); 
      iter = stat_table.find(masterName);
      if(iter!=stat_table.end()){
	  int n = iter->second;
	  stat_table[masterName] = ++n;
      }else stat_table[masterName] = 1;
      ++count;

      vector<oaModInst*> fnout;
      DesignTool::getFanouts(modInst,fnout,1);
      if(fnout.size() > maxFanout) maxFanout = fnout.size();

      if(fnout.size() == 0) continue;
      it = foCounter.find((int)fnout.size());
      if(it != foCounter.end()){ 
	  int n = iter->second;
	  foCounter[(int)fnout.size()] = ++n;
      }else{
	  foCounter[(int)fnout.size()] = 1; 
      }
  }

  double avgFanout = 0.0; 
  int totalCnt = 0;
  for(it=foCounter.begin(); it!=foCounter.end(); ++it){
      //std::cout<<"First "<<it->first<<" Second "<<it->second<<endl;
      avgFanout += (double(it->first)*double(it->second));
      totalCnt += it->second;
  }
  avgFanout = avgFanout/double(totalCnt);

 
  std::cout << "Gate_Count_Statistics" << std::endl;
  cout << " Max # LibSize:" << maxLibSize << std::endl;
  cout << " Max # Fanouts:" << maxFanout << std::endl;
  cout << " Average Fanouts:" << avgFanout << std::endl;
  std::cout << " Total Gates: " << count << std::endl;
  for(iter=stat_table.begin(); iter!=stat_table.end(); ++iter){
      std::cout << iter->first << std::setw(10)
		<< iter->second << std::endl;
  }
  std::cout << std::endl;

  /*cout << " Fanout Size Statistics " << endl;
  for(int n=0; n<foCounter.size(); ++n){
      cout << " Gates with " << n+1 << " fanouts:" << foCounter[n] << endl;
  }*/
}
/*---------------------------------------------------------*/
void
Util::changeToMinCellSize(oaDesign *design, Timer *timer){
    oaModule *top = design->getTopModule();
    assert(top);
    oaIter<oaModInst> modIter = top->getInsts();
    while(oaModInst *inst = modIter.getNext()){
	CellData *cell = CellData::get(inst);
	assert(cell);
	if(cell->getDownSizeCount()){
	    oaString minSize = cell->getMinSize();
	    cell->swap(minSize);
	    cell->commitSwap();
	}
    }
    timer->clearDesignTimingData();
    timer->updateAll();
}
/*---------------------------------------------------------*/
void
Util::reportTimingInfo(oaDesign *des, Timer *t){
    oaOccurrence *occ = des->getTopOccurrence();
    assert(occ);
    oaIter<oaOccInstTerm> oitIter ( occ->getInstTerms());
    while(oaOccInstTerm* oit = oitIter.getNext()){
	TPoint *tp = TPoint::get(oit);
	if(tp->type == TIMING_POINT_SIGNAL_IN)
	    std::cout << "Input Pin:";
	else
	    std::cout << "Output Pin:";
	std::cout << Util::getBlockName(oit) << std::endl;	
	std::cout << "  RiseArr:" << tp->getRiseArr()
	    << " FallArr" << tp->getFallArr() << std::endl;
	std::cout << "  RiseDelay" << tp->riseDelay
	    << " FallDelay" << tp->fallDelay << std::endl;
	std::cout << "  NetDelay:" << tp->netDelay << std::endl;
	std::cout << "  RiseReq:" << tp->getRiseReq()
	    << " FallReq" << tp->getFallReq() << std::endl;
	std::cout << "  Slack:" << t->getSlack(oit) << std::endl; 
	std::cout << std::endl;
    }
}
/*---------------------------------------------------------*/
void
Util::reportCellTiming(CellData *cell, Timer *t){
    oaIter<oaOccInstTerm> oitIter ( cell->occInst->getInstTerms());
    while(oaOccInstTerm* oit = oitIter.getNext()){
	TPoint *tp = TPoint::get(oit);
	if(tp->type == TIMING_POINT_SIGNAL_IN)
	    std::cout << "Input Pin:";
	else
	    std::cout << "Output Pin:";
	std::cout << Util::getBlockName(oit) << std::endl;	
	std::cout << "  RiseArr:" << tp->getRiseArr()
	    << " FallArr" << tp->getFallArr() << std::endl;
	std::cout << "  RiseDelay" << tp->riseDelay
	    << " FallDelay" << tp->fallDelay << std::endl;
	std::cout << "  NetDelay:" << tp->netDelay << std::endl;
	std::cout << "  RiseReq:" << tp->getRiseReq()
	    << " FallReq" << tp->getFallReq() << std::endl;
	std::cout << "  Slack:" << t->getSlack(oit) << std::endl; 
	std::cout << std::endl;
    }
}
/*---------------------------------------------------------*/
void
Util::reportAllCellTiming(oaDesign *design, Timer *t){
    std::cout << "Reporting all cell timing info" << std::endl;
    oaModule *top = design->getTopModule();
    oaIter<oaModInst> miIter(top->getInsts());
    while(oaModInst *inst = miIter.getNext()){
	CellData *cell = CellData::get(inst);
	assert(cell);
	oaIter<oaOccInstTerm> oitIter (cell->occInst->getInstTerms());
	while(oaOccInstTerm* oit = oitIter.getNext()){
	    TPoint *tp = TPoint::get(oit);
	    if(tp->type == TIMING_POINT_SIGNAL_IN)
		std::cout << "Input Pin:";
	    else
		std::cout << "Output Pin:";
	    std::cout << Util::getBlockName(oit) << std::endl;	
	    std::cout << "  RiseArr:" << tp->getRiseArr()
		<< " FallArr" << tp->getFallArr() << std::endl;
	    std::cout << "  RiseDelay" << tp->riseDelay
		<< " FallDelay" << tp->fallDelay << std::endl;
	    std::cout << "  NetDelay:" << tp->netDelay << std::endl;
	    std::cout << "  RiseReq:" << tp->getRiseReq()
		<< " FallReq" << tp->getFallReq() << std::endl;
	    std::cout << "  Slack:" << t->getSlack(oit) << std::endl; 
	    std::cout << std::endl;
	}
    }
}
/*---------------------------------------------------------*/
void
Util::reportTimingFlag(oaDesign *des){
    oaOccurrence *occ = des->getTopOccurrence();
    assert(occ);
    oaIter<oaOccTerm> otIter ( occ->getTerms());
    while(oaOccTerm* ot = otIter.getNext()){
	TPoint *tp = TPoint::get(ot);
	std::cout << Util::getBlockName(ot) 
	    << ": " << tp->atValid << " " 
	    << tp->ratValid << std::endl;	
	 
    }
    oaIter<oaOccInstTerm> oitIter ( occ->getInstTerms());
    while(oaOccInstTerm* oit = oitIter.getNext()){
	TPoint *tp = TPoint::get(oit);
	if(tp->type == TIMING_POINT_CLOCK){
	TPoint *tp1 = TPoint::getClock(oit);
	std::cout << Util::getBlockName(oit) 
	    << ": " << tp1->atValid << " " 
	    << tp1->ratValid << std::endl;	
	    continue;
	}
	std::cout << Util::getBlockName(oit) 
	    << ": " << tp->atValid << " " 
	    << tp->ratValid << std::endl;	
    }
}
/*---------------------------------------------------------*/
void
Util::printAllNetData(oaDesign *design){
    oaOccurrence *occ = design->getTopOccurrence();
    oaIter<oaOccTerm> tIter = occ->getTerms();
    while(oaOccTerm *term = tIter.getNext()){
	TPoint *tp = TPoint::get(term);
	assert(tp);
	if(tp->type == TIMING_POINT_PI){
	    std::cout << Util::getBlockName(term)
		<< " - load:" << tp->netLoad << std::endl;
	}
	else if(tp->type == TIMING_POINT_PO){
	    std::cout << Util::getBlockName(term)
		<< " - netDelay:" << tp->netDelay << std::endl;
	}
	
    }
    oaIter<oaOccInstTerm> itIter2 = occ->getInstTerms();
    while(oaOccInstTerm *iTerm = itIter2.getNext()){
	TPoint *tp = TPoint::get(iTerm);
	assert(tp);
	if(tp->type == TIMING_POINT_SIGNAL_IN){
	    std::cout << Util::getBlockName(iTerm)
		<< " - netDelay:" << tp->netDelay << std::endl;
	}
	else if(tp->type == TIMING_POINT_SIGNAL_OUT){
	    std::cout << Util::getBlockName(iTerm)
		<< " - load:" << tp->netLoad << std::endl;
	}
    }
}
/*---------------------------------------------------------*/
std::string
Util::getBlockName(oaOccObject *oPtr) 
{
    oaString f;
    oaNativeNS ns;
    std::string s;
    if (oPtr->isOccTerm()) {
        static_cast<oaOccTerm*>(oPtr)->getName(ns, f);
        s = std::string(f);
    } else {
        oaOccInst *inst = static_cast<oaOccInstTerm*>(oPtr)->getInst();
        inst->getName(ns, f);
        s = std::string(f);
        static_cast<oaOccInstTerm*>(oPtr)->getTermName(ns, f);
        s = s + "/" + std::string(f);
    }
    return s;
}
/*---------------------------------------------------------*/
/*oaModule*
::findMaster(const char *s){
    oaNativeNS NS;
    oaString cellName; 
    Util *u = new Util();
if(libertyLib->getAccess(oacReadLibAccess,0)){
    oaIter<oaCell> cellIter (libertyLib->getCells());
    while(oaCell *cell = cellIter.getNext()){
	cell->getName(NS,cellName);
	if(
    oaModule *master;
	master = modInst->getMasterModule();
	master->getName(NS, cName);
	if(strcmp(s,cName) == 0)
		break;
    }
}
    cout << "cName: " << cName << endl;
    cout << "s: " << s << endl;
    return master;
   
}*/
/*---------------------------------------------------------*/
}//namespace
