/************************************************************
* 
* File: oagTimerSubTimer.cpp
* Author: Santiago Mok (smok@ee.ucla.edu)
* Created: 03-12-2011
* Last Modified: Wed 06 Apr 2011 07:37:39 PM PDT
*
************************************************************/

#include "oaDesignDB.h"
#include "oagTimerTPoint.h"
#include "oagTimerTPointMaster.h"
#include "oagTimerCellData.h"
#include "oagTimerDesignTool.h"
#include "oagTimerSubTimer.h"
#include "oagTimerElmoreWireModel.h"

#include <assert.h>

//#define DEBUG
namespace oagTimer {
    using namespace std;
    using namespace oa;
/*---------------------------------------------------------*/
// SubTPoint
/*---------------------------------------------------------*/
SubTPoint::SubTPoint(){
    riseArr = fallArr = -DBL_MAX;
    riseSlew = fallSlew = -DBL_MAX;
    riseReq = fallReq = DBL_MAX; 
    delay = -DBL_MAX;
}
SubTPoint::~SubTPoint(){}
/*---------------------------------------------------------*/
double
SubTPoint::getSlew(){
    return (riseSlew > fallSlew) ? riseSlew : fallSlew; 
}
/*---------------------------------------------------------*/
double
SubTPoint::getArr(){
    return (riseArr > fallArr) ? riseArr : fallArr; 
}
/*---------------------------------------------------------*/
double
SubTPoint::getReq(){
    return (riseReq < fallReq) ? riseReq : fallReq; 
}
/*---------------------------------------------------------*/
double
SubTPoint::getRiseSlack(){
    double riseSlack = riseReq - riseArr;
    return riseSlack;
}
/*---------------------------------------------------------*/
double
SubTPoint::getFallSlack(){
    double fallSlack = fallReq - fallArr;
    return fallSlack;
}
/*---------------------------------------------------------*/
double
SubTPoint::getSlack(){
    double riseSlack = riseReq - riseArr;
    double fallSlack = fallReq - fallArr;
    return (riseSlack < fallSlack) ? riseSlack : fallSlack; 
}
/*---------------------------------------------------------*/
// SubTimer
/*---------------------------------------------------------*/
SubTimer::SubTimer(oaDesign *des, Timer *t, oaOccInst *i, oaString c){
    design = des;
    timer = t;
    occInst = i; 
    masterCell = c;
    loadGiven = false;
}
SubTimer::~SubTimer(){
    tpMap.clear();
    faninVec.clear();
    fanoutVec.clear();
}
/*---------------------------------------------------------*/
SubTPoint
SubTimer::getSubTP(oaOccInstTerm *i){
    map<oaOccInstTerm*,SubTPoint>::iterator tpMapIter;
    tpMapIter = tpMap.find(i);
    if(tpMapIter == tpMap.end()){
	std::cout << "Error in subtimer timing graph" << std::endl;
	exit(1);
    }
    return tpMapIter->second;
}
/*---------------------------------------------------------*/
void
SubTimer::setLoad(double l){
    currLoad = l;
    loadGiven = true;
}
/*---------------------------------------------------------*/
double
SubTimer::getCellDelayEstimate(oaOccInstTerm *i){
    if(tpMap.empty()) buildTimingGraph();
    SubTPoint sTP = getSubTP(i);
    return sTP.delay;
}
/*---------------------------------------------------------*/
double
SubTimer::getWorstSlack(){

    if(tpMap.empty()) buildTimingGraph();
    double wSlack = DBL_MAX;
    oaIter<oaOccInstTerm> itIter = occInst->getInstTerms();
    while(oaOccInstTerm *iTerm = itIter.getNext()){
	TPoint *tp = TPoint::get(iTerm);
	assert(tp);
	if(tp->type == TIMING_POINT_SIGNAL_IN){
	    SubTPoint sTP = getSubTP(iTerm);
	    if(sTP.getSlack() < wSlack) wSlack = sTP.getSlack();
	}
    }
    return wSlack;
}
/*---------------------------------------------------------*/
double
SubTimer::getDeltaDelay(){
    if(tpMap.empty()) buildTimingGraph();
    //printAll();

    double dDelay = -DBL_MAX;

    /*for(int i=0; i<fanoutVec.size(); i++){
	oaOccInst *inst = fanoutVec[i];
	oaOccInstTerm *outInstTerm = DesignTool::getOccOutInstTerm(inst);
	double delta = getDeltaArr(outInstTerm);	
	if(delta > dDelay) dDelay = delta;
    }*/
    
    oaOccInstTerm *outITerm = DesignTool::getOccOutInstTerm(occInst);
    TPoint *outPin = TPoint::get(outITerm);
    //if(outPin->isToPO){
	double delta = getDeltaArr(outITerm);	
	if(delta > dDelay) dDelay = delta;
    //}
    return dDelay;
}
/*---------------------------------------------------------*/
double 
SubTimer::getDeltaArr(oaOccInstTerm *it){
    if(tpMap.empty()) buildTimingGraph();
    //printAll();
    map<oaOccInstTerm*,SubTPoint>::iterator tpMapIter;
    tpMapIter = tpMap.find(it);
    if(tpMapIter == tpMap.end()){
	std::cout << "Error in subtimer timing graph" << std::endl;
	exit(1);
    }
    TPoint *tp = TPoint::get(it);
    assert(tp);
    double newRiseArr = tpMapIter->second.riseArr;
    double currRiseArr = tp->getRiseArr();
    double dRise = newRiseArr - currRiseArr;

    double newFallArr = tpMapIter->second.fallArr;
    double currFallArr = tp->getFallArr();
    double dFall = newFallArr - currFallArr;
    double d = (dRise > dFall) ? dRise : dFall;
    return d;
}
/*---------------------------------------------------------*/
double 
SubTimer::getDeltaSlack(){
    if(tpMap.empty()) buildTimingGraph();
    //printAll();
    double dSlack = DBL_MAX;

    /*for(int i=0; i<faninVec.size(); i++){
	oaOccInst *inst = faninVec[i];
	double delta = getMinDeltaSlack(inst);
	if(delta < dSlack) dSlack = delta;
    }*/

    oaIter<oaOccInstTerm> itIter = occInst->getInstTerms();
    while(oaOccInstTerm *iTerm = itIter.getNext()){
	//TPoint *inPin = TPoint::get(iTerm);
	//assert(inPin);
	//if(inPin->isFromPI){
	    double delta = getDeltaSlack(iTerm);
	    if(delta < dSlack) dSlack = delta;
	//}
    }
    return dSlack;
}
/*---------------------------------------------------------*/
double 
SubTimer::getMinDeltaSlack(oaOccInst *i){
    double minSlack = DBL_MAX;
    oaIter<oaOccInstTerm> itIter = occInst->getInstTerms();
    while(oaOccInstTerm *iTerm = itIter.getNext()){
	TPoint *inPin = TPoint::get(iTerm);
	assert(inPin);
	if(inPin->type == TIMING_POINT_SIGNAL_IN){
	    double delta = getDeltaSlack(iTerm);
	    if(delta < minSlack) minSlack = delta;
	}
    }
    return minSlack;
}
/*---------------------------------------------------------*/
double 
SubTimer::getDeltaSlack(oaOccInstTerm *it){

    map<oaOccInstTerm*,SubTPoint>::iterator tpMapIter;
    tpMapIter = tpMap.find(it);
    if(tpMapIter == tpMap.end()){
	std::cout << "Error in subtimer timing graph" << std::endl;
	exit(1);
    }
    TPoint *tp = TPoint::get(it);
    assert(tp);
    double newRiseSlack = tpMapIter->second.getRiseSlack();
    double currRiseSlack = tp->getRiseSlack();
    double dRise = currRiseSlack - newRiseSlack;

    double newFallSlack = tpMapIter->second.getFallSlack();
    double currFallSlack = tp->getFallSlack();
    double dFall = currFallSlack - newFallSlack;
    /*std::cout << Util::getBlockName(it) << 
	" curr(r/f):" << currRiseSlack << "/" << currFallSlack
	 << " new(r/f):" << newRiseSlack << "/" << newFallSlack << std::endl; */
    double d = (dRise < dFall) ? dRise : dFall;
    return d;
}
/*---------------------------------------------------------*/
void
SubTimer::buildTimingGraph(){
#if defined(DEBUG)
    std::cout << "buildTimingGraph()" << std::endl;
#endif
    faninVec.clear(); fanoutVec.clear();
    oaIter<oaOccInstTerm> itIter = occInst->getInstTerms();
    while(oaOccInstTerm *currInstTerm = itIter.getNext()){
	getOtherInsts(currInstTerm);
    }
    updateArr();
    updateReq();
}
/*---------------------------------------------------------*/
void
SubTimer::getOtherInsts(oaOccInstTerm *instTerm){
#if defined(DEBUG)
    std::cout << "getOtherInsts(" << Util::getBlockName(instTerm) << ")" << std::endl;
#endif
    TPoint *currTP =  TPoint::get(instTerm);
    assert(currTP);
    oaOccInst *inst;

    oaIter<oaOccInstTerm> itIter = instTerm->getNet()->getInstTerms();
    while(oaOccInstTerm *iTerm = itIter.getNext()){
	inst = iTerm->getInst();
	if(inst == occInst) continue;
	CellData *cell = CellData::get(inst->getModInst());
	assert(cell);
	if(cell->master->isSequential) continue;

	TPoint *otherTP = TPoint::get(iTerm);
	assert(otherTP);
	if(currTP->type == TIMING_POINT_SIGNAL_IN){
	    if(otherTP->type == TIMING_POINT_SIGNAL_OUT)
		faninVec.push_back(inst); 
	}else if(currTP->type == TIMING_POINT_SIGNAL_OUT){
	    if(otherTP->type == TIMING_POINT_SIGNAL_IN)
		fanoutVec.push_back(inst); 
	}
    }
}
/*---------------------------------------------------------*/
double
SubTimer::getCapLoad(oaOccInstTerm *i, oaString otherSize){
#if defined(DEBUG)
    std::cout << "getCapLoad (" << Util::getBlockName(i)  << "," << otherSize << ")" << std::endl;
#endif
    double load = 0.0;
    oaIter<oaOccInstTerm> itIter = i->getNet()->getInstTerms();
    while(oaOccInstTerm *iTerm = itIter.getNext()){
	TPoint *tp = TPoint::get(iTerm);
	assert(tp);
	if(tp->type == TIMING_POINT_SIGNAL_IN){
	    CellData *cell = CellData::get(iTerm->getInst()->getModInst());
	    assert(cell);
	    if(iTerm->getInst() == occInst){
		load += Util::getCellInputCap(iTerm,otherSize,timer);
	    }else{
		load += Util::getCellInputCap(iTerm,cell->master->name,timer);
	    }
	}
    }

    TPoint *outPin = TPoint::get(i);
    assert(outPin);
    load += outPin->netLoad;
    return load;
}
/*---------------------------------------------------------*/
void 
SubTimer::updateArr(){
#if defined(DEBUG)
    std::cout << "updateArr()" << std::endl;
#endif
    //Update timing for all fanin
    for(int i=0; i<faninVec.size(); i++){
	oaOccInstTerm *outInstTerm = DesignTool::getOccOutInstTerm(faninVec[i]);
	double load = getCapLoad(outInstTerm,masterCell);
	CellData *cell = CellData::get(faninVec[i]->getModInst());
	assert(cell);
	oaIter<oaOccInstTerm> iTermIter = faninVec[i]->getInstTerms();	
	while(oaOccInstTerm *iTerm = iTermIter.getNext()){
	    TPoint *tp = TPoint::get(iTerm);
	    assert(tp);
	    if(tp->type == TIMING_POINT_SIGNAL_IN){
#if defined(DEBUG)
		std::cout << " update(" << Util::getBlockName(iTerm) << ")" << std::endl;
#endif
		computeArr(iTerm,cell->master->name, load);
	    }
	}
    }

    //Update timing for current cell
    oaOccInstTerm *currOutInstTerm = DesignTool::getOccOutInstTerm(occInst);
    TPoint *currOutTP = TPoint::get(currOutInstTerm);
    assert(currOutTP);
    if(!loadGiven) currLoad = currOutTP->load;
    oaIter<oaOccInstTerm> instTermIter = occInst->getInstTerms();	
    while(oaOccInstTerm *iTerm = instTermIter.getNext()){
	TPoint *tp = TPoint::get(iTerm);
	assert(tp);
	if(tp->type == TIMING_POINT_SIGNAL_IN){
#if defined(DEBUG)
	    std::cout << " update(" << Util::getBlockName(iTerm) << ")" << std::endl;
#endif
	    computeArr(iTerm, masterCell, currLoad);
	}
    }

    //Update timing for all immediate fanout
    for(int j=0; j<fanoutVec.size(); j++){
	oaOccInstTerm *outInstTerm = DesignTool::getOccOutInstTerm(fanoutVec[j]);
	TPoint *outTP = TPoint::get(outInstTerm);
	assert(outTP);
	double load = outTP->load ;
	CellData *cell = CellData::get(fanoutVec[j]->getModInst());
	assert(cell);
	oaIter<oaOccInstTerm> iTermIter = fanoutVec[j]->getInstTerms();	
	while(oaOccInstTerm *iTerm = iTermIter.getNext()){
	    TPoint *tp = TPoint::get(iTerm);
	    assert(tp);
	    if(tp->type == TIMING_POINT_SIGNAL_IN){
#if defined(DEBUG)
		std::cout << " update(" << Util::getBlockName(iTerm) << ")" << std::endl;
#endif
		computeArr(iTerm, cell->master->name, load);
	    }
	}
    }
}
/*---------------------------------------------------------*/
void
SubTimer::computeArr(oaOccInstTerm *inpTerm, oaString cellMaster, double load){

#if defined(DEBUG)
    std::cout << "computeArr(" << Util::getBlockName(inpTerm) << "," << cellMaster
	<< "," << load << ")" << std::endl;
#endif

    getOtherArr(inpTerm);
    oaOccInstTerm *outInstTerm = DesignTool::getOccOutInstTerm(inpTerm->getInst());
    map<oaOccInstTerm*,SubTPoint>::iterator tpMapIter;
    tpMapIter = tpMap.find(inpTerm);

    if(tpMapIter == tpMap.end()){
	std::cout << "Error: no input arrival time propagated forward" << std::endl;
	exit(1);
    }else{
	SubTPoint subTP = tpMapIter->second;
	double riseSlew = subTP.riseSlew;
	double fallSlew = subTP.fallSlew;
	double riseArr = subTP.riseArr;
	double fallArr = subTP.fallArr;

	//double delay = -DBL_MAX;
	oaModule* topCell = DesignTool::getCellTopMod(cellMaster);
	assert(topCell);
	oaModTerm *outTerm;
	oaIter<oaModTerm> termIter = topCell->getTerms();
	while(oaModTerm *term = termIter.getNext()){
	    oaTermType type(term->getTermType());
	    if(type == oacOutputTermType){
		outTerm = term;
	    }
	}
	TPointMaster *otherTM = TPointMaster::get(outTerm);
	assert(otherTM);
	oaModule *master = inpTerm->getInst()->getModInst()->getMasterModule();
	assert(master);

	TPointMaster::pathVector::iterator i;
	for (i = otherTM->inPaths.begin(); i != otherTM->inPaths.end(); ++i) {
	    assert(i->other);
	    oaName name;
	    i->other->getName(name);
	    oaModTerm *inTerm = oaModTerm::find(master,name);
	    assert(inTerm);
	    oaOccInstTerm *other = timer->getOccFromMod(inpTerm->getInst(), inTerm);
	    if (!other) continue;
	    if(other == inpTerm){
		double d = 0.0;
		double t3;
		if (i->transition & TPointMaster::TRANSITION_SRC_RISE) {
		    d = i->delay->lookup(load, riseSlew, otherTM->loadLimit);
		    t3 = d + riseArr;
#if defined(DEBUG)
		    std::cout << " src_rise_arr:" << t3 << "=" 
			<< d << "(" << riseSlew << "," << load << ")+" << riseArr << std::endl;
#endif

		} 
		else if (i->transition & TPointMaster::TRANSITION_SRC_FALL)
		{
		    d = i->delay->lookup(load, fallSlew, otherTM->loadLimit);
		    t3 = d + fallArr;
#if defined(DEBUG)
		    std::cout << " src_fall_arr:" << t3 << "=" 
			<< d << "(" << fallSlew << "," << load << ")+" << riseArr << std::endl;
#endif
		}
		
		if(d > subTP.delay){ 
		    subTP.delay = d; 
		    tpMap[inpTerm] = subTP;
		}
		map<oaOccInstTerm*,SubTPoint>::iterator iter;
		iter = tpMap.find(outInstTerm);
		SubTPoint otherTP;
		if(iter != tpMap.end()){
		    otherTP = iter->second;
		}

		if (i->transition & TPointMaster::TRANSITION_DST_RISE) {
		    if (t3 > otherTP.riseArr)  {
#if defined(DEBUG)
			std::cout << " DST_RISE ARR:" << t3 << std::endl;
#endif
			otherTP.riseArr = t3;
		    }
		}
		else if (i->transition & TPointMaster::TRANSITION_DST_FALL) {
		    if (t3 > otherTP.fallArr) {
#if defined(DEBUG)
			std::cout << " DST_RISE FALL:" << t3 << std::endl;
#endif
			otherTP.fallArr = t3;
		    }
		}
		//Slew
#define propagateSlew
		//otherTP.riseSlew = 0.01; otherTP.fallSlew = 0.01;
#if defined(propagateSlew)
		// Output Slew Update
		double sl3;
		if (i->transition & TPointMaster::TRANSITION_DST_RISE) {
		    if (i->transition & TPointMaster::TRANSITION_SRC_RISE) {
			if (i->slew) {
			    sl3 = i->slew->lookup(load, riseSlew, otherTM->loadLimit);
#if defined(DEBUG)
			    std::cout << " slew_rise->rise:" 
				<< sl3 << "(" << riseSlew << "," << load << ")" << std::endl;
#endif
			} else {
			    sl3 = riseSlew;
			}
		    } 
		    else if (i->transition & 
			     TPointMaster::TRANSITION_SRC_FALL) {
			if (i->slew) {
			    sl3 = i->slew->lookup(load, fallSlew, otherTM->loadLimit);
#if defined(DEBUG)
			    std::cout << " slew_fall->rise:" 
				<< sl3 << "(" << fallSlew << "," << load << ")" << std::endl;
#endif
			} else {
			    sl3 = fallSlew;
			}
		    } 
		    if (sl3 > otherTP.riseSlew) {
#if defined(DEBUG)
			std::cout << " DST_RISE SLEW:" << sl3 << std::endl;
#endif
			otherTP.riseSlew = sl3;
		    }
		} else if (i->transition & TPointMaster::TRANSITION_DST_FALL){
		    if (i->transition & TPointMaster::TRANSITION_SRC_RISE) {
			if (i->slew) {
			    sl3 = i->slew->lookup(load, riseSlew, otherTM->loadLimit);
#if defined(DEBUG)
			    std::cout << " slew_rise->fall:" 
				<< sl3 << "(" << fallSlew << "," << load << ")" << std::endl;
#endif
			} else {
			    sl3 = riseSlew;
			}
		    } else if (i->transition & 
			       TPointMaster::TRANSITION_SRC_FALL) {
			if (i->slew) {
			    sl3 = i->slew->lookup(load, fallSlew, otherTM->loadLimit);
#if defined(DEBUG)
			    std::cout << " slew_fall->fall:" 
				<< sl3 << "(" << fallSlew << "," << load << ")" << std::endl;
#endif
			} else {
			    sl3 = fallSlew;
			}
		    } 
		    if (sl3 > otherTP.fallSlew) {
#if defined(DEBUG)
			std::cout << " DST_FALL SLEW:" << sl3 << std::endl;
#endif
			otherTP.fallSlew = sl3;
		    }
		}
#endif
		tpMap[outInstTerm] = otherTP;
	    }
	}
    }
}
/*---------------------------------------------------------*/
void
SubTimer::getOtherArr(oaOccInstTerm *i){
#if defined(DEBUG)
    std::cout << "getOtherArr(" << Util::getBlockName(i) << ")" << std::endl;
#endif

    bool noOther = true;
    TPoint *currTP = TPoint::get(i);
    assert(currTP);
    //computeNetDelay(i);
    if(currTP->isFromPI){
	noOther = false;
	SubTPoint subTP;
	subTP.riseArr = timer->getRiseArr(i);
	subTP.fallArr = timer->getFallArr(i);
	subTP.riseSlew = timer->getRiseSlew(i);
	subTP.fallSlew = timer->getFallSlew(i);
	tpMap[i] = subTP;
#if defined(DEBUG)
	std::cout << " fromPI:" << subTP.riseArr << "/" << subTP.fallArr
	    << "-" << subTP.riseSlew << "/" << subTP.fallSlew << std::endl;
#endif
    }

    map<oaOccInstTerm*,SubTPoint>::iterator tpMapIter;
    oaIter<oaOccInstTerm> iTermIter = i->getNet()->getInstTerms();	
    while(oaOccInstTerm *otherTerm = iTermIter.getNext()){
	TPoint *otherTP = TPoint::get(otherTerm);
	assert(otherTP);
	if(otherTP->type == TIMING_POINT_SIGNAL_OUT){
	    noOther = false;
#if defined(DEBUG)
	    std::cout << " from " << Util::getBlockName(otherTerm) << std::endl;
#endif
	    tpMapIter = tpMap.find(otherTerm);
	    if(tpMapIter == tpMap.end()){
		SubTPoint subTP;
		subTP.riseArr = timer->getRiseArr(i);
		subTP.fallArr = timer->getFallArr(i);
		subTP.riseSlew = timer->getRiseSlew(i);
		subTP.fallSlew = timer->getFallSlew(i);
		tpMap[i] = subTP;
#if defined(DEBUG)
		std::cout << " NF " << subTP.riseArr << "/" << subTP.fallArr
		    << "-" << subTP.riseSlew << "/" << subTP.fallSlew << std::endl;
#endif
	    }else{
		SubTPoint subTP;
		SubTPoint otherSubTP = tpMapIter->second;
		subTP.riseArr = otherSubTP.riseArr + currTP->netDelay;
		subTP.fallArr = otherSubTP.fallArr + currTP->netDelay;
		subTP.riseSlew = otherSubTP.riseSlew;
		subTP.fallSlew = otherSubTP.fallSlew;
		tpMap[i] = subTP;
#if defined(DEBUG)
		std::cout << " F" << subTP.riseArr << "/" << subTP.fallArr
		    << "-" << subTP.riseSlew << "/" << subTP.fallSlew << std::endl;
#endif
	    }
	}
    }
    if(noOther){
	SubTPoint subTP;
	subTP.riseArr = 0.0;
	subTP.fallArr = 0.0;
	subTP.riseSlew = 0.0;
	subTP.fallSlew = 0.0;
	tpMap[i] = subTP;
    }
}
/*---------------------------------------------------------*/
void 
SubTimer::updateReq(){
#if defined(DEBUG)
    std::cout << "updateReq()" << std::endl;
#endif
    //Update timing for all immediate fanout
    for(int j=0; j<fanoutVec.size(); j++){
	oaOccInstTerm *outInstTerm = DesignTool::getOccOutInstTerm(fanoutVec[j]);
	TPoint *outTP = TPoint::get(outInstTerm);
	assert(outTP);
	double load = outTP->load ;
	CellData *cell = CellData::get(fanoutVec[j]->getModInst());
	assert(cell);
	oaIter<oaOccInstTerm> iTermIter = fanoutVec[j]->getInstTerms();	
	while(oaOccInstTerm *iTerm = iTermIter.getNext()){
	    TPoint *tp = TPoint::get(iTerm);
	    assert(tp);
	    if(tp->type == TIMING_POINT_SIGNAL_IN){
#if defined(DEBUG)
		std::cout << " updateReq(" << Util::getBlockName(iTerm) 
		    << ")" << std::endl;
#endif
		computeReq(iTerm, cell->master->name, load);
	    }
	}
    }

    //Update timing for current cell
    oaOccInstTerm *currOutInstTerm = DesignTool::getOccOutInstTerm(occInst);
    TPoint *currOutTP = TPoint::get(currOutInstTerm);
    assert(currOutTP);
    if(!loadGiven) currLoad = currOutTP->load;
    oaIter<oaOccInstTerm> instTermIter = occInst->getInstTerms();	
    while(oaOccInstTerm *iTerm = instTermIter.getNext()){
	TPoint *tp = TPoint::get(iTerm);
	assert(tp);
	if(tp->type == TIMING_POINT_SIGNAL_IN){
#if defined(DEBUG)
	    std::cout << " updateReq(" << Util::getBlockName(iTerm) 
		<< ")" << std::endl;
#endif
	    computeReq(iTerm, masterCell, currLoad);
	}
    }

    //Update timing for all fanin
    for(int i=0; i<faninVec.size(); i++){
	oaOccInstTerm *outInstTerm = DesignTool::getOccOutInstTerm(faninVec[i]);
	double load = getCapLoad(outInstTerm,masterCell);
	CellData *cell = CellData::get(faninVec[i]->getModInst());
	assert(cell);
	oaIter<oaOccInstTerm> iTermIter = faninVec[i]->getInstTerms();	
	while(oaOccInstTerm *iTerm = iTermIter.getNext()){
	    TPoint *tp = TPoint::get(iTerm);
	    assert(tp);
	    if(tp->type == TIMING_POINT_SIGNAL_IN){
#if defined(DEBUG)
		std::cout << " updateReq(" << Util::getBlockName(iTerm) 
		    << ")" << std::endl;
#endif
		computeReq(iTerm,cell->master->name, load);
	    }
	}
    }
}
/*---------------------------------------------------------*/
void
SubTimer::computeReq(oaOccInstTerm *inpTerm, oaString cellMaster, double load){
#if defined(DEBUG)
    std::cout << "computeReq(" << Util::getBlockName(inpTerm) << "," << cellMaster
	<< "," << load << ")" << std::endl;
#endif

    oaOccInstTerm *outInstTerm = DesignTool::getOccOutInstTerm(inpTerm->getInst());

    map<oaOccInstTerm*,SubTPoint>::iterator tpMapInIter, tpMapOutIter;
    tpMapInIter = tpMap.find(inpTerm);
    tpMapOutIter = tpMap.find(outInstTerm);

    if(tpMapInIter == tpMap.end()){
	std::cout << "Error: no input arrival time propagated forward" << std::endl;
	exit(1);
    }
    if(tpMapOutIter == tpMap.end()){
	std::cout << "Error: no output require arrival time propagated back" 
	    << std::endl;
	exit(1);
    }
    SubTPoint subTP_in = tpMapInIter->second;
    SubTPoint subTP_out = tpMapOutIter->second;
    getOtherReq(outInstTerm,subTP_out);

    double riseSlew = subTP_in.riseSlew;
    double fallSlew = subTP_in.fallSlew;

    //double delay = -DBL_MAX;
    oaModule* topCell = DesignTool::getCellTopMod(cellMaster);
    assert(topCell);
    oaModTerm *outTerm;
    oaIter<oaModTerm> termIter = topCell->getTerms();
    while(oaModTerm *term = termIter.getNext()){
	oaTermType type(term->getTermType());
	if(type == oacOutputTermType){
	    outTerm = term;
	}
    }
    TPointMaster *otherTM = TPointMaster::get(outTerm);
    assert(otherTM);
    oaModule *master = inpTerm->getInst()->getModInst()->getMasterModule();
    assert(master);

    TPointMaster::pathVector::iterator i;
    for (i = otherTM->inPaths.begin(); i != otherTM->inPaths.end(); ++i) {
	assert(i->other);
	oaName name;
	i->other->getName(name);
	oaModTerm *inTerm = oaModTerm::find(master,name);
	assert(inTerm);
	oaOccInstTerm *other = timer->getOccFromMod(inpTerm->getInst(), inTerm);
	if (!other) continue;
	if(other == inpTerm){
	    double d = 0.0;
	    double t3 = 0.0;
	    if (i->transition & TPointMaster::TRANSITION_DST_RISE) {
		t3 = subTP_out.riseReq;
	    } else if (i->transition & TPointMaster::TRANSITION_DST_FALL){
		t3 = subTP_out.fallReq;
	    }

	    if (i->transition & TPointMaster::TRANSITION_SRC_RISE) {
		double otherReq = t3;
		d = i->delay->lookup(load, riseSlew, otherTM->loadLimit);
		t3 = t3 - d;
		if(t3 < subTP_in.riseReq){
		    subTP_in.riseReq = t3;
#if defined(DEBUG)
		std::cout << " src_rise_req:" << t3 << "=" 
		    << otherReq << "-" << d 
		    << "(" << riseSlew << "," << load << ")" << std::endl;
#endif
		}

	    } 
	    else if (i->transition & TPointMaster::TRANSITION_SRC_FALL)
	    {
		double otherReq = t3;
		d = i->delay->lookup(load, fallSlew, otherTM->loadLimit);
		t3 = t3 - d;
		if(t3 < subTP_in.fallReq){
		    subTP_in.fallReq = t3;
#if defined(DEBUG)
		std::cout << " src_fall_req:" << t3 << "=" 
		    << otherReq << "-" << d 
		    << "(" << fallSlew << "," << load << ")" << std::endl;
#endif
		}
	    }
	    tpMap[inpTerm] = subTP_in;
	}
    }
}
/*---------------------------------------------------------*/
void
SubTimer::getOtherReq(oaOccInstTerm *i, SubTPoint &subTP){
#if defined(DEBUG)
    std::cout << "getOtherReq(" << Util::getBlockName(i) << ")" << std::endl;
#endif

    TPoint *currTP = TPoint::get(i);
    assert(currTP);
    if(currTP->isToPO){
	subTP.riseReq = timer->getRiseReq(i);
	subTP.fallReq = timer->getFallReq(i);
	tpMap[i] = subTP;
#if defined(DEBUG)
	std::cout << " fromPO:" << subTP.riseReq << "/" << subTP.fallReq << std::endl;
#endif
    }

    map<oaOccInstTerm*,SubTPoint>::iterator tpMapIter;
    oaIter<oaOccInstTerm> iTermIter = i->getNet()->getInstTerms();	
    while(oaOccInstTerm *otherTerm = iTermIter.getNext()){
	TPoint *otherTP = TPoint::get(otherTerm);
	assert(otherTP);
	if(otherTP->type == TIMING_POINT_SIGNAL_IN){
#if defined(DEBUG)
	    std::cout << " from " << Util::getBlockName(otherTerm) << std::endl;
#endif
	    tpMapIter = tpMap.find(otherTerm);
	    if(tpMapIter == tpMap.end()){
		subTP.riseReq = timer->getRiseReq(i);
		subTP.fallReq = timer->getFallReq(i);
		tpMap[i] = subTP;
#if defined(DEBUG)
		std::cout << " NF " << subTP.riseReq 
		    << "/" << subTP.fallReq << std::endl;
#endif
	    }else{
		//computeNetDelay(otherTerm);
		SubTPoint otherSubTP = tpMapIter->second;
		double otherRiseReq = otherSubTP.riseReq - otherTP->netDelay;
		if(otherRiseReq < subTP.riseReq)
		    subTP.riseReq = otherRiseReq;

		double otherFallReq = otherSubTP.fallReq - otherTP->netDelay;
		if(otherFallReq < subTP.fallReq)
		    subTP.fallReq = otherFallReq;
		tpMap[i] = subTP;
#if defined(DEBUG)
		std::cout << " F " << subTP.riseReq << "/" << subTP.fallReq << std::endl;
#endif
	    }
	}
    }
} 
#define Constant 1.0
/*---------------------------------------------------------*/
/*void
SubTimer::computeNetDelay(oaOccInstTerm *i){

    _outputMap outputPin; 
    _inputMap inputPin;
    oaOccNet *net = i->getNet();
    oaNet *blockNet = net->getNet();
    double netDelay = timer->_elmoreWireModel->getWireDelay(blockNet,outputPin,inputPin);

    TPoint *tp = TPoint::get(i);
    assert(tp);
    assert(tp->type == TIMING_POINT_SIGNAL_IN);

    oaInstTerm *instTerm = i->getInstTerm();
    _inputMap::iterator iTIter;
    iTIter = inputPin.find(instTerm);
    if( iTIter != inputPin.end()){
	tp->netDelay = Constant*iTIter->second;
    }else{ 
	tp->netDelay = Constant*netDelay; 
    }
}*/
/*---------------------------------------------------------*/
void
SubTimer::print(oaOccInstTerm *i){
    SubTPoint s = tpMap[i];
    std::cout << Util::getBlockName(i)
	<< "\n slew(r/f):" << s.riseSlew << "/" << s.fallSlew
	<< "\n arr(r/f):" << s.riseArr << "/" << s.fallArr
	<< "\n req(r/f):" << s.riseReq << "/" << s.fallReq
	<< "\n slack(r/f):" << s.getRiseSlack() << "/" << s.getFallSlack()
	<< "/" << s.getSlack()
	<< std::endl;
}
/*---------------------------------------------------------*/
void
SubTimer::printAll(){
    //Print all fanin
    for(int i=0; i<faninVec.size(); i++){
	oaIter<oaOccInstTerm> iTermIter = faninVec[i]->getInstTerms();	
	while(oaOccInstTerm *iTerm = iTermIter.getNext()){
	    print(iTerm);
	}
    }

    //Print current cell
    oaIter<oaOccInstTerm> instTermIter = occInst->getInstTerms();	
    while(oaOccInstTerm *iTerm = instTermIter.getNext()){
	print(iTerm);
	/*SubTPoint s = tpMap[iTerm];
	std::cout << Util::getBlockName(iTerm)
	    << "\n slew(r/f):" << s.riseSlew << "/" << s.fallSlew
	    << "\n arr(r/f):" << s.riseArr << "/" << s.fallArr
	    << "\n req(r/f):" << s.riseReq << "/" << s.fallReq
	    << "\n slack(r/f):" << s.getRiseSlack() << "/" << s.getFallSlack()
	    << "/" << s.getSlack()
	    << std::endl;
	    */
    }

    //Update timing for all immediate fanout
    for(int j=0; j<fanoutVec.size(); j++){
	oaIter<oaOccInstTerm> iTermIter = fanoutVec[j]->getInstTerms();	
	while(oaOccInstTerm *iTerm = iTermIter.getNext()){
	    print(iTerm);
	    /*SubTPoint s = tpMap[iTerm];
	    std::cout << Util::getBlockName(iTerm)
		<< "\n slew(r/f):" << s.riseSlew << "/" << s.fallSlew
		<< "\n arr(r/f):" << s.riseArr << "/" << s.fallArr
		<< "\n req(r/f):" << s.riseReq << "/" << s.fallReq
		<< "\n slack(r/f):" << s.getRiseSlack() << "/" << s.getFallSlack()
		<< "/" << s.getSlack()
		<< std::endl;
		*/
	}
    }

}
/*---------------------------------------------------------*/
}//namespace
