/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Zhong Xiu <zxiu@andrew.cmu.edu>

ChangeLog:
2004-09-15: ChangeLog started
*/

#include "oaDesignDB.h"
#include "oagTimerReport.h"
#include <iostream>
#include <iomanip>
#include <list>
#include "oagTimerExtDelay.h"
#include "oagTimerSdcData.h"
#include "oagTimerSdcParserInt.h"

using namespace oa;

namespace oagTimer {

Report::Report(oaDesign *design, Timer *timer) :
    _timer(timer)
{
    assert(design);
    _occ = design->getTopOccurrence();
    assert(_occ);
    assert(timer);
}

Report::~Report()
{
    // nothing
}

/*!
  Reports the timing information of all terms and inst terms in the design.
*/
void
Report::reportAll()
{
    std::cout << "Updating arrival times ..." << std::endl;
    _timer->updateAllArr();
    std::cout << "Updating required times ..." << std::endl;
    _timer->updateAllReq();
    std::cout << "Generating report ..." << std::endl;
    oaIter<oaOccTerm> termIter(_occ->getTerms());
    std::list<std::string> names;
    std::list<std::string>::iterator it, it1; 
    while (oaOccTerm *term = termIter.getNext()) {
        std::string s = _timer->getBlockName(term);
        for (it = names.begin(); it != names.end(); ++it) {
            if (s < (*it)) {
                names.insert(it, s);
                break;
            }
        }
        if (it == names.end()) {
            names.push_back(s);
        }
    }
    it1 = names.end();
    oaIter<oaOccInstTerm> instTermIter(_occ->getInstTerms());
    while (oaOccInstTerm *term = instTermIter.getNext()) {
        std::string s = _timer->getBlockName(term);
        for (it = it1; it != names.end(); ++it) {
            if (s < (*it)) {
                names.insert(it, s);
                break;
            }
        }
        if (it == names.end()) {
            names.push_back(s);
        }
    }
    for (it = names.begin(); it != names.end(); ++it) {
        oaOccObject *block = _timer->findOccObj(_occ, (*it));
        TPoint *p = TPoint::get(block);
        if (p->multiClockData.size() == 0) {
            continue;
        }

        std::cout << std::setw(20) << "Pin" 
            << std::setw(20) << "Load"
            << std::endl; 
        std::cout << std::setw(20) << (*it);
        print(20, p->getLoad());
        std::cout << std::endl;
        std::cout << std::setw(20) << "Clock" 
            << std::setw(7)  << "Multi" 
            << std::setw(13) << "Rise Slew" 
            << std::setw(20) << "Fall Slew" 
            << std::setw(20) << "Rise Slack" 
            << std::setw(20) << "Fall Slack" 
            << std::setw(20) << "Rise Arrival"
            << std::setw(20) << "Fall Arrival"
            << std::endl;
        TPoint *p1;
        if ((p->type == TIMING_POINT_CLOCK) ||
                (p->multiClockData.size() == 0)) {
            p1 = TPoint::getClock(block);
            if (p1->multiClockData.size() == 0) {
                continue;
            }
            std::cout << std::setw(10) << "As Clock";
            for (unsigned int k = 0; k < p1->multiClockData.size(); ++k) { 
                timingData *d =  p1->multiClockData[k];
                propagatedClock *propClk = 
                    sdcParseData.propagatedClocks[d->clockIndex];
                std::cout << std::setw(10) << propClk->clk->name << 
                std::setw(5) << sdcParseData.propagatedClocks[d->clockIndex]->m;
                print(15, p1->getRiseSlew(k));
                print(20, p1->getFallSlew(k));
                print(20, p1->getRiseReq(k)-p1->getRiseArr(k));
                print(20, p1->getFallReq(k)-p1->getFallArr(k));
                print(20, p1->getRiseArr(k));
                print(20, p1->getFallArr(k));
                std::cout << std::endl;
            }
        }
        for (unsigned int k = 0; k < p->multiClockData.size(); ++k) { 
            timingData *d =  p->multiClockData[k];
            propagatedClock *propClk = 
                sdcParseData.propagatedClocks[d->clockIndex];
            std::cout << std::setw(20) << propClk->clk->name << 
            std::setw(5) << sdcParseData.propagatedClocks[d->clockIndex]->m;
            print(15, p->getRiseSlew(k));
            print(20, p->getFallSlew(k));
            print(20, p->getRiseReq(k)-p->getRiseArr(k));
            print(20, p->getFallReq(k)-p->getFallArr(k));
            print(20, p->getRiseArr(k));
            print(20, p->getFallArr(k));
            std::cout << std::endl;
        }
        std::cout << std::endl;
    }
}

//-----------------------------------------------------------------------------

/*! 
  This function reports the timing information in the path vector.
  \param path the vector which contains the path information
*/
void
Report::pathReport(nodesSlopeDir & path)
{
    if (path.empty()) {
        std::cout << " The path is empty! " << std::endl;
        return;
    }
    std::cout << std::endl;
    std::cout << " Reporting path timing information " << std::endl;
    TPoint *p;
    if ((path[0].first)->isOccTerm()) {
        p = TPoint::get(path[0].first);
        if (p->isPI()) {
            TimerExtDelay *dp = static_cast<TimerExtDelay*>
                (TimerExtDelay::termAppDef->
                 get((static_cast<oaOccTerm*>(path[0].first))));
            if (dp) {
                std::cout << "input_delay";
                print(80, dp->delay);
                std::cout << std::endl;
            }
        }
    }
    for (unsigned i=0; i<path.size(); i++) {
        TPoint *p = TPoint::get(path[i].first);
        if (p->multiClockData.size() == 0) {
            continue;
        }
        std::cout << std::setw(20) << "Pin" 
            << std::setw(20) << "Load"
            << std::endl; 
        std::cout << std::setw(20) << _timer->getBlockName(path[i].first);
        print(20, p->getLoad());
        std::cout << std::endl;
        std::cout << std::setw(20) << "Clock" 
            << std::setw(7)  << "Multi" 
            << std::setw(13) << "Rise Slew" 
            << std::setw(20) << "Fall Slew" 
            << std::setw(20) << "Rise Slack" 
            << std::setw(20) << "Fall Slack" 
            << std::setw(20) << "Rise Arrival"
            << std::setw(20) << "Fall Arrival"
            << std::endl;
        TPoint *p1;
        if ((p->type == TIMING_POINT_CLOCK) ||
                (p->multiClockData.size() == 0)) {
            p1 = TPoint::getClock(path[i].first);
            if (p1->multiClockData.size() == 0) {
                continue;
            }
            std::cout << std::setw(10) << "As Clock";
            for (unsigned int k = 0; k < p1->multiClockData.size(); ++k) { 
                timingData *d =  p1->multiClockData[k];
                propagatedClock *propClk = 
                    sdcParseData.propagatedClocks[d->clockIndex];
                std::cout << std::setw(10) << propClk->clk->name << 
                std::setw(5) << sdcParseData.propagatedClocks[d->clockIndex]->m;
                print(15, p1->getRiseSlew(k));
                print(20, p1->getFallSlew(k));
                print(20, p1->getRiseReq(k)-p1->getRiseArr(k));
                print(20, p1->getFallReq(k)-p1->getFallArr(k));
                print(20, p1->getRiseArr(k));
                print(20, p1->getFallArr(k));
                std::cout << std::endl;
            }
        }
        for (unsigned int k = 0; k < p->multiClockData.size(); ++k) { 
            timingData *d =  p->multiClockData[k];
            propagatedClock *propClk = 
                sdcParseData.propagatedClocks[d->clockIndex];
            std::cout << std::setw(20) << propClk->clk->name << 
            std::setw(5) << sdcParseData.propagatedClocks[d->clockIndex]->m;
            print(15, p->getRiseSlew(k));
            print(20, p->getFallSlew(k));
            print(20, p->getRiseReq(k)-p->getRiseArr(k));
            print(20, p->getFallReq(k)-p->getFallArr(k));
            print(20, p->getRiseArr(k));
            print(20, p->getFallArr(k));
            std::cout << std::endl;
        }
        std::cout << std::endl;
    }
}

//-----------------------------------------------------------------------------

/*!
  This function print a double value.
*/
void
Report::print(int w, const double d)
{
    std::cout.setf(std::ios::fixed, std::ios::floatfield);
    std::cout.precision(7);

    if(d < DBL_MAX && d > -DBL_MAX) {
        if (w)
            std::cout << std::setw(w) << d;
        else
            std::cout << d;
    }
    else if (d < DBL_MAX) {
        if (w)
            std::cout << std::setw(w) << "-Infinity";
        else
            std::cout << "-Infinity";
    }
    else {
        if (w)
            std::cout << std::setw(w) << "Infinity";
        else
            std::cout << "Infinity";
    }
}

//-----------------------------------------------------------------------------

/*! 
  This function prints the information of nets in a vector. 
  \param nNets the vector of nets with slack values
*/ 
void
Report::printNets(netsSlacks  & nNets)
{
  oaNativeNS oaNs;
  std::cout << "The following nets exist in the vector." << std::endl;

  for (unsigned i=0; i<nNets.size(); i++) {
    oaString nextNetName;
    oaName name;
    nNets[i].first->getName(name);
    name.get(oaNs, nextNetName);
    std::cout << "\t" << (const char *)nextNetName << "\t";
    print(0, nNets[i].second);
    std::cout << std::endl;
  }
}

//-----------------------------------------------------------------------------

/*! This function invokes the net iterator for the cellview
    and prints the names of the nets one by one. */
void 
Report::printNets()
{
  // The net iterator for the block is initialized.
  oaIter<oaOccNet> netIterator(_occ->getNets());
  oaOccNet *nextNet;
  oaNativeNS oaNs;

  std::cout << "The following nets exist in this cellView." << std::endl;

  // Run the iterator to get the next net present in the cellView.
  while ((nextNet = netIterator.getNext())) {
    oaString nextNetName;
    oaName name;
    nextNet->getName(name);
    name.get(oaNs, nextNetName);
    std::cout << "\t" << (const char *)nextNetName << std::endl;
  }
}

//-----------------------------------------------------------------------------

/*! 
  This function reports the slacks for all endpoints. 
*/
void
Report::reportEndpointsSlacks()
{
  _timer->getWorstSlack();

  oaNativeNS ns;
  oaString f;

  std::cout << " **** Endpoints slacks: " << std::endl;

  // 1. Primary outputs
  oaIter<oaOccTerm> termIter(_occ->getTerms());
  while (oaOccTerm *i = termIter.getNext()) {
    TPoint *p = TPoint::get(i);
    if (p->isPO()) {
      i->getName(ns, f);
      std::cout << std::setw(20) << f;
      print(20, p->getSlack());
      std::cout << std::endl;
    }
  }

  // 2. InstTerm Clocks
  for (std::set<oaOccObject*>::iterator i=_timer->clockPins.begin();
       i!=_timer->clockPins.end(); ++i)
  {
    //TPoint *p = static_cast<TPoint*>(myAppDef->get(*i));
    oaModTerm *masterTerm = (static_cast<oaOccInstTerm*>(*i))
            ->getModInstTerm()->getTerm();
    assert(masterTerm);
    TPointMaster *tm = TPointMaster::get(masterTerm);
    assert(tm);
    oaOccInst *inst = (static_cast<oaOccInstTerm*>(*i))->getInst();
    TPointMaster::pathVector::iterator t = tm->inPaths.begin();
    assert(t->other);
    oaOccInstTerm *other = Timer::getOccFromMod(inst, t->other);
    if (!other) {
      continue;
    }
    TPoint *otherTP = TPoint::get(other);
    assert(otherTP);
    inst->getName(ns, f);
    std::cout << std::setw(10) << f;
    other->getTermName(ns, f);
    std::cout << std::setw(10) << f;
    print(20, otherTP->getSlack());
    std::cout << std::endl;
  }
}


}
