/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Zhong Xiu <zxiu@andrew.cmu.edu>

ChangeLog:
2004-09-15: ChangeLog started
*/

#include <assert.h>
#include <iostream>

#include "oagTimerModel.h"
#include "oagTimerTimer.h"

//#define DEBUG

namespace oagTimer {

//-----------------------------------------------------------------------------

/*! The constructor, initialize all the members. */
TimerModel::TimerModel() {
  loadSize = slewSize = dataSize = 0;
  originalSlewMajor = false;
}

//-----------------------------------------------------------------------------

/*! The constructor, copy. */
TimerModel::TimerModel(const TimerModel &c) : 
    loadAxis(c.loadAxis), slewAxis(c.slewAxis), tableData(c.tableData) 
{
  loadSize = c.loadSize;
  slewSize = c.slewSize;
  dataSize = c.dataSize;
  originalSlewMajor = c.originalSlewMajor;
}

//-----------------------------------------------------------------------------

/*! Push one number into the load axis. */
int
TimerModel::pushLoadAxis(double d) {
#if defined(DEBUG)
  std::cerr << "LOAD " << d << std::endl;
#endif
  //std::cout << "LoadSize:" << loadSize  << " " << d << std::endl;
  if (loadSize && d < loadAxis[loadSize-1]) {
    std::cerr << "Load axis is not monotone" << std::endl;
    return 1;
  }
  if (originalSlewMajor) {
    assert(slewSize);
  }
  else {
    assert(!slewSize);
  }
  loadAxis.push_back(d);
  ++loadSize;
  return 0;
}

//-----------------------------------------------------------------------------

/*! Push one number into the clock slew axis. */
int
TimerModel::pushClockSlewAxis(double d) {
#if defined(DEBUG)
  std::cerr << "CLOCKSLEW " << d << std::endl;
#endif
  if (originalSlewMajor) {
    assert(slewSize);
  }
  else {
    assert(!slewSize);
  }
  loadAxis.push_back(d);
  ++loadSize;
  return 0;
}

//-----------------------------------------------------------------------------

/*! Push one number into the input slew axis. */
int
TimerModel::pushInputSlewAxis(double d) {
#if defined(DEBUG)
  std::cerr << "INPUTSLEW " << d << std::endl;
#endif
  /*
  if (slewSize && d < slewAxis[slewSize-1]) {
    std::cerr << "Slew axis is not monotone" << std::endl;
    return 1;
  }
  */
  slewAxis.push_back(d);
  ++slewSize;
  if (!loadSize) {
    originalSlewMajor = true;
  }
  return 0;
}

//-----------------------------------------------------------------------------

/*! Push one number into the slew axis. */
int
TimerModel::pushSlewAxis(double d) {
#if defined(DEBUG)
  std::cerr << "SLEW " << d << std::endl;
#endif
  slewAxis.push_back(d);
  ++slewSize;
  if (!loadSize) {
    originalSlewMajor = true;
  }
  return 0;
}

//-----------------------------------------------------------------------------

/*! Reserve the data space for the table. */
void
TimerModel::reserveData() {
  assert(!dataSize);
  if (!loadSize) {
    loadAxis.push_back(0.0);
    loadSize = 1;
  }
  if (!slewSize) {
    slewAxis.push_back(0.0);
    slewSize = 1;
  }
  ptr = 0;
  if (loadSize == 1) dataSize = slewSize * slewSize;
  else if (slewSize == 1) dataSize = loadSize * loadSize;
  else dataSize = loadSize * slewSize;
  tableData.resize(dataSize);
}

//-----------------------------------------------------------------------------

/*! Push one number into the data table. */
int
TimerModel::pushData(double d) {
    //std::cout << "Push Data: " << d << std::endl;
  if (!dataSize) {
    reserveData();
  }
  if (ptr == dataSize) {
    std::cerr << loadSize << " " << slewSize << " ";
    std::cerr << dataSize << " " << std::endl;
    std::cerr << "Too many data items in model" << std::endl;
    return 1;
  }
  unsigned int loc;
  if (originalSlewMajor) {
    unsigned int dSlew = ptr / loadSize;
    unsigned int dLoad = ptr % loadSize;
    loc = dLoad * slewSize + dSlew;
  }
  else {
    loc = ptr;
  }
#if defined(DEBUG)
  std::cout << "LOAD INDEX " << (loc / slewSize) << " SLEW INDEX " <<
      (loc % slewSize) << " DATA " << d << std::endl;
#endif
  assert(loc < dataSize);
  tableData[loc] = d;
  ++ptr;
  return 0;
}

//-----------------------------------------------------------------------------

/*! Clear all the vectors. */
void
TimerModel::clear() {
  loadSize = slewSize = dataSize = 0;
  originalSlewMajor = false;
  loadAxis.clear();
  slewAxis.clear();
  tableData.clear();
}

//-----------------------------------------------------------------------------

/*! 
  Compute the delay value or slew rate from the lookup table. 
  Will print a warning message if the load exceeds the load limit.

  \param load the load value at this point
  \param slew the slew rate at this point
  \param loadLimit the load limit at this point
*/
DelayType
TimerModel::lookup(double load, DelayType slew, double loadLimit) {
  unsigned int loadIndex, loadIndex2, slewIndex, slewIndex2;

  bool checkDelayCalc = false;
  assert(loadSize);
  if (loadSize == 1) {
    loadIndex = loadIndex2 = 0;
  }
  else {
    for (loadIndex = 0; loadIndex < loadSize; ++loadIndex) {
      if (loadAxis[loadIndex] >= load) {
        break;
      }
    }
    if (loadIndex == 0) {
      loadIndex = 1;
      loadIndex2 = 0;

      if (load < 0.0) {
	if (Timer::displayWarning) {
	  std::cerr << "Warning: load below range " << load;
	  std::cerr << " " << slew << std::endl;
	}
#if defined DEBUG
	std::cout << "In load below range warning: loadSize => " << loadSize << " loadIndex=>" << loadIndex << " loadIndex2=>" << loadIndex2 <<  " loadAxis[loadIndex]:" << loadAxis[loadIndex] << std::endl;
#endif
      }
    }
    else if (loadIndex == loadSize) {
      loadIndex = loadSize-1;
      loadIndex2 = loadSize-2;

      if (load > loadLimit) {
	checkDelayCalc = true;
	if (Timer::displayWarning) {
	  std::cerr << "Warning: load above range " << load;
	  std::cerr << " " << slew << std::endl;
	}
#if defined DEBUG
	std::cout << "In load above range warning: loadSize=>" << loadSize << " load=>" << load << " loadLimit=>" << loadLimit << std::endl;
#endif
      }
    }
    else {
      loadIndex2 = loadIndex - 1;
    }
  }

  assert(slewSize);
  if (slewSize == 1) {
    slewIndex = slewIndex2 = 0;
    DelayType t;
    if (loadIndex == loadIndex2) {
      t = tableData[0];
    }
    else {
      t = tableData[loadIndex2]+(tableData[loadIndex]-
         tableData[loadIndex2])*(load-loadIndex2)/(loadIndex-loadIndex2);
    }
    return t;
  }
  else {
    for (slewIndex = 0; slewIndex < slewSize; ++slewIndex) {
      if (slew <= slewAxis[slewIndex]) {
        break;
      }
    }
    if (slewIndex == 0) {
      slewIndex = 1;
      slewIndex2 = 0;

      if (slew < 0.0) {
#if defined DEBUG
	std::cout << "In slew below range warning: slewSize=>" << slewSize << " slewIndex=>" << slewIndex << " slewIndex2=>" << slewIndex2 << " slewAxis[slewIndex]=>" << slewAxis[slewIndex] << std::endl;
#endif
	if (Timer::displayWarning) {
	  std::cerr << "Warning: slew below range " << load;
	  std::cerr << " " << slew << std::endl;
	}
      }
    }
    else if (slewIndex == slewSize) {
      slewIndex = slewSize-1;
      slewIndex2 = slewSize-2;

#if defined DEBUG
	std::cout << "In slew above range warning: slewSize=>" << slewSize << " slewIndex=>" << slewIndex << " slewIndex2=>" << slewIndex2 << std::endl;
#endif
      if (Timer::displayWarning) {
	  std::cerr << "Warning: slew above range " << slew;
	  std::cerr << " " << load << std::endl;
      }
    }
    else {
      slewIndex2 = slewIndex - 1;
    }
  }

  double t = tableData[loadIndex2 * slewSize + slewIndex2]
      * (loadAxis[loadIndex]-load) * (slewAxis[slewIndex]-slew);
  t += tableData[loadIndex2 * slewSize + slewIndex]
      * (loadAxis[loadIndex]-load) * (slew-slewAxis[slewIndex2]);
  t += tableData[loadIndex * slewSize + slewIndex2]
      * (load-loadAxis[loadIndex2]) * (slewAxis[slewIndex]-slew);
  t += tableData[loadIndex * slewSize + slewIndex]
      * (load-loadAxis[loadIndex2]) * (slew-slewAxis[slewIndex2]);

  t = t/(loadAxis[loadIndex]-loadAxis[loadIndex2]);
  t = t/(slewAxis[slewIndex]-slewAxis[slewIndex2]);

  DelayType dt = t;
#if defined DEBUG
  if(checkDelayCalc){
  std::cout << "slew => " << slew << " slewSize => " << slewSize << " slewIndex => " << slewIndex <<" slewIndex2 => " << slewIndex2 << std::endl;
  std::cout << "load => " << load << " loadSize => "<< loadSize << " loadIndex => " << loadIndex << "  " << "loadIndex2 => " << loadIndex2 << std::endl;

  std::cout << "slewAxis1 => " << slewAxis[slewIndex] << " " << "slewAxis2 => " << slewAxis[slewIndex2] << std::endl;
  std::cout << "loadAxis1 => " << loadAxis[loadIndex] << " " << "loadAxis2 => " << loadAxis[loadIndex2] << std::endl;
  std::cout << "tableData1 => " << tableData[loadIndex2*slewSize+slewIndex2] << " " << std::endl;
  std::cout << "tableData2 => " << tableData[loadIndex2*slewSize+slewIndex] << " " << std::endl;
  std::cout << "tableData3 => " << tableData[loadIndex*slewSize+slewIndex2] << " " << std::endl;
  std::cout << "tableData4 => " << tableData[loadIndex*slewSize+slewIndex] << " " << std::endl;
  std::cout << t << std::endl;
  }
#endif 
  
  return dt;
}

//-----------------------------------------------------------------------------

}
