/************************************************************
* 
* File: oagTimerLR.cpp
* Author: Santiago Mok (smok@ee.ucla.edu)
* Created: 01-26-2011
* Last Modified: Thu 23 Jun 2011 10:40:26 PM PDT
*
************************************************************/
#include "oagTimerLR.h"
#include "oagTimerUtil.h"
#include "oagTimerDesignTool.h"
#include "oagTimerLibParserInt.h"
#include "oagTimerSubTimer.h"

#include "solverIF.h"

#include "assert.h"
#include <time.h>
#include <vector>

//#define DEBUG
//#define DEBUG1
//#define LMPO
namespace oagTimer{

using namespace oa;
using namespace std;
using namespace solver;

/*---------------------------------------------------------*/
LR::LR(oaDesign *d, Timer *t){
      design = d;
      timing = t;
      tau = 0.0;
      factor = 0.0;

      iterRefine = false;
};
/*---------------------------------------------------------*/
LR::~LR(){}
/*---------------------------------------------------------*/
void
LR::runDebug(){

    timing->printDetailTiming();
    G = DesignTool::getTopological(design,timing); 
    for(int i=0; i<G.size(); i++){
        CellData *cell = CellData::get(G[i]);
        assert(cell);
        vector<oaString> sizesData = cell->getAllSizes(); 
        //std::cout << cell->instName << " datasize:" << sizesData.size() << std::endl;
        //if(cell->instName != "g11") continue;
        for(int j=0; j<sizesData.size(); j++){
            oaString toSize = sizesData[j];
            //oaString toSize = cell->master->name;
            std::cout << "\n" << Util::getInstName(G[i]) 
                << "(" << cell->master->name << " -> " << toSize << ")" << std::endl;
            SubTimer currInstTimer(design,timing,cell->occInst, toSize);
            oaOccInstTerm *outTerm = DesignTool::getOccOutInstTerm(cell->occInst);
            TPoint *outPin = TPoint::get(outTerm);
            //currInstTimer.setLoad(outPin->load);
            double dDelay = currInstTimer.getDeltaDelay();
            double dSlack = currInstTimer.getDeltaSlack();
            oaIter<oaOccInstTerm> itIter = cell->occInst->getInstTerms();
            while(oaOccInstTerm *inTerm = itIter.getNext()){
                TPoint *tp = TPoint::get(inTerm);
                if(tp->type == TIMING_POINT_SIGNAL_IN){
                    double delay = currInstTimer.getCellDelayEstimate(inTerm);
                    std::cout << " d:" << delay << std::endl;
                }
            }
            
            std::cout << " (dD/dS):" << dDelay << "/" << dSlack << std::endl;
            /*cell->swap(toSize);
            ut.invalidateFanIOTiming(design,timing,cell->inst);
            timing->updateAll();
            printAll(cell->occInst);    

            //Reverse to original cell
            cell->reverseSwap();
            ut.invalidateFanIOTiming(design,timing,cell->inst);
            timing->updateAll();
            */
             
            //break; 
        }
        //break;
    }
}
/*---------------------------------------------------------*/
void
LR::printAll(oaOccInst *inst){
    //std::cout << "Print Actual Timing" << std::endl;
    //Print for each fanin
    oaIter<oaOccInstTerm> itIter = inst->getInstTerms();
    while(oaOccInstTerm *iTerm = itIter.getNext()){
        TPoint *tp = TPoint::get(iTerm);
        assert(tp);
        if(tp->type == TIMING_POINT_SIGNAL_IN){
            oaIter<oaOccInstTerm> itIter2 = iTerm->getNet()->getInstTerms();
            while(oaOccInstTerm *iTerm2 = itIter2.getNext()){
                TPoint *otherTP = TPoint::get(iTerm2);
                assert(otherTP);
                if(otherTP->type == TIMING_POINT_SIGNAL_OUT){
                    printTiming(iTerm2->getInst());     
                }
            }
        }
    }
    //Print current cell
    printTiming(inst);

    //Print for each fanout
    oaIter<oaOccInstTerm> itIter3 = inst->getInstTerms();
    while(oaOccInstTerm *iTerm = itIter3.getNext()){
        TPoint *tp = TPoint::get(iTerm);
        assert(tp);
        if(tp->type == TIMING_POINT_SIGNAL_OUT){
            oaIter<oaOccInstTerm> itIter2 = iTerm->getNet()->getInstTerms();
            while(oaOccInstTerm *iTerm2 = itIter2.getNext()){
                TPoint *otherTP = TPoint::get(iTerm2);
                assert(otherTP);
                if(otherTP->type == TIMING_POINT_SIGNAL_IN){
                    printTiming(iTerm2->getInst());     
                }
            }
        }
    }
}
/*---------------------------------------------------------*/
void
LR::printTiming(oaOccInst *inst){
    oaIter<oaOccInstTerm> itIter = inst->getInstTerms();
    while(oaOccInstTerm *iTerm = itIter.getNext()){
        double riseSlew = timing->getRiseSlew(iTerm);
        double fallSlew = timing->getFallSlew(iTerm);
        double riseArr = timing->getRiseArr(iTerm);
        double fallArr = timing->getFallArr(iTerm);
        double riseReq = timing->getRiseReq(iTerm);
        double fallReq = timing->getFallReq(iTerm);
        std::cout << Util::getBlockName(iTerm)
            << "\n slew(r/f):" << riseSlew << "/" << fallSlew
            << "\n arr(r/f):" << riseArr << "/" << fallArr
            << "\n req(r/f):" << riseReq << "/" << fallReq
            << std::endl;
    }
}
/*---------------------------------------------------------*/
void
LR::runDP(int type){
    time_t start; 
    time_t end;
    time (&start);

    Util ut;
    std::cout << "Initial Design Power:" 
        << ut.getTotalLeakPower(design) 
        << " target:" << timing->getPeriod() << std::endl;

    setLMIndex();

    G = DesignTool::getTopological(design,timing); 
    int graphSize = G.size(); 
    double hTol = 0.0;
    double lTol = 0.0;
    double tolHFactor = 0.0;
    double tolLFactor = 0.0;

    double currPower;
    double currWObj;
    tolSet = false;
    
    double minalpha;
    if(libParseData.libType.compare("footprint") == 0 ){
        alpha1=0.001;
        //alpha1=0.2;
        //alpha1=0.288;
         minalpha= 1e-4;
        //minalpha= 0.115;
        factor = 1e-6; //nangate
        tolLFactor = 0.002;
        tolHFactor = 0.20;
    } else if(libParseData.libType.compare("nangate") == 0){
        alpha1=0.01;
        //alpha1=0.2;
        //alpha1=0.288;
         minalpha= 1e-4;
        //minalpha= 0.115;
        factor = 1e-6; //nangate
        tolLFactor = 0.002;
        tolHFactor = 0.20;
    }else if(libParseData.libType.compare("st") == 0
            || libParseData.libType.compare("st_vt") == 0){
        alpha1=0.2;
        minalpha= 1e-6;
        //minalpha= 0.12;
        factor = 1e5;  //st
        tolLFactor = 0.002;
        tolHFactor = 0.20;
    }else if(libParseData.libType.compare("eyechart") == 0){
        alpha1 = 0.1;
        factor = 1e-3;  //st
        tolLFactor = 0.002;
        tolHFactor = 0.20;
        minalpha = 0.001;
    }else{ 
        std::cout << "Error in LR optimization: power scaling factor not set" << std::endl;
    }
    alpha2=tau*alpha1;
    int cnt = 0;
    //int noMoveCount = 0;
    while(cnt < 500){
        cnt++;
        map<oaModInst*,oaString> currSizes;
        saveSizing(currSizes);
        currPower = ut.getTotalLeakPower(design);
        currWObj = getLRObjective();

        double prevArr = timing->getWorstArr();
        if(type == 0){
            runDPAssign();
        }else{
            runAssign();
        }
        double newWObj = getLRObjective();
        double worstArr = timing->getWorstArr();
        double dDelay = prevArr-worstArr;

        double worstSlack = timing->getWorstSlack();
        //std::cout << "wSlack:" << worstSlack << std::endl;
//#if defined(DEBUG)
        if(tolSet){
            std::cout << cnt << ": alpha:" << alpha1 //<< "/" << alpha2 
                        << " / " << dDelay
                        << " / " << worstArr 
                        << " / " << ut.getTotalLeakPower(design) << std::endl; 
            std::cout << "(newObj/currObj):" << newWObj << " / " << currWObj << " ";
            std::cout << "wSlack:" << worstSlack << std::endl;
        }
//#endif
        if(!tolSet){
            tolSet = true;
            lTol = (worstArr-timing->getPeriod())*tolLFactor;
            hTol = (worstArr-timing->getPeriod())*tolHFactor;
#if defined(DEBUG)
            std::cout << "tolerance:" << lTol << "/" << hTol << std::endl;
            std::cout << "MinSizeDelay:" << worstArr 
                << "/" << worstSlack << " - power:" << ut.getTotalLeakPower(design) << std::endl;
#endif
        }
        else if(dDelay > hTol){ //too fast
            if(alpha1 < minalpha && (worstArr > timing->getPeriod()) ){
                // nothing -- delay decreasing too fast (taken for small alpha threshold
                double current_best_timing = timing->getWorstArr();
                double current_best_power = ut.getTotalLeakPower(design);
                std::cout << "power:" << current_best_power << " - timing:" << current_best_timing << std::endl;
            }else{
                reloadSizing(currSizes);
                reverseLM();
                alpha1 *= 0.50;
            }
        }else if(dDelay < 0){   //delay increased!
            //reloadSizing(currSizes);
            //reverseLM();
            alpha1 *= 0.75;
        }else if(dDelay < lTol){ //too slow
            if(dDelay == 0){
                alpha1 *= 1.2;
            }else{
                alpha1 *= 1.1;
            }
            if(alpha1 > 2000){
                alpha1 = 0.01;
            }
        }else{      
            // nothing -- regular decrement
            double current_best_timing = timing->getWorstArr();
            double current_best_power = ut.getTotalLeakPower(design);
            std::cout << "power:" << current_best_power << " - timing:" << current_best_timing << std::endl;
        }
        //Stoppping condition
        if(dDelay < hTol){
            if(worstSlack >= 0){
                std::cout << "Ending: delay met - " << worstArr 
                    << " power:" << ut.getTotalLeakPower(design) << std::endl;
                break;
            }
        }
        if (alpha1 < 1e-10) {
          std::cout << "alpha near zero - breaking" << endl;
          break;
        }
        alpha2 = tau*alpha1;
        updateLM();
    }
    if (cnt >= 500)
        std::cout<<"Reached max iters "<<endl;
    time (&end);
    std::cout << "Total Runtime:" << difftime(end,start) << " (s)" << std::endl;
}
/*---------------------------------------------------------*/
void
LR::projectLM(){
    solverIF s;
    s.initializeSolver(_N); // # OF VARIABLES
    for(int i=1; i<_N+1; i++){
        s.setLowerBound(i, 0.0);
    }

    // Set linear part of the objective 
    double val[_N];
    int inds[_N];
    int pos = 0;
    oaOccurrence *occ = design->getTopOccurrence();
    oaIter<oaOccInstTerm> oitIter(occ->getInstTerms());
    while(oaOccInstTerm *oit = oitIter.getNext()){ 
        TPoint *tp = TPoint::get(oit);
        assert(tp);
        if(tp->type == TIMING_POINT_SIGNAL_IN){
            oaOccInst *inst = oit->getInst();
            CellData *cell = CellData::get(inst->getModInst());
            assert(cell);
            if(cell->master->isSequential) continue;

            if(tp->lm != 0){
                val[pos] = -2.0*tp->lm;
                inds[pos] = tp->lmIndex;
                ++pos;
            }
        }
    }
#if defined(LMPO)
    oaIter<oaOccTerm> otIter(occ->getTerms());
    while(oaOccTerm *ot = otIter.getNext()){ 
        TPoint *tp = TPoint::get(ot);
        assert(tp);
        if(tp->type == TIMING_POINT_PO){
            if(timing->getArr(ot) == DBL_MAX) continue;
            if(tp->lm != 0){
                val[pos] = -2.0*tp->lm;
                inds[pos] = tp->lmIndex;
                ++pos;
            }
        }
    }
#endif
    /*std::cout << "\nCheck Linear Part (pos:" << pos << ")" << std::endl;
    for(int i=0; i<pos; ++i){
        std::cout << i << ":" << inds[i] << " - " << val[i]/(-2.0) << std::endl;
    }*/
    s.setObjective(pos, val, inds);
    
    // Set constraint A*vec(lm)=0
    for(int i=0; i<G.size(); i++){
        vector<int> indices = getIndices(G[i]);
        if(indices.empty()) continue;

        int varCnt = (int)indices.size();
        double cval[varCnt];
        int cinds[varCnt];

        for(int j=0; j<varCnt; j++){    
            if(indices[j] < 0) cval[j] = -1.0;
            else cval[j] = 1.0;
            cinds[j] = abs(indices[j]);
        }
        s.addConstraint(varCnt, cval, cinds, BD_EQ, 0.0);
    }

    // Set quadratic part of the objective
    for(int i=1; i<_N+1; i++){
        s.addQuadObjectiveEntry(2.0, i, i);
    }

    double LM[_N];
    s.solveQP(LM);

    // Replace TPoint lm with new LM solutions
    oaIter<oaOccInstTerm> oitIter2(occ->getInstTerms());
    while(oaOccInstTerm *oit = oitIter2.getNext()){ 
        TPoint *tp = TPoint::get(oit);
        assert(tp);
        if(tp->type == TIMING_POINT_SIGNAL_IN){
            oaOccInst *inst = oit->getInst();
            CellData *cell = CellData::get(inst->getModInst());
            assert(cell);
            if(cell->master->isSequential) continue;
            /*std::cout << Util::getBlockName(oit) << ":" << tp->lm 
                << ":" << LM[tp->lmIndex-1]<< std::endl;*/
            tp->lm = LM[tp->lmIndex-1];
        }
    }
#if defined(LMPO)
    oaIter<oaOccTerm> otIter2(occ->getTerms());
    while(oaOccTerm *ot = otIter2.getNext()){ 
        TPoint *tp = TPoint::get(ot);
        assert(tp);
        if(tp->type == TIMING_POINT_PO){
            if(timing->getArr(ot) == DBL_MAX) continue;
            //std::cout << Util::getBlockName(ot) << ":" << tp->lm 
                //<< ":" << LM[tp->lmIndex-1]<< std::endl;
            tp->lm = LM[tp->lmIndex-1];
        }
    }
#endif
}
/*---------------------------------------------------------*/
double
LR::getMaxInArr(oaOccInst *inst){
    double maxArr = -DBL_MAX;
    oaIter<oaOccInstTerm> oitIter(inst->getInstTerms());
    while(oaOccInstTerm *oit = oitIter.getNext()){ 
        TPoint *tp = TPoint::get(oit);
        assert(tp);
        if(tp->type == TIMING_POINT_SIGNAL_IN){
            double arr = timing->getArr(oit);
            //std::cout << Util::getBlockName(oit) << "-arr:" << arr << std::endl;
            if(arr > maxArr) maxArr = arr;
        }
    }
    return maxArr;
}
/*---------------------------------------------------------*/
/*void
LR::updateLM(){
    int indx=0;
    oaOccurrence *occ = design->getTopOccurrence();
    
    oaIter<oaOccInst> occIter(occ->getInsts());
    while(oaOccInst *occInst = occIter.getNext()){ 
        double maxArr = getMaxInArr(occInst);
        oaIter<oaOccInstTerm> oitIter(occInst->getInstTerms());
        while(oaOccInstTerm *oit = oitIter.getNext()){ 
            TPoint *tp = TPoint::get(oit);
            assert(tp);
            if(tp->type == TIMING_POINT_SIGNAL_IN){
                tp->lm_old = tp->lm;
                double arr = timing->getArr(oit);
                //std::cout << "maxArr:" << maxArr << " arr:" << arr << std::endl;
                tp->lm = tp->lm + alpha1*(maxArr - arr);
            }
        }
    }
    oaIter<oaOccTerm> otIter(occ->getTerms());
    while(oaOccTerm *ot = otIter.getNext()){ 
        TPoint *tp = TPoint::get(ot);
        assert(tp);
        if(tp->type == TIMING_POINT_PO){
            if(timing->getArr(ot) == DBL_MAX) continue;
            tp->lm_old = tp->lm;
            double dArr = timing->getArr(ot) - timing->getPeriod();
            //std::cout << Util::getBlockName(ot) << "- dArr:" << dArr << std::endl;
                tp->lm = tp->lm + alpha1*(dArr);
        }
    }
    //std::cout << "Before Projection" << std::endl;
    //printLMValues();
    projectLM();
    //std::cout << "After Projection" << std::endl;
    //printLMValues();
    //checkLMValues();
}*/
/*---------------------------------------------------------*/
void
LR::updateLM(){
    int indx=0;
    oaOccurrence *occ = design->getTopOccurrence();
    oaIter<oaOccInstTerm> oitIter(occ->getInstTerms());
    while(oaOccInstTerm *oit = oitIter.getNext()){ 
        TPoint *tp = TPoint::get(oit);
        assert(tp);
        if(tp->type == TIMING_POINT_SIGNAL_IN){
            oaOccInst *inst = oit->getInst();
            CellData *cell = CellData::get(inst->getModInst());
            assert(cell);
            if(cell->master->isSequential) continue;
            tp->lm_old = tp->lm;
            double slack = timing->getSlack(oit);
            if(slack < 0) {
                //std::cout << "sw" << Util::getBlockName(oit) << ":" << slack << std::endl;
                tp->lm = tp->lm - alpha1*slack;
            }
            else 
                tp->lm = tp->lm - alpha2*slack;
        }
    }
#if defined(LMPO)
    oaIter<oaOccTerm> otIter(occ->getTerms());
    while(oaOccTerm *ot = otIter.getNext()){ 
        TPoint *tp = TPoint::get(ot);
        assert(tp);
        if(tp->type == TIMING_POINT_PO){
            if(timing->getArr(ot) == DBL_MAX) continue;
            tp->lm_old = tp->lm;
            double slack = timing->getSlack(ot);
            if(slack < 0) 
                tp->lm = tp->lm - alpha1*slack;
            else 
                tp->lm = tp->lm - alpha2*slack;
        }
    }
#endif
    //std::cout << "Before Projection" << std::endl;
    //printLMValues();
    projectLM();
    //std::cout << "After Projection" << std::endl;
    //printLMValues();
    //checkLMValues();
}
/*---------------------------------------------------------*/
void
LR::reverseLM(){
    oaOccurrence *occ = design->getTopOccurrence();
    oaIter<oaOccInstTerm> oitIter(occ->getInstTerms());
    while(oaOccInstTerm *oit = oitIter.getNext()){ 
        TPoint *tp = TPoint::get(oit);
        assert(tp);
        if(tp->type == TIMING_POINT_SIGNAL_IN){
            oaOccInst *inst = oit->getInst();
            CellData *cell = CellData::get(inst->getModInst());
            assert(cell);
            if(cell->master->isSequential) continue;
            tp->lm = tp->lm_old;
        }
    }
#if defined(LMPO)
    oaIter<oaOccTerm> otIter(occ->getTerms());
    while(oaOccTerm *ot = otIter.getNext()){ 
        TPoint *tp = TPoint::get(ot);
        assert(tp);
        if(tp->type == TIMING_POINT_PO){
            if(timing->getArr(ot) == DBL_MAX) continue;
            tp->lm = tp->lm_old;
        }
    }
#endif
}
/*---------------------------------------------------------*/
void
LR::setLMIndex(){
    int n = 0;
    oaOccurrence *occ = design->getTopOccurrence();
    oaIter<oaOccInstTerm> oitIter(occ->getInstTerms());
    while(oaOccInstTerm *oit = oitIter.getNext()){ 
        TPoint *tp = TPoint::get(oit);
        assert(tp);
        if(tp->type == TIMING_POINT_SIGNAL_IN){
            oaOccInst *inst = oit->getInst();
            CellData *cell = CellData::get(inst->getModInst());
            assert(cell);
            if(cell->master->isSequential) continue;

            tp->lmIndex = ++n;
        }
    }
#if defined(LMPO)
    oaIter<oaOccTerm> otIter(occ->getTerms());
    while(oaOccTerm *ot = otIter.getNext()){ 
        TPoint *tp = TPoint::get(ot);
        assert(tp);
        if(tp->type == TIMING_POINT_PO){
            if(timing->getArr(ot) == DBL_MAX) continue;
            tp->lmIndex = ++n;
        }
    }
#endif
    _N = n;
}
/*---------------------------------------------------------*/
vector<int> 
LR::getIndices(oaModInst *inst){
    vector<int> indices;
    oaOccNet *net;
    bool toPO = true;
    oaOccInst *occInst = DesignTool::getOccInst(design, inst);
    //Get the input LMIndex;
    oaIter<oaOccInstTerm> oitIter(occInst->getInstTerms()); 
    while(oaOccInstTerm *oit = oitIter.getNext()){
        TPoint *tp = TPoint::get(oit);
        assert(tp);
        if(tp->type == TIMING_POINT_SIGNAL_IN){
            indices.push_back(tp->lmIndex);
        }else if(tp->type == TIMING_POINT_SIGNAL_OUT){
            net = oit->getNet();
        }
    }
    //Get the output LMIndex
    oaIter<oaOccInstTerm> netIter(net->getInstTerms()); 
    while(oaOccInstTerm *oit = netIter.getNext()){
        TPoint *tp = TPoint::get(oit);
        assert(tp);
        if(tp->type == TIMING_POINT_SIGNAL_IN){
            oaOccInst *inst = oit->getInst();
            CellData *cell = CellData::get(inst->getModInst());
            assert(cell);
            if(cell->master->isSequential) continue;

            toPO = false;
            indices.push_back(-1*tp->lmIndex);
        }
    }
#if defined(LMPO)
    //Get PO LMIndex
    oaIter<oaOccTerm> netTermIter(net->getTerms()); 
    while(oaOccTerm *ot = netTermIter.getNext()){
        TPoint *tp = TPoint::get(ot);
        assert(tp);
        if(tp->type == TIMING_POINT_PO){
            indices.push_back(-1*tp->lmIndex);
        }
    }
#endif
    if(toPO) indices.clear();
    return indices;
}
/*---------------------------------------------------------*/
double
LR::getDelaySum(){
    double s = 0.0;
    oaOccurrence *occ = design->getTopOccurrence();
    oaIter<oaOccInstTerm> oitIter(occ->getInstTerms());
    while(oaOccInstTerm *oit = oitIter.getNext()){ 
        TPoint *tp = TPoint::get(oit);
        assert(tp);
        if(tp->type == TIMING_POINT_SIGNAL_IN){
            oaOccInst *inst = oit->getInst();
            CellData *cell = CellData::get(inst->getModInst());
            assert(cell);
            if(cell->master->isSequential) continue;
            //std::cout << Util::getCellDelay(oit) << std::endl;
            s += tp->lm*Util::getCellDelay(oit);
        }
    }
    return s;
}
/*---------------------------------------------------------*/
void 
LR::saveSizing(map<oaModInst*,oaString> &m){
    if(!m.empty()) m.clear();
    oaModule *mod = design->getTopModule();
    oaIter<oaModInst> mIter(mod->getInsts());
    while(oaModInst *mInst = mIter.getNext()){
        CellData *c = CellData::get(mInst);
        assert(c);
        m[mInst] = c->master->name; 
    }
}
/*---------------------------------------------------------*/
void 
LR::reloadSizing(map<oaModInst*,oaString> &m){
    map<oaModInst*,oaString>::iterator it;
    for(it=m.begin(); it!=m.end(); ++it){
        CellData *c = CellData::get(it->first);
        assert(c);
        if(c->master->name == it->second) continue;
        c->swap(it->second);
        c->commitSwap();
    }
    timing->clearDesignTimingData();
    timing->updateAll();
}
/*---------------------------------------------------------*/
void
LR::printLMAssignment(){
    //Check LM assignment
    oaOccurrence *occ = design->getTopOccurrence();
    oaIter<oaOccInstTerm> oitIter(occ->getInstTerms());
    while(oaOccInstTerm *oit = oitIter.getNext()){ 
        TPoint *tp = TPoint::get(oit);
        assert(tp);
        if(tp->type == TIMING_POINT_SIGNAL_IN){
            oaOccInst *inst = oit->getInst();
            CellData *cell = CellData::get(inst->getModInst());
            assert(cell);
            if(cell->master->isSequential) continue;

            std::cout << Util::getBlockName(oit) 
                << ":" << tp->lmIndex << std::endl;
        }
    }
    /*oaIter<oaOccTerm> otIter(occ->getTerms());
    while(oaOccTerm *ot = otIter.getNext()){ 
        TPoint *tp = TPoint::get(ot);
        assert(tp);
        if(tp->type == TIMING_POINT_PO){
            if(timing->getArr(ot) == DBL_MAX) continue;
            std::cout << Util::getBlockName(ot) 
                << ":" << tp->lmIndex << std::endl;
        }
    }*/
    /*for(int i=0; i<G.size(); i++){
        vector<int> indx = getIndices(G[i]);
        CellData *cell = CellData::get(G[i]);
        std::cout << cell->instName << ":";
        for(int j=0; j<indx.size(); j++){       
            std::cout << " " << abs(indx[j]);
        }
        std::cout << std::endl;
    }*/
}
/*---------------------------------------------------------*/
void
LR::printLMValues(){
    std::cout << "Print LM Values ...:" << std::endl;
    oaOccurrence *occ = design->getTopOccurrence();
    oaIter<oaOccInstTerm> oitIter(occ->getInstTerms());
    while(oaOccInstTerm *oit = oitIter.getNext()){ 
        TPoint *tp = TPoint::get(oit);
        assert(tp);
        if(tp->type == TIMING_POINT_SIGNAL_IN){
            oaOccInst *inst = oit->getInst();
            CellData *cell = CellData::get(inst->getModInst());
            assert(cell);
            if(cell->master->isSequential) continue;
            std::cout << Util::getBlockName(oit) 
                << "(" << tp->lmIndex << "):" << tp->lm << " : " << tp->lm_old << std::endl; 
                //<< " - " << fabs((tp->lm - tp->lm_old)/tp->lm) 
                //<< " : " << fabs((tp->lm - tp->lm_old)/tp->lm_old) << std::endl;

        }
    }
    /*oaIter<oaOccTerm> otIter(occ->getTerms());
    while(oaOccTerm *ot = otIter.getNext()){ 
        TPoint *tp = TPoint::get(ot);
        assert(tp);
        if(tp->type == TIMING_POINT_PO){
            if(timing->getArr(ot) == DBL_MAX) continue;
            std::cout << Util::getBlockName(ot) 
                << "(" << tp->lmIndex << "):" << tp->lm << " : " << tp->lm_old << std::endl; 
                //<< " - " << fabs((tp->lm - tp->lm_old)/tp->lm) 
                //<< " : " << fabs((tp->lm - tp->lm_old)/tp->lm_old) << std::endl;
        }
    }*/
}
/*---------------------------------------------------------*/
void
LR::checkLMValues(){
    for(int i=0; i<G.size(); i++){
        CellData *cell = CellData::get(G[i]);
        assert(cell);
        double sum = checkSum(G[i]);
        std::cout << cell->instName << ":" << sum; 
        if(fabs(sum) > 1e-6) std::cout << " error ";
        std::cout << std::endl;
    }
}
/*---------------------------------------------------------*/
double
LR::checkSum(oaModInst *inst){
    oaOccNet *net;
    bool toPO = true;
    oaOccInst *occInst = DesignTool::getOccInst(design, inst);
    double sum = 0.0;
    //Add the input LMValues;
    oaIter<oaOccInstTerm> oitIter(occInst->getInstTerms()); 
    while(oaOccInstTerm *oit = oitIter.getNext()){
        TPoint *tp = TPoint::get(oit);
        assert(tp);
        if(tp->type == TIMING_POINT_SIGNAL_IN){
            sum += tp->lm;
        }else if(tp->type == TIMING_POINT_SIGNAL_OUT){
            net = oit->getNet();
        }
    }
    //Get the output LMIndex
    oaIter<oaOccInstTerm> netIter(net->getInstTerms()); 
    while(oaOccInstTerm *oit = netIter.getNext()){
        TPoint *tp = TPoint::get(oit);
        assert(tp);
        if(tp->type == TIMING_POINT_SIGNAL_IN){
            oaOccInst *inst = oit->getInst();
            CellData *cell = CellData::get(inst->getModInst());
            assert(cell);
            if(cell->master->isSequential) continue;

            toPO = false;
            sum -= tp->lm;
        }
    }
    oaIter<oaOccTerm> netTermIter(net->getTerms()); 
    while(oaOccTerm *ot = netTermIter.getNext()){
        TPoint *tp = TPoint::get(ot);
        assert(tp);
        if(tp->type == TIMING_POINT_PO){
            sum -= tp->lm;
        }
    }
    if(toPO) return 0;
    return sum;
}
/*---------------------------------------------------------*/
void 
LR::runAssign(){
std::cout << "Running old assignment method" << std::endl;
    bool flag;
    do{
        flag = false;
        for(int i=0; i<G.size(); i++){
            double dSum = DBL_MAX;
            double pSum = DBL_MAX;
            double minSum = getDelaySum()+(ut.getTotalLeakPower(design)*factor);
            CellData *cell = CellData::get(G[i]);
            assert(cell);
            //std::cout << "minSum:" << minSum << std::endl;
            //std::cout << "Inst:" << cell->instName <<
                //" - " << cell->master->name << std::endl;
            vector<oaString> sizes = cell->getOtherSizes();
            //std::cout << " Sizes:" << sizes.size() << std::endl;
            for(int j=0; j<sizes.size(); j++){
                //std::cout << " ->" << sizes[j] << std::endl;
                cell->swap(sizes[j]);
                ut.invalidateFanIOTiming(design,timing,cell->inst);
                timing->updateAll();

                dSum = getDelaySum();
                pSum = ut.getTotalLeakPower(design)*factor;
                //std::cout << " " << dSum << "/" << pSum << std::endl;
                if( (dSum+pSum) >= minSum){
                    cell->reverseSwap();
                    ut.invalidateFanIOTiming(design,timing,cell->inst);
                    timing->updateAll();
                    //std::cout << " Not TAKEN" << std::endl;
                }else{
                    minSum = dSum+pSum;
                    cell->commitSwap();
                    flag = true;
                    //std::cout << " TAKEN" << std::endl;
                }
            }
        }
    }while(flag);
}
/*---------------------------------------------------------*/
// WObj = sum(power) + sum(lm(ik)*(dik)) + sum(lm*(Tout-Ttarget));
double
LR::getLRObjective(){
   
    double lrObj = 0.0;
    oaOccurrence *occ = design->getTopOccurrence();
    oaIter<oaOccInstTerm> instTermIter = occ->getInstTerms();
    while(oaOccInstTerm *iTerm = instTermIter.getNext()){
        TPoint *tp = TPoint::get(iTerm);
        assert(tp);
        if(tp->type == TIMING_POINT_SIGNAL_IN){
            oaOccInst *inst = iTerm->getInst();
            CellData *cell = CellData::get(inst->getModInst());
            assert(cell);
            if(cell->master->isSequential) continue;
            double delay = Util::getCellDelay(iTerm);
            lrObj += tp->lm*(delay+tp->netDelay);
        }
    }
    /*oaIter<oaOccInst> occIter(occ->getInsts());
    while(oaOccInst *occInst = occIter.getNext()){ 
        double maxArr = getMaxInArr(occInst);
        oaIter<oaOccInstTerm> oitIter(occInst->getInstTerms());
        while(oaOccInstTerm *oit = oitIter.getNext()){ 
            TPoint *tp = TPoint::get(oit);
            assert(tp);
            if(tp->type == TIMING_POINT_SIGNAL_IN){
                double arr = timing->getArr(oit);
                lrObj += tp->lm*(maxArr - arr);
            }
        }
    }
    oaIter<oaOccTerm> otIter(occ->getTerms());
    while(oaOccTerm *ot = otIter.getNext()){ 
        TPoint *tp = TPoint::get(ot);
        assert(tp);
        if(tp->type == TIMING_POINT_PO){
            if(timing->getArr(ot) == DBL_MAX) continue;
            double dArr = timing->getArr(ot) - timing->getPeriod();
            //std::cout << Util::getBlockName(ot) << "- dArr:" << dArr << std::endl;
            lrObj += tp->lm*(dArr);
        }
    }*/
    Util ut;
    lrObj += ut.getTotalLeakPower(design)*factor;
    return lrObj;
}
/*---------------------------------------------------------*/
// WObj = sum(power) + sum(lm(ik)*(dik));
/*double
LR::getLRObjective(){
   
    double lrObj = 0.0;
    oaOccurrence *occ = design->getTopOccurrence();
    oaIter<oaOccInstTerm> instTermIter = occ->getInstTerms();
    while(oaOccInstTerm *iTerm = instTermIter.getNext()){
        TPoint *tp = TPoint::get(iTerm);
        assert(tp);
        if(tp->type == TIMING_POINT_SIGNAL_IN){
            double delay = Util::getCellDelay(iTerm);
            lrObj += tp->lm*(delay+tp->netDelay);
        }
    }
    Util ut;
    lrObj += ut.getTotalLeakPower(design)*factor;
    return lrObj;
}*/

/*---------------------------------------------------------*/
// DP Implementation
/*---------------------------------------------------------*/
void
LR::runDPAssign(){
#if defined(DEBUG)
    std::cout << "DPAssignment" << std::endl;
#endif
    //std::cout<<"Initial COUNTER =0 "<<endl;
    iterRefine = false;
    int refineCount = 0;
    double globalBestObj = DBL_MAX;
    bool outerObjImprove = false;
    //do{
    
        //std::cout << " DP: outer loop iteration:" << ++refineCount << std::endl;
        //double bestObj = globalBestObj;
        double bestObj = getLRObjective();
        double bestPower = ut.getTotalLeakPower(design);
        outerObjImprove = false;
        bool objImprove = false;
        map<oaModInst*,oaString> bestSizing;
        
        int refineTimeCount = 0;
        do{
            //std::cout << " Refine Timing Iteration:" << ++refineTimeCount << std::endl;
            ++refineTimeCount;
            objImprove = false;
            map<oaModInst*,oaString> currSizing;
            saveSizing(currSizing);

            // Initial Data Check (Backward Search)
            DpRelaxation();
            // Dp Assignment (Forward Search)
            DpRestoration();

            //assignDPSolutions;
            if(dpSolutions.empty()) std::cout << "No DP Solution" << std::endl;
            
            map<oaOccInst*, oaString>::iterator iter;
            for(iter=dpSolutions.begin(); iter!=dpSolutions.end(); iter++){
                CellData *cell = CellData::get(iter->first->getModInst());
                assert(cell);
                oaString newSize = iter->second;
                if(newSize == cell->master->name) continue;

                cell->swap(newSize);
                cell->commitSwap();
            }
            timing->clearDesignTimingData();
            timing->updateAll();

            Util ut;
            double currTiming = timing->getWorstArr();
            double currPower = ut.getTotalLeakPower(design);
            double obj = getLRObjective();
            /*std::cout << " DP-wobj:" << alpha1 
                << ":" << obj << ":" << currPower  
                << "/bestObj:" << bestObj << ":" << bestPower << std::endl;
                */
            if(obj < bestObj){
                //std::cout << " SavingBest-obj:" << bestObj << "/" << obj << " currTiming:" << currTiming << " power:" << currPower << std::endl;
                bestObj = obj;
                bestPower = currPower; 
                objImprove = true;
                saveSizing(bestSizing);
                if(obj < globalBestObj){
                    globalBestObj = obj;
                    outerObjImprove = true;
                }
            }else{
                if(!bestSizing.empty()){
                    reloadSizing(bestSizing);
                } else
                    reloadSizing(currSizing);
            }

            //ERASE OF DPDATA*
            oaOccurrence *occ = design->getTopOccurrence();
            oaIter<oaOccInstTerm> itIter = occ->getInstTerms();
            while(oaOccInstTerm *i = itIter.getNext()){
                TPoint *tp = TPoint::get(i);
                tp->lrDataVec.clear();
            }
        }while(objImprove && refineTimeCount < MAX_INNER_ITERATION);
        //iterRefine = true;
        
    //}while(outerObjImprove);
}
/*---------------------------------------------------------*/
void
LR::DpRelaxation(){
#if defined(DEBUG)
    std::cout << "DpRelaxation()" << std::endl;
#endif
    vector<oaModInst*> revTop = DesignTool::getReverseTopological(design,timing); 
    //for each gate in reverse topological order
    for(int i=0; i<revTop.size(); i++){
        //std::cout << Util::getInstName(revTop[i]) << std::endl;
        CellData *cell = CellData::get(revTop[i]);
        assert(cell);

        oaOccInstTerm *outTerm = DesignTool::getOccOutInstTerm(cell->occInst);
        mergeFanoutSolutions(outTerm);

        vector<oaString> sizesData = cell->getAllSizes(); 
        /*std::cout << cell->instName << "(" << cell->master->name << "):" 
            << sizesData.size() << std::endl;*/
        oaIter<oaOccInstTerm> itTerm = cell->occInst->getInstTerms();
        while(oaOccInstTerm *iTerm = itTerm.getNext()){
            TPoint *currPin = TPoint::get(iTerm);
            assert(currPin);
            if(currPin->type == TIMING_POINT_SIGNAL_IN){
                for(int j=0; j<sizesData.size(); j++){   //for each size options
                    //std::cout << " " << Util::getBlockName(iTerm) << "(" << sizesData[j] 
                        //<< "):" << std::endl;
                    if(iterRefine && cell->isMultiFanin){
                        oaString solution = dpSolutions[iTerm->getInst()]; 
                        if(solution != sizesData[j]) continue;
                    }
                    lrData lr;
                    lr.cellSize = sizesData[j];
                    lr.inCap = Util::getCellInputCap(iTerm,sizesData[j],timing);
                    lr.fObj = getFanoutsMinObjSum(iTerm,sizesData[j]); 
                    currPin->lrDataVec.push_back(lr);
                }
            }
        }
    }
    //DEBUG: Print Circuit Graph Solutions so far
#if defined(DEBUG)
    std::cout << "Exit DpRelaxation()" << std::endl;
#endif
}
/*---------------------------------------------------------*/
//#define verifyRelaxation
double
LR::getFanoutsMinObjSum(oaOccInstTerm *in, oaString currSize){
#if defined(DEBUG)
    std::cout << "getFanoutsMinObjSum("
        << Util::getBlockName(in) << "," << currSize << ")" << std::endl;
#endif
    TPoint *inPin = TPoint::get(in);
    assert(inPin);
    oaOccInstTerm *outTerm = DesignTool::getOccOutInstTerm(in->getInst());
    TPoint *outPin = TPoint::get(outTerm);
    assert(outPin);
    outPin->netDelay = Util::getMaxNetDelay(outTerm);
   
    //For each fanouts
    if(outPin->lrDataVec.empty()){ 
        double load = outPin->netLoad; 
        double wDelay = inPin->lm*
            (Util::getCellDelayEstimate(in,currSize,load,timing)+outPin->netDelay);
        double obj = wDelay + (Util::getCellLeakagePower(currSize)*factor);
        return obj;
    }
    double minObj = DBL_MAX;
    for(int i=0; i<outPin->lrDataVec.size(); i++){ 
        double load = outPin->lrDataVec[i].inCap + outPin->netLoad; 
        double wDelay = inPin->lm*
            (Util::getCellDelayEstimate(in,currSize,load,timing)+outPin->netDelay);
        double obj = outPin->lrDataVec[i].fObj + wDelay 
            + (Util::getCellLeakagePower(currSize)*factor);
#if defined(verifyRelaxation)
        std::cout << " fObjSum:" << obj
            << "=" << outPin->lrDataVec[i].fObj << "+" 
            << inPin->lm*Util::getCellDelayEstimate(in,currSize,load,timing) 
            << "+" <<  Util::getCellLeakagePower(currSize)*factor << "(" << load << ")" 
            << std::endl;
#endif
        if(obj < minObj){
            minObj = obj;
            outPin->lrDataIndex = i;
        }
    }
    return minObj;
}
/*---------------------------------------------------------*/
void 
LR::mergeFanoutSolutions(oaOccInstTerm *outTerm){
#if defined(DEBUG)
    std::cout << "mergeFanoutSolutions(" << Util::getBlockName(outTerm) << ")" << std::endl;
#endif
    //Need to take care FF case
    TPoint *outPin = TPoint::get(outTerm);
    assert(outPin);
    outPin->lrDataVec.clear();
    oaOccNet *net = outTerm->getNet();
    oaIter<oaOccInstTerm> itIter = net->getInstTerms();
    while(oaOccInstTerm *instTerm = itIter.getNext()){
        TPoint *otherTP = TPoint::get(instTerm);
        assert(otherTP);
        if(otherTP->type == TIMING_POINT_SIGNAL_IN){
            CellData *cell = CellData::get(instTerm->getInst()->getModInst());
            assert(cell);
            if(cell->master->isSequential){
                lrData ld;
                ld.inCap = Util::getCellInputCap(instTerm,cell->master->name,timing);
                ld.fObj = 0.0;
                ld.fanoutMap.clear();
                outPin->lrDataVec.push_back(ld);
                continue;
            }
            if(outPin->lrDataVec.empty()){
                /*map<oaOccInstTerm*,oaString> sizeMap;
                for(int i=0; i<otherTP->lrDataVec.size(); i++){
                    otherTP->lrDataVec[i].fanoutMap[instTerm] = otherTP->lrDataVec[i].cellSize;
                }*/
                copyVecData(otherTP->lrDataVec,outPin->lrDataVec);
            }else{
                //std::cout << "merge:" << Util::getBlockName(instTerm) << std::endl;
                vector<lrData> tmpVec;
                for(int i=0; i<outPin->lrDataVec.size(); i++){
                    for(int j=0; j<otherTP->lrDataVec.size(); j++){
                        lrData ld;
                        /*ld.fanoutMap = outPin->lrDataVec[i].fanoutMap;
                        ld.fanoutMap[instTerm] = otherTP->lrDataVec[j].cellSize;*/
                        ld.inCap = outPin->lrDataVec[i].inCap + otherTP->lrDataVec[j].inCap;
                        ld.fObj = outPin->lrDataVec[i].fObj + otherTP->lrDataVec[j].fObj;
                        tmpVec.push_back(ld);
                    }
                }
                outPin->lrDataVec.clear();
                copyVecData(tmpVec,outPin->lrDataVec);
                /*std::cout << " Check FanoutMap copy " << std::endl;
                for(int i=0; i<outPin->lrDataVec.size(); i++){
                    map<oaOccInstTerm*,oaString>::iterator iter;
                    for(iter=outPin->lrDataVec[i].fanoutMap.begin();
                            iter!=outPin->lrDataVec[i].fanoutMap.end(); ++iter){
                        std::cout << " " << Util::getBlockName(iter->first)
                            << "(" << iter->second << ")"; 
                    }
                    std::cout << std::endl;
                }*/
            }
            pruneSolutions(outPin->lrDataVec);
        }
    }
}
/*---------------------------------------------------------*/
void
LR::copyVecData(vector<lrData> &src, vector<lrData> &dst){
#if defined(DEBUG) 
    std::cout << "copyVecData" << std::endl;
#endif
    for(int i=0; i<src.size(); i++){
        lrData d;
        d.cellSize = src[i].cellSize;
        d.inCap = src[i].inCap;
        d.fObj = src[i].fObj;
        //d.fanoutMap = src[i].fanoutMap;
        dst.push_back(d);
    }
}
/*---------------------------------------------------------*/
void 
LR::pruneSolutions(vector<lrData> &d){
#if defined(DEBUG)
    std::cout << "pruneSolutions()" << std::endl;
#endif
    //DEBUG: Check pruned solutions
    /*std::cout << " *DEBUG - List Size " << d.size() << ":" << std::endl;
    for(int i=0; i<d.size(); i++){
        std::cout << d[i].inCap << " " << d[i].fObj << std::endl;
    }*/
    if(d.empty()) return;
    vector<lrData>::iterator curr, next;
    curr = d.begin();
    next = curr+1;
    while(curr != d.end()){
        while(next != d.end()){
            if(curr->inCap >= next->inCap){ 
                if(curr->fObj >= next->fObj){
                    curr = d.erase(curr);
                    if(curr != d.end()) 
                        next = curr+1;
                    continue;
                }
            }else{
                if(curr->fObj <= next->fObj){
                    next = d.erase(next);
                    continue;
                }
            }
            next++;
        }
        curr++;
        if(curr != d.end()) 
                next = curr+1;
    }
    //DEBUG: Check pruned solutions
    /*std::cout << " * After Pruning - List Size " << d.size() << ":" << std::endl;
    for(int i=0; i<d.size(); i++){
        std::cout << d[i].inCap << " " << d[i].fObj << std::endl;
    }*/
}
/*---------------------------------------------------------*/
void
LR::DpRestoration(){
#if defined(DEBUG1)
    std::cout << "DpRestoration()" << std::endl;
#endif
    //printDPGraph();
    dpSolutions.clear();
    //for each gate in topological order
    for(int i=0; i<G.size(); i++){
        //std::cout << Util::getInstName(G[i]) << std::endl;
        CellData *cell = CellData::get(G[i]);
        assert(cell);
        oaString solution = getSolution(cell);

        //save solution
        dpSolutions[cell->occInst] = solution;

        //save arr and forward solution
        oaOccInstTerm *outTerm = DesignTool::getOccOutInstTerm(cell->occInst);
        TPoint *outPin = TPoint::get(outTerm);
        assert(outPin);
        outPin->lrSolution = solution;
    }
    //Print Solutions
    //std::cout << "Solutions:" << std::endl;
    map<oaOccInst*,oaString>::iterator it;
    for(it=dpSolutions.begin(); it!=dpSolutions.end(); ++it){
        CellData *cell = CellData::get(it->first->getModInst());
        assert(cell);
        if (cell->master->name != it->second) {
            //std::cout << cell->instName << " Changed (curr:" << cell->master->name << ")" 
            //<< "->" << it->second << std::endl;
        }
    }
#if defined(DEBUG1)
    std::cout << "Exit DpRestoration()" << std::endl;
#endif
}
/*---------------------------------------------------------*/
oaString
LR::getSolution(CellData *c){
#if defined(DEBUG1)
    std::cout << "getSolution(" << c->instName << ")" << std::endl;
#endif
    oaString bestSize = c->master->name;
    double minObj = DBL_MAX;
    vector<oaString> sizesData = c->getAllSizes(); 
    //cout << "N Sizes :" << sizesData.size() << endl;
    //For each size option of current cell
    for(int i=0; i<sizesData.size(); i++){
        oaString currSize = sizesData[i];
        double wObj = 0.0;
        double maxfObj = -DBL_MAX;

        //For each input pin
        //std::cout << "for each input pin" << std::endl;
        oaIter<oaOccInstTerm> itIter = c->occInst->getInstTerms();
        while(oaOccInstTerm *inTerm = itIter.getNext()){
            TPoint *inPin = TPoint::get(inTerm);
            assert(inPin);
            if(inPin->type == TIMING_POINT_SIGNAL_IN){
                //std::cout << " " << Util::getBlockName(inTerm) ;

                //If input arc is coming from an output of another cell
                oaIter<oaOccInstTerm> iter = inTerm->getNet()->getInstTerms();
                while(oaOccInstTerm *j = iter.getNext()){
                    TPoint *otherTP = TPoint::get(j);
                    assert(otherTP);
                    if(otherTP->type == TIMING_POINT_SIGNAL_OUT){       
                        double obj = getMaxWeightedFaninObj(j,inTerm,currSize);
                        //std::cout << "\n   from:" << Util::getBlockName(j)
                            //<< " - " << obj << std::endl;
                        wObj += obj;    
                    }
                }
                for(int j=0; j<inPin->lrDataVec.size(); j++){
                    if(inPin->lrDataVec[j].cellSize == currSize){
                        if(inPin->lrDataVec[j].fObj > maxfObj)
                            maxfObj = inPin->lrDataVec[j].fObj;
                    }
                }
            }
        }
        wObj += maxfObj;
        //std::cout << " wObj:" << wObj << std::endl; 
        if(wObj < minObj){
            minObj = wObj;
            bestSize = currSize;
            //std::cout << " bestSize:" << bestSize << std::endl;
        }
    }
    return bestSize;
}
/*---------------------------------------------------------*/
double
LR::getMaxWeightedFaninObj(oaOccInstTerm *out, oaOccInstTerm *other, oaString otherSize){
#if defined(DEBUG1)
    std::cout << "getMaxWeightedFaninObj(" 
        << Util::getBlockName(out) << "," << Util::getBlockName(other) << "," 
        << otherSize << ")"  << std::endl;
#endif
    double maxWDelay = -DBL_MAX;
    TPoint *otherTP = TPoint::get(other);
    assert(otherTP);
    TPoint *outPin = TPoint::get(out);
    assert(outPin);

    CellData *cell = CellData::get(other->getInst()->getModInst());
    assert(cell);
    double currLoad = Util::getCellInputCap(other,cell->master->name,timing);
    double load = outPin->load - currLoad 
        + Util::getCellInputCap(other,otherSize,timing);

    //Handle FF as fanin case
    oaString solution;
    CellData *faninCell = CellData::get(out->getInst()->getModInst());
    assert(faninCell);
    if(faninCell->master->isSequential) solution = faninCell->master->name;
    else solution = outPin->lrSolution;

    oaIter<oaOccInstTerm> instTermIter = out->getInst()->getInstTerms();
    while(oaOccInstTerm *i = instTermIter.getNext()){
        TPoint *inPin = TPoint::get(i);
        assert(inPin);
        if(inPin->type == TIMING_POINT_SIGNAL_IN){      
            double delay = 
                (Util::getCellDelayEstimate(i,solution,load,timing)+otherTP->netDelay);
            double wDelay = inPin->lm*delay;
            //std::cout << "   based on:" << Util::getBlockName(i) << " - " << wDelay;
            if(wDelay > maxWDelay) maxWDelay = wDelay;
        }
    }
    maxWDelay += (Util::getCellLeakagePower(solution)*factor);
    //std::cout << "\n   +" << (Util::getCellLeakagePower(solution)*factor);
    return maxWDelay;
}
/*---------------------------------------------------------*/
void
LR::printDPGraph(){
    for(int i=0; i<G.size(); i++){
        std::cout << Util::getInstName(G[i]) << std::endl;
        CellData *cell = CellData::get(G[i]);
        assert(cell);
        oaIter<oaOccInstTerm> itTerm = cell->occInst->getInstTerms();
        while(oaOccInstTerm *iTerm = itTerm.getNext()){
            TPoint *currPin = TPoint::get(iTerm);
            assert(currPin);
            if(currPin->type == TIMING_POINT_SIGNAL_IN){
                std::cout << " " << Util::getBlockName(iTerm)
                    << "(curr:" << cell->master->name << ")" << std::endl;
                for(int j=0; j<currPin->lrDataVec.size(); j++){
                    std::cout << "  " << currPin->lrDataVec[j].cellSize
                        << " fObj:" << currPin->lrDataVec[j].fObj << std::endl;
                }
            }
        }
    }
}
/*---------------------------------------------------------*/
}//namespace
        
// TEMP
