/************************************************************
* oagTimerElmoreWireModel.h
* Perform Elmore delay analysis on interconnects
* by: Santiago Mok <smok@ee.ucla.edu>
* advised by: Prof. Puneet Gupta
* Electrical Engr Dept, UCLA
*
************************************************************/
#if !defined(oagTimerElmoreWireModel_P)
#define oagTimerElmoreWireModel_P

#include "oaDesignDB.h"
#include "oagTimerWireModel.h"
//#include "oagTimerTPointMaster.h"
#include "oagTimerLibParserInt.h"
#include <map>
#include <vector>

namespace oagTimer {

using namespace oa;
using namespace std;

struct _inputTermsData{
    oaFloat res;
    oaFloat elmoreValue;
    vector<oaNode*> topNode;
};
struct _pathData{
    oaFloat totalRes;
    oaFloat totalDownCap;
    oaFloat totalElmoreValue;
};
struct nodeData{
    oaNode *current;
    vector<oaFloat> _res;
    vector<oaNode*> _nextNode;
    /*
    oaFloat left_res;
    oaFloat right_res;
    oaNode *left;
    oaNode *right;
    */
};

typedef map<oaNode*,double> inputPinMap; 
typedef map<oaNode*,_pathData> _PathMap;
typedef map<oaNode*,_inputTermsData> _inputPinMap;
typedef map<oaNode*,oaTerm*> termMap;
typedef map<oaNode*,oaInstTerm*> instTermMap;
typedef map<oaTerm*,DelayType> _outputMap;
typedef map<oaInstTerm*,DelayType> _inputMap;

class ElmoreWireModel : public WireModel{
    public:
	//Constructor
	ElmoreWireModel();
	
	//Destructor
	~ElmoreWireModel();

private:
	//Check if the term or instTerm is a Pin Node
	bool isPinNode(oaNode *node);
	
	/*!
	  Get the gate input capacitance for elmore delay calculations
	  \param n The node of the input pin
	  \return the gate input pin capacitance
	*/
	double getInputPinCap(oaNode* n);
	/*!
	  If node branches or Input Pin follow down, save the
	  current path values
	  \param _node current branching node
	  \param _res current path resistor
	  \param _elm current path elmore value
	*/
	void storeTopPath(oaNode *currentNode, oaFloat _res, oaFloat _elm);

	/*!
	  Save the current input pin elmore value
	  \param currNode current input pin node
	  \param _res current path resistor
	  \param _elm current path elmore value
	*/
	void storeInpPin(oaNode *currNode, oaFloat _res, oaFloat _elm);
	/*!
	  Update and sum Top Path existing elmore Delay Value
	  \param _currCap Current Path capacitor value to update the elmore value
	  of the top path
	*/
	void updateTopPathElmoreValue(oaFloat _currCap);

	/*! Check if it has reached all downstream nodes */
	bool isTopPathNode(oaNode *node);
	
	//Clear current path resistor value and current path elmore value 
	void clearCurrentPath();
	
	//clear the summed path capacitor value
	void clearPathCap();

	//Clear all data;
	void clearAll();
	
	/*!
	  Search for left and right groundedNode and store it in the structure
	  \param node struct that hold current node data 
	  \param prevsNode pointer to previous node
	*/
	//void findNextNode(oaNode *currNode,oaNode *prevsNode, elmElmts &elmts);
	void findNext(nodeData &node, oaNode *previousNode);

	/*!
	  Recursive function that start from the input of a Term or output of an instTerm, tranverses the wire 
	  to get the parasitics(resistance and capacitance) value for elmore delay calculations, and end 
	  until all the wire terminals are reached (output of a Term or input of an instTerm)
	  \param parentNode pointer to current node
	  \param prevs pointer to previous node
	*/
	void oaTraversePath(oaNode* parentNode, oaNode *previousNode);
    
	/*! Return the corresponding  output term to the given node 
	  \param n node of the  output term 
	*/
	oaTerm *getTerm(oaNode *n); 


	/*! Return the corresponding  input instTerm to the given node 
	  \param n node of the  input instTerm 
	*/
	oaInstTerm *getInstTerm(oaNode *n);
    
    public:
	/*!
	  Calculate and return effective capacitance
	*/
    double ElmoreWireModel::getWireCapEff(double netLoad, double netRes, DelayType Td, DelayType Tx);
	
	/*!
	  Get total resistor values in the wire network
	*/
	double ElmoreWireModel::getWireRes(oaNet *net);

	/*!
	  Set the unit factor for cap_unit and time unit
	  according to the specification in the .lib library
	  \param time The time unit 
	  \param cap The cap unit
	*/
	void setUnit(const char* time,const char* cap);
	
	DelayType getWireCap(oaNet *net);
	DelayType getWireDelay(oaNet *net);
	DelayType getWireDelay(oaNet *net, _outputMap &outputPin, _inputMap &inputPin);
//Variables
    public:
	oaNode *parentNode;
    private:
	oaFloat
	    _currPathRes, _currNodeCap, _currPathCap, _retPathCap, _currElmorePath;
	oaAnalysisPoint *_currentAP;
	vector<oaNode*> _currNodeVector;
	inputPinMap inputPinCap;
	_PathMap topPath;
	_PathMap luMap;
	_inputPinMap inputTerm;
	termMap _terms;
	instTermMap _instTerms;	
	bool branch;
	bool pinBranch; 
	double time_factor, cap_factor;
	
};

}

#endif
