/************************************************************
* 
* File: oagTimerDuetSensitivity.cpp
* Author: Santiago Mok (smok@ee.ucla.edu)
* Created: 09-02-2009
* Last Modified: Thu 23 Jun 2011 10:38:48 PM PDT
*
************************************************************/

#include "oaDesignDB.h"
#include "oagTimerTimer.h"
#include "oagTimerDuetSensitivity.h"
#include "oagTimerLibParserInt.h"
#include "oagTimerReport.h"
#include "oagTimerSubTimer.h"
#include "string.h"
#include <time.h>
#include <assert.h>

#define MODE_READ 'r'

//#define DEBUG
namespace oagTimer{
/*---------------------------------------------------------*/
using namespace oa;
using namespace std;
/*---------------------------------------------------------*/
// Constructor
DuetSensitivity::DuetSensitivity(oaDesign *d,Timer *t){
	design = d;
	timing = t;
	debug_flag = false;
	flag = false;
	//capacity = 0;
}
// ~Destructor
DuetSensitivity::~DuetSensitivity(){}
/*---------------------------------------------------------*/
float
DuetSensitivity::getSensitivity(oaInst *old_inst, oaInst *new_inst) {
    double cellLeak_old = Util::getCellLeakagePower(old_inst);
    double cellLeak_new = Util::getCellLeakagePower(new_inst);
    return (cellLeak_new - cellLeak_old); 
}
/*---------------------------------------------------------*/
void
DuetSensitivity::enableDebug(){
#if defined(DEBUG)
    std::cout << "enable Debug" << std::endl;
#endif
    debug_flag = true;
}
/*---------------------------------------------------------*/
void
DuetSensitivity::initDesign(){
    if(!Design_list.empty()) Design_list.clear();
    oaOccurrence *occ = design->getTopOccurrence();
    assert(occ);
    oaIter<oaOccInst> occIter (occ->getInsts());
    while(oaOccInst *occInst = occIter.getNext()){
	Design_list.push_back(occInst->getModInst());
	toUpdMap[occInst] = 0;
	//++capacity;
    }
    //For statistics
    numOfRoots = numOfRootsWithgreaterFO4 
	= numOfRootsWithgreaterDepth10 = numOfRootsWithgreaterDepth20
	= maxDepth = maxFanout = 0;
}
/*---------------------------------------------------------*/
void
DuetSensitivity::mergeList(){
#if defined(DEBUG)
    std::cout << "mergeList()" << std::endl;
#endif
    /*list<SensType>::iterator iter;
    for(iter=sensList_temp.begin(); iter!=sensList_temp.end(); ++iter){
	sensList.push_back(*iter);
    }
    sensList_temp.clear();
    */
    if(!sensList.empty()) sensList.clear();
    bestList.sort(incr);
    regList.sort(decr);
    list<SensType>::iterator bestIter, regIter;
    for(bestIter=bestList.begin(); bestIter!=bestList.end(); ++bestIter){
	sensList.push_back(*bestIter);
    }
    for(regIter=regList.begin(); regIter!=regList.end(); ++regIter){
	sensList.push_back(*regIter);
    }
    bestList.clear(); regList.clear();
}
/*---------------------------------------------------------*/
void
DuetSensitivity::getSensData(CellData *c){
    vector<oaString> size_vec = c->getLessThanCurrentSize();
    if(size_vec.empty()) return;
    oaOccInst *occInst = DesignTool::getOccInst(design,c->inst);
    oaOccInstTerm *outInstTerm = DesignTool::getOccOutInstTerm(occInst);

    for(vector<oaString>::iterator it=size_vec.begin();
	    it!=size_vec.end(); ++it){
	oaModule* topCell = DesignTool::getCellTopMod(*it);
	assert(topCell);
	oaModTerm *outTerm;
	oaIter<oaModTerm> termIter = topCell->getTerms();
	while(oaModTerm *term = termIter.getNext()){
	    oaTermType type(term->getTermType());
	    if(type == oacOutputTermType){
		outTerm = term;
	    }
	}
	TPointMaster *new_tm = TPointMaster::get(outTerm);
	assert(new_tm);
	SensType sd;
	double dPower = (Util::getCellLeakagePower(topCell)
	    - c->master->leakage_power);
	if(dPower >= 0) continue;
	sd.modInst = c->inst;
	sd.occInst = c->occInst;
	sd.name = *it;
	sd.dPower = fabs(dPower);
	//sd.dDelay = Util::getDeltaDelay(design,timing,c,sd.name);
	SubTimer subTP(design,timing,c->occInst,sd.name);
	sd.dDelay = subTP.getDeltaDelay(); 
	
	if(sd.dDelay <= 0.0){ 
	    if(sd.dDelay == 0.0)
		sd.sens = 0.0; 
	    else
		sd.sens = sd.dPower/sd.dDelay;
	    //sensList.push_back(sd);
	    bestList.push_back(sd);
	}else{ //normal case
	    sd.sens = sd.dPower/sd.dDelay;
	    //sensList_temp.push_back(sd);
	    regList.push_back(sd);
	}
    }
}
/*---------------------------------------------------------*/
void
DuetSensitivity::createSensList(){
#if defined(DEBUG)
    std::cout << "CREATESENSLIST()" << std::endl;
#endif
    sensList.clear(); sensList_temp.clear();
    if(!bestList.empty()) bestList.clear(); 
    if(!regList.empty()) regList.clear();
    for(vector<oaModInst*>::iterator it = Design_list.begin();
	    it!=Design_list.end(); ++it){
	CellData *cell = CellData::get(*it);
	assert(cell);
	if(cell->master->isSequential) continue;
        cout << cell->master->name << " / " << cell->getDownSizeCount() << endl;
        if(cell->getDownSizeCount()){
	    getSensData(cell);
	}
	/*vector<oaString> lessSizes = cell->getLessThanCurrentSize();
	vector<oaString> greaterSizes = cell->getGreaterThanCurrentSize();
	std::cout << " Current:" << cell->instName << ":" << cell->master->name 
	    << " " << lessSizes.size() << " " << greaterSizes.size() << std::endl;
	std::cout << "  Less:" << std::endl;
	    for(int i=0; i<lessSizes.size(); ++i){
		std::cout << "    " << lessSizes[i] << std::endl;
	    }
	std::cout << "  Greater:" << std::endl;
	    for(int i=0; i<greaterSizes.size(); ++i){
		std::cout << "    " << greaterSizes[i] << std::endl;
	    }
	    */
    }
    //sensList.sort(incr);
    //sensList_temp.sort(decr);
    mergeList();
    resetUpdList();
    //printSensList();
}
/*---------------------------------------------------------*/
void
DuetSensitivity::resetUpdList(){

    map<oaOccInst*,int>::iterator iter;
    for(iter=toUpdMap.begin(); iter!=toUpdMap.end(); iter++){
	iter->second = 0;
    }
}
/*---------------------------------------------------------*/
void 
DuetSensitivity::markInst(oaOccInst *inst){
    //Get all imd fanin/fanout that need to update sensitivities
    //toUpdVec.push_back(inst);
    toUpdMap[inst] = 1;
    oaIter<oaOccInstTerm> itIter = inst->getInstTerms();
    while(oaOccInstTerm *currTerm = itIter.getNext()){
	TPoint *tp = TPoint::get(currTerm);
	assert(tp);
	oaOccNet *net = currTerm->getNet();
	oaIter<oaOccInstTerm> itIter2 = net->getInstTerms();
	if(tp->type == TIMING_POINT_SIGNAL_IN){
	    while(oaOccInstTerm *otherTerm = itIter2.getNext()){
		TPoint *otherTP = TPoint::get(otherTerm);
		assert(otherTP);
		if(otherTP->type == TIMING_POINT_SIGNAL_OUT){
		    toUpdMap[otherTerm->getInst()] = 1;
		}
	    }
	}else if(tp->type == TIMING_POINT_SIGNAL_OUT){
	    while(oaOccInstTerm *otherTerm = itIter2.getNext()){
		TPoint *otherTP = TPoint::get(otherTerm);
		assert(otherTP);
		if(otherTP->type == TIMING_POINT_SIGNAL_IN){
		    toUpdMap[otherTerm->getInst()] = 1;
		}
	    }
	}
    }
}
/*---------------------------------------------------------*/
void
DuetSensitivity::splitCurrList(){
    if(sensList.empty()) return;

    bestList.clear(); regList.clear();
    list<SensType>::iterator iter;
    for(iter=sensList.begin(); iter!=sensList.end(); ++iter){
	if(iter->sens <= 0) 
	    bestList.push_back(*iter);
	else
	    regList.push_back(*iter);
    }
}
/*---------------------------------------------------------*/
void
DuetSensitivity::updateSensList(){
#if defined(DEBUG)
    std::cout << "updateSensList ()" << std::endl;
#endif

    //markInst(inst);
    //Search the current sensitivity list and update it
    list<SensType>::iterator iter;
    for(iter=sensList.begin(); iter!=sensList.end();){
	if(needUpdate(iter->occInst)){	
	    SubTimer subTP(design,timing,iter->occInst,iter->name);
	    double dDelay = subTP.getDeltaDelay(); 
	    iter->dDelay = dDelay; 

	    if(dDelay <= 0.0){ 
		if(dDelay == 0.0)
		    iter->sens = 0.0; 
		else
		    iter->sens = iter->dPower/iter->dDelay;
	    }else{
		iter->sens = iter->dPower/iter->dDelay;
		sensList_temp.push_back(*iter);
		iter = sensList.erase(iter);
		continue;
	    }
	}
	iter++;
    }
    splitCurrList();
    /*sensList.sort(incr);
    sensList_temp.sort(decr);
    */
    mergeList();
    resetUpdList();
}
/*---------------------------------------------------------*/
bool
DuetSensitivity::needUpdate(oaOccInst* inst){
    map<oaOccInst*,int>::iterator iter;
    iter = toUpdMap.find(inst);
    if(iter != toUpdMap.end()){
	if(iter->second == 1) return true;
	return false;
    }
    return false;
}
/*---------------------------------------------------------*/
void 
DuetSensitivity::remove(oaModInst* currInst){
    list<SensType>::iterator iter = sensList.begin();
    while(iter != sensList.end()){
	if(iter->modInst == currInst){
	    iter=sensList.erase(iter);
	    continue;
	}
	++iter;
    }
}
/*---------------------------------------------------------*/
void
DuetSensitivity::run(){
#if defined(DEBUG)
    std::cout << "run()" << std::endl;
#endif
    oaNativeNS NS;
    time_t start; 
    time_t end;
    time (&start);

    Util ut;
    initDesign();
    bool any_move_taken = false;
    int trial = 0;
    double tcPeriod = timing->getPeriod();
    std::cout << "Initial Cell Leakage:" << Util::getTotalLeakPower(design) 
	<< " Target Constraint:" << tcPeriod << std::endl;
    int count = 0;
    do{
	++trial;
	double I = Util::getTotalLeakPower(design);
	createSensList();
	any_move_taken = false;
	while(!sensList.empty()){
	    SensType data = sensList.front();
	    sensList.pop_front();
	    CellData *cell = CellData::get(data.modInst);
	    assert(cell);
	    cell->swap(data.name);
	    ut.invalidateFanIOTiming(design,timing,cell->inst);
	    timing->updateAll();

	    DelayType worst_slack = timing->getWorstSlack();
	    if(worst_slack >= 0){
		remove(data.modInst);
		any_move_taken = true;
		cell->commitSwap();

		markInst(cell->occInst);
		updateSensList();
	    }else{
		cell->reverseSwap();
		ut.invalidateFanIOTiming(design,timing,cell->inst);
		timing->updateAll();
	    }
	}
	double F = Util::getTotalLeakPower(design);
	if(I-F > 0) std::cout << "Gain:" << I-F << std::endl;
	if(flag) break;
    }while(any_move_taken);

    cout << "Final CellLeakage:" << Util::getTotalLeakPower(design) << endl;
    time (&end);
    std::cout << "Runtime:" << difftime(end,start) << " (s)" << std::endl;
}
/*---------------------------------------------------------*/
/* Debug List */
/*---------------------------------------------------------*/
void
DuetSensitivity::printSensList(){
    list<SensType>::iterator iter; 
    for(iter=sensList.begin(); iter!=sensList.end(); ++iter){
	print(*iter);
    }
}
/*---------------------------------------------------------*/
void
DuetSensitivity::print(SensType s){
    CellData *cell = CellData::get(s.modInst);
    assert(cell);
    std::cout << " Cell:" << cell->instName << "("
	<< cell->master->name << "->" << s.name << ") :" 
	<< s.dPower << "/" << s.dDelay << " -- " << s.sens << std::endl;
}
/*---------------------------------------------------------*/
void
DuetSensitivity::printDuetList(sensitivityList list, duetType type,int pos){
    Util ut;
    int cnt = 0;
    std::cout << "Type - " << type << " Initial List:" << std::endl;
    sensitivityList::iterator iter; 
    for(iter=list.begin(); iter!=list.end(); ++iter){
	++cnt;
	oaModInst* currInst = iter->inst;
	std::cout << " Cell:" << Util::getInstName(currInst)
	    << " current:" << iter->currInst << " new:" << iter->newInst 
	    << " Delta_Power:" << iter->delta_power;
	if(type == DUET_POWER_DELAY){
	    std::cout << " Delta_Delay:" << iter->delta_delay << std::endl; 
	    std::cout << "  delay_new:" << iter->delay_new 
	    << " delay_current:" << iter->delay_current << std::endl;
	}else{
	    std::cout << "  slack_new:" << iter->slack_new
	    << " slack_current:" << iter->slack_current
	    << " Delta_Slack:" << iter->delta_slack << std::endl;
	}
	    std::cout << "  sensitivity:" << iter->delta << std::endl;
	if(cnt == pos) return;
    }
}
/*---------------------------------------------------------*/
}//namespace

/*---------------------------------------------------------*/
/*void
DuetSensitivity::printSensList(){
    list<SensType>::iterator iter;
    for(iter=sensList.begin(); iter!=sensList.end(); ++iter){
	SensType data = *iter;
	CellData *cell = CellData::get(data.modInst);
	assert(cell);
	std::cout << "* " << cell->instName << ":" << cell->master->name
	    << "->" << data.name << std::endl;
	std::cout << "   dPower:" << data.dPower << std::endl;
	std::cout << "   dDelay:" << data.dDelay << std::endl;
	std::cout << "   sens:" << data.sens << std::endl;
    }
}*/
