/// \file oagSswLogger.h

#if !defined(oagSswLogger_P)
#define oagSswLogger_P

#include <iostream>
#include <map>

/// \def oagSsw_LOG(logger, message)
/// \brief Logs a message if a logger is enabled, prepending a header
/// string.
///
/// No newline will be appended to the message.
/// The message can consist of multiple arguments:
/// \code OAGSSW_LOG(logger, "Number of objects: " << numObjs);
/// The message expression is never evaluated when the logger is disabled.
#define OAGSSW_LOG(logger, message)        \
  do {                                  \
    if ((logger)->isEnabled()) {        \
      (logger)->logHeader() << message; \
    }                                   \
  } while (0)

/// \brief Logs a message line if a logger is enabled, prepending a header
///
/// string and appending a newline.
///
/// The message can consist of multiple arguments:
/// \code OAGSSW_LOG(logger, "Number of objects: " << numObjs);
/// The message expression is never evaluated when the logger is disabled.
#define OAGSSW_LOGLN(logger, message)              \
  do {                                          \
    if ((logger)->isEnabled()) {                \
      (logger)->logHeader() << message;         \
      (logger)->getStream() << endl;            \
    }                                           \
  } while (0)

namespace oagSsw
{

/// \brief Output logger.
class Logger {
  public:
    /// \brief Returns a reference to the logger with the given name.
    ///
    /// Unless previously enabled, the logger returned is disabled.
    static Logger           *getInstance(const std::string &name);
    
    bool                    isEnabled()     { return enabled; }
    void                    enable()        { enabled = true; }
    void                    disable()       { enabled = false; }
    
    /// Sets the output stream for a logger.
    void                    setStream(std::ostream & out)   { this->stream = &out; }
    std::ostream            &getStream()                    { return *stream; }

    template<class T>
    Logger                  &operator<<(const T & t) { *stream << t; return *this; }
    /// Outputs the current time and logger name.
    Logger                  &logHeader();
    
  private:
                            Logger(const std::string & name)
                                : name(name),
                                  enabled(false),
                                  stream(&std::cout) {}

  // Prevent implicit copy constructor and assignment operator
                            Logger(Logger & other);
    Logger                  &operator=(Logger & other); 

    std::string             name;
    bool                    enabled;
    std::ostream            *stream;

    static std::map<std::string, Logger *> *instances;
};

}

#endif
