#include <iomanip>
#include <stdexcept>
#include <sstream>
#include <string>
#include "oagSswBitVector.h"

using namespace std;
using namespace oa;

namespace oagSsw
{

const oa::oaUInt4 BitVector::BITS_PER_WORD = 8 * sizeof(BitVector::Word);

oa::oaUInt4
BitVector::getWordIndex(oa::oaUInt4 bitIndex)
{ 
  return bitIndex / BITS_PER_WORD; 
}

// 0 is most sig. bit, so that comparison is lexicographic on bits
inline oa::oaUInt4
BitVector::getWordOffset(oa::oaUInt4 bitIndex)
{ 
  return BITS_PER_WORD - 1 - bitIndex % BITS_PER_WORD; 
}

oa::oaUInt4
BitVector::getNumWords(oa::oaUInt4 numBits)
{ 
  return numBits == 0 ? 0 : 1 + getWordIndex(numBits - 1); 
}

BitVector::BitVector()
{
    numBits = 0;
}

oa::oaUInt4
BitVector::size() const
{
    return numBits;
}

void
BitVector::setSize(oaUInt4 numBits)
{
    words.resize(getNumWords(numBits));
    this->numBits = numBits;
}

bool
BitVector::getBit(oa::oaUInt4 bitIndex) const
{
    Word word;

    word = words[getWordIndex(bitIndex)];
    return (word >> getWordOffset(bitIndex)) & 1;
}

void
BitVector::setBit(oa::oaUInt4 bitIndex,
                  bool value)
{ 
    Word word;
    
    word = words[getWordIndex(bitIndex)];
    if (value)
        word |= 1 << getWordOffset(bitIndex);
    else
        word &= ~(1 << getWordOffset(bitIndex));
    words[getWordIndex(bitIndex)] = word;
}

bool
BitVector::operator<(const BitVector &other) const
{
    if (size() != other.size()) {
        ostringstream oss;

        oss << "BitVector lengths do not match: "
            << size() << " and " << other.size();

        throw invalid_argument(oss.str());
    }

    for (oaUInt4 i = 0; i < words.size(); i++) {
        // Compare words from last to first instead of vice versa because the
        // differences are most likely to be at the end, in the most recently
        // added bits.
        oaUInt4 j       = words.size() - 1 - i;
        Word word       = words[j];
        Word otherWord  = other.words[j];

        if (word < otherWord) {
            return true;
        } else if (word > otherWord) {
            return false;
        }
    }
 
    return false;
}

bool
BitVector::operator==(const BitVector &other) const
{
    if (size() != other.size()) {
        ostringstream oss;

        oss << "BitVector lengths do not match: "
            << size() << " and " << other.size();

        throw invalid_argument(oss.str());
    }

    for (oaUInt4 i = 0; i < words.size(); i++) {
        // Compare words from last to first instead of vice versa because the
        // differences are most likely to be at the end, in the most recently
        // added bits.
        oaUInt4 j       = words.size() - 1 - i;

        if (words[j] != other.words[j])
            return false;
    }
 
    return true;
}

void
BitVector::clear()
{
    words.clear();
    numBits = 0;
}

void
BitVector::print(std::ostream &out)
{
    ios::fmtflags flags = out.flags();

    for (oaUInt4 i = 0; i < getNumWords(numBits); i++) {
        out << setfill('0') << setw(8) << setbase(16) << words[i];
    }
    out << endl;

    out.flags(flags);
}

}
