/*
Author: Kai-hui Chang <changkh@eecs.umich.edu>
*/

#include <iostream>
#include <string>
#include <set>
#include "oagResynSim.h"
#include "oagResynCktMan.h"
#include "oagResynUtil.h"

using namespace std;
using namespace oa;

// #define DEBUG

namespace oagResyn {

// *****************************************************************************
// Sim()
//
/// \brief Constructor.
//
// *****************************************************************************
Sim::Sim(oa::oaDesign *design) {
    setDesign(design);
    assert(design);
    block= design->getTopBlock();
    assert(block);
    this->design = design;
    myCkt= cktManager.designGetInternalCkt(design);
    myUserPattern= NULL;
}

// *****************************************************************************
// setDesign()
//
/// \brief Set the design for simulation
//
// *****************************************************************************
void Sim::setDesign(oa::oaDesign *design) {
    assert(design);
    block= design->getTopBlock();
    assert(block);
    this->design = design;
    myCkt= cktManager.designGetInternalCkt(design);
}

// *****************************************************************************
// getStates()
//
/// \brief Gets all state instances inside this design. 
// *****************************************************************************
void Sim::getStates(vector<oaInst*>& result) {
    result= myCkt->stateInsts;
}

// *****************************************************************************
// getAllNets()
//
/// \brief Gets all nets in the circuit (oaBitNet).
// *****************************************************************************
void Sim::getAllNets(vector<oaNet*>& result) {
    set<oaNet*>::iterator setIter;
    
    result.clear();
    for (setIter= myCkt->allBitNets.begin(); setIter != myCkt->allBitNets.end(); setIter++)
        result.push_back(*setIter);
}

// *****************************************************************************
// getClockNets()
//
/// \brief Gets all nets marked as clock by DFF's clock.
// *****************************************************************************
void Sim::getClockNets(vector<oaNet*>& result) {
    set<oaNet*>::iterator setIter;
    
    result.clear();
    for (setIter= myCkt->clockNets.begin(); setIter != myCkt->clockNets.end(); setIter++)
        result.push_back(*setIter);
}

// *****************************************************************************
// getInputs()
//
/// \brief Gets all PI nets inside this design.
// *****************************************************************************
void Sim::getInputs(vector<oaNet*>& result) {
    result= myCkt->inputNets;
}

// *****************************************************************************
// getOutputs()
//
/// \brief Gets all PO nets inside this design.
// *****************************************************************************
void Sim::getOutputs(vector<oaNet*>& result) {
    result= myCkt->outputNets;
}

// *****************************************************************************
// simulateInst()
//
/// \brief Simulate an instance. 
/// 
/// Inputs are from its input wires and outputs are sent to output wires. If isw is 1, copy the value to nextSimVec instead of simVec.
// *****************************************************************************
void Sim::simulateInst(oaInst* inInst, int isw = 0)
{
    libCell* cell;
    unsigned long type= 0;
    unsigned int outputOrder;
    bool processed= false;
    SimulationVector result= 0, result2;
    simInst* simInst2;
    oaInstTerm* instTerm;
    oaNet* myNet2;
    simWire* myWire2;
    unsigned int portCount= 0;
    unsigned int i, j;

    simInst2= myCkt->instGetSimInst(inInst);
    cell= simInst2->cell;
    oaIter<oaInstTerm> oaITIter(inInst->getInstTerms());
    oaIter<oaInstTerm> oaITIter2(inInst->getInstTerms());
    #ifdef DEBUG
        cout<<"Simulating inst: "<<getNameFromOA(inInst)<<endl;
    #endif
   
    if (simInst2->inputSimWires.empty())
    {
        simInst2->inputSimWires.insert(simInst2->inputSimWires.end(), cell->inputNo, NULL);
        // Determine input port order, must be the same as cell
        oaIter<oaInstTerm> oaITIter2(inInst->getInstTerms());
        i= 0;
        while (oaInstTerm* instTerm2= oaITIter2.getNext())
        {
            oaTerm*  term = instTerm2->getTerm();
            oaTermType termType = term->getTermType();
            if (termType == oacOutputTermType)
                continue;
            // determine portCount (position of port)
            for (portCount= 0; portCount < cell->inputNo; portCount++)
            {
                if (cell->inputNames[portCount].compare(getNameFromOA(instTerm2->getTerm())) == 0)
                    break;
            }
            myNet2= instTerm2->getNet();
            myWire2= myCkt->netGetSimWire(myNet2);
            simInst2->inputOrder.push_back(portCount);
            simInst2->inputSimWires[portCount]= myWire2;
        }
    }
    #ifdef DEBUG
    for (i= 0; i < cell->inputNo; i++)
        cout<<"Input("<<i<<"): "<<simInst2->inputSimWires[i]->simVec<<endl;
    #endif
    if (simInst2->outputOrder.empty())
    {
        i= 0;
        simInst2->outputSimWires.insert(simInst2->outputSimWires.end(), cell->outputNames.size(), NULL);
        for (instTerm= oaITIter.getNext(); instTerm; instTerm= oaITIter.getNext())
        {
            // only want output
            oaTerm* term = instTerm->getTerm();
            oaTermType termType = term->getTermType();
            if( termType == oacInputTermType)
            {
                continue;
            }
            for (outputOrder= 0; outputOrder < cell->outputNames.size(); outputOrder++)
            {
                if (cell->outputNames[outputOrder].compare(getNameFromOA(instTerm->getTerm())) == 0)
                    break;
            }
            myNet2= instTerm->getNet();
            myWire2= myCkt->netGetSimWire(myNet2);
            simInst2->outputOrder.push_back(outputOrder);
            simInst2->outputSimWires[outputOrder]= myWire2;
            i++;
        }
    }
    i= 0;
    for (instTerm= oaITIter2.getNext(); instTerm; instTerm= oaITIter2.getNext())
    {
        // only want output
        oaTerm* term = instTerm->getTerm();
        oaTermType termType = term->getTermType();
        if( termType == oacInputTermType)
        {
            continue;
        }
        outputOrder= simInst2->outputOrder[i];
        if (cell->inputNo <= 4)
            type= cell->tTables[outputOrder].to_ulong();
        if (cell->inputNo == 1)
        {
            switch (type)
            {
            case 1: // INV
                result= ~(simInst2->inputSimWires[0]->simVec);
                processed= true;
                break;
            case 2: // BUF
                result= (simInst2->inputSimWires[0]->simVec);
                processed= true;
                break;
            }
        }
        else if (cell->inputNo == 2)
        {
            switch (type)
            {
            case 8: // AND
                result= (simInst2->inputSimWires[0]->simVec) & (simInst2->inputSimWires[1]->simVec);
                processed= true;
                break;
            case 7: //NAND
                result= (simInst2->inputSimWires[0]->simVec) & (simInst2->inputSimWires[1]->simVec);
                result= ~result;
                processed= true;
                break;
            case 14: // OR
                result= (simInst2->inputSimWires[0]->simVec) | (simInst2->inputSimWires[1]->simVec);
                processed= true;
                break;
            case 1: // NOR
                result= (simInst2->inputSimWires[0]->simVec) | (simInst2->inputSimWires[1]->simVec);
                result= ~result;
                processed= true;
                break;
            case 6: //XOR
                result= (simInst2->inputSimWires[0]->simVec) ^ (simInst2->inputSimWires[1]->simVec);
                processed= true;
                break;
            case 9: //XNOR
                result= ~((simInst2->inputSimWires[0]->simVec) ^ (simInst2->inputSimWires[1]->simVec));
                processed= true;
                break;
            }
        }
        else if (cell->inputNo == 3)
        {
            switch (type)
            {
            case 7: // AOI21
                result= (simInst2->inputSimWires[0]->simVec) & (simInst2->inputSimWires[1]->simVec);
                result |= (simInst2->inputSimWires[2]->simVec);
                result= ~result;
                processed= true;
                break;
            case 202: // MX2X1
                result= (simInst2->inputSimWires[1]->simVec) & (simInst2->inputSimWires[2]->simVec);
                result2= (simInst2->inputSimWires[0]->simVec) & ~(simInst2->inputSimWires[2]->simVec);
                result |= result2;
                processed= true;
                break;
            case 1: // NOR
                result= (simInst2->inputSimWires[1]->simVec) | (simInst2->inputSimWires[2]->simVec) | (simInst2->inputSimWires[0]->simVec);
                result= ~result;
                processed= true;
                break;
            case 127: // NAND
                result= (simInst2->inputSimWires[1]->simVec) & (simInst2->inputSimWires[2]->simVec) & (simInst2->inputSimWires[0]->simVec);
                result= ~result;
                processed= true;
                break;
            case 31: // OAI21
                result= ~(((simInst2->inputSimWires[0]->simVec) | (simInst2->inputSimWires[1]->simVec)) & (simInst2->inputSimWires[2]->simVec)); 
                processed= true;
                break;
            }
        }
        else if (cell->inputNo == 4)
        {
            switch (type)
            {
            case 4383: // OAI22
                result= (simInst2->inputSimWires[0]->simVec) | (simInst2->inputSimWires[1]->simVec);
                result2= (simInst2->inputSimWires[2]->simVec) | (simInst2->inputSimWires[3]->simVec);
                result= result & result2;
                result= ~result;
                processed= true;
                break;
            case 1911: // AOI22
                result= (simInst2->inputSimWires[0]->simVec) & (simInst2->inputSimWires[1]->simVec);
                result2= (simInst2->inputSimWires[2]->simVec) & (simInst2->inputSimWires[3]->simVec);
                result= result | result2;
                result= ~result;
                processed= true;
                break;
            case 32767: // NAND4
                result= (simInst2->inputSimWires[0]->simVec) & (simInst2->inputSimWires[1]->simVec);
                result2= (simInst2->inputSimWires[2]->simVec) & (simInst2->inputSimWires[3]->simVec);
                result= result & result2;
                result= ~result;
                processed= true;
                break;
            case 65534: // OR4
                result= (simInst2->inputSimWires[0]->simVec) | (simInst2->inputSimWires[1]->simVec);
                result2= (simInst2->inputSimWires[2]->simVec) | (simInst2->inputSimWires[3]->simVec);
                result= result | result2;
                processed= true;
                break;
            case 1: // NOR4
                result= (simInst2->inputSimWires[0]->simVec) | (simInst2->inputSimWires[1]->simVec);
                result2= (simInst2->inputSimWires[2]->simVec) | (simInst2->inputSimWires[3]->simVec);
                result= result | result2;
                result= ~result;
                processed= true;
                break;
            }
        }
        #ifdef DEBUG
        processed= false;
        #endif
        if (processed == false)
        {
            // Use truth table, calculate bit-by-bit
            unsigned int minterm, bitCount;
            simWire* myWire;


            result2= 0;
            for (bitCount= 0; bitCount < sizeof(SimulationVector) * 8; bitCount++)
            {
                minterm= 0;
                for (j= 0; j < cell->inputNo; j++)
                {
                    myWire= simInst2->inputSimWires[j];
                    if ((myWire->simVec & (1 << bitCount)) != 0)
                    {
                        minterm |= (1 << j);
                    }
                }
                if (cell->tTables[outputOrder][minterm] == true)
                    result2 |= (1 << bitCount);
            }
            #ifdef DEBUG
            if (result != result2)
                cout<<"Mismatch: "<<result<<", "<<result2<<endl;
            #endif
            result= result2;
        }

        #ifdef DEBUG
        cout<<"Output "<<outputOrder<<": "<<result<<endl;
        #endif
        myNet2= instTerm->getNet();
        myWire2= simInst2->outputSimWires[outputOrder];
        if (myWire2)
        {
            oaBitNet* myBitNet= static_cast<oaBitNet*>(myNet2);
            if (myBitNet->getEquivalentNets().getCount() > 0)
            {
                setVector(myNet2, result, isw);
            }
            else
            {
                if (isw == 0)
                    myWire2->simVec= result;
                else
                    myWire2->nextSimVec= result;
            }
        }
        i++;
    }
    #ifdef DEBUG
    cout<<endl;
    #endif
}

// *****************************************************************************
// setVector()
//
/// \brief Set vector to a net. 
//
// *****************************************************************************
void Sim::setVector(oaNet* net, SimulationVector vec, int isw)
{
    if (net->getNumBits() > 1)
    {
        cout<<"Simulation vector must be set on single-bit nets."<<endl;
        cout<<"The problematic net name is: "<<getNameFromOA(net)<<endl;
        return;
    }
   
    myCkt->setVector(net, vec, isw);
}

// *****************************************************************************
// setStateVector()
//
/// \brief Set vector to a state register. 
///
/// Will search Q and QN.
//
// *****************************************************************************
void Sim::setStateVector(oaInst* inst, SimulationVector vec)
{
    simWire* myWire;
    oaNet* net;

    oaIter<oaInstTerm>   oaITIter(inst->getInstTerms(oacInstTermIterNotImplicit | oacInstTermIterEquivNets));
    while (oaInstTerm* instTerm = oaITIter.getNext())
    {
        oaTerm* term = instTerm->getTerm();
        oaTermType termType = term->getTermType();
        if( termType == oacOutputTermType)
        {
            net= instTerm->getNet();
            myWire= myCkt->netGetSimWire(net);
            oaBitNet* myBitNet= static_cast<oaBitNet*>(net);
            if (myWire->isRegOutQN)
            {
                if (myBitNet->getEquivalentNets().getCount() > 0)
                    setVector(net, ~vec, 0);
                else
                    myWire->simVec= ~vec;
            }
            else
            {
                if (myBitNet->getEquivalentNets().getCount() > 0)
                    setVector(net, vec, 0);
                else
                    myWire->simVec= vec;
            }
        }
    }
}

// *****************************************************************************
// getVector()
//
/// \brief Get vector from a net. 
//
// *****************************************************************************
void Sim::getVector(oaNet* net, SimulationVector& vec)
{
    simWire* myWire;

    if (net->getNumBits() > 1)
    {
        cout<<"Simulation vector only available on single-bit nets."<<endl;
        cout<<"The problematic net name is: "<<getNameFromOA(net)<<endl;
        return;
    }
    myWire= myCkt->netGetSimWire(net);
    vec= myWire->simVec;
}

// *****************************************************************************
// generateRandomInputVectors()
//
/// \brief Generate random input vectors.
//
// *****************************************************************************
void Sim::generateRandomInputVectors()
{
    vector<oaNet*>::iterator netIter;

    for (netIter= myCkt->inputNets.begin(); netIter != myCkt->inputNets.end(); netIter++)
    {
        setVector(*netIter, randomVector());
    }
}

// *****************************************************************************
// generateRandomStateVectors()
//
/// \brief Generate random state vectors.
//
// *****************************************************************************
void Sim::generateRandomStateVectors()
{
    vector<oaInst*>::iterator instIter;

    for (instIter= myCkt->stateInsts.begin(); instIter != myCkt->stateInsts.end(); instIter++)
    {
        setStateVector(*instIter, randomVector());
    }
}

// *****************************************************************************
// runFull()
//
/// \brief Do one-cycle full simulation.
//
// *****************************************************************************
void Sim::runFull()
{
    vector<oaInst*>::iterator instIter;
    for (instIter= myCkt->levelizedInst.begin(); instIter != myCkt->levelizedInst.end();
         instIter++)
    {
        simulateInst(*instIter);
    }
}

// *****************************************************************************
// nextCycle()
//
/// \brief Do another cycle of full simulation. 
///
/// Values will pass through state registers.
//
// *****************************************************************************
void Sim::nextCycle()
{
    // Propagate state register's inputs to outputs
    vector<oaNet*>::iterator netIter;
    simWire* myWire, *myWire2;
    oaInst* myOAInst;
    SimulationVector vec;
    simInst* myInst;
    stateSimWireCache myCache;
    vector<stateSimWireCache>::iterator cacheIter;
    vector<simWire*>::iterator wireIter;
    

    #ifdef DEBUG
    cout<<"Propagate states"<<endl;
    #endif
    if (stateCache.empty())
    {
        for (netIter= myCkt->stateInNets.begin(); netIter != myCkt->stateInNets.end(); netIter++)
        {
            myWire= myCkt->netGetSimWire(*netIter);
            myCache.dataWire= myWire;
            oaIter<oaInstTerm>   oaITIter((*netIter)->getInstTerms(oacInstTermIterNotImplicit | oacInstTermIterEquivNets));
            while (oaInstTerm* instTerm = oaITIter.getNext())
            {
                oaTerm* term = instTerm->getTerm();
                oaTermType termType = term->getTermType();
                if( termType == oacOutputTermType)
                    continue;
                myOAInst= instTerm->getInst();
                myInst= myCkt->instGetSimInst(myOAInst);
                if (myInst->isStateReg)
                {
                    oaIter<oaInstTerm>   oaITIter2(myOAInst->getInstTerms(oacInstTermIterNotImplicit | oacInstTermIterEquivNets));
                    while (oaInstTerm* instTerm2 = oaITIter2.getNext())
                    {
                        oaTerm* term2 = instTerm2->getTerm();
                        oaTermType termType2 = term2->getTermType();
                        if( termType2 == oacInputTermType)
                            continue;
                        oaNet* myNet2= instTerm2->getNet();
                        oaBitNet*  bitNet= static_cast<oaBitNet*>(myNet2);
                        myWire2= myCkt->netGetSimWire(myNet2);
                        if (myWire2 == NULL)
                            continue;
                        // collect all wires connected to mynet2
                        if (myWire2->isRegOutQN)
                            myCache.QN.push_back(myWire2);
                        else
                            myCache.Q.push_back(myWire2);

                        if (bitNet->getEquivalentNets().getCount() > 0)
                        {
                            oaBitNet* bitNet2;
                            oaIter<oaBitNet>   oaBNIter(bitNet->getEquivalentNets());
                            for (bitNet2 = oaBNIter.getNext(); bitNet2; bitNet2 = oaBNIter.getNext())
                            {
                                simWire* myWire= myCkt->netGetSimWire(bitNet2);
                                if (myWire2->isRegOutQN)
                                    myCache.QN.push_back(myWire);
                                else
                                    myCache.Q.push_back(myWire);
                            }
                        }
                    }
                }
            }
            stateCache.push_back(myCache);
            myCache.Q.clear();
            myCache.QN.clear();
        }
    }
    for (cacheIter= stateCache.begin(); cacheIter != stateCache.end(); cacheIter++)
    {
        vec= (*cacheIter).dataWire->simVec;
        for (wireIter= (*cacheIter).Q.begin(); wireIter != (*cacheIter).Q.end(); wireIter++)
        {
            (*wireIter)->simVec= vec;
        }
        vec= ~vec;
        for (wireIter= (*cacheIter).QN.begin(); wireIter != (*cacheIter).QN.end(); wireIter++)
        {
            (*wireIter)->simVec= vec;
        }
    }
    runFull();
}

// *****************************************************************************
// randomVector()
//
/// \brief Generate a random vector.
//
// *****************************************************************************
SimulationVector Sim::randomVector()
{
    unsigned int i;
    const int N_RANDS = UINT_MAX / RAND_MAX;

    SimulationVector result = 0;
    for (i= 0; i < sizeof(SimulationVector) / sizeof(int); i++)
    {
        for(int i = 0; i<N_RANDS; ++i)
            result += static_cast<SimulationVector>(rand()) << (i * sizeof(int) * 8);
    }

    return result;
}

// *****************************************************************************
// getStateNets()
//
/// \brief Get nets connected to the outputs of state registers.
//
// *****************************************************************************
void Sim::getStateNets(vector<oaNet*>& result)
{
    result= myCkt->stateNets;
}

// *****************************************************************************
// ~sim()
//
/// \brief Destructor.
//
// *****************************************************************************
Sim::~Sim()
{
    if (myUserPattern)
        delete(myUserPattern);
}

// *****************************************************************************
// doRandomSimulation()
//
/// \brief Do random simulation for one cycle.
//
// *****************************************************************************
void Sim::doRandomSimulation()
{
    vector<oaNet*> nets;
    
    generateRandomInputVectors();
    generateRandomStateVectors();
    runFull();
}

// *****************************************************************************
// incrementalSim
//
/// \brief Do incremental simulation for the net.
/// It is assumed that its value is changed to newVec. Simulation values of all other nets must have been available before calling incrementalSim, for example, by calling runFull.
//
// *****************************************************************************
void Sim::incrementalSim(oaNet* wire, SimulationVector newVec)
{
    // See my master's thesis for the simulation algorithm
    list<oaNet*> activeNodes;
    list<oaInst*> activeGates;
    oaNet* net2;
    simWire* wire2;
    simInst* simInst2;
    oaInst* inst2;
    
    wire2= myCkt->netGetSimWire(wire);
    wire2->nextSimVec= newVec;
    activeNodes.push_back(wire);

    do
    {
        // Update active Nodes
        while (activeNodes.empty() == false)
        {
            wire= activeNodes.front();
            wire2= myCkt->netGetSimWire(wire);
            wire2->marked= false;
            activeNodes.pop_front();
            setVector(wire, wire2->nextSimVec, 0);
            
            oaIter<oaInstTerm>   oaITIter(wire->getInstTerms(oacInstTermIterNotImplicit | oacInstTermIterEquivNets));
            while (oaInstTerm* instTerm = oaITIter.getNext())
            {
                oaTerm* term = instTerm->getTerm();
                oaTermType termType = term->getTermType();
                if (termType == oacInputTermType)
                    continue;
                inst2= instTerm->getInst();
                simInst2= myCkt->instGetSimInst(inst2);
                if (simInst2 && simInst2->marked == false && simInst2->isStateReg == false)
                {
                    simInst2->marked= true;
                    activeGates.push_back(inst2);
                }
            }
        }
        while (activeGates.empty() == false)
        {
            
            inst2= activeGates.front();
            activeGates.pop_front();
            simInst2= myCkt->instGetSimInst(inst2);
            simInst2->marked= false;
            simulateInst(inst2, 1);
            oaIter<oaInstTerm>   oaITIter(wire->getInstTerms(oacInstTermIterNotImplicit | oacInstTermIterEquivNets));
            while (oaInstTerm* instTerm = oaITIter.getNext())
            {
                oaTerm* term = instTerm->getTerm();
                oaTermType termType = term->getTermType();
                if (termType == oacInputTermType)
                    continue;
                net2= instTerm->getNet();
                wire2= myCkt->netGetSimWire(net2);
                if (wire2->nextSimVec != wire2->simVec && wire2->marked == false)
                {
                    wire2->marked= true;
                    activeNodes.push_back(net2);
                }
            }
        }
    } while (activeNodes.empty() == false);
}

/// \brief Advance input to next cycle.
/// Get new input vectors and apply them to primary inputs. If the user
/// has provided data file, inputs will be from the file. Otherwise random
/// inputs will be returned. If user data has been exhausted, random
/// inputs will be used.
///
void Sim::advanceInputs()
{
    vector<SimulationVector> vecs;
    unsigned int i;
    
    if (myUserPattern && myUserPattern->isFinished() == false)
    {
        if (myUserPattern->getVector(vecs) == false)
        {
            generateRandomInputVectors();
            return;
        }   
        for (i= 0; i < inputVecNets.size(); i++)
        {
            setVector(inputVecNets[i], vecs[i]);
        }
    }
    else
    {
        generateRandomInputVectors();
    }
}

/// Set user pattern file.
/// \param fileName file name 
/// \return true if success.
bool Sim::setUserPatternFile(const char* fileName)
{
    vector<oaInst*>::iterator instIter;

    myUserPattern= new (userPattern);
    // If input file not available, set states to random
    if (myUserPattern->setFile(design, fileName) == false)
    {
        delete(myUserPattern);
        myUserPattern= NULL;
        generateRandomStateVectors();
        return false;
    }
    // if input file available, set states to 0
    for (instIter= myCkt->stateInsts.begin(); instIter != myCkt->stateInsts.end(); instIter++)
        setStateVector(*instIter, 0);
    myUserPattern->getNets(inputVecNets);
    return true;
}

/// \brief Step to next cycle.
/// Will update inputs and do next cycle simulation
void Sim::step()
{
    advanceInputs();
    nextCycle();
}

/// \brief If true, user patterns have been exhausted
bool Sim::isFinished()
{
    if (myUserPattern)
    {
        if (myUserPattern->isFinished())
        {
            return true;   
        }
    }
    return false;
}

/// \brief use state file to initialize states.
bool Sim::useStateFile(const char* fileName)
{
    ifstream iFile;
    vector<oaInst*> insts;
    oaInst* inst;
    vector<oaInst*>::iterator instIter;
    SimulationVector simVal;
    int result;
    
    string token;
    oaBlock* block= design->getTopBlock();
    iFile.open(fileName, ios::in);
    if (iFile.is_open())
    {
        while (util::getToken(iFile, token) == 0)
        {
            const oa::oaNativeNS nativeNS;
            oaSimpleName name1(nativeNS, static_cast<const oaChar*>(token.c_str()));
            inst= oaInst::find(block, name1);
            if  (inst == NULL)
            {
                cout<<"Instance "<<token<<" not found, ignored."<<endl;
            }
            insts.push_back(inst);
        }
        // Read values
        for (instIter= insts.begin(); instIter != insts.end(); instIter++)
        {
            result= util::getToken(iFile, token);
            if (result == 1)
                continue;
            else if (result == 2)
            {
                iFile.close();
                break;
            }
            else if (result == 0)
            {
                if (SimVectorSize == 32)
                    simVal= atol(token.c_str());
                else
                    simVal= atoll(token.c_str());
                if (*instIter)
                    setStateVector(*instIter, simVal);
            }
        }
        return true;
    }
    else
        return false;
}

} // End of namespace
// vim: ci et sw=4 sts=4
