/*
Author: Kai-hui Chang <changkh@eecs.umich.edu>
*/

#include <iostream>
#include <fstream>
#include <string>
#include <set>
#include <list>
#include <algorithm>
#include "oagResynOptInternalCkt.h"
#include "oagResynUtil.h"

using namespace std;
using namespace oa;

// #define DEBUG

namespace oagResyn {

// *****************************************************************************
// optInternalCkt()
//
/// \brief Constructor.
//
// *****************************************************************************
optInternalCkt::optInternalCkt(oa::oaDesign *design) {
    assert(design);
    block= design->getTopBlock();
    assert(block);
    this->design = design;
    tie0Wire= NULL;
    tie1Wire= NULL;
    setup();
}

// *****************************************************************************// analyzeCircuit()
//
/// \brief Analyze the circuit. 
///
/// Identify state registers, PIs and POs. Also build cell library for simulation. Called by setup.
//
// *****************************************************************************
void optInternalCkt::analyzeCircuit() {
    set<oaInst*> stateSet;
    set<oaInst*>::iterator setIter;

    // Identify PIs/POs
    try
    {
        oaIter<oaTerm>   oaTIter(block->getTerms());
        while (oaTerm*  term = oaTIter.getNext())
        {
            oaTermType termType = term->getTermType();
            oaNet* net= term->getNet();
            if( termType == oacOutputTermType)
            {
                oaIter<oaBitNet>   oaBNIter(net->getSingleBitMembers());
                while (oaNet*  bitNet = (oaNet*)oaBNIter.getNext())
                    outputNets.push_back(bitNet);
            }
            if( termType == oacInputTermType)
            {
                oaIter<oaBitNet>   oaBNIter(net->getSingleBitMembers());
                while (oaNet*  bitNet = (oaNet*)oaBNIter.getNext())
                    inputNets.push_back(bitNet);
            }
        }
        oaIter<oaNet>   oaNetIter(block->getNets());
        // Find all bitNets
        while (oaNet*  net = oaNetIter.getNext())
        {
            oaIter<oaBitNet>   oaBNIter(net->getSingleBitMembers());
            while (oaNet*  bitNet = (oaNet*)oaBNIter.getNext())
            {
                if (net->getNumBits() == 1 && net != bitNet)
                    allBitNets.insert(net);
                else
                    allBitNets.insert(bitNet);
            }
        }
    }
    catch (oa::oaException &excp)
    {
        printf("Error: %s\n", (const char *)excp.getMsg());
        return;
    }
    
    try
    {
        oaIter<oaInst>   oaInstIter(block->getInsts());
        while (oaInst*  inst = oaInstIter.getNext()) 
        {
            string modName;
            
            instGetModName(inst, modName);
            // Check if module name contains DFF.
            if (modName.find(dffKey.c_str()) != string::npos)
                stateSet.insert(inst);
            else
                libManager.addCell(modName);
        }
    }
    catch (oa::oaException &excp)
    {
        printf("Error: %s\n", (const char *)excp.getMsg());
        return;
    }
    // Move stateSet to stateInsts
    for (setIter= stateSet.begin(); setIter != stateSet.end(); setIter++)
    {
        stateInsts.push_back(*setIter);
    }
}

// *****************************************************************************
// getStates()
//
/// \brief Gets all state instances inside this design. 
// *****************************************************************************
void optInternalCkt::getStates(vector<oaInst*>& result) {
    result= stateInsts;
}

// *****************************************************************************
// getAllNets()
//
/// \brief Gets all nets in the circuit (oaBitNet).
// *****************************************************************************
void optInternalCkt::getAllNets(vector<oaNet*>& result) {
    set<oaNet*>::iterator setIter;
    
    result.clear();
    for (setIter= allBitNets.begin(); setIter != allBitNets.end(); setIter++)
        result.push_back(*setIter);
}

// *****************************************************************************
// getClockNets()
//
/// \brief Gets all nets marked as clock by DFF's clock.
// *****************************************************************************
void optInternalCkt::getClockNets(vector<oaNet*>& result) {
    set<oaNet*>::iterator setIter;
    
    result.clear();
    for (setIter= clockNets.begin(); setIter != clockNets.end(); setIter++)
        result.push_back(*setIter);
}

// *****************************************************************************
// getInputs()
//
/// \brief Gets all PI nets inside this design.
// *****************************************************************************
void optInternalCkt::getInputs(vector<oaNet*>& result) {
    result= inputNets;
}

// *****************************************************************************
// getOutputs()
//
/// \brief Gets all PO nets inside this design.
// *****************************************************************************
void optInternalCkt::getOutputs(vector<oaNet*>& result) {
    result= outputNets;
}

// *****************************************************************************
// printInOutState()
//
/// \brief Print all PI/PO/State
// *****************************************************************************
void optInternalCkt::printInOutState()
{
    vector<oaInst*>::iterator instIter;
    vector<oaNet*>::iterator netIter;
    oaInst* myInst;
    oaNet* myNet;

    cout<<"Input terminals:"<<endl;
    for (netIter= inputNets.begin(); netIter != inputNets.end(); netIter++)
    {
        myNet= *netIter;
        cout<<getNameFromOA(myNet)<<endl;
    }
    cout<<"Output terminals:"<<endl;
    for (netIter= outputNets.begin(); netIter != outputNets.end(); netIter++)
    {
        myNet= *netIter;
        cout<<getNameFromOA(myNet)<<endl;
    }
    cout<<"State registers:"<<endl;
    for (instIter= stateInsts.begin(); instIter != stateInsts.end(); instIter++)
    {
        myInst= *instIter;
        cout<<getNameFromOA(myInst)<<endl;
    }
}

// *****************************************************************************
// instGetModName()
//
/// \brief Gets the module name of an instance
// *****************************************************************************
void optInternalCkt::instGetModName(oaInst* inInst, string& modName) {
    oa::oaNativeNS ns;
    oa::oaString myName;
    oa::oaInstHeader* myHeader;
                        
    myHeader= inInst->getHeader();
    myHeader->getCellName(ns, myName);
    modName= static_cast<const char*>(myName);
}

// *****************************************************************************
// buildInternalCircuit()
//
/// \brief Build internal circuit for simulation.
// *****************************************************************************
void optInternalCkt::buildInternalCircuit()
{
    optSimInst* newInst;
    optSimWire* newWire;
    optSimPort* newPort;
    string cellName;
    map<oaNet*, optSimWire*>::iterator wireMapIter;
    map<oaInst*, optSimInst*>::iterator instMapIter;
    vector<oaInst*>::iterator oaInstIter;
    vector<oaNet*>::iterator oaWireIter;
    vector<optSimPort*>::iterator portIter, portIter2;
    set<oaNet*>::iterator netIter;

    // Build all wires first and the map from nets to wires. Only one wire
    // will be created for equivalent net
    try
    {
        // Create all nets first
        for (netIter= allBitNets.begin(); netIter != allBitNets.end();
             netIter++)
        {
            oaNet* net= *netIter;
            oaBitNet* bitNet= (oaBitNet*)net;
            bool found;
            bool isTie0, isTie1;
            
            // Also identify tie0/tie1 here
            if (tie0.compare(getNameFromOA(net)) == 0)
            {
                isTie1= false;
                isTie0= true;
            }
            else if (tie1.compare(getNameFromOA(net)) == 0)
            {
                isTie1= true;
                isTie0= false;
            } 
            else
            {
                isTie0= false;
                isTie1= false;
            }

            if (bitNet->getEquivalentNets().getCount() > 0)
            {
                oaBitNet*  bitNet2;

                found= false;
                oaIter<oaBitNet>   oaBNIter(bitNet->getEquivalentNets());
                for (bitNet2 = oaBNIter.getNext(); bitNet2; bitNet2 = oaBNIter.getNext())
                {
                    // Search if any wire already exists
                    wireMapIter= wireMap.find(bitNet2);
                    if (wireMapIter != wireMap.end())
                    {
                        found= true;
                        break;
                    }
                }
                if (found == true)
                {
                    // A wire already exists for this equivalent wire.
                    // Use that.
                    newWire= (*wireMapIter).second;
                    wireMap[bitNet]= newWire;
                }
                else
                {
                    // No equivalent wire exist yet. Create it.
                    newWire= (optSimWire*) new(optSimWire);
                    wireMap[net]= newWire;
                    newWire->origWire= net;
                }
            }
            else
            {
                newWire= (optSimWire*) new(optSimWire);
                wireMap[net]= newWire;
                newWire->origWire= net;
            }
            if ((isTie0 && tie0Wire == NULL) || (isTie1 && tie1Wire == NULL))
            {
                newWire->isPI= true;
                newWire->driver= NULL;
                if (isTie0)
                    tie0Wire= newWire;
                else
                    tie1Wire= newWire;
            } 
        }
    }
    catch (oa::oaException &excp)
    {
        printf("Error: %s\n", (const char *)excp.getMsg());
        return;
    }
    // Now build instance net connections
    try
    {
        oaIter<oaInst>   oaInstIter(block->getInsts());
        int inst_count= 0;
        while (oaInst*  inst = oaInstIter.getNext())
        {
            inst_count++;
            // Add inst to circuit
            newInst= (optSimInst*)new(optSimInst);        
            newInst->origInst= inst;
            instMap[inst]= newInst;
            instGetModName(inst, cellName);
            // May be NULL if it is stateRegister
            newInst->cell= libManager.lookUpCell(cellName);
            oaIter<oaInstTerm>   oaITIter(inst->getInstTerms());
            while (oaInstTerm*  instTerm = oaITIter.getNext())
            {
                oaTerm* term = instTerm->getTerm();
                newPort= (optSimPort*)new(optSimPort);
                newPort->origPort= term;
                newPort->origInstTerm= instTerm;
                newPort->inst1= newInst;
                newPort->wire1= NULL;
                oaTermType termType = term->getTermType();
                if( termType == oacOutputTermType)
                {
                    newInst->outputs.push_back(newPort);
                }
                else if( termType == oacInputTermType)
                {
                    newInst->inputs.push_back(newPort);
                }
                else
                {
                    cout<<"TermType not known."<<termType<<endl;
                }
                oaNet* net = instTerm->getNet();
                if (net == NULL)
                    continue;
                wireMapIter= wireMap.find(net);
                if (wireMapIter == wireMap.end())
                {   
                    cout<<"Net not found: "<<getNameFromOA(net)<<endl;
                    newWire= (optSimWire*) new(optSimWire);
                    wireMap[net]= newWire;
                    newWire->origWire= net;
                }
                else
                    newWire= (*wireMapIter).second;
                newPort->wire1= newWire;
                if( termType == oacOutputTermType)
                {
                    newWire->driver= newPort;
                }
                else if( termType == oacInputTermType)
                {
                    newWire->outputs.push_back(newPort);
                }
                else
                {
                    cout<<"TermType not known."<<termType<<endl;
                }
            }
        }
        if (inst_count == 0)
        {
            cerr<<"Error: No instance has been found in the block domain. Cannot build internal data structure."<<endl;
            throw;
        }
    }
    catch (oa::oaException &excp)
    {
        printf("Error: %s\n", (const char *)excp.getMsg());
        return;
    }
    // Mark PI/PO/State
    for (oaInstIter= stateInsts.begin(); oaInstIter != stateInsts.end(); oaInstIter++)
    {   
        instMapIter= instMap.find(*oaInstIter);
        if (instMapIter == instMap.end())
        {
            cout<<"Could not find instance for state register."<<endl;
        }
        else
        {
            (*instMapIter).second->isStateReg= true;
            for (portIter= (*instMapIter).second->outputs.begin(); 
                 portIter != (*instMapIter).second->outputs.end();
                 portIter++)
            {
                if ((*portIter)->wire1 == NULL)
                    continue;
                if (dffQ.compare(getNameFromOA((*portIter)->origPort)) == 0)
                    (*portIter)->wire1->isRegOut= true;
                else if (dffQN.compare(getNameFromOA((*portIter)->origPort)) == 0)
                {
                    (*portIter)->wire1->isRegOut= true;
                    (*portIter)->wire1->isRegOutQN= true;
                    // Search for Q and save it
                    for (portIter2= (*instMapIter).second->outputs.begin(); 
                        portIter2 != (*instMapIter).second->outputs.end();
                        portIter2++)
                        if (dffQ.compare(getNameFromOA((*portIter2)->origPort)) == 0)
                            (*portIter)->wire1->regQWire= (*portIter2)->wire1;
                }
                stateNets.push_back((*portIter)->wire1->origWire);
            }
            for (portIter= (*instMapIter).second->inputs.begin(); 
                 portIter != (*instMapIter).second->inputs.end();
                 portIter++)
            {
                if ((*portIter)->wire1 == NULL)
                    continue;
                if (dffD.compare(getNameFromOA((*portIter)->origPort)) == 0)
                {
                    (*portIter)->wire1->isRegD= true;
                    stateInWires.push_back((*portIter)->wire1);
                    stateInNets.push_back((*portIter)->wire1->origWire);
                }
                else if (dffCLK.compare(getNameFromOA((*portIter)->origPort)) == 0)
                {
                    clockNets.insert((*portIter)->wire1->origWire);
                }
            }
        }
    }
    for (oaWireIter= inputNets.begin(); oaWireIter != inputNets.end(); oaWireIter++)
    {
        wireMapIter= wireMap.find(*oaWireIter);
        if (wireMapIter == wireMap.end())
        {
            cout<<"Could not find wire."<<endl;
        }
        else
        {
            (*wireMapIter).second->isPI= true;
            (*wireMapIter).second->driver= NULL;
        }
    }
    for (oaWireIter= outputNets.begin(); oaWireIter != outputNets.end(); oaWireIter++)
    {
        wireMapIter= wireMap.find(*oaWireIter);
        if (wireMapIter == wireMap.end())
        {
            cout<<"Could not find wire."<<endl;
        }
        else
            (*wireMapIter).second->isPO= true;
    }
    // Now adjust the port order of insts so that they match cell library's
    // for correct simulation
    for (instMapIter= instMap.begin(); instMapIter != instMap.end(); instMapIter++)
    {
        optSimInst *myInst= (*instMapIter).second;
        vector<optSimPort*> newPorts;
        vector<string>::iterator strIter;
        libCell* myCell= myInst->cell;
        vector<optSimPort*>::iterator portIter;
        string nameStr;
       
        // Not necessary for state registers
        if (myCell == NULL)
            continue;
        // input
        for (strIter= myCell->inputNames.begin(); strIter != myCell->inputNames.end(); strIter++)
        {
            // search instance's port for match
            for (portIter= myInst->inputs.begin(); portIter != myInst->inputs.end(); portIter++)
            {
                nameStr= getNameFromOA((*portIter)->origPort);
                if (nameStr.compare(*strIter) == 0)
                {
                    newPorts.push_back(*portIter);
                    break;
                }
            }
            if (portIter == myInst->inputs.end())
                cout<<"Could not find port in reordering"<<endl;
        }
        myInst->inputs= newPorts;
        newPorts.clear();
        for (strIter= myCell->outputNames.begin(); strIter != myCell->outputNames.end(); strIter++)
        {
            // search instance's port for match
            for (portIter= myInst->outputs.begin(); portIter != myInst->outputs.end(); portIter++)
            {
                nameStr= getNameFromOA((*portIter)->origPort);
                if (nameStr.compare(*strIter) == 0)
                {
                    newPorts.push_back(*portIter);
                    break;
                }
            }
            // Unconnected output, put a port without wire to maintain
            // the logic relationship
            if (portIter == myInst->outputs.end())
            {
                optSimPort* newPort;
                
                newPort= new(optSimPort);
                newPort->wire1= NULL;
                newPort->inst1= myInst;
                newPort->origPort= NULL;
                newPort->origInstTerm= NULL;
                newPorts.push_back(newPort);
            }
        }
        myInst->outputs= newPorts;
        newPorts.clear();
    }
}

// *****************************************************************************
// printInternalCircuit()
//
/// \brief Print internal circuit, for debugging
// *****************************************************************************
void optInternalCkt::printInternalCircuit()
{
    map<oaInst*, optSimInst*>::iterator instMapIter;

    for (instMapIter= instMap.begin(); instMapIter != instMap.end(); instMapIter++)
    {
        printInternalInst((*instMapIter).second);
    }
    
}

// *****************************************************************************
// printInternalInst()
//
/// \brief Print internal inst, for debugging
// *****************************************************************************
void optInternalCkt::printInternalInst(optSimInst* inst)
{
    string str;
    vector<optSimPort*>::iterator portIter;

    if (inst == NULL)
        cout<<"No name instance"<<endl;
    instGetModName(inst->origInst, str);
    cout<<"Instance: "<<getNameFromOA(inst->origInst)<<", module= "<<str;
    if (inst->isStateReg)
        cout<<", is state register";
    cout<<endl;
    cout<<"Level: "<<inst->level<<endl;
    cout<<"Inputs:"<<endl;
    for (portIter= inst->inputs.begin(); portIter != inst->inputs.end(); portIter++)
    {
        cout<<getNameFromOA((*portIter)->origPort)<<" -> "<<getNameFromOA((*portIter)->wire1->origWire);
        if ((*portIter)->wire1->isPI)
            cout<<", is primary input";
        cout<<endl;
    }
    cout<<"Outputs:"<<endl;
    for (portIter= inst->outputs.begin(); portIter != inst->outputs.end(); portIter++)
    {
        cout<<getNameFromOA((*portIter)->origPort)<<" -> "<<getNameFromOA((*portIter)->wire1->origWire);
        if ((*portIter)->wire1->isPO)
            cout<<", is primary output";
        cout<<endl;
    }
    cout<<"Cell info:"<<endl;
//    if (inst->cell)
//        libManager.printCell(inst->cell);
}

// *****************************************************************************
// levelizeInternalCircuit()
//
/// \brief Levelize the internal circuit for simulation.
// *****************************************************************************
void optInternalCkt::levelizeInternalCircuit()
{
    map<oaInst*, optSimInst*>::iterator instMapIter;
    map<oaNet*, optSimWire*>::iterator wireMapIter;
    optSimWire* myWire;
    optSimInst* myInst;
    vector<optSimInst*> sInsts;
    vector<optSimInst*>::iterator instIter;
    vector<optSimPort*>::iterator portIter;
    unsigned int count, count2;
    int level;

    levelizedInst.clear();
    // Use wire's simVec to determine if all inputs are available. If 0,
    // not yet available. If 1, already available.
    for (wireMapIter= wireMap.begin(); wireMapIter != wireMap.end(); wireMapIter++)
    {
        myWire= (*wireMapIter).second;
        if (myWire->isPI || myWire->isRegOut)
            myWire->simVec= 1;
        else
            myWire->simVec= 0;
    }
    // Add combinational instances to sInsts
    for (instMapIter= instMap.begin(); instMapIter != instMap.end(); instMapIter++)
    {
        myInst= (*instMapIter).second;
        if (myInst->isStateReg == false)
            sInsts.push_back(myInst);
        myInst->visited= false;
    }
    count= 0;
    count2= 0;
    while (levelizedInst.size() < sInsts.size())
    {
        count2++;
        if (count2 > sInsts.size())
            break;
        for (instIter= sInsts.begin(); instIter != sInsts.end(); instIter++)
        {
            myInst= *instIter;
            if (myInst->visited)
                continue;
            level= 0;
            for (portIter= myInst->inputs.begin(); portIter != myInst->inputs.end(); portIter++)
            {
                myWire= (*portIter)->wire1;
                if (myWire && myWire->simVec == 0)
                    break;
                if (myWire && myWire->driver && myWire->driver->inst1)
                {
                    if (myWire->driver->inst1->level > level)
                        level= myWire->driver->inst1->level;
                }
            }
            if (portIter == myInst->inputs.end())
            {
                myInst->level= level + 1;
                levelizedInst.push_back(myInst);
                myInst->visited= true;
                count++;
                for (portIter= myInst->outputs.begin(); portIter != myInst->outputs.end(); portIter++)
                {
                    myWire= (*portIter)->wire1;
                    if (myWire)
                        myWire->simVec= 1;
                }
            }
        }
    }
    if (count != sInsts.size())
    {
        unsigned int i, j;
        int errs;
        
        cout<<"Could not levelize internal circuit due to gates:"<<endl;
        for (i= 0; i < sInsts.size(); i++)
        {
            for (j= 0; j < levelizedInst.size(); j++)
            {
                if (sInsts[i] == levelizedInst[j])
                    break;
            }
            if (j == levelizedInst.size())
            {
                myInst= sInsts[i];
                errs= 0;
                while (myInst != NULL && myInst->isStateReg == false && errs < 30)
                {
                    cout<<getNameFromOA(myInst->origInst)<<", ";
                    for (portIter= myInst->inputs.begin(); portIter != myInst->inputs.end(); portIter++)
                    {
                        myWire= (*portIter)->wire1;
                        if (myWire && myWire->driver && myWire->driver->inst1)
                        {
                            if (myWire->driver->inst1->level == 0)
                            {
                                myInst= myWire->driver->inst1;
                                break;
                            }
                        }
                    }
                    errs++;
                }
                cout<<endl;
            }
        }
        cout<<endl;
    }
    // clear visited flag of instances
    for (instIter= levelizedInst.begin(); instIter != levelizedInst.end(); instIter++)
        (*instIter)->visited= false;
    // Set tie0/tie1
    if (tie0Wire)
        tie0Wire->simVec= 0;
    if (tie1Wire)
        tie1Wire->simVec= ~0;
}

// *****************************************************************************
// getStateNets()
//
/// \brief Get nets connected to the outputs of state registers.
//
// *****************************************************************************
void optInternalCkt::getStateNets(vector<oaNet*>& result)
{
    result= stateNets;
}

// *****************************************************************************
// getStateInNets()
//
/// \brief Get nets connected to the inputs of state registers.
//
// *****************************************************************************
void optInternalCkt::getStateInNets(vector<oaNet*>& result)
{
    result= stateInNets;
}


// *****************************************************************************
// setup()
//
/// \brief Setup internal data structures.
///
/// Note that setKeyWords must be called before this routine to be effective.
//
// *****************************************************************************
void optInternalCkt::setup()
{
    analyzeCircuit();
    libManager.genLogicInfo(design);
    buildInternalCircuit();
    levelizeInternalCircuit();
}

// *****************************************************************************
// cleanUp()
//
/// \brief Destroy datastructures built and free the memory.
//
// *****************************************************************************
void optInternalCkt::cleanUp()
{
    map<oaNet*, optSimWire*>::iterator wireIter;
    map<oaInst*, optSimInst*>::iterator instIter;
    optSimInst* inst; 
    vector<optSimPort*>::iterator portIter;
    set<optSimWire*> wireSet;
    set<optSimWire*>::iterator setIter;

    // Free wireMap. Add to a set first because multiple nets may map to the
    // same wire due to equivalent nets.
    for (wireIter= wireMap.begin(); wireIter != wireMap.end(); wireIter++)
    {
        wireSet.insert((*wireIter).second);
    }
    for (setIter= wireSet.begin(); setIter != wireSet.end(); setIter++)
    {
        delete(*setIter);
    }
    // Free instances and ports
    for (instIter= instMap.begin(); instIter != instMap.end(); instIter++)
    {
        inst= (*instIter).second;
        for (portIter= inst->inputs.begin(); portIter != inst->inputs.end(); portIter++)
        {
            delete(*portIter);
        }
        for (portIter= inst->outputs.begin(); portIter != inst->outputs.end(); portIter++)
        {
            delete(*portIter);
        }
        delete(inst);
    }
    stateInsts.clear();
    inputNets.clear();
    outputNets.clear();
    stateNets.clear();
    wireMap.clear();
    instMap.clear();
    levelizedInst.clear();
    stateInWires.clear();
    allBitNets.clear();
}

// *****************************************************************************
// ~optInternalCkt()
//
/// \brief Destructor.
//
// *****************************************************************************
optInternalCkt::~optInternalCkt()
{
    cleanUp();
}

// *****************************************************************************
// getInputConeSimInsts()
//
/// \brief Return the SimInst in the input cones of nets.
//
// *****************************************************************************
void optInternalCkt::getInputConeSimInsts(vector<oaNet*>& nets, vector<optSimInst*>& insts)
{
    vector<oaNet*>::iterator netIter;
    vector<optSimPort*>::iterator portIter;
    vector<optSimInst*>::iterator instIter;
    optSimInst* inst, *inst2;
    optSimPort* port, *port2;
    optSimWire* wire;
    vector<optSimInst*> stack;
    vector<optSimInst*> visitedInst;

    insts.clear();
    // Get wires from nets. Add instances to stack.
    for (netIter= nets.begin(); netIter != nets.end(); netIter++)
    {
        wire= wireMap[*netIter];
        if (wire->driver && wire->driver->inst1)
        {
            inst= wire->driver->inst1;
            if (inst->visited == false)
            {
                stack.push_back(inst);
                inst->visited= true;
                visitedInst.push_back(inst);
            }
        }
    }
    while (!stack.empty())
    {
        inst= stack.back();
        stack.pop_back();
        if (inst->isStateReg == false)
        {
            insts.push_back(inst);
            // Traverse all its inputs
            for (portIter= inst->inputs.begin(); portIter != inst->inputs.end(); portIter++)
            {
                port= *portIter;
                wire= port->wire1;
                if (wire == NULL || wire->driver == NULL || wire->isPI)
                    continue;
                port2= wire->driver;
                inst2= port2->inst1;
                if (inst2->visited == false)
                {
                    inst2->visited= true;
                    visitedInst.push_back(inst2);
                    stack.push_back(inst2);
                }
            }
        }
    }
    // clear visited flag
    for (instIter= visitedInst.begin(); instIter != visitedInst.end(); instIter++)
    {
        (*instIter)->visited= false;
    }
}

// *****************************************************************************
// optSimInstsGetSimWireIO()
//
/// \brief Return the simwires of PIs/POs given insts. 
///
/// Registers' outputs are also PIs, and inputs are also POs. Note that for a register's Q and QN, only Q will be returned as input, unless only QN exists and Q does not exist. All the nets that appeared and are not PIs/POs will be returned in allNets.
//
// *****************************************************************************
void optInternalCkt::optSimInstsGetSimWireIO(vector<optSimInst*>& insts, vector<optSimWire*>& inputs, vector<optSimWire*>& outputs, vector<optSimWire*>& allNets)
{
    vector<optSimInst*>::iterator instIter;
    optSimInst* inst;
    optSimPort* port;
    optSimWire* wire;
    set<optSimWire*> wireSet, wireSet2;
    vector<optSimPort*>::iterator portIter;
    set<optSimWire*>::iterator setIter;

    // collect wires
    for (instIter= insts.begin(); instIter != insts.end(); instIter++)
    {
        inst= *instIter;
        for (portIter= inst->inputs.begin(); portIter != inst->inputs.end(); portIter++)
        {
            port= *portIter;
            wire= port->wire1;
            if (wire->isPI || wire->isPO || wire->isRegOut || (wire->isRegOutQN && wire->regQWire == NULL) || wire->isRegD)
                wireSet.insert(wire);
            else
                wireSet2.insert(wire);
        }
        for (portIter= inst->outputs.begin(); portIter != inst->outputs.end(); portIter++)
        {
            port= *portIter;
            wire= port->wire1;
            if (wire->isPI || wire->isPO || wire->isRegOut || (wire->isRegOutQN && wire->regQWire == NULL) || wire->isRegD)
                wireSet.insert(wire);
            else
                wireSet2.insert(wire);
        }
    }
    // Now put those wires to correct vectors
    for (setIter= wireSet.begin(); setIter != wireSet.end(); setIter++)
    {
        wire= *setIter;
        if (wire->isPI || wire->isRegOut || (wire->isRegOutQN && wire->regQWire == NULL))
        {
//            cout<<"Add PI: "<<getNameFromOA(wire->origWire)<<endl;
            inputs.push_back(wire);
        }
        if (wire->isPO || wire->isRegD)
        {
//            cout<<"Add PO: "<<getNameFromOA(wire->origWire)<<endl;
            outputs.push_back(wire);
        }
    }
    // Out wireSet2 to allNets
    for (setIter= wireSet2.begin(); setIter != wireSet2.end(); setIter++)
    {
        wire= *setIter;
        allNets.push_back(wire);
    }
}

// *****************************************************************************
// findWireClose()
//
/// \brief Find wires close to the origWire.
///
/// Up to maxNumber wires will be returned. OrigWire is not included in the returned wires. Wires in the output cone of origWire will not be returned.
//
// *****************************************************************************
void optInternalCkt::findWireClose(optSimWire* origWire, int maxNumber, vector<optSimWire*>& wires)
{
    list<optSimWire*> queue;
    set<optSimWire*> outputConeWires;
    optSimWire* wire2, *wire3;
    vector<optSimPort*>::iterator portIter, portIter2;
    optSimInst* inst2;
    optSimPort* port2;
    vector<optSimWire*> markedWires;
    vector<optSimWire*>::iterator wireIter;

    getOutputConeWires(origWire, outputConeWires);
    wires.clear();
    queue.push_back(origWire);
    origWire->marked= true;
    markedWires.push_back(origWire);
    while (queue.empty() == false)
    {
        wire2= queue.front();
        queue.pop_front();
        if (outputConeWires.find(wire2) == outputConeWires.end())
        {
            if (wire2 != origWire)
                wires.push_back(wire2);
            if (wires.size() > static_cast<unsigned int>(maxNumber))
                break;
        }
        // Try wire2's fanouts
        for (portIter= wire2->outputs.begin(); portIter != wire2->outputs.end(); portIter++)
        {
            port2= *portIter;
            if (port2 == NULL)  
                continue;
            inst2= port2->inst1;
            if (inst2 == NULL)
                continue;
            if (inst2->isStateReg)
                continue;
            // Add input to queue
            for (portIter2= inst2->inputs.begin(); portIter2 != inst2->inputs.end(); portIter2++)
            {
                wire3= (*portIter2)->wire1;
                if (wire3 && wire3->marked == false)
                {
                    wire3->marked= true;
                    markedWires.push_back(wire3);
                    queue.push_back(wire3);
                }
            }
            // Add output to queue
            for (portIter2= inst2->outputs.begin(); portIter2 != inst2->outputs.end(); portIter2++)
            {
                wire3= (*portIter)->wire1;
                if (wire3 && wire3->marked == false)
                {
                    wire3->marked= true;
                    markedWires.push_back(wire3);
                    queue.push_back(wire3);
                }
            }
        }
        port2= wire2->driver;
        if (port2 == NULL)  
            continue;
        inst2= port2->inst1;
        if (inst2 == NULL)
            continue;
        if (inst2->isStateReg)
            continue;
        // Add input to queue
        for (portIter= inst2->inputs.begin(); portIter != inst2->inputs.end(); portIter++)
        {
            wire3= (*portIter)->wire1;
            if (wire3 && wire3->marked == false)
            {
                wire3->marked= true;
                markedWires.push_back(wire3);
                queue.push_back(wire3);
            }
        }
        // Add output to queue
        for (portIter= inst2->outputs.begin(); portIter != inst2->outputs.end(); portIter++)
        {
            wire3= (*portIter)->wire1;
            if (wire3 && wire3->marked == false)
            {
                wire3->marked= true;
                markedWires.push_back(wire3);
                queue.push_back(wire3);
            }
        }
    }
    for (wireIter= markedWires.begin(); wireIter != markedWires.end(); wireIter++)
        (*wireIter)->marked= false;
cout<<endl;        
}

// *****************************************************************************
// getOutputConeWires
//
/// \brief Find wires in the output cone of origWire.
//
// *****************************************************************************
void optInternalCkt::getOutputConeWires(optSimWire* origWire, set<optSimWire*>& outputConeWires)
{
    list<optSimWire*> queue;
    optSimWire* wire2, *wire3;
    vector<optSimPort*>::iterator portIter, portIter2;
    optSimInst* inst2;
    vector<optSimWire*> markedWires;
    vector<optSimWire*>::iterator wireIter;

    outputConeWires.clear();
    queue.push_back(origWire);
    while (queue.empty() == false)
    {
        wire2= queue.front();
        queue.pop_front();
        outputConeWires.insert(wire2);
        for (portIter= wire2->outputs.begin(); portIter != wire2->outputs.end(); portIter++)
        {
            inst2= (*portIter)->inst1;
            if (inst2 == NULL)
                continue;
            if (inst2->isStateReg)
                continue;
            for (portIter2= inst2->outputs.begin(); portIter2 != inst2->outputs.end(); portIter2++)
            {
                wire3= (*portIter2)->wire1;
                if (wire3 == NULL)
                    continue;
                if (wire3->marked == false)
                {
                    markedWires.push_back(wire3);
                    wire3->marked= true;
                    queue.push_back(wire3);
                }
            }
        }
    }
    for (wireIter= markedWires.begin(); wireIter != markedWires.end(); wireIter++)
        (*wireIter)->marked= false;
}

// *****************************************************************************
// removePort
//
/// \brief Remove a port from the wire's fanout
//
// *****************************************************************************
void optSimWire::removePort(optSimPort* inPort)
{
    vector<optSimPort*>::iterator portIter;

    for (portIter= outputs.begin(); portIter != outputs.end(); portIter++)
    {
        if (*portIter == inPort)
        {
            outputs.erase(portIter);
            inPort->origInstTerm->removeFromNet();
            inPort->wire1= NULL;
            break;
        }
    }
}

// *****************************************************************************
// getOutputConeInstWires
//
/// \brief Find insts and wires in the output cone of origWire. 
///
/// Wires that are needed by insts are also included. Used for simulation.
//
// *****************************************************************************
void optInternalCkt::getOutputConeInstWires(optSimWire* origWire, set<optSimInst*>& insts, set<optSimWire*>& outputConeWires)
{
    list<optSimWire*> queue;
    optSimWire* wire2, *wire3;
    vector<optSimPort*>::iterator portIter, portIter2;
    optSimInst* inst2;
    vector<optSimWire*> markedWires;
    vector<optSimWire*>::iterator wireIter;

    outputConeWires.clear();
    queue.push_back(origWire);
    markedWires.push_back(origWire);
    origWire->marked= true;
    outputConeWires.insert(origWire);
    insts.clear();
    while (queue.empty() == false)
    {
        wire2= queue.front();
        queue.pop_front();
        for (portIter= wire2->outputs.begin(); portIter != wire2->outputs.end(); portIter++)
        {
            inst2= (*portIter)->inst1;
            if (inst2 == NULL)
                continue;
            if (inst2->isStateReg)
                continue;
            insts.insert(inst2);
            for (portIter2= inst2->inputs.begin(); portIter2 != inst2->inputs.end(); portIter2++)
            {
                wire3= (*portIter2)->wire1;
                if (wire3)
                    outputConeWires.insert(wire3);

            }
            for (portIter2= inst2->outputs.begin(); portIter2 != inst2->outputs.end(); portIter2++)
            {
                wire3= (*portIter2)->wire1;
                if (wire3 == NULL)
                    continue;
                if (wire3->marked == false)
                {
                    markedWires.push_back(wire3);
                    wire3->marked= true;
                    queue.push_back(wire3);
                    outputConeWires.insert(wire3);
                }
            }
        }
    }
    for (wireIter= markedWires.begin(); wireIter != markedWires.end(); wireIter++)
        (*wireIter)->marked= false;
    // for debug
    if (false)
    {
        set<optSimWire*>::iterator setIter;

        cout<<"Output Cone:"<<endl;
        for (setIter= outputConeWires.begin(); setIter != outputConeWires.end(); setIter++)
        {
            cout<<getNameFromOA((*setIter)->origWire)<<" ";
        }
        cout<<endl;        
    }
}

// *****************************************************************************
// connectWire
//
/// \brief Connect the port to a simWire. OA database will be updated.
///
// *****************************************************************************
void optSimPort::connectWire(optSimWire* newWire)
{
    oaTermType termType = origPort->getTermType();
    
    wire1= newWire;
    origInstTerm->addToNet(newWire->origWire);
    if( termType == oacOutputTermType)
    {
        newWire->driver= this;
    }
    else if( termType == oacInputTermType)
    {
        newWire->outputs.push_back(this);
    }

}

// *****************************************************************************
// removeFromWire
//
/// \brief Disconnect the port from its simWire. OA database will be updated.
///
// *****************************************************************************
void optSimPort::removeFromWire()
{
    if (wire1)
        wire1->removePort(this);
}

// *****************************************************************************
// destroy
//
/// \brief Destroy the port in OA database.
///
// *****************************************************************************
void optSimPort::destroy()
{
   origPort->destroy(); 
}

// *****************************************************************************
// destroyWire
//
/// \brief Destroy the wire in OA database.
///
// *****************************************************************************
void optInternalCkt::destroyWire(optSimWire* iWire)
{
    std::map<oa::oaNet*, optSimWire*>::iterator mapIter;
    std::set<oa::oaNet*>::iterator setIter;
    std::vector<oa::oaNet*>::iterator vecIter;
    std::vector<optSimWire*>::iterator vecSIter;

    mapIter= wireMap.find(iWire->origWire);
    wireMap.erase(mapIter);
    setIter= allBitNets.find(iWire->origWire);
    if (setIter != allBitNets.end())
        allBitNets.erase(setIter);
    vecIter= find(inputNets.begin(), inputNets.end(), iWire->origWire);
    if (vecIter != inputNets.end())
        inputNets.erase(vecIter);
    vecIter= find(outputNets.begin(), outputNets.end(), iWire->origWire);
    if (vecIter != outputNets.end())
        outputNets.erase(vecIter);
    vecIter= find(stateNets.begin(), stateNets.end(), iWire->origWire);
    if (vecIter != stateNets.end())
        stateNets.erase(vecIter);
    vecIter= find(stateInNets.begin(), stateInNets.end(), iWire->origWire);
    if (vecIter != stateInNets.end())
        stateInNets.erase(vecIter);
    vecSIter= find(stateInWires.begin(), stateInWires.end(), iWire);
    if (vecSIter != stateInWires.end())
        stateInWires.erase(vecSIter);
    iWire->origWire->destroy(); 
}

// *****************************************************************************
// addWire
//
/// \brief Add a wire to the circuit. OA database will be updated.
///
// *****************************************************************************
optSimWire* optInternalCkt::addWire(string& name)
{
    oaNet* newNet;
    optSimWire* newWire;
    
    oa::oaNativeNS ns;
    oaName name2(ns, name.c_str());
    newNet= oaNet::create(block, name2);
    newWire= new(optSimWire);
    newWire->origWire= newNet;
    allBitNets.insert(newNet);
    wireMap[newNet]= newWire;
    return newWire;
}

// *****************************************************************************
// destroyInst
//
/// \brief Destroy the inst in OA database.
///
// *****************************************************************************
void optInternalCkt::destroyInst(optSimInst* iInst)
{
    std::map<oa::oaInst*, optSimInst*>::iterator mapIter;
    vector<oa::oaInst*>::iterator vecIter;
    vector<optSimPort*>::iterator portIter;

    // Remove ports from wires first.
    for (portIter= iInst->inputs.begin(); portIter != iInst->inputs.end();
         portIter++)
        (*portIter)->removeFromWire();
    for (portIter= iInst->outputs.begin(); portIter != iInst->outputs.end();
         portIter++)
        (*portIter)->removeFromWire();
    mapIter= instMap.find(iInst->origInst);
    instMap.erase(mapIter);
    vecIter= find(stateInsts.begin(), stateInsts.end(), iInst->origInst);
    if (vecIter != stateInsts.end())
        stateInsts.erase(vecIter);
    iInst->origInst->destroy(); 
    delete iInst;
}

// *****************************************************************************
// addInst
//
/// \brief Add an inst to the circuit. OA database will be updated.
///
// *****************************************************************************
optSimInst* optInternalCkt::addInst(string& name, string& cellType)
{
    optSimInst* newInst;
    oaScalarInst* newSInst;
    oaTransform xtran(0, 0);
    oa::oaNativeNS ns;
    oaScalarName name2(ns, name.c_str());
    oaScalarName type2(ns, cellType.c_str());
    oaScalarName libName, viewName;
    libCell* modCell;
    unsigned int i;
    optSimPort* newPort;
    oaInstTerm* newTerm;
    
    design->getLibName(libName);
    design->getViewName(viewName);
    newSInst= oaScalarInst::create(block, libName, type2,
              viewName, name2, xtran);
    newInst= new(optSimInst);
    newInst->origInst= static_cast<oaInst*>(newSInst);
    instMap[newSInst]= newInst;    
    // Add ports/instTerms. They are not connected to any net yet.
    modCell= libManager.lookUpCell(cellType);
    if (modCell == NULL)
    {
        oaDesign* myDesign;
        const oaNativeNS nativeNS;
        // Not a cell in library. Search OA database
        oaScalarName cellName(nativeNS, cellType.c_str());
        oaScalarName libName, viewName;
        design->getLibName(libName);
        design->getViewName(viewName);
        myDesign= oaDesign::find(libName, cellName, viewName);
        if (myDesign == NULL)
        {
            myDesign= oaDesign::find(libName, cellName, viewName);
            if (myDesign == NULL)
            {
                oaScalarName str(nativeNS, "netlist");
                myDesign= oaDesign::find(libName, cellName, str);
                if (myDesign == NULL)
                {
                    myDesign= oaDesign::open(libName, cellName, str, 'r');
                    if (myDesign == NULL)
                    {
                        cout<<"Cannot find cell "<<cellType<<endl;
                        return NULL;
                    }
                }
            }
        }
        newSInst->setMaster(myDesign);
        oaBlock* cBlock= myDesign->getTopBlock();
        oaIter<oaNet> oaInstIter(cBlock->getNets());
        while (oaNet* net2= oaInstIter.getNext())
        {
            oaName termName;
            string name2;
            
            net2->getName(termName);
            name2= getNameFromOA(net2);
            if (name2.compare("GRND") == 0)
                continue;
            if (name2.compare("POWR") == 0)
                continue;
            if (name2.compare("tie0") == 0)
                continue;
            if (name2.compare("tie1") == 0)
                continue;
            oaIter<oaTerm> oaNTIter(net2->getTerms());
            while (oaTerm* term2= oaNTIter.getNext())
            {
                newTerm= oaInstTerm::create(NULL, newSInst, term2);
                newPort= new(optSimPort);
                newPort->inst1= newInst;
                newPort->origInstTerm= newTerm;
                newPort->origPort= newTerm->getTerm();
                oaTermType termType = newPort->origPort->getTermType();
                if (termType == oacOutputTermType)
                    newInst->outputs.push_back(newPort);
                else 
                    newInst->inputs.push_back(newPort);
            }
        }
        newInst->cell= modCell;
        return newInst;
    }
    newSInst->setMaster(modCell->design);
    // Create inputs
    for (i= 0; i < modCell->inputNo; i++)
    {
        const oa::oaName termName2(ns, static_cast<const oaChar*>(modCell->inputNames[i].c_str()));
        newTerm= oaInstTerm::create(NULL, newSInst, termName2);
        newPort= new(optSimPort);
        newPort->inst1= newInst;
        newPort->origInstTerm= newTerm;
        newPort->origPort= newTerm->getTerm();
        newInst->inputs.push_back(newPort);
    }
    // Create outputs
    for (i= 0; i < modCell->outputNames.size(); i++)
    {
        const oa::oaName termName2(ns, static_cast<const oaChar*>(modCell->outputNames[i].c_str()));
        newTerm= oaInstTerm::create(NULL, newSInst, termName2);
        newPort= new(optSimPort);
        newPort->inst1= newInst;
        newPort->origInstTerm= newTerm;
        newPort->origPort= newTerm->getTerm();
        newInst->outputs.push_back(newPort);
    }
    newInst->cell= modCell;
    return newInst;
}


// *****************************************************************************
// getInputConeInsts()
//
/// \brief Return the oaInst in the input cones of nets along with PIs.
//
// *****************************************************************************
void optInternalCkt::getInputConeInsts(vector<oaNet*>& nets, vector<oaInst*>& insts, vector<oaNet*>& inputNets)
{
    vector<oaNet*>::iterator netIter;
    vector<oaInst*> stack;
    vector<oaInst*> visitedInst;
    vector<oaInst*>::iterator instIter;
    oaInstTerm* instTerm, *driverTerm;
    oaInst* inst, *inst2;
    optSimInst* simInst2, *simInst3;
    oaNet* net2;
    set<oaNet*> inputSet;
    set<oaNet*>::iterator setIter;

    insts.clear();
    inputNets.clear();
    // Get wires from nets. Add instances to stack.
    for (netIter= nets.begin(); netIter != nets.end(); netIter++)
    {
        instTerm= util::netGetInstDriver(*netIter);
        if (instTerm)
            inst= instTerm->getInst();
        else
        {
            inputSet.insert(*netIter);
            continue;
        }
        if (inst)
        {
            simInst2= instMap[inst];
            if (simInst2->visited == false)
            {
                stack.push_back(inst);
                simInst2->visited= true;
                visitedInst.push_back(inst);
            }
            if (simInst2->isStateReg)
                inputSet.insert(*netIter);
        }
    }
    while (!stack.empty())
    {
        inst= stack.back();
        stack.pop_back();
        simInst2= instMap[inst];
        if (simInst2->isStateReg == false)
        {
            insts.push_back(inst);
            // Traverse all its inputs. 
            oaIter<oaInstTerm>   oaITIter(inst->getInstTerms(oacInstTermIterNotImplicit | oacInstTermIterEquivNets));
            for (instTerm = oaITIter.getNext(); instTerm; instTerm = oaITIter.getNext())
            {
                oaTerm* term = instTerm->getTerm();
                oaTermType termType = term->getTermType();
                if( termType == oacInputTermType)
                {
                    net2= instTerm->getNet();
                    if (net2 == NULL)
                        continue;
                    driverTerm= util::netGetInstDriver(net2);
                    if (driverTerm == NULL)
                    {
                        inputSet.insert(net2);
                        continue;
                    }
                    inst2= driverTerm->getInst();
                    simInst3= instMap[inst2];
                    if (simInst3->visited == false)
                    {
                        simInst3->visited= true;
                        visitedInst.push_back(inst2);
                        stack.push_back(inst2);
                        if (simInst3->isStateReg)
                        {
                            inputSet.insert(net2);
                        }
                    }
                }
            }
        }
    }
    // clear visited flag
    for (instIter= visitedInst.begin(); instIter != visitedInst.end(); instIter++)
    {
        simInst3= instMap[*instIter];
        simInst3->visited= false;
    }
    for (setIter= inputSet.begin(); setIter != inputSet.end(); setIter++)
        inputNets.push_back(*setIter);
}

// *****************************************************************************
// getInputSimInsts()
//
/// \brief Return up to instNO optSimInsts in the input cones of wires. Use BFS
//
// *****************************************************************************
void optInternalCkt::getInputSimInsts(std::vector<optSimWire*>& wires,
         std::vector<optSimInst*>& insts, int instNO)
{
    int instCount= 0;
    vector<optSimWire*>::iterator wireIter;
    vector<optSimInst*> markedInsts;
    vector<optSimPort*>::iterator portIter;
    vector<optSimInst*>::iterator instIter;
    list<optSimInst*> queue;
    optSimInst* inst3, *inst2;

    for (wireIter= wires.begin(); wireIter != wires.end(); wireIter++)
    {
        queue.push_back((*wireIter)->driver->inst1);
        (*wireIter)->driver->inst1->marked= true;
        markedInsts.push_back((*wireIter)->driver->inst1);
        insts.clear();
    }
    while (!queue.empty())
    {
        inst2= queue.front();
        queue.pop_front();
        insts.push_back(inst2);
        instCount++;
        if (instCount >= instNO)
            break;
        for (portIter= inst2->inputs.begin(); portIter != inst2->inputs.end();
             portIter++)
        {
            if ((*portIter)->wire1 && (*portIter)->wire1->driver)
                inst3= (*portIter)->wire1->driver->inst1;
            else
                continue;
            if (inst3->isStateReg == false && inst3->inputs.empty() == false &&
                inst3->marked == false)
            {
                inst3->marked= true;
                markedInsts.push_back(inst3);
                queue.push_back(inst3);
            }
        }
    }
    for (instIter= markedInsts.begin(); instIter != markedInsts.end(); 
         instIter++)
        (*instIter)->marked= false;
}

// *****************************************************************************
// generateBlif()
//
/// \brief Generate a blif file for the given instances and their inputs/outputs.
//
// *****************************************************************************
void optInternalCkt::generateBlif(std::string fileName, 
                                  std::vector<optSimInst*>& insts,
                                  std::vector<optSimWire*>& inputs,
                                  std::vector<optSimWire*>& outputs)
{
    fstream ofile;
    vector<optSimWire*>::iterator wireIter;
    vector<optSimInst*>::iterator instIter;
    unsigned int c;
    vector<optSimPort*>::iterator portIter;
    optSimInst* inst2;
    libCell* cell;
    
    ofile.open(fileName.c_str(), ios::out);
    ofile<<".model ckt"<<endl;
    ofile<<".inputs ";
    for (wireIter= inputs.begin(); wireIter != inputs.end(); wireIter++)
        ofile<<getNameFromOA((*wireIter)->origWire)<<" ";
    ofile<<endl<<".outputs ";
    for (wireIter= outputs.begin(); wireIter != outputs.end(); wireIter++)
        ofile<<getNameFromOA((*wireIter)->origWire)<<" ";
    ofile<<endl;
    for (instIter= insts.begin(); instIter != insts.end(); instIter++)
    {
        inst2= *instIter;
        ofile<<".names ";
        for (c= 0; c < inst2->outputs.size(); c++)
        {
            long long i, j, mintermNO;
            for (portIter= inst2->inputs.begin(); portIter != inst2->inputs.end();
                 portIter++)
            {   
                ofile<<getNameFromOA((*portIter)->wire1->origWire)<<" ";
            }
            ofile<<getNameFromOA(inst2->outputs[c]->wire1->origWire)<<endl;
            cell= inst2->cell;
            // Now generate truth table.
            mintermNO= 1 << cell->inputNames.size();
            for (i= 0; i < mintermNO; i++)
            {
                if (cell->tTables[c][i])
                {
                    for (j= 0; j < cell->inputNames.size(); j++)
                    {
                        if (i & (1 << j))
                            ofile<<"1";
                        else
                            ofile<<"0";
                    }
                    ofile<<" 1"<<endl;
                }
            }
        }
    }
    ofile<<".end";
    ofile.close();
}

// *****************************************************************************
// optSimInstsGetSubcktIO()
//
/// \brief Given a subcircuit (instances), find its input and output wires. Internal wires are returned in others.
//
// *****************************************************************************
void optInternalCkt::optSimInstsGetSubcktIO(std::vector<optSimInst*>& insts,
                                            std::vector<optSimWire*>& inputs,
                                            std::vector<optSimWire*>& outputs,
                                            std::vector<optSimWire*>& others)
{
    vector<optSimInst*>::iterator instIter;
    optSimInst* inst2;
    int i;
    vector<optSimPort*> ports;
    vector<optSimPort*>::iterator portIter;
    optSimPort* port2;
    optSimWire* wire2;
    vector<optSimWire*> wires;
    vector<optSimWire*>::iterator wireIter;
    
    inputs.clear();
    outputs.clear();
    others.clear();
    
    for (instIter= insts.begin(); instIter != insts.end(); instIter++)
    {
        inst2= *instIter;
        for (i= 0; i < 2; i++)
        {
            if (i == 0)
                ports= inst2->inputs;
            else
                ports= inst2->outputs;
            for (portIter= ports.begin(); portIter != ports.end(); portIter++)
            {
                port2= *portIter;
                wire2= port2->wire1;
                if (find(wires.begin(), wires.end(), wire2) == wires.end())
                {
                    wires.push_back(wire2);
                    wire2->tmpflag1= 0; // tmpflag1 is fanin number
                    wire2->tmpflag2= 0; // tmpflag2 is fanout number
                }
                if (i == 0)
                    wire2->tmpflag2++;
                else
                    wire2->tmpflag1++;
            }
        }
    }
    for (wireIter= wires.begin(); wireIter != wires.end(); wireIter++)
    {
        wire2= *wireIter;
        if (wire2->tmpflag1 == 0)
            inputs.push_back(wire2);
        else if (wire2->tmpflag2 < static_cast<int>(wire2->outputs.size()) && 
            wire2->tmpflag1 > 0)
            outputs.push_back(wire2);
        else
            others.push_back(wire2);
    }
}

/// Remove driver of a net
void optSimWire::removeDriver()
{
    driver->origInstTerm->removeFromNet();
    driver->wire1= NULL;
    driver= NULL;
}

// *****************************************************************************
// getAllInsts()
//
/// \brief Gets all insts in the circuit.
// *****************************************************************************
void optInternalCkt::getAllInsts(vector<oaInst*>& result) 
{
    std::map<oa::oaInst*, optSimInst*>::iterator mapIter;

    result.clear();
    for (mapIter= instMap.begin(); mapIter != instMap.end(); mapIter++)
    {
        result.push_back((*mapIter).first);
    }
}

// *****************************************************************************
// getAllCombiInsts()
//
/// \brief Gets all combinationalinsts in the circuit.
// *****************************************************************************
void optInternalCkt::getAllCombiInsts(vector<oaInst*>& result) 
{
    std::map<oa::oaInst*, optSimInst*>::iterator mapIter;

    result.clear();
    for (mapIter= instMap.begin(); mapIter != instMap.end(); mapIter++)
    {
        if ((*mapIter).second->isStateReg == false)
            result.push_back((*mapIter).first);
    }
}

// *****************************************************************************
// getAllCombiInsts()
//
/// \brief Gets all combinationalinsts in the circuit.
// *****************************************************************************
int optInternalCkt::getMaxLevel() 
{
    std::vector<optSimInst*>::iterator instIter;
    int result= 0;

    for (instIter= levelizedInst.begin(); instIter != levelizedInst.end();
         instIter++)
    {
        if ((*instIter)->level > result)
            result= (*instIter)->level;
    }
    return result;
}

} // End of namespace
// vim: ci et sw=4 sts=4
