
/*
    Author: Kai-hui Chang <changkh@eecs.umich.edu>
*/

///
/// \file oagResynLibMan.h
/// Library manager.

#if !defined(oagResynLibMan_P)
#define oagResynLibMan_P

#include "oaDesignDB.h"
#include <map>
#include <vector>
#include <set>
#include <bitset>

namespace oagResyn {

/// Set maxCellInput to the maximum number of inputs in your cell library
#define maxCellInput 8
#define tTableSize (1 << maxCellInput)

/// Stores SAT information for a cell.
class libSATCell {
public:
    unsigned int inputNO; ///< Literal NO 1 to inputNO
    unsigned int outputNO; ///< Literal NO inputNO+1 to inputNO + outputNO 
    unsigned int extraNO; ///< Number of extra literals, NO > inputNO+outputNO
    /// Each vector is a clause. Each clause is a vector, which contains 
    /// literals in the clause. 
    std::vector< std::vector<int> > clauses;
};

/// Save logic information of cells.
class libCell {
public:
    libCell () {satCell= NULL;};
    bool type; // True: use logic value. False: Use Occurrence.
    unsigned int inputNo; ///< Number of inputs
    std::vector<std::bitset<tTableSize> > tTables; ///< Truth tables for each output
    std::vector<std::string> inputNames, outputNames; ///< Input/output names
    libSATCell* satCell; ///< Points to SAT information.
    oa::oaDesign* design; ///< OADesign of the cell
};

typedef std::map<std::string, libCell*> cellMapType;

/// Manages cell library, mainly for logic information.
class LibMan {
public:
    LibMan(); 
    /// Define some basic gate types.
    enum cellLibType {BUFFER, INV, AND, OR, XOR, NAND, NOR, XNOR};
    void addCell(std::string& cellName);
    void getCellNames(std::vector<std::string>& strvec);
    libCell* lookUpCell(std::string& cellName);
    libCell* lookUpCellType(cellLibType type);
    void printLib();
    void printCell(libCell* cell);
    bool genLogicInfo(oa::oaDesign *design);
    bool useHardCodeCNF; /// If true, use hardcoded CNF 

protected:
    bool genCellLogic(libCell* myCell, oa::oaDesign *myDesign);
    bool genCellSAT(libCell* myLibCell, oa::oaDesign *myDesign);
    cellMapType cellNameMap;
    std::map<cellLibType, libCell*> cellTypeMap;
    std::vector<std::string> cellNames;
};

}
#endif
// vim: ci et sw=4 sts=4
