
/*
Author: Kai-hui Chang <changkh@eecs.umich.edu>
*/

#include "oagResynOptSim.h"
#include "oagResynUtil.h"
#include "oagResynEquiCheck.h"
#include "oagResynSATIntf.h"
#include "oagResynPrimeMap.h"
#include <sys/times.h>
#include <unistd.h>

using namespace std;
using namespace oa;

namespace oagResyn {

// *****************************************************************************
// equiCheck()
//
/// \brief Constructor
//
// *****************************************************************************
equiCheck::equiCheck()
{
    randomPatternNumber= 1024;
    currVectorNumber= 0;
    designs[0]= NULL;
    designs[1]= NULL;
    nets[0]= NULL;
    nets[1]= NULL;
    showRuntime= false;
}


// *****************************************************************************
// checkEquiTwoNets()
//
/// \brief Check equivalency of two nets. 
///
/// \param design oaDesign
/// \param net1 first net to be checked
/// \param net2 second net to be checked
/// \param vecNumber the number of counterexamples required; the actual vectors returned will be returned by this argument
/// \param inputs returns the primary inputs in the fanin of net1 and net2
/// \param inputVecs returns the input vectors that can distinguish the nets
/// \param outVec1 returns the simulation value for net1 using counterexamples
/// \param outVec2 returns the simulation value for net2 using counterexamples
/// \return If equivalent, return true; otherwise return false. 
//
// *****************************************************************************
bool equiCheck::checkEquiTwoNets(oaDesign* design, oaNet* net1, oaNet* net2,
                             int& vecNumber, vector<oaNet*>& inputs,
                             vector<SimulationVector>& inputVecs,
                             SimulationVector& outVec1, 
                             SimulationVector& outVec2)
{
    bool result;
    
    nets[0]= net1;
    nets[1]= net2;
    designs[0]= design;
    if (randomPatternNumber > 0)
    {
        result= checkEquiSimSat(design, net1, net2, vecNumber, inputs, 
                               inputVecs, outVec1, outVec2);
    }
    else
    {
        result= checkEquiSat(design, net1, net2, vecNumber, inputs, 
                               inputVecs, outVec1, outVec2);
    }
    counterExampleNO= vecNumber;
    PI1= inputs;
    counterExamples= inputVecs;
    counterExampleOutput1.push_back(outVec1);
    counterExampleOutput2.push_back(outVec2);
    PO1.push_back(net1);
    PO2.push_back(net2);
    twoDesign= false;
    twoNet= true;
    equi= result;
    return result;
}

// *****************************************************************************
// checkEquiSat()
//
/// \brief Check equivalency of two nets using pure SAT. 
///
/// Called by checkEquiTwoNets.
//
// *****************************************************************************
bool equiCheck::checkEquiSat(oaDesign* design, oaNet* net1, oaNet* net2,
                             int& vecNumber, vector<oaNet*>& inputs,
                             vector<SimulationVector>& inputVecs,
                             SimulationVector& outVec1, 
                             SimulationVector& outVec2)
{
    vector<oaNet*> nets;
    vector<oaInst*> insts;
    clauseType outputCNF;
    clauseType comparatorCNF;
    map<oaNet*, int> litNOMap;
    int largestLitNO, litNO2;
    vector<int> input1, input2;
    int outputLitNO;
    vector<int> clause;
    oagResynSATIntf mySat;
    optInternalCkt* myCkt;
    bool result;
    int foundVecNumber= 0;
    unsigned int i;

    if (net1 == net2)
        return true;
    if (vecNumber > static_cast<int>(SimVectorSize))
    {
        cout<<"Can only return <= "<<SimVectorSize<<" vectors."<<endl;
        vecNumber= SimVectorSize;
    }
    myCkt= cktManager.designGetOptInternalCkt(design);
    nets.push_back(net1);
    nets.push_back(net2);
    myCkt->getInputConeInsts(nets, insts, inputs);
    mySatCkt.genInsts(design, 1, insts, outputCNF, litNOMap, largestLitNO);
    input1.push_back(litNOMap[net1]);
    input2.push_back(litNOMap[net2]);
    mySatCkt.genComparator(largestLitNO+1, input1, input2, comparatorCNF, outputLitNO);
    outputCNF.insert(outputCNF.end(), comparatorCNF.begin(), comparatorCNF.end());
    // Force output to 1
    clause.push_back(outputLitNO);
    outputCNF.push_back(clause);
    //printEquiCheckCNF(outputCNF, litNOMap, outputLitNO);
    // Call SAT and see if it is satisfiable...
    mySat.addCNF(outputCNF);
    mySat.solve();
    result= !(mySat.isSat());
    inputVecs.clear();
    if (mySat.isSat() == true && vecNumber > 0)
    {
        for (i= 0; i < inputs.size(); i++)
            inputVecs.push_back(0);
        outVec1= 0;
        outVec2= 0;
    }
    // Need to build input/output vectors
    while (mySat.isSat() == true && foundVecNumber < vecNumber)
    {
        // Get input vectors and output responses
        for (i= 0; i < inputVecs.size(); i++)
        {
            if (mySat.getLitValue(litNOMap[inputs[i]]) == 1)
                inputVecs[i] |= (1 << foundVecNumber);
        }
        if (mySat.getLitValue(litNOMap[net1]) == 1)
            outVec1 |= (1 << foundVecNumber);
        if (mySat.getLitValue(litNOMap[net2]) == 1)
            outVec2 |= (1 << foundVecNumber);
        foundVecNumber++;
        // add -inputVecs and solve again
        for (i= 0; i < inputVecs.size(); i++)
        {
            litNO2= litNOMap[inputs[i]];
            if (mySat.getLitValue(litNO2) == 1)
                clause.push_back(-litNO2);
            else
                clause.push_back(litNO2);
        }
        mySat.addClause(clause);
        clause.clear();
        mySat.solve();
    }
    vecNumber= foundVecNumber;
    return result;
}

// *****************************************************************************
// checkEquiSimSat()
//
/// \brief Check equivalency of two nets using simulation + SAT. 
///
/// Called by checkEquiTwoNets.
//
// *****************************************************************************
bool equiCheck::checkEquiSimSat(oaDesign* design, oaNet* net1, oaNet* net2,
                                int& vecNumber, vector<oaNet*>& inputs,
                                vector<SimulationVector>& inputVecs,
                                SimulationVector& outVec1, 
                                SimulationVector& outVec2)
{
    unsigned int i, j;
    int diffVecNumber;
    optInternalCkt* myCkt;
    optSimWire* wire1, *wire2, *iwire;
    vector<oaNet*> nets;
    vector<oaInst*> insts;
    vector<oaNet*>::iterator netIter;
    int bitmask;
    bool result;
   
    diffVecNumber= 0;
    outVec1= 0;
    outVec2= 0;
    inputs.clear();
    inputVecs.clear();
    myCkt= cktManager.designGetOptInternalCkt(design);
    if (currVectorNumber == 0)
        preSimulation(design);
    if (vecNumber > static_cast<int>(SimVectorSize))
        vecNumber= SimVectorSize;

    nets.push_back(net1);
    nets.push_back(net2);
    myCkt->getInputConeInsts(nets, insts, inputs);
    for (i= 0; i < inputs.size(); i++)
        inputVecs.push_back(0);
    // Compare signatures
    wire1= myCkt->netGetSimWire(net1);
    wire2= myCkt->netGetSimWire(net2);
    
    for (i= 0; i < currVectorNumber; i++)
    {
        if (util::intVecGetBit(wire1->signature, i) != util::intVecGetBit(wire2->signature, i))
        {
            bitmask= 1 << diffVecNumber;
            if (util::intVecGetBit(wire1->signature, i))
                outVec1 |= bitmask;
            if (util::intVecGetBit(wire2->signature, i))
                outVec2 |= bitmask;
            j= 0;
            for (netIter= inputs.begin(); netIter != inputs.end(); netIter++)
            {
                iwire= myCkt->netGetSimWire(*netIter);
                if (util::intVecGetBit(iwire->signature, i))
                {
                    inputVecs[j] |= bitmask;
                }
                j++;
            }
            diffVecNumber++;
            if (diffVecNumber == vecNumber)
                break;
        }
    }
    if (diffVecNumber > 0)  
    {
        vecNumber= diffVecNumber;
        return false;
    }
    // Simulation cannot distinguish two nets. Need to use SAT.
    result= checkEquiSat(design, net1, net2, vecNumber, inputs, inputVecs,
                         outVec1, outVec2);
    // Add vectors for simulation
    addPattern(design, inputs, inputVecs, vecNumber);

    return result;
}

// *****************************************************************************
// printEquiCheckCNF()
//
/// \brief Print equivalence checking CNF info. For debugging.
//
// *****************************************************************************
void equiCheck::printEquiCheckCNF(clauseType& inputCNF, map<oaNet*, int>& litNOMap, int outputLitNO)
{
    map<oaNet*, int>::iterator mapIter;
    oaNet* wire1;
    
    cout<<"Output is "<<outputLitNO<<endl;
    cout<<"Wire map is "<<endl;
    for (mapIter= litNOMap.begin(); mapIter != litNOMap.end(); mapIter ++)
    {
        wire1= (*mapIter).first;
        cout<<getNameFromOA(wire1)<<" "<<(*mapIter).second<<endl;
    }
    cout<<"CNF is "<<endl;
    mySatCkt.printClause(inputCNF);
}

// *****************************************************************************
// checkTwoCkts()
//
/// \brief Check equivalency of two circuits. 
///
/// \param design1 oaDesign for design 1
/// \param insts1 set of instances in design 1
/// \param inputs1 inputs to the set of instances in design 1
/// \param outputs1 outputs of the set of instances in design 1
/// \param design2 oaDesign for design 2
/// \param insts2 set of instances in design 2
/// \param inputs2 inputs to the set of instances in design 2
/// \param outputs2 outputs of the set of instances in design 2
/// \param vecNumber the number of counterexamples required; the actual vectors returned will be returned by this argument
/// \param inputVecs returns the input vectors that can distinguish the nets
/// \param outputVec1 returns the output vectors for design 1 
/// \param outputVec2 returns the output vectors for design 2 
/// \return true if equivalent.
//
// *****************************************************************************
bool equiCheck::checkTwoCkts(oaDesign* design1,
                            vector<oaInst*>& insts1, vector<oaNet*>& inputs1,
                            vector<oaNet*>& outputs1,
                            oaDesign* design2,
                            vector<oaInst*>& insts2, vector<oaNet*>& inputs2,
                            vector<oaNet*>& outputs2,
                            int& vecNumber, vector<SimulationVector>& inputVecs,
                            vector<SimulationVector>& outputVec1, 
                            vector<SimulationVector>& outputVec2)
{
    clauseType outputCNF, simCNF;
    clauseType comparatorCNF;
    vector<int> clause, inputNO1, inputNO2;
    map<oaNet*, int> litNOMap1, litNOMap2;
    int largestLitNO;
    int litNO, outputLitNO;
    unsigned int i;
    bool result;
    oagResynSATIntf mySat;
    int foundVecNumber= 0;
    int litNO2;
    optInternalCkt *ckt1, *ckt2;
   
    designs[0]= design1;
    designs[1]= design2;
    assert(inputs1.size() == inputs2.size());
    assert(outputs1.size() == outputs2.size());
    if (vecNumber > static_cast<int>(SimVectorSize))
    {
        cout<<"Can only return <= "<<SimVectorSize<<" vectors."<<endl;
        vecNumber= SimVectorSize;
    }
    ckt1= cktManager.designGetOptInternalCkt(design1);
    ckt2= cktManager.designGetOptInternalCkt(design2);
    inputVecs.clear();
    outputVec1.clear();
    outputVec2.clear();
    mySatCkt.genInsts(design1, 1, insts1, outputCNF, litNOMap1, largestLitNO);
    mySatCkt.genInsts(design2, largestLitNO + 1, insts2, simCNF, litNOMap2, largestLitNO);
    outputCNF.insert(outputCNF.end(), simCNF.begin(), simCNF.end());
    litNO= largestLitNO + 1;
    // Check inputs/outputs. If anything is not used, just assign a new
    // litNO
    for (i= 0; i < inputs1.size(); i++)
    {
        map<oaNet*, int>::iterator mapIter;

        mapIter= litNOMap1.find(inputs1[i]);
        if (mapIter == litNOMap1.end())
            litNOMap1[inputs1[i]]= portGetLitNO(inputs1[i], litNOMap1, litNO);
        mapIter= litNOMap2.find(inputs2[i]);
        if (mapIter == litNOMap2.end())
            litNOMap2[inputs2[i]]= portGetLitNO(inputs2[i], litNOMap2, litNO);
    }
    for (i= 0; i < outputs1.size(); i++)
    {
        map<oaNet*, int>::iterator mapIter;

        mapIter= litNOMap1.find(outputs1[i]);
        if (mapIter == litNOMap1.end())
            litNOMap1[outputs1[i]]= portGetLitNO(outputs1[i], litNOMap1, litNO);
        mapIter= litNOMap2.find(outputs2[i]);
        if (mapIter == litNOMap2.end())
            litNOMap2[outputs2[i]]= portGetLitNO(outputs2[i], litNOMap2, litNO);
    }
    // Handle tie0/tie1
    {
        map<oaNet*, int>::iterator mapIter;
        for (mapIter= litNOMap1.begin(); mapIter != litNOMap1.end(); mapIter++)
        {
            if (tie0.compare(getNameFromOA((*mapIter).first)) == 0)
            {
                clause.push_back(-(*mapIter).second);
                outputCNF.push_back(clause);
                clause.clear();
            }
        }
        for (mapIter= litNOMap1.begin(); mapIter != litNOMap1.end(); mapIter++)
        {
            if (tie1.compare(getNameFromOA((*mapIter).first)) == 0)
            {
                clause.push_back((*mapIter).second);
                outputCNF.push_back(clause);
                clause.clear();
            }
        }
        for (mapIter= litNOMap2.begin(); mapIter != litNOMap2.end(); mapIter++)
        {
            if (tie0.compare(getNameFromOA((*mapIter).first)) == 0)
            {
                clause.push_back(-(*mapIter).second);
                outputCNF.push_back(clause);
                clause.clear();
            }
        }
        for (mapIter= litNOMap2.begin(); mapIter != litNOMap2.end(); mapIter++)
        {
            if (tie1.compare(getNameFromOA((*mapIter).first)) == 0)
            {
                clause.push_back((*mapIter).second);
                outputCNF.push_back(clause);
                clause.clear();
            }
        }
    }
    // connect inputs.
    for (i= 0; i < inputs1.size(); i++)
    {
        clause.push_back(litNOMap1[inputs1[i]]);
        clause.push_back(-litNOMap2[inputs2[i]]);
        outputCNF.push_back(clause);
        clause.clear();
        clause.push_back(-litNOMap1[inputs1[i]]);
        clause.push_back(litNOMap2[inputs2[i]]);
        outputCNF.push_back(clause);
        clause.clear();
    }
    for (i= 0; i < outputs1.size(); i++)
    {
        inputNO1.push_back(litNOMap1[outputs1[i]]);
        inputNO2.push_back(litNOMap2[outputs2[i]]);
    }
    // Add comparator to outputs
    mySatCkt.genComparator(litNO, inputNO1, inputNO2, comparatorCNF, outputLitNO);
    outputCNF.insert(outputCNF.end(), comparatorCNF.begin(), comparatorCNF.end());
    // Force output to 1
    clause.push_back(outputLitNO);
    outputCNF.push_back(clause);
    clause.clear();
    mySat.addCNF(outputCNF);
    /* For debug 
    cout<<endl<<"Map1:"<<endl;
    printEquiCheckCNF(outputCNF, litNOMap1, outputLitNO);
    cout<<"Map2:"<<endl;
    printEquiCheckCNF(outputCNF, litNOMap2, outputLitNO);
       End of debug */
    
    mySat.solve();
    
    /* For debug 
    cout<<"Solution:";
    for (int i= 1; i <= outputLitNO; i++)
    {
        cout<<" ";
        if (mySat.getLitValue(i) == -1)
            cout<<"-";
        cout<<i;
    }
    * End of debug */

    result= !mySat.isSat();
    if (mySat.isSat() == true && vecNumber > 0)
    {
        // build inputVecs, outputVecs
        for (i= 0; i < inputs1.size(); i++)
            inputVecs.push_back(0);
        for (i= 0; i < outputs1.size(); i++)
        {
            outputVec1.push_back(0);
            outputVec2.push_back(0);
        }
    }
    // Need to build input/output vectors
    while (mySat.isSat() == true && foundVecNumber < vecNumber)
    {
        // Get input vectors and output responses
        for (i= 0; i < inputVecs.size(); i++)
        {
            if (mySat.getLitValue(litNOMap1[inputs1[i]]) == 1)
                inputVecs[i] |= (1 << foundVecNumber);
        }
        for (i= 0; i < outputVec1.size(); i++)
        {
            if (mySat.getLitValue(inputNO1[i]) == 1)
                outputVec1[i] |= (1 << foundVecNumber);
            if (mySat.getLitValue(inputNO2[i]) == 1)
                outputVec2[i] |= (1 << foundVecNumber);
        }
        foundVecNumber++;
        // add -inputVecs and solve again
        for (i= 0; i < inputVecs.size(); i++)
        {
            litNO2= litNOMap1[inputs1[i]];
            if (mySat.getLitValue(litNO2) == 1)
            {
                clause.push_back(-litNO2);
            }
            else 
            {
                clause.push_back(litNO2);
            }
        }
        mySat.addClause(clause);
        clause.clear();
        mySat.solve();
    }
    vecNumber= foundVecNumber;
    return result;
}

// *****************************************************************************
// checkTwoDesigns()
//
/// \brief Check equivalency between two designs.
///
/// Similar to checkTwoCkts except the whole design will be checked, and inputs/output must be primary inputs/outputs of the whole design.
//
// *****************************************************************************
bool equiCheck::checkTwoDesigns(oaDesign* design1,
                            vector<oaNet*>& inputs1,
                            vector<oaNet*>& outputs1,
                            oaDesign* design2,
                            vector<oaNet*>& inputs2,
                            vector<oaNet*>& outputs2,
                            int& vecNumber, vector<SimulationVector>& inputVecs,
                            vector<SimulationVector>& outputVec1, 
                            vector<SimulationVector>& outputVec2)
{
    optInternalCkt *ckt1, *ckt2;
    vector<oaInst*> insts1, insts2;
    vector<oaNet*> nets1, nets2;
    vector<optSimWire*> inputSim1, outputSim1, outputSim2;
    unsigned int i, j;
    bool mismatch;
    int bitmask;
    int diffVecNumber= 0;

    designs[0]= design1;
    designs[1]= design2;
    ckt1= cktManager.designGetOptInternalCkt(design1);
    ckt2= cktManager.designGetOptInternalCkt(design2);
    if (inputs1.size() != inputs2.size())
    {
        cout<<"Input sizes are not the same, cannot do equivalence checking."<<endl;
        return false;
    }
    if (outputs1.size() != outputs2.size())
    {
        cout<<"Output sizes are not the same, cannot do equivalence checking."<<endl;
        return false;
    }
    if (randomPatternNumber > 0)
    {
        // Clear previous signatures (if any)
        clearSignature(design1);
        clearSignature(design2);
        // Perform simulation first
        inputVecs.insert(inputVecs.end(), inputs1.size(), 0);
        outputVec1.insert(outputVec1.end(), outputs1.size(), 0);
        outputVec2.insert(outputVec2.end(), outputs2.size(), 0);
        preSimulateTwoDesigns(design1, inputs1, design2, inputs2);
        for (i= 0; i < inputs1.size(); i++)
        {
            inputSim1.push_back(ckt1->netGetSimWire(inputs1[i]));
        }
        for (i= 0; i < outputs1.size(); i++)
        {
            outputSim1.push_back(ckt1->netGetSimWire(outputs1[i]));
            outputSim2.push_back(ckt2->netGetSimWire(outputs2[i]));
        }
        for (i= 0; i < currVectorNumber; i++)
        {
            mismatch= false;
            for (j= 0; j < outputs1.size(); j++)
            {
                if (util::intVecGetBit(outputSim1[j]->signature, i) !=
                    util::intVecGetBit(outputSim2[j]->signature, i))
                {
//cout<<"Wire: "<<getNameFromOA(outputSim1[j]->origWire)<<", "<<util::intVecGetBit(outputSim1[j]->signature, i)<<endl;
                    mismatch= true;
                    break;
                }
            }
            if (mismatch == true)
            {
                bitmask= 1 << diffVecNumber;
                for (j= 0; j < outputs1.size(); j++)
                {
                    if (util::intVecGetBit(outputSim1[j]->signature, i))
                        outputVec1[j] |= bitmask;
                    if (util::intVecGetBit(outputSim2[j]->signature, i))
                        outputVec2[j] |= bitmask;
                }
                for (j= 0; j < inputs1.size(); j++)
                {
                    if (util::intVecGetBit(inputSim1[j]->signature, i))
                        inputVecs[j] |= bitmask;
                }
                diffVecNumber++;
                if (diffVecNumber == vecNumber)
                    break;
            }
        }
        if (diffVecNumber > 0)  
        {
            vecNumber= diffVecNumber;
            return false;
        }
    }
    ckt1->getInputConeInsts(outputs1, insts1, nets1);
    ckt2->getInputConeInsts(outputs2, insts2, nets2);
    return checkTwoCkts(design1, insts1, inputs1, outputs1, design2,
                       insts2, inputs2, outputs2,
                       vecNumber, inputVecs, outputVec1, outputVec2);
}

// *****************************************************************************
// portGetLitNO()
//
/// \brief Get literal NO of a port
///
/// Given a port (not used by any instance), search for equivalent net and return its LitNO.
//
// *****************************************************************************
int equiCheck::portGetLitNO(oaNet* inNet, map<oaNet*, int>& litNOMap, int& litNO)
{
    map<oaNet*, int>::iterator mapIter;
    oaBitNet*  bitNet2, *bitNet;
    
    bitNet= static_cast<oaBitNet*>(inNet);
    oaIter<oaBitNet>   oaBNIter(bitNet->getEquivalentNets());
    bitNet2= oaBNIter.getNext();
    do
    {
        mapIter= litNOMap.find(bitNet2);
        if (mapIter != litNOMap.end())
        {
            return (*mapIter).second;
        }
        bitNet2 = oaBNIter.getNext();
    } while (bitNet2);
    litNOMap[inNet]= litNO++;
    return litNO - 1;
}

// *****************************************************************************
// getNetSignature()
//
/// \brief Return the signature of a net.
//
// *****************************************************************************
void equiCheck::getNetSignature(oaDesign* design, oaNet* net, vector<SimulationVector>& signature)
{
    optInternalCkt* myCkt;
    optSimWire* myWire;

    myCkt= cktManager.designGetOptInternalCkt(design);
    myWire= myCkt->netGetSimWire(net);
    signature= myWire->signature;
}

// *****************************************************************************
// preSimulation()
//
/// \brief Pre-simulate the circuit for randomPatternNumber patterns and generate signatures.
//
// *****************************************************************************
void equiCheck::preSimulation(oaDesign* design)
{
    optInternalCkt* myCkt;
    myCkt= cktManager.designGetOptInternalCkt(design);
    unsigned int step, last, i, size;
    optSim mySim(design);
    vector<oaNet*> allBitNets;
    optSimWire* myWire;
    vector<oaNet*>::iterator netIter;
    step= randomPatternNumber / SimVectorSize;
    last= randomPatternNumber % (SimVectorSize);
    if (last)   
        step++;
        
    myCkt->getAllNets(allBitNets);
    for (i= 0; i < step; i++)
    {
        if (last && step == (i - 1))
            size= last;
        else
            size= SimVectorSize;
        // Do random simulation
        mySim.doRandomSimulation();
        for (netIter= allBitNets.begin(); netIter != allBitNets.end(); netIter++)
        {
            myWire= myCkt->netGetSimWire(*netIter);
            if (myWire)
            {
                util::vecAddInt(myWire->signature, myWire->simVec, 
                                currVectorNumber, size);
            }
        }
        currVectorNumber+= size;
    }
}

// *****************************************************************************
// addPattern()
//
/// \brief Simulate additional patterns.
///
/// vecNumber "patterns" corresponding to "inputs" will be simulated, and vecNumber must be <= SimVectorSize.
//
// *****************************************************************************
void equiCheck::addPattern(oaDesign* design, vector<oaNet*>& inputs,
                           vector<SimulationVector>& patterns, int vecNumber)
{
    optInternalCkt* myCkt;
    optSimWire* myWire;
    vector<oaNet*>::iterator netIter;
    unsigned int i;
    optSim mySim(design);
    vector<oaNet*> allBitNets;

    myCkt= cktManager.designGetOptInternalCkt(design);
    // Set input values
    i= 0;
    for (netIter= inputs.begin(); netIter != inputs.end(); netIter++)
    {
        myCkt->setVector(*netIter, patterns[i], 0);
    }
    mySim.runFull();
    // Add simVec to signature
    myCkt->getAllNets(allBitNets);
    for (netIter= allBitNets.begin(); netIter != allBitNets.end(); netIter++)
    {
        myWire= myCkt->netGetSimWire(*netIter);
        if (myWire)
        {
            util::vecAddInt(myWire->signature, myWire->simVec, currVectorNumber,
                            vecNumber);
        }
    }
    currVectorNumber+= vecNumber;
}

// *****************************************************************************
// checkTwoDesignsAutoMap()
//
/// \brief Check two designs, ports will be automatically mapped. 
///
/// Inputs/outputs with identical names between both designs will be mapped automatically. The nets sill be returned in netInput/netOutput. Counterexamples will also be returned. The number of required counterexamples should be specified in vecNumber, and the actual number returned will be updated there.
//
// *****************************************************************************
bool equiCheck::checkTwoDesignsAutoMap(oaDesign* design1, oaDesign* design2,
                        vector<oaNet*>& netInput1, vector<oaNet*>& netOutput1,
                        vector<oaNet*>& netInput2, vector<oaNet*>& netOutput2,
                        int& vecNumber, vector<SimulationVector>& inputVecs,
                        vector<SimulationVector>& outputVec1, 
                        vector<SimulationVector>& outputVec2)
{
    optInternalCkt* ckt1, *ckt2;
    vector<oaNet*> netStateIn1, netStateIn2, netStateOut1, netStateOut2, tmpNet;
    vector<oaInst*> insts1, insts2;
    vector<oaNet*>::iterator wireIter;
    bool result;
    oaBlock* block2;
    struct tms tv1, tv2;
  
    if (showRuntime)
        times(&tv1);
    designs[0]= design1;
    designs[1]= design2;
    ckt1= cktManager.designGetOptInternalCkt(design1);
    ckt2= cktManager.designGetOptInternalCkt(design2);
    block2= design2->getTopBlock();
    ckt1->getOutputs(netOutput1);
    ckt1->getInputs(netInput1);
    ckt1->getStateInNets(netStateIn1);
    ckt2->getStateInNets(netStateIn2);
    ckt1->getStateNets(netStateOut1);
    netOutput1.insert(netOutput1.end(), netStateIn1.begin(), netStateIn1.end());
    netInput1.insert(netInput1.end(), netStateOut1.begin(), netStateOut1.end());

    for (wireIter= netInput1.begin(); wireIter != netInput1.end(); wireIter++)
    {
        oaNet* net2;
        oaName name1;
        
        (*wireIter)->getName(name1);
        net2= oaNet::find(block2, name1);
        if (net2 == NULL)
        {
            cout<<"Cannot find net "<<getNameFromOA(*wireIter)<<" in design 2."<<endl;
            equi= false;
            return false;
        }
        netInput2.push_back(net2);
    }
    for (wireIter= netOutput1.begin(); wireIter != netOutput1.end(); wireIter++)
    {
        oaNet* net2;
        oaName name1;
        
        (*wireIter)->getName(name1);
        net2= oaNet::find(block2, name1);
        if (net2 == NULL)
        {
            cout<<"Cannot find net "<<getNameFromOA(*wireIter)<<" in design 2."<<endl;
            equi= false;
            return false;
        }
        netOutput2.push_back(net2);
    }

    result= checkTwoDesigns(design1, netInput1, netOutput1,
                        design2, netInput2, netOutput2,
                        vecNumber, inputVecs, outputVec1, outputVec2);
    PI1= netInput1;
    PO1= netOutput1;
    PI2= netInput2;
    PO2= netOutput2;
    counterExampleNO= vecNumber;
    counterExamples= inputVecs;
    counterExampleOutput1= outputVec1;
    counterExampleOutput2= outputVec2;
    equi= result;
    twoDesign= true;
    twoNet= false;
    if (showRuntime)
    {
        double t;
        times(&tv2);
        t= (double)(tv2.tms_utime - tv1.tms_utime)/ sysconf(_SC_CLK_TCK);
        cout<<"Equivalence checking time: "<<t<<endl;
    }
    return result;
}

// *****************************************************************************
// preSimulateTwoDesigns()
//
/// \brief Pre-simulate two circuits for randomPatternNumber patterns and generate signatures. 
///
/// The same inputs will be applied.
//
// *****************************************************************************
void equiCheck::preSimulateTwoDesigns(oaDesign* design1, vector<oaNet*>& inputs1,
                                    oaDesign* design2, vector<oaNet*>& inputs2)
{
    optInternalCkt* myCkt1, *myCkt2;
    myCkt1= cktManager.designGetOptInternalCkt(design1);
    myCkt2= cktManager.designGetOptInternalCkt(design2);
    unsigned int step, last, i, size, j;
    optSim mySim1(design1);
    optSim mySim2(design2);
    vector<oaNet*> allBitNets1, allBitNets2;
    optSimWire* myWire;
    vector<oaNet*>::iterator netIter;

    step= randomPatternNumber / SimVectorSize;
    last= randomPatternNumber % (SimVectorSize);
    if (last)   
        step++;
        
    myCkt1->getAllNets(allBitNets1);
    myCkt2->getAllNets(allBitNets2);
    for (i= 0; i < step; i++)
    {
        if (last && step == (i - 1))
            size= last;
        else
            size= SimVectorSize;
        // Apply inputs to both
        for (j= 0; j < inputs1.size(); j++)
        {
            SimulationVector vec;
            vec= mySim1.randomVector();
            
            mySim1.setVector(inputs1[j], vec, 0);
            mySim2.setVector(inputs2[j], vec, 0);
        }
        
        // Do simulation
        mySim1.runFull();
        mySim2.runFull();
        for (netIter= allBitNets1.begin(); netIter != allBitNets1.end(); netIter++)
        {
            myWire= myCkt1->netGetSimWire(*netIter);
            if (myWire)
            {
                util::vecAddInt(myWire->signature, myWire->simVec, 
                                currVectorNumber, size);
            }
        }
        for (netIter= allBitNets2.begin(); netIter != allBitNets2.end(); netIter++)
        {
            myWire= myCkt2->netGetSimWire(*netIter);
            if (myWire)
            {
                util::vecAddInt(myWire->signature, myWire->simVec, 
                                currVectorNumber, size);
            }
        }
        currVectorNumber+= size;
    }
}

// *****************************************************************************
// getSimilarityFactor()
//
/// \brief Get similarity factor. 
///
/// Let N be the total number of signals (wires) used by the two circuits. Out of those N signals, we distinguish M matching signals --- a signal is considered matching if and only if both circuits include signals with an identical signature.  Then, the similarity factor between ckt1 and ckt2 is (M / N) * 100%. This metric is computed by our tools and returned to the user.
//
// *****************************************************************************
double equiCheck::getSimilarityFactor(oaDesign* design1, oaDesign* design2,
                                      vector<oaNet*>& netInput1, 
                                      vector<oaNet*>& netInput2)
{
    unsigned int step, i;
    vector<simFactorSort> vec1, vec2;
    vector<simFactorSort>::iterator vecIter1, vecIter2;
    unsigned int count= 0;
    vector<oaNet*> allBitNets1, allBitNets2;
    vector<oaNet*>::iterator netIter;
    optInternalCkt* myCkt1, *myCkt2;
    optSimWire* myWire;
    SimulationVector vec;
    simFactorSort tmpSFS;
    bool inc1, inc2;
    struct tms tv1, tv2;
   
    if (showRuntime)
        times(&tv1);
    myCkt1= cktManager.designGetOptInternalCkt(design1);
    myCkt2= cktManager.designGetOptInternalCkt(design2);
    
    if (netInput1.size() != netInput2.size() || netInput1.empty())
    {
        cout<<"Number of input is different or empty."<<endl;
        similarityFactor= 0.0;
        return 0.0;
    }
    if (currVectorNumber == 0)
        preSimulateTwoDesigns(design1, netInput1, design2, netInput2);

    step= randomPatternNumber / SimVectorSize;
    
    myCkt1->getAllNets(allBitNets1);
    myCkt2->getAllNets(allBitNets2);
    

    for (netIter= allBitNets1.begin(); netIter != allBitNets1.end(); netIter++)
    {
        myWire= myCkt1->netGetSimWire(*netIter);
        if (myWire)
        {
            vec= 0;
            for (i= 0; i < step; i++)
            {
                vec ^= myWire->signature[i] * PRIME_NUMBERS[i % 1000];
            }
            tmpSFS.vec= vec;
            tmpSFS.net= *netIter;
            vec1.push_back(tmpSFS);
        }
    }
    for (netIter= allBitNets2.begin(); netIter != allBitNets2.end(); netIter++)
    {
        myWire= myCkt2->netGetSimWire(*netIter);
        if (myWire)
        {
            vec= 0;
            for (i= 0; i < step; i++)
            {
                vec ^= myWire->signature[i] * PRIME_NUMBERS[i % 1000];
            }
            tmpSFS.vec= vec;
            tmpSFS.net= *netIter;
            vec2.push_back(tmpSFS);
        }
    }
    // Sort both vectors
    sort(vec1.begin(), vec1.end());
    sort(vec2.begin(), vec2.end());
    vecIter1= vec1.begin();
    vecIter2= vec2.begin();
     
    do
    {
        if ((*vecIter1).vec < (*vecIter2).vec)
        {
            inc1= true;
            inc2= false;
        }
        else if ((*vecIter1).vec > (*vecIter2).vec)
        {
            inc1= false;
            inc2= true;
        }
        else 
        {
            inc1= true;
            inc2= true;
        }
        if (inc1)
        {
            vec= (*vecIter1).vec;
            while (vecIter1 != vec1.end())
            {
                if (inc2 == false)
                {
                    count++;
                    diffNets2.push_back((*vecIter2).net);
                }
                vecIter1++;
                if( vecIter1 == vec1.end() || vec != (*vecIter1).vec)
                    break;
            }
        }
        if (inc2)
        {
            vec= (*vecIter2).vec;
            while (vecIter2 != vec2.end())
            {
                if (inc1 == false)
                {
                    count++;
                    diffNets1.push_back((*vecIter2).net);
                }
                vecIter2++;
                if (vecIter2 == vec2.end() || vec != (*vecIter2).vec)
                    break;
            }
        }
    } while (vecIter1 != vec1.end() && vecIter2 != vec2.end());
    similarityFactor= (1.0 - static_cast<double>(count) / static_cast<double>(vec1.size() + vec2.size())) * 100.0;
    if (showRuntime)
    {
        double t;
        times(&tv2);
        t= (double)(tv2.tms_utime - tv1.tms_utime)/ sysconf(_SC_CLK_TCK);
        cout<<"Similarity factor calculation time: "<<t<<endl;
    }
    // Add instance level to map
    {
        oaBlock* block= design1->getTopBlock();
        
        oaIter<oaInst>   oaInstIter(block->getInsts());
        while (oaInst*  inst = oaInstIter.getNext())
        {
            if (myCkt1->instGetSimInst(inst))
                instLevelMap1[inst]= myCkt1->instGetSimInst(inst)->level;
        }
        block= design2->getTopBlock();
        oaIter<oaInst>   oaInstIter2(block->getInsts());
        while (oaInst*  inst = oaInstIter2.getNext())
        {
            if (myCkt2->instGetSimInst(inst))
                instLevelMap2[inst]= myCkt2->instGetSimInst(inst)->level;
        }
    }
    return similarityFactor;
}

/// Return counterexample
const vector<SimulationVector>& equiCheck::getCounterExample()
{
    return counterExamples;
}

/// Return counterexample output1
const vector<SimulationVector>& equiCheck::getCounterExampleOutput1()
{
    return counterExampleOutput1;
}

/// Return counterexample output2
const vector<SimulationVector>& equiCheck::getCounterExampleOutput2()
{
    return counterExampleOutput2;
}

/// Return primary output 1
const vector<oaNet*>& equiCheck::getPrimaryOutputs1()
{
    return PO1;
}

/// Return primary output 2
const vector<oaNet*>& equiCheck::getPrimaryOutputs2()
{
    return PO2;
}

/// Return primary input 1
const vector<oaNet*>& equiCheck::getPrimaryInputs1()
{
    return PI1;
}

/// Return primary input 2
const vector<oaNet*>& equiCheck::getPrimaryInputs2()
{
    return PI2;
}

const vector<oaNet*>& equiCheck::getDiffNets1()
{
    return diffNets1;
}

const vector<oaNet*>& equiCheck::getDiffNets2()
{
    return diffNets2;
}

/// Return number of counterexamples
int equiCheck::getNumCounterExamples()
{
    return counterExampleNO;
}

/// Return if the result is for two nets
bool equiCheck::isTwoNet()
{
    return twoNet;
}

/// Return if the result is for two designs
bool equiCheck::isTwoDesign()
{
    return twoDesign;
}

/// Return if the result is equivalent
bool equiCheck::isEquivalent()
{
    return equi;
}

/// Return the similarity factor previously calculated and saved.
double equiCheck::getSimilarityFactor()
{
    return similarityFactor;
}

string equiCheck::getLib1()
{
    string str;
    
    oa::oaNativeNS ns;
    oa::oaString name;
    if (designs[0] == NULL)
        return str;
    designs[0]->getLibName(ns, name);
    str= static_cast<const char*>(name);
    return str;
}
string equiCheck::getLib2()
{
    string str;
    
    oa::oaNativeNS ns;
    oa::oaString name;
    if (designs[1] == NULL)
        return str;
    designs[1]->getLibName(ns, name);
    str= static_cast<const char*>(name);
    return str;
}    

string equiCheck::getCell1()
{
    string str;
    oa::oaNativeNS ns;
    oa::oaString name;
    
    if (designs[0] == NULL)
        return str;
    designs[0]->getCellName(ns, name);
    str= static_cast<const char*>(name);
    return str;
}
string equiCheck::getCell2()
{
    string str;
    oa::oaNativeNS ns;
    oa::oaString name;
    
    if (designs[1] == NULL)
        return str;
    designs[1]->getCellName(ns, name);
    str= static_cast<const char*>(name);
    return str;
}

string equiCheck::getNet1()
{
    string str;
    oa::oaNativeNS ns;
    oa::oaString name;
    if (nets[0] == NULL)
        return str;
    nets[0]->getName(ns, name);
    str= static_cast<const char*>(name);
    return str;
}

string equiCheck::getNet2()
{
    string str;
    oa::oaNativeNS ns;
    oa::oaString name;
    if (nets[1] == NULL)
        return str;
    nets[1]->getName(ns, name);
    str= static_cast<const char*>(name);
    return str;
}

const std::map<oa::oaInst*, int>& equiCheck::getInstLevel1()
{
    return instLevelMap1;
}

const std::map<oa::oaInst*, int>& equiCheck::getInstLevel2()
{
    return instLevelMap2;
}

/// Clear signatures before performing equivalence checking
void equiCheck::clearSignature(oaDesign* design)
{
    optInternalCkt* myCkt1;
    vector<oaNet*> allBitNets1;
    optSimWire* myWire;
    vector<oaNet*>::iterator netIter;

    myCkt1= cktManager.designGetOptInternalCkt(design);
    myCkt1->getAllNets(allBitNets1);
    for (netIter= allBitNets1.begin(); netIter != allBitNets1.end(); netIter++)
    {
        myWire= myCkt1->netGetSimWire(*netIter);
        if (myWire)
            myWire->signature.clear();
    }
}

} // end of namespace

// vim: ci et sw=4 sts=4
