
/*
Author: Kai-hui Chang <changkh@eecs.umich.edu>
*/

#include "oagResyn.h"
#include "oagUtilOption.h"
#include "oagUtilOptionParser.h"
#include "oagResynSim.h"
#include "oagResynOptSim.h"
#include "oagResynEquiCheck.h"
#include "oagResynOptCktOper.h"

using namespace std;
using namespace oa;

namespace oagResyn {

/// Keyword to identify DFF's Q. 
string dffQ("Q");
/// Keyword to identify DFF's QN. 
string dffQN("QN");
/// Keyword to identify DFF's clock. 
string dffCLK("CK");
/// Keyword to identify DFF's D. 
string dffD("D");
/// Keyword to identify DFF. 
string dffKey("DFF");
/// Keyword to identify tie1. 
string tie1("tie1");
/// Keyword to identify tie0. 
string tie0("tie0");

LibMan libManager;
cktMan cktManager;
bool libertyRead= false;

// *****************************************************************************
// setKeyWords()
//
/// \brief Set keywords for recognizing important elements. 
///
/// Default: DFFKEY=DFF, DFFD=D, DFFQ=Q, DFFQN=QN, DFFCLK=CLK, TIE1= tie1, TIE0= tie0.
///
/// \param key DFFKEY: the keyword to identify DFFs, DFFD: DFF's D, DFFQ: DFF's Q, DFFQN: DFF's QN, DFFCLK: DFF's clk, TIE1: tie1, TIE0: tie0. 
/// \param str The string for the keyword.
//
// *****************************************************************************
void setKeyWords(keyWords key, char* str)
{
    switch (key)
    {
    case DFFKEY:
        dffKey= str;
        break;
    case DFFQ:
        dffQ= str;
        break;
    case DFFQN:
        dffQN= str;
        break;
    case DFFCLK:
        dffCLK= str;
        break;
    case DFFD:
        dffD= str;
        break;
    case TIE0:
        tie0= str;
        break;
    case TIE1:
        tie1= str;
        break; 
    }
}


// *****************************************************************************
// doOptSimulation()
//
/// \brief Perform simulation using optimized internal circuit.
//
// *****************************************************************************
void doOptSimulation(oa::oaDesign *design, int cycles, string& patternFile, string& stateFile) 
{
    optSim sim(design);
    vector<oaNet*> nets;
    vector<oaInst*> insts;
    vector<oaNet*>::iterator netIter;
    SimulationVector vec;
   
    if (patternFile.empty() == false)
    {
        sim.setUserPatternFile(patternFile.c_str());
        sim.advanceInputs();
    }
    else
    {
        sim.generateRandomInputVectors();
        sim.generateRandomStateVectors();
    }
    sim.useStateFile(stateFile.c_str());
    cout<<"Clocks found: ";
    sim.getClockNets(nets);
    for (netIter= nets.begin(); netIter != nets.end(); netIter++)
    {
        cout<<getNameFromOA(*netIter)<<endl;
    }
    cout<<endl;
//    sim.printInternalCircuit();
//    sim.libManager.printLib();
    
    cout<<"(cycle " <<  0 << ")" << std::endl;
    // Print input vectors
    cout << "(initial input vectors)" << std::endl;
    sim.getInputs(nets);
    for (netIter= nets.begin(); netIter != nets.end(); netIter++)
    {
        sim.getVector(*netIter, vec);
        cout<<getNameFromOA(*netIter)<<": "<<vec<<endl;
    }
    // Print state vectors
    cout << "(initial state vectors)" << std::endl;
    sim.getStateNets(nets);
    for (netIter= nets.begin(); netIter != nets.end(); netIter++)
    {
        sim.getVector(*netIter, vec);
        cout<<getNameFromOA(*netIter)<<": "<<vec<<endl;
    }
    sim.runFull();
    for(int iter = 0; iter < cycles+1; ++iter) 
    {
        cout << "(output vectors)" << std::endl;
        // Print output vectors
        sim.getOutputs(nets);
        for (netIter= nets.begin(); netIter != nets.end(); netIter++)
        {
            sim.getVector(*netIter, vec);
            cout<<getNameFromOA(*netIter)<<": "<<vec<<endl;
        }
        // next cycle
        std::cout << "----------------------------------------" << std::endl;
        cout<<"(cycle " << iter + 1 << ")" << std::endl;
        // Advance one cycle
        sim.advanceInputs();
        sim.nextCycle();
        cout << "(input vectors)" << std::endl;
        sim.getInputs(nets);
        for (netIter= nets.begin(); netIter != nets.end(); netIter++)
        {
            sim.getVector(*netIter, vec);
            cout<<getNameFromOA(*netIter)<<": "<<vec<<endl;
        }
    }
}

// *****************************************************************************
// doSimulation()
//
/// \brief Perform simulation using OAGear native data structures.
//
// *****************************************************************************
void doSimulation(oa::oaDesign *design, int cycles, string& patternFile, string& stateFile) 
{
    Sim sim(design);
    vector<oaNet*> nets;
    vector<oaInst*> insts;
    vector<oaNet*>::iterator netIter;
    SimulationVector vec;
   
    if (patternFile.empty() == false)
    {
        sim.setUserPatternFile(patternFile.c_str());
        sim.advanceInputs();
    }
    else
    {
        sim.generateRandomInputVectors();
        sim.generateRandomStateVectors();
    }
    sim.useStateFile(stateFile.c_str());
    cout<<"Clocks found: ";
    sim.getClockNets(nets);
    for (netIter= nets.begin(); netIter != nets.end(); netIter++)
    {
        cout<<getNameFromOA(*netIter)<<endl;
    }
    cout<<endl;
//    sim.printInternalCircuit();
//    sim.libManager.printLib();
    
    cout<<"(cycle " <<  0 << ")" << std::endl;
    // Print input vectors
    cout << "(initial input vectors)" << std::endl;
    sim.getInputs(nets);
    for (netIter= nets.begin(); netIter != nets.end(); netIter++)
    {
        sim.getVector(*netIter, vec);
        cout<<getNameFromOA(*netIter)<<": "<<vec<<endl;
    }
    // Print state vectors
    cout << "(initial state vectors)" << std::endl;
    sim.getStateNets(nets);
    for (netIter= nets.begin(); netIter != nets.end(); netIter++)
    {
        sim.getVector(*netIter, vec);
        cout<<getNameFromOA(*netIter)<<": "<<vec<<endl;
    }
    sim.runFull();
    for(int iter = 0; iter < cycles+1; ++iter) 
    {
        cout << "(output vectors)" << std::endl;
        // Print output vectors
        sim.getOutputs(nets);
        for (netIter= nets.begin(); netIter != nets.end(); netIter++)
        {
            sim.getVector(*netIter, vec);
            cout<<getNameFromOA(*netIter)<<": "<<vec<<endl;
        }
        // next cycle
        std::cout << "----------------------------------------" << std::endl;
        cout<<"(cycle " << iter + 1 << ")" << std::endl;
        sim.advanceInputs();
        sim.nextCycle();
        cout << "(input vectors)" << std::endl;
        sim.getInputs(nets);
        for (netIter= nets.begin(); netIter != nets.end(); netIter++)
        {
            sim.getVector(*netIter, vec);
            cout<<getNameFromOA(*netIter)<<": "<<vec<<endl;
        }
    }
}

// Open liberary 2
void openLib2(const std::string& lib2)
{
    // library files can be handled here because they are not design
    // specific
    if (lib2.empty() == false) {
        const oa::oaNativeNS nativeNS;
        const oa::oaVerilogNS verilogNS;
        oa::oaString lib2OptString = lib2.c_str();
        const oa::oaScalarName lib2Name(nativeNS, lib2OptString);
        if ((oa::oaLib::find(lib2Name)))
            ;
        else if(!oa::oaLib::exists(lib2OptString)) {
            oa::oaLib::create(lib2Name, lib2OptString);
            oa::oaLibDefList *defList
                = oa::oaLibDefList::get("lib.defs", 'a');
            oa::oaLibDef::create(defList, lib2Name, lib2OptString);
            defList->save();
        } else {
            oa::oaLib::open(lib2Name, lib2OptString);
        }
    }
}


// *****************************************************************************
// getSimulator()
/// \brief Get a simulatorInterface.
// *****************************************************************************
std::auto_ptr<simulatorInterface> getSimulator(int argc, const char **argv) throw(oagResynErr, oa::oaException)
{
    int seed;
    string view, libDefFile;

    oagUtil::OptionParser options("Simple test program to demostarte the use "
              "of the simulation class.");

    oagUtil::Option *libOpt = options.add("lib", "Input and output library "
            "name", true, "library");
    oagUtil::Option *viewOpt = options.add("view", "Input and output view name "
            "(default: netlist)", false, "view");
    oagUtil::Option *cellOpt = options.add("cell", "Input cell "
            "name", true, "cell");
    oagUtil::Option *libertyOpt = options.add("liberty", ".lib library file", 
            true, "file");
    oagUtil::Option *seedOpt = options.add("seed", "Use a specific random seed"
            " (default: use system time)", false, "int");
    oagUtil::Option *useOpt = options.add("useOpt", "Use optimized simulator based on custom data structure",
            false);
    oagUtil::Option *libdefsOpt = options.add("libDef", "Use a specific lib.defs"
            " (default: use current directory)", false, "/path/to/lib.defs");
    oagUtil::Option *lib2Opt = options.add("lib2", "Second input library name", 
            false, "library");
    oagUtil::Option *patternFileOpt = options.add("patternFile", "Input pattern file", 
            false, "file");
    oagUtil::Option *stateFileOpt = options.add("stateFile", "Initial state file", 
            false, "file");

    if (!options.parse(argc, argv)) {
        std::cerr << options.getMessage();
        throw oagResynErr("Option parsing error");
    }
    if (seedOpt->isGiven()) {
        seed= atoi(seedOpt->getValue());
    } else {
        seed= 0;
    }
    oa::oaDesignInit();
    if (viewOpt->isGiven())
        view= viewOpt->getValue();
    else
        view= "netlist";
    if (libdefsOpt->isGiven())
        libDefFile= libdefsOpt->getValue();
    string lib2;
    if (lib2Opt->isGiven())
        lib2= lib2Opt->getValue();
    string patternFile;
    if (patternFileOpt->isGiven())
        patternFile= patternFileOpt->getValue();
    string stateFile;
    if (stateFileOpt->isGiven())
        stateFile= stateFileOpt->getValue();
    return getSimulator(libOpt->getValue(), cellOpt->getValue(), 
               libertyOpt->getValue(), seed, useOpt->isGiven(),
               patternFile, stateFile, view, libDefFile, lib2);
}

// *****************************************************************************
// getSimulator()
/// \brief Get a simulatorInterface.
// *****************************************************************************
std::auto_ptr<simulatorInterface> getSimulator(const std::string& lib,
               const std::string& cell, const std::string& liberty,
               const int seed, const bool useOpt, 
               const std::string& patternFile,
               const std::string& stateFile,
               const std::string& view, const std::string& libDefFile,
               const std::string& lib2)
{
    oa::oaDesign *design;
    try {
    
        oa::oaDesignInit();
        if (libDefFile.empty())
            oa::oaLibDefList::openLibs();
        else
            oa::oaLibDefList::openLibs(libDefFile.c_str());
        openLib2(lib2);
            
        oagFunc::initialize();

        const oa::oaNativeNS nativeNS;
        const oa::oaVerilogNS verilogNS;
        oa::oaLib *library;
        const oa::oaScalarName libName(nativeNS, lib.c_str());
        const oa::oaScalarName viewName(nativeNS, view.c_str());
        // Find, create, or load the first library

        if ((library = oa::oaLib::find(libName))) {
            ;    
        } else if(!oa::oaLib::exists(lib.c_str())) {
            library = oa::oaLib::create(libName, lib.c_str());
            oa::oaLibDefList *defList = oa::oaLibDefList::get("lib.defs", 'a');
            oa::oaLibDef::create(defList, libName, lib.c_str());
            defList->save();
        } else {
            library = oa::oaLib::open(libName, lib.c_str());
        }


        if (libertyRead == false)
        {
            oagFunc::readLiberty(library, viewName, liberty.c_str());
            libertyRead= true;
        }

        // Open design

        const char *cellString = cell.c_str();
        const oa::oaScalarName cellName(nativeNS, cellString);
        if ((design = oa::oaDesign::find(libName, cellName, viewName))) {
            ;
        } else {
            // open for read only
            design = oa::oaDesign::open(libName, cellName, viewName, 'r');
        }

        if (!design) {
            std::cerr << "ERROR: Could not open design" << std::endl;
            throw;
        }

        if (seed != 0)
            srand(seed);
        else
            srand(time(NULL));
        if (useOpt)
        {
            auto_ptr<optSim> mySim(new optSim(design));
            mySim->setUserPatternFile(patternFile.c_str());
            mySim->useStateFile(stateFile.c_str());
            mySim->runFull();
            optSim* myTmpSim = mySim.release();
            simulatorInterface* mySimIntp = static_cast<simulatorInterface*>(myTmpSim);
            auto_ptr<simulatorInterface> rval( mySimIntp);
            return rval;
            //return static_cast<auto_ptr<simulatorInterface> >(mySim);
        }
        else
        {
            auto_ptr<Sim> mySim(new Sim(design));
            mySim->setUserPatternFile(patternFile.c_str());
            mySim->useStateFile(stateFile.c_str());
            mySim->runFull();

            Sim* myTmpSim = mySim.release();
            simulatorInterface* mySimIntp = static_cast<simulatorInterface*>(myTmpSim);
            auto_ptr<simulatorInterface> rval( mySimIntp);
            return rval;
//            return static_cast<auto_ptr<simulatorInterface> >(mySim);
        }
    } catch(oa::oaException &e) {
        std::cerr << "UMSim ERROR: " << e.getMsg() << std::endl;
        throw;
    }
}

// *****************************************************************************
// simulationMain()
/// \brief Sample main function to invoke the simulator.
// *****************************************************************************
int simulationMain(int argc, const char **argv) {
    oagUtil::OptionParser options("Simple test program to demostarte the use "
              "of the simulation class.");

    oagUtil::Option *libOpt = options.add("lib", "Input and output library "
            "name", true, "library");
    oagUtil::Option *viewOpt = options.add("view", "Input and output view name "
            "(default: netlist)", false, "view");
    oagUtil::Option *cellOpt = options.add("cell", "Input cell "
            "name", true, "cell");
    oagUtil::Option *libertyOpt = options.add("liberty", ".lib library file", 
            true, "file");
    oagUtil::Option *verilogOpt = options.add("verilog", "Verilog file", false, 
            "file");
    oagUtil::Option *lib2Opt = options.add("lib2", "Second input library name", 
            false, "library");
    oagUtil::Option *cycleOpt = options.add("cycles", "Number of addt'l cycles to "
            "simulate (default: 0)", false, "int");
    oagUtil::Option *seedOpt = options.add("seed", "Use a specific random seed"
            " (default: use system time)", false, "int");
    oagUtil::Option *useOpt = options.add("useOpt", "Use optimized simulator based on custom data structure",
            false);
    oagUtil::Option *libdefsOpt = options.add("libDef", "Use a specific lib.defs"
            " (default: use current directory)", false, "/path/to/lib.defs");
    oagUtil::Option *patternFileOpt = options.add("patternFile", "Input pattern file", 
            false, "file");
    oagUtil::Option *stateFileOpt = options.add("stateFile", "Initial state file", 
            false, "file");

    if (!options.parse(argc, argv)) {
        std::cerr << options.getMessage();
        return(0);
    }
 
    string libdeffile;
    if (libdefsOpt->isGiven())
    {
        libdeffile= libdefsOpt->getValue();

        if(!libdeffile.empty())
        {
            ifstream checkFile(libdeffile.c_str());
            if(!checkFile.good())
            {
                cerr<<"No such file: \""<<libdeffile<<"\"."<<endl;
                return(0);
            }
        }
    }


    const char *libString = libOpt->getValue();
    const char *viewString 
        = viewOpt->isGiven() ? viewOpt->getValue(): "netlist";

    try {

        // Initialize OpenAccess

        oa::oaDesignInit();
        if(!libdeffile.empty())
        {
            oa::oaLibDefList::openLibs(libdeffile.c_str());
        }
        else
        {
            oa::oaLibDefList::openLibs();
        }
        oagFunc::initialize();

        const oa::oaNativeNS nativeNS;
        const oa::oaVerilogNS verilogNS;
        oa::oaLib *library;
        const oa::oaScalarName libName(nativeNS, libString);
        const oa::oaScalarName viewName(nativeNS, viewString);

        // Find, create, or load the first library

        if ((library = oa::oaLib::find(libName))) {
            ;    
        } else if(!oa::oaLib::exists(libString)) {
            library = oa::oaLib::create(libName, libString);
            oa::oaLibDefList *defList = oa::oaLibDefList::get("lib.defs", 'a');
            oa::oaLibDef::create(defList, libName, libString);
            defList->save();
        } else {
            library = oa::oaLib::open(libName, libString);
        }

        // Find, create, or load an additional library (if given)

        if (lib2Opt->isGiven()) {
            oa::oaString lib2OptString = lib2Opt->getValue();
            const oa::oaScalarName lib2Name(nativeNS, lib2OptString);   
            // todo - use the function from above.
            if ((oa::oaLib::find(lib2Name)))
                ;    
            else if(!oa::oaLib::exists(lib2OptString)) {
                oa::oaLib::create(lib2Name, lib2OptString);
                oa::oaLibDefList *defList 
                    = oa::oaLibDefList::get("lib.defs", 'a');
                oa::oaLibDef::create(defList, lib2Name, lib2OptString);
                defList->save();
            } else {
                oa::oaLib::open(lib2Name, lib2OptString);
            }
        }

        // Read designs from a Liberty file

        if (libertyOpt->isGiven()) {
            // read in a cell library
            if (libertyRead == false)
            {
                oagFunc::readLiberty(library, viewName, libertyOpt->getValue());
                libertyRead= true;
            }
        }

        // Read designs from a Verilog file

        if (verilogOpt->isGiven()) {
            // read in functional Verilog description
            oagFunc::readVerilog(library, viewName, verilogOpt->getValue());
        }

        // Open design

        const char *cellString = cellOpt->getValue();
        const oa::oaScalarName cellName(nativeNS, cellString);
        oa::oaDesign *design;
        if ((design = oa::oaDesign::find(libName, cellName, viewName))) {
            ;
        } else {
            // open for read only
            design = oa::oaDesign::open(libName, cellName, viewName, 'r');
        }

        if (!design) {
            std::cerr << "ERROR: Could not open design" << std::endl;
            return(1);
        }
        
        // Seed random generator w/ user-specified num or system time
        
        if (seedOpt->isGiven()) {
            srand(atoi(seedOpt->getValue()));
        } else {
            srand(time(NULL));
        }
        
        // Run simulation

        int cycles = 0;
        if (cycleOpt->isGiven()) {
            cycles = atoi(cycleOpt->getValue());
        }
        
        string patternFile;
        if (patternFileOpt->isGiven())
            patternFile= patternFileOpt->getValue();

        string stateFile;
        if (stateFileOpt->isGiven())
            stateFile= stateFileOpt->getValue();

        // Add my code here.
        if (useOpt->isGiven())
            doOptSimulation(design, cycles, patternFile, stateFile);
        else
            doSimulation(design, cycles, patternFile, stateFile);
        

        /* not necessary
        mySim.analyzeCircuit();
        mySim.printInOutState();
        mySim.libManager.genLogicInfo(design);
        mySim.libManager.printLib();
        mySim.buildInternalCircuit();
        mySim.printInternalCircuit(); */
        
        // Close design

        oa::oaDesign *openDesign;
        oa::oaIter<oa::oaDesign> designIter(oa::oaDesign::getOpenDesigns());
        while((openDesign = designIter.getNext())) {
            oa::oaString designString;
            openDesign->getTopModule()->getName(nativeNS, designString);
            openDesign->purge();
        }
        // Handle errors

    } catch(oa::oaException &e) {
        std::cerr << "ERROR: " << e.getMsg() << std::endl;
        assert(false);
        return 1;
    }

    return 0;
}

// *****************************************************************************
// getEquiResultTwoNets()
/// \brief Get equivalence checking result for two nets.
// *****************************************************************************
std::auto_ptr<equiCheckOutputInterface> getEquiResultTwoNets(
              const std::string& lib, 
              const std::string& cell, 
              const std::string& net1, 
              const std::string& net2, 
              const std::string& liberty,
              const int randomCycle, const int seed, 
              const std::string& view, 
              const std::string& libDefFile,
              const std::string& lib2)
{
    oa::oaDesign *design;
    try {
    
        oa::oaDesignInit();
        if (libDefFile.empty())
            oa::oaLibDefList::openLibs();
        else
            oa::oaLibDefList::openLibs(libDefFile.c_str());
        openLib2(lib2);   
        oagFunc::initialize();

        const oa::oaNativeNS nativeNS;
        const oa::oaVerilogNS verilogNS;
        oa::oaLib *library;
        const oa::oaScalarName libName(nativeNS, lib.c_str());
        const oa::oaScalarName viewName(nativeNS, view.c_str());
        // Find, create, or load the first library

        if ((library = oa::oaLib::find(libName))) {
            ;    
        } else if(!oa::oaLib::exists(lib.c_str())) {
            library = oa::oaLib::create(libName, lib.c_str());
            oa::oaLibDefList *defList = oa::oaLibDefList::get("lib.defs", 'a');
            oa::oaLibDef::create(defList, libName, lib.c_str());
            defList->save();
        } else {
            library = oa::oaLib::open(libName, lib.c_str());
        }

        if (libertyRead == false)
        {
            oagFunc::readLiberty(library, viewName, liberty.c_str());
            libertyRead= true;
        }

        // Open design

        const char *cellString = cell.c_str();
        const oa::oaScalarName cellName(nativeNS, cellString);
        if ((design = oa::oaDesign::find(libName, cellName, viewName))) {
            ;
        } else {
            // open for read only
            design = oa::oaDesign::open(libName, cellName, viewName, 'r');
        }

        if (!design) {
            std::cerr << "ERROR: Could not open design" << std::endl;
            throw;
        }

        if (seed != 0)
            srand(seed);
        else
            srand(time(NULL));
        oa::oaName net1Name(verilogNS, net1.c_str());
        oa::oaName net2Name(verilogNS, net2.c_str());
        oa::oaNet *n1= oa::oaNet::find(design->getTopBlock(), net1Name);
        oa::oaNet *n2= oa::oaNet::find(design->getTopBlock(), net2Name);
        if (n1 == NULL || n2 == NULL)
        {
            throw oagResynErr("Net not found");
        }

        auto_ptr<equiCheck> myEquiCheck(new equiCheck);
        myEquiCheck->randomPatternNumber= randomCycle;
        SimulationVector v1, v2;
        vector<SimulationVector> inputVecs;
        int no= 32;
        vector<oaNet*> nets;
        no= 32;
        myEquiCheck->checkEquiTwoNets(design, n1, n2, 
                               no, nets, inputVecs, v1, v2); 

        equiCheck* myTmpCheck = myEquiCheck.release();
        equiCheckOutputInterface* myIntp = static_cast<equiCheckOutputInterface*>(myTmpCheck);
        auto_ptr<equiCheckOutputInterface> rval( myIntp);
        return rval;
        //return static_cast<auto_ptr<equiCheckOutputInterface> >(myEquiCheck);
    } catch(oa::oaException &e) {
        std::cerr << "UMSim ERROR: " << e.getMsg() << std::endl;
        throw;
    }
}

// *****************************************************************************
// getEquiResult()
/// \brief Get equivalence checking result for two nets/designs.
// *****************************************************************************
std::auto_ptr<equiCheckOutputInterface> getEquiResult(int argc, const char **argv) throw(oagResynErr, oa::oaException)
{
    int seed;
    string view, libDefFile;
    int rndCycle;

    oagUtil::OptionParser options("Simple test program to demostarte the use "
              "of the equivalence checking class.");

    oagUtil::Option *libOpt = options.add("lib", "Input and output library "
            "name", true, "library");
    oagUtil::Option *viewOpt = options.add("view", "Input and output view name "
            "(default: netlist)", false, "view");
    oagUtil::Option *cellOpt = options.add("cell1", "Input cell "
            "name", true, "cell");
    oagUtil::Option *cell2Opt = options.add("cell2", "Input cell "
            "name", false, "cell");
    oagUtil::Option *libertyOpt = options.add("liberty", ".lib library file", 
            true, "file");
    oagUtil::Option *seedOpt = options.add("seed", "Use a specific random seed"
            " (default: use system time)", false, "int");
    oagUtil::Option *libdefsOpt = options.add("libDef", "Use a specific lib.defs"
            " (default: use current directory)", false, "/path/to/lib.defs");
    oagUtil::Option *net1Opt = options.add("net1", "Net 1 for checking",
                 false, "net");
    oagUtil::Option *net2Opt = options.add("net2", "Net 2 for checking",
                false, "net");
    oagUtil::Option *rndCycleOpt = options.add("randomPattern", "number of random cycle",
                false, "int");
    oagUtil::Option *lib2Opt = options.add("lib2", "Second input library name",
                false, "library");

    if (!options.parse(argc, argv)) {
        std::cerr << options.getMessage();
        throw oagResynErr("Option parsing error");
    }
    if (seedOpt->isGiven()) {
        seed= atoi(seedOpt->getValue());
    } else {
        seed= 0;
    }
    if (rndCycleOpt->isGiven()) {
        rndCycle= atoi(rndCycleOpt->getValue());
        if (rndCycle < 0)
            rndCycle= 0;
    } else {
        rndCycle= 1024;
    }
    if (viewOpt->isGiven())
        view= viewOpt->getValue();
    else
        view= "netlist";
    if (libdefsOpt->isGiven())
        libDefFile= libdefsOpt->getValue();
    string lib2;
    if (lib2Opt->isGiven())
        lib2= lib2Opt->getValue();

    if (net1Opt->isGiven() && net2Opt->isGiven())
    {
        return getEquiResultTwoNets(
               libOpt->getValue(), 
               cellOpt->getValue(), 
               net1Opt->getValue(),
               net2Opt->getValue(),
               libertyOpt->getValue(), 
               rndCycle,
               seed, 
               view, 
               libDefFile,
               lib2);
    }
    else
    {
        return getEquiResultTwoDesigns(
                libOpt->getValue(), 
               cellOpt->getValue(), 
               cell2Opt->getValue(), 
               libertyOpt->getValue(), 
               rndCycle, seed,
               view, libDefFile,
               lib2);
    }
}

// *****************************************************************************
// getEquiResultTwoDesigns()
/// \brief Get equivalence checking result for two designs.
// *****************************************************************************
std::auto_ptr<equiCheckOutputInterface> getEquiResultTwoDesigns(
              const std::string& lib, 
              const std::string& cell1, 
              const std::string& cell2, 
              const std::string& liberty,
              const int randomCycle, const int seed, 
              const std::string& view, 
              const std::string& libDefFile,
              const std::string& lib2)
{
    oa::oaDesign *design, *design2;
    try {
        oa::oaDesignInit();
    
        if (libDefFile.empty())
            oa::oaLibDefList::openLibs();
        else
            oa::oaLibDefList::openLibs(libDefFile.c_str());
        openLib2(lib2);    
        oagFunc::initialize();

        const oa::oaNativeNS nativeNS;
        const oa::oaVerilogNS verilogNS;
        oa::oaLib *library;
        const oa::oaScalarName libName(nativeNS, lib.c_str());
        const oa::oaScalarName viewName(nativeNS, view.c_str());
        // Find, create, or load the first library

        if ((library = oa::oaLib::find(libName))) {
            ;    
        } else if(!oa::oaLib::exists(lib.c_str())) {
            library = oa::oaLib::create(libName, lib.c_str());
            oa::oaLibDefList *defList = oa::oaLibDefList::get("lib.defs", 'a');
            oa::oaLibDef::create(defList, libName, lib.c_str());
            defList->save();
        } else {
            library = oa::oaLib::open(libName, lib.c_str());
        }

        if (libertyRead == false)
        {
            oagFunc::readLiberty(library, viewName, liberty.c_str());
            libertyRead= true;
        }

        // Open design

        const char *cellString = cell1.c_str();
        const oa::oaScalarName cellName(nativeNS, cellString);
        if ((design = oa::oaDesign::find(libName, cellName, viewName))) {
            ;
        } else {
            // open for read only
            design = oa::oaDesign::open(libName, cellName, viewName, 'r');
        }

        if (!design) {
            std::cerr << "ERROR: Could not open design" << std::endl;
            throw;
        }
        const char *cellString2 = cell2.c_str();
        const oa::oaScalarName cell2Name(nativeNS, cellString2);
        if ((design2 = oa::oaDesign::find(libName, cell2Name, viewName))) {
            ;
        } else {
            // open for read only
            design2 = oa::oaDesign::open(libName, cell2Name, viewName, 'r');
        }

        if (!design2) {
            std::cerr << "ERROR: Could not open design" << std::endl;
            throw;
        }

        if (seed != 0)
            srand(seed);
        else
            srand(time(NULL));

        auto_ptr<equiCheck> myEquiCheck(new equiCheck);
        myEquiCheck->randomPatternNumber= randomCycle;
        vector<oaNet*> PI1, PI2, PO1, PO2;
        int no= 32;
        vector<SimulationVector> counterExamples, counterExampleOutput1,
                                 counterExampleOutput2;
        myEquiCheck->checkTwoDesignsAutoMap(design, design2,
                               PI1, 
                               PO1, 
                               PI2, 
                               PO2, 
                               no,
                               counterExamples,
                               counterExampleOutput1,
                               counterExampleOutput2
                               );
        // Similarity factor. To calculate it, random cycle cannot be 0
        if (randomCycle == 0)
            myEquiCheck->randomPatternNumber= 1024;
        myEquiCheck->getSimilarityFactor(design, design2, PI1, PI2);

        equiCheck* myTmpCheck = myEquiCheck.release();
        equiCheckOutputInterface* myIntp = static_cast<equiCheckOutputInterface*>(myTmpCheck);
        auto_ptr<equiCheckOutputInterface> rval( myIntp);
        return rval;
        //return static_cast<auto_ptr<equiCheckOutputInterface> >(myEquiCheck);
    } catch(oa::oaException &e) {
        std::cerr << "UMSim ERROR: " << e.getMsg() << std::endl;
        throw;
    }
}

// *****************************************************************************
// increVeriMain()
/// \brief Sample main function to invoke the incremental verification 
/// experiment.
// *****************************************************************************
std::auto_ptr<equiCheckOutputInterface> increVeriMain(int argc, const char **argv) throw(oagResynErr, oa::oaException)
{
    oagUtil::OptionParser options("Program for incremental verification experiments");

    oagUtil::Option *libOpt = options.add("lib", "Input and output library "
            "name", true, "library");
    oagUtil::Option *viewOpt = options.add("view", "Input and output view name "
            "(default: netlist)", false, "view");
    oagUtil::Option *cellOpt = options.add("cell", "Input cell "
            "name", true, "cell");
    oagUtil::Option *libertyOpt = options.add("liberty", ".lib library file", 
            true, "file");
    oagUtil::Option *verilogOpt = options.add("verilog", "Verilog file", false, 
            "file");
    oagUtil::Option *lib2Opt = options.add("lib2", "Second input library name", 
            false, "library");
    oagUtil::Option *seedOpt = options.add("seed", "Use a specific random seed"
            " (default: use system time)", false, "int");
    oagUtil::Option *libdefsOpt = options.add("libDef", "Use a specific lib.defs"
            " (default: use current directory)", false, "/path/to/lib.defs");
    oagUtil::Option *resynOpt = options.add("resyn", "Resynthesize the circuit in stead of injecting bugs",
            false);

    if (!options.parse(argc, argv)) {
        std::cerr << options.getMessage();
        throw oagResynErr("Argument error");
    }
 
    string libdeffile;
    if (libdefsOpt->isGiven())
    {
        libdeffile= libdefsOpt->getValue();

        if(!libdeffile.empty())
        {
            ifstream checkFile(libdeffile.c_str());
            if(!checkFile.good())
            {
                cerr<<"No such file: \""<<libdeffile<<"\"."<<endl;
                throw oagResynErr("No such file");
            }
        }
    }


    const char *libString = libOpt->getValue();
    const char *viewString 
        = viewOpt->isGiven() ? viewOpt->getValue(): "netlist";

    try {

        // Initialize OpenAccess

        oa::oaDesignInit();
        if(!libdeffile.empty())
        {
            oa::oaLibDefList::openLibs(libdeffile.c_str());
        }
        else
        {
            oa::oaLibDefList::openLibs();
        }
        oagFunc::initialize();

        const oa::oaNativeNS nativeNS;
        const oa::oaVerilogNS verilogNS;
        oa::oaLib *library;
        const oa::oaScalarName libName(nativeNS, libString);
        const oa::oaScalarName viewName(nativeNS, viewString);
        // oa::oaViewType *viewType = oa::oaViewType::get(oa::oacNetlist);

        // Find, create, or load the first library

        if ((library = oa::oaLib::find(libName))) {
            ;    
        } else if(!oa::oaLib::exists(libString)) {
            library = oa::oaLib::create(libName, libString);
            oa::oaLibDefList *defList = oa::oaLibDefList::get("lib.defs", 'a');
            oa::oaLibDef::create(defList, libName, libString);
            defList->save();
        } else {
            library = oa::oaLib::open(libName, libString);
        }

        // Find, create, or load an additional library (if given)

        if (lib2Opt->isGiven()) {
            oa::oaString lib2OptString = lib2Opt->getValue();
            const oa::oaScalarName lib2Name(nativeNS, lib2OptString);   
            // todo - use the function from above.
            if ((oa::oaLib::find(lib2Name)))
                ;    
            else if(!oa::oaLib::exists(lib2OptString)) {
                oa::oaLib::create(lib2Name, lib2OptString);
                oa::oaLibDefList *defList 
                    = oa::oaLibDefList::get("lib.defs", 'a');
                oa::oaLibDef::create(defList, lib2Name, lib2OptString);
                defList->save();
            } else {
                oa::oaLib::open(lib2Name, lib2OptString);
            }
        }

        // Read designs from a Liberty file

        if (libertyOpt->isGiven()) {
            // read in a cell library
            if (libertyRead == false)
            {
                oagFunc::readLiberty(library, viewName, libertyOpt->getValue());
                libertyRead= true;
            }
        }

        // Read designs from a Verilog file

        if (verilogOpt->isGiven()) {
            // read in functional Verilog description
            oagFunc::readVerilog(library, viewName, verilogOpt->getValue());
        }

        // Open design

        const char *cellString = cellOpt->getValue();
        const oa::oaScalarName cellName(nativeNS, cellString);
        oa::oaDesign *design;
        if ((design = oa::oaDesign::find(libName, cellName, viewName))) {
            ;
        } else {
            // open for read only
            design = oa::oaDesign::open(libName, cellName, viewName, 'r');
        }

        if (!design) {
            std::cerr << "ERROR: Could not open design" << std::endl;
            throw oagResynErr("Could not open design");
        }
        
        // Seed random generator w/ user-specified num or system time
        
        string cellname2;

        cellname2= cellOpt->getValue();
        cellname2= cellname2 + "_buggy";
        int seed;

        if (seedOpt->isGiven()) {
            seed= atoi(seedOpt->getValue());
            srand(seed);
        } else {
            seed= 0;
            srand(time(NULL));
        }

        string viewString;

        if (viewOpt->isGiven())
            viewString = viewOpt->getValue();
        else
            viewString = "netlist";
        string libDefFile;
        if (libdefsOpt->isGiven())
            libDefFile= libdefsOpt->getValue();
        string lib2;
        if (lib2Opt->isGiven())
            lib2= lib2Opt->getValue();

        // To do: Save the design to another name ("name_buggy"). Open it, 
        // inject error, then call equivalence checker.
        oaScalarName oaCellName2(nativeNS, cellname2.c_str());
        design->saveAs(libName, oaCellName2, viewName);
        // open the second design
        oa::oaDesign *design2;
        if ((design2 = oa::oaDesign::find(libName, oaCellName2, viewName))) {
            ;
        } else {
            // open for read only
            design2 = oa::oaDesign::open(libName, oaCellName2, viewName, 'r');
        }
        optInternalCkt* ckt2;
        optSimInst* tgtGate;
        
        ckt2= cktManager.designGetOptInternalCkt(design2);
        optCktOper cktOper1(ckt2);
        if (resynOpt->isGiven())
        {
            do
            {
                tgtGate= cktOper1.randomResynthesize();
            } while (tgtGate == NULL);
        }
        else
        {
            do
            {
                tgtGate= cktOper1.injectRandomBug();
            } while (tgtGate == NULL);
        }
        ckt2->levelizeInternalCircuit();
        return getEquiResultTwoDesigns(
               libOpt->getValue(), 
               cellOpt->getValue(), 
               cellname2,
               libertyOpt->getValue(), 
               1024, seed,
               viewString, libDefFile,
               lib2);
    } catch(oa::oaException &e) {
        std::cerr << "ERROR: " << e.getMsg() << std::endl;
        throw;
    }
}

} // end of namespace


// vim: ci et sw=4 sts=4
