
/*
Author: Kai-hui Chang <changkh@eecs.umich.edu>
*/

#include <iostream>
#include "oaDesignDB.h"
#include "oagFunc.h"
#include "oagFuncOccGraph.h"
#include "oagUtilOption.h"
#include "oagUtilOptionParser.h"
#include "oagResynEquiCheck.h"
#include "oagResyn.h"

using namespace oagResyn;
using namespace std;
using namespace oa;

void checkTwoNet(oaDesign* design1, oaNet* net1, oaNet* net2, int rndCycle)
{
    oagResyn::equiCheck myEquiCheck;
    bool result;
    vector<SimulationVector> inputVecs;
    SimulationVector outVec1, outVec2;
    vector<oaNet*> inputs;
    unsigned int i;
    int vecNumber;

    vecNumber= 32;
    myEquiCheck.randomPatternNumber= rndCycle;
    result= myEquiCheck.checkEquiTwoNets(design1, net1, net2, vecNumber, 
                inputs, inputVecs, outVec1, outVec2);
    if (result == true)
        cout<<"Equivalent"<<endl;
    else
    {
        cout<<"Not equivalent"<<endl;
        cout<<"Vectors returned: "<<vecNumber<<endl;
        cout<<"Inputs: "<<endl;
        for (i= 0; i < inputs.size(); i++)
        {
            cout<<getNameFromOA(inputs[i])<<": "<<inputVecs[i]<<endl;
        }
        cout<<"Outputs (net1, net2):"<<endl;
        cout<<outVec1<<", "<<outVec2<<endl;
    }
}

void checkEqui(oaDesign* design1, oaDesign* design2, int rndCycle)
{
    equiCheck equiCheck1;
    vector<oaInst*> insts1, insts2;
    bool result;
    int vecNumber;
    vector<SimulationVector> inputVecs, outputVec1, outputVec2;
    unsigned int i;
    vector<oaNet*> netOutput1, netOutput2, netInput1, netInput2;
  
    vecNumber= 32;
    equiCheck1.randomPatternNumber= rndCycle;
    result= equiCheck1.checkTwoDesignsAutoMap(design1, design2, 
                        netInput1, netOutput1, netInput2, netOutput2,
                        vecNumber, inputVecs, outputVec1, outputVec2);
    // To calculate similarity factor, random simulation must be allowed
    equiCheck1.randomPatternNumber= 1024;
    cout<<"Similarity factor: "<<equiCheck1.getSimilarityFactor(design1, design2, netInput1, netInput2)<<endl;
    if (result == true)
        cout<<"Equivalent"<<endl;
    else
    {
        cout<<"Not equivalent"<<endl;
        cout<<"Number of vectors returned: "<<vecNumber<<endl;
        cout<<"Input vectors:"<<endl;
        for (i= 0; i < inputVecs.size(); i++)
        {
            cout<<getNameFromOA(netInput1[i])<<": "<<inputVecs[i]<<endl;
        }
        cout<<endl;
        cout<<"Output vectors (cell1, cell2):"<<endl;
        for (i= 0; i < outputVec1.size(); i++)
        {
            cout<<getNameFromOA(netOutput1[i])<<": "<<outputVec1[i]<<", "<<outputVec2[i]<<endl;
        }
        cout<<endl;
    }
}

// *****************************************************************************// main()
//
// *****************************************************************************
int main(int argc, const char **argv) {

    std::auto_ptr<equiCheckOutputInterface> equiOut;
    /*
    try
    {
        equiOut= getEquiResult(argc, argv);
        if (equiOut->isEquivalent() == true)
            cout<<"Equivalent"<<endl;

    } catch(oa::oaException &e) {
        std::cerr << "ERROR: " << e.getMsg() << std::endl;
        assert(false);
        return 1;
    } catch(oagResynErr& e)
    {
        std::cerr<<e.what()<<endl;
        return 1;
    }
    return 0;
    */
    oagUtil::OptionParser options("Simple test program to demostarte the use "
              "of the equivalence checker. To check two circuits, specify both cell1 and cell2. Inputs/outputs with the same names will be tied together. To check two nets in a circuit, specify cell1, net1 and net2.");

    oagUtil::Option *libOpt = options.add("lib", "Input and output library "
            "name", true, "library");
    oagUtil::Option *viewOpt = options.add("view", "Input and output view name "
            "(default: netlist)", false, "view");
    oagUtil::Option *cellOpt = options.add("cell1", "Input cell "
            "name", true, "cell");
    oagUtil::Option *cell2Opt = options.add("cell2", "Input cell "
            "name", false, "cell");
    oagUtil::Option *libertyOpt = options.add("liberty", ".lib library file", 
            false, "file");
    oagUtil::Option *net1Opt = options.add("net1", "Net 1 for checking", 
            false, "net");
    oagUtil::Option *net2Opt = options.add("net2", "Net 2 for checking", 
            false, "net");
    oagUtil::Option *lib2Opt = options.add("lib2", "Second input library name", 
            false, "library");
    oagUtil::Option *seedOpt = options.add("seed", "Use a specific random seed"
            " (default: use system time)", false, "int");
    oagUtil::Option *rndCycleOpt = options.add("randomPattern", "number of random cycle",
                false, "int");

    if (!options.parse(argc, argv)) {
        std::cerr << options.getMessage();
        exit(0);
    }

    const char *libString = libOpt->getValue();
    const char *viewString 
        = viewOpt->isGiven() ? viewOpt->getValue(): "netlist";

    try {

        // Initialize OpenAccess

        oa::oaDesignInit();
        oa::oaLibDefList::openLibs();
        oagFunc::initialize();

        const oa::oaNativeNS nativeNS;
        const oa::oaVerilogNS verilogNS;
        oa::oaLib *library;
        const oa::oaScalarName libName(nativeNS, libString);
        const oa::oaScalarName viewName(nativeNS, viewString);
        // oa::oaViewType *viewType = oa::oaViewType::get(oa::oacNetlist);

        // Find, create, or load the first library

        if ((library = oa::oaLib::find(libName))) {
            ;    
        } else if(!oa::oaLib::exists(libString)) {
            library = oa::oaLib::create(libName, libString);
            oa::oaLibDefList *defList = oa::oaLibDefList::get("lib.defs", 'a');
            oa::oaLibDef::create(defList, libName, libString);
            defList->save();
        } else {
            library = oa::oaLib::open(libName, libString);
        }

        if (seedOpt->isGiven()) {
            srand(atoi(seedOpt->getValue()));
        } else {
            srand(time(NULL));
        }
        // Find, create, or load an additional library (if given)

        if (lib2Opt->isGiven()) {
            oa::oaString lib2OptString = lib2Opt->getValue();
            const oa::oaScalarName lib2Name(nativeNS, lib2OptString);   
            // todo - use the function from above.
            if ((oa::oaLib::find(lib2Name)))
                ;    
            else if(!oa::oaLib::exists(lib2OptString)) {
                oa::oaLib::create(lib2Name, lib2OptString);
                oa::oaLibDefList *defList 
                    = oa::oaLibDefList::get("lib.defs", 'a');
                oa::oaLibDef::create(defList, lib2Name, lib2OptString);
                defList->save();
            } else {
                oa::oaLib::open(lib2Name, lib2OptString);
            }
        }

        // Read designs from a Liberty file

        if (libertyOpt->isGiven()) {
            // read in a cell library
            oagFunc::readLiberty(library, viewName, libertyOpt->getValue());
        }


        // Open design

        const char *cellString = cellOpt->getValue();
        const oa::oaScalarName cellName(nativeNS, cellString);
        oa::oaDesign *design;
        if ((design = oa::oaDesign::find(libName, cellName, viewName))) {
            ;
        } else {
            // open for read only
            design = oa::oaDesign::open(libName, cellName, viewName, 'r');
        }

        if (!design) {
            std::cerr << "ERROR: Could not open design" << std::endl;
            exit(1);
        }
        int rndCycle;

        if (rndCycleOpt->isGiven())
            rndCycle= atoi(rndCycleOpt->getValue());
        else
            rndCycle= 1024;
        // Add my code here.
        if (cell2Opt->isGiven())
        {
            const char *cellString2 = cell2Opt->getValue();
            const oa::oaScalarName cellName2(nativeNS, cellString2);
            oa::oaDesign *design2;
            if ((design2 = oa::oaDesign::find(libName, cellName2, viewName))) {
                ;
            } else {
                // open for read only
                design2 = oa::oaDesign::open(libName, cellName2, viewName, 'r');
            }

            if (!design2) {
                std::cerr << "ERROR: Could not open design" << std::endl;
                exit(1);
            }
            checkEqui(design, design2, rndCycle);
        }
        if (net1Opt->isGiven() && net2Opt->isGiven()) 
        {
            oa::oaName net1Name(verilogNS, net1Opt->getValue());
            oa::oaName net2Name(verilogNS, net2Opt->getValue());
            oa::oaNet *net1= oa::oaNet::find(design->getTopBlock(), net1Name);
            oa::oaNet *net2= oa::oaNet::find(design->getTopBlock(), net2Name);
            if (!net1) 
            {
                std::cerr << "ERROR: Unknown net " << net1Opt->getValue() << std::endl;
                exit(0);
            }
            if (!net2)
            {
                std::cerr << "ERROR: Unknown net " << net2Opt->getValue() << std::endl;
                exit(0);
            }
            checkTwoNet(design, net1, net2, rndCycle);
        }
        
        // Close design

        oa::oaDesign *openDesign;
        oa::oaIter<oa::oaDesign> designIter(oa::oaDesign::getOpenDesigns());
        while((openDesign = designIter.getNext())) {
            oa::oaString designString;
            openDesign->getTopModule()->getName(nativeNS, designString);
            openDesign->purge();
        }

        // Handle errors

    } catch(oa::oaException &e) {
        std::cerr << "ERROR: " << e.getMsg() << std::endl;
        assert(false);
        return 1;
    }

    return 0;
}


// vim: ci et sw=4 sts=4
