/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron Hurst <ahurst@eecs.berkeley.edu>

ChangeLog:
2005-08-18: ChangeLog started
*/

#include <iostream>
#include "oaDesignDB.h"
#include "oagFunc.h"
#include "oagFuncManager.h"
#include "oagUtilOption.h"
#include "oagUtilOptionParser.h"
#include "oagMapperSimple.h"

using namespace std;


// *****************************************************************************
// findOpenLib()
//
/// \brief Find an already open, opens from disk, or creates a library.
///
// *****************************************************************************
oa::oaLib *
findOpenLib(const oa::oaScalarName libName, const char *libString) {
    oa::oaLib *lib = NULL;

    if ((lib = oa::oaLib::find(libName)))
        ;
    else if (!oa::oaLib::exists(libString)) {
        lib = oa::oaLib::create(libName, libString);
        oa::oaLibDefList *defList = oa::oaLibDefList::get("lib.defs", 'a');
        oa::oaLibDef::create(defList, libName, libString);
        defList->save();
    } else {
        lib = oa::oaLib::open(libName, libString); 
    }

    return lib;
}


// *****************************************************************************
// findOpenDesign()
//
/// \brief Find an already open, opens from disk, or creates a design.
///
/// Opens designs from disk for read.
//
// *****************************************************************************
oa::oaDesign *
findOpenDesign(oa::oaScalarName libName, oa::oaScalarName viewName,
               const char *cellString) {
    const oa::oaNativeNS nativeNS;
    const oa::oaScalarName cellName(nativeNS, cellString);
    oa::oaDesign *design = NULL;

    if ((design = oa::oaDesign::find(libName, cellName, viewName))) {
        ;
    } else {
        // open for read only
        try {
            return oa::oaDesign::open(libName, cellName, viewName, 'r');
        } catch (oa::oaException &e) {}
    }
    return design;
}


// *****************************************************************************
// main()
//
// *****************************************************************************
int main(int argc, const char **argv) {

    // Define the options of interest

    oagUtil::OptionParser options("Map a design to AND, NOT and sequential "
            "cells.");

    oagUtil::Option *designLibOpt = options.add("design_lib", "Input library "
            "name for the design to be mapped", true, "lib");
    oagUtil::Option *designViewOpt = options.add("design_view", "Input view "
            "name for the design to be mapped (default: netlist)", false, 
            "view");
    oagUtil::Option *cellOpt = options.add("cell", "The cell to be mapped "
            "(including hierarchical instances)", true, "cell");
    oagUtil::Option *libraryLibOpt = options.add("library_lib", "Input library "
            "name for the library with the AND, NOT and sequential cells", 
            true, "library");
    oagUtil::Option *libraryViewOpt = options.add("library_view", "Input view "
            "name for the library (default: netlist)", false, "view");
    oagUtil::Option *andCellOpt = options.add("and_cell", "AND cell used for "
            "mapping", true, "cell");
    oagUtil::Option *notCellOpt = options.add("not_cell", "NOT cell used for "
            "mapping", true, "cell");
    oagUtil::Option *seqCellOpt = options.add("seq_cell", "Sequential cell "
            "used for mapping", true, "cell");
    oagUtil::Option *clockNetOpt = options.add("clock_net", "Single global clock net name",
             false, "net");
    oagUtil::Option *clockTriggerOpt = options.add("clock_trig", "Single global clock net trigger type (default: pos)",
             false, "pos|neg");
    oagUtil::Option *resetNetOpt = options.add("reset_net", "Single global reset net name",
             false, "net");
    oagUtil::Option *resetTriggerOpt = options.add("reset_trig", "Single global reset net trigger type (default: pos)",
             false, "pos|neg");
    oagUtil::Option *flattenOpt = options.add("flatten", "Flatten the cell "
            "before mapping", false);
    oagUtil::Option *saveOpt = options.add("save", "Save the design", false);

    if (!options.parse(argc, argv)) {
        cerr << options.getMessage();
        exit(0);
    }

    // Get the names supplied, or fill in the default values.

    const char *libraryLibString = libraryLibOpt->getValue();
    const char *designLibString = designLibOpt->getValue();
    const char *libraryViewString 
        = libraryViewOpt->isGiven() ? libraryViewOpt->getValue() : "netlist";
    const char *designViewString 
        = designViewOpt->isGiven() ? designViewOpt->getValue() : "netlist";
    const char *cellString = cellOpt->getValue();

    try {

        // Initialize OpenAccess

        oa::oaDesignInit();
        oa::oaLibDefList::openLibs();
        oagFunc::initialize();

        const oa::oaNativeNS nativeNS;
        const oa::oaVerilogNS verilogNS;
        const oa::oaScalarName libraryLibName(nativeNS, libraryLibString);
        const oa::oaScalarName libraryViewName(nativeNS, libraryViewString);
        const oa::oaScalarName designLibName(nativeNS, designLibString);
        const oa::oaScalarName designViewName(nativeNS, designViewString);

        // Find, create, or load the libraries.
        findOpenLib(libraryLibName, libraryLibString);
        findOpenLib(designLibName, designLibString);

        // Find or open the design to map

        const oa::oaScalarName cellName(nativeNS, cellString);
        oa::oaDesign *design;
        if ((design = oa::oaDesign::find(designLibName, cellName, designViewName)))
            ;
        else if (saveOpt->isGiven()) {
            // open for read and write
            design = oa::oaDesign::open(designLibName, cellName, designViewName, 'a');
        } else {
            // open for read only
            design = oa::oaDesign::open(designLibName, cellName, designViewName, 'r');
        }    

        // Find or open the library cells

        oa::oaDesign *andDesign = findOpenDesign(libraryLibName, 
                libraryViewName, andCellOpt->getValue());
        oa::oaDesign *notDesign = findOpenDesign(libraryLibName, 
                libraryViewName, notCellOpt->getValue());
        oa::oaDesign *seqDesign = findOpenDesign(libraryLibName, 
                libraryViewName, seqCellOpt->getValue());

        // Flatten

        if (flattenOpt->isGiven()) {
            cout << "Flattening \'" << cellString << "\' ... " << endl;
            // oagFunc::collapseAllInstancesAllLevels(design->getTopModule());
            // question - Why is this in comments? -- Christoph
        }

        // Map

        cout << "Mapping \'" << cellString << "\' ... " << endl;
        oagMapper::Simple simpleMapper(andDesign, notDesign, seqDesign);
        
        if (clockNetOpt->isGiven()) {
            oa::oaScalarName clockName(verilogNS, clockNetOpt->getValue());
            oa::oaModNet *clockNet = oa::oaModNet::find(design->getTopModule(), clockName);
            if (!clockNet) {
                cerr << "ERROR: Could not find clock net: " << clockNetOpt->getValue() << endl;
                exit(0);
            }
            if (clockNet->getNumBits() != 1) {
                cerr << "ERROR: Clock net not a single bit net: " << clockNetOpt->getValue() << endl;
                exit(0);                
            }
            oagMapper::Simple::TriggerType clockTrigger = oagMapper::Simple::POSEDGE;
            if (clockTriggerOpt->isGiven()) {
                if (!strcmp(clockTriggerOpt->getValue(), "pos")) {
                    clockTrigger = oagMapper::Simple::POSEDGE;
                } else if (!strcmp(clockTriggerOpt->getValue(), "neg")) {
                    clockTrigger = oagMapper::Simple::NEGEDGE;
                } else {
                    cerr << "ERROR: Unknown trigger type.  Use \"pos\" or \"neg\"" << endl;
                    exit(0);
                }
            }
            simpleMapper.useSingleGlobalClock(oagFunc::toBitNet(clockNet), clockTrigger);
        }
        if (resetNetOpt->isGiven()) {
            oa::oaScalarName resetName(verilogNS, resetNetOpt->getValue());
            oa::oaModNet *resetNet = oa::oaModNet::find(design->getTopModule(), resetName);
            if (!resetNet) {
                cerr << "ERROR: Could not find reset net: " << resetNetOpt->getValue() << endl;
                exit(0);
            }
            if (resetNet->getNumBits() != 1) {
                cerr << "ERROR: Reset net not a single bit net: " << resetNetOpt->getValue() << endl;
                exit(0);
            }
            oagMapper::Simple::TriggerType resetTrigger = oagMapper::Simple::POSEDGE;
            if (resetTriggerOpt->isGiven()) {
                if (!strcmp(resetTriggerOpt->getValue(), "pos")) {
                    resetTrigger = oagMapper::Simple::POSEDGE;
                } else if (!strcmp(resetTriggerOpt->getValue(), "neg")) {
                    resetTrigger = oagMapper::Simple::NEGEDGE;
                } else {
                    cerr << "ERROR: Unknown trigger type.  Use \"pos\" or \"neg\"" << endl;
                    exit(0);
                }
            }
            simpleMapper.useSingleGlobalReset(oagFunc::toBitNet(resetNet), resetTrigger);
        }
        
        simpleMapper.techmap(design->getTopModule());
        simpleMapper.mergeEquivalentNets(design->getTopModule());

        // Create a pointer to technology database in library

        oa::oaLibDMData *libData = oa::oaLibDMData::open(designLibName, 'a');
        assert(libData);
        if(oa::oaStringProp *prop =
           static_cast<oa::oaStringProp*> (oa::oaProp::find(libData, "techLibName"))) {
            prop->setValue(libraryLibString);
        } else {
            oa::oaStringProp::create(libData, "techLibName", libraryLibString);
        }
        libData->save();
        libData->purge();

        // Save 

        if (saveOpt->isGiven()) {
            cout << "Saving \'" << cellString << "\' ... " << endl;
            design->save();
        }

        // Close

        oa::oaDesign *openDesign;
        oa::oaIter<oa::oaDesign> designIter(oa::oaDesign::getOpenDesigns());
        while((openDesign = designIter.getNext())) {
            oa::oaString designNameString;
            openDesign->getTopModule()->getName(nativeNS, designNameString);
            openDesign->purge();
        }

    } catch(oa::oaException &e) {
        cerr << "ERROR: " << e.getMsg() << endl;
        assert(false);
        return 1;
    }

    return 0;
}
