/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-08-18: ChangeLog started
*/

#if !defined(oagMapperUtils_P)
#define oagMapperUtils_P

#include "oaDesignDB.h"
#include "oagFunc.h"
#include "oagFuncModGraph.h"
#include <list>

using namespace std;

namespace oagMapper {

// *****************************************************************************
// Utils
//
/// \brief A utility class for commonly used mapping-related functions.
//
// *****************************************************************************

class Utils {
  friend class Table;

  public:
  
    Utils();

    typedef enum{POSEDGE, NEGEDGE} TriggerType;

    /// \name Informational queries
    /// @{
               
    void                printGateUsage(oa::oaModule *target);
       
    /// @}
    /// \name Structural clean-up operations
    /// @{

    void                mergeEquivalentNets(oa::oaModule *module);
    void                removeDanglingNets(oa::oaModule *module);
    
    /// @}
    /// \name Important library cells
    /// @{
    
    oa::oaDesign       *andGate;
    oa::oaModTerm      *andInput[2], *andOutput;
    
    oa::oaDesign       *notGate;
    oa::oaModTerm      *notInput, *notOutput;
    
    oa::oaDesign       *seqGate;
    oa::oaModTerm      *seqInput, *seqOutput;
    oa::oaModTerm      *seqReset, *seqPreset, *seqClock;
    TriggerType         seqResetTrigger, seqPresetTrigger, seqClockTrigger;
    
    /// @}
    /// \name Identification of library cell terminals by function
    /// @{
    
  public:
    oa::oaDesign *      identifyAnd(const std::list<oa::oaDesign*> &lib);
    void                identifyAndTerminals();
    oa::oaDesign *      identifyNot(const std::list<oa::oaDesign*> &lib);
    void                identifyNotTerminals();
    oa::oaDesign *      identifySeq(const std::list<oa::oaDesign*> &lib);
    void                identifySeqTerminalsByName();

    void                setAnd(oa::oaDesign *gate) { andGate = gate; }
    void                setNot(oa::oaDesign *gate) { notGate = gate; }
    void                setSeq(oa::oaDesign *gate) { seqGate = gate; }

    oa::oaDesign *      getAnd() { return andGate; }
    oa::oaDesign *      getNot() { return notGate; }
    oa::oaDesign *      getSeq() { return seqGate; }

    /// @}
    /// \name Mapping sequential signals
    /// @{
    
    void                addGlobalClock(oa::oaModBitNet *clock, const TriggerType trig = POSEDGE) {
      /// \brief Manually identifies a net as a global clock.
      /// \param clock the global clock net
      /// \param trig trigger type (POSEDGE or NEGEDGE)
      if (trig == POSEDGE) posClocks.insert(clock); else negClocks.insert(clock);
    }
    void                addGlobalReset(oa::oaModBitNet *reset, const TriggerType trig = POSEDGE) {
      /// \brief Manually identifies a net as a global reset.
      /// \param reset the global reset net
      /// \param trig trigger type (POSEDGE or NEGEDGE)
      if (trig == POSEDGE) posAsyncResets.insert(reset); else negAsyncResets.insert(reset);
    }
    void                identifyControls(oa::oaModule *module);
    void                identifyControls(oagFunc::ModRef seq);
    void                removeAsyncResetsFromLogic(oa::oaModule *module);

    void                connectAllControls(oagFunc::ModRef ref, oa::oaModInst *inst);
 
  protected:

    set<oa::oaModBitNet*>     posAsyncResets, negAsyncResets; 
    set<oa::oaModBitNet*>     posClocks, negClocks;
    
    void                      connectControl(oa::oaModTerm *term, 
                                             oa::oaModInst *inst,
                                             oa::oaModBitNet *net,
                                             bool inverted = false);
   
    /// @}

};

}
#endif
