/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron Hurst <ahurst@eecs.berkeley.edu>

ChangeLog:
2005-07-20: ChangeLog started
*/

#include <iostream>
#include "oaDesignDB.h"
#include "oagFuncOccGraph.h"
#include "oagFuncManager.h"
#include "oagFuncPrint.h"
#include "oagFuncSynthesis.h"
#include "oagUtilOption.h"
#include "oagUtilOptionParser.h"


// *****************************************************************************
// findOpenLib()
//
/// \brief Find an already open, opens from disk, or creates a library.
///
// *****************************************************************************
oa::oaLib *
findOpenLib(const oa::oaScalarName libName, const char *libString) {
    oa::oaLib *lib = NULL;

    if ((lib = oa::oaLib::find(libName)))
        ;
    else if (!oa::oaLib::exists(libString)) {
        lib = oa::oaLib::create(libName, libString);
        oa::oaLibDefList *defList = oa::oaLibDefList::get("lib.defs", 'a');
        oa::oaLibDef::create(defList, libName, libString);
        defList->save();
    } else {
        lib = oa::oaLib::open(libName, libString); 
    }

    return lib;
}


// *****************************************************************************
// main()
//
// *****************************************************************************
int main(int argc, const char **argv) {

    std::cout << "*****************************************************************************" << std::endl;
    std::cout << "Tool:   verilog2oaFunc   Package:    oagFunc     version:  01/2007" << std::endl;
    std::cout << "*****************************************************************************" << std::endl;
    std::cout << std::endl;

    oagUtil::OptionParser options("Convert Verilog to OpenAccess with "
        "functional information");

    oagUtil::Option *libOpt = options.add("lib", "Input and output library "
            "name", true, "library");
    oagUtil::Option *viewOpt = options.add("view", "Input and output view name "
            "(default: netlist)", false, "view");
    oagUtil::Option *cellOpt = options.add("cell", "Print info about this cell "
            "or module, flatten or uniquify it", false, "cell");
    oagUtil::Option *libertyOpt = options.add("liberty", ".lib library file", 
            false, "file");
    oagUtil::Option *verilogOpt = options.add("verilog", "Verilog file", false, 
            "file");
    oagUtil::Option *leafLibsOpt = options.add("leafLibs", "Library files for leaf cells",
             false, "\"libList\"");
    oagUtil::Option *leafViewsOpt = options.add("leafViews", "Library views for leaf cells", 
             false, "\"viewList\"");
    oagUtil::Option *nextStatesOpt = options.add("next_states", "Print the "
            "next states", false);
    oagUtil::Option *reuseOpt = options.add("reuse", "Re-use existing structures "
            "and append functional descriptions", false);
    oagUtil::Option *hierarchicalNamesOpt = options.add("hierarchical_names", 
            "Print the hierarchcal names", false);
    oagUtil::Option *flattenOpt = options.add("flatten", "Flatten the cell "
            "(cell needs to be specified)", false);
    oagUtil::Option *uniquifyOpt = options.add("uniquify", "Uniquify the cell "
            "(only possible if -flatten is not used)", false, "cell");

    if (!options.parse(argc, argv)) {
        std::cerr << options.getMessage();
        exit(0);
    }

    const char *libString = libOpt->getValue();
    const char *viewString 
        = viewOpt->isGiven() ? viewOpt->getValue(): "netlist";

    try {

        // Initialize OpenAccess

        oa::oaDesignInit();
        oa::oaLibDefList::openLibs();
        oagFunc::initialize();

        const oa::oaNativeNS nativeNS;
        const oa::oaScalarName libName(nativeNS, libString);
        const oa::oaScalarName viewName(nativeNS, viewString);

        // Find, create, or load the design library and view

        oa::oaLib *library = findOpenLib(libName, libString);
        assert(library);

        // Find, create, or load additional libraries (if given)
        
        if (leafLibsOpt->isGiven()) {
            assert(strlen(leafLibsOpt->getValue())<256);
            char opt[256];
            strcpy(opt, leafLibsOpt->getValue());
            const char *tok = strtok(opt, "\" ,");
            while(tok) {
              oa::oaScalarName leafLibName(nativeNS, tok);
              oa::oaLib *leafLib = findOpenLib(leafLibName, tok);
              leafLib->getAccess(oa::oacWriteLibAccess);
              assert(leafLib);
              oagFunc::Synthesis::addLeafLibrary(leafLib);
              tok = strtok(NULL, "\" ,");
            }
            if (leafViewsOpt->isGiven()) {
              assert(strlen(leafViewsOpt->getValue())<256);
              char opt[256];
              strcpy(opt, leafViewsOpt->getValue());
              char *tok = strtok(opt, "\" ,");
              while(tok) {
                oa::oaScalarName leafViewName(nativeNS, tok);
                oagFunc::Synthesis::addLeafView(leafViewName);
                tok = strtok(NULL, "\" ,");
              }
            } else {
                std::cout << "WARNING: No leaf views provided. Assuming \"netlist\" view" << endl;
                oa::oaScalarName leafViewName(nativeNS, "netlist");
                oagFunc::Synthesis::addLeafView(leafViewName);
            }
        }

        // Should existing structures be re-used? 
         
        if (reuseOpt->isGiven()) { 
            oagFunc::Synthesis::setOverwriteStructure(false); 
        }

        // Read designs from a Liberty file

        if (libertyOpt->isGiven()) {
            // read in a cell library
            oagFunc::readLiberty(library, viewName, libertyOpt->getValue());
        }

        // Read designs from a Verilog file

        if (verilogOpt->isGiven()) {
            // read in functional Verilog description
            oagFunc::readVerilog(library, viewName, verilogOpt->getValue());
        }

        // Print statistics about a module (if given)

        if (cellOpt->isGiven()) {

            const char *cellString = cellOpt->getValue();
            const oa::oaScalarName cellName(nativeNS, cellString);
            oa::oaDesign *design;
            if ((design = oa::oaDesign::find(libName, cellName, viewName))) {
                ;
            } else {
                // open for read and write
                design = oa::oaDesign::open(libName, cellName, viewName, 'a');
            }

            printFunctionality(design, nextStatesOpt->isGiven(), 
                    hierarchicalNamesOpt->isGiven());

            if (flattenOpt->isGiven()) {
                std::cout << "Flattening \'" << cellString << "\' ..." 
                    << std::endl;
                oagFunc::collapseAllInstancesAllLevels(design->getTopModule());

                printFunctionality(design, nextStatesOpt->isGiven(), 
                        hierarchicalNamesOpt->isGiven());

            } else if (uniquifyOpt->isGiven()) {

                const oa::oaVerilogNS verilogNS;
                oa::oaSimpleName occName(verilogNS, uniquifyOpt->getValue());
                oa::oaOccInst *occInst = oa::oaOccInst::find(design->getTopOccurrence(), occName);

                if (!occInst) {
                    std::cerr << "ERROR: Could not find instance: " << uniquifyOpt->getValue() << std::endl;
                    exit(0);
                }

                std::cout << "Uniquifying \'" << uniquifyOpt->getValue() << "\' ..." << std::endl;

                oa::oaOccurrence *occurrence = occInst->getMasterOccurrence();
                assert(occurrence);
                oagFunc::uniquify(occurrence);

                printFunctionality(design, nextStatesOpt->isGiven(), hierarchicalNamesOpt->isGiven());

            }
        }

        // Save and close

        std::cout << "Saving designs: ";
        oa::oaDesign *openDesign;
        oa::oaIter<oa::oaDesign> designIter(oa::oaDesign::getOpenDesigns());
        while((openDesign = designIter.getNext())) {
            if (openDesign->getLib() == library && openDesign->getMode() != 'r') {
                oa::oaString designString;
                openDesign->getTopModule()->getName(nativeNS, designString);
                std::cout << designString << " ";
                openDesign->save();
            }
            openDesign->purge();
        }
        std::cout << std::endl << std::endl;

        // Handle errors

    } catch(oa::oaException &e) {
        std::cerr << "ERROR: " << e.getMsg() << std::endl;
        return 1;
    }

    return 0;
}
