/* (c) Copyright 2004-2007, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2007-01-21: ChangeLog started
*/

#include "oaDesignDB.h"
#include "oagFunc.h"
#include "oagFuncManager.h"
#include "oagFuncModGraph.h"
#include <list>
#include <string>
#include <fstream>

#include "oagFuncDebug.h"

namespace oagFunc {


// *****************************************************************************
// getBaseName()
//
/// \brief Returns the identifier (without range) of a net.
///
/// The supplied net can be scalar, a vector bit, or a vector net.
///
/// \param name the net name
/// \return string
//
// *****************************************************************************
string
getBaseName(oa::oaName &name) {

  oa::oaString str;  
  const oa::oaVerilogNS verilogNS;

  switch(name.getType()) {
  case oa::oacScalarNameType: {
    oa::oaScalarName *scalarName = name.getScalar();
    assert(scalarName);
    scalarName->get(verilogNS, str);
    return string(str);
  }
  case oa::oacVectorNameType: {
    oa::oaVectorName *vectorName = name.getVector();
    assert(vectorName);
    vectorName->getBaseName(verilogNS, str);
    return string(str);
  }
  case oa::oacVectorBitNameType: {
    oa::oaVectorBitName *vectorBitName = name.getVectorBit(); 
    assert(vectorBitName);
    vectorBitName->getBaseName(verilogNS, str);
    return string(str);
  }          
  default:
    cerr << "ERROR: Unsupported name type" << endl;
    QUIT_ON_ERROR;
  }

  return string("");
}


// *****************************************************************************
// writeNetDef()
//
/// \brief Writes a net in a Verilog net definition.
///
/// This function is needed because bus net definitions have the range
/// declaration before the identifier, unlike their later references inside
/// the module (and the format returned by OpenAccess in the VerilogNS).
///
/// \param ss the string stream onto which to append the net definition
/// \param net the net
//
// *****************************************************************************
void
writeNetDef(stringstream & ss, oa::oaModNet *net) {
  assert(net);
  
  oa::oaString str;
  const oa::oaVerilogNS verilogNS;

  oa::oaName name;
  net->getName(name);
  switch(name.getType()) {
  case oa::oacScalarNameType: {
    oa::oaScalarName *scalarName = name.getScalar();
    assert(scalarName);
    scalarName->get(verilogNS, str);
    ss << str;
    break;
  }
  case oa::oacVectorNameType: {
    oa::oaVectorName *vectorName = name.getVector();
    assert(vectorName);
    ss << "[" << vectorName->getStart()
       << ":" << vectorName->getStop() << "] ";
    vectorName->getBaseName(verilogNS, str);
    ss << str;    
    break;
  }
  case oa::oacVectorBitNameType: {
    oa::oaVectorBitName *vectorBitName = name.getVectorBit(); 
    assert(vectorBitName);
    ss << "[" << vectorBitName->getIndex() << "] ";
    vectorBitName->getBaseName(verilogNS, str);
    ss << str; 
    break;
  }          
  default:
    cerr << "ERROR: Unsupported name type" << endl;
    QUIT_ON_ERROR;
  }
}


// *****************************************************************************
// refToVerilogName()
//
/// \brief Generates a Verilog net name for a ModRef.
///
/// The name of a ModRef with Ref XXX will be "ai_XXX".
///
/// The constant string NULL_ASSIGNMENT can be modified to adjust the 
/// default Verilog assignment of null references.
///
/// \param ref
/// \return name
//
// *****************************************************************************
string
refToVerilogName(ModRef ref) {
  // const char *NULL_ASSIGNMENT = "1'b0";
  const char *NULL_ASSIGNMENT = "1'bx";

  stringstream result;
  if (ModGraph::isNull(ref)) {
    result << NULL_ASSIGNMENT;
  } else if (ModGraph::isInverted(ref)) {
    result << "~" << "ai_" 
            << ModGraph::getNonInverted(ref).ref;
  } else {
    result << "ai_" << ref.ref;
  }
  return result.str();
}


// *****************************************************************************
// writeVerilog()
//
/// \brief Write an OpenAccess design (and functional description) to a Verilog file.
///
/// The Verilog description will be appended to the end of the file.
///
/// \li Modules, instances, scalar nets, bus nets, and ports/terminals are 
///     mapped directly from OpenAccess database objects to Verilog
/// \li Nets which are undriven by a signal but marked equivalent (through
///     the OpenAccess markEquivalent method) to another net that is driven
///     by a signal will be assigned using a direct Verilog "assign"
/// \li All AND and TERMINAL nodes are mapped to new Verilog nets names "ai_XXX"
///     where XXX is the uninverted Ref of the node.  The functionality
///     (including the inversions on inputs) is described using simple
///     "assign" statements, one per node
///
/// If there are pre-existing OpenAccess nets of a conflicting name "ai_XXX", an
/// error will be generated.
///
/// \param design
/// \param filename
//
// *****************************************************************************
void            
writeVerilog(oa::oaDesign *design, const char *filename) {
    assert(design);
    assert(filename);
 
    const oa::oaVerilogNS verilogNS;

    ofstream outFile;
    outFile.open(filename, ios::app); 
    if (!outFile) {
      cerr << "ERROR: Could not open file " << filename << " for output" << endl;
      QUIT_ON_ERROR;
    }

    outFile << "// Written by oagFunc Verilog writer" << endl << endl;
     
    // for each module...
    oa::oaModule *module;
    oa::oaIter<oa::oaModule> moduleIter(design->getModules());
    while((module = moduleIter.getNext())) {
      oa::oaString str;
      
      // 1. write module definition
      module->getName(verilogNS, str);
      outFile << "module " << str << "(";

      // 1.a. portlist
      bool first = true;
      oa::oaModTerm *term;
      oa::oaIter<oa::oaModTerm> termIter(module->getTerms(oacTermIterNotImplicit));
      while((term = termIter.getNext())) {
        oa::oaName termName, netName;

        term->getNet()->getName(netName);
        term->getName(termName);

        if (!first) {
          outFile << ", ";
        }

        if (getBaseName(termName) == getBaseName(netName)) {
          outFile << getBaseName(termName);
        } else {
          outFile << "." << getBaseName(termName) << "(" 
                  << getBaseName(netName) << ")";
        }

        first = false;
      }
      outFile << ");" << endl;

      // 2. definitions

      // 2.a. inputs/outputs
      termIter.reset();
      while((term = termIter.getNext())) {
        stringstream ss;
        writeNetDef(ss, term->getNet());
        if(term->getTermType() == oa::oacInputTermType) {
          outFile << "\t" << "input " << ss.str() << ";" << endl;
        } else if(term->getTermType() == oa::oacOutputTermType) {
          outFile << "\t" << "output " << ss.str() << ";" << endl;
        }
      }

      // 2.b. nets
      oa::oaModNet *net;
      oa::oaIter<oa::oaModNet> netIter(module->getNets());
      while((net = netIter.getNext())) {
        // do not write if this is already a terminal
        if (net->getTerms().isEmpty()) {
          stringstream ss;
          writeNetDef(ss, net);
          if (net->getSigType() == oa::oacSignalSigType) {
            outFile << "\t" << "wire " << ss.str() << ";" << endl;
          } else if (net->getSigType() == oa::oacTieHiSigType) {
            outFile << "\t" << "supply1 " << ss.str() << ";" << endl;
          }  else if (net->getSigType() == oa::oacTieLoSigType) {
            outFile << "\t" << "supply0 " << ss.str() << ";" << endl;
          }
        }
      }

      // 2.c. a/i graph
      list<ModRef> outputNodes, all;
      if (Manager::hasManager(module->getDesign())) {
        ModGraph::getOutputs(module, outputNodes);
        ModGraph::getTransitiveFanin(outputNodes, all, true, true);
        all.splice(all.end(), outputNodes);
        
        for(list<ModRef>::iterator nodeIter = all.begin();
            nodeIter != all.end(); nodeIter++) {
          ModRef ref = *nodeIter;
          if (ModGraph::isSequential(ref)) {
            outFile << "\t" << "reg " << "ai_" << ref.ref << ";" << endl;
          } else {
            outFile << "\t" << "wire " << "ai_" << ref.ref << ";" << endl;
          }
        }
        outFile << endl;
      }      

      // 3. equivalent nets
      oa::oaIter<oa::oaModNet> singleNetIter(module->getNets(oacNetIterSingleBit));
      while((net = netIter.getNext())) {     
        oa::oaModBitNet *bitNet = toBitNet(net);
        oa::oaModBitNet *driverNet = ModGraph::findDriverOfEquivalentNets(bitNet);
        if (driverNet) {
          bitNet->getName(verilogNS, str);
          outFile << "\t" << "assign " << str << " = ";
          driverNet->getName(verilogNS, str);
          outFile << str << ";" << endl;
        }
      }

      // 4. a/i graph
      if (Manager::hasManager(module->getDesign())) {
        for(list<ModRef>::iterator nodeIter = all.begin();
            nodeIter != all.end(); nodeIter++) {
          ModRef ref = *nodeIter;
          
          // check for naming conflict
          char aiNetString[64];
          sprintf(aiNetString, "ai_%d", ref.ref);
          if (oa::oaModScalarNet::find(module, oa::oaScalarName(verilogNS, aiNetString))) {
            cerr << "ERROR : Module has pre-existing net of conflicting name type \"ai_XXX\"" << endl;
            QUIT_ON_ERROR;
          }
          
          if (ModGraph::isTerminal(ref)) {
            // TERMINAL node
            
            oa::oaModBitNet *net = ModGraph::getNetToAiConnection(ref);
            if (net) {
              net->getName(verilogNS, str);
            }
            
            ModRef driver = ModGraph::getTerminalDriver(ref);
            if (!ModGraph::isNull(driver)) {
              // driven by another node            
              outFile << "\t" << "assign " << aiNetString << " = " 
                      << refToVerilogName(driver) << ";" << endl;
              
              if (net && net->getSigType() == oa::oacSignalSigType) {
                // net driven by terminal
                outFile << "\t" << "assign " << str << " = " 
                        << "ai_" << ref.ref << ";" << endl;
              }
              
            } else if (net) {
              // driven by net
              outFile << "\t" << "assign " << aiNetString 
                      << " = " << str << ";" << endl;
            } else {
              // driven by nothing
            }
            
          } else if (ModGraph::isAnd(ref)) {
            
            // AND node
            ModRef left = ModGraph::getAndLeft(ref),
              right = ModGraph::getAndRight(ref);
            outFile << "\t" << "assign " << aiNetString << " = ";
            
            outFile << refToVerilogName(left) << " & "
                    << refToVerilogName(right) << ";" << endl;
            
          } else if (ModGraph::isSequential(ref)) {
            
            // SEQ node
            outFile << "\t" << "always @(";
            oagAi::Node::SequentialData *data = oagFunc::ModGraph::getSequentialData(ref);
            if (!data) {
              cerr << "WARNING: Sequential node without any control signals (i.e. clocks) defined."
                   << " Using generic clock \"clk\"" << endl;
              outFile << "posedge clk";
            } else {
              first = true;
              for(map<string, oagAi::Ref>::iterator sigIter = data->signals.begin();
                  sigIter != data->signals.end(); sigIter++) {
                if (!first) {
                  outFile << " or ";
                }
                first = false;
                if (sigIter->first.find("posedge") != sigIter->first.npos) {
                  outFile << "posedge " << refToVerilogName(ModRef(sigIter->second,module));
                }
                if (sigIter->first == "clocked_on") {
                  cerr << "WARNING: Latch-type sequential nodes are not yet supported."
                       << " Written as a posedge-triggered register." << endl;
                  outFile << "posedge " << refToVerilogName(ModRef(sigIter->second,module));                
                }
                if (sigIter->first.find("negedge") != sigIter->first.npos) {
                  outFile << "negedge " << refToVerilogName(ModRef(sigIter->second,module));
                }
              }
            }
            outFile << ")" << endl;
            
            ModRef nextState = ModGraph::getNextState(ref);
            outFile << "\t\t" << aiNetString << " <= " << refToVerilogName(nextState)
                    << ";" << endl;
            
          } else if (ref == ModGraph::constantZero(module)) {
            outFile << "\t" << "assign " << aiNetString << " = 1'b0;" << endl;
          }
        }
      }
      
      // 5. module instantiations
      oa::oaModInst *inst;
      oa::oaIter<oa::oaModInst> instIter(module->getInsts());
      while((inst = instIter.getNext())) {
        oa::oaModule *master = inst->getMasterModule();
        if (!master) {
          cerr << "ERROR: Could not bind to master module" << endl;
          QUIT_ON_INTERNAL_ERROR;
        }
        master->getName(verilogNS, str);
        outFile << "\t" << str << " ";

        inst->getName(verilogNS, str);
        outFile << str << " (";

        // are the connections by name or position?
        bool first = true;
        if (inst->usesTermPositions()) {
          // connected by position

          set<int> validPositions;
          oa::oaModInstTerm *instTerm;
          oa::oaIter<oa::oaModInstTerm> instTermIter(inst->getInstTerms());
          while((instTerm = instTermIter.getNext())) {
            validPositions.insert(instTerm->getTermPosition());
          }
          for(int i=*(validPositions.begin()); i<=*(validPositions.rbegin()); i++) {
            assert(validPositions.find(i) != validPositions.end());
            instTerm = oa::oaModInstTerm::find(inst, i);
            assert(instTerm);
            if (!first) {
              outFile << ", ";
            }
            oa::oaModNet *net = instTerm->getNet();
            if (net) {
              first = false;
              net->getName(verilogNS, str);
              outFile << str;
            } else {
              outFile << " "; // no assignment
            }
          }
          
        } else {
          // connected by name

          oa::oaModInstTerm *instTerm;
          oa::oaIter<oa::oaModInstTerm> instTermIter(inst->getInstTerms());
          while((instTerm = instTermIter.getNext())) {
            oa::oaModNet *net = instTerm->getNet();
            if (net) {
              if (!first) {
                outFile << ", ";
              }
              first = false;

              oa::oaName termName;
              instTerm->getTermName(termName);
              net->getName(verilogNS, str);
              outFile << "." << getBaseName(termName) << "(" << str << ")";
            }
          }
        }

        outFile << ");" << endl;
      }

      outFile << "endmodule" << endl << endl;
    }

    outFile.close();
}


// *****************************************************************************
// writeVerilog()
//
/// \brief Write an OpenAccess library (and functional descriptions) to a Verilog file.
///
/// The Verilog descriptions will be appended to the end of the file.
///
/// \param lib
/// \param filename
//
// *****************************************************************************
void            
writeVerilog(oa::oaLib *lib, oa::oaView *view, const char *filename) {
    assert(lib);
    assert(filename);
    assert(view);

    oa::oaScalarName libName;
    lib->getName(libName);
    oa::oaScalarName viewName;
    view->getName(viewName);
    
    oa::oaCellView *cellView;
    oa::oaIter<oa::oaCellView> cellViewIter(view->getCellViews());
    while((cellView = cellViewIter.getNext())) {
      oa::oaScalarName cellName;
      cellView->getCell()->getName(cellName);

      oa::oaDesign *design = oa::oaDesign::find(libName, cellName, viewName);
      if (design) {
        writeVerilog(design, filename);
      }
    }
}

}
