/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-05-31: ChangeLog started
*/

#include "oaDesignDB.h"
#include "oagFuncManager.h"
#include "oagFuncModGraph.h"
#include "oagFuncVerilogSynthesis.h"
#include <list>
#include <set>
#include <string>
#include <sstream>

#include "oagFuncDebug.h"

#define sign(a)  ( (a)==0?0: ((a)<0?-1:1) )
#define abs(a)   ( (a)>=0?(a):-(a) )
#define max(a,b) ( (a)>(b)?(a):(b) )
#define min(a,b) ( (a)>(b)?(b):(a) )

namespace oagFunc {

// *****************************************************************************
// Class static variable definitions.
// *****************************************************************************

VerilogDesign::Module                *VerilogSynthesis::currentVmodule;
VerilogSynthesis::ParameterValues    *VerilogSynthesis::currentParams;
set<string>                           VerilogSynthesis::finishedModules;
map<string, VerilogSynthesis::Bounds> VerilogSynthesis::twoDimRegisters;


// *****************************************************************************
// synthesize()
//
/// \brief Synthesize a Verilog design into OpenAccess
///
/// Builds an OpenAccess module for every unparameterized Verilog module.
//
// *****************************************************************************
void            
VerilogSynthesis::synthesize(VerilogDesign *vDesign) {
    assert(vDesign);

    // create a namespace object that will be used through the synthesis
    nameSpace = new oa::oaVerilogNS();

    // synthesize the unparameterized versions of all modules in the design
    for(list<VerilogDesign::Module*>::iterator modIter = vDesign->modules.begin(); 
            modIter != vDesign->modules.end(); ++modIter) {
        VerilogDesign::Module *vModule = *modIter;
      
        if(finishedModules.find(vModule->name) == finishedModules.end()) {
            list<ConstantValue> emptyParameters; // i.e. unparameterized
            synthesizeModule(vModule, emptyParameters);
        }
    }

    // clean up
    delete nameSpace;
}


// *****************************************************************************
// getParameterizedModuleName()
//
/// \brief Creates a module name to reflect a set of parameters
//
/// Returns a string to identify a version of a module with a particular set
/// of parameters.
///
/// Currently, the parameter values are appended to the module name
/// and delimited with the "_" character.
//
// *****************************************************************************
string
VerilogSynthesis::getParameterizedModuleName(VerilogDesign::Module *vModule, 
                                             list<ConstantValue> &parameters) {
    assert(vModule);

    string              parameterizedModuleName(vModule->name);
    ParameterValues     localParameters;
    bool                nonDefault = false;

    // if no parameters, then all values must be default
    if (parameters.size() == 0)
        return parameterizedModuleName;

    // scan the parameters by position
    list<ConstantValue>::iterator it1 = parameters.begin();
    list<VerilogDesign::Declaration*>::iterator it2 = vModule->parameters.begin();
    while(it2 != vModule->parameters.end()) {
        // compute the default value
        ConstantValue c;
        evaluateConstantExpression(c, (*it2)->value, &localParameters );
    
        if (it1 == parameters.end()) {
            // append the default value to the module name
            stringstream suffix;
            suffix << "_" << c.intValue;
            appendSuffix(parameterizedModuleName, suffix.str());

            localParameters[(*it2)->name] = c;
            ++it2;
        } else {
            // test if we have differed from the default settings
            if (it1->intValue != c.intValue) {
                nonDefault = true;
            }
       
            // append this parameter value to the module name
            stringstream suffix;
            suffix << "_" << it1->intValue;
            appendSuffix(parameterizedModuleName, suffix.str());

            localParameters[(*it2)->name] = *it1;
            ++it1; ++it2;
        }
    }
    
    // are there too many parameters?
    if (it1 != parameters.end()) {
        cerr << "WARNING: Too many parameters for module " << vModule->name << ".  Ignoring extra" << endl;
    }
    
    // if no parameter still match default, no suffix is necessary
    if (!nonDefault) {
        parameterizedModuleName = vModule->name;
    }
    
    return parameterizedModuleName;
}
    
    
// *****************************************************************************
// synthesizeModule()
//
/// \brief Synthesize a Verilog module into an OpenAccess object
///
/// Builds an OpenAccess module for a Verilog module and a set of parameters.
///
/// Because the parameters can significantly affect the structure of netlist,
/// parameterization must be done at this point.  The different versions of
/// the same module with differing parameters will be identified by name.
//
// *****************************************************************************
void
VerilogSynthesis::synthesizeModule(VerilogDesign::Module *vModule, 
                                   list<ConstantValue> &parameters) {
    assert(vModule);
    assert(nameSpace);
    currentVmodule = vModule;
    
    // set up the local parameter value lookup table
    currentParams = new ParameterValues;
    assert(currentParams);
    list<ConstantValue>::iterator it1 = parameters.begin();
    list<VerilogDesign::Declaration*>::iterator it2 = vModule->parameters.begin();
    
    while(it2 != vModule->parameters.end()) {
        // compute the default value
        ConstantValue c;
        evaluateConstantExpression(c, (*it2)->value, currentParams );
        if (it1 == parameters.end()) {
            (*currentParams)[(*it2)->name] = c;
            ++it2;
        } else {
            (*currentParams)[(*it2)->name] = *it1;
            ++it1; ++it2;
        }
    }
    
    // are there too many parameters?
    if (it1 != parameters.end()) {
        cerr << "WARNING: Too many parameters for module " << vModule->name << ".  Ignoring extra" << endl;
    }

    // create the module
    string parameterizedModuleName = getParameterizedModuleName(vModule, parameters);
    DEBUG_PRINTLN("module " << parameterizedModuleName)        
    currentModule = createModule(parameterizedModuleName);
    currentManager = NULL;
    
    // synthesis
    synthesizeModuleNets();
    synthesizeModuleTerms();
    synthesizeModuleAssigns();
    synthesizeModuleInsts();
    synthesizeModuleFunc();

    // if not structural, reflect structural equivalences in graph
    if (currentManager) {
      ModGraph::connectEquivalentNetsInGraph(currentModule);
    }
    
    // clean up
    currentVmodule = NULL;
    assert(currentParams);
    delete currentParams;
    currentParams = NULL;

    // store the completed module
    finishedModules.insert(parameterizedModuleName);
    DEBUG_PRINTLN("endmodule " << parameterizedModuleName);
}


// *****************************************************************************
// synthesizeModuleNets()
//
/// \brief Synthesize the nets in a Verilog module into OpenAccess objects
///
/// Builds OpenAccess nets for a Verilog wires, integers, and regs.
///
// *****************************************************************************
void
VerilogSynthesis::synthesizeModuleNets() {

    // create a net for each declaration

    for(list<VerilogDesign::Declaration*>::iterator decIter = currentVmodule->declarations.begin();
        decIter != currentVmodule->declarations.end(); ++decIter) {
        VerilogDesign::Declaration *vDeclaration = *decIter;
        oa::oaModNet *net = NULL;
        list<oa::oaModBitNet*> bits;
        
        ConstantValue start, stop;
        if (vDeclaration->start == NULL) {
            if (vDeclaration->start2D == NULL) {
                if (!(net = findScalarNet(vDeclaration->name))) {
                    net = createScalarNet(vDeclaration->name);
                }
                DEBUG_PRINT("\t net " << vDeclaration->name);
            } else {
                ConstantValue start2D, stop2D;
                evaluateConstantExpression(start2D, vDeclaration->start2D, currentParams); 
                evaluateConstantExpression(stop2D, vDeclaration->stop2D, currentParams);
                Bounds bounds;
                bounds.upper = max(start2D.intValue, stop2D.intValue); bounds.lower = 
                                                                         min(start2D.intValue, stop2D.intValue);
                twoDimRegisters[vDeclaration->name] = bounds;
                for(int i=bounds.lower; i<=bounds.upper; i++) {
                    // construct suffix to identify net in 2-d vector 
                    string vectorName(vDeclaration->name);
                    stringstream suffix;
                    suffix << "_" << i << "_";
                    appendSuffix(vectorName, suffix.str());
                    if (!(net = findScalarNet(vectorName))) {
                        net = createScalarNet(vectorName);
                    }
                }
                DEBUG_PRINT("\t net " << vDeclaration->name << "<" << start2D.intValue 
                            << ":" << stop2D.intValue << ">");
            }
            // for all bits
            // bits.push_back();
        } else {
            evaluateConstantExpression(start, vDeclaration->start, currentParams); 
            evaluateConstantExpression(stop, vDeclaration->stop, currentParams);
            if (vDeclaration->start2D == NULL) {
                if (!(net = findBusNet(vDeclaration->name))) {            
                    net = createBusNet(vDeclaration->name, start.intValue, stop.intValue);
                }
                DEBUG_PRINT("\t net " << vDeclaration->name << "[" << start.intValue 
                            << ":" << stop.intValue << "]");
            } else {
                // assume that multi-dimensional nets are supply lines
                if (vDeclaration->type == VerilogDesign::Declaration::SUPPLY0 ||
                    vDeclaration->type == VerilogDesign::Declaration::SUPPLY1) {
                    cerr << "ERROR: Two dimensional supply nets are not supported" << endl;
                    QUIT_ON_ERROR;
                } 
                ConstantValue start2D, stop2D;
                evaluateConstantExpression(start2D, vDeclaration->start2D, currentParams); 
                evaluateConstantExpression(stop2D, vDeclaration->stop2D, currentParams);
                Bounds bounds;
                bounds.upper = max(start2D.intValue, stop2D.intValue);
                bounds.lower = min(start2D.intValue, stop2D.intValue);
                twoDimRegisters[vDeclaration->name] = bounds;
                for(int i=bounds.lower; i<=bounds.upper; i++) {
                    // construct suffix to identify net in 2-d vector 
                    string vectorName(vDeclaration->name);
                    stringstream suffix;
                    suffix << "_" << i << "_";
                    appendSuffix(vectorName, suffix.str());
                    if (!(net = findScalarNet(vectorName))) {                    
                        net = createBusNet(vectorName, start.intValue, stop.intValue);
                    }
                }
                DEBUG_PRINT("\t net " << vDeclaration->name << "[" << start.intValue 
                            << ":" << stop.intValue << "]" 
                            << "<" << start2D.intValue << ":" << stop2D.intValue << ">");
            }
        }
        
        switch(vDeclaration->type) {
        case VerilogDesign::Declaration::INPUT:
            DEBUG_PRINTMORE(" (input)" << endl);
            break;
        case VerilogDesign::Declaration::OUTPUT:
            DEBUG_PRINTMORE(" (output)" << endl);
            break;
        case VerilogDesign::Declaration::WIRE:
            DEBUG_PRINTMORE(" (wire)" << endl);
            break;
        case VerilogDesign::Declaration::SUPPLY0: {
            DEBUG_PRINTMORE(" (supply0)" << endl);
            // tie to supply
            oa::oaModScalarNet *supplyNet = oa::oaModScalarNet::find(currentModule, 
                                                                     oa::oaScalarName(oa::oaNativeNS(), 
                                                                                      "tie0"));
            for(list<oa::oaModBitNet*>::iterator bitIter = bits.begin();
                bitIter != bits.end(); bitIter++) {
              (*bitIter)->makeEquivalent(supplyNet);
            }
            break;
            }
        case VerilogDesign::Declaration::SUPPLY1: {
            DEBUG_PRINTMORE(" (supply1)" << endl);
            // tie to supply
            oa::oaModScalarNet *supplyNet = oa::oaModScalarNet::find(currentModule, 
                                                                     oa::oaScalarName(oa::oaNativeNS(), 
                                                                                      "tie1"));
            for(list<oa::oaModBitNet*>::iterator bitIter = bits.begin();
                bitIter != bits.end(); bitIter++) {
              (*bitIter)->makeEquivalent(supplyNet);
            }
            break;
            }
        case VerilogDesign::Declaration::INOUT:
            DEBUG_PRINTMORE(" (inout)" << endl);
            break;
        case VerilogDesign::Declaration::REG:
            DEBUG_PRINTMORE(" (reg)" << endl);
            break;
        default:
            QUIT_ON_INTERNAL_ERROR;
            break;
        }

    }
}


// *****************************************************************************
// synthesizeModuleTerms()
//
/// \brief Synthesize the ports in a Verilog module into OpenAccess objects
///
/// Builds OpenAccess terms for a Verilog ports.
///
// *****************************************************************************
void
VerilogSynthesis::synthesizeModuleTerms() {

  // create a term for each port (and check that it has been declared inside the module)
  
  int position = 0;
  for(list<VerilogDesign::Port*>::iterator portIter = currentVmodule->ports->begin();
      portIter != currentVmodule->ports->end(); ++portIter, ++position) {
      VerilogDesign::Port *vPort = *portIter;
      DEBUG_PRINTLN("\t port " << position << " " << vPort->externalName);
      
      // search for a corresponding declaration
      bool foundCorrespondingDeclaration = false;
      VerilogDesign::Declaration *vDeclaration = NULL;
      for(list<VerilogDesign::Declaration*>::iterator decIter = currentVmodule->declarations.begin();
          decIter != currentVmodule->declarations.end(); ++decIter) {
          vDeclaration = *decIter;
          if (vDeclaration->name == vPort->internalName) {
              foundCorrespondingDeclaration = true;
              break;
          }
      }
      if (!foundCorrespondingDeclaration) {
        cerr << "ERROR: Port " << vPort->internalName << " not declared in module body" << endl;
        QUIT_ON_ERROR;
      }
      oa::oaModNet *net = findNet(vPort->internalName);
      assert(net);
    
      assert(vDeclaration);
      switch(vDeclaration->type) {
      case VerilogDesign::Declaration::INPUT:
        createTerm(net, vPort->externalName.c_str(), oa::oacInputTermType, position);
        break;
      case VerilogDesign::Declaration::OUTPUT:
        createTerm(net, vPort->externalName.c_str(), oa::oacOutputTermType, position);
        break;
      case VerilogDesign::Declaration::INOUT:
        createTerm(net, vPort->externalName.c_str(), oa::oacInputOutputTermType, position);
        break;            
      default:
        QUIT_ON_INTERNAL_ERROR;
      }
      
  }
}


// *****************************************************************************
// synthesizeModuleAssigns()
//
/// \brief Synthesize the continuous assignments in a Verilog module
///
/// One of several things is done with a continuous assignment.  If the assignment
/// involves a constant or two nets, the objects are marked as equivalent nets
/// in OpenAccess.  If the assignment involves any sort of functional operator,
/// the assign will be implemented through the attached AI graph.
///
// *****************************************************************************
void
VerilogSynthesis::synthesizeModuleAssigns() {

    // continuous assignments
    
    for(list<VerilogDesign::Assignment*>::iterator assignIter = currentVmodule->assignments->begin();
        assignIter != currentVmodule->assignments->end();
        ++assignIter) {
        VerilogDesign::Assignment *assignment = *assignIter;
        
        LvalRefBus lval;
        MultiRefBus value;
        evaluateLval(lval, assignment->lval, NULL, currentParams);
        evaluateExpression(value, assignment->value, NULL, currentParams);
        DEBUG_PRINTLN("\t assign [" << lval.size() << "] = [" << value.size() << "]");
        
        LvalRefBus::reverse_iterator leftIter = lval.rbegin();
        MultiRefBus::reverse_iterator rightIter = value.rbegin();
        while(leftIter != lval.rend()) {
            assert((*leftIter)->conditionalLval.empty());
            oa::oaModBitNet *net = (*leftIter)->unconditionalLval;
            assert(net);

            if (rightIter == value.rend()) {
                assignMultiRef(net, constantZero());
            } else {
                assignMultiRef(net, *rightIter);
                ++rightIter;
            }   
            
            ++leftIter;
        }

    }
}


// *****************************************************************************
// synthesizeModuleInsts()
//
/// \brief Synthesize the instantiations in a Verilog module into OpenAccess objects
///
/// Builds OpenAccess insts for a Verilog structural instantiations.
///
// *****************************************************************************
void
VerilogSynthesis::synthesizeModuleInsts() {

    // instantiations
    
    for(list<VerilogDesign::Instantiation*>::iterator instIter = currentVmodule->instantiations->begin();
        instIter != currentVmodule->instantiations->end();
        ++instIter) {
        VerilogDesign::Instantiation *instantiation = *instIter;
        oa::oaModule *master = NULL;
        
        // is this an instantiation of a primitive, a Verilog module, or an external preexisting cell?

        // a primitive?
        if (instantiation->primitive != VerilogDesign::Instantiation::ISNT_PRIMITIVE) {
            MultiRefBus rvals;
            MultiRef    rval;
            assert(instantiation->connections->back()->name == "in");
            evaluateExpression(rvals, instantiation->connections->back()->value, NULL, currentParams);
            
            switch(instantiation->primitive) {
              case VerilogDesign::Instantiation::AND:
                rval = reductionAnd(rvals);
                break;
              case VerilogDesign::Instantiation::NAND:
                rval = notOf(reductionAnd(rvals));
                break;
              case VerilogDesign::Instantiation::OR:
                rval = reductionOr(rvals);
                break;
              case VerilogDesign::Instantiation::NOR:
                rval = notOf(reductionOr(rvals));
                break;
              case VerilogDesign::Instantiation::XOR:
                rval = reductionXor(rvals);
                break;
              case VerilogDesign::Instantiation::XNOR:
                rval = notOf(reductionXor(rvals));
                break;
              case VerilogDesign::Instantiation::NOT:
                assert(rvals.size() == 1);
                rval = notOf(rvals.front());
                break;
              case VerilogDesign::Instantiation::BUF:
                assert(rvals.size() == 1);
                rval = rvals.front();
                break;
              case VerilogDesign::Instantiation::ISNT_PRIMITIVE:
                break;
              default:
                QUIT_ON_INTERNAL_ERROR;
            }
           
            LvalRefBus  lvals;
            assert(instantiation->connections->front()->name == "out");
            evaluateLval(lvals, instantiation->connections->front()->value, NULL, currentParams);
            
            DEBUG_PRINTLN("\t primitive [" << lvals.size() << "] = [" << rvals.size() << "]");
            LvalRefBus::iterator leftIter = lvals.begin();
            while(leftIter != lvals.end()) {
                assert((*leftIter)->conditionalLval.empty());
                oa::oaModBitNet *net = (*leftIter)->unconditionalLval;
                assert(net);

                assignMultiRef(net, rval);
                ++leftIter;
            }
            
            continue;
        }

        assert(instantiation->parameters);

        // find the Verilog definition of this module
        VerilogDesign::Module *instVmodule = NULL;
        for(list<VerilogDesign::Module*>::iterator modIter = currentVmodule->design->modules.begin(); 
            modIter != currentVmodule->design->modules.end(); ++modIter) {
            if ((*modIter)->name == instantiation->type) {
                instVmodule = *modIter;
                break;
            }
        }

        // an instantiation of a Verilog module
        if (instVmodule) {

            // decode parameters
            list<ConstantValue> instParameters;
            for(list<VerilogDesign::Expression*>::iterator paramIter = instantiation->parameters->begin(); 
                paramIter != instantiation->parameters->end(); 
                ++paramIter) {
                // evaluate value of constant expression
                ConstantValue c;
                evaluateConstantExpression(c, *paramIter, currentParams);
                instParameters.push_back(c);
            }
        
            // get module name
            string parameterizedInstName = getParameterizedModuleName(instVmodule, instParameters);
            DEBUG_PRINTLN("\t instance of Verilog module " << parameterizedInstName 
                          << " named " << instantiation->name);
    
            // has this module already been synthesized?
            master = findModule(parameterizedInstName);
            if (master == NULL) {
                oa::oaModule *lastModule = currentModule;
                VerilogDesign::Module *lastVmodule = currentVmodule;
                Manager *lastManager = currentManager;
                ParameterValues *lastParams = currentParams;
                synthesizeModule(instVmodule, instParameters);
                master = currentModule;
                currentParams = lastParams;
                currentModule = lastModule;
                currentManager = lastManager;
                currentVmodule = lastVmodule;
            }

        }

        // an instantiation of a leaf library cell
        else {

            DEBUG_PRINTLN("\t instance of leaf cell " << instantiation->type 
                          << " named " << instantiation->name);

            master = findModule(instantiation->type);
            // can't find  this cell
            if (master == NULL) { 
                cerr << "ERROR: No Verilog or pre-existing definition of module " << instantiation->type << endl;
                QUIT_ON_ERROR;
            }

            // can't parameterize an external cell
            if (!instantiation->parameters->empty()) {
                cerr << "ERROR: The non-Verilog module " << instantiation->type
                     << " can not be parameterized" << endl;
                QUIT_ON_ERROR;
            }

        }
        
        // create instance
        oa::oaModInst *inst = instantiateModule(master->getDesign(), instantiation->name);

        // connect ports
        for(list<VerilogDesign::PortConnection*>::iterator conIter = instantiation->connections->begin();
            conIter != instantiation->connections->end();
            ++conIter) {
            VerilogDesign::PortConnection *connection = *conIter;
            
            // evaluate the expression to be connected
            MultiRefBus value;
            if (connection->value == NULL) {
                // if the port is unconnected, ignore
                if (connection->name.empty()) {
                    DEBUG_PRINTLN("\t\t " << connection->position << " <- floating");
                } else {
                    DEBUG_PRINTLN("\t\t " << connection->name << " <- floating");
                }
                continue;    
            } else {
                evaluateExpression(value, connection->value, NULL, currentParams);
            }
            
            // construct dummy net name
            string dummyName(instantiation->name);
            if (connection->name.empty()) {
              stringstream suffix;
              suffix << "_" << connection->position << "_connection_";
              appendSuffix(dummyName, suffix.str());
            } else {
              stringstream suffix;
              suffix << "_" << connection->name << "_connection_";
              appendSuffix(dummyName, suffix.str());
            }
            
            if (value.size() == 1) {
                // single-bit connection
                
                oa::oaModBitNet *net;
                // if this is an oagAi::Ref, create a dummy net
                if (value.front().type == AIREF) {
                    net = createScalarNet(dummyName);
                    assignMultiRef(net, value.front());
                } else if (value.front().type == NET) {
                    assert(value.front().net);
                    // if this is an implicit net (i.e. a vector bit), 
                    // a dummy explicit net needs to be created for the connection
                    if (value.front().net->isImplicit()) {
                        net = createScalarNet(dummyName);
                        net->makeEquivalent(value.front().net);
                    } else {
                        net = value.front().net;
                    }
                } else {
                    // a net of NEITHER type
                    QUIT_ON_INTERNAL_ERROR;
                }
                
                if (connection->name.empty()) {
                    connectPort(inst, net, connection->position);
                } else {
                    connectPort(inst, net, connection->name);
                }
            } else {
                // bus connection
                
                // 1. make a dummy bus net of the appropriate width
                oa::oaModBusNet *bus = createBusNet(dummyName, value.size()-1, 0);
        
                // 2. assign bus net bits to be equivalent to supplied nets
                MultiRefBus::iterator bitIter = value.begin();
                int b = 0;
                while(bitIter != value.end()) {
                    MultiRef m = *bitIter;
                    if (m.type == AIREF) {
                        assignMultiRef(bus->getBit(b), m);
                    }
                    else if (m.type == NET) {
                        m.net->makeEquivalent(bus->getBit(b));
                    } else {
                        // a net of NEITHER type
                        QUIT_ON_INTERNAL_ERROR;
                    }
                    ++bitIter; ++b;                
                }
        
                // 3. connect bus net to InstTerm
                if (connection->name.empty()) {
                    connectPort(inst, bus, connection->position);
                } else {
                    connectPort(inst, bus, connection->name);
                }
            }
            
            if (connection->name.empty()) {
                DEBUG_PRINTLN("\t\t " << connection->position << " <- [" << value.size() << "]");
            } else {
                DEBUG_PRINTLN("\t\t " << connection->name << " <- [" << value.size() << "]");
            }
        }
    }
}


// *****************************************************************************
// synthesizeModuleFunc()
//
/// \brief Synthesize the behavioral descriptions in a Verilog module into an AI description
///
/// Builds the associated AI graph to describe the Verilog behavior description.
///
// *****************************************************************************
void
VerilogSynthesis::synthesizeModuleFunc() {

    // synthesize always blocks
    
    for(list<VerilogDesign::AlwaysBlock*>::iterator alwaysIter = currentVmodule->alwaysBlocks->begin();
        alwaysIter != currentVmodule->alwaysBlocks->end();
        ++alwaysIter) {
        VerilogDesign::AlwaysBlock *always = *alwaysIter;
        
        DEBUG_PRINTLN("\t always block");
        set<oa::oaModBitNet *> normalTriggers, posTriggers, negTriggers;
        
        // categorize triggers
        for(list<VerilogDesign::Trigger*>::iterator triggerIter = always->triggers->begin();
            triggerIter != always->triggers->end();
            ++triggerIter) {
            VerilogDesign::Trigger *trigger = *triggerIter;

            MultiRefBus t;
            evaluateExpression(t, trigger->net, NULL, currentParams);
            for(MultiRefBus::iterator bitIter = t.begin(); bitIter != t.end(); ++bitIter) {
                assert(bitIter->type == NET);
                if (trigger->type == VerilogDesign::Trigger::POSEDGE) {
                    posTriggers.insert(bitIter->net);
                }
                else if (trigger->type == VerilogDesign::Trigger::NEGEDGE) {
                    negTriggers.insert(bitIter->net);
                }
                else {
                    normalTriggers.insert(bitIter->net);   
                }
            }
        }

        ProceduralState state;
        if (posTriggers.empty() && negTriggers.empty()) {
            DEBUG_PRINTLN("\t\t combinational/latch");
            state.isRegister = false;
        } else {
            DEBUG_PRINTLN("\t\t register");
            state.isRegister = true;
        }
        
        synthesizeBehavioral(always->action, state, currentParams);

/*
        // UNIMPLEMENTED
        // verify that always triggers are complete
        for(set<oa::oaModBitNet*>::iterator it = state.dependencies.begin(); 
            it != state.dependencies.end(); it++) {
            if (triggers.find(*it) == triggers.end()) {
                (*it)->getName(*nameSpace, str);
                cerr << "WARNING: net " << str << " not included in event trigger list" << endl;
            }
        }
        // verify that triggers don't include lvals
*/

        // set the functionality for all of the modified nets
        DEBUG_PRINTLN("\t nets assigned: ");
        for(map<oa::oaModBitNet*, MultiRef>::iterator it = state.blockingAssignments.begin(); 
            it != state.blockingAssignments.end(); it++) {
            if (state.isRegister) {
                stringstream str;
                oa::oaName name;
                it->first->getName(name);
                if (name.getVectorBit()) {
                    oa::oaString baseName;
                    name.getVectorBit()->getBaseName(baseName);
                    str << baseName << "_" << name.getVectorBit()->getIndex();
                } else {
                    oa::oaString scalarName;
                    assert(name.getScalar());
                    name.getScalar()->get(scalarName);
                    str << scalarName;
                }
                cerr << "WARNING: Blocking assignment in register " << str << endl;
                MultiRef seqRef = seq(it->second, str.str());
                assignMultiRef(it->first, seqRef);
                
                // append all of the negedge triggers as "trigger_XXX_negedge" and
                // posedge as "trigger_XXX_posedge"
                // the tech mapper will need to distinguish clocks from reset/preset
                int triggerID = 0;
                assert(seqRef.type == AIREF);
                for(set<oa::oaModBitNet *>::iterator trigIter = posTriggers.begin(); 
                    trigIter != posTriggers.end(); trigIter++, triggerID++) {
                    oa::oaModBitNet *trigNet = *trigIter;
                    stringstream label;
                    label << "trigger_" << triggerID << "_posedge";
                    annotateAsynchronousSignal(seqRef.ai, label.str(), MultiRef(trigNet));
                }
                for(set<oa::oaModBitNet *>::iterator trigIter = negTriggers.begin(); 
                    trigIter != negTriggers.end(); trigIter++, triggerID++) {
                    oa::oaModBitNet *trigNet = *trigIter;
                    stringstream label;
                    label << "trigger_" << triggerID << "_negedge";
                    annotateAsynchronousSignal(seqRef.ai, label.str(), MultiRef(trigNet));
                }
                
            } else {
                assignMultiRef(it->first, it->second);
            }
        }
        for(map<oa::oaModBitNet*, MultiRef>::iterator it = state.nonblockingAssignments.begin(); 
            it != state.nonblockingAssignments.end(); it++) {
            if (state.isRegister) {
                stringstream str;
                oa::oaName name;
                it->first->getName(name);
                if (name.getVectorBit()) {
                    oa::oaString baseName;
                    name.getVectorBit()->getBaseName(baseName);
                    str << baseName << "_" << name.getVectorBit()->getIndex();
                } else {
                    assert(name.getScalar());
                    oa::oaString scalarName;
                    name.getScalar()->get(scalarName);
                    str << scalarName;
                }
                
                MultiRef seqRef = seq(it->second, str.str());
                assignMultiRef(it->first, seqRef);
                
                // append all of the negedge triggers as "trigger_XXX_negedge" 
                // and posedge as "trigger_XXX_posedge"
                // the tech mapper will need to distinguish clocks from reset/preset
                int triggerID = 0;
                assert(seqRef.type == AIREF);
                for(set<oa::oaModBitNet *>::iterator trigIter = posTriggers.begin(); 
                    trigIter != posTriggers.end(); trigIter++, triggerID++) {
                    oa::oaModBitNet *trigNet = *trigIter;
                    stringstream label;
                    label << "trigger_" << triggerID << "_posedge";
                    annotateAsynchronousSignal(seqRef.ai, label.str(), MultiRef(trigNet));
                }
                for(set<oa::oaModBitNet *>::iterator trigIter = negTriggers.begin(); 
                    trigIter != negTriggers.end(); trigIter++, triggerID++) {
                    oa::oaModBitNet *trigNet = *trigIter;
                    stringstream label;
                    label << "trigger_" << triggerID << "_negedge";
                    annotateAsynchronousSignal(seqRef.ai, label.str(), MultiRef(trigNet));
                }
                
            } else {
                assignMultiRef(it->first, it->second);
            }
        }        
    }
}


// *****************************************************************************
// synthesizeBehavioral()
//
/// \brief Synthesize a behavioral statement
//
// *****************************************************************************
void
VerilogSynthesis::synthesizeBehavioral(VerilogDesign::Statement *statement, 
                                       ProceduralState &state, 
                                       ParameterValues *parameters) {
    assert(nameSpace);
    assert(statement);
    switch(statement->type) {

      case VerilogDesign::Statement::NOP:
        break;
      
      case VerilogDesign::Statement::BLOCK:

        // there may be local variables defined for this block
        if (statement->begin_end.declarations != NULL)
        for(list<VerilogDesign::Declaration*>::iterator declIter = 
              statement->begin_end.declarations->begin();
            declIter != statement->begin_end.declarations->end();
            declIter++) {
            VerilogDesign::Declaration *vDeclaration = *declIter;
            // UNIMPLEMENTED: prefix register name with block name
        
            ConstantValue start, stop;
            if (vDeclaration->start == NULL) {
                if (vDeclaration->start2D == NULL) {
                    if (!(findScalarNet(vDeclaration->name))) {
                        createScalarNet(vDeclaration->name);
                    }
                    DEBUG_PRINT("\t net " << vDeclaration->name);
                } else {
                    ConstantValue start2D, stop2D;
                    evaluateConstantExpression(start2D, vDeclaration->start2D, parameters); 
                    evaluateConstantExpression(stop2D, vDeclaration->stop2D, parameters);
                    Bounds bounds;
                    bounds.upper = max(start2D.intValue, stop2D.intValue);
                    bounds.lower = min(start2D.intValue, stop2D.intValue);
                    twoDimRegisters[vDeclaration->name] = bounds;
                    for(int i=bounds.lower; i<=bounds.upper; i++) {
                        // construct suffix to identify net in 2-d vector 
                        string vectorName(vDeclaration->name);
                        stringstream suffix;
                        suffix << "_" << i << "_";
                        appendSuffix(vectorName, suffix.str());
                        if (!(findScalarNet(vectorName))) {
                            createScalarNet(vectorName);
                        }
                    }
                    DEBUG_PRINT("\t net " << vDeclaration->name << "<" 
                                << start2D.intValue << ":" << stop2D.intValue << ">");
                }
            } else {
                evaluateConstantExpression(start, vDeclaration->start, parameters); 
                evaluateConstantExpression(stop, vDeclaration->stop, parameters);
                if (vDeclaration->start2D == NULL) {
                    if (!(findBusNet(vDeclaration->name))) {
                        createBusNet(vDeclaration->name, start.intValue, stop.intValue);
                    }
                    DEBUG_PRINT("\t net " << vDeclaration->name << "[" 
                                << start.intValue << ":" << stop.intValue << "]");
                } else {
                    ConstantValue start2D, stop2D;
                    evaluateConstantExpression(start2D, vDeclaration->start2D, parameters); 
                    evaluateConstantExpression(stop2D, vDeclaration->stop2D, parameters);
                    Bounds bounds;
                    bounds.upper = max(start2D.intValue, stop2D.intValue); 
                    bounds.lower = min(start2D.intValue, stop2D.intValue);
                    twoDimRegisters[vDeclaration->name] = bounds;
                    for(int i=bounds.lower; i<=bounds.upper; i++) {
                        // construct suffix to identify net in 2-d vector 
                        string vectorName(vDeclaration->name);
                        stringstream suffix;
                        suffix << "_" << i << "_";
                        appendSuffix(vectorName, suffix.str());
                        if (!(findBusNet(vectorName))) {                        
                            createBusNet(vectorName, start.intValue, stop.intValue);
                        }
                    }
                    DEBUG_PRINT("\t net " << vDeclaration->name << "[" << start.intValue 
                                << ":" << stop.intValue << "]" 
                                << "<" << start2D.intValue << ":" << stop2D.intValue << ">");
                }
            }
    
            switch(vDeclaration->type) {
            case VerilogDesign::Declaration::REG:
                DEBUG_PRINTMORE(" (reg)" << endl);
                break;
            default:
                QUIT_ON_INTERNAL_ERROR;
                break;
            }
        }

        DEBUG_PRINTLN("\t\t block");
        for(list<VerilogDesign::Statement*>::iterator stateIter = statement->begin_end.block->begin();
            stateIter != statement->begin_end.block->end(); ++stateIter) {
            synthesizeBehavioral(*stateIter, state, parameters);
        }
        break;
        
      case VerilogDesign::Statement::IF: {

        DEBUG_PRINTLN("\t\t if");
        MultiRefBus condition;
        assert(statement->ifc.condition);
        evaluateExpression(condition, statement->ifc.condition, &state, parameters);
        MultiRef select(reductionOr(condition));
        
        ProceduralState ifState(state), elseState(state);
        assert(statement->ifc.ifTrue);
        synthesizeBehavioral(statement->ifc.ifTrue, ifState, parameters);
        if (statement->ifc.ifFalse) {
            synthesizeBehavioral(statement->ifc.ifFalse, elseState, parameters);
        }
        
        // construct mux for all variables w/ different assignments
        for(map<oa::oaModBitNet*, MultiRef>::iterator it = ifState.blockingAssignments.begin(); 
            it != ifState.blockingAssignments.end(); it++) {
            oa::oaString str;
            it->first->getName(*nameSpace, str);
        
            if (elseState.blockingAssignments.find(it->first) != elseState.blockingAssignments.end()) {
                // combinational

                state.blockingAssignments[it->first] = mux(select, 
                                                           elseState.blockingAssignments[it->first], 
                                                           it->second);
            } else if (state.isRegister) {
                // mux previous state
            
                state.blockingAssignments[it->first] = mux(select, MultiRef(it->first), it->second); 
            } else {
                // implicit latch... transparent when condition
                state.blockingAssignments[it->first] = latch(select, it->second);
                
                oa::oaString str;
                it->first->getName(*nameSpace, str);
                cerr << "NOTE: Implicit latch on net " << currentVmodule->name << "." << str << endl;
            }
        }
        for(map<oa::oaModBitNet*, MultiRef>::iterator it = elseState.blockingAssignments.begin(); 
            it != elseState.blockingAssignments.end(); it++) {

            if (ifState.blockingAssignments.find(it->first) != ifState.blockingAssignments.end()) {
                ;
            } else if (state.isRegister) {
                // mux previous state
            
                state.blockingAssignments[it->first] = mux(select, it->second, MultiRef(it->first)); 
            } else {
                // implicit latch... transparent when !select
                state.blockingAssignments[it->first] = latch(notOf(select), it->second);
                
                oa::oaString str;
                it->first->getName(*nameSpace, str);
                cerr << "NOTE: Implicit latch on net " << currentVmodule->name << "." << str << endl;
            }
        }   

        // construct mux for all variables w/ different assignments
        for(map<oa::oaModBitNet*, MultiRef>::iterator it = ifState.nonblockingAssignments.begin(); 
            it != ifState.nonblockingAssignments.end(); it++) {
            oa::oaString str;
            it->first->getName(*nameSpace, str);
        
            if (elseState.nonblockingAssignments.find(it->first) != elseState.nonblockingAssignments.end()) {
                // combinational

                state.nonblockingAssignments[it->first] = mux(select, elseState.nonblockingAssignments[it->first], it->second);
            } else if (state.isRegister) {
                // mux previous state
            
                state.nonblockingAssignments[it->first] = mux(select, MultiRef(it->first), it->second); 
            } else {
                // implicit latch... transparent when condition
                state.nonblockingAssignments[it->first] = latch(select, it->second);
                
                oa::oaString str;
                it->first->getName(*nameSpace, str);
                cerr << "NOTE: Implicit latch on net " << currentVmodule->name << "." << str << endl;
            }
        }
        for(map<oa::oaModBitNet*, MultiRef>::iterator it = elseState.nonblockingAssignments.begin(); 
            it != elseState.nonblockingAssignments.end(); it++) {
            if (ifState.nonblockingAssignments.find(it->first) != ifState.nonblockingAssignments.end()) {
                ;
            } else if (state.isRegister) {
                // mux previous state
            
                state.nonblockingAssignments[it->first] = mux(select, it->second, MultiRef(it->first)); 
            } else {
                // implicit latch... transparent when !select
                state.nonblockingAssignments[it->first] = latch(notOf(select), it->second);
                
                oa::oaString str;
                it->first->getName(*nameSpace, str);
                cerr << "NOTE: Implicit latch on net " << currentVmodule->name << "." << str << endl;
            }
        }
        
        break;
        }

      case VerilogDesign::Statement::CASEZ:
      case VerilogDesign::Statement::CASEX:
      case VerilogDesign::Statement::CASE: {
        
        // DEBUG_PRINTLN("\t\t case num_labels=" << statement->cases->size());
        
        // 1. compute condition
        MultiRefBus          condition;
        evaluateExpression(condition, statement->ifc.condition, &state, parameters);
        
        list<ProceduralState *> caseActions;
        MultiRefBus             caseSelects;
        set<oa::oaModBitNet*>   modifiedBlockingAssignments, modifiednonblockingAssignments;
        ProceduralState        *defaultAction = NULL;
        MultiRef                defaultSelect = constantZero();
        
        // 2. compute the procedural state for each case
        // 3. compute the select lines for each case
        for(list<VerilogDesign::Case*>::iterator caseIter = statement->ifc.cases->begin();
            caseIter != statement->ifc.cases->end();
            ++caseIter) {
            VerilogDesign::Case *c = *caseIter;
            assert(c);
            
            ProceduralState *action = NULL;
            
            if (c->isDefault) {
                // this is the default case
                DEBUG_PRINTLN("\t\t default case");
            
                if (defaultAction != NULL) {
                    cerr << "ERROR: Multiple default cases" << endl;
                    QUIT_ON_ERROR;
                }

                action = new ProceduralState(state);
                synthesizeBehavioral(c->action, *action, parameters);            
                defaultAction = action;
                
            } else {
                // this is not a default case
                DEBUG_PRINTLN("\t\t case label conditions=" << c->conditions->size());
            
                MultiRef select = constantZero();
                bool     onePossiblyTrueCase = false;
                for(list<VerilogDesign::Expression*>::iterator condIter = c->conditions->begin();
                    condIter != c->conditions->end();
                    ++condIter) {
                    
                    MultiRefBus caseCondition;
                    VerilogDesign::Expression *condExpr = *condIter;
                    evaluateExpression(caseCondition, condExpr, &state, parameters);
                    zeroExpand(condition, caseCondition);
                    MultiRef caseMatchesCondition;
                    
                    if (isConstantExpression(condExpr, parameters) && 
                        isConstantExpression(statement->ifc.condition, parameters)) {

                        ConstantValue caseConstant, conditionConstant;
                        evaluateConstantExpression(caseConstant, condExpr, parameters);
                        evaluateConstantExpression(conditionConstant, statement->ifc.condition, parameters);
                        if (caseConstant.intValue == conditionConstant.intValue) {
                            caseMatchesCondition = constantOne();
                            DEBUG_PRINTLN("\t\t\t\t case always true");
                            onePossiblyTrueCase = true;
                        } else {
                            caseMatchesCondition = constantZero();
                            DEBUG_PRINTLN("\t\t\t\t case always false");                            
                        }

                    } else if (statement->type == VerilogDesign::Statement::CASEX) {
                        MultiRefBus l;
                        int dontCareMask = (condExpr->type == VerilogDesign::Expression::PRIMARY && 
                                            condExpr->primary->type == VerilogDesign::Primary::CONST ? condExpr->primary->number.xMask: 0);
                        MultiRefBus::reverse_iterator l1Iter = condition.rbegin(), l2Iter = caseCondition.rbegin();
                        while(l1Iter != condition.rend() && l2Iter != caseCondition.rend()) {
                            if (condExpr->type != VerilogDesign::Expression::PRIMARY || 
                                condExpr->primary->type != VerilogDesign::Primary::CONST || dontCareMask%2 == 0) {
                                l.push_back( notOf(xorOf(*l1Iter, *l2Iter)) );
                            }
                            dontCareMask = dontCareMask >> 1;
                            ++l1Iter; ++l2Iter;
                        }
                        onePossiblyTrueCase = true;
                        caseMatchesCondition = reductionAnd(l);
                    } else if (statement->type == VerilogDesign::Statement::CASEZ) {
                        MultiRefBus l;
                        int dontCareMask = (condExpr->type == VerilogDesign::Expression::PRIMARY && 
                                            condExpr->primary->type == VerilogDesign::Primary::CONST ? condExpr->primary->number.zMask: 0);
                        MultiRefBus::reverse_iterator l1Iter = condition.rbegin(), l2Iter = caseCondition.rbegin();
                        while(l1Iter != condition.rend() && l2Iter != caseCondition.rend()) {
                            if (condExpr->type != VerilogDesign::Expression::PRIMARY || 
                                condExpr->primary->type != VerilogDesign::Primary::CONST || dontCareMask%2 == 0) {
                                l.push_back( notOf(xorOf(*l1Iter, *l2Iter)) );
                            }
                            dontCareMask = dontCareMask >> 1;
                            ++l1Iter; ++l2Iter;
                        }
                        onePossiblyTrueCase = true;
                        caseMatchesCondition = reductionAnd(l);
                    } else {
                        onePossiblyTrueCase = true;
                        caseMatchesCondition = equalTo(caseCondition, condition);
                    }

                    select = orOf(select, caseMatchesCondition);
                }
                
                if (onePossiblyTrueCase) {
                    caseSelects.push_back(select);
                    if (action == NULL) {
                        action = new ProceduralState(state);
                        synthesizeBehavioral(c->action, *action, parameters);
                    }
                    caseActions.push_back(action);
                }
                defaultSelect = orOf(defaultSelect, select);                
            }
            
            // identify the nets that have been modified
            if (action != NULL) {
                for(map<oa::oaModBitNet*, MultiRef>::iterator postActionAssignment = 
                      action->blockingAssignments.begin(); 
                    postActionAssignment != action->blockingAssignments.end(); 
                    ++postActionAssignment) {

                    map<oa::oaModBitNet*, MultiRef>::iterator preActionAssignment = 
                      state.blockingAssignments.find(postActionAssignment->first);
                    // either the net hadn't been assigned before, or it was assigned to something different?
                    if (preActionAssignment == state.blockingAssignments.end() || 
                        !(postActionAssignment->second == preActionAssignment->second)) {
                        modifiedBlockingAssignments.insert(postActionAssignment->first);
                    }
                }
                for(map<oa::oaModBitNet*, MultiRef>::iterator postActionAssignment = 
                      action->nonblockingAssignments.begin(); 
                    postActionAssignment != action->nonblockingAssignments.end(); 
                    ++postActionAssignment) {

                    map<oa::oaModBitNet*, MultiRef>::iterator preActionAssignment = 
                      state.nonblockingAssignments.find(postActionAssignment->first);

                    // either the net hadn't been assigned before, or it was assigned to something different?
                    if (preActionAssignment == state.nonblockingAssignments.end() || 
                        !(postActionAssignment->second == preActionAssignment->second)) {
                        modifiednonblockingAssignments.insert(postActionAssignment->first);
                    }
                }
            }
        }
       
        defaultSelect = notOf(defaultSelect);
        
        // 4. for each set variable, build n-way mux.
        // 5. also compute the case where the value is nextState
        
        for(set<oa::oaModBitNet*>::iterator bitIter = modifiedBlockingAssignments.begin(); 
            bitIter != modifiedBlockingAssignments.end(); ++bitIter) {
            MultiRefBus        muxTerms;
            MultiRefBus        latchTerms;
            
            MultiRefBus::iterator              selectsIter = caseSelects.begin();
            list<ProceduralState *>::iterator  actionIter = caseActions.begin();
            
            while(selectsIter != caseSelects.end() && actionIter != caseActions.end()) {
                    
                if ((*actionIter)->blockingAssignments.find(*bitIter) == (*actionIter)->blockingAssignments.end()) {
                    // implicit latch
                    latchTerms.push_back(*selectsIter);
                } else {
                    // mux term
                    muxTerms.push_back(andOf(*selectsIter, (*actionIter)->blockingAssignments[*bitIter]));
                }
                
                ++selectsIter; ++actionIter;      
            }

            // handle default case, if it exists
            if (defaultAction != NULL) {
                if (defaultAction->blockingAssignments.find(*bitIter) == defaultAction->blockingAssignments.end()) {
                    // implicit latch
                    latchTerms.push_back(defaultSelect);
                } else {
                    // mux term
                    muxTerms.push_back(andOf(defaultSelect, defaultAction->blockingAssignments[*bitIter]));
                }
            }
                
            if (latchTerms.empty()) {
                oa::oaString str;
                (*bitIter)->getName(*nameSpace, str);
                DEBUG_PRINTLN("\t\t\t "<< str << " = case_mux" );
               
                state.blockingAssignments[*bitIter] = reductionOr(muxTerms);
            } else {
                oa::oaString str;
                (*bitIter)->getName(*nameSpace, str);
                cerr << "NOTE: Implicit latch on net " << currentVmodule->name << "." << str << endl;
                state.blockingAssignments[*bitIter] = latch(reductionOr(latchTerms), reductionOr(muxTerms));
            }

        }

        for(set<oa::oaModBitNet*>::iterator bitIter = modifiednonblockingAssignments.begin(); 
            bitIter != modifiednonblockingAssignments.end(); ++bitIter) {
            MultiRefBus        muxTerms;
            MultiRefBus        latchTerms;
            
            MultiRefBus::iterator              selectsIter = caseSelects.begin();
            list<ProceduralState *>::iterator  actionIter = caseActions.begin();
            
            while(selectsIter != caseSelects.end() && actionIter != caseActions.end()) {
                    
                if ((*actionIter)->nonblockingAssignments.find(*bitIter) == (*actionIter)->nonblockingAssignments.end()) {
                    // implicit latch
                    latchTerms.push_back(*selectsIter);
                } else {
                    // mux term
                    muxTerms.push_back(andOf(*selectsIter, (*actionIter)->nonblockingAssignments[*bitIter]));
                }
                
                ++selectsIter; ++actionIter;      
            }

            // handle default case, if it exists
            if (defaultAction != NULL) {
                if (defaultAction->nonblockingAssignments.find(*bitIter) == defaultAction->nonblockingAssignments.end()) {
                    // implicit latch
                    latchTerms.push_back(defaultSelect);
                } else {
                    // mux term
                    muxTerms.push_back(andOf(defaultSelect, defaultAction->nonblockingAssignments[*bitIter]));
                }
            }
                
            if (latchTerms.empty()) {
                oa::oaString str;
                (*bitIter)->getName(*nameSpace, str);
                DEBUG_PRINTLN("\t\t\t "<< str << " = case_mux" );
               
                state.nonblockingAssignments[*bitIter] = reductionOr(muxTerms);
            } else {
                oa::oaString str;
                (*bitIter)->getName(*nameSpace, str);
                cerr << "NOTE: Implicit latch on net " << currentVmodule->name << "." << str << endl;
                state.nonblockingAssignments[*bitIter] = latch(reductionOr(latchTerms), reductionOr(muxTerms));
            }

        }

        // free memory
        for(list<ProceduralState *>::iterator actionIter = caseActions.begin(); 
            actionIter != caseActions.end(); actionIter++) {
            delete (*actionIter);
        }
        
        break;
        }
        
      case VerilogDesign::Statement::BLOCKING_ASSIGNMENT: {

        LvalRefBus lval;
        MultiRefBus rval;
        evaluateLval(lval, statement->assign.lval, &state, parameters);
        evaluateExpression(rval, statement->assign.rval, &state, parameters);

        LvalRefBus::reverse_iterator leftIter = lval.rbegin();
        MultiRefBus::reverse_iterator rightIter = rval.rbegin();
        while(leftIter != lval.rend()) {
            MultiRef value;
            if (rightIter == rval.rend()) {
                value = constantZero();
            } else {
                value = *rightIter;
                ++rightIter;
            }
            
            if ((*leftIter)->type == VerilogSynthesis::LVAL_UNCONDITIONAL) {
                // unconditional assignment
                
                oa::oaModBitNet *net = (*leftIter)->unconditionalLval;
                state.blockingAssignments[net] = value;

                oa::oaString str;
                net->getName(*nameSpace, str);
                if (state.blockingAssignments[net].type == NET) {
                    oa::oaString str2;
                    state.blockingAssignments[net].net->getName(*nameSpace, str2);
                    DEBUG_PRINTLN("\t\t\t " << str << " = " << str2);
                } else {
                    DEBUG_PRINTLN("\t\t\t " << str << " = (func)");
                }
            } else if ((*leftIter)->type == LVAL_CONDITIONAL) {
                // conditional assignment
                
                for(list<ConditionalLvalRef>::iterator condIter = (*leftIter)->conditionalLval.begin();
                    condIter != (*leftIter)->conditionalLval.end();
                    ++condIter) {
                    
                    if (state.blockingAssignments.find(condIter->lval) != state.blockingAssignments.end()) {
                        // combinational

                        state.blockingAssignments[condIter->lval] = mux(condIter->condition, 
                                                                        state.blockingAssignments[condIter->lval], 
                                                                        value);
                    } else if (state.isRegister) {
                        // mux previous state
            
                        state.blockingAssignments[condIter->lval] = mux(condIter->condition, 
                                                                        MultiRef(condIter->lval), 
                                                                        value); 
                    } else {
                        // implicit latch... transparent when condition
                        state.blockingAssignments[condIter->lval] = latch(condIter->condition, value);
                
                        oa::oaString str;
                        condIter->lval->getName(*nameSpace, str);
                        cerr << "NOTE: Implicit latch on net " << currentVmodule->name << "." << str << endl;
                    }
                }
                
                DEBUG_PRINTLN("\t\t\t (mux) <= (func)");
            } else if ((*leftIter)->type == LVAL_FUNCTION) {
                assert(state.isFunction);            
                state.functionAssignments[(*leftIter)->functionLvalName][(*leftIter)->functionLvalBit] = value;
                DEBUG_PRINTLN("\t\t\t " << (*leftIter)->functionLvalName << " <= (func)");
            } else {
                QUIT_ON_INTERNAL_ERROR;
            }
            
            ++leftIter;
        }        
        
        break;
        }

      case VerilogDesign::Statement::NONBLOCKING_ASSIGNMENT: {
      
        LvalRefBus lval;
        MultiRefBus rval;
        evaluateLval(lval, statement->assign.lval, &state, parameters);
        evaluateExpression(rval, statement->assign.rval, &state, parameters);

        LvalRefBus::reverse_iterator leftIter = lval.rbegin();
        MultiRefBus::reverse_iterator rightIter = rval.rbegin();
        while(leftIter != lval.rend()) {
            MultiRef value;
            if (rightIter == rval.rend()) {
                value = constantZero();
            } else {
                value = *rightIter;
                ++rightIter;
            }
            
            if ((*leftIter)->type == VerilogSynthesis::LVAL_UNCONDITIONAL) {
                // unconditional assignment
                
                oa::oaModBitNet *net = (*leftIter)->unconditionalLval;
                state.nonblockingAssignments[net] = value;

                oa::oaString str;
                net->getName(*nameSpace, str);
                if (state.nonblockingAssignments[net].type == NET) {
                    oa::oaString str2;
                    state.nonblockingAssignments[net].net->getName(*nameSpace, str2);
                    DEBUG_PRINTLN("\t\t\t " << str << " <= " << str2);
                } else {
                    DEBUG_PRINTLN("\t\t\t " << str << " <= (func)");
                }
            } else if ((*leftIter)->type == LVAL_CONDITIONAL) {
                // conditional assignment
                
                for(list<ConditionalLvalRef>::iterator condIter = (*leftIter)->conditionalLval.begin();
                    condIter != (*leftIter)->conditionalLval.end();
                    ++condIter) {
                    
                    if (state.nonblockingAssignments.find(condIter->lval) != state.nonblockingAssignments.end()) {
                        // combinational

                        state.nonblockingAssignments[condIter->lval] = mux(condIter->condition, 
                                                                           state.nonblockingAssignments[condIter->lval], 
                                                                           value);
                    } else if (state.isRegister) {
                        // mux previous state
            
                        state.nonblockingAssignments[condIter->lval] = mux(condIter->condition, 
                                                                           MultiRef(condIter->lval), 
                                                                           value); 
                    } else {
                        // implicit latch... transparent when condition
                        state.nonblockingAssignments[condIter->lval] = latch(condIter->condition, value);
                
                        oa::oaString str;
                        condIter->lval->getName(*nameSpace, str);
                        cerr << "NOTE: Implicit latch on net " << currentVmodule->name << "." << str << endl;
                    }
                }
                
                DEBUG_PRINTLN("\t\t\t (mux) <= (func)");
            } else if ((*leftIter)->type == LVAL_FUNCTION) {
                cerr << "ERROR: Non-blocking assignments are not supported inside functions" << endl;
                QUIT_ON_ERROR;
            } else {
                QUIT_ON_INTERNAL_ERROR;
            }
            
            ++leftIter;
        }        
        
        break;
        }
 
      default:
        QUIT_ON_INTERNAL_ERROR;
    }
}


// *****************************************************************************
// evaluateConstantExpression()
//
/// \brief Evaluate an expression (that should be constant).
//
/// An error will be return if the expression is not constant.
//
// *****************************************************************************
void
VerilogSynthesis::evaluateConstantExpression(ConstantValue &result, VerilogDesign::Expression *expression, 
                                             ParameterValues *parameters) {
    assert(expression);

    ConstantValue c1, c2, c3;
    switch(expression->type) {
    
      case VerilogDesign::Expression::PRIMARY:
      
        if (expression->primary->type == VerilogDesign::Primary::CONST) {
            result.intValue = expression->primary->number.intValue;
            result.bitWidth = expression->primary->number.bitWidth;
        } else if (expression->primary->type == VerilogDesign::Primary::NET) {
            ParameterValues::iterator it;
            if ((it = parameters->find(expression->primary->name)) == parameters->end()) {
                cerr << "ERROR: Unknown parameter " << expression->primary->name << endl;
                QUIT_ON_ERROR;
            }
            result = it->second;
        } else if (expression->primary->type == VerilogDesign::Primary::FUNCTION_CALL) {
            cerr << "ERROR: Function calls are not supported in constant expressions" << endl;
            QUIT_ON_ERROR;
        } else {
            QUIT_ON_INTERNAL_ERROR;
        }
      break;
      
      case VerilogDesign::Expression::BUNDLE:

        // bundle may be replicated
        ConstantValue replication;
        
        if (expression->bundle->replication == NULL) {
            replication.intValue = 1;
        } else {
            evaluateConstantExpression(replication, expression->bundle->replication, parameters);
    
            if (replication.intValue != 1) {
                cerr << "ERROR: Bundle replication not supported in constant expressions" << endl;
                QUIT_ON_ERROR;
            }
        }

        result.intValue = 0;
        result.bitWidth = 0;
        for(list<VerilogDesign::Expression*>::iterator exprIter = expression->bundle->members->begin();
            exprIter != expression->bundle->members->end(); 
            ++exprIter) {
            ConstantValue partial;
            evaluateConstantExpression(partial, *exprIter, parameters);
            result.intValue = (result.intValue << partial.bitWidth) + partial.intValue;
            result.bitWidth += partial.bitWidth;
        }
            
        DEBUG_PRINTLN("\t\t bundle [" << result.bitWidth << "] replication= " << replication.intValue);
        break;
        
      case VerilogDesign::Expression::ADD:
        evaluateConstantExpression(c1, expression->op1, parameters);
        evaluateConstantExpression(c2, expression->op2, parameters);
        result.intValue = c1.intValue + c2.intValue;
        result.bitWidth = max(c1.bitWidth, c2.bitWidth)+1;
      break;
      case VerilogDesign::Expression::SUBTRACT:
        evaluateConstantExpression(c1, expression->op1, parameters);
        evaluateConstantExpression(c2, expression->op2, parameters);
        result.intValue = c1.intValue - c2.intValue;
        result.bitWidth = max(c1.bitWidth, c2.bitWidth);
      break;
      case VerilogDesign::Expression::MULTIPLY:
        evaluateConstantExpression(c1, expression->op1, parameters);
        evaluateConstantExpression(c2, expression->op2, parameters);
        result.intValue = c1.intValue * c2.intValue;
        result.bitWidth = c1.bitWidth+c2.bitWidth;
      break;
      case VerilogDesign::Expression::DIVIDE:
        evaluateConstantExpression(c1, expression->op1, parameters);
        evaluateConstantExpression(c2, expression->op2, parameters);
        result.intValue = c1.intValue / c2.intValue;
        result.bitWidth = c1.bitWidth;
      break;
      case VerilogDesign::Expression::MODULO:
        evaluateConstantExpression(c1, expression->op1, parameters);
        evaluateConstantExpression(c2, expression->op2, parameters);
        result.intValue = c1.intValue % c2.intValue;
        result.bitWidth = c2.bitWidth;
      break;
      case VerilogDesign::Expression::BITWISE_AND:
        evaluateConstantExpression(c1, expression->op1, parameters);
        evaluateConstantExpression(c2, expression->op2, parameters);
        result.intValue = c1.intValue & c2.intValue;
        result.bitWidth = max(c1.bitWidth, c2.bitWidth);
      break;
      case VerilogDesign::Expression::BITWISE_NOT:
        evaluateConstantExpression(c1, expression->op1, parameters);
        result.intValue = (~c1.intValue);
        result.bitWidth = c1.bitWidth;
      break;
      case VerilogDesign::Expression::BITWISE_OR:
        evaluateConstantExpression(c1, expression->op1, parameters);
        evaluateConstantExpression(c2, expression->op2, parameters);
        result.intValue = c1.intValue | c2.intValue;
        result.bitWidth = max(c1.bitWidth, c2.bitWidth);
      break;
      case VerilogDesign::Expression::BITWISE_XOR:
        evaluateConstantExpression(c1, expression->op1, parameters);
        evaluateConstantExpression(c2, expression->op2, parameters);
        result.intValue = c1.intValue ^ c2.intValue;
        result.bitWidth = max(c1.bitWidth, c2.bitWidth);
      break;
      case VerilogDesign::Expression::LEFT_SHIFT:
        evaluateConstantExpression(c1, expression->op1, parameters);
        evaluateConstantExpression(c2, expression->op2, parameters);
        result.intValue = c1.intValue << c2.intValue;
        result.bitWidth = c1.bitWidth + c2.intValue;
      break;
      case VerilogDesign::Expression::RIGHT_SHIFT:
        evaluateConstantExpression(c1, expression->op1, parameters);
        evaluateConstantExpression(c2, expression->op2, parameters);
        result.intValue = c1.intValue >> c2.intValue;
        result.bitWidth = c1.bitWidth;
      break;
      case VerilogDesign::Expression::GREATER_THAN:
        evaluateConstantExpression(c1, expression->op1, parameters);
        evaluateConstantExpression(c2, expression->op2, parameters);
        result.intValue = (c1.intValue > c2.intValue ? 1 : 0);
        result.bitWidth = 1;
      break;
      case VerilogDesign::Expression::LESS_THAN:
        evaluateConstantExpression(c1, expression->op1, parameters);
        evaluateConstantExpression(c2, expression->op2, parameters);
        result.intValue = (c1.intValue < c2.intValue ? 1 : 0);
        result.bitWidth = 1;
      break;
      case VerilogDesign::Expression::GREATER_THAN_EQUAL:
        evaluateConstantExpression(c1, expression->op1, parameters);
        evaluateConstantExpression(c2, expression->op2, parameters);
        result.intValue = (c1.intValue >= c2.intValue ? 1 : 0);
        result.bitWidth = 1;
      break;
      case VerilogDesign::Expression::LESS_THAN_EQUAL:
        evaluateConstantExpression(c1, expression->op1, parameters);
        evaluateConstantExpression(c2, expression->op2, parameters);
        result.intValue = (c1.intValue <= c2.intValue ? 1 : 0);
        result.bitWidth = 1;
      break;
      case VerilogDesign::Expression::EQUAL:
        evaluateConstantExpression(c1, expression->op1, parameters);
        evaluateConstantExpression(c2, expression->op2, parameters);
        result.intValue = (c1.intValue == c2.intValue ? 1 : 0);
        result.bitWidth = 1;
      break;
      case VerilogDesign::Expression::NOTEQUAL:
        evaluateConstantExpression(c1, expression->op1, parameters);
        evaluateConstantExpression(c2, expression->op2, parameters);
        result.intValue = (c1.intValue != c2.intValue ? 1 : 0);
        result.bitWidth = 1;
      break;
      case VerilogDesign::Expression::LOGICAL_AND:
        evaluateConstantExpression(c1, expression->op1, parameters);
        evaluateConstantExpression(c2, expression->op2, parameters);
        result.intValue = (c1.intValue != 0 && c2.intValue != 0 ? 1 : 0);
        result.bitWidth = 1;
      break;
      case VerilogDesign::Expression::LOGICAL_OR:
        evaluateConstantExpression(c1, expression->op1, parameters);
        evaluateConstantExpression(c2, expression->op2, parameters);
        result.intValue = (c1.intValue != 0 || c2.intValue != 0 ? 1 : 0);
        result.bitWidth = 1;
      break;
      case VerilogDesign::Expression::LOGICAL_NOT:
        evaluateConstantExpression(c1, expression->op1, parameters);
        evaluateConstantExpression(c2, expression->op2, parameters);
        result.intValue = (c1.intValue == 0 ? 1 : 0);
        result.bitWidth = 1;
      break;
      case VerilogDesign::Expression::IF_ELSE:
        evaluateConstantExpression(c1, expression->op1, parameters);
        evaluateConstantExpression(c2, expression->op2, parameters);
        evaluateConstantExpression(c3, expression->op3, parameters);
        if (c1.intValue != 0) {
            result.intValue = c2.intValue;
            result.bitWidth = c2.bitWidth;
        } else {
            result.intValue = c3.intValue;
            result.bitWidth = c3.bitWidth;
        }
      break;
      default:
        cerr << "ERROR: Invalid operator in constant expression: " << expression->type << endl;
        break;
    }

}


// *****************************************************************************
// isConstantExpression()
//
/// \brief Tests if an expression is a constant or not
///
/// This returns true only if all leaves of the expression are constants.  Any
/// expressions that involve nets or registers, regardless of whether the nets
/// or regs are themselves constant will return false.  Also, any expressions
/// that involve redundancy and are functionally constant (i.e. x & !x) will
/// not be identified and also return false.
//
// *****************************************************************************
bool            
VerilogSynthesis::isConstantExpression(VerilogDesign::Expression *expression, 
                                       ParameterValues *parameters) {
    assert(expression);
    switch(expression->type) {
    
      case VerilogDesign::Expression::PRIMARY:
        if (expression->primary->type == VerilogDesign::Primary::CONST) {
            return true;
        } if (expression->primary->type == VerilogDesign::Primary::NET) {
            ParameterValues::iterator it;
            if ((it = parameters->find(expression->primary->name)) == parameters->end()) {
                return false;
            }
            return true;
        } else {
            return false;
        }
      case VerilogDesign::Expression::BUNDLE:
        for(list<VerilogDesign::Expression*>::iterator exprIter = expression->bundle->members->begin();
            exprIter != expression->bundle->members->end(); 
            ++exprIter) {
            if (!isConstantExpression(*exprIter, parameters)) {
                return false;
            }
        }
        return true;
      case VerilogDesign::Expression::BITWISE_NOT:
      case VerilogDesign::Expression::LOGICAL_NOT:
        return isConstantExpression(expression->op1, parameters);
      case VerilogDesign::Expression::ADD:
      case VerilogDesign::Expression::SUBTRACT:
      case VerilogDesign::Expression::MULTIPLY:
      case VerilogDesign::Expression::DIVIDE:
      case VerilogDesign::Expression::MODULO:
      case VerilogDesign::Expression::BITWISE_AND:
      case VerilogDesign::Expression::BITWISE_OR:
      case VerilogDesign::Expression::BITWISE_XOR:
      case VerilogDesign::Expression::LEFT_SHIFT:
      case VerilogDesign::Expression::RIGHT_SHIFT:
      case VerilogDesign::Expression::LESS_THAN:
      case VerilogDesign::Expression::GREATER_THAN:
      case VerilogDesign::Expression::LESS_THAN_EQUAL:
      case VerilogDesign::Expression::GREATER_THAN_EQUAL:
      case VerilogDesign::Expression::LOGICAL_AND:
      case VerilogDesign::Expression::LOGICAL_OR:
      case VerilogDesign::Expression::EQUAL:
      case VerilogDesign::Expression::NOTEQUAL:
        return isConstantExpression(expression->op1, parameters) && 
               isConstantExpression(expression->op2, parameters);
      case VerilogDesign::Expression::IF_ELSE:
        return isConstantExpression(expression->op1, parameters) && 
               isConstantExpression(expression->op2, parameters) && 
               isConstantExpression(expression->op3, parameters);
      default:
        return false;
    }

}


// *****************************************************************************
// evaluateLval()
//
/// \brief Evaluate an lval
//
// *****************************************************************************
void            
VerilogSynthesis::evaluateLval(LvalRefBus &result, VerilogDesign::Expression *expression, 
                               ProceduralState *state, ParameterValues *parameters) {    

    assert(expression);
    result.clear();

    MultiRefBus op1, op3;
    switch(expression->type) {
    
      case VerilogDesign::Expression::PRIMARY: {
        
        // is this a numerical constant?
        if (expression->primary->type == VerilogDesign::Primary::CONST) {
                cerr << "ERROR: Lval can not be a constant" << endl;
                QUIT_ON_ERROR;

        } else if (expression->primary->type == VerilogDesign::Primary::NET) {
        
            // is this a parameter?
            ParameterValues::iterator it;
            if ((it = parameters->find(expression->primary->name)) != parameters->end()) {
                cerr << "ERROR: Lval can not be a parameter" << endl;
                QUIT_ON_ERROR;
            }

            // is this in the context of a function?
            else if (state && state->isFunction) {
                map<string, FunctionVariableAssignment>::iterator it = 
                  state->functionAssignments.find(expression->primary->name);
                if (it == state->functionAssignments.end()) {
                    cerr << "ERROR: Unknown net: " << expression->primary->name << endl;
                    QUIT_ON_ERROR;
                }
                
                FunctionVariableAssignment bitAssignments = it->second;
                if (expression->primary->range.start == NULL) {
                    for(int b=bitAssignments.start; 
                        (bitAssignments.start<bitAssignments.stop ? b<=bitAssignments.stop : b>=bitAssignments.stop); 
                        b+=(bitAssignments.start<bitAssignments.stop?1:-1)) {
                        result.push_back(new LvalRef(expression->primary->name, b));
                    }
                } else {
                    
                    if (!isConstantExpression(expression->primary->range.start, parameters) 
                        || !isConstantExpression(expression->primary->range.stop, parameters)) {
                        cerr << "ERROR: Range expressions must be constant" << endl;
                        QUIT_ON_ERROR;
                    }
                    
                    ConstantValue start, stop;
                    evaluateConstantExpression(start, expression->primary->range.start, parameters);
                    evaluateConstantExpression(stop, expression->primary->range.stop, parameters);
                    for(int b=start.intValue; 
                        (start.intValue<stop.intValue?b<=stop.intValue:b>=stop.intValue); 
                        b+=(start.intValue<stop.intValue?1:-1)) {
                        if (b > bitAssignments.start && b > bitAssignments.stop ||
                            b < bitAssignments.start && b < bitAssignments.stop) {
                            cerr << "ERROR: Bus index " << expression->primary->name << "[" 
                                 << b << "] is out of range.  Range is [" << bitAssignments.start 
                                 << ":" << bitAssignments.stop << "]" << endl;
                            QUIT_ON_ERROR;
                        }
                        result.push_back(new LvalRef(expression->primary->name, b));
                    }
                }
                DEBUG_PRINTLN("\t\t\t\t lval " << expression->primary->name);
                
                break;
            }
        
            // is this a 2D net?
            else if (twoDimRegisters.find(expression->primary->name) != twoDimRegisters.end()) {
                if (expression->primary->range.start == NULL || 
                    expression->primary->range.start != expression->primary->range.stop) {
                    cerr << "ERROR: Net " << expression->primary->name 
                         << " is two-dimensional.  Index required" << endl;
                    QUIT_ON_ERROR;
                }
                Bounds bounds(twoDimRegisters[expression->primary->name]);
                if (isConstantExpression(expression->primary->range.start, parameters)) {
                    // attach wires to appropriate net

                    // construct suffix to identify net in 2-d vector 
                    ConstantValue c;
                    evaluateConstantExpression(c, expression->primary->range.start, parameters);
                    if (c.intValue < bounds.lower || c.intValue > bounds.upper) {
                        cerr << "ERROR: Net vector index out of bounds" << endl;
                        QUIT_ON_ERROR;
                    }
                    string vectorName(expression->primary->name);
                    stringstream suffix;
                    suffix << "_" << c.intValue << "_";
                    appendSuffix(vectorName, suffix.str());

                    oa::oaModScalarNet *scalarNet;
                    oa::oaModBusNet *busNet;
                    if ((scalarNet = findScalarNet(vectorName))) {
                        result.push_back(new LvalRef(scalarNet));
                    } else if ((busNet = findBusNet(vectorName))) {
                        int start = busNet->getStart();
                        int stop = busNet->getStop();
                        for(int b=start; (start<stop?b<=stop:b>=stop); b+=(start<stop?1:-1)) {
                            oa::oaModBusNetBit *bitNet = findBusNetBit(vectorName, b);
                            assert(bitNet);
                            result.push_back(new LvalRef(bitNet));
                        }
                    } else {
                        QUIT_ON_INTERNAL_ERROR;
                    }
                    
                    DEBUG_PRINTLN("\t\t\t\t lval <" << c.intValue << ">: " << expression->primary->name);        
                } else {
                    // build mux

                    evaluateExpression(op1, expression->primary->range.start, state, parameters); 
                    int low = bounds.lower;
                    int high = min(bounds.upper, 1<<op1.size());
                    bool firstConditional = true;
                    for(int i=low; i<=high; i++) {
                        MultiRefBus constantBits;
                        multiBitConstant(constantBits, i);
                        zeroExpand(constantBits, op1);
                        MultiRef select(equalTo(constantBits, op1));
                        
                        // construct suffix to identify net in 2-d vector 
                        string vectorName(expression->primary->name);
                        stringstream suffix;
                        suffix << "_" << i << "_";
                        appendSuffix(vectorName, suffix.str());
                        
                        oa::oaModScalarNet *scalarNet;
                        oa::oaModBusNet *busNet;
                        if ((scalarNet = findScalarNet(vectorName))) {
                            if (firstConditional) {
                                result.push_back(new LvalRef);
                            }
                            result.front()->addConditional(scalarNet, select);                     
                        } else if ((busNet = findBusNet(vectorName))) {
                            int start = busNet->getStart(), stop = busNet->getStop();
                            if (firstConditional) {
                                for(int b=start; (start<stop?b<=stop:b>=stop); b+=(start<stop?1:-1)) {
                                    result.push_back(new LvalRef);
                                }
                            }                            
                            LvalRefBus::iterator bitIter = result.begin();
                            for(int b=0; b<=abs(start-stop) && bitIter != result.end(); ++b, ++bitIter) {
                                oa::oaModBitNet *bitNet = busNet->getBit(b);
                                assert(bitNet);
                                (*bitIter)->addConditional(bitNet, select);
                            }
                        } else {
                            QUIT_ON_INTERNAL_ERROR;
                        }
                        firstConditional = false;
                    }
            
                    DEBUG_PRINTLN("\t\t\t\t lval <mux>: " << expression->primary->name);
                }
                
            // is this a net?
            } else if (expression->primary->range.start == NULL) {
                oa::oaModScalarNet *scalarNet;
                oa::oaModBusNet *busNet;
                if ((scalarNet = findScalarNet(expression->primary->name))) {
                    result.push_back(new LvalRef(scalarNet));
                } else if ((busNet = findBusNet(expression->primary->name))) {
                    int start = busNet->getStart();
                    int stop = busNet->getStop();
                    for(int b=start; (start<stop?b<=stop:b>=stop); b+=(start<stop?1:-1)) {
                        oa::oaModBusNetBit *bitNet = findBusNetBit(expression->primary->name, b);
                        assert(bitNet);
                        result.push_back(new LvalRef(bitNet));
                    }
                } else {
                    if (IMPLICIT_NET_WARNINGS) {
                        cerr << "NOTE: Implicit net " << expression->primary->name << ".  Assuming scalar wire" << endl;
                    }
                    result.push_back(new LvalRef(createScalarNet(expression->primary->name)));
                }
                DEBUG_PRINTLN("\t\t\t\t lval: " << expression->primary->name);
            } else if (expression->primary->range.start == expression->primary->range.stop && 
                       !isConstantExpression(expression->primary->range.start, parameters)) {
                // bit index is not known at compile time.  build a multiplexor
                oa::oaModBusNet *busNet = findBusNet(expression->primary->name);
                if (busNet == NULL) {
                    cerr << "ERROR: Unknown net " << expression->primary->name << ".  Busses can not be implicit" << endl;
                    QUIT_ON_ERROR;
                }
                evaluateExpression(op1, expression->primary->range.start, state, parameters);
                
                result.push_back(new LvalRef);
                int start = busNet->getStart(), stop = busNet->getStop();
                for(int i=0; i<=abs(stop-start); i++) {
                    // only need to mux bits that are indexable by select
                    if (i*sign(stop-start)+start < (1 << op1.size())) {
                        MultiRefBus constantBits;
                        multiBitConstant(constantBits, i*sign(stop-start)+start);
                        zeroExpand(constantBits, op1);
                        oa::oaModBitNet *bitNet = busNet->getBit(i);
                        assert(bitNet);
                        result.back()->addConditional(bitNet, equalTo(constantBits, op1));
                    }
                } 
                DEBUG_PRINTLN("\t\t\t\t lval [mux]: " << expression->primary->name);
            } else {
            
                if (!isConstantExpression(expression->primary->range.start, parameters) || 
                    !isConstantExpression(expression->primary->range.stop, parameters)) {
                    cerr << "ERROR: Range expressions must be constant" << endl;
                    QUIT_ON_ERROR;
                }
                    
                ConstantValue start, stop;
                evaluateConstantExpression(start, expression->primary->range.start, parameters);
                evaluateConstantExpression(stop, expression->primary->range.stop, parameters);
                for(int b=start.intValue; 
                    (start.intValue<stop.intValue ? b<=stop.intValue : b>=stop.intValue); 
                    b+=(start.intValue<stop.intValue?1:-1)) {
                    oa::oaModBusNetBit *bitNet = findBusNetBit(expression->primary->name, b);
                    if (!bitNet) {
                        cerr << "ERROR: Bus index " << expression->primary->name << "[" << b 
                             << "] does not exist or is out of range" << endl;
                        QUIT_ON_ERROR;
                    }
                
                    result.push_back(new LvalRef(bitNet));
                }
                DEBUG_PRINTLN("\t\t\t\t lval [" << start.intValue << ":" 
                              << stop.intValue << "]: " << expression->primary->name);
            }

        } else if (expression->primary->type == VerilogDesign::Primary::FUNCTION_CALL) {
            cerr << "ERROR: Lval can not be a function call" << endl;
            QUIT_ON_ERROR;
        } else {
            QUIT_ON_INTERNAL_ERROR;
        }

        break;
      }
      
      case VerilogDesign::Expression::BUNDLE:

        // bundle may be replicated
        ConstantValue replication;
        if (expression->bundle->replication == NULL) {
            replication.intValue = 1;
        } else {
            evaluateConstantExpression(replication, expression->bundle->replication, parameters);
        }

        for(int i=0; i<replication.intValue; i++)
            for(list<VerilogDesign::Expression*>::iterator exprIter = expression->bundle->members->begin();
                exprIter != expression->bundle->members->end(); 
                ++exprIter) {
                LvalRefBus partial;
                evaluateLval(partial, *exprIter, state, parameters);
                result.splice(result.end(), partial);
            }
            
        DEBUG_PRINTLN("\t\t\t\t lval bundle [" << result.size() << "] replication= " << replication.intValue);
        break;

      default:

        QUIT_ON_INTERNAL_ERROR;
      }
}

// *****************************************************************************
// evaluateExpression()
//
/// \brief Evaluate an expression
//
// *****************************************************************************
void            
VerilogSynthesis::evaluateExpression(MultiRefBus &result, VerilogDesign::Expression *expression, 
                                     ProceduralState *state, ParameterValues *parameters) {    
    assert(expression);
    assert(parameters);

    result.clear();
    
    MultiRefBus op1, op2, op3;
    switch(expression->type) {
    
      case VerilogDesign::Expression::PRIMARY: {
        assert(expression->primary);

        // is this a numerical constant?
        if (expression->primary->type == VerilogDesign::Primary::CONST) {
            if (expression->primary->number.negative) {
                cerr << "ERROR: Negative numbers are currently unimplemented" << endl;
                QUIT_ON_ERROR;
            }
            multiBitConstant(result, expression->primary->number.intValue, expression->primary->number.bitWidth);
            DEBUG_PRINTLN("\t\t\t\t const [" << result.size() << "]");
            break;

        } else if (expression->primary->type == VerilogDesign::Primary::NET) {
        
            // is this a parameter?
            ParameterValues::iterator it;
            if ((it = parameters->find(expression->primary->name)) != parameters->end()) {
                multiBitConstant(result, it->second.intValue, it->second.bitWidth);
                DEBUG_PRINTLN("\t\t\t\t param [" << result.size() << "]");
                break;
            }

            // is this in the context of a function?
            else if (state && state->isFunction) {
                map<string, FunctionVariableAssignment>::iterator it = 
                    state->functionAssignments.find(expression->primary->name);
                if (it == state->functionAssignments.end()) {
                    cerr << "ERROR: Unknown net: " << expression->primary->name << endl;
                    QUIT_ON_ERROR;
                }
                
                FunctionVariableAssignment bitAssignments = it->second;
                if (expression->primary->range.start == NULL) {
                    for(int b=bitAssignments.start; 
                        bitAssignments.start<bitAssignments.stop?b<=bitAssignments.stop:b>=bitAssignments.stop; 
                        b+=(bitAssignments.start<bitAssignments.stop?1:-1)) {
                        result.push_back(bitAssignments[b]);
                    }
                } else {
                    
                    if (!isConstantExpression(expression->primary->range.start, parameters) || 
                        !isConstantExpression(expression->primary->range.stop, parameters)) {
                        cerr << "ERROR: Range expressions must be constant" << endl;
                        QUIT_ON_ERROR;
                    }
                    
                    ConstantValue start, stop;
                    evaluateConstantExpression(start, expression->primary->range.start, parameters);
                    evaluateConstantExpression(stop, expression->primary->range.stop, parameters);
                    for(int b=start.intValue; (start.intValue<stop.intValue?b<=stop.intValue:b>=stop.intValue); 
                        b+=(start.intValue<stop.intValue?1:-1)) {
                        if (b > bitAssignments.start && b > bitAssignments.stop || 
                            b < bitAssignments.start && b < bitAssignments.stop) {
                            cerr << "ERROR: Bus index " << expression->primary->name 
                                 << "[" << b << "] is out of range.  Range is [" 
                                 << bitAssignments.start << ":" << bitAssignments.stop << "]" 
                                 << endl;
                            QUIT_ON_ERROR;
                        }
                        result.push_back(bitAssignments[b]);
                    }
                }
                DEBUG_PRINTLN("\t\t\t\t funcval " << expression->primary->name);
                
                break;
            }
            
            // is this a 2D net?
            else if (twoDimRegisters.find(expression->primary->name) != twoDimRegisters.end()) {
                if (expression->primary->range.start == NULL || 
                    expression->primary->range.start != expression->primary->range.stop) {
                    cerr << "ERROR: Net " << expression->primary->name << " is two-dimensional.  Index required" << endl;
                    QUIT_ON_ERROR;
                }
                Bounds bounds(twoDimRegisters[expression->primary->name]);
                if (isConstantExpression(expression->primary->range.start, parameters)) {
                    // attach wires to appropriate net

                    // construct suffix to identify net in 2-d vector 
                    ConstantValue c;
                    evaluateConstantExpression(c, expression->primary->range.start, parameters);
                    if (c.intValue < bounds.lower || c.intValue > bounds.upper) {
                        cerr << "ERROR: Net vector index out of bounds" << endl;
                        QUIT_ON_ERROR;
                    }
                    string vectorName(expression->primary->name);
                    stringstream suffix;
                    suffix << "_" << c.intValue << "_";
                    appendSuffix(vectorName, suffix.str());

                    oa::oaModScalarNet *scalarNet;
                    oa::oaModBusNet *busNet;
                    if ((scalarNet = findScalarNet(vectorName))) {
                        result.push_back(getContextualValue(scalarNet, state));
                    } else if ((busNet = findBusNet(vectorName))) {
                        int start = busNet->getStart();
                        int stop = busNet->getStop();
                        for(int b=start; (start<stop?b<=stop:b>=stop); b+=(start<stop?1:-1)) {
                            oa::oaModBusNetBit *bitNet = findBusNetBit(vectorName, b);
                            assert(bitNet);
                            result.push_back(getContextualValue(bitNet, state));
                        }
                    } else {
                        QUIT_ON_INTERNAL_ERROR;
                    }
                    
                    DEBUG_PRINTLN("\t\t\t\t net <" << c.intValue << ">: " << expression->primary->name);

                } else {
                    // build mux

                    evaluateExpression(op1, expression->primary->range.start, state, parameters);
                    int low = bounds.lower;
                    int high = min(bounds.upper, 1<<op1.size());
                    MultiRefBus *muxTerms = new MultiRefBus[high-low+1];
                    for(int i=low; i<=high; i++) {
                        MultiRefBus constantBits;
                        multiBitConstant(constantBits, i);
                        zeroExpand(constantBits, op1);
                        MultiRef select(equalTo(constantBits, op1));
                        
                        // construct suffix to identify net in 2-d vector 
                        string vectorName(expression->primary->name);
                        stringstream suffix;
                        suffix << "_" << i << "_";
                        appendSuffix(vectorName, suffix.str());
                        
                        oa::oaModScalarNet *scalarNet;
                        oa::oaModBusNet *busNet;
                        if ((scalarNet = findScalarNet(vectorName))) {
                            muxTerms[i-low].push_back(andOf(getContextualValue(scalarNet, state), select));                     
                        } else if ((busNet = findBusNet(vectorName))) {
                            int start = busNet->getStart(), stop = busNet->getStop();
                            for(int b=0; b<=abs(stop-start); b++) {
                                oa::oaModBitNet *bitNet = busNet->getBit(b);
                                assert(bitNet);
                                muxTerms[i-low].push_back(andOf(getContextualValue(bitNet, state), select));
                            }
                        } else {
                            QUIT_ON_INTERNAL_ERROR;
                        }
                    }

                    while(!muxTerms[0].empty()) {
                        MultiRefBus terms;
                        for(int i=low; i<=high; i++) {
                            terms.push_back(muxTerms[i-low].front());
                            muxTerms[i-low].pop_front();
                        }
                        result.push_back(reductionOr(terms));
                    }
            
                    DEBUG_PRINTLN("\t\t\t\t net <mux>: " << expression->primary->name);
                    delete [] muxTerms;
                }
                
            // is this a 1-D net?
            } else if (expression->primary->range.start == NULL) {
                oa::oaModScalarNet *scalarNet;
                oa::oaModBusNet *busNet;
                if ((scalarNet = findScalarNet(expression->primary->name))) {
                    result.push_back(getContextualValue(scalarNet, state));
                } else if ((busNet = findBusNet(expression->primary->name))) {
                    int start = busNet->getStart();
                    int stop = busNet->getStop();
                    for(int b=start; (start<stop?b<=stop:b>=stop); b+=(start<stop?1:-1)) {
                        oa::oaModBusNetBit *bitNet = findBusNetBit(expression->primary->name, b);
                        assert(bitNet);
                        result.push_back(getContextualValue(bitNet, state));
                    }
                } else {
                    if (IMPLICIT_NET_WARNINGS) {
                        cerr << "NOTE: Implicit net " << expression->primary->name 
                             << ".  Assuming scalar wire" << endl;
                    }
                    result.push_back(MultiRef(createScalarNet(expression->primary->name)));
                }
                DEBUG_PRINTLN("\t\t\t\t net: " << expression->primary->name);
            } else if (expression->primary->range.start == expression->primary->range.stop && 
                       !isConstantExpression(expression->primary->range.start, parameters)) {
                // bit index is not known at compile time.  build a multiplexor
                oa::oaModBusNet *busNet = findBusNet(expression->primary->name);
                if (busNet == NULL) {
                    cerr << "ERROR: Unknown net " << expression->primary->name 
                         << ".  Busses can not be implicit" << endl;
                    QUIT_ON_ERROR;
                }
                evaluateExpression(op1, expression->primary->range.start, state, parameters);
                
                MultiRefBus muxTerms;
                int start = busNet->getStart(), stop = busNet->getStop();
                for(int i=0; i<=abs(stop-start); i++) {
                    // only need to mux bits that are indexable by select
                    if (i*sign(stop-start)+start < (1 << op1.size())) {
                        MultiRefBus constantBits;
                        multiBitConstant(constantBits, i*sign(stop-start)+start);
                        zeroExpand(constantBits, op1);
                        oa::oaModBitNet *bitNet = busNet->getBit(i);
                        assert(bitNet);
                        muxTerms.push_back(andOf(MultiRef(bitNet), equalTo(constantBits, op1)));
                    } 
                }
                result.push_back(reductionOr(muxTerms));
                DEBUG_PRINTLN("\t\t\t\t net [mux]: " << expression->primary->name);
            } else {
            
                if (!isConstantExpression(expression->primary->range.start, parameters) || 
                    !isConstantExpression(expression->primary->range.stop, parameters)) {
                    cerr << "ERROR: Range expressions must be constant" << endl;
                    QUIT_ON_ERROR;
                }
            
                ConstantValue start, stop;
                evaluateConstantExpression(start, expression->primary->range.start, parameters);
                evaluateConstantExpression(stop, expression->primary->range.stop, parameters);
                for(int b=start.intValue; (start.intValue<stop.intValue?b<=stop.intValue:b>=stop.intValue); 
                    b+=(start.intValue<stop.intValue?1:-1)) {
                    oa::oaModBusNetBit *bitNet = findBusNetBit(expression->primary->name, b);
                    if (!bitNet) {
                        cerr << "ERROR: Bus net bit " << expression->primary->name << "[" << b 
                             << "] does not exist or is out of range" << endl;
                    }
                
                    result.push_back(MultiRef(bitNet));
                }
                DEBUG_PRINTLN("\t\t\t\t net [" << start.intValue << ":" << stop.intValue 
                              << "]: " << expression->primary->name);
            }

        } else if (expression->primary->type == VerilogDesign::Primary::FUNCTION_CALL) {
            assert(currentVmodule);

            // find the function defintion of this name
            VerilogDesign::Function *definition = NULL;
            VerilogDesign::Primary *invocation = expression->primary;
            for(list<VerilogDesign::Function*>::iterator fIter = currentVmodule->functions->begin();
                fIter != currentVmodule->functions->end();
                ++fIter) {
                if (invocation->name == (*fIter)->name) {
                    definition = *fIter;
                    break;
                }
            }
            if (definition == NULL) {
                cerr << "ERROR: Unknown function: " << invocation->name << endl;
                QUIT_ON_ERROR;            
            }

            DEBUG_PRINTLN("\t\t\t\t function call " << invocation->name << "()");

            // create a ProceduralState.  Because of the independent scope of the
            // function call, no procedural states are inhereted from the current one
            ProceduralState functionState;
            ParameterValues functionParameters(*parameters);
            functionState.isFunction = true;
            
            // declare the function variable for the output
            ConstantValue start, stop;
            if (definition->start) {
                // bus variable
                evaluateConstantExpression(start, definition->start, parameters);
                evaluateConstantExpression(stop, definition->stop, parameters);
                FunctionVariableAssignment bitAssignments(start.intValue, stop.intValue);
                for(int b=start.intValue; (start.intValue<stop.intValue ? b<=stop.intValue : b>=stop.intValue); 
                    b+=(start.intValue<stop.intValue?1:-1)) {
                    bitAssignments[b] = constantZero();
                }
                functionState.functionAssignments[definition->name] = bitAssignments;
            } else {
                // scalar variable
                FunctionVariableAssignment bitAssignments;
                bitAssignments[0] = constantZero();
                functionState.functionAssignments[definition->name] = bitAssignments;
            }

            // declare the function variable for the other declarations
            // and assign the state of the inputs to be that of the arguments passed
            list<VerilogDesign::Expression *>::iterator argIter = invocation->arguments->begin();
            list<VerilogDesign::Declaration *>::iterator declIter = definition->declarations->begin();
            while (declIter != definition->declarations->end()) {
                VerilogDesign::Declaration *declaration = *declIter;
                
                // an input declaration
                if (declaration->type == VerilogDesign::Declaration::INPUT) {
                    if (declaration->start2D!=NULL) {
                        cerr << "ERROR: Two dimensional nets are not supported inside functions" << endl;
                        QUIT_ON_ERROR;
                    }
                    
                    // evaluate the matching argument
                    if (argIter == invocation->arguments->end()) {
                        cerr << "ERROR: Too few arguments to function call" << endl;
                        QUIT_ON_ERROR;
                    }
                    MultiRefBus argResult;
                    evaluateExpression(argResult, *argIter, state, parameters);
                    ++argIter;

                    DEBUG_PRINTLN("\t\t\t\t\t input " << declaration->name);
                    if (!declaration->start) {
                        // scalar variable
                        FunctionVariableAssignment bitAssignments;
                        bitAssignments[0] = argResult.back();
                        functionState.functionAssignments[declaration->name] = bitAssignments;
                    } else {
                        // bus variable
                        ConstantValue start, stop;
                        evaluateConstantExpression(start, declaration->start, parameters);
                        evaluateConstantExpression(stop, declaration->stop, parameters);
                        FunctionVariableAssignment bitAssignments(start.intValue, stop.intValue);
                        MultiRefBus::reverse_iterator argResultIter = argResult.rbegin();
                        for(int b=stop.intValue; (start.intValue<stop.intValue?b>=start.intValue:b<=start.intValue); 
                            b+=(start.intValue<stop.intValue?-1:1)) {
                            if (argResultIter == argResult.rend()) {
                                bitAssignments[b] = constantZero();
                            } else {
                                bitAssignments[b] = *argResultIter;
                            }
                            ++argResultIter;
                        }
                        functionState.functionAssignments[declaration->name] = bitAssignments;
                    }

                // a reg declaration                    
                } else if (declaration->type == VerilogDesign::Declaration::REG) {
                    if (declaration->start2D!=NULL) {
                        cerr << "ERROR: Two dimensional nets are not supported inside functions" << endl;
                        QUIT_ON_ERROR;
                    }
                    
                    DEBUG_PRINTLN("\t\t\t\t\t reg " << declaration->name);
                    if (!declaration->start) {
                        // scalar variable
                        FunctionVariableAssignment bitAssignments;
                        bitAssignments[0] = constantZero();
                        functionState.functionAssignments[declaration->name] = bitAssignments;
                    } else {
                        // bus variable
                        ConstantValue start, stop;
                        evaluateConstantExpression(start, declaration->start, parameters);
                        evaluateConstantExpression(stop, declaration->stop, parameters);
                        FunctionVariableAssignment bitAssignments(start.intValue, stop.intValue);
                        for(int b=start.intValue; 
                            (start.intValue<stop.intValue?b<=stop.intValue:b>=stop.intValue); 
                            b+=(start.intValue<stop.intValue?1:-1)) {
                            bitAssignments[b] = constantZero();
                        }
                        functionState.functionAssignments[declaration->name] = bitAssignments;
                    }
                    
                // UNIMPLEMENTED: allow parameters inside functions
                } else {
                    cerr << "ERROR: Invalid declaration for function.  Only input and reg declarations are allowed" << endl;
                    QUIT_ON_ERROR;
                }
                ++declIter;            
            }

            // left over arguments?
            if (argIter != invocation->arguments->end()) {
                cerr << "ERROR: Too many arguments to function call" << endl;
                QUIT_ON_ERROR;
            }

            // evaluate function
            synthesizeBehavioral(definition->action, functionState, &functionParameters);
            
            // return the final state of the bits w/ same name as function
            FunctionVariableAssignment resultAssignment(functionState.functionAssignments[expression->primary->name]);
            for(int b=resultAssignment.start; 
                (resultAssignment.start<resultAssignment.stop ? b<=resultAssignment.stop : b>=resultAssignment.stop); 
                b+=(resultAssignment.start<resultAssignment.stop?1:-1)) {
                result.push_back(resultAssignment[b]);
            }
        } else {
            QUIT_ON_INTERNAL_ERROR;
        }

        break;
      }
      
      case VerilogDesign::Expression::BUNDLE:

        // bundle may be replicated
        ConstantValue replication;
        if (expression->bundle->replication == NULL) {
            replication.intValue = 1;
        } else {
            evaluateConstantExpression(replication, expression->bundle->replication, parameters);
        }

        for(int i=0; i<replication.intValue; i++)
            for(list<VerilogDesign::Expression*>::iterator exprIter = expression->bundle->members->begin();
                exprIter != expression->bundle->members->end(); 
                ++exprIter) {
                MultiRefBus partial;
                evaluateExpression(partial, *exprIter, state, parameters);
                result.splice(result.end(), partial);
            }
            
        DEBUG_PRINTLN("\t\t bundle [" << result.size() << "] replication= " << replication.intValue);
        break;
        
      case VerilogDesign::Expression::BITWISE_AND: {

        evaluateExpression(op1, expression->op1, state, parameters);
        evaluateExpression(op2, expression->op2, state, parameters);
        zeroExpand(op1, op2);
        assert(op1.size() == op2.size());
        MultiRefBus::iterator op1Iter = op1.begin(), op2Iter = op2.begin();
        while(op1Iter != op1.end() && op2Iter != op2.end()) {
            result.push_back( andOf(*op1Iter, *op2Iter) );
            ++op1Iter; ++op2Iter;
        }
        
        break;
        }
        
      case VerilogDesign::Expression::BITWISE_NOT: {

        evaluateExpression(op1, expression->op1, state, parameters);
        MultiRefBus::iterator op1Iter = op1.begin();
        while(op1Iter != op1.end()) {
            result.push_back( notOf(*op1Iter) );
            ++op1Iter;
        }
        
        break;
        }

      case VerilogDesign::Expression::BITWISE_OR: {

        evaluateExpression(op1, expression->op1, state, parameters);
        evaluateExpression(op2, expression->op2, state, parameters);
        zeroExpand(op1, op2);
        assert(op1.size() == op2.size());
        MultiRefBus::iterator op1Iter = op1.begin(), op2Iter = op2.begin();
        while(op1Iter != op1.end() && op2Iter != op2.end()) {
            result.push_back( orOf(*op1Iter, *op2Iter) );
            ++op1Iter; ++op2Iter;
        }
        
        break;
        }

      case VerilogDesign::Expression::BITWISE_XOR:{

        evaluateExpression(op1, expression->op1, state, parameters);
        evaluateExpression(op2, expression->op2, state, parameters);
        zeroExpand(op1, op2);
        assert(op1.size() == op2.size());
        MultiRefBus::iterator op1Iter = op1.begin(), op2Iter = op2.begin();
        while(op1Iter != op1.end() && op2Iter != op2.end()) {
            result.push_back( xorOf(*op1Iter, *op2Iter) );
            ++op1Iter; ++op2Iter;
        }
        
        break;
        }

      case VerilogDesign::Expression::BITWISE_NOR: {

        evaluateExpression(op1, expression->op1, state, parameters);
        evaluateExpression(op2, expression->op2, state, parameters);
        zeroExpand(op1, op2);
        assert(op1.size() == op2.size());
        MultiRefBus::iterator op1Iter = op1.begin(), op2Iter = op2.begin();
        while(op1Iter != op1.end() && op2Iter != op2.end()) {
            result.push_back( notOf(orOf(*op1Iter, *op2Iter)) );
            ++op1Iter; ++op2Iter;
        }
        
        break;
        }

      case VerilogDesign::Expression::BITWISE_NAND: {

        evaluateExpression(op1, expression->op1, state, parameters);
        evaluateExpression(op2, expression->op2, state, parameters);
        zeroExpand(op1, op2);
        assert(op1.size() == op2.size());
        MultiRefBus::iterator op1Iter = op1.begin(), op2Iter = op2.begin();
        while(op1Iter != op1.end() && op2Iter != op2.end()) {
            result.push_back( notOf(andOf(*op1Iter, *op2Iter)) );
            ++op1Iter; ++op2Iter;
        }
        
        break;
        }
        
      case VerilogDesign::Expression::BITWISE_XNOR: {

        evaluateExpression(op1, expression->op1, state, parameters);
        evaluateExpression(op2, expression->op2, state, parameters);
        zeroExpand(op1, op2);
        assert(op1.size() == op2.size());
        MultiRefBus::iterator op1Iter = op1.begin(), op2Iter = op2.begin();
        while(op1Iter != op1.end() && op2Iter != op2.end()) {
            result.push_back( notOf(xorOf(*op1Iter, *op2Iter)) );
            ++op1Iter; ++op2Iter;
        }
        
        break;
        }
        
      case VerilogDesign::Expression::LOGICAL_AND:

        evaluateExpression(op1, expression->op1, state, parameters);
        evaluateExpression(op2, expression->op2, state, parameters);
        result.push_back(andOf(reductionOr(op1), reductionOr(op2)));
        break;
        
      case VerilogDesign::Expression::LOGICAL_NOT:

        evaluateExpression(op1, expression->op1, state, parameters);
        result.push_back(notOf(reductionOr(op1)));
        break;
        
      case VerilogDesign::Expression::LOGICAL_OR:

        evaluateExpression(op1, expression->op1, state, parameters);
        evaluateExpression(op2, expression->op2, state, parameters);
        result.push_back(orOf(reductionOr(op1), reductionOr(op2)));
        break;
        
      case VerilogDesign::Expression::REDUCTION_AND:
        
        evaluateExpression(op1, expression->op1, state, parameters);
        result.push_back(reductionAnd(op1));
        break;
        
      case VerilogDesign::Expression::REDUCTION_OR:

        evaluateExpression(op1, expression->op1, state, parameters);
        result.push_back(reductionOr(op1));
        break;

      case VerilogDesign::Expression::REDUCTION_XOR:
      
        evaluateExpression(op1, expression->op1, state, parameters);
        result.push_back(reductionXor(op1));
        break;
        
      case VerilogDesign::Expression::REDUCTION_NAND: 

        evaluateExpression(op1, expression->op1, state, parameters);
        result.push_back(notOf(reductionAnd(op1)));
        break;
        
      case VerilogDesign::Expression::REDUCTION_NOR: 

        evaluateExpression(op1, expression->op1, state, parameters);
        result.push_back(notOf(reductionOr(op1)));
        break;
        
      case VerilogDesign::Expression::REDUCTION_XNOR:

        evaluateExpression(op1, expression->op1, state, parameters);
        result.push_back(notOf(reductionXor(op1)));
        break;
        
      case VerilogDesign::Expression::LESS_THAN:

        evaluateExpression(op1, expression->op1, state, parameters);
        evaluateExpression(op2, expression->op2, state, parameters);
        zeroExpand(op1, op2);
        assert(op1.size() == op2.size());
        result.push_back(lessThan(op1, op2));
        break;
        
      case VerilogDesign::Expression::LESS_THAN_EQUAL:

        evaluateExpression(op1, expression->op1, state, parameters);
        evaluateExpression(op2, expression->op2, state, parameters);
        zeroExpand(op1, op2);
        assert(op1.size() == op2.size());
        result.push_back(notOf(lessThan(op2, op1)));
        break;
        
      case VerilogDesign::Expression::GREATER_THAN:

        evaluateExpression(op1, expression->op1, state, parameters);
        evaluateExpression(op2, expression->op2, state, parameters);
        zeroExpand(op1, op2);
        assert(op1.size() == op2.size());
        result.push_back(lessThan(op2, op1));
        break;

      case VerilogDesign::Expression::GREATER_THAN_EQUAL:

        evaluateExpression(op1, expression->op1, state, parameters);
        evaluateExpression(op2, expression->op2, state, parameters);
        zeroExpand(op1, op2);
        assert(op1.size() == op2.size());
        result.push_back(notOf(lessThan(op1, op2)));
        break;
        
      case VerilogDesign::Expression::EQUAL:

        evaluateExpression(op1, expression->op1, state, parameters);
        evaluateExpression(op2, expression->op2, state, parameters);
        zeroExpand(op1, op2);
        assert(op1.size() == op2.size());
        result.push_back(equalTo(op1, op2));
        break;
        
      case VerilogDesign::Expression::NOTEQUAL:

        evaluateExpression(op1, expression->op1, state, parameters);
        evaluateExpression(op2, expression->op2, state, parameters);
        zeroExpand(op1, op2);
        assert(op1.size() == op2.size());
        result.push_back(notOf(equalTo(op1, op2)));
        break;
        
      case VerilogDesign::Expression::IF_ELSE: {

        evaluateExpression(op1, expression->op1, state, parameters);
        evaluateExpression(op2, expression->op2, state, parameters);
        evaluateExpression(op3, expression->op3, state, parameters);
        MultiRef condition = reductionOr(op1);

        zeroExpand(op2, op3);
        assert(op2.size() == op3.size());
        MultiRefBus::iterator op2Iter = op2.begin(), op3Iter = op3.begin();
        while(op2Iter != op2.end() && op3Iter != op3.end()) {
            result.push_back( mux(condition, *op2Iter, *op3Iter) );
            ++op2Iter; ++op3Iter;
        }
        
        break;
        }

      case VerilogDesign::Expression::LEFT_SHIFT:

        evaluateExpression(op1, expression->op1, state, parameters);
        
        if (isConstantExpression(expression->op2, parameters)) {
            // shift signal a constant number of bits by connection
            
            ConstantValue c;
            evaluateConstantExpression(c, expression->op2, parameters);
            if (c.intValue < 0) {
                cerr << "ERROR: Constant shift length must be greater than zero" << endl;
                QUIT_ON_ERROR;                
            }
            int size = op1.size();
            if (c.intValue >= size) {
                cerr << "WARNING: Shift length is greater than length of operand.  Result will be constant zero." << endl;   
            }
        
            MultiRefBus::reverse_iterator bitIter = op1.rbegin();
            for(int i=0; i<size; i++) {
                if(i<c.intValue) {
                    result.push_front(constantZero());
                } else {
                    result.push_front(*bitIter);
                    ++bitIter;
                }
            }        
        } else {
            // shift signal a variable number of bits by multiplexing
           
            evaluateExpression(op2, expression->op2, state, parameters);
            // the number of shift possibilities is the minimum of the bits to be shifted
            // and the largest number that can be represented by the shift value
            unsigned int size = min(op1.size(), static_cast<unsigned int>(1 << op2.size()));
            MultiRefBus selects;
            for(unsigned int i=0; i<size; ++i) {
                MultiRefBus constantBits;
                multiBitConstant(constantBits, i);
                zeroExpand(constantBits, op2);
                selects.push_back(equalTo(constantBits, op2));
            }

            // build a multiplexor for each bit    
            for(MultiRefBus::iterator bitIter = op1.begin(); bitIter != op1.end(); ++bitIter) {
                MultiRefBus muxTerms;            

                // build term for each bit
                MultiRefBus::iterator inputIter(bitIter);
                MultiRefBus::iterator selectIter = selects.begin();
                while(inputIter != op1.end() && selectIter != selects.end()) {
                    muxTerms.push_back(andOf(*selectIter, *inputIter));
                    ++selectIter;
                    if (inputIter == op1.begin()) {
                        inputIter = op1.end();
                    } else {
                        ++inputIter;
                    }
                }

                result.push_back(reductionOr(muxTerms));            
            }
        }
        break;
      
      case VerilogDesign::Expression::RIGHT_SHIFT:

        evaluateExpression(op1, expression->op1, state, parameters);
        
        if (isConstantExpression(expression->op2, parameters)) {
            // shift signal a constant number of bits by connection
            
            ConstantValue c;
            evaluateConstantExpression(c, expression->op2, parameters);
            if (c.intValue < 0) {
                cerr << "ERROR: Constant shift length must be greater than zero" << endl;
                QUIT_ON_ERROR;                
            }
            int size = op1.size();
            if (c.intValue >= size) {
                cerr << "WARNING: Shift length is greater than length of operand.  Result will be constant zero." << endl;   
            }
        
            MultiRefBus::iterator bitIter = op1.begin();
            for(int i=0; i<size; i++) {
                if(i<c.intValue) {
                    result.push_back(constantZero());
                } else {
                    result.push_back(*bitIter);
                    ++bitIter;
                }
            }        
        } else {
            // shift signal a variable number of bits by multiplexing
           
            evaluateExpression(op2, expression->op2, state, parameters);
            // the number of shift possibilities is the minimum of the bits to be shifted
            // and the largest number that can be represented by the shift value
            unsigned int size = min(op1.size(), static_cast<unsigned int>(1 << op2.size()));
            MultiRefBus selects;
            for(unsigned int i=0; i<size; i++) {
                MultiRefBus constantBits;
                multiBitConstant(constantBits, i);
                zeroExpand(constantBits, op2);
                selects.push_back(equalTo(constantBits, op2));
            }

            // build a multiplexor for each bit    
            for(MultiRefBus::iterator bitIter = op1.begin(); bitIter != op1.end(); ++bitIter) {
                MultiRefBus muxTerms;            

                // build term for each bit
                MultiRefBus::iterator inputIter(bitIter);
                MultiRefBus::iterator selectIter = selects.begin();
                while(inputIter != op1.end() && selectIter != selects.end()) {
                    muxTerms.push_back(andOf(*selectIter, *inputIter));
                    ++inputIter; ++selectIter;
                }

                result.push_back(reductionOr(muxTerms));            
            }
        }
        break;
        
      case VerilogDesign::Expression::ADD:
      
        evaluateExpression(op1, expression->op1, state, parameters);
        evaluateExpression(op2, expression->op2, state, parameters);
        zeroExpand(op1, op2);
        assert(op1.size() == op2.size());
        arithmeticAdd(result, op1, op2);
        break;
        
      case VerilogDesign::Expression::SUBTRACT:

        evaluateExpression(op1, expression->op1, state, parameters);
        evaluateExpression(op2, expression->op2, state, parameters);
        zeroExpand(op1, op2);
        assert(op1.size() == op2.size());
        arithmeticSubtract(result, op1, op2);
        break;
        
      case VerilogDesign::Expression::NEGATE:
      
        evaluateExpression(op1, expression->op1, state, parameters);
        op2.clear();
        zeroExpand(op1, op2);
        assert(op1.size() == op2.size());
        arithmeticSubtract(result, op2, op1);
        break;
        
      case VerilogDesign::Expression::MULTIPLY:
      
        evaluateExpression(op1, expression->op1, state, parameters);
        evaluateExpression(op2, expression->op2, state, parameters);
        arithmeticMultiply(result, op1, op2);
        break;
        
      case VerilogDesign::Expression::DIVIDE:

        evaluateExpression(op1, expression->op1, state, parameters);
        evaluateExpression(op2, expression->op2, state, parameters);
        zeroExpand(op1, op2);
        assert(op1.size() == op2.size());
        arithmeticDivide(result, op1, op2);
        break;
        
      case VerilogDesign::Expression::MODULO:

        evaluateExpression(op1, expression->op1, state, parameters);
        evaluateExpression(op2, expression->op2, state, parameters);
        zeroExpand(op1, op2);
        assert(op1.size() == op2.size());
        arithmeticModulo(result, op1, op2);
        break;

      default:
        QUIT_ON_INTERNAL_ERROR;
        break;
    }

    for(MultiRefBus::iterator it = result.begin(); it != result.end(); it++)
        assert(it->type != NEITHER);   
}


// *****************************************************************************
// getContextualValue()
//
/// \brief Return the MultiRef for a particular net, including any procedural assignment that it may have.
//
// *****************************************************************************
MultiRef
VerilogSynthesis::getContextualValue(oa::oaModBitNet *net, ProceduralState *state) {
    map<oa::oaModBitNet*, MultiRef>::iterator it;
    if (state && (it = state->blockingAssignments.find(net)) != state->blockingAssignments.end() ) {
        return it->second;
    } else {
        return MultiRef(net);
    }
}


// *****************************************************************************
// appendSuffix()
//
/// \brief Appends a suffix onto a string to construct a valid Verilog name
///
/// This routine is necessary because escaped Verilog names must end in a space.
///
/// \param root the string on to which the suffix is appended
/// \param suffix
//
// *****************************************************************************
void
VerilogSynthesis::appendSuffix(string &root, const string suffix) {
  if (root[root.length()-1] == ' ') {
    root.erase(root.length());
    root += suffix + " ";
  } else {
    root += suffix;
  }
}


// *****************************************************************************
// LvalRef constructors
//
/// \brief Constructs an Lval for a function variable.
///
/// \param c the identifier of the function variable to assign
/// \param b the bit of the variable to assign
//
// *****************************************************************************
VerilogSynthesis::LvalRef::LvalRef(string c, int b) { 
    type = LVAL_FUNCTION; 
    functionLvalName = c; 
    functionLvalBit = b; 
}


// *****************************************************************************
// LvalRef constructors
//
/// \brief Constructs an Lval that is an unconditional assignment.
///
/// \param n the net to be assigned
//
// *****************************************************************************
VerilogSynthesis::LvalRef::LvalRef(oa::oaModBitNet* n) { 
    type = VerilogSynthesis::LVAL_UNCONDITIONAL; 
    unconditionalLval = n; 
}


// *****************************************************************************
// LvalRef constructors
//
/// \brief Constructs an Lval that is a conditional assignment.
///
/// The nets and conditions for assignment are supplied in subsequent calls to
/// addConditional.
//
// *****************************************************************************
VerilogSynthesis::LvalRef::LvalRef() { 
    type = LVAL_CONDITIONAL; 
}


// *****************************************************************************
// LvalRef::addConditional
//
/// \brief Adds another conditional assignment.
///
/// \param n the net to be conditionally assigned
/// \param c the condition under which it is to be assigned
//
// *****************************************************************************
void
VerilogSynthesis::LvalRef::addConditional(oa::oaModBitNet *n, MultiRef c) {
    assert(type == LVAL_CONDITIONAL);
    conditionalLval.push_back(ConditionalLvalRef(n,c));    
}
                        
}
