/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-05-31: ChangeLog started
*/

#if !defined(oagFuncVerilogDesign_P)
#define oagFuncVerilogDesign_P

#include <string>
#include <list>

namespace oagFunc {

// *****************************************************************************
// VerilogDesign
//
/// \brief An entire Verilog design.
//
// *****************************************************************************

class VerilogDesign {

  private:

    /// If DELETE_UPON_DESTRUCTION is set true,
    /// the destructor of a VerilogDesign will
    /// deallocate all of the memory in the subcomponents.
    static const bool DELETE_UPON_DESTRUCTION = true;

  public:
  
    class AlwaysBlock;
    class Assignment;
    class Bundle;
    class Case;
    class Declaration;
    class Expression;
    class Function;
    class Instantiation;
    class Module;
    class Port;
    class PortConnection;
    class Primary;
    class Statement;
    class Trigger;

    std::list<Module *>       modules;

    // *****************************************************************************
    // Module
    //
    /// \brief A single Verilog module
    //
    /// This module may spawn several OpenAccess modules if multiple
    /// parameterizations exist.
    //
    // *****************************************************************************

    class Module {
      public:
        std::string                                name;

        VerilogDesign                             *design;
        std::list<Port*>                          *ports;
        std::list<Declaration*>                    declarations;
        std::list<Declaration*>                    parameters;
        std::list<Declaration*>                   *parameterOverrides;
        std::list<Assignment*>                    *assignments;
        std::list<AlwaysBlock*>                   *alwaysBlocks;
        std::list<Statement*>                     *initialBlocks;
        std::list<Function*>                      *functions;
        std::list<Instantiation*>                 *instantiations;

        Module() {
          design = NULL;
          ports = NULL;
          assignments = NULL;
          alwaysBlocks = NULL;
          initialBlocks = NULL;
          functions = NULL;
          instantiations = NULL;
        };
        // Destructor
        ~Module();
    };

    // *****************************************************************************
    // Declaration
    //
    /// \brief A wire, port, reg, or data type declaration.
    ///
    /// Five declaration types are currently supported:
    /// \li wire
    /// \li input, output, inout
    /// \li reg
    //
    // *****************************************************************************

    class Declaration {
      public:

        std::string              name;
        typedef enum{UNKNOWN, 
                     INPUT, 
                     OUTPUT, 
                     WIRE, 
                     REG,
                     INOUT, 
                     PARAMETER,
                     SUPPLY0,
                     SUPPLY1,
                     TRI,
                     WIREAND, // not commonly used
                     WIREOR,
                     TRI0,
                     TRI1,
                     TRIAND,
                     TRIOR
                     } Type;
        Type                     type;

        Expression              *start, *stop;
        Expression              *start2D, *stop2D;
    
        Expression              *value;

        Declaration() { start = stop = start2D = stop2D = value = NULL; }
        // Destructor
        ~Declaration();
    };

    // *****************************************************************************
    // Function
    //
    /// \brief A function definition
    //
    // *****************************************************************************

    class Function {
      public:

        std::string                 name;
    
        Expression                 *start, *stop;

        std::list<Declaration *>   *declarations;
        Statement                  *action;

        Function() { start = stop = NULL; declarations = NULL; action = NULL; }
        // Destructor
        ~Function();
    };

    // *****************************************************************************
    // Port
    //
    /// \brief A port definition (from inside a portlist).
    //
    // *****************************************************************************

    class Port {
      public:
          
        std::string       externalName;
        int               position;
        std::string       internalName;

        // Destructor
        ~Port() { }
    };
    
    // *****************************************************************************
    // Trigger
    //
    /// \brief A procedural trigger.
    ///
    /// A trigger may be one of three types:
    /// \li posedge
    /// \li negedge
    /// \li both edges/any change
    //
    // *****************************************************************************

    class Trigger {
      public:
        typedef enum{UNKNOWN, 
                     BOTH, 
                     POSEDGE, 
                     NEGEDGE}    Type;        
        Type                     type;
        Expression              *net;

        Trigger() { net = NULL; }
        // Destructor
        ~Trigger();
    };

    // *****************************************************************************
    // Assignment
    //
    /// \brief A continuous assignment.
    //
    // *****************************************************************************

    class Assignment {
      public:

        Expression        *lval;
        Expression        *value;

        Assignment() { lval = value = NULL; }
        // Destructor
        ~Assignment();
    };

    // *****************************************************************************
    // AlwaysBlock
    //
    /// \brief An always block.
    //
    // *****************************************************************************

    class AlwaysBlock {
      public:   
        
        std::list<Trigger*>  *triggers; 
        Statement            *action;

        AlwaysBlock() { triggers = NULL; action = NULL; }
        // Destructor
        ~AlwaysBlock();
    };


    // *****************************************************************************
    // Case
    //
    /// \brief A single case inside of a case statement.
    //
    // *****************************************************************************

    class Case {
      public:
        std::list<Expression*> *conditions;
        Statement              *action;
        bool                    isDefault;

        Case() { conditions = NULL; action = NULL; }
        // Destructor
        ~Case();
    };

    // *****************************************************************************
    // Statement
    //
    /// \brief A procedural statement.
    ///
    /// The statement can be one of the following types:
    /// \li case
    /// \li if / if-else
    /// \li blocking assignment
    /// \li non-blocking assignment
    /// \li begin-end block
    //
    // *****************************************************************************

    class Statement {
      public:

        typedef enum{NOP, 
                     BLOCK, 
                     IF, 
                     CASE, CASEX, CASEZ,
                     BLOCKING_ASSIGNMENT, 
                     NONBLOCKING_ASSIGNMENT} Type;              
        Type                                 type;
    
        // for if/case statements...
        struct {
        Expression                *condition;
        Statement                 *ifTrue;
        Statement                 *ifFalse;
    
        std::list<Case*>          *cases;
        } ifc;

        // for assign statements...
        struct {
        Expression                *lval;
        Expression                *rval;
        } assign;
    
        // for begin/end statements...
        struct {
        std::list<Statement *>    *block;
        std::list<Declaration*>   *declarations;
        } begin_end;

        std::string                name;

        Statement() {
          type = NOP; 
          ifc.condition = NULL;
          ifc.ifTrue = ifc.ifFalse = NULL;
          ifc.cases = NULL;
          assign.lval = assign.rval = NULL;
          begin_end.block = NULL;
          begin_end.declarations = NULL;
        }
        // Destructor
        ~Statement();
    };

    // *****************************************************************************
    // Instantiation
    //
    /// \brief A module instantiation.
    //
    // *****************************************************************************

    class Instantiation {
      public: 
          
        std::string                     name;
        std::string                     type;
        typedef enum { ISNT_PRIMITIVE,
                       AND, NAND,
                       OR, NOR,
                       XOR, XNOR,
                       NOT, BUF } PrimitiveType;
        PrimitiveType             primitive;

        std::list<Expression*>         *parameters;
        std::list<PortConnection*>     *connections;

        Instantiation() { parameters = NULL; connections = NULL; }
        // Destructor
        ~Instantiation();
    };

    // **************************************************************************
    // PortConnection
    //
    /// \brief A port connection inside of a module instantiation.
    //
    // *****************************************************************************

    class PortConnection {
      public:
  
        std::string               name;
        int                       position;
        Expression               *value;
        
        PortConnection() { position = -1; value = NULL; }
        PortConnection(const std::string &n, int p, Expression *v) { name = n; position = p; value = v; }

        // Destructor
        ~PortConnection();
    };
            
    // *****************************************************************************
    // Expression
    //
    /// \brief An expression.
    ///
    /// The expression is broken into single operations and built recursively.
    ///
    /// Each decomposed expression may be involve one operator from the following classes:
    /// \li arithmetic: +, -, *, /, %
    /// \li bit-wise: ~, &, |, ^, ~&, ~|, ~^, ^~
    /// \li logical: !, &&, ||
    /// \li shift: <<, >>
    /// \li conditional:  ?/:
    /// \li reduction: &, |, ^, ~&, ~|, ~^, ^~
    /// \li equality: ==, !=, ===, !==
    /// \li comparison: <, >, <=, >=
    /// \li bundle
    /// \li primary
    //
    // *****************************************************************************

    class Expression {
      public:

      typedef enum{UNKNOWN, 
                   PRIMARY, 
                   BUNDLE,
                   BITWISE_AND, BITWISE_NAND, 
                   BITWISE_OR, BITWISE_NOR,
                   BITWISE_XOR, BITWISE_XNOR,
                   BITWISE_NOT,
                   LOGICAL_AND, LOGICAL_NOT, LOGICAL_OR,
                   REDUCTION_AND, REDUCTION_OR, REDUCTION_XOR,
                   REDUCTION_NAND, REDUCTION_NOR, REDUCTION_XNOR,
                   LESS_THAN, LESS_THAN_EQUAL,
                   GREATER_THAN, GREATER_THAN_EQUAL,
                   EQUAL, NOTEQUAL,
                   IF_ELSE, 
                   LEFT_SHIFT, RIGHT_SHIFT,
                   ADD, SUBTRACT, 
                   MULTIPLY, 
                   DIVIDE, MODULO, 
                   NEGATE}           Operator;
        Operator                     type;

        union {
            Expression               *op1;
            Primary                  *primary;
            Bundle                   *bundle;
        };
        
        Expression                   *op2, *op3;
        
        Expression()                                { op1 = op2 = op3 = NULL; type = UNKNOWN; }
        Expression(Primary *p)                      { primary = p; type = PRIMARY; }
        Expression(Bundle *b)                       { bundle = b; type = BUNDLE; }
        Expression(Expression *e1)                  { op1 = e1; op2 = op3 = NULL; type = UNKNOWN; }
        Expression(Expression *e1, Expression *e2)  { op1 = e1; op2 = e2; op3 = NULL; type = UNKNOWN; } 
        Expression(Expression *e1, 
                   Expression *e2,
                   Expression *e3)                  { op1 = e1; op2 = e2; op3 = e3; type = UNKNOWN; } 

        // Destructor
        ~Expression();
    };

    // *****************************************************************************
    // Primary
    //
    /// \brief An expression primary.
    ///
    /// The primary can either be:
    /// \li a constant value
    /// \li a parameter
    /// \li a net or register
    /// \li a function call
    //
    // *****************************************************************************

    class Primary {
      public:

        typedef enum{UNKNOWN, 
                     NET, 
                     CONST, 
                     FUNCTION_CALL} Type;      
        Type                        type;
          
        std::string                 name;

        struct {
            bool                    negative;
            unsigned int            intValue;
            int                     bitWidth;
            unsigned int            xMask, zMask;
        } number;
    
        struct {       
            Expression              *start;
            Expression              *stop;
        } range;

        std::list<Expression *>     *arguments; 
      
        Primary(const std::string &n) { 
          // scalar net primary (of name n)
          name = n; range.start = range.stop = NULL; type = NET; 
          arguments = NULL;
        }
        Primary(unsigned int v, unsigned int w) {
          // constant primary (of value v and bitwidth w)
          range.start = range.stop = NULL; 
          number.intValue = v; number.bitWidth = w; 
          type = CONST; number.negative = false; 
          arguments = NULL;
        }
        Primary(const std::string &n, Expression *i) { 
          // single-bit net primary (of name n and index i)
          name = n; range.start = range.stop = i; type = NET; 
          arguments = NULL;
        }
        Primary(const std::string &n, Expression *f, Expression *l) {
          // multi-bit net primary (of name n and indices f to l)
          name = n; range.start = f; range.stop = l; type = NET; 
          arguments = NULL;
        }
        Primary(const std::string &n, std::list<Expression *> *args) { 
          // functional call primary (of name n with arguments)
          name = n; arguments = args; type = FUNCTION_CALL; 
          range.start = range.stop = NULL; 
        }

        // Destructor
        ~Primary();
    };
    
    // *****************************************************************************
    // Bundle
    //
    /// \brief A concatenation of expressions.
    //
    // *****************************************************************************

    class Bundle {
      public:
          
        std::list<Expression*>    *members;    
        Expression                *replication;

        Bundle(std::list<Expression*> *m)                      { members = m; replication = NULL;}
        Bundle(std::list<Expression*> *m, class Expression *r) { members = m; replication = r;}
        // Destructor
        ~Bundle();
    };

    ~VerilogDesign();
};

}

#endif
