/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-05-31: ChangeLog startedo
*/

#if !defined(oagFuncSynthesis_P)
#define oagFuncSynthesis_P

#include "oaDesignDB.h"
#include "oagAiGraph.h"
#include "oagFunc.h"
#include "oagFuncMultiRef.h"
#include "oagFuncVerilogDesign.h"
#include "oagFuncManager.h"
#include <list>
#include <set>
#include <string>

using namespace std;

namespace oagFunc {

// *****************************************************************************
// Synthesis
//
/// \brief Class for synthesizing various design objects into OpenAccess functional designs.
///
/// All calls are static functions.  This class is meant as a utility box for
/// synthesizing new designs.
//
// *****************************************************************************

class Synthesis {

  public:
  
    static void                    setLibrary(oa::oaLib *library) { currentLibrary = library; }
    static void                    setView(oa::oaView *view) { currentView = view; }
    static void                    setOverwriteStructure(bool overwrite);
    static void                    addLeafLibrary(oa::oaLib *library) { leafLibs.push_back(library); }
    static void                    addLeafView(const oa::oaScalarName &viewName) { 
      leafViews.push_back(viewName); 
    }

  protected:

// *****************************************************************************
// Type definitions for various objects.
// *****************************************************************************

typedef struct {int intValue; int bitWidth;}              ConstantValue;
typedef struct {int upper; int lower;}                    Bounds;

// *****************************************************************************
 
    /// \name Creating OpenAccess Structure
    /// @{

    static oa::oaModule*           createModule(const string identifier);
    static oa::oaModScalarNet*     createScalarNet(const string identifier);
    static oa::oaModBusNet*        createBusNet(const string identifier, int start, int stop);
    static oa::oaModTerm*          createTerm(oa::oaModNet* net, const string identifier, 
                                              oa::oaTermType direction, 
                                              unsigned int portPosition = oacNullIndex);
    static oa::oaModInst*          instantiateModule(oa::oaDesign *master, const string name);   
    static void                    connectPort(oa::oaModInst *inst, oa::oaModNet* net, 
                                               unsigned int portPosition);
    static void                    connectPort(oa::oaModInst *inst, oa::oaModNet* net, 
                                               const string port);
    /// @}
    /// \name Finding Named Objects
    /// @{

    static oa::oaModNet*           findNet(const string identifier);
    static oa::oaModScalarNet*     findScalarNet(const string identifier);
    static oa::oaModBusNet*        findBusNet(const string identifier);
    static oa::oaModBusNetBit*     findBusNetBit(const string identifier, const int bit);
    static oa::oaModule*           findModule(const string identifier);
    static bool                    checkForExistenceOfDesign(oa::oaScalarName libName,
                                                             oa::oaScalarName cellName,
                                                             oa::oaScalarName viewName);

    /// @}
    /// \name Connecting OA and AI Objects
    /// @{

    static void                    assignMultiRef(oa::oaModBitNet *net, const MultiRef e);
    static void                    annotateAsynchronousSignal(oagAi::Ref sequential, 
                                                              const string label, 
                                                              MultiRef signal);

    /// @}
    /// \name Primitive Logic
    /// @{

    static MultiRef                notOf(MultiRef e);
    static MultiRef                orOf(MultiRef e1, MultiRef e2);
    static MultiRef                andOf(MultiRef e1, MultiRef e2);
    static MultiRef                xorOf(MultiRef e1, MultiRef e2);
    static MultiRef                mux(MultiRef select, MultiRef in0, MultiRef in1);
    static void                    fullAdder(MultiRef &sum, MultiRef &carryOut, 
                                             MultiRef e1, MultiRef e2, MultiRef carryIn);
    
    /// @}
    /// \name Multi-bit Comparison and Selection Operations
    /// @{
    
    static MultiRef                reductionOr(MultiRefBus &l);
    static MultiRef                reductionAnd(MultiRefBus &l);
    static MultiRef                reductionXor(MultiRefBus &l);
    static MultiRef                lessThan(MultiRefBus &l1, MultiRefBus &l2);
    static MultiRef                equalTo(MultiRefBus &l1, MultiRefBus &l2);
    static void                    mux(MultiRefBus &result, MultiRef select, 
                                       MultiRefBus &in0, MultiRefBus &in1);
    /// @}
    /// \name Arithmetic Operations
    /// @{

    static void                    arithmeticAdd(MultiRefBus &result, 
                                                 MultiRefBus &l1, MultiRefBus &l2);
    static void                    arithmeticSubtract(MultiRefBus &result, 
                                                      MultiRefBus &l1, MultiRefBus &l2);
    static void                    arithmeticSubtract(MultiRefBus &result, MultiRef &negFlag, 
                                                      MultiRefBus &l1, MultiRefBus &l2);
    static void                    arithmeticMultiply(MultiRefBus &result, 
                                                      MultiRefBus &l1, MultiRefBus &l2);
    static void                    arithmeticDivide(MultiRefBus &result, 
                                                    MultiRefBus &l1, MultiRefBus &l2);
    static void                    arithmeticModulo(MultiRefBus &result, 
                                                    MultiRefBus &l1, MultiRefBus &l2);

    /// @}
    /// \name Sequential Elements    
    /// @{

    static MultiRef                latch(MultiRef enable, MultiRef in, const string name = "");
    static MultiRef                seq(MultiRef in, const string name = "");

    /// @}
    /// \name Constants
    /// @{

    static MultiRef                constantZero();
    static MultiRef                constantOne();
    static void                    zeroExpand(MultiRefBus &l1, MultiRefBus &l2);
    static void                    multiBitConstant(MultiRefBus &result, int value, int bits = 0);

    /// @}
    /// \name State Information
    /// @{

    static oa::oaNameSpace        *nameSpace;    
    static oa::oaLib              *currentLibrary;
    static oa::oaView             *currentView;
    static Manager                *currentManager;
    static oa::oaModule           *currentModule;
    static bool                    overwriteStructure;

    // leaf libraries
    static list<oa::oaLib*>        leafLibs;
    static list<oa::oaScalarName>  leafViews;

    // naming settings
    static const unsigned int      NAME_LENGTH_LIMIT = 256;

    /// @}    

};

}

#endif
