/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-11-08: ChangeLog started
*/

#include <iostream>
#include "oagFuncSimOcc.h"

#include "oagFuncDebug.h"

using namespace std;

namespace oagFunc {


// *****************************************************************************
// SimOcc()
//
/// \brief Constructor.
//
// *****************************************************************************
SimOcc::SimOcc(oa::oaDesign *design) { 
    assert(design);
    assert(design->getTopOccurrence());
    clear();
    this->design = design;
    // determine BITS_PER_RAND
    BITS_PER_RAND = 0;
    for(unsigned int i=1; i<sizeof(int)*8; i++) {
      if (RAND_MAX >= (1<<i)) {
        BITS_PER_RAND++;
      }
    }
}


// *****************************************************************************
// clear()
//
/// \brief Clears all previously set and computed vectors.
//
// *****************************************************************************   
void
SimOcc::clear() {
    data.clear();
    toBeSimulated.clear();
}


// *****************************************************************************
// setVector()
//
/// \brief Sets the simulation vector of a reference.
//
/// \param ref
/// \param vec
//
// *****************************************************************************   
void
SimOcc::setVector(OccRef &ref, SimulationVector vec) {
    if (OccGraph::isInverted(ref)) {
        data[OccGraph::getNonInverted(ref)] = ~vec;
    } else {
        data[ref] = vec;
    }
    toBeSimulated.push_back(OccGraph::getNonInverted(ref));
}


// *****************************************************************************
// generateRandomStateVectors()
//
/// \brief Generates a set of random vectors at all state bits.
//
// *****************************************************************************
void
SimOcc::generateRandomStateVectors() {
    list<OccRef> states;
    OccGraph::getStates(design->getTopOccurrence(), states);
    for(list<OccRef>::iterator it=states.begin(); it!=states.end(); ++it) {
        setVector(*it, randomVector());
    }
}


// *****************************************************************************
// generateRandomInputVectors()
//
/// \brief Generates a set of random vectors at all inputs.
//
// *****************************************************************************
void
SimOcc::generateRandomInputVectors() {
    list<OccRef> inputs;
    OccGraph::getInputs(design->getTopOccurrence(), inputs);
    for(list<OccRef>::iterator it=inputs.begin(); it!=inputs.end(); ++it) {
        setVector(*it, randomVector());
    }
}


// *****************************************************************************
// randomVector()
//
/// \brief Returns a random simulation vector.
//
/// \return a random vector
//
// *****************************************************************************
SimulationVector
SimOcc::randomVector() {
    const int RAND_MASK = (1 << BITS_PER_RAND) - 1;

    SimulationVector result = 0;
    int bits = sizeof(int)*8;
    while(bits>0) {
      result = (result << BITS_PER_RAND) | (rand() & RAND_MASK);
      bits -= BITS_PER_RAND;
    }
    return result;
}

    
// *****************************************************************************
// runFull()
//
/// \brief Propagates simulation vectors through all combinational logic.
//
// *****************************************************************************
void
SimOcc::runFull() {

    toBeSimulated.clear();
    set<OccRef> alreadyVisited;
 
    // seed all the start points
    
    // inputs and states
    list<OccRef> inputList, stateList;
    OccGraph::getInputs(design->getTopOccurrence(), inputList);
    OccGraph::getStates(design->getTopOccurrence(), stateList);
    for(list<OccRef>::iterator it = inputList.begin();
        it != inputList.end(); ++it) {
        toBeSimulated.push_back(*it);
        alreadyVisited.insert(*it);
    }
    for(list<OccRef>::iterator it = stateList.begin();
        it != stateList.end(); ++it) {
        toBeSimulated.push_back(*it);
        alreadyVisited.insert(*it);
    }    
    // constants
    list<OccRef> zeroList, oneList;
    OccGraph::getConstants(design->getTopOccurrence(), zeroList, oneList);
    for(list<OccRef>::iterator it = zeroList.begin();
        it != zeroList.end(); ++it) {
        data[*it] = 0;
        toBeSimulated.push_back(*it);
        alreadyVisited.insert(*it);
    }
    for(list<OccRef>::iterator it = oneList.begin();
        it != oneList.end(); ++it) {
        data[*it] = ~(0);
        toBeSimulated.push_back(*it);
        alreadyVisited.insert(*it);
    }
 
    while(!toBeSimulated.empty()) {
        oagFunc::OccRef ref = toBeSimulated.front();
        assert(!OccGraph::isInverted(ref));
        toBeSimulated.pop_front();

        #ifdef DEBUG
        oa::oaString occPathString;
        if (ref.occurrence->getOccInst()) {
            ref.occurrence->getOccInst()->getPathName(oa::oaVerilogNS(), occPathString);
        } else {
            occPathString = "(top)";
        }
        DEBUG_PRINT("- ai " << occPathString << "." << ref.ref << "\t")
        #endif

        switch(OccGraph::getNodeType(ref)) {
          case oagAi::Node::AND: {
            DEBUG_PRINTMORE("and")

            OccRef left = OccGraph::getAndLeft(ref);
            OccRef nonInvLeft = OccGraph::getNonInverted(left);
            OccRef right = OccGraph::getAndRight(ref);
            OccRef nonInvRight = OccGraph::getNonInverted(right);

            SimulationVector leftVec;
            if(OccGraph::getNodeType(nonInvLeft) == oagAi::Node::CONSTANT0) {
                leftVec = 0;
            } else if(data.find(nonInvLeft) == data.end()) {
                DEBUG_PRINTMORE(" need " << nonInvLeft.ref << endl)
                alreadyVisited.erase(ref);
                continue;
            } else {
                leftVec = data[nonInvLeft];
            }
            leftVec = OccGraph::isInverted(left) ? ~leftVec : leftVec;
            
            SimulationVector rightVec;
            if(OccGraph::getNodeType(nonInvRight) == oagAi::Node::CONSTANT0) {
                rightVec = 0;
            } else if(data.find(nonInvRight) == data.end()) {
                DEBUG_PRINTMORE(" need " << nonInvRight.ref << endl)
                alreadyVisited.erase(ref);
                continue;
            } else {
                rightVec = data[nonInvRight];
            }
            rightVec = OccGraph::isInverted(right) ? ~rightVec : rightVec;
            
            data[ref] = leftVec & rightVec;
            
            break;
            }

          case oagAi::Node::SEQUENTIAL: {
            DEBUG_PRINTMORE("seq")
            assert(data.find(ref) != data.end());            
            break;
            }
            
          case oagAi::Node::TERMINAL: {
            DEBUG_PRINTMORE("terminal")
            assert(data.find(ref) != data.end());
            break;
            }
            
         default:
            break;
        }
        
        DEBUG_PRINTMORE(" vector=" << data[ref])
        
        // first follow immediate fanout (to AND nodes)
        list<oagFunc::OccRef> fanoutList;
        OccGraph::getFanout(ref, fanoutList);

        DEBUG_PRINTMORE(" (immediate fanout= " << fanoutList.size() << " ")
        for(list<OccRef>::iterator fanoutIter = fanoutList.begin();
                fanoutIter != fanoutList.end(); ++fanoutIter) {
            OccRef fanoutRef = *fanoutIter;
            assert(!OccGraph::isInverted(fanoutRef));
            if (OccGraph::getNodeType(fanoutRef) == oagAi::Node::AND &&
                alreadyVisited.find(fanoutRef) == alreadyVisited.end()) {
                toBeSimulated.push_back(fanoutRef);
                alreadyVisited.insert(fanoutRef);
            }
        }
        
        // follow connections through OA hierarchy (to other terminal nodes)
        std::set<oagFunc::OccRef> connectedRefs;
        std::set<oa::oaOccBitNet*> connectedNets;
        OccGraph::getAllConnections(ref, connectedNets, connectedRefs,
                                    true, false, true, true, false, false);
                  
        DEBUG_PRINTMORE("though hierarchy= " << connectedRefs.size())
        for(std::set<OccRef>::iterator connectedIter = connectedRefs.begin();
                connectedIter != connectedRefs.end(); ++connectedIter) {
            OccRef connectedRef = *connectedIter;
            OccRef nonInvConnectedRef = OccGraph::getNonInverted(connectedRef);
            assert(OccGraph::getNodeType(connectedRef) == oagAi::Node::TERMINAL);
            
            if (alreadyVisited.find(nonInvConnectedRef) != alreadyVisited.end()) {
                ;
            } else if (OccGraph::isInverted(connectedRef)) { 
                data[nonInvConnectedRef] = ~data[ref];
                toBeSimulated.push_back(nonInvConnectedRef); 
                alreadyVisited.insert(nonInvConnectedRef);
            } else {
                data[nonInvConnectedRef] = data[ref];
                toBeSimulated.push_back(nonInvConnectedRef);
                alreadyVisited.insert(nonInvConnectedRef);                
            }
        }
        
        DEBUG_PRINTMORE(")" << endl)
    }
}


// *****************************************************************************
// getVector()
//
/// \brief Returns the simulation vector stored for a reference.
///
/// \param ref
/// \param vec
/// \return true, if the vector is valid (has been simulated)
//
// *****************************************************************************
bool    
SimOcc::getVector(const OccRef &ref, SimulationVector &vec) {
    if (!toBeSimulated.empty() && data.find(ref) == data.end()) {
        return false;
    }
    
    vec = data[ref];
    return true;
}


// *****************************************************************************
// nextCycle()
//
/// \brief Advances all of the state bits to the next state value.
//
// *****************************************************************************
void    
SimOcc::nextCycle() {
    list<OccRef> stateList;
    OccGraph::getStates(design->getTopOccurrence(), stateList);
    for(list<OccRef>::iterator it = stateList.begin();
        it != stateList.end(); ++it) {
        assert(!OccGraph::isInverted(*it));
        OccRef nextState = OccGraph::getNextState(*it);
        OccRef nonInvNextState = OccGraph::getNonInverted(nextState);
        if (OccGraph::isNull(nextState)) {
            // no next state input.  do nothing
        } else if (OccGraph::isInverted(nextState)) {
            data[*it] = ~data[nonInvNextState];
        } else {
            data[*it] = data[nonInvNextState];
        }
        toBeSimulated.push_back(*it);
    }    
}

}
