/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-05-31: ChangeLog started
*/

#include <iostream>
#include "oagAiGraph.h"
#include "oagFuncQueryAi.h"

#include "oagFuncDebug.h"

using namespace std;

namespace oagFunc {

// *****************************************************************************
// QueryAi()
//
/// \brief Constructor.
///
/// \param graph the Ai Graph of interest
/// \param engine a ReasoningEngine to manipulate functional information
//
// *****************************************************************************
QueryAi::QueryAi(oagAi::Graph *graph, GenericReasoningEngine *engine) { 
    assert(graph);
    assert(engine);
    this->graph = graph; this->engine = engine; 
}

                            
// *****************************************************************************
// getNextState()
//
/// \brief Returns the next state function of a sequential node.
//
// *****************************************************************************
GenericFunction
QueryAi::getNextState(oagAi::Ref ref) {
    assert(graph->getNodeType(ref) == oagAi::Node::SEQUENTIAL);
    return get(graph->getNextState(ref));
}


// *****************************************************************************
// get()
//
/// \brief Returns the function of an Ai Ref.
//
// *****************************************************************************
GenericFunction
QueryAi::get(oagAi::Ref ref) {
    // initialize
    visitedSet.clear();
    
    return getRecursively(ref);
}


// *****************************************************************************
// getRecursively()
//
/// \brief Returns the function of an Ai Ref.
//
// *****************************************************************************
GenericFunction
QueryAi::getRecursively(oagAi::Ref ref) {
    GenericFunction result;

    // is this result currently cached?
    if (precomputedCache.find(ref) != precomputedCache.end()) {
        return precomputedCache[ref];
    }

    DEBUG_PRINT("- ai " << ref << "\t")

    if (graph->isInverted(ref)) {
        DEBUG_PRINTMORE("not " << endl)
        result = engine->genericNotOf(getRecursively(graph->notOf(ref)));
        precomputedCache[ref] = result;
        return result;
    }

    // is there a user-specified function at this point?
    if (setFunctions.find(ref) != setFunctions.end()) {
        return setFunctions[ref];
    }

    switch(graph->getNodeType(ref)) {
      case oagAi::Node::AND: {
        DEBUG_PRINTMORE("and" << endl)
        
        // check for cycle
        if (visitedSet.find(ref) != visitedSet.end()) {
            cerr << "ERROR: Combinational cycle detected" << endl;
            QUIT_ON_ERROR;
        } else {
            visitedSet.insert(ref);
        }
        
        result = engine->genericAndOf(getRecursively(graph->getAndLeft(ref)), getRecursively(graph->getAndRight(ref)));
        break;
        }
        
      case oagAi::Node::CONSTANT0:
        DEBUG_PRINTMORE("zero" << endl)
        return engine->genericGetZero();
        break;  
        
      case oagAi::Node::SEQUENTIAL:
        DEBUG_PRINTMORE("seq" << endl)
        cerr << "ERROR: Unassigned state bit: " << ref << endl;
        QUIT_ON_ERROR;
        break;
        
      case oagAi::Node::TERMINAL: 
        DEBUG_PRINTMORE("terminal" << endl)
        
        if (!graph->isNull(graph->getTerminalDriver(ref))) {
            result = getRecursively(graph->getTerminalDriver(ref));
            break;
        }
        
        cerr << "ERROR: Unassigned node with no predecessor: " << ref << endl;
        QUIT_ON_ERROR;
        break;
        
      default:
        cerr << "ERROR: Broken AI node: " << ref << endl;
        QUIT_ON_INTERNAL_ERROR;
    }
    
    precomputedCache[ref] = result;
    return result;
}


// *****************************************************************************
// get()
//
/// \brief Returns the function of an Ai Ref given an explicit cut.
//
// *****************************************************************************
GenericFunction
QueryAi::get(oagAi::Ref ref, oagAi::Graph::Cut & cut) {
    static_cast<void>(ref);
    static_cast<void>(cut);
    // UNIMPLEMENTED
    QUIT_ON_INTERNAL_ERROR
    return 0;
}


// *****************************************************************************
// set()
//
/// \brief Specifies the function at an Ai Ref.
///
/// The Ai Ref can be an inverted reference.  The inverse of the provided
/// function will be computed immediately and set to the noninverted reference.
//
// *****************************************************************************
void
QueryAi::set(oagAi::Ref ref, GenericFunction f) {
    if (graph->isInverted(ref)) {
        setFunctions[graph->notOf(ref)] = engine->genericNotOf(f);
    } else {
        setFunctions[ref] = f;
    }

    // clear the entire precomputed cache
    precomputedCache.clear();
}
   

// *****************************************************************************
// unset()
//
/// \brief Clears any explicitly specified function at an Ai Ref.
//
// ***************************************************************************** 
void
QueryAi::unset(oagAi::Ref ref) {
    map<oagAi::Ref, GenericFunction>::iterator it = setFunctions.find(graph->getNonInverted(ref));

    if (it == setFunctions.end()) {
        return;
    }

    setFunctions.erase(it);
    
    // clear the entire precomputed cache
    precomputedCache.clear();
}

}
