/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron Hurst <ahurst@eecs.berkeley.edu>

ChangeLog:
2005-07-20: ChangeLog started
*/

#include <iostream>
#include "oaDesignDB.h"
#include "oagFuncOccGraph.h"
#include "oagFuncManager.h"

// *****************************************************************************
// getRefName()
//
/// \brief Searches for an equivalent net with a name, otherwise returns "UNNAMED".
///
/// \param ref
/// \return a name for the reference
///
// *****************************************************************************
oa::oaString 
getRefName(oagFunc::OccRef ref) {

    const oa::oaVerilogNS verilogNS;

    set<oagFunc::OccRef> connectedRefs;
    set<oa::oaOccBitNet*> connectedNets;
    oagFunc::OccGraph::getAllConnections(ref, connectedNets, connectedRefs, true, false, false, false, false);

    if (connectedNets.size() != 0) {
        oa::oaString netNameString;            
        (*connectedNets.begin())->getPathName(verilogNS, netNameString);
        return netNameString;
    }

    return oa::oaString("UNNAMED");
}


// *****************************************************************************
// printFunctionality()
//
/// \brief Prints some information about the functional description of a design.
///
/// \param design
/// \param nextStates print the full set of state bits
/// \param hierarchicalNames print net hierarchical path names
/// \return a name for the reference
///
// *****************************************************************************
void 
printFunctionality(oa::oaDesign *design, bool nextStates, 
        bool hierarchicalNames) {
    const oa::oaNativeNS nativeNS;
    const oa::oaVerilogNS verilogNS;

    std::cout << "---------------------------------------" << std::endl;

    // Print module name

    oa::oaModule *module = design->getTopModule();
    oa::oaString str;
    module->getName(nativeNS, str);
    std::cout << "MODULE: " << str << std::endl;
    
    // Get (or create) Manager
    
    oagFunc::Manager *manager = oagFunc::Manager::get(design);

    // Print statistics

    if (manager->isStructural()) {
        std::cout << "\tlocal module is entirely structural" << std::endl;
    }
    
    std::cout << "\tlocal AI | " << "and    " << manager->getLocalAndAIsize() 
        << std::endl;
    std::cout << "\t         | " << "state  " << manager->getLocalSeqNodeCount() 
        << std::endl;
    std::cout << "\t         | " << "total  " << manager->getLocalAIsize() 
        << std::endl;

    std::cout << "\thier  AI | " << "and    " << manager->getHierarchicalAndAIsize() 
        << std::endl;
    std::cout << "\t         | " << "state  " << manager->getHierarchicalSeqNodeCount() 
        << std::endl;
    std::cout << "\t         | " << "total  " << manager->getHierarchicalAIsize() 
        << std::endl;

    std::cout << "\tinsts    | " << "total  " << manager->getNumTotalInstances() 
        << std::endl;
    std::cout << "\t         | " << "local  " << module->getInsts(oacInstIterSingleBit).getCount()
        << std::endl;
    std::cout << "\t         | " << "leaf   " << manager->getNumLeafInstances() 
        << std::endl;
    
    // Print terminals
    
    oa::oaOccurrence *occurrence = design->getTopOccurrence();
    list<oagFunc::OccRef> inputs, outputs, states;
    oagFunc::OccGraph::getInputs(occurrence, inputs);
    oagFunc::OccGraph::getOutputs(occurrence, outputs);
    oagFunc::OccGraph::getStates(occurrence, states);
    std::cout << "\tbits     | " << "input  " << inputs.size() 
        << std::endl; 
    std::cout << "\t         | " << "output " << outputs.size()
        << std::endl;
    std::cout << "\t         | " << "state  " << states.size() 
        << std::endl;
    std::cout << "\tterminals..." << std::endl;
    for(list<oagFunc::OccRef>::iterator it = outputs.begin(); 
            it!=outputs.end(); it++) {
        oa::oaOccBitNet *net = oagFunc::OccGraph::getNetToAiConnection(*it);
        assert(net);
        net->getName(verilogNS, str); 
        std::cout << "\t\tterminal: " << str << " (output)" << std::endl;
	//change by donaldpatch, which seems not to work	
//std::cout << "\t\tterminal: " << str << " (output)" << (*it).ref << std::endl;
    }    
    for(list<oagFunc::OccRef>::iterator it = inputs.begin(); 
            it!=inputs.end(); it++) {
        oa::oaOccBitNet *net = oagFunc::OccGraph::getNetToAiConnection(*it);
        assert(net);    
        net->getName(verilogNS, str);
        std::cout << "\t\tterminal: " << str << " (input)" << std::endl;
	//change by donaldpatch, which seems not to work
//std::cout << "\t\tterminal: " << str << " (input)" << it->ref << std::endl;
    }
    
    // Print states

    if (nextStates) {
        std::cout << "\tstates..." << std::endl;
        for(list<oagFunc::OccRef>::iterator it = states.begin(); 
                it!=states.end(); it++) {
            oa::oaOccBitNet *net = oagFunc::OccGraph::getNetToAiConnection(*it);
            if (!net) {
                str = "anonymous";
            }
            if (hierarchicalNames) {
                net->getPathName(verilogNS, str);
            } else {
                net->getName(verilogNS, str);
            }
            std::cout << "\t\tstate bit: " << str << std::endl;
        }
    }
    //change by donaldpatch, which seems not to work
    //manager->getGraph()->print();

    std::cout << "---------------------------------------" << std::endl;
}

//------------------------------------------------------------------------------
