/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2007-01-25: ChangeLog started
*/

#include "oaDesignDB.h"
#include "oagFunc.h"
#include "oagFuncManager.h"
#include "oagFuncModGraph.h"
#include "oagFuncOccGraph.h"
#include <iostream>

// #define DEBUG

#include "oagFuncDebug.h"

using namespace std;

namespace oagFunc {

// *****************************************************************************
// copy()
//
/// \brief Copies a module, including its functional description.
///
/// Copies the nets, instances, terminals, and functional description.
///
/// Only portions of the graph that are in the transitive fan-in of the
/// source module's outputs are copied.  Any equivalent nodes and their
/// transitive fan-ins are also copied.  Only graph structures that
/// are unconnected to the outputs and aren't marked as equivalent
/// are ignored.
///
/// The copy can either be created as another module inside of the same
/// design as the source, or a new design can be created with the copy as
/// the top module.  If the makeNewDesign parameter is true, then a new
/// design will be created.
///
/// The name of the new design or module is the original with "_####" appended,
/// where #### is the lowest number that is not already used as a name.
///
/// \param source the original module
/// \param makeNewDesign if true, the copy will be placed into a new design
/// \return a copy
//
// *****************************************************************************
oa::oaModule*
copy(oa::oaModule *source, bool makeNewDesign) {
    assert(source);

    const oa::oaNativeNS      nativeNS;

    oa::oaString sourceString;
    source->getName(nativeNS, sourceString);
    DEBUG_PRINTLN("\tcopying module: " << sourceString)

    oa::oaModule *target;
    if (makeNewDesign) {

        // open design
        oa::oaScalarName    libName, viewName;
        source->getDesign()->getLibName(libName);
        source->getDesign()->getViewName(viewName);
        oa::oaScalarName    cellName;

        // find first unused cell name XXXXX_#
        int suffix = 0;
        do {
            suffix++;
            char suffixString[10];
            sprintf(suffixString, "%d",suffix);
            cellName = oa::oaScalarName(nativeNS, sourceString+"_"+suffixString);
        } while(oa::oaDesign::exists(libName, cellName, viewName));

        // create new design
        oa::oaDesign *newDesign = oa::oaDesign::open(libName, 
                                                     cellName, 
                                                     viewName, 
                                                     oa::oaViewType::get(oa::oacNetlist), 'w');

        target = oa::oaModule::create(newDesign);
        newDesign->setTopModule(target);
    } else {
    
        // find first unused module name XXXXX_#
        oa::oaScalarName    moduleName;
        int suffix = 0;
        do {
            suffix++;
            char suffixString[10];
            sprintf(suffixString, "%d",suffix);
            moduleName = oa::oaScalarName(nativeNS, sourceString+"_"+suffixString);
        } while(!oa::oaModule::find(source->getDesign(), moduleName));
    
       target = oa::oaModule::create(source->getDesign(), moduleName);
    }
    assert(target);

    // map original objects to copied/collapsed objects
    map<oa::oaModNet*, oa::oaModNet*> copiedNets;
    map<oa::oaModInst*, oa::oaModInst*> copiedInsts;
    map<ModRef, ModRef> copiedRefs;
    
    // 1. copy all insts
    DEBUG_PRINTLN("\t\tcopying insts")

    oa::oaModInst *subInst;
    oa::oaIter<oa::oaModInst> instIter(source->getInsts());
    while((subInst = instIter.getNext())) {
    
        // create corresponding copied inst
        oa::oaModInst *copiedInst;
        oa::oaString subInstString;
        subInst->getName(nativeNS, subInstString);
        if (subInst->isModDesignInst()) {
            copiedInst = oa::oaModScalarInst::create(target, 
                            static_cast<oa::oaModDesignInst *>(subInst)->getMaster(), 
                            oa::oaScalarName(nativeNS, subInstString));
        } else {
            copiedInst = oa::oaModModuleScalarInst::create(target, 
                            subInst->getMasterModule(), 
                            oa::oaScalarName(nativeNS, subInstString));
        }

        copiedInsts[subInst] = copiedInst;
    }    

    // 2. copy all nets
    DEBUG_PRINTLN("\t\tcopying nets")

    oa::oaModNet *subNet;
    oa::oaIter<oa::oaModNet> netIter(source->getNets());
    while((subNet = netIter.getNext())) {
    
        // create corresponding copied net
        oa::oaModNet *copiedNet;
        if (copiedNets.find(subNet) == copiedNets.end()) {
            // create a corresponding net
            oa::oaName subNetName;
            subNet->getName(subNetName);
            switch(subNetName.getType()) {
              case oa::oacScalarNameType: {
                oa::oaString subNetScalarString;
                subNetName.getScalar()->get(nativeNS, subNetScalarString);
                oa::oaString copiedNetString(subNetScalarString);
                if (subNet->isGlobal()) {
                    // the names of global nets should be copied exactly
                    copiedNetString = subNetScalarString;
                }
                
                copiedNet = oa::oaModScalarNet::create(target, 
                                oa::oaScalarName(nativeNS, copiedNetString));
                break;
                }
              case oa::oacVectorNameType: {
                oa::oaString subNetBaseString;
                subNetName.getVector()->getBaseName(nativeNS, subNetBaseString);
                oa::oaString copiedNetString(subNetBaseString);
                if (subNet->isGlobal()) {
                    // the names of global nets should be copied exactly
                    copiedNetString = subNetBaseString;
                }
                
                copiedNet = oa::oaModBusNet::create(target, 
                                oa::oaScalarName(nativeNS, copiedNetString),
                                subNetName.getVector()->getStart(),
                                subNetName.getVector()->getStop(),
                                subNetName.getVector()->getStep());                
                break;
                }
              case oa::oacVectorBitNameType: {
                oa::oaString subNetBaseString;
                subNetName.getVector()->getBaseName(nativeNS, subNetBaseString);
                oa::oaString copiedNetString(subNetBaseString);                
                if (subNet->isGlobal()) {
                    // the names of global nets should be copied exactly
                    copiedNetString = subNetBaseString;
                }
                
                copiedNet = oa::oaModBusNet::create(target, 
                                oa::oaScalarName(nativeNS, copiedNetString),
                                subNetName.getVectorBit()->getIndex(),
                                subNetName.getVectorBit()->getIndex(),
                                1);
                break;
                }
              default:
                cerr << "ERROR: Can not copy unsupported net type, that is, other than scalar or bus." << endl;
                QUIT_ON_INTERNAL_ERROR;
            }
            
            copiedNets[subNet] = copiedNet;
        } else {
            // this net was connected to a terminal and need not be copied
            copiedNet = copiedNets[subNet];
        }
    
        // also map implicit vector bits AND copy net equivalences
        oa::oaModBitNet *subBitNet;
        oa::oaIter<oa::oaModBitNet> bitNetIter(subNet->getSingleBitMembers());
        while((subBitNet = bitNetIter.getNext())) {
        
            // insert single bit into map
            oa::oaName subBitNetName;
            oa::oaModBitNet *copiedBitNet;
            if (subBitNetName.getType() == oa::oacVectorBitNameType) {
                oa::oaScalarName subNetBaseName;
                subBitNetName.getVectorBit()->getBaseName(subNetBaseName);
                copiedBitNet = oa::oaModBusNetBit::find(target, 
                                    subNetBaseName, 
                                    subBitNetName.getVectorBit()->getIndex());
                assert(copiedBitNet);
                copiedNets[copiedBitNet] = subBitNet;
            } else {
                // this net is a scalar or single bit, so we already have the mapping
                copiedBitNet = toBitNet(copiedNets[subBitNet]);
                assert(copiedBitNet);
            }
            
            // copy net equivalences, if any
            oa::oaModBitNet *equivSubBitNet;
            oa::oaIter<oa::oaModBitNet> equivSubBitNetIter(subBitNet->getEquivalentNets());
            while((equivSubBitNet = equivSubBitNetIter.getNext())) {
                // if we've already copied the equivalent bit net, mark the equivalence
                // on the copies.  if not, the equivalence will be marked when that net
                // is copied.
                if (copiedNets.find(equivSubBitNet) != copiedNets.end()) {
                    copiedBitNet->makeEquivalent(toBitNet(copiedNets[equivSubBitNet]));
                }
            }
        }
        
        // copy globality
        if (subNet->isGlobal()) {
            copiedNet->setGlobal(true);
        }
    }

    // 3. copy all terms
    DEBUG_PRINTLN("\t\tcopying terms")
    
    oa::oaModTerm *sourceTerm;
    oa::oaIter<oa::oaModTerm> termIter(source->getTerms());
    while((sourceTerm = termIter.getNext())) {
        oa::oaModNet *sourceNet = sourceTerm->getNet();
        assert(sourceNet);
        oa::oaModNet *targetNet = copiedNets[sourceNet];
        assert(targetNet);
        oa::oaName sourceName;
        sourceTerm->getName(sourceName);
        oa::oaModTerm *targetTerm = oa::oaModTerm::create(targetNet, 
                                        sourceName, sourceTerm->getTermType());
        targetTerm->setPosition(sourceTerm->getPosition());
    }
    
    // 4. copy all instterms
    DEBUG_PRINTLN("\t\tcopying instterms")
    // they will be connected to the copied nets upon creation

    oa::oaModInstTerm *instTerm;
    oa::oaIter<oa::oaModInstTerm> instTermIter2(source->getInstTerms());
    while((instTerm = instTermIter2.getNext())) {
    
        oa::oaModNet *copiedNet = NULL;
        if (instTerm->getNet() != NULL) {
            assert(copiedNets.find(instTerm->getNet()) != copiedNets.end());
            copiedNet = copiedNets[instTerm->getNet()];
        }
        
        assert(copiedInsts.find(instTerm->getInst()) != copiedInsts.end());
        oa::oaModInst *copiedInst = copiedInsts[instTerm->getInst()];
    
        // create corresponding copied instterm
        oa::oaModInstTerm *copiedInstTerm;
        if (instTerm->usesTermPosition()) {
            copiedInstTerm = oa::oaModInstTerm::create(copiedNet, copiedInst, instTerm->getTermPosition());
        } else {
            oa::oaName termName;
            instTerm->getTermName(termName);
            copiedInstTerm = oa::oaModInstTerm::create(copiedNet, copiedInst, termName);
        }
    }
    
    // 5. copy graph
    DEBUG_PRINTLN("\t\tcopying graph");
    
    list<ModRef> toBeVisited;
   
    oa::oaIter<oa::oaModNet> netIter2(source->getNets(oacNetIterSingleBit));
    while((subNet = netIter2.getNext())) {
        ModRef subRef;
        if (!ModGraph::isNull(subRef = ModGraph::getNetToAiConnection(toBitNet(subNet)))) {
            toBeVisited.push_back(subRef);
        }
    }
    
    // copy each node
    while(!toBeVisited.empty()) {
        ModRef subRef = toBeVisited.front();
        toBeVisited.pop_front();

        // has this node already been copied?
        if (copiedRefs.find(subRef) != copiedRefs.end()) {
            continue;        
        }

        // copy node and recurse on predecessors
        ModRef copiedRef, nextRef;
        switch(ModGraph::getNodeType(subRef)) {
            case oagAi::Node::TERMINAL:
                copiedRef = ModGraph::newTerminal(ModGraph::getNull(target));
                if (!ModGraph::isNull(nextRef = ModGraph::getTerminalDriver(subRef))) {
                    toBeVisited.push_back(nextRef);          
                }
                break;
            case oagAi::Node::SEQUENTIAL:
                copiedRef = ModGraph::newSequential(ModGraph::getNull(target));
                if (!ModGraph::isNull(nextRef = ModGraph::getNextState(subRef))) {
                    toBeVisited.push_back(nextRef);          
                }                
                break;
            case oagAi::Node::AND:
                copiedRef = ModGraph::newAnd(ModGraph::getNull(target), ModGraph::getNull(target));
                toBeVisited.push_back(ModGraph::getAndLeft(subRef));
                toBeVisited.push_back(ModGraph::getAndRight(subRef));
                break;
            default:
                break;
        }

        // recurse on equivalent structures
        list<ModRef> equivs;
        ModGraph::getEquivalents(subRef, equivs);
        for(list<ModRef>::iterator it = equivs.begin(); it!=equivs.end(); ++it) {
            if (!(subRef == *it)) { // no need to revisit current node
                toBeVisited.push_back(*it);
            }
        }

        copiedRefs[subRef] = copiedRef;        
    }
    
    // reconnect graph
    for(map<ModRef, ModRef>::iterator it=copiedRefs.begin(); it!=copiedRefs.end(); ++it) {
        ModRef nextRef;
        assert(ModGraph::getNodeType(it->first) == ModGraph::getNodeType(it->second));
        switch(ModGraph::getNodeType(it->first)) {
            case oagAi::Node::TERMINAL:
                if (!ModGraph::isNull(nextRef = ModGraph::getTerminalDriver(it->first))) {
                    ModGraph::setTerminalDriver(it->second, copiedRefs[nextRef]);
                }
                break;
            case oagAi::Node::SEQUENTIAL:
                if (!ModGraph::isNull(nextRef = ModGraph::getNextState(it->first))) {
                    ModGraph::setNextState(it->second, copiedRefs[nextRef]);
                }                
                break;
            case oagAi::Node::AND:
                ModGraph::setAndLeft(it->second, copiedRefs[ModGraph::getAndLeft(it->first)]);
                ModGraph::setAndRight(it->second, copiedRefs[ModGraph::getAndRight(it->first)]);
                break;
            default:
                break;
        }
    }
    
    // connect copied nets to copied graph
    netIter2.reset();
    while((subNet = netIter2.getNext())) {
        ModRef subRef;
        if (!ModGraph::isNull(subRef = ModGraph::getNetToAiConnection(toBitNet(subNet)))) {
            ModGraph::setNetToAiConnection(toBitNet(copiedNets[subNet]), copiedRefs[subRef]);
        }
    }
    
    DEBUG_PRINTLN("\t\t...done")
    return target;
}


// *****************************************************************************
// collapseOneInstanceOneLevel()
//
/// \brief Collapses the contents of an instance into the enclosing module.
///
/// Further levels of hierarchy inside of the instance will remain intact.
///
/// \param inst the instance to collapse
//
// *****************************************************************************
void
collapseOneInstanceOneLevel(oa::oaModInst *inst) {
    assert(inst);
    oa::oaString instString;
    inst->getName(oa::oaVerilogNS(), instString);
    oa::oaModule *source = inst->getMasterModule();
    assert(source);
    oa::oaModule *target = inst->getModule();
    assert(target);

    // ---------- NEEDS TO BE DEBUGGED ------------
    QUIT_ON_INTERNAL_ERROR;

    // map original objects to copied/collapsed objects
    map<oa::oaModNet*, oa::oaModNet*> copiedNets;
    map<oa::oaModInst*, oa::oaModInst*> copiedInsts;

    // 1. copy all insts
 
    oa::oaModInst *subInst;
    oa::oaIter<oa::oaModInst> instIter(source->getInsts());
    while((subInst = instIter.getNext())) {
    
        // create corresponding copied inst
        oa::oaModInst *copiedInst;
        oa::oaString subInstString;
        subInst->getName(oa::oaVerilogNS(), subInstString);
        if (subInst->isModDesignInst()) {
            copiedInst = oa::oaModScalarInst::create(target, 
                            static_cast<oa::oaModDesignInst *>(subInst)->getMaster(), 
                            oa::oaScalarName(oa::oaVerilogNS(), instString+"_"+subInstString));
        } else {
            if (!inst->isModModuleInst()) {
                // we're attempting to collapse a ModModInst of a DesignInst
                // but a submodule can't be instantiated from another design
                cerr << "ERROR: Can not copy a ModModuleInst into another design" << endl;
                QUIT_ON_ERROR;
            }
            copiedInst = oa::oaModModuleScalarInst::create(target, 
                            subInst->getMasterModule(), 
                            oa::oaScalarName(oa::oaVerilogNS(), instString+"_"+subInstString));
        }

        copiedInsts[subInst] = copiedInst;
    }    

    // 2. remove terminals and extend nets into collapsed instance
    
    oa::oaModInstTerm *instTerm;
    oa::oaIter<oa::oaModInstTerm> instTermIter(inst->getInstTerms());
    while((instTerm = instTermIter.getNext())) {
        oa::oaModNet *existingNet = instTerm->getNet();
        if (!existingNet) {
            // this instance terminal is not connected
            continue;
        }
        oa::oaModTerm *subTerm = instTerm->getTerm();
        assert(subTerm);
        oa::oaModNet *subNet = subTerm->getNet();
        assert(subNet);
        copiedNets[subNet] = existingNet;
    }

    // 3. copy all other nets

    oa::oaModNet *subNet;
    oa::oaIter<oa::oaModNet> netIter(source->getNets());
    while((subNet = netIter.getNext())) {
    
        // create corresponding copied net
        oa::oaModNet *copiedNet;
        if (copiedNets.find(subNet) == copiedNets.end()) {
            // create a corresponding net
            oa::oaName subNetName;
            subNet->getName(subNetName);
            switch(subNetName.getType()) {
              case oa::oacScalarNameType: {
                oa::oaString subNetScalarString;
                subNetName.getScalar()->get(oa::oaVerilogNS(), subNetScalarString);
                oa::oaString copiedNetString(instString+"_"+subNetScalarString);
                if (subNet->isGlobal()) {
                    // the names of global nets should be copied exactly
                    copiedNetString = subNetScalarString;
                }
                
                copiedNet = oa::oaModScalarNet::create(target, 
                                oa::oaScalarName(oa::oaVerilogNS(), copiedNetString));
                break;
                }
              case oa::oacVectorNameType: {
                oa::oaString subNetBaseString;
                subNetName.getVector()->getBaseName(oa::oaVerilogNS(), subNetBaseString);
                oa::oaString copiedNetString(instString+"_"+subNetBaseString);
                if (subNet->isGlobal()) {
                    // the names of global nets should be copied exactly
                    copiedNetString = subNetBaseString;
                }
                
                copiedNet = oa::oaModBusNet::create(target, 
                                oa::oaScalarName(oa::oaVerilogNS(), copiedNetString),
                                subNetName.getVector()->getStart(),
                                subNetName.getVector()->getStop(),
                                subNetName.getVector()->getStep());                
                break;
                }
              case oa::oacVectorBitNameType: {
                oa::oaString subNetBaseString;
                subNetName.getVector()->getBaseName(oa::oaVerilogNS(), subNetBaseString);
                oa::oaString copiedNetString(instString+"_"+subNetBaseString);                
                if (subNet->isGlobal()) {
                    // the names of global nets should be copied exactly
                    copiedNetString = subNetBaseString;
                }
                
                copiedNet = oa::oaModBusNet::create(target, 
                                oa::oaScalarName(oa::oaVerilogNS(), copiedNetString),
                                subNetName.getVectorBit()->getIndex(),
                                subNetName.getVectorBit()->getIndex(),
                                1);
                break;
                }
              default:
                cerr << "ERROR: Can not copy unsupported net type, that is, other than scalar or bus." << endl;
                QUIT_ON_INTERNAL_ERROR;
            }
            
            copiedNets[subNet] = copiedNet;
        } else {
            // this net was connected to a terminal and need not be copied
            copiedNet = copiedNets[subNet];
        }
    
        // also map implicit vector bits AND copy net equivalences
        oa::oaModBitNet *subBitNet;
        oa::oaIter<oa::oaModBitNet> bitNetIter(subNet->getSingleBitMembers());
        while((subBitNet = bitNetIter.getNext())) {
        
            // insert single bit into map
            oa::oaName subBitNetName;
            oa::oaModBitNet *copiedBitNet;
            if (subBitNetName.getType() == oa::oacVectorBitNameType) {
                oa::oaScalarName subNetBaseName;
                subBitNetName.getVectorBit()->getBaseName(subNetBaseName);
                copiedBitNet = oa::oaModBusNetBit::find(target, 
                                    subNetBaseName, 
                                    subBitNetName.getVectorBit()->getIndex());
                assert(copiedBitNet);
                copiedNets[copiedBitNet] = subBitNet;
            } else {
                // this net is a scalar or single bit, so we already have the mapping
                copiedBitNet = toBitNet(copiedNets[subBitNet]);
                assert(copiedBitNet);
            }
            
            // copy net equivalences, if any
            oa::oaModBitNet *equivSubBitNet;
            oa::oaIter<oa::oaModBitNet> equivSubBitNetIter(subBitNet->getEquivalentNets());
            while((equivSubBitNet = equivSubBitNetIter.getNext())) {
                // if we've already copied the equivalent bit net, mark the equivalence
                // on the copies.  if not, the equivalence will be marked when that net
                // is copied.
                if (copiedNets.find(equivSubBitNet) != copiedNets.end()) {
                    copiedBitNet->makeEquivalent(toBitNet(copiedNets[equivSubBitNet]));
                }
            }
        }
        
        // copy globality
        if (subNet->isGlobal()) {
            copiedNet->setGlobal(true);
        }
    }
    
    // 4. copy all instterms
    // they will be connected to the copied nets upon creation

    oa::oaIter<oa::oaModInstTerm> instTermIter2(source->getInstTerms());
    while((instTerm = instTermIter2.getNext())) {
    
        oa::oaModNet *copiedNet = NULL;
        if (instTerm->getNet() != NULL) {
            assert(copiedNets.find(instTerm->getNet()) != copiedNets.end());
            copiedNet = copiedNets[instTerm->getNet()];
        }
        
        assert(copiedInsts.find(instTerm->getInst()) != copiedInsts.end());
        oa::oaModInst *copiedInst = copiedInsts[instTerm->getInst()];
    
        // create corresponding copied instterm
        oa::oaModInstTerm *copiedInstTerm;
        if (instTerm->usesTermPosition()) {
            copiedInstTerm = oa::oaModInstTerm::create(copiedNet, copiedInst, instTerm->getTermPosition());
        } else {
            oa::oaName termName;
            instTerm->getTermName(termName);
            copiedInstTerm = oa::oaModInstTerm::create(copiedNet, copiedInst, termName);
        }
    }
    
    // 5. copy graph
    
    map<ModRef, ModRef> copiedRefs;
    list<ModRef> toBeVisited;
   
    oa::oaIter<oa::oaModNet> netIter2(source->getNets(oacNetIterSingleBit));
    while((subNet = netIter2.getNext())) {
        ModRef subRef;
        if (!ModGraph::isNull(subRef = ModGraph::getNetToAiConnection(toBitNet(subNet)))) {
            toBeVisited.push_back(subRef);
        }
    }
    
    // copy each node
    while(!toBeVisited.empty()) {
        ModRef subRef = toBeVisited.front();
        toBeVisited.pop_front();

        // has this node already been copied?
        if (copiedRefs.find(subRef) != copiedRefs.end()) {
            continue;        
        }

        // copy node and recurse on predecessors
        ModRef copiedRef, nextRef;
        switch(ModGraph::getNodeType(subRef)) {
            case oagAi::Node::TERMINAL:
                copiedRef = ModGraph::newTerminal(ModGraph::getNull(target));
                if (!ModGraph::isNull(nextRef = ModGraph::getTerminalDriver(subRef))) {
                    toBeVisited.push_back(nextRef);          
                }
                break;
            case oagAi::Node::SEQUENTIAL:
                copiedRef = ModGraph::newSequential(ModGraph::getNull(target));
                if (!ModGraph::isNull(nextRef = ModGraph::getNextState(subRef))) {
                    toBeVisited.push_back(nextRef);          
                }                
                break;
            case oagAi::Node::AND:
                copiedRef = ModGraph::newAnd(ModGraph::getNull(target), ModGraph::getNull(target));
                toBeVisited.push_back(ModGraph::getAndLeft(subRef));
                toBeVisited.push_back(ModGraph::getAndRight(subRef));
                break;
            default:
                break;
        }

        // recurse on equivalent structures
        list<ModRef> equivs;
        ModGraph::getEquivalents(subRef, equivs);
        for(list<ModRef>::iterator it = equivs.begin(); it!=equivs.end(); ++it) {
            if (!(subRef == *it)) { // no need to revisit current node
                toBeVisited.push_back(*it);
            }
        }

        copiedRefs[subRef] = copiedRef;        
    }
    
    // reconnect graph
    for(map<ModRef, ModRef>::iterator it=copiedRefs.begin(); it!=copiedRefs.end(); ++it) {
        ModRef nextRef;
        switch(ModGraph::getNodeType(it->first)) {
            case oagAi::Node::TERMINAL:
                if (!ModGraph::isNull(nextRef = ModGraph::getTerminalDriver(it->first))) {
                    ModGraph::setTerminalDriver(it->second, copiedRefs[nextRef]);
                }
                break;
            case oagAi::Node::SEQUENTIAL:
                if (!ModGraph::isNull(nextRef = ModGraph::getNextState(it->first))) {
                    ModGraph::setNextState(it->second, copiedRefs[nextRef]);
                }                
                break;
            case oagAi::Node::AND:
                ModGraph::setAndLeft(it->second, copiedRefs[ModGraph::getAndLeft(it->first)]);
                ModGraph::setAndRight(it->second, copiedRefs[ModGraph::getAndRight(it->first)]);
                break;
            default:
                break;
        }
    }
    
    // connect copied nets to copied graph
    netIter2.reset();
    while((subNet = netIter2.getNext())) {
        ModRef subRef;
        if (!ModGraph::isNull(subRef = ModGraph::getNetToAiConnection(toBitNet(subNet)))) {
            ModGraph::setNetToAiConnection(toBitNet(copiedNets[subNet]), copiedRefs[subRef]);
        }
    }
    
    // 6. destroy instance and instterms
    
    oa::oaIter<oa::oaModInstTerm> instTermIter3(inst->getInstTerms());
    while((instTerm = instTermIter.getNext())) {
        instTerm->destroy();
    }
    inst->destroy();
    
}


// *****************************************************************************
// partialCopyNode()
//
/// \brief Partially copies a node into the destModule.
///
/// The node is created and only the type is set.  The inputs are all null.
///
/// \param original
/// \param destModule
//
// *****************************************************************************
ModRef
partialCopyNode(OccRef original, oa::oaModule *destModule) {
    switch(OccGraph::getNodeType(original)) {
      case oagAi::Node::AND:
        return ModGraph::newAnd(ModGraph::getNull(destModule), 
                                ModGraph::getNull(destModule));
      case oagAi::Node::CONSTANT0:
        return ModGraph::constantZero(destModule);  
      case oagAi::Node::SEQUENTIAL:
        return ModGraph::newSequential(ModGraph::getNull(destModule));
      case oagAi::Node::TERMINAL: 
        return ModGraph::newTerminal(ModGraph::getNull(destModule));
      default:
        cerr << "ERROR: Broken AI node: " << original.ref << endl;
        QUIT_ON_INTERNAL_ERROR;
    }
}


// *****************************************************************************
// partialCopyNode()
//
/// \brief Partially copies a node into the destModule.
///
/// The node is created and only the type is set.  The inputs are all null.
///
/// \param original
/// \param destModule
//
// *****************************************************************************
ModRef
partialCopyNode(ModRef original, oa::oaModule *destModule) {
    switch(ModGraph::getNodeType(original)) {
      case oagAi::Node::AND:
        return ModGraph::newAnd(ModGraph::getNull(destModule), 
                                ModGraph::getNull(destModule));
      case oagAi::Node::CONSTANT0:
        return ModGraph::constantZero(destModule);  
      case oagAi::Node::SEQUENTIAL:
        return ModGraph::newSequential(ModGraph::getNull(destModule));
      case oagAi::Node::TERMINAL: 
        return ModGraph::newTerminal(ModGraph::getNull(destModule));
      default:
        cerr << "ERROR: Broken AI node: " << original.ref << endl;
        QUIT_ON_INTERNAL_ERROR;
    }
}


// *****************************************************************************
// collapse_otherTerm()
//
/// \brief Finishes other TERMINAL nodes into the destModule.
///
// *****************************************************************************
void
collapse_otherTerm(ModRef collapsedDriver,
                   std::set<OccRef> &connectedRefs,
                   map<oagFunc::OccRef, oagFunc::ModRef> &alreadyCollapsed,
                   oa::oaModule *destModule) {

  // early complete all other connected terminals
  for(std::set<OccRef>::iterator connectedIter = connectedRefs.begin();
      connectedIter != connectedRefs.end(); ++connectedIter) {
    OccRef connectedRef = *connectedIter;
    OccRef nonInvConnectedRef = OccGraph::getNonInverted(connectedRef);
      
    if (OccGraph::getNodeType(connectedRef) != oagAi::Node::TERMINAL) {
      continue;
    }
    
    if (alreadyCollapsed.find(nonInvConnectedRef) == alreadyCollapsed.end()) {
      alreadyCollapsed[nonInvConnectedRef] = partialCopyNode(nonInvConnectedRef, destModule);
    }
    ModRef collapsedOther = alreadyCollapsed[nonInvConnectedRef];

    if (OccGraph::isInverted(connectedRef)) {
      ModGraph::setTerminalDriver(collapsedOther, ModGraph::notOf(collapsedDriver));
    } else {
      ModGraph::setTerminalDriver(collapsedOther, collapsedDriver);
    }
  }
}


// *****************************************************************************
// collapse_finishTerm()
//
/// \brief Finishes copying a TERMINAL node into the destModule.
///
// *****************************************************************************
void
collapse_finishTerm(OccRef ref, list<oagFunc::OccRef> &toBeCompleted,
                    map<oagFunc::OccRef, oagFunc::ModRef> &alreadyCollapsed,
                    map<oagFunc::OccRef, oagFunc::ModRef> &instInputs,
                    oa::oaModule *destModule) {
  
  ModRef collapsedParent = alreadyCollapsed[ref];

  if (!ModGraph::isNull(ModGraph::getTerminalDriver(collapsedParent))) {
    return;
  }
            
  oagFunc::ModRef collapsedDriver;
  oagFunc::OccRef uncollapsedDriver;
  
  std::set<oagFunc::OccRef> connectedRefs;
  std::set<oa::oaOccBitNet*> connectedNets;
  OccGraph::getAllConnections(ref, connectedNets, connectedRefs,
                              false, true, true, true, false);
        
  DEBUG_PRINTMORE(" (nets " << connectedNets.size() << " nodes " << connectedRefs.size() << ") ");

  // for every connected net... look for a tied zero or one
  for(std::set<oa::oaOccBitNet*>::iterator connectedIter = connectedNets.begin();
      connectedIter != connectedNets.end() && ModGraph::isNull(collapsedDriver);
      ++connectedIter) {
    oa::oaString netPathString, netNameString;
    (*connectedIter)->getPathName(oa::oaVerilogNS(), netPathString);
    DEBUG_PRINTMORE(netPathString << " ");
    (*connectedIter)->getName(oa::oaVerilogNS(), netNameString);
    if (netNameString == "tie0") {
      DEBUG_PRINTMORE(" driven by constant" << endl);
      collapsedDriver = ModGraph::constantZero(destModule);
    } else if (netNameString == "tie1") {
      DEBUG_PRINTMORE(" driven by constant" << endl);
      collapsedDriver = ModGraph::constantOne(destModule);                
    }
  }

  // for every connected ref... look for a driver
  for(std::set<OccRef>::iterator connectedIter = connectedRefs.begin();
      connectedIter != connectedRefs.end(); 
      ++connectedIter) {
    OccRef connectedRef = *connectedIter;
    OccRef nonInvConnectedRef = OccGraph::getNonInverted(connectedRef);
    
    // driven by an input
    if (instInputs.find(nonInvConnectedRef) != instInputs.end()) {
      DEBUG_PRINTMORE(" driven by input" << endl);
      uncollapsedDriver = connectedRef;
      // there should only be one driver
      if (!ModGraph::isNull(collapsedDriver)) {
        // (except for constants)
        if (ModGraph::getNonInverted(collapsedDriver) != 
            ModGraph::constantZero(collapsedDriver.module) ||
            ModGraph::getNonInverted(instInputs[nonInvConnectedRef]) != 
            ModGraph::constantZero(collapsedDriver.module)) {
          cout << "WARNING: Multiple drivers of ";
          ref.print(true);
        }
      }
      collapsedDriver = instInputs[nonInvConnectedRef];
      if (OccGraph::isInverted(connectedRef)) {
        collapsedDriver = ModGraph::notOf(collapsedDriver);
      }
    }
      
    // driven elsewhere
    if (OccGraph::getNodeType(connectedRef) != oagAi::Node::TERMINAL) {
      DEBUG_PRINTMORE(" driven in graph by ");
#if defined(DEBUG)
      connectedRef.print(true);
#endif
      
      if (alreadyCollapsed.find(nonInvConnectedRef) == alreadyCollapsed.end()) {
        alreadyCollapsed[nonInvConnectedRef] = partialCopyNode(nonInvConnectedRef, destModule);
        toBeCompleted.push_front(nonInvConnectedRef);
      }
      uncollapsedDriver = connectedRef;
      // there should only be one driver
      if (!ModGraph::isNull(collapsedDriver)) {
        // (except for constants)
        if (ModGraph::getNonInverted(collapsedDriver) != 
            ModGraph::constantZero(collapsedDriver.module) ||
            ModGraph::getNonInverted(alreadyCollapsed[nonInvConnectedRef]) != 
            ModGraph::constantZero(collapsedDriver.module)) {
          cout << "WARNING: Multiple drivers of ";
          ref.printName();
          cout << " = ";
          collapsedDriver.printName();
          cout << " and ";
          connectedRef.printName();
        }
      }
      collapsedDriver = alreadyCollapsed[nonInvConnectedRef];
      if (OccGraph::isInverted(connectedRef)) {
        collapsedDriver = ModGraph::notOf(collapsedDriver);
      }
    }                    
  }

  if (ModGraph::isNull(collapsedDriver)) {
    cerr << "WARNING: Floating AI node: ";
    ref.print(true); 
    collapsedDriver = ModGraph::getNull(destModule);
  }

  /*
  oa::oaString n;
  oa::oaOccBitNet *o = OccGraph::getNetToAiConnection(ref);
  o->getName(oa::oaVerilogNS(), n);
  cout << collapsedParent.ref << " " << collapsedDriver.ref << " " << n << endl;
  */

  ModGraph::setTerminalDriver(collapsedParent, collapsedDriver);
  collapse_otherTerm(collapsedDriver, connectedRefs, alreadyCollapsed, destModule);
}


// *****************************************************************************
// collapse_finishSeq()
//
/// \brief Finishes copying a SEQUENTIAL node into the destModule.
///
// *****************************************************************************
void
collapse_finishSeq(OccRef ref, list<oagFunc::OccRef> &toBeCompleted,
                   map<oagFunc::OccRef, oagFunc::ModRef> &alreadyCollapsed,
                   oa::oaModule *destModule) {

  ModRef collapsedParent = alreadyCollapsed[ref];
  OccRef original = OccGraph::getNonInverted(OccGraph::getNextState(ref));

  if (alreadyCollapsed.find(original) == alreadyCollapsed.end()) {
    alreadyCollapsed[original] = partialCopyNode(original, destModule);
    toBeCompleted.push_front(original);
  }
  ModRef collapsed = alreadyCollapsed[original];
  if (OccGraph::isInverted(OccGraph::getNextState(ref))) {
    ModGraph::setNextState(collapsedParent, ModGraph::notOf(collapsed));
  } else {
    ModGraph::setNextState(collapsedParent, collapsed);
  }
  
  // collapse seq signals
  oagAi::Node::SequentialData *originalData = OccGraph::getSequentialData(ref);
  oagAi::Node::SequentialData *collapsedData = ModGraph::getSequentialData(collapsedParent);
  collapsedData->abstractionLevel = originalData->abstractionLevel;
  
  switch(originalData->abstractionLevel) {
  case oagAi::Node::SequentialData::BLACK_BOX:
    
    DEBUG_PRINTMORE("\tBLACK_BOX " << " signals=" << originalData->signals.size() << endl)
      for(map<string, oagAi::Ref>::iterator signalIter = originalData->signals.begin();
          signalIter != originalData->signals.end(); ++signalIter) {
        DEBUG_PRINTLN("\t" << signalIter->first)
          
          original = OccGraph::getNonInverted(OccRef(signalIter->second, ref.occurrence));
        if (alreadyCollapsed.find(original) == alreadyCollapsed.end()) {
          alreadyCollapsed[original] = partialCopyNode(original, destModule);
          toBeCompleted.push_front(original);
        }
        collapsed = alreadyCollapsed[original];
        if (OccGraph::isInverted(OccRef(signalIter->second, ref.occurrence))) {
          collapsedData->signals[signalIter->first] = ModGraph::notOf(collapsed).ref;
        } else {
          collapsedData->signals[signalIter->first] = collapsed.ref;
        }
        
      }
    break;
    
  case oagAi::Node::SequentialData::PRIMITIVE:
    
    DEBUG_PRINTMORE("\tPRIMITIVE" << endl)              
      original = OccGraph::getNonInverted(OccRef(originalData->gate, ref.occurrence));
    if (alreadyCollapsed.find(original) == alreadyCollapsed.end()) {
      alreadyCollapsed[original] = partialCopyNode(original, destModule);
     toBeCompleted.push_front(original);
    }
    collapsed = alreadyCollapsed[original];
    if (OccGraph::isInverted(OccRef(originalData->gate, ref.occurrence))) {
      collapsedData->gate = ModGraph::notOf(collapsed).ref;
    } else {
      collapsedData->gate = collapsed.ref;
    }
    break;
    
  case oagAi::Node::SequentialData::FEEDBACK:
    DEBUG_PRINTMORE("\tFEEDBACK" << endl)              
      break;
  default:
    QUIT_ON_INTERNAL_ERROR;
  }
  
}


// *****************************************************************************
// collapseOneInstanceAllLevels()
//
/// \brief Collapses the entire hierarchy in one instance into the enclosing module.
///
/// Moves the functionality implemented inside the entire hierarchy of an instance
/// into the module in which it occurs.  The instance will be removed from the module.
///
/// The preserveNets and preserveStateNets features are currently unimplemented.
///
/// \param inst the instance to collapse
/// \param preserveStateNets UNIMPLEMENTED.  preserve state nets
/// \param preserveAllNets UNIMPLEMENTED.  preserve all nets
//
// *****************************************************************************
void
collapseOneInstanceAllLevels(oa::oaModInst *inst,
                             bool preserveStateNets, bool preserveAllNets) {
    assert(inst);
    oa::oaString instString;
    inst->getName(oa::oaVerilogNS(), instString);
    DEBUG_PRINT("Flattening inst " << instString << " all levels ...")

    // *destModule == module into which graph nodes are collapsed
    oa::oaModule *destModule = inst->getModule();
    assert(destModule);
    // *destDesign == design whose top module the graph nodes are collapsed
    oa::oaDesign *destDesign = destModule->getDesign();
    assert(destDesign);
    if (destDesign->getTopModule() != destModule) {
        cerr << "ERROR: Destination module must be the top module of a design" << endl;
        QUIT_ON_ERROR;
    }
    // *sourceDesign == design from which nodes are collapsed
    oa::oaDesign *sourceDesign = inst->getMasterModule()->getDesign();
    assert(sourceDesign);
    if (sourceDesign->getTopModule() != inst->getMasterModule()) {
        cerr << "ERROR: Source instance must be the top module of a design" << endl;
        QUIT_ON_ERROR;
    }
    // *sourceOcc == occurrence from which nodes are collapsed
    oa::oaOccurrence *sourceOcc = sourceDesign->getTopOccurrence();
    assert(sourceOcc);
    
    map<oagFunc::OccRef, oagFunc::ModRef> alreadyCollapsed;
    list<oagFunc::OccRef> toBeCompleted;
    map<oagFunc::OccRef, oagFunc::ModRef> instInputs;
    OccRef original;
    ModRef collapsed, collapsedParent;

    // prepare for the traversal
    // every output term is a starting point, every input term is a stopping point
    oa::oaOccTerm *term;
    oa::oaIter<oa::oaOccTerm> termIter(sourceOcc->getTerms(oacTermIterSingleBit));
    while((term = termIter.getNext())) {
#if defined(DEBUG)
        // get name
        oa::oaString termString;
        term->getName(oa::oaVerilogNS(), termString);
#endif        

        // seed outputs as starting points
        if (term->getTermType() == oa::oacOutputTermType) {
            DEBUG_PRINTLN("output " << termString)
            oa::oaOccBitNet *net = toBitNet(term->getNet());
            assert(net);

            // partially copy node, add to toBeCompleted list
            OccRef sourceRef = OccGraph::getNetToAiConnection(net);
            alreadyCollapsed[sourceRef] = partialCopyNode(sourceRef, destModule);
            toBeCompleted.push_back(sourceRef);
            
            #ifdef DEBUG
            sourceRef.print(true);
            #endif
        } else {
            DEBUG_PRINTLN("input " << termString)
            oa::oaOccBitNet *net = toBitNet(term->getNet());
            assert(net);

            // partially copy node
            OccRef sourceRef = OccGraph::getNetToAiConnection(net);
            alreadyCollapsed[sourceRef] = partialCopyNode(sourceRef, destModule);
            instInputs[sourceRef] = alreadyCollapsed[sourceRef];
            
            #ifdef DEBUG
            sourceRef.print();
            #endif
        }
    }

    // complete the copy of all nodes, partially copy new ones as discovered
    while(!toBeCompleted.empty()) {
        oagFunc::OccRef ref = toBeCompleted.front();
        toBeCompleted.pop_front();

        // this node should have already been partially copied
        assert(alreadyCollapsed.find(ref) != alreadyCollapsed.end());
        collapsedParent = alreadyCollapsed[ref];
        assert(ModGraph::getNodeType(collapsedParent) == OccGraph::getNodeType(ref));

        #ifdef DEBUG
        ref.print(true);
        #endif

        // partially copy inputs, and point to new nodes
        switch(OccGraph::getNodeType(ref)) {
          case oagAi::Node::AND: {
            
            original = OccGraph::getNonInverted(OccGraph::getAndLeft(ref));
            if (alreadyCollapsed.find(original) == alreadyCollapsed.end()) {
                alreadyCollapsed[original] = partialCopyNode(original, destModule);
                toBeCompleted.push_front(original);
            }
            collapsed = alreadyCollapsed[original];
            if (OccGraph::isInverted(OccGraph::getAndLeft(ref))) {
                ModGraph::setAndLeft(collapsedParent, ModGraph::notOf(collapsed));
            } else {
                ModGraph::setAndLeft(collapsedParent, collapsed);
            }
            
            original = OccGraph::getNonInverted(OccGraph::getAndRight(ref));
            if (alreadyCollapsed.find(original) == alreadyCollapsed.end()) {
                alreadyCollapsed[original] = partialCopyNode(original, destModule);
                toBeCompleted.push_front(original);
            }
            collapsed = alreadyCollapsed[original];
            if (OccGraph::isInverted(OccGraph::getAndRight(ref))) {
                ModGraph::setAndRight(collapsedParent, ModGraph::notOf(collapsed));
            } else {
                ModGraph::setAndRight(collapsedParent, collapsed);
            }
            
            break;
            }
          case oagAi::Node::CONSTANT0:
            break;  
          case oagAi::Node::SEQUENTIAL:
            collapse_finishSeq(ref, toBeCompleted, alreadyCollapsed, destModule);
            break;
          case oagAi::Node::TERMINAL: {
            collapse_finishTerm(ref, toBeCompleted, alreadyCollapsed, instInputs, destModule);
            break;
                       
            break;
          }
          default:
            cerr << "ERROR: Broken AI node: ";
            ref.print(true);
            QUIT_ON_INTERNAL_ERROR;
        }
    }
    
    // stitch new graph into old context
    oa::oaModInstTerm *instTerm;
    oa::oaIter<oa::oaModInstTerm> instTermIter(inst->getInstTerms(oacInstTermIterSingleBit));
    while((instTerm = instTermIter.getNext())) {
        oa::oaModTerm *term = instTerm->getTerm();
        assert(term);
        // find the equivalent term in the source occurrence
        oa::oaName termName;
        term->getName(termName);
        oa::oaOccTerm *sourceTerm = oa::oaOccTerm::find(sourceOcc, termName);
        assert(sourceTerm);

        if (term->getTermType() == oa::oacOutputTermType) {
            ModRef destRef = ModGraph::prepareNetToAiConnection(toBitNet(instTerm->getNet()));
            OccRef sourceRef = OccGraph::getNetToAiConnection(toBitNet(sourceTerm->getNet()));
            ModGraph::setTerminalDriver(destRef, alreadyCollapsed[sourceRef]);
        } else {
            ModRef destRef = ModGraph::prepareNetToAiConnection(toBitNet(instTerm->getNet()));
            OccRef sourceRef = OccGraph::getNetToAiConnection(toBitNet(sourceTerm->getNet()));
            ModGraph::setTerminalDriver(alreadyCollapsed[sourceRef], destRef);
        }
    }
    inst->destroy();
}


// *****************************************************************************
// collapseAllInstancesOneLevel()
//
/// \brief Collapses one level of hierarchy inside of a module.
///
/// Moves the functionality implemented inside the entire hierarchy below a module
/// into the module itself.  All instances will be removed from the module.
///
/// The preserveNets and preserveStateNets features are currently unimplemented.
///
/// \param module the modules whose instances are to be collapsed
//
// *****************************************************************************
void
collapseAllInstancesOneLevel(oa::oaModule *module) {
    assert(module);
    
    oa::oaModInst *inst;
    oa::oaIter<oa::oaModInst> instIter(module->getInsts());
    while((inst = instIter.getNext())) {
        collapseOneInstanceOneLevel(inst);
    }
}


// *****************************************************************************
// collapseAllInstancesAllLevels()
//
/// \brief Collapses all hierarchy inside of a module.
///
/// The resulting module will not contain any instances.
///
/// The preserveNets and preserveStateNets features are currently unimplemented.
///
/// \param module the modules whose instances are to be collapsed
/// \param preserveStateNets UNIMPLEMENTED.  preserve state nets
/// \param preserveAllNets UNIMPLEMENTED.  preserve all nets
//
// *****************************************************************************
void
collapseAllInstancesAllLevels(oa::oaModule *module,
                              bool preserveStateNets, bool preserveAllNets) {

    assert(module);

    oa::oaModInst *inst;
    oa::oaIter<oa::oaModInst> instIter(module->getInsts());
    while((inst = instIter.getNext())) {
        collapseOneInstanceAllLevels(inst, preserveStateNets, preserveAllNets);
    }
}


// *****************************************************************************
// uniquify()
//
/// \brief Uniquifies an occurrence, creating new modules if necessary.
///
/// This function ensures that this is the only occurrence of this module, 
/// creating a new module variant if necessary. The ancestors of this 
/// occurrence will be uniquified as well, stopping when reaching an ancestor 
/// that is unique.
///
/// The newly uniquified occurrence is returned.  As its module is guaranteed
/// to be unique to this occurrence, the functionality of that module can be
/// modified without affecting other portions of the design.
/// 
/// The uniquified copies will be created using the oagFunc::copy() routine
/// and their names will be generated according to its rules.
///
/// NOTE: Currently, ModModInsts can not be uniquified.
///
/// \param occurrence the occurrence to uniquify
/// \return the uniquified occurrence
//
// *****************************************************************************
oa::oaOccurrence *
uniquify(oa::oaOccurrence *occurrence) {
    assert(occurrence);
    oa::oaOccurrence *topOcc = occurrence->getTopOccurrence();    

    // construct the path name of this occurrence
    oa::oaSimpleName pathName;
    occurrence->getOccInst()->getPathName(pathName);
    oa::oaString pathNameString;
    pathName.get(pathNameString);
    DEBUG_PRINTLN("\tpath name=" << pathNameString);

    // every module that contains this instance and is itself
    // instantiated more than once needs to be uniquified.
    // the uniquification proceeds from the top down.

    unsigned int pathNameMember = 0;
    oa::oaOccurrence *currentOcc = topOcc;

    while(true) {
        oa::oaString moduleNameString;
        currentOcc->getModule()->getName(oa::oaVerilogNS(), moduleNameString);
        DEBUG_PRINTLN("\t\tinstances of module " << moduleNameString << ": " 
                      << currentOcc->getModule()->getOccurrences(topOcc).getCount())
        
        if (currentOcc->getModule()->getOccurrences(topOcc).getCount() > 1) {
        
            // copy this portion of the tree
            oa::oaModule *uniquifiedModule = copy(currentOcc->getModule(), true);
            DEBUG_PRINTLN("\t\t\tinitial insts of uniquified mod: " 
                          << uniquifiedModule->getOccurrences(topOcc).getCount())

            // update instance to point to copied subtree
            oa::oaModInst *modInst = currentOcc->getOccInst()->getModInst();
            /*
            DEBUG_PRINTLN("count: " << modInst->getOccInsts(topOcc).getCount() 
                 << " occinst: " 
                 << oa::oaIter<oa::oaOccInst>(modInst->getOccInsts(topOcc)).getNext()
                 << " occ:  " 
                 << oa::oaIter<oa::oaOccInst>(modInst->getOccInsts(topOcc)).getNext()->getMasterOccurrence()
                 << " occmod: "
                 << oa::oaIter<oa::oaOccInst>(modInst->getOccInsts(topOcc)).getNext()->getMasterOccurrence()->getModule()
                 << " mod: " << modInst->getMasterModule() )
            */
            if (modInst->isModDesignInst()) {
                currentOcc->destroy();
                static_cast<oa::oaModDesignInst*>(modInst)->setMaster(uniquifiedModule->getDesign());
                currentOcc = oa::oaIter<oa::oaOccInst>(modInst->getOccInsts(topOcc)).getNext()->getMasterOccurrence();
            } else {
                cerr << "ERROR: Can not uniquify a ModModInst" << endl;
                QUIT_ON_ERROR;
            }
            DEBUG_PRINTLN("\t\t\tfinal insts of uniquified mod: " 
                          << uniquifiedModule->getOccurrences(topOcc).getCount())  
            assert(uniquifiedModule->getOccurrences(topOcc).getCount() == 1);
        }
        
        if (pathNameMember == pathName.getNumMembers()) {
            break;
        }
            
        // proceed down path hierarchy
        DEBUG_PRINTLN("\t\tdelving into instance " << pathName[pathNameMember].getValue());
        
        oa::oaOccInst *nextInst =  oa::oaOccInst::find(currentOcc, 
            oa::oaScalarName(oa::oaNativeNS(), pathName[pathNameMember].getValue()));
        assert(nextInst);
        currentOcc = nextInst->getMasterOccurrence();
        assert(currentOcc);
        
        ++pathNameMember;
    }

    return NULL;
}


// *****************************************************************************
// isUnique()
//
/// \brief Returns true if an occurrence is the single one of a module in its design.
///
/// \param occurrence
/// \return true, if this is the single occurrence of a module in its design
//
// *****************************************************************************
bool
isUnique(oa::oaOccurrence *occurrence) {
        assert(occurrence);

        // count how many times each module is instantiatied in this design.
        // start at the top of the hierarchy and explore down the tree.

        map<oa::oaModule*,int> count;
        list<oa::oaOccurrence*> toBeVisited;
        toBeVisited.push_back(occurrence->getTopOccurrence());
        while(!toBeVisited.empty()) {
                oa::oaOccurrence *currentOcc = toBeVisited.front();
                toBeVisited.pop_front();

                // further explore each of the instances
                oa::oaOccInst *inst;
                oa::oaIter<oa::oaOccInst> instIter(currentOcc->getInsts());
                while((inst = instIter.getNext())) {
                        oa::oaOccurrence *nextOcc = inst->getMasterOccurrence();
                        if (!nextOcc) {
                                cerr << "WARNING: Unknown submodule, occurrence can not be bound" << endl;
                        } else {
                                toBeVisited.push_back(nextOcc);
                        }
                }

                // increment the occurrence count (starting at 1 if this is the first)
                if (count.find(currentOcc->getModule()) == count.end()) {
                        count[currentOcc->getModule()] = 1;
                } else {
                        ++count[currentOcc->getModule()];
                }
        }

        // an occurrence is unique if it and every parent (recursively to the top)
        // is instantiated no more than once

        oa::oaOccurrence *currentOcc = occurrence;
        while(currentOcc != currentOcc->getTopOccurrence()) {
                if (count[currentOcc->getModule()] > 1) {
                        return false;
                }
                currentOcc = currentOcc->getOccInst()->getOccurrence();
        }

        return true;
}


// *****************************************************************************
// extract()
//
/// \brief Extracts and reimplements all local functionality in a new submodule.
///
/// A new module is created with the name <module name>_extracted. 
/// If the makeNewDesign parameter is true, this will be placed into a 
/// new design of the same name, otherwise, into the same design as the source. 
/// All functionality that was implemented inside the source module is
/// moved to the new design.  A single instance of the new design is
/// created inside the source and connected so that the behavior of
/// the source remains unchanged.
///
/// Any hierarchy that is contained inside the source module will not be
/// altered in any way.  The name and numbers of other instances and the functional
/// descriptions of those instances will not be changed.  Only the functional
/// behavior that is local to the source module is moved to the new
/// extracted one.
///
/// After extraction, source module will be purely structural.
///
/// This operation is particularly useful for exporting and importing the
/// functional component of a design while preserving the existing
/// hierarchy.  Algorithms and tools that do not support hierarchy (even to
/// the extent where it can be ignored) will find this useful, as the types of
/// objects in the extracted module are very simple: single bit inputs and 
/// outputs and an internal A/I graph between them.
///
/// \param source the source module
/// \param makeNewDesign if true, the copy will be placed into a new design
/// \return the extracted submodule
//
// *****************************************************************************
oa::oaModule*
extract(oa::oaModule *source, bool makeNewDesign) {
    assert(source);
    
    const oa::oaNativeNS nativeNS;

    // create new name
    oa::oaString     sourceString;
    source->getName(nativeNS, sourceString);
    oa::oaScalarName extractedName(nativeNS, sourceString+"_extracted");

    // 1. create target/extracted module
    oa::oaDesign *targetDesign = NULL;
    oa::oaModule *target = NULL;
    if (makeNewDesign) {
        // create new design

        oa::oaScalarName     libName, viewName;
        source->getDesign()->getLibName(libName);
        source->getDesign()->getViewName(viewName);

        targetDesign = oa::oaDesign::open(libName, 
                                          extractedName, 
                                          viewName, 
                                          oa::oaViewType::get(oa::oacNetlist), 'w');

        if (targetDesign) {
          target = oa::oaModule::create(targetDesign, extractedName);
          targetDesign->setTopModule(target);
        }
    } else {
        target = oa::oaModule::create(source->getDesign(), extractedName);
    }
    
    // did this fail?
    if (!target) {
      cerr << "ERROR: Could not create new extracted module (and/or design) of name " 
           << sourceString << "_extracted" << endl;
      QUIT_ON_ERROR;
    }

    list<ModRef> outputNodes;
    map<ModRef, oa::oaModNet*> inputNodes;
    map<ModRef, ModRef> extractedNodes;
    extractedNodes[ModGraph::getNull(source)] = ModGraph::getNull(target);
    extractedNodes[ModGraph::constantZero(source)] = ModGraph::constantZero(target);

    // 1. b. create one instance of the target module inside the source
    oa::oaModInst *targetInst;
    if (makeNewDesign) {
      targetInst = oa::oaModScalarInst::create(source, targetDesign);
    } else {
      targetInst = oa::oaModModuleScalarInst::create(source, target);
    }

    // 2. identify input terms
    //    these are all nets connected to input Term or output InstTerm
    // 3. identify output terms
    //    these are all nets connected to output Term or input instTerm
    oa::oaModNet *net;
    oa::oaIter<oa::oaModNet> netIter(source->getNets(oacNetIterSingleBit));
    while((net = netIter.getNext())) {
      ModRef parentNode = ModGraph::getNetToAiConnection(toBitNet(net));
      bool input = false, output = false;

      // connected terms
      oa::oaModTerm *term;
      oa::oaIter<oa::oaModTerm> termIter(net->getTerms(oacTermIterAll|oacTermIterEquivNets));
      while((term = termIter.getNext())) {
        if (term->getTermType() == oa::oacInputTermType) {
          input = true;
        } else if (term->getTermType() == oa::oacOutputTermType) {
          output = true;
        }
      }
      // connected inst terms
      oa::oaModInstTerm *instTerm;
      oa::oaIter<oa::oaModInstTerm> instTermIter(net->getInstTerms(oacInstTermIterAll|oacInstTermIterEquivNets));
      while((instTerm = instTermIter.getNext())) {
        term = instTerm->getTerm();
        assert(term);
        if (term->getTermType() == oa::oacInputTermType) {
          output = true;
        } else if (term->getTermType() == oa::oacOutputTermType) {
          input = true;
        }
      }

      // create and connect terminals in new module
      if (input) {

        // delay creation of input terminals
        // to determine if they are in the fanin of any
        // outputs or state bits
        inputNodes[parentNode] = net;

      } else if (output) {
        // create the terminal

        // a. net inside extracted module
        oa::oaModScalarNet *termNet = oa::oaModScalarNet::create(target);
        oa::oaScalarName termNetName;
        termNet->getName(termNetName);
        ModRef childNode = ModGraph::prepareNetToAiConnection(termNet);
        extractedNodes[parentNode] = childNode;
        outputNodes.push_back(parentNode);
        // b. term inside extracted module
        term = oa::oaModScalarTerm::create(termNet, termNetName, oa::oacOutputTermType);
        // c. inst term inside source module
        if (net->isImplicit()) {
          oa::oaModScalarNet *scalarNet = oa::oaModScalarNet::create(source);
          scalarNet->makeEquivalent(toBitNet(net));
          net = scalarNet;
        }
        instTerm = oa::oaModInstTerm::create(net, targetInst, term);
      }
    }

    // 3. copy all graph structure
    list<ModRef> all;
    ModGraph::getTransitiveFanin(outputNodes, all, true, true);
    all.splice(all.end(), outputNodes);

    // 3.a. partial pre-copy
    for(list<ModRef>::iterator nodeIter = all.begin(); nodeIter!=all.end(); nodeIter++) {
      ModRef node = *nodeIter;
      node = ModGraph::getNonInverted(node);

      // has this already been extracted?
      if (extractedNodes.find(node) != extractedNodes.end()) {
        continue;
      }

      // is this terminal an input?
      if (inputNodes.find(node) != inputNodes.end()) {
        // create the terminal

        // a. net inside extracted module
        oa::oaModScalarNet *termNet = oa::oaModScalarNet::create(target);
        oa::oaScalarName termNetName;
        termNet->getName(termNetName);
        ModRef childNode = ModGraph::prepareNetToAiConnection(termNet);
        extractedNodes[node] = childNode;
        // b. term inside extracted module
        oa::oaModTerm *term = oa::oaModScalarTerm::create(termNet, termNetName, oa::oacInputTermType);
        // c. inst term inside source module
        oa::oaModNet *net = inputNodes[node];
        if (net->isImplicit()) {
          oa::oaModScalarNet *scalarNet = oa::oaModScalarNet::create(source);
          scalarNet->makeEquivalent(toBitNet(net));
          net = scalarNet;
        }
        oa::oaModInstTerm::create(net, targetInst, term);
      } else {

        // otherwise, create a new node
        extractedNodes[node] = partialCopyNode(node, target);
      }

      // is this a sequential node?
      if (ModGraph::isSequential(node)) {
        // partial copy the asynchronous logic (in the SequentialData structure)

        // recurse on all triggers
        oagAi::Node::SequentialData *oldData = ModGraph::getSequentialData(node);
        if (oldData) {
          // can only support black box view
          assert(oldData->abstractionLevel == oagAi::Node::SequentialData::BLACK_BOX);
          for(map<string,oagAi::Ref>::iterator trigIter = oldData->signals.begin();
              trigIter != oldData->signals.end(); trigIter++) {
            // recurse on trigger fanin
            ModRef trigger = ModGraph::getNonInverted(ModRef(trigIter->second, source));
            ModGraph::getTransitiveFanin(trigger, all, true, true);
            all.push_back(trigger);
          }
        }
      }
    }

    // 3.b. complete copy
    for(list<ModRef>::iterator nodeIter = all.begin(); nodeIter!=all.end(); nodeIter++) {
      ModRef node = *nodeIter;
      node = ModGraph::getNonInverted(node);
      ModRef newNode = extractedNodes[node];

      switch(ModGraph::getNodeType(node)) {
      case oagAi::Node::CONSTANT0:
        break;

      case oagAi::Node::AND: {
        ModRef oldLeft = ModGraph::getAndLeft(node);
        assert(extractedNodes.find(ModGraph::getNonInverted(oldLeft)) != extractedNodes.end());
        ModRef newLeft = extractedNodes[ModGraph::getNonInverted(oldLeft)];
        if (ModGraph::isInverted(oldLeft)) {
          newLeft = ModGraph::notOf(newLeft);
        }
        ModRef oldRight = ModGraph::getAndRight(node);
        assert(extractedNodes.find(ModGraph::getNonInverted(oldRight)) != extractedNodes.end());
        ModRef newRight = extractedNodes[ModGraph::getNonInverted(oldRight)];
        if (ModGraph::isInverted(oldRight)) {
          newRight = ModGraph::notOf(newRight);
        }
        ModGraph::setAndLeft(newNode, newLeft);
        ModGraph::setAndRight(newNode, newRight);
        break;
        }

      case oagAi::Node::SEQUENTIAL: {
        ModRef oldNextState = ModGraph::getNextState(node);
        assert(extractedNodes.find(ModGraph::getNonInverted(oldNextState)) != extractedNodes.end());
        ModRef newNextState = extractedNodes[ModGraph::getNonInverted(oldNextState)];
        if (ModGraph::isInverted(oldNextState)) {
          newNextState = ModGraph::notOf(newNextState);
        }
        ModGraph::setNextState(newNode, newNextState);

        // finish SequentialData
        oagAi::Node::SequentialData *oldData = ModGraph::getSequentialData(node);
        oagAi::Node::SequentialData *newData = ModGraph::getSequentialData(extractedNodes[node]); 
        if (oldData) {
          // can only support black box view
          assert(oldData->abstractionLevel == oagAi::Node::SequentialData::BLACK_BOX);
          // copy each signal
          for(map<string,oagAi::Ref>::iterator trigIter = oldData->signals.begin();
              trigIter != oldData->signals.end(); trigIter++) {
            // continue on trigger fanin
            ModRef oldTrigger = ModRef(trigIter->second, source);
            assert(extractedNodes.find(ModGraph::getNonInverted(oldTrigger)) != extractedNodes.end());
            ModRef newTrigger = extractedNodes[ModGraph::getNonInverted(oldTrigger)];
            if (ModGraph::isInverted(oldTrigger)) {
              newTrigger = ModGraph::notOf(newTrigger);
            }
            // copy data
            newData->signals[trigIter->first] = newTrigger.ref;
          }
        }

        break;
        }
      case oagAi::Node::TERMINAL: {
        
        ModRef oldDriver = ModGraph::getTerminalDriver(node);
        assert(extractedNodes.find(ModGraph::getNonInverted(oldDriver)) != extractedNodes.end());
        ModRef newDriver = extractedNodes[ModGraph::getNonInverted(oldDriver)];
        if (ModGraph::isInverted(oldDriver)) {
          newDriver = ModGraph::notOf(newDriver);
        }
        ModGraph::setTerminalDriver(newNode, newDriver);

        break;
        }
      default:
        QUIT_ON_INTERNAL_ERROR;
        break;
      }
    }

    // 4. obliterate existing functional description in source
    //    set the driver of every TERMINAL node to be null
    netIter.reset();
    while((net = netIter.getNext())) {
      ModRef termNode = ModGraph::getNetToAiConnection(toBitNet(net));
      if (ModGraph::isTerminal(termNode)) {
        ModGraph::detach(termNode);
      }
    }
    
    return target;
}


// ------------------- the code below is not currently in use ------------------

#if 0
// *****************************************************************************
// flattenName()
//
/// \brief Creates a flattened version of a name.
///
/// The hierarchical delimiters are replaced with underscores.
///
/// \param name a hierarchical name
/// \return a string with a flattened version of the name
//
// *****************************************************************************
oa::oaString
flattenName(oa::oaName &name) {
    const oa::oaVerilogNS verilogNS;
    oa::oaString hierString, flatString;
    name.get(verilogNS, hierString);
    
    for(unsigned int i = 0; i < hierString.getLength(); ++i) {
        if (hierString[i] == '/') {
            flatString += '_';
        } else {
            flatString += hierString[i];   
        }
    }
    
    return flatString;
}
#endif

}
