/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron Hurst <ahurst@eecs.berkeley.edu>

ChangeLog:
2005-08-01: ChangeLog started
*/

#include <iostream>
#include "oaDesignDB.h"
#include "oagFunc.h"
#include "oagFuncManager.h"
#include "oagFuncQueryOcc.h"
#include "oagUtilOption.h"
#include "oagUtilOptionParser.h"
#include "oagFuncDdReasoningEngine.h"
#include <map>
#include <set>

#include "oagFuncDebug.h"

struct EquivalentInfo {
    bool    simpleNameCompare;
    map<oa::oaString, oa::oaString> mapFromOneToTwo;
    map<oa::oaString, oa::oaString> mapFromTwoToTwo;
};


// *****************************************************************************
// findOpenLib()
//
/// \brief Find an already open, opens from disk, or creates a library.
///
// *****************************************************************************
oa::oaLib *
findOpenLib(const oa::oaScalarName libName, const char *libString) {
    oa::oaLib *lib = NULL;

    if ((lib = oa::oaLib::find(libName)))
        ;
    else if (!oa::oaLib::exists(libString)) {
        lib = oa::oaLib::create(libName, libString);
        oa::oaLibDefList *defList = oa::oaLibDefList::get("lib.defs", 'a');
        oa::oaLibDef::create(defList, libName, libString);
        defList->save();
    } else {
        lib = oa::oaLib::open(libName, libString); 
    }

    return lib;
}


// *****************************************************************************
// readEquivalentNetMap()
//
/// \brief Reads a text file that describe the name mapping between equivalent nets in two designs.
///
/// Each line of the file describes a 1-to-1 equivalence.  The two nets names
/// are separated by a comma, and the line is terminated with a newline.  The
/// nets should be specified by their full hierarchical path name (in Verilog
/// format), such as:
///
/// \li u1.u4.\my_reg[24] .some_net
///
/// If there is a many-to-1 equivalence-- for example, if two nets in were 
/// determined to be functionally equivalent and collapsed into a single
/// representative-- this can be legally specified with multiple lines in
/// the map file.  A net can appear in more than one equivalence.
//
// *****************************************************************************
void 
readEquivalentNetMap(const char *filename, 
                     EquivalentInfo &equivInfo) {

    equivInfo.simpleNameCompare = false;

    FILE *mapFile = fopen(filename, "r");
    if (!mapFile) {
        std::cerr << "ERROR: Could not open name mapping file \'" 
            << filename << "\'." << std::endl;
        exit(0);
    }
    
    char c = ' ';
    char line[2048];

    while(c != EOF) {
        // read a line
        int pos = 0;
        while((c = fgetc(mapFile)) != EOF && c != '\n') {
            line[pos++] = c;
        }
        line[pos] = '\0';

        // process line
        oa::oaString equiv1Name, equiv2Name;
        char *token;
        
        // ignore lines that do not have two non-NULL tokens
        if ((token = strtok(line, ",\r\n")) && strlen(token) > 0) {
            equiv1Name = token;
        } else {
            continue;
        }
        if ((token = strtok(NULL, ",\r\n")) && strlen(token) > 0) {
            equiv2Name = token;
        } else {
            continue;
        }
        
        if (equivInfo.mapFromOneToTwo.find(equiv1Name) == equivInfo.mapFromOneToTwo.end()) {
            equivInfo.mapFromOneToTwo[equiv1Name] = equiv2Name;
            DEBUG_PRINTLN("\"" << equiv1Name << "\"" 
                 << " -> " << "\"" << equiv2Name << "\"")
        } else {
            // is this is part of a one to many mapping?
            equivInfo.mapFromTwoToTwo[equiv2Name] = equivInfo.mapFromOneToTwo[equiv1Name];
            DEBUG_PRINTLN("\"" << equiv1Name << "\""
                << " -> " << "\"" << equiv2Name << "\"")        
        }
    }
}


// *****************************************************************************
// setVariablesForMatchingRefs()
//
/// \brief Takes two lists of refs, and creates and sets a new variable for each matching pair.
///
/// The query1 object is the object in which the function of the refs in occRefs1
/// should be set, and the query2 object is the object in which the function
/// of the refs in occRefs2 should be set.
///
/// If two or more refs have any connected nets whose names have been marked
/// as equivalent, those refs are set with a single variable.  
///
/// If a ref can not be matched, a warning is displayed and a variable is
/// assigned to that ref.
///
/// \param occRefs1
/// \param query1
/// \param occRefs2
/// \param query2
/// \param equivInfo
/// \param engine
///
// *****************************************************************************
void
setVariablesForMatchingNets(list<oagFunc::OccRef> &occRefs1, 
                            oagFunc::QueryOcc &query1,
                            list<oagFunc::OccRef> &occRefs2, 
                            oagFunc::QueryOcc &query2,
                            EquivalentInfo &equivInfo,
                            oagFunc::DdReasoningEngine &engine) {

    std::cout << "Setting variables for " 
            << occRefs1.size() << " nets in design1 and " 
            << occRefs2.size() << " nets in design2 ..." << std::endl;

    oa::oaVerilogNS verilogNS;
    map<oa::oaString, void*> setVariables;
    set<oa::oaString> matchedVariables;
    oa::oaString netNameString;

    // first pass: a new variable is created for every ref in the
    // second design
    for(list<oagFunc::OccRef>::iterator it = occRefs2.begin(); 
        it!=occRefs2.end(); ++it) {
        
        // get all connected named nets
        set<oagFunc::OccRef> connectedRefs;
        set<oa::oaOccBitNet*> connectedNets;
        oagFunc::OccGraph::getAllConnections(*it, connectedNets, connectedRefs, true, true, true, true, true);
        DEBUG_PRINTLN("list2 ref: " << it->ref << " has " << connectedNets.size() << " connected nets")
        
        // first, has the variable already been set on any connected net?
        void *var = NULL;
        netNameString = "UNNAMED";
        DEBUG_PRINT("\t")
        for(set<oa::oaOccBitNet*>::iterator netIter = connectedNets.begin();
            netIter != connectedNets.end(); ++netIter) {
            (*netIter)->getPathName(verilogNS, netNameString);
            DEBUG_PRINTMORE("  \"" << netNameString << "\"")
            
            if (equivInfo.mapFromTwoToTwo.find(netNameString) != equivInfo.mapFromTwoToTwo.end()) {
                netNameString = equivInfo.mapFromTwoToTwo[netNameString];
                DEBUG_PRINTMORE("->\"" << netNameString << "\"")                
            }
            if (setVariables.find(netNameString) != setVariables.end()) {
                var = setVariables[netNameString];
                break;
            }            
        }
        DEBUG_PRINTMORE("\n");
        
        // second, create a new variable if no variable has been set
        if (!var) {
            var = engine.newVariable();
        }
        query2.set(*it, var);
        
        // third, set the variable on every connected net
        for(set<oa::oaOccBitNet*>::iterator netIter = connectedNets.begin();
            netIter != connectedNets.end(); ++netIter) {
            (*netIter)->getPathName(verilogNS, netNameString);

            setVariables[netNameString] = var;
            if (equivInfo.mapFromTwoToTwo.find(netNameString) != equivInfo.mapFromTwoToTwo.end()) {
                netNameString = equivInfo.mapFromTwoToTwo[netNameString];
            }
            setVariables[netNameString] = var;
        }
    }
    
    // second pass: if the net can be matched, then the same variable
    // from the corresponding ref in the second design, otherwise a new
    // one is created
    for(list<oagFunc::OccRef>::iterator it = occRefs1.begin(); 
        it!=occRefs1.end(); ++it) {
        
        // get all connected named nets
        set<oagFunc::OccRef> connectedRefs;
        set<oa::oaOccBitNet*> connectedNets;
        oagFunc::OccGraph::getAllConnections(*it, connectedNets, connectedRefs, true, true, true, true, true);
        DEBUG_PRINTLN("list1 ref: " << it->ref << " has " << connectedNets.size() << " connected nets")
        
        // has the variable already been set on any connected net?
        void *var = NULL;
        netNameString = "UNNAMED";
        DEBUG_PRINT("\t")        
        for(set<oa::oaOccBitNet*>::iterator netIter = connectedNets.begin();
            netIter != connectedNets.end(); ++netIter) {
            (*netIter)->getPathName(verilogNS, netNameString);
            DEBUG_PRINTMORE("  \"" << netNameString << "\"")            
            
            if (equivInfo.mapFromOneToTwo.find(netNameString) != equivInfo.mapFromOneToTwo.end()) {
                netNameString = equivInfo.mapFromOneToTwo[netNameString];
                DEBUG_PRINTMORE("->\"" << netNameString << "\"")                
                
                if (setVariables.find(netNameString) != setVariables.end()) {
                    var = setVariables[netNameString];
                    matchedVariables.insert(netNameString);
                    break;
                }            
            }
            
            else if (equivInfo.simpleNameCompare) {
                DEBUG_PRINTMORE("->\"" << netNameString << "\"")                
                
                if (setVariables.find(netNameString) != setVariables.end()) {
                    var = setVariables[netNameString];
                    matchedVariables.insert(netNameString);
                    break;
                }
            }
        }
        DEBUG_PRINTMORE("\n");

        if (!var) {
            std::cout << "WARNING: Net " << netNameString 
                 << " not present in design 2" << std::endl;
            var = engine.newVariable();
        }
        query1.set(*it, var);
        
    }
    
    // third pass: for any refs in the second design that weren't eventually
    // matched by a ref in the first design, a warning is displayed
    for(list<oagFunc::OccRef>::iterator it = occRefs2.begin(); 
        it!=occRefs2.end(); ++it) {

        // get all connected named nets
        set<oagFunc::OccRef> connectedRefs;
        set<oa::oaOccBitNet*> connectedNets;
        oagFunc::OccGraph::getAllConnections(*it, connectedNets, connectedRefs, true, true, true, true, true);
        
        // first, has the variable already been set on any connected net?
        bool matchedSomething = false;
        netNameString = "UNNAMED";
        for(set<oa::oaOccBitNet*>::iterator netIter = connectedNets.begin();
            netIter != connectedNets.end(); ++netIter) {
            (*netIter)->getPathName(verilogNS, netNameString);
            
            if (equivInfo.mapFromTwoToTwo.find(netNameString) != equivInfo.mapFromTwoToTwo.end()) {
                netNameString = equivInfo.mapFromTwoToTwo[netNameString];
            }
            if (matchedVariables.find(netNameString) != matchedVariables.end()) {
                matchedSomething = true;
                break;
            }            
        }
        
        if (!matchedSomething) {
            std::cout << "WARNING: Net " << netNameString 
                 << " not present in design 1" << std::endl;
        }
    }

}



// *****************************************************************************
// testVariablesForMatchingNets()
//
/// \brief Takes two lists of refs, and tests the functionality of each matching pair.
///
/// The query1 object is the object in which the function of the refs in occRefs1
/// should be queried, and the query2 object is the object in which the function
/// of the refs in occRefs2 should be queried.
///
/// If a ref can not be matched, a warning is displayed.
///
/// \param occRefs1
/// \param query1
/// \param occRefs2
/// \param query2
/// \param compareNextStateFunctions
/// \param equivInfo
/// \param engine
/// \param printGoodCubes
/// \param printBadCubes
///
// *****************************************************************************
void
testVariablesForMatchingNets(list<oagFunc::OccRef> &occRefs1, 
                            oagFunc::QueryOcc &query1,
                            list<oagFunc::OccRef> &occRefs2, 
                            oagFunc::QueryOcc &query2,
                            bool compareNextStateFunctions,
                            EquivalentInfo &equivInfo,
                            oagFunc::DdReasoningEngine &engine,
                            bool printGoodCubes, bool printBadCubes) {

    oa::oaVerilogNS verilogNS;
    map<oa::oaString, DdNode *> computedFunctions;
    set<oa::oaString> matchedVariables;
    oa::oaString netNameString;

    // first pass: a new variable is created for every ref in the
    // second design
    for(list<oagFunc::OccRef>::iterator it = occRefs2.begin(); 
        it!=occRefs2.end(); ++it) {
        
        // get all connected named nets
        set<oagFunc::OccRef> connectedRefs;
        set<oa::oaOccBitNet*> connectedNets;
        oagFunc::OccGraph::getAllConnections(*it, connectedNets, connectedRefs, true, true, true, true, true);
        DEBUG_PRINTLN("list2 ref: " << it->ref << " has " << connectedNets.size() << " connected nets")
        
        // first, has the variable already been set on any connected net?
        DdNode *f2 = NULL;
        netNameString = "UNNAMED";
        DEBUG_PRINT("\t")
        for(set<oa::oaOccBitNet*>::iterator netIter = connectedNets.begin();
            netIter != connectedNets.end(); ++netIter) {
            (*netIter)->getPathName(verilogNS, netNameString);
            DEBUG_PRINTMORE("  \"" << netNameString << "\"")
            
            if (equivInfo.mapFromTwoToTwo.find(netNameString) != equivInfo.mapFromTwoToTwo.end()) {
                netNameString = equivInfo.mapFromTwoToTwo[netNameString];
                DEBUG_PRINTMORE("->\"" << netNameString << "\"")
            }
            if (computedFunctions.find(netNameString) != computedFunctions.end()) {
                f2 = computedFunctions[netNameString];
                break;
            }            
        }
        DEBUG_PRINTMORE("\n");
        
        // second, get the function if it hasn't been computed
        if (!f2) {
            if (compareNextStateFunctions) {
                // this should be a state bit.  we want to compare
                // its next state function
                assert(oagFunc::OccGraph::isSequential(*it));
                oagFunc::OccRef nextStateFunction = oagFunc::OccGraph::getNextState(*it);            
                f2 = static_cast<DdNode*>(query2.get(nextStateFunction));
            } else {
                f2 = static_cast<DdNode*>(query2.get(*it));
            }
        }
        
        // third, set the variable on every connected net
        for(set<oa::oaOccBitNet*>::iterator netIter = connectedNets.begin();
            netIter != connectedNets.end(); ++netIter) {
            (*netIter)->getPathName(verilogNS, netNameString);

            computedFunctions[netNameString] = f2;
            if (equivInfo.mapFromTwoToTwo.find(netNameString) != equivInfo.mapFromTwoToTwo.end()) {
                netNameString = equivInfo.mapFromTwoToTwo[netNameString];
            }
            computedFunctions[netNameString] = f2;
        }
    }
    
    // second pass: if the net can be matched, then the same variable
    // from the corresponding ref in the second design, otherwise a new
    // one is created
    for(list<oagFunc::OccRef>::iterator it = occRefs1.begin(); 
        it!=occRefs1.end(); ++it) {
        
        // get all connected named nets
        set<oagFunc::OccRef> connectedRefs;
        set<oa::oaOccBitNet*> connectedNets;
        oagFunc::OccGraph::getAllConnections(*it, connectedNets, connectedRefs, true, true, true, true, true);
        DEBUG_PRINTLN("list1 ref: " << it->ref << " has " << connectedNets.size() << " connected nets")
        
        // has the variable already been set on any connected net?
        DdNode *f2 = NULL;
        netNameString = "UNNAMED";
        DEBUG_PRINT("\t")
        for(set<oa::oaOccBitNet*>::iterator netIter = connectedNets.begin();
            netIter != connectedNets.end(); ++netIter) {
            (*netIter)->getPathName(verilogNS, netNameString);
            DEBUG_PRINTMORE("  \"" << netNameString << "\"")
            
            if (equivInfo.mapFromOneToTwo.find(netNameString) != equivInfo.mapFromOneToTwo.end()) {
                netNameString = equivInfo.mapFromOneToTwo[netNameString];
                DEBUG_PRINTMORE("->\"" << netNameString << "\"")
                
                if (computedFunctions.find(netNameString) != computedFunctions.end()) {
                    f2 = computedFunctions[netNameString];
                    matchedVariables.insert(netNameString);
                    break;
                }
            }

            else if (equivInfo.simpleNameCompare) {
                DEBUG_PRINTMORE("->\"" << netNameString << "\"")                
                
                if (computedFunctions.find(netNameString) != computedFunctions.end()) {
                    f2 = computedFunctions[netNameString];
                    matchedVariables.insert(netNameString);
                    break;
                }
            }
        }
        DEBUG_PRINTMORE("\n");
        
        if (!f2) {
            std::cout << "\tNONCOMPARABLE: Net " << netNameString 
                 << " not present in design 2" << std::endl;
        } else {
        
            DdNode *f1;
            if (compareNextStateFunctions) {
                // this should be a state bit.  we want to compare
                // its next state function
                assert(oagFunc::OccGraph::isSequential(*it));
                oagFunc::OccRef nextStateFunction = oagFunc::OccGraph::getNextState(*it);            
                f1 = static_cast<DdNode*>(query1.get(nextStateFunction));
            } else {
                f1 = static_cast<DdNode*>(query1.get(*it));
            }

            if (f1 == f2) {
                std::cout << "\tEQUIVALENT: Net " << netNameString << std::endl;
                if (printGoodCubes) {
                    std::cout << std::endl;
                    engine.print(f1);
                    std::cout << "-------------------------------------------" << std::endl;
                }
            } else {
                std::cout << "\tNOT equivalent: Net " << netNameString << std::endl;
                if (printBadCubes) {
                    std::cout << std::endl << "Cell1: " << std::endl;
                    engine.print(f1);
                    std::cout << std::endl << "Cell2: " << std::endl;
                    engine.print(f2);
                    std::cout << std::endl << "Difference: " << std::endl;
                    engine.print(engine.xorOf(f1, f2));
                    std::cout << "-------------------------------------------" << std::endl;
                }
            }
        }
    }
    
    // third pass: for any refs in the second design that weren't eventually
    // matched by a ref in the first design, a warning is displayed
    for(list<oagFunc::OccRef>::iterator it = occRefs2.begin(); 
        it!=occRefs2.end(); ++it) {

        // get all connected named nets
        set<oagFunc::OccRef> connectedRefs;
        set<oa::oaOccBitNet*> connectedNets;
        oagFunc::OccGraph::getAllConnections(*it, connectedNets, connectedRefs, true, true, true, true, true);
        
        // first, has the variable already been set on any connected net?
        bool matchedSomething = false;
        netNameString = "UNNAMED";
        for(set<oa::oaOccBitNet*>::iterator netIter = connectedNets.begin();
            netIter != connectedNets.end(); ++netIter) {
            (*netIter)->getPathName(verilogNS, netNameString);
            
            if (equivInfo.mapFromTwoToTwo.find(netNameString) != equivInfo.mapFromTwoToTwo.end()) {
                netNameString = equivInfo.mapFromTwoToTwo[netNameString];
            }
            if (matchedVariables.find(netNameString) != matchedVariables.end()) {
                matchedSomething = true;
                break;
            }            
        }
        
        if (!matchedSomething) {
            std::cout << "\tNONCOMPARABLE: Net " << netNameString 
                 << " not present in design 1" << std::endl;
        }
    }
}


//------------------------------------------------------------------------------

// *****************************************************************************
// main()
//
// *****************************************************************************
int main(int argc, const char **argv) {

    std::cout << "*****************************************************************************" << std::endl;
    std::cout << "Tool:   equivCheck      Package:    oagFunc     version:  01/2007" << std::endl;
    std::cout << "*****************************************************************************" << std::endl;
    std::cout << std::endl;

    oagUtil::OptionParser options("Check the equivalence of two designs using BDDs.");

    oagUtil::Option *cell1Opt = options.add("cell1", "Input cell name of the first design", true, "cell");
    oagUtil::Option *cell2Opt = options.add("cell2", "Input cell name of the second design", true, "cell");
    oagUtil::Option *lib1Opt = options.add("lib", "Input library name for the first design", true, "lib");
    oagUtil::Option *lib2Opt = options.add("lib", "Input library name for the second design (default: same lib)", false, "lib");
    oagUtil::Option *view1Opt = options.add("lib", "Input view name for the first design (default: netlist)", false, "lib");
    oagUtil::Option *view2Opt = options.add("lib", "Input view name for the second design (default: netlist)", false, "lib");
    oagUtil::Option *mapOpt = options.add("map", "Input file describing names of equivalent states", false, "file");
    oagUtil::Option *net1Opt = options.add("net1", "Net name to check equivalence in first design", false, "net");
    oagUtil::Option *net2Opt = options.add("net2", "Net name to check equivalence in second design", false, "net");
    oagUtil::Option *allOpt = options.add("all", "Check equivalence of all states and primary outputs", false);
    oagUtil::Option *printGoodCubesOpt = options.add("print_good_cubes", "Print cubes for comparing signals", false);
    oagUtil::Option *printBadCubesOpt = options.add("print_bad_cubes", "Print cubes for miscomparing signals", false);

    if (!options.parse(argc, argv)) {
        std::cerr << options.getMessage();
        exit(0);
    }


    const char *cell1String = cell1Opt->getValue();
    const char *cell2String = cell2Opt->getValue();
    const char *lib1String = lib1Opt->getValue();
    const char *lib2String 
        = lib2Opt->isGiven() ? lib2Opt->getValue() : lib1String;
    const char *view1String 
        = view1Opt->isGiven() ? view1Opt->getValue() : "netlist";
    const char *view2String 
        = view2Opt->isGiven() ? view2Opt->getValue() : "netlist";

    std::cout << "Checking equivalence for design \'" << cell1String 
        << "\' and design \'" << cell2String << "\' ..." << std::endl;

    try {

        // Initialize OpenAccess

        oa::oaDesignInit();
        oa::oaLibDefList::openLibs();
        oagFunc::initialize();

        const oa::oaNativeNS  nativeNS;
        const oa::oaVerilogNS verilogNS;
        const oa::oaScalarName lib1Name(nativeNS, lib1String);
        const oa::oaScalarName view1Name(nativeNS, view1String);
        const oa::oaScalarName lib2Name(nativeNS, lib2String);
        const oa::oaScalarName view2Name(nativeNS, view2String);
        // oa::oaViewType *viewType = oa::oaViewType::get(oa::oacNetlist);

        // Find, create, or load the design library and view
        
        oa::oaLib *library1 = findOpenLib(lib1Name, lib1String);
        assert(library1);
        library1->getAccess(oa::oacReadLibAccess);
        oa::oaLib *library2 = findOpenLib(lib2Name, lib2String);
        assert(library2);
        library2->getAccess(oa::oacReadLibAccess);
        
        // Read equivalent net mapping, if given

        EquivalentInfo equivInfo;
        if (mapOpt->isGiven()) {
            readEquivalentNetMap(mapOpt->getValue(), equivInfo);
        } else {
            equivInfo.simpleNameCompare = true;
            std::cout << "Using simple name matching" << std::endl;
        }

        // Open designs (read only)

        oa::oaDesign *cell1 = oa::oaDesign::open(lib1Name, 
                oa::oaScalarName(nativeNS, cell1String), 
                view1Name, 'r');
        oa::oaDesign *cell2 = oa::oaDesign::open(lib2Name, 
                oa::oaScalarName(nativeNS, cell2String), 
                view2Name, 'r');

        oa::oaOccurrence *occurrence1 = cell1->getTopOccurrence();
        assert(occurrence1);
        oa::oaOccurrence *occurrence2 = cell2->getTopOccurrence();
        assert(occurrence2);

        // Create reasoning engine

        oagFunc::DdReasoningEngine  engine;

        // Create query objects

        oagFunc::QueryOcc query1(cell1, &engine);
        oagFunc::QueryOcc query2(cell2, &engine);

        // Define functionality for inputs

        std::cout << "Matching inputs ..." << std::endl;
        list<oagFunc::OccRef> inputOccRefs1, inputOccRefs2;    
        oagFunc::OccGraph::getInputs(occurrence1, inputOccRefs1);
        oagFunc::OccGraph::getInputs(occurrence2, inputOccRefs2);

        setVariablesForMatchingNets(inputOccRefs1, query1, inputOccRefs2, query2,
                equivInfo, engine);

        // Define functionality for states
        
        std::cout << "Matching states ..." << std::endl;
        list<oagFunc::OccRef> stateOccRefs1, stateOccRefs2;
        oagFunc::OccGraph::getStates(occurrence1, stateOccRefs1);
        oagFunc::OccGraph::getStates(occurrence2, stateOccRefs2);
        
        setVariablesForMatchingNets(stateOccRefs1, query1, stateOccRefs2, query2,
                equivInfo, engine);

        // Compare two nets

        if (net1Opt->isGiven() && net2Opt->isGiven()) {

            oa::oaName net1Name(verilogNS, net1Opt->getValue());
            oa::oaName net2Name(verilogNS, net2Opt->getValue());
            oa::oaOccNet *net1 = oa::oaOccNet::find(cell1->getTopOccurrence(), net1Name);
            if (!net1) {
                std::cerr << "ERROR: Unknown net " << net1Opt->getValue() << std::endl;
                exit(0);    
            }
            if (net1->getNumBits() != 1) {
                cerr << "ERROR: Not a single bit net: " << net1Opt->getValue()<< std::endl;
                exit(0);
            }

            oa::oaOccNet *net2 = oa::oaOccNet::find(cell2->getTopOccurrence(), net2Name);
            if (!net2) {
                std::cerr << "ERROR: Unknown net " << net2Opt->getValue() << std::endl;
                exit(0);    
            }
            if (net2->getNumBits() != 1) {
                cerr << "ERROR: Not a single bit net: " << net2Opt->getValue()<< std::endl;
                exit(0);
            }

            DdNode *f1 
                = static_cast<DdNode*>(query1.get(oagFunc::toBitNet(net1)));
            DdNode *f2 
                = static_cast<DdNode*>(query2.get(oagFunc::toBitNet(net2)));

            std::cout << "-------------------------------------------" 
                << std::endl;
            if (f1 == f2) {
                std::cout << net1Opt->getValue() << "," << net2Opt->getValue() 
                    << " are functionally EQUIVALENT." << std::endl;
                if (printGoodCubesOpt->isGiven()) {
                    std::cout << std::endl;
                    engine.print(f1);
                }
            } else {
                std::cout << net1Opt->getValue() << "," << net2Opt->getValue()
                    << " are NOT functionally equivalent." << std::endl;
                if (printBadCubesOpt->isGiven()) {
                    std::cout << std::endl << "Difference: " << std::endl;
                   engine.print(engine.xorOf(f1, f2));
                }
            }
            std::cout << "-------------------------------------------" << std::endl;

            // compare all outputs / states

        } else if (allOpt->isGiven()) {

            std::cout << "Comparing outputs..." << std::endl;
            list<oagFunc::OccRef> outputOccRefs1, outputOccRefs2;
            oagFunc::OccGraph::getOutputs(occurrence1, outputOccRefs1);
            oagFunc::OccGraph::getOutputs(occurrence2, outputOccRefs2);

            testVariablesForMatchingNets(outputOccRefs1, query1, outputOccRefs2, query2,
                    false, equivInfo, engine, 
                    printGoodCubesOpt->isGiven(), printBadCubesOpt->isGiven());

            std::cout << "Comparing next state functions..." << std::endl;
            testVariablesForMatchingNets(stateOccRefs1, query1, stateOccRefs2, query2,
                    true, equivInfo, engine, 
                    printGoodCubesOpt->isGiven(), printBadCubesOpt->isGiven());         
        }

    } catch(oa::oaException &e) {
        std::cerr << "ERROR: " << e.getMsg() << std::endl;
        return 1;
    }

    return 0;
}
